/*
 * Decompiled with CFR 0.152.
 */
package schemacrawler.tools.databaseconnector;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.DriverPropertyInfo;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import schemacrawler.schemacrawler.exceptions.DatabaseAccessException;
import schemacrawler.schemacrawler.exceptions.InternalRuntimeException;
import schemacrawler.schemacrawler.exceptions.WrappedSQLException;
import schemacrawler.tools.databaseconnector.SingleUseUserCredentials;
import schemacrawler.tools.databaseconnector.UserCredentials;
import us.fatehi.utility.Utility;
import us.fatehi.utility.string.StringFormat;

public final class DatabaseConnectionSource
implements Supplier<Connection> {
    private static final Logger LOGGER = Logger.getLogger(DatabaseConnectionSource.class.getName());
    private final Map<String, String> connectionProperties;
    private final String connectionUrl;
    private UserCredentials userCredentials;

    private static Properties safeProperties(Properties properties) {
        Properties logProperties = new Properties(properties);
        logProperties.remove("password");
        return logProperties;
    }

    public DatabaseConnectionSource(String connectionUrl) {
        this(connectionUrl, null);
    }

    public DatabaseConnectionSource(String connectionUrl, Map<String, String> connectionProperties) {
        this.connectionUrl = Utility.requireNotBlank((String)connectionUrl, (String)"No database connection URL provided");
        this.connectionProperties = connectionProperties;
        this.userCredentials = new SingleUseUserCredentials();
    }

    @Override
    public Connection get() {
        String user = this.userCredentials.getUser();
        String password = this.userCredentials.getPassword();
        return this.getConnection(user, password);
    }

    public String getConnectionUrl() {
        return this.connectionUrl;
    }

    public Driver getJdbcDriver() throws SQLException {
        return this.getJdbcDriver(this.connectionUrl);
    }

    public UserCredentials getUserCredentials() {
        return this.userCredentials;
    }

    public void setUserCredentials(UserCredentials userCredentials) {
        this.userCredentials = Objects.requireNonNull(userCredentials, "No user credentials provided");
    }

    public String toString() {
        String jdbcDriverClass = "<unknown>";
        try {
            Driver jdbcDriver = this.getJdbcDriver();
            jdbcDriverClass = jdbcDriver.getClass().getName();
        }
        catch (SQLException e) {
            jdbcDriverClass = "<unknown>";
        }
        StringBuilder builder = new StringBuilder(1024);
        builder.append("driver=").append(jdbcDriverClass).append(System.lineSeparator());
        builder.append("url=").append(this.connectionUrl).append(System.lineSeparator());
        return builder.toString();
    }

    private Properties createConnectionProperties(String connectionUrl, String user, String password) {
        Properties jdbcConnectionProperties;
        List<String> skipProperties = Arrays.asList("server", "host", "port", "database", "urlx", "user", "password", "url");
        try {
            Driver jdbcDriver = this.getJdbcDriver(connectionUrl);
            DriverPropertyInfo[] propertyInfo = jdbcDriver.getPropertyInfo(this.connectionUrl, new Properties());
            HashMap<String, Boolean> jdbcDriverProperties = new HashMap<String, Boolean>();
            for (DriverPropertyInfo driverPropertyInfo : propertyInfo) {
                String jdbcPropertyName = driverPropertyInfo.name.toLowerCase();
                if (skipProperties.contains(jdbcPropertyName)) continue;
                jdbcDriverProperties.put(jdbcPropertyName, driverPropertyInfo.required);
            }
            jdbcConnectionProperties = new Properties();
            if (user != null) {
                jdbcConnectionProperties.put("user", user);
            }
            if (password != null) {
                jdbcConnectionProperties.put("password", password);
            }
            if (this.connectionProperties != null) {
                for (Map.Entry entry : this.connectionProperties.entrySet()) {
                    String property = (String)entry.getKey();
                    String value = (String)entry.getValue();
                    if (!jdbcDriverProperties.containsKey(property.toLowerCase()) || value == null) continue;
                    jdbcConnectionProperties.put(property, value);
                }
            }
        }
        catch (SQLException e) {
            throw new InternalRuntimeException("Could not get connection properties", (Throwable)e);
        }
        return jdbcConnectionProperties;
    }

    private Connection getConnection(String user, String password) {
        if (Utility.isBlank((CharSequence)user)) {
            LOGGER.log(Level.WARNING, "Database user is not provided");
        }
        if (Utility.isBlank((CharSequence)password)) {
            LOGGER.log(Level.WARNING, "Database password is not provided");
        }
        Properties jdbcConnectionProperties = this.createConnectionProperties(this.connectionUrl, user, password);
        try {
            LOGGER.log(Level.INFO, (Supplier<String>)new StringFormat("Making connection to %s%nfor user '%s', with properties %s", new Object[]{this.connectionUrl, user, DatabaseConnectionSource.safeProperties(jdbcConnectionProperties)}));
            Driver driver = this.getJdbcDriver(this.connectionUrl);
            Connection connection = driver.connect(this.connectionUrl, jdbcConnectionProperties);
            LOGGER.log(Level.INFO, (Supplier<String>)new StringFormat("Opened database connection <%s>", new Object[]{connection}));
            jdbcConnectionProperties.remove("password");
            return connection;
        }
        catch (SQLException e) {
            String username = user != null ? String.format("user '%s'", user) : "unspecified user";
            throw new DatabaseAccessException(String.format("Could not connect to <%s>, for <%s>, with properties <%s>", this.connectionUrl, username, DatabaseConnectionSource.safeProperties(jdbcConnectionProperties)), e);
        }
    }

    private Driver getJdbcDriver(String connectionUrl) throws SQLException {
        try {
            return DriverManager.getDriver(connectionUrl);
        }
        catch (SQLException e) {
            throw new WrappedSQLException(String.format("Could not find a suitable JDBC driver for database connection URL <%s>", this.connectionUrl), e);
        }
    }
}

