/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "NetworkProcess.h"

#include "AppPrivacyReport.h"
#include "ArgumentCoders.h"
#include "Attachment.h"
#include "CacheModel.h"
#include "DataReference.h"
#include "Decoder.h"
#include "DownloadID.h"
#include "DownloadManager.h"
#include "FormDataReference.h"
#include "HandleMessage.h"
#include "NavigatingToAppBoundDomain.h"
#include "NetworkProcessCreationParameters.h"
#include "NetworkProcessMessages.h"
#include "QuotaIncreaseRequestIdentifier.h"
#include "SandboxExtension.h"
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include "ShouldGrandfatherStatistics.h"
#endif
#include "WebCoreArgumentCoders.h"
#include "WebPageNetworkParameters.h"
#include "WebPageProxyIdentifier.h"
#if ENABLE(SERVICE_WORKER)
#include "WebPushMessage.h"
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include "WebResourceLoadStatisticsStore.h"
#endif
#include "WebsiteData.h"
#include "WebsiteDataFetchOption.h"
#include "WebsiteDataStoreParameters.h"
#include "WebsiteDataType.h"
#include <WebCore/CertificateInfo.h>
#include <WebCore/ClientOrigin.h>
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <WebCore/CrossSiteNavigationDataTransfer.h>
#endif
#if USE(CURL)
#include <WebCore/CurlProxySettings.h>
#endif
#include <WebCore/HTTPCookieAcceptPolicy.h>
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <WebCore/NetworkStorageSession.h>
#endif
#include <WebCore/PageIdentifier.h>
#include <WebCore/PrivateClickMeasurement.h>
#include <WebCore/ProcessIdentifier.h>
#include <WebCore/RegistrableDomain.h>
#include <WebCore/ResourceError.h>
#include <WebCore/ResourceRequest.h>
#include <WebCore/ResourceResponse.h>
#include <WebCore/SecurityOriginData.h>
#if USE(SOUP)
#include <WebCore/SoupNetworkProxySettings.h>
#endif
#include <WebCore/StoredCredentialsPolicy.h>
#include <optional>
#include <pal/SessionID.h>
#if (ENABLE(INTELLIGENT_TRACKING_PREVENTION) && ENABLE(APP_BOUND_DOMAINS))
#include <wtf/HashSet.h>
#endif
#include <wtf/OptionSet.h>
#include <wtf/Seconds.h>
#include <wtf/URLHash.h>
#include <wtf/Vector.h>
#include <wtf/WallTime.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace NetworkProcess {

void CreateNetworkConnectionToWebProcess::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<IPC::Attachment>&&, WebCore::HTTPCookieAcceptPolicy&&)>&& completionHandler)
{
    std::optional<std::optional<IPC::Attachment>> connectionIdentifier;
    decoder >> connectionIdentifier;
    if (!connectionIdentifier) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<WebCore::HTTPCookieAcceptPolicy> cookieAcceptPolicy;
    decoder >> cookieAcceptPolicy;
    if (!cookieAcceptPolicy) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*connectionIdentifier), WTFMove(*cookieAcceptPolicy));
}

void CreateNetworkConnectionToWebProcess::cancelReply(CompletionHandler<void(std::optional<IPC::Attachment>&&, WebCore::HTTPCookieAcceptPolicy&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<IPC::Attachment>>::create(), IPC::AsyncReplyError<WebCore::HTTPCookieAcceptPolicy>::create());
}

void CreateNetworkConnectionToWebProcess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<IPC::Attachment>& connectionIdentifier, WebCore::HTTPCookieAcceptPolicy cookieAcceptPolicy)
{
    encoder.get() << connectionIdentifier;
    encoder.get() << cookieAcceptPolicy;
    connection.sendSyncReply(WTFMove(encoder));
}

void FetchWebsiteData::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebKit::WebsiteData&&)>&& completionHandler)
{
    std::optional<WebKit::WebsiteData> websiteData;
    decoder >> websiteData;
    if (!websiteData) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*websiteData));
}

void FetchWebsiteData::cancelReply(CompletionHandler<void(WebKit::WebsiteData&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebKit::WebsiteData>::create());
}

void FetchWebsiteData::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::WebsiteData& websiteData)
{
    encoder.get() << websiteData;
    connection.sendSyncReply(WTFMove(encoder));
}

void DeleteWebsiteData::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteWebsiteData::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteWebsiteData::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void DeleteWebsiteDataForOrigins::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteWebsiteDataForOrigins::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteWebsiteDataForOrigins::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void RenameOriginInWebsiteData::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void RenameOriginInWebsiteData::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void RenameOriginInWebsiteData::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void CancelDownload::callReply(IPC::Decoder& decoder, CompletionHandler<void(IPC::DataReference&&)>&& completionHandler)
{
    std::optional<IPC::DataReference> resumeData;
    decoder >> resumeData;
    if (!resumeData) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*resumeData));
}

void CancelDownload::cancelReply(CompletionHandler<void(IPC::DataReference&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<IPC::DataReference>::create());
}

void CancelDownload::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const IPC::DataReference& resumeData)
{
    encoder.get() << resumeData;
    connection.sendSyncReply(WTFMove(encoder));
}

void RequestResource::callReply(IPC::Decoder& decoder, CompletionHandler<void(IPC::DataReference&&, WebCore::ResourceResponse&&, WebCore::ResourceError&&)>&& completionHandler)
{
    std::optional<IPC::DataReference> data;
    decoder >> data;
    if (!data) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<WebCore::ResourceResponse> response;
    decoder >> response;
    if (!response) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<WebCore::ResourceError> error;
    decoder >> error;
    if (!error) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*data), WTFMove(*response), WTFMove(*error));
}

void RequestResource::cancelReply(CompletionHandler<void(IPC::DataReference&&, WebCore::ResourceResponse&&, WebCore::ResourceError&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<IPC::DataReference>::create(), IPC::AsyncReplyError<WebCore::ResourceResponse>::create(), IPC::AsyncReplyError<WebCore::ResourceError>::create());
}

void RequestResource::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const IPC::DataReference& data, const WebCore::ResourceResponse& response, const WebCore::ResourceError& error)
{
    encoder.get() << data;
    encoder.get() << response;
    encoder.get() << error;
    connection.sendSyncReply(WTFMove(encoder));
}

void SetAllowsAnySSLCertificateForWebSocket::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void FlushCookies::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void FlushCookies::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void FlushCookies::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetCacheModelSynchronouslyForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void ProcessWillSuspendImminentlyForTestingSync::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void PrepareToSuspend::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void PrepareToSuspend::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void PrepareToSuspend::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void ClearPrevalentResource::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearPrevalentResource::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearPrevalentResource::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void ClearUserInteraction::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearUserInteraction::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearUserInteraction::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void DumpResourceLoadStatistics::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&)>&& completionHandler)
{
    std::optional<String> dumpedStatistics;
    decoder >> dumpedStatistics;
    if (!dumpedStatistics) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*dumpedStatistics));
}

void DumpResourceLoadStatistics::cancelReply(CompletionHandler<void(String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create());
}

void DumpResourceLoadStatistics::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& dumpedStatistics)
{
    encoder.get() << dumpedStatistics;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void UpdatePrevalentDomainsToBlockCookiesFor::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void UpdatePrevalentDomainsToBlockCookiesFor::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void UpdatePrevalentDomainsToBlockCookiesFor::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void IsGrandfathered::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> isGrandfathered;
    decoder >> isGrandfathered;
    if (!isGrandfathered) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*isGrandfathered));
}

void IsGrandfathered::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsGrandfathered::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isGrandfathered)
{
    encoder.get() << isGrandfathered;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void IsPrevalentResource::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> isPrevalent;
    decoder >> isPrevalent;
    if (!isPrevalent) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*isPrevalent));
}

void IsPrevalentResource::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsPrevalentResource::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isPrevalent)
{
    encoder.get() << isPrevalent;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void IsVeryPrevalentResource::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> isVeryPrevalent;
    decoder >> isVeryPrevalent;
    if (!isVeryPrevalent) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*isVeryPrevalent));
}

void IsVeryPrevalentResource::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsVeryPrevalentResource::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isVeryPrevalent)
{
    encoder.get() << isVeryPrevalent;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetAgeCapForClientSideCookies::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetAgeCapForClientSideCookies::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetAgeCapForClientSideCookies::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetLastSeen::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetLastSeen::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetLastSeen::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void MergeStatisticForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void MergeStatisticForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void MergeStatisticForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void InsertExpiredStatisticForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void InsertExpiredStatisticForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void InsertExpiredStatisticForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetPrevalentResource::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrevalentResource::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrevalentResource::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetPrevalentResourceForDebugMode::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrevalentResourceForDebugMode::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrevalentResourceForDebugMode::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void IsResourceLoadStatisticsEphemeral::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> isEphemeral;
    decoder >> isEphemeral;
    if (!isEphemeral) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*isEphemeral));
}

void IsResourceLoadStatisticsEphemeral::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsResourceLoadStatisticsEphemeral::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isEphemeral)
{
    encoder.get() << isEphemeral;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void HadUserInteraction::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> hadUserInteraction;
    decoder >> hadUserInteraction;
    if (!hadUserInteraction) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*hadUserInteraction));
}

void HadUserInteraction::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void HadUserInteraction::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool hadUserInteraction)
{
    encoder.get() << hadUserInteraction;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void IsRelationshipOnlyInDatabaseOnce::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> hadUserInteraction;
    decoder >> hadUserInteraction;
    if (!hadUserInteraction) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*hadUserInteraction));
}

void IsRelationshipOnlyInDatabaseOnce::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsRelationshipOnlyInDatabaseOnce::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool hadUserInteraction)
{
    encoder.get() << hadUserInteraction;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void HasLocalStorage::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> hadUserInteraction;
    decoder >> hadUserInteraction;
    if (!hadUserInteraction) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*hadUserInteraction));
}

void HasLocalStorage::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void HasLocalStorage::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool hadUserInteraction)
{
    encoder.get() << hadUserInteraction;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void GetAllStorageAccessEntries::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<String>&&)>&& completionHandler)
{
    std::optional<Vector<String>> domains;
    decoder >> domains;
    if (!domains) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*domains));
}

void GetAllStorageAccessEntries::cancelReply(CompletionHandler<void(Vector<String>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<String>>::create());
}

void GetAllStorageAccessEntries::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& domains)
{
    encoder.get() << domains;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void IsRegisteredAsRedirectingTo::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> isRedirectingTo;
    decoder >> isRedirectingTo;
    if (!isRedirectingTo) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*isRedirectingTo));
}

void IsRegisteredAsRedirectingTo::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsRegisteredAsRedirectingTo::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isRedirectingTo)
{
    encoder.get() << isRedirectingTo;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void IsRegisteredAsSubFrameUnder::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> isSubframeUnder;
    decoder >> isSubframeUnder;
    if (!isSubframeUnder) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*isSubframeUnder));
}

void IsRegisteredAsSubFrameUnder::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsRegisteredAsSubFrameUnder::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isSubframeUnder)
{
    encoder.get() << isSubframeUnder;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void IsRegisteredAsSubresourceUnder::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> isSubresourceUnder;
    decoder >> isSubresourceUnder;
    if (!isSubresourceUnder) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*isSubresourceUnder));
}

void IsRegisteredAsSubresourceUnder::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsRegisteredAsSubresourceUnder::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isSubresourceUnder)
{
    encoder.get() << isSubresourceUnder;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void DomainIDExistsInDatabase::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> exists;
    decoder >> exists;
    if (!exists) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*exists));
}

void DomainIDExistsInDatabase::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void DomainIDExistsInDatabase::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool exists)
{
    encoder.get() << exists;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void LogUserInteraction::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void LogUserInteraction::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void LogUserInteraction::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void ResetParametersToDefaultValues::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResetParametersToDefaultValues::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResetParametersToDefaultValues::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void ScheduleClearInMemoryAndPersistent::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ScheduleClearInMemoryAndPersistent::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ScheduleClearInMemoryAndPersistent::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void ScheduleCookieBlockingUpdate::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ScheduleCookieBlockingUpdate::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ScheduleCookieBlockingUpdate::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void ScheduleStatisticsAndDataRecordsProcessing::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ScheduleStatisticsAndDataRecordsProcessing::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ScheduleStatisticsAndDataRecordsProcessing::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void StatisticsDatabaseHasAllTables::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> hasAllTables;
    decoder >> hasAllTables;
    if (!hasAllTables) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*hasAllTables));
}

void StatisticsDatabaseHasAllTables::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void StatisticsDatabaseHasAllTables::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool hasAllTables)
{
    encoder.get() << hasAllTables;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetCacheMaxAgeCapForPrevalentResources::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetCacheMaxAgeCapForPrevalentResources::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetCacheMaxAgeCapForPrevalentResources::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetGrandfathered::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetGrandfathered::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetGrandfathered::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void GetResourceLoadStatisticsDataSummary::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<WebKit::WebResourceLoadStatisticsStore::ThirdPartyData>&&)>&& completionHandler)
{
    std::optional<Vector<WebKit::WebResourceLoadStatisticsStore::ThirdPartyData>> thirdPartyData;
    decoder >> thirdPartyData;
    if (!thirdPartyData) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*thirdPartyData));
}

void GetResourceLoadStatisticsDataSummary::cancelReply(CompletionHandler<void(Vector<WebKit::WebResourceLoadStatisticsStore::ThirdPartyData>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<WebKit::WebResourceLoadStatisticsStore::ThirdPartyData>>::create());
}

void GetResourceLoadStatisticsDataSummary::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebKit::WebResourceLoadStatisticsStore::ThirdPartyData>& thirdPartyData)
{
    encoder.get() << thirdPartyData;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetGrandfatheringTime::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetGrandfatheringTime::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetGrandfatheringTime::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetMaxStatisticsEntries::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetMaxStatisticsEntries::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetMaxStatisticsEntries::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetMinimumTimeBetweenDataRecordsRemoval::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetMinimumTimeBetweenDataRecordsRemoval::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetMinimumTimeBetweenDataRecordsRemoval::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetPruneEntriesDownTo::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPruneEntriesDownTo::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPruneEntriesDownTo::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetShouldClassifyResourcesBeforeDataRecordsRemoval::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetShouldClassifyResourcesBeforeDataRecordsRemoval::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetShouldClassifyResourcesBeforeDataRecordsRemoval::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetNotifyPagesWhenDataRecordsWereScanned::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetNotifyPagesWhenDataRecordsWereScanned::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetNotifyPagesWhenDataRecordsWereScanned::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetIsRunningResourceLoadStatisticsTest::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetIsRunningResourceLoadStatisticsTest::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetIsRunningResourceLoadStatisticsTest::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetResourceLoadStatisticsDebugMode::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetResourceLoadStatisticsDebugMode::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetResourceLoadStatisticsDebugMode::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetVeryPrevalentResource::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetVeryPrevalentResource::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetVeryPrevalentResource::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetSubframeUnderTopFrameDomain::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetSubframeUnderTopFrameDomain::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetSubframeUnderTopFrameDomain::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetSubresourceUnderTopFrameDomain::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetSubresourceUnderTopFrameDomain::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetSubresourceUnderTopFrameDomain::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetSubresourceUniqueRedirectTo::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetSubresourceUniqueRedirectTo::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetSubresourceUniqueRedirectTo::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetSubresourceUniqueRedirectFrom::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetSubresourceUniqueRedirectFrom::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetSubresourceUniqueRedirectFrom::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetTimeToLiveUserInteraction::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetTimeToLiveUserInteraction::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetTimeToLiveUserInteraction::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetTopFrameUniqueRedirectTo::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetTopFrameUniqueRedirectTo::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetTopFrameUniqueRedirectTo::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetTopFrameUniqueRedirectFrom::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetTopFrameUniqueRedirectFrom::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetTopFrameUniqueRedirectFrom::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void ResetCacheMaxAgeCapForPrevalentResources::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResetCacheMaxAgeCapForPrevalentResources::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResetCacheMaxAgeCapForPrevalentResources::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetCrossSiteLoadWithLinkDecorationForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetCrossSiteLoadWithLinkDecorationForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetCrossSiteLoadWithLinkDecorationForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void ResetCrossSiteLoadsWithLinkDecorationForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResetCrossSiteLoadsWithLinkDecorationForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResetCrossSiteLoadsWithLinkDecorationForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void DeleteCookiesForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteCookiesForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteCookiesForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void HasIsolatedSession::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> hasIsolatedSession;
    decoder >> hasIsolatedSession;
    if (!hasIsolatedSession) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*hasIsolatedSession));
}

void HasIsolatedSession::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void HasIsolatedSession::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool hasIsolatedSession)
{
    encoder.get() << hasIsolatedSession;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void CloseITPDatabase::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void CloseITPDatabase::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void CloseITPDatabase::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if (ENABLE(INTELLIGENT_TRACKING_PREVENTION) && ENABLE(APP_BOUND_DOMAINS))

void SetAppBoundDomainsForResourceLoadStatistics::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetAppBoundDomainsForResourceLoadStatistics::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetAppBoundDomainsForResourceLoadStatistics::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetShouldDowngradeReferrerForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetShouldDowngradeReferrerForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetShouldDowngradeReferrerForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetThirdPartyCookieBlockingMode::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetThirdPartyCookieBlockingMode::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetThirdPartyCookieBlockingMode::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetShouldEnbleSameSiteStrictEnforcementForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetShouldEnbleSameSiteStrictEnforcementForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetShouldEnbleSameSiteStrictEnforcementForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetFirstPartyWebsiteDataRemovalModeForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetFirstPartyWebsiteDataRemovalModeForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetFirstPartyWebsiteDataRemovalModeForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetToSameSiteStrictCookiesForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetToSameSiteStrictCookiesForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetToSameSiteStrictCookiesForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetFirstPartyHostCNAMEDomainForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetFirstPartyHostCNAMEDomainForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetFirstPartyHostCNAMEDomainForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetThirdPartyCNAMEDomainForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetThirdPartyCNAMEDomainForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetThirdPartyCNAMEDomainForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void SyncLocalStorage::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SyncLocalStorage::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SyncLocalStorage::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void DumpPrivateClickMeasurement::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&)>&& completionHandler)
{
    std::optional<String> privateClickMeasurementState;
    decoder >> privateClickMeasurementState;
    if (!privateClickMeasurementState) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*privateClickMeasurementState));
}

void DumpPrivateClickMeasurement::cancelReply(CompletionHandler<void(String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create());
}

void DumpPrivateClickMeasurement::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& privateClickMeasurementState)
{
    encoder.get() << privateClickMeasurementState;
    connection.sendSyncReply(WTFMove(encoder));
}

void ClearPrivateClickMeasurement::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearPrivateClickMeasurement::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearPrivateClickMeasurement::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetPrivateClickMeasurementOverrideTimerForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementOverrideTimerForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementOverrideTimerForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void MarkAttributedPrivateClickMeasurementsAsExpiredForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void MarkAttributedPrivateClickMeasurementsAsExpiredForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void MarkAttributedPrivateClickMeasurementsAsExpiredForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetPrivateClickMeasurementEphemeralMeasurementForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementEphemeralMeasurementForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementEphemeralMeasurementForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SimulatePrivateClickMeasurementSessionRestart::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SimulatePrivateClickMeasurementSessionRestart::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SimulatePrivateClickMeasurementSessionRestart::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetPrivateClickMeasurementTokenPublicKeyURLForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementTokenPublicKeyURLForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementTokenPublicKeyURLForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetPrivateClickMeasurementTokenSignatureURLForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementTokenSignatureURLForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementTokenSignatureURLForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetPrivateClickMeasurementAttributionReportURLsForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementAttributionReportURLsForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementAttributionReportURLsForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void MarkPrivateClickMeasurementsAsExpiredForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void MarkPrivateClickMeasurementsAsExpiredForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void MarkPrivateClickMeasurementsAsExpiredForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetPCMFraudPreventionValuesForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPCMFraudPreventionValuesForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPCMFraudPreventionValuesForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetPrivateClickMeasurementAppBundleIDForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementAppBundleIDForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetPrivateClickMeasurementAppBundleIDForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void ClosePCMDatabase::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClosePCMDatabase::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClosePCMDatabase::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void SetServiceWorkerFetchTimeoutForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void ResetServiceWorkerFetchTimeoutForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void ResetQuota::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResetQuota::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResetQuota::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void ClearStorage::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearStorage::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearStorage::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(APP_BOUND_DOMAINS)

void HasAppBoundSession::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> hasAppBoundSession;
    decoder >> hasAppBoundSession;
    if (!hasAppBoundSession) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*hasAppBoundSession));
}

void HasAppBoundSession::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void HasAppBoundSession::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool hasAppBoundSession)
{
    encoder.get() << hasAppBoundSession;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(APP_BOUND_DOMAINS)

void ClearAppBoundSession::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearAppBoundSession::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearAppBoundSession::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(IOS_FAMILY) && !PLATFORM(MACCATALYST)

void ClearServiceWorkerEntitlementOverride::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearServiceWorkerEntitlementOverride::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearServiceWorkerEntitlementOverride::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void UpdateBundleIdentifier::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void UpdateBundleIdentifier::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void UpdateBundleIdentifier::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void ClearBundleIdentifier::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearBundleIdentifier::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearBundleIdentifier::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#if PLATFORM(COCOA)

void AppPrivacyReportTestingData::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebKit::AppPrivacyReportTestingData&&)>&& completionHandler)
{
    std::optional<WebKit::AppPrivacyReportTestingData> data;
    decoder >> data;
    if (!data) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*data));
}

void AppPrivacyReportTestingData::cancelReply(CompletionHandler<void(WebKit::AppPrivacyReportTestingData&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebKit::AppPrivacyReportTestingData>::create());
}

void AppPrivacyReportTestingData::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::AppPrivacyReportTestingData& data)
{
    encoder.get() << data;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void ClearAppPrivacyReportTestingData::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearAppPrivacyReportTestingData::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearAppPrivacyReportTestingData::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void CountNonDefaultSessionSets::callReply(IPC::Decoder& decoder, CompletionHandler<void(size_t&&)>&& completionHandler)
{
    std::optional<size_t> count;
    decoder >> count;
    if (!count) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*count));
}

void CountNonDefaultSessionSets::cancelReply(CompletionHandler<void(size_t&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<size_t>::create());
}

void CountNonDefaultSessionSets::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, size_t count)
{
    encoder.get() << count;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(SERVICE_WORKER)

void GetPendingPushMessages::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<WebKit::WebPushMessage>&&)>&& completionHandler)
{
    std::optional<Vector<WebKit::WebPushMessage>> messages;
    decoder >> messages;
    if (!messages) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*messages));
}

void GetPendingPushMessages::cancelReply(CompletionHandler<void(Vector<WebKit::WebPushMessage>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<WebKit::WebPushMessage>>::create());
}

void GetPendingPushMessages::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebKit::WebPushMessage>& messages)
{
    encoder.get() << messages;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(SERVICE_WORKER)

void ProcessPushMessage::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> didSucceed;
    decoder >> didSucceed;
    if (!didSucceed) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*didSucceed));
}

void ProcessPushMessage::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void ProcessPushMessage::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool didSucceed)
{
    encoder.get() << didSucceed;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void DeletePushAndNotificationRegistration::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&)>&& completionHandler)
{
    std::optional<String> errorMessage;
    decoder >> errorMessage;
    if (!errorMessage) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*errorMessage));
}

void DeletePushAndNotificationRegistration::cancelReply(CompletionHandler<void(String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create());
}

void DeletePushAndNotificationRegistration::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& errorMessage)
{
    encoder.get() << errorMessage;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetOriginsWithPushAndNotificationPermissions::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<WebCore::SecurityOriginData>&&)>&& completionHandler)
{
    std::optional<Vector<WebCore::SecurityOriginData>> origins;
    decoder >> origins;
    if (!origins) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*origins));
}

void GetOriginsWithPushAndNotificationPermissions::cancelReply(CompletionHandler<void(Vector<WebCore::SecurityOriginData>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<WebCore::SecurityOriginData>>::create());
}

void GetOriginsWithPushAndNotificationPermissions::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::SecurityOriginData>& origins)
{
    encoder.get() << origins;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace NetworkProcess

} // namespace Messages

namespace WebKit {

void NetworkProcess::didReceiveNetworkProcessMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::NetworkProcess::InitializeNetworkProcess::name())
        return IPC::handleMessage<Messages::NetworkProcess::InitializeNetworkProcess>(connection, decoder, this, &NetworkProcess::initializeNetworkProcess);
    if (decoder.messageName() == Messages::NetworkProcess::CreateNetworkConnectionToWebProcess::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::CreateNetworkConnectionToWebProcess>(connection, decoder, this, &NetworkProcess::createNetworkConnectionToWebProcess);
#if USE(SOUP)
    if (decoder.messageName() == Messages::NetworkProcess::SetIgnoreTLSErrors::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetIgnoreTLSErrors>(connection, decoder, this, &NetworkProcess::setIgnoreTLSErrors);
#endif
#if USE(SOUP)
    if (decoder.messageName() == Messages::NetworkProcess::UserPreferredLanguagesChanged::name())
        return IPC::handleMessage<Messages::NetworkProcess::UserPreferredLanguagesChanged>(connection, decoder, this, &NetworkProcess::userPreferredLanguagesChanged);
#endif
#if USE(SOUP)
    if (decoder.messageName() == Messages::NetworkProcess::SetNetworkProxySettings::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetNetworkProxySettings>(connection, decoder, this, &NetworkProcess::setNetworkProxySettings);
#endif
#if USE(SOUP)
    if (decoder.messageName() == Messages::NetworkProcess::PrefetchDNS::name())
        return IPC::handleMessage<Messages::NetworkProcess::PrefetchDNS>(connection, decoder, this, &NetworkProcess::prefetchDNS);
#endif
#if USE(SOUP)
    if (decoder.messageName() == Messages::NetworkProcess::SetPersistentCredentialStorageEnabled::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetPersistentCredentialStorageEnabled>(connection, decoder, this, &NetworkProcess::setPersistentCredentialStorageEnabled);
#endif
#if USE(CURL)
    if (decoder.messageName() == Messages::NetworkProcess::SetNetworkProxySettings::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetNetworkProxySettings>(connection, decoder, this, &NetworkProcess::setNetworkProxySettings);
#endif
    if (decoder.messageName() == Messages::NetworkProcess::ClearCachedCredentials::name())
        return IPC::handleMessage<Messages::NetworkProcess::ClearCachedCredentials>(connection, decoder, this, &NetworkProcess::clearCachedCredentials);
    if (decoder.messageName() == Messages::NetworkProcess::AddWebsiteDataStore::name())
        return IPC::handleMessage<Messages::NetworkProcess::AddWebsiteDataStore>(connection, decoder, this, &NetworkProcess::addWebsiteDataStore);
    if (decoder.messageName() == Messages::NetworkProcess::DestroySession::name())
        return IPC::handleMessage<Messages::NetworkProcess::DestroySession>(connection, decoder, this, &NetworkProcess::destroySession);
    if (decoder.messageName() == Messages::NetworkProcess::FetchWebsiteData::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::FetchWebsiteData>(connection, decoder, this, &NetworkProcess::fetchWebsiteData);
    if (decoder.messageName() == Messages::NetworkProcess::DeleteWebsiteData::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::DeleteWebsiteData>(connection, decoder, this, &NetworkProcess::deleteWebsiteData);
    if (decoder.messageName() == Messages::NetworkProcess::DeleteWebsiteDataForOrigins::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::DeleteWebsiteDataForOrigins>(connection, decoder, this, &NetworkProcess::deleteWebsiteDataForOrigins);
    if (decoder.messageName() == Messages::NetworkProcess::RenameOriginInWebsiteData::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::RenameOriginInWebsiteData>(connection, decoder, this, &NetworkProcess::renameOriginInWebsiteData);
    if (decoder.messageName() == Messages::NetworkProcess::DownloadRequest::name())
        return IPC::handleMessage<Messages::NetworkProcess::DownloadRequest>(connection, decoder, this, &NetworkProcess::downloadRequest);
    if (decoder.messageName() == Messages::NetworkProcess::ResumeDownload::name())
        return IPC::handleMessage<Messages::NetworkProcess::ResumeDownload>(connection, decoder, this, &NetworkProcess::resumeDownload);
    if (decoder.messageName() == Messages::NetworkProcess::CancelDownload::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::CancelDownload>(connection, decoder, this, &NetworkProcess::cancelDownload);
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::NetworkProcess::PublishDownloadProgress::name())
        return IPC::handleMessage<Messages::NetworkProcess::PublishDownloadProgress>(connection, decoder, this, &NetworkProcess::publishDownloadProgress);
#endif
    if (decoder.messageName() == Messages::NetworkProcess::RequestResource::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::RequestResource>(connection, decoder, this, &NetworkProcess::requestResource);
    if (decoder.messageName() == Messages::NetworkProcess::ApplicationDidEnterBackground::name())
        return IPC::handleMessage<Messages::NetworkProcess::ApplicationDidEnterBackground>(connection, decoder, this, &NetworkProcess::applicationDidEnterBackground);
    if (decoder.messageName() == Messages::NetworkProcess::ApplicationWillEnterForeground::name())
        return IPC::handleMessage<Messages::NetworkProcess::ApplicationWillEnterForeground>(connection, decoder, this, &NetworkProcess::applicationWillEnterForeground);
    if (decoder.messageName() == Messages::NetworkProcess::ContinueWillSendRequest::name())
        return IPC::handleMessage<Messages::NetworkProcess::ContinueWillSendRequest>(connection, decoder, this, &NetworkProcess::continueWillSendRequest);
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::NetworkProcess::SetQOS::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetQOS>(connection, decoder, this, &NetworkProcess::setQOS);
#endif
    if (decoder.messageName() == Messages::NetworkProcess::FlushCookies::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::FlushCookies>(connection, decoder, this, &NetworkProcess::flushCookies);
    if (decoder.messageName() == Messages::NetworkProcess::AllowSpecificHTTPSCertificateForHost::name())
        return IPC::handleMessage<Messages::NetworkProcess::AllowSpecificHTTPSCertificateForHost>(connection, decoder, this, &NetworkProcess::allowSpecificHTTPSCertificateForHost);
    if (decoder.messageName() == Messages::NetworkProcess::AllowTLSCertificateChainForLocalPCMTesting::name())
        return IPC::handleMessage<Messages::NetworkProcess::AllowTLSCertificateChainForLocalPCMTesting>(connection, decoder, this, &NetworkProcess::allowTLSCertificateChainForLocalPCMTesting);
    if (decoder.messageName() == Messages::NetworkProcess::SetCacheModel::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetCacheModel>(connection, decoder, this, &NetworkProcess::setCacheModel);
    if (decoder.messageName() == Messages::NetworkProcess::PrepareToSuspend::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::PrepareToSuspend>(connection, decoder, this, &NetworkProcess::prepareToSuspend);
    if (decoder.messageName() == Messages::NetworkProcess::ProcessDidResume::name())
        return IPC::handleMessage<Messages::NetworkProcess::ProcessDidResume>(connection, decoder, this, &NetworkProcess::processDidResume);
    if (decoder.messageName() == Messages::NetworkProcess::PreconnectTo::name())
        return IPC::handleMessage<Messages::NetworkProcess::PreconnectTo>(connection, decoder, this, &NetworkProcess::preconnectTo);
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::ClearPrevalentResource::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ClearPrevalentResource>(connection, decoder, this, &NetworkProcess::clearPrevalentResource);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::ClearUserInteraction::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ClearUserInteraction>(connection, decoder, this, &NetworkProcess::clearUserInteraction);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::DumpResourceLoadStatistics::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::DumpResourceLoadStatistics>(connection, decoder, this, &NetworkProcess::dumpResourceLoadStatistics);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetResourceLoadStatisticsEnabled::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetResourceLoadStatisticsEnabled>(connection, decoder, this, &NetworkProcess::setResourceLoadStatisticsEnabled);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetResourceLoadStatisticsLogTestingEvent::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetResourceLoadStatisticsLogTestingEvent>(connection, decoder, this, &NetworkProcess::setResourceLoadStatisticsLogTestingEvent);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::UpdatePrevalentDomainsToBlockCookiesFor::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::UpdatePrevalentDomainsToBlockCookiesFor>(connection, decoder, this, &NetworkProcess::updatePrevalentDomainsToBlockCookiesFor);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::IsGrandfathered::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::IsGrandfathered>(connection, decoder, this, &NetworkProcess::isGrandfathered);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::IsPrevalentResource::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::IsPrevalentResource>(connection, decoder, this, &NetworkProcess::isPrevalentResource);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::IsVeryPrevalentResource::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::IsVeryPrevalentResource>(connection, decoder, this, &NetworkProcess::isVeryPrevalentResource);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetAgeCapForClientSideCookies::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetAgeCapForClientSideCookies>(connection, decoder, this, &NetworkProcess::setAgeCapForClientSideCookies);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetLastSeen::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetLastSeen>(connection, decoder, this, &NetworkProcess::setLastSeen);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::MergeStatisticForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::MergeStatisticForTesting>(connection, decoder, this, &NetworkProcess::mergeStatisticForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::InsertExpiredStatisticForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::InsertExpiredStatisticForTesting>(connection, decoder, this, &NetworkProcess::insertExpiredStatisticForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetPrevalentResource::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPrevalentResource>(connection, decoder, this, &NetworkProcess::setPrevalentResource);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetPrevalentResourceForDebugMode::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPrevalentResourceForDebugMode>(connection, decoder, this, &NetworkProcess::setPrevalentResourceForDebugMode);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::IsResourceLoadStatisticsEphemeral::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::IsResourceLoadStatisticsEphemeral>(connection, decoder, this, &NetworkProcess::isResourceLoadStatisticsEphemeral);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::HadUserInteraction::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::HadUserInteraction>(connection, decoder, this, &NetworkProcess::hadUserInteraction);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::IsRelationshipOnlyInDatabaseOnce::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::IsRelationshipOnlyInDatabaseOnce>(connection, decoder, this, &NetworkProcess::isRelationshipOnlyInDatabaseOnce);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::HasLocalStorage::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::HasLocalStorage>(connection, decoder, this, &NetworkProcess::hasLocalStorage);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::GetAllStorageAccessEntries::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::GetAllStorageAccessEntries>(connection, decoder, this, &NetworkProcess::getAllStorageAccessEntries);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::IsRegisteredAsRedirectingTo::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::IsRegisteredAsRedirectingTo>(connection, decoder, this, &NetworkProcess::isRegisteredAsRedirectingTo);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::IsRegisteredAsSubFrameUnder::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::IsRegisteredAsSubFrameUnder>(connection, decoder, this, &NetworkProcess::isRegisteredAsSubFrameUnder);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::IsRegisteredAsSubresourceUnder::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::IsRegisteredAsSubresourceUnder>(connection, decoder, this, &NetworkProcess::isRegisteredAsSubresourceUnder);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::DomainIDExistsInDatabase::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::DomainIDExistsInDatabase>(connection, decoder, this, &NetworkProcess::domainIDExistsInDatabase);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::LogFrameNavigation::name())
        return IPC::handleMessage<Messages::NetworkProcess::LogFrameNavigation>(connection, decoder, this, &NetworkProcess::logFrameNavigation);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::LogUserInteraction::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::LogUserInteraction>(connection, decoder, this, &NetworkProcess::logUserInteraction);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::ResetParametersToDefaultValues::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ResetParametersToDefaultValues>(connection, decoder, this, &NetworkProcess::resetParametersToDefaultValues);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::ScheduleClearInMemoryAndPersistent::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ScheduleClearInMemoryAndPersistent>(connection, decoder, this, &NetworkProcess::scheduleClearInMemoryAndPersistent);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::ScheduleCookieBlockingUpdate::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ScheduleCookieBlockingUpdate>(connection, decoder, this, &NetworkProcess::scheduleCookieBlockingUpdate);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::ScheduleStatisticsAndDataRecordsProcessing::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ScheduleStatisticsAndDataRecordsProcessing>(connection, decoder, this, &NetworkProcess::scheduleStatisticsAndDataRecordsProcessing);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::StatisticsDatabaseHasAllTables::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::StatisticsDatabaseHasAllTables>(connection, decoder, this, &NetworkProcess::statisticsDatabaseHasAllTables);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetCacheMaxAgeCapForPrevalentResources::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetCacheMaxAgeCapForPrevalentResources>(connection, decoder, this, &NetworkProcess::setCacheMaxAgeCapForPrevalentResources);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetGrandfathered::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetGrandfathered>(connection, decoder, this, &NetworkProcess::setGrandfathered);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::GetResourceLoadStatisticsDataSummary::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::GetResourceLoadStatisticsDataSummary>(connection, decoder, this, &NetworkProcess::getResourceLoadStatisticsDataSummary);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetGrandfatheringTime::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetGrandfatheringTime>(connection, decoder, this, &NetworkProcess::setGrandfatheringTime);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetMaxStatisticsEntries::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetMaxStatisticsEntries>(connection, decoder, this, &NetworkProcess::setMaxStatisticsEntries);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetMinimumTimeBetweenDataRecordsRemoval::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetMinimumTimeBetweenDataRecordsRemoval>(connection, decoder, this, &NetworkProcess::setMinimumTimeBetweenDataRecordsRemoval);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetPruneEntriesDownTo::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPruneEntriesDownTo>(connection, decoder, this, &NetworkProcess::setPruneEntriesDownTo);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetShouldClassifyResourcesBeforeDataRecordsRemoval::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetShouldClassifyResourcesBeforeDataRecordsRemoval>(connection, decoder, this, &NetworkProcess::setShouldClassifyResourcesBeforeDataRecordsRemoval);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetNotifyPagesWhenDataRecordsWereScanned::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetNotifyPagesWhenDataRecordsWereScanned>(connection, decoder, this, &NetworkProcess::setNotifyPagesWhenDataRecordsWereScanned);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetIsRunningResourceLoadStatisticsTest::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetIsRunningResourceLoadStatisticsTest>(connection, decoder, this, &NetworkProcess::setIsRunningResourceLoadStatisticsTest);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetResourceLoadStatisticsDebugMode::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetResourceLoadStatisticsDebugMode>(connection, decoder, this, &NetworkProcess::setResourceLoadStatisticsDebugMode);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetVeryPrevalentResource::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetVeryPrevalentResource>(connection, decoder, this, &NetworkProcess::setVeryPrevalentResource);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetSubframeUnderTopFrameDomain::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetSubframeUnderTopFrameDomain>(connection, decoder, this, &NetworkProcess::setSubframeUnderTopFrameDomain);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetSubresourceUnderTopFrameDomain::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetSubresourceUnderTopFrameDomain>(connection, decoder, this, &NetworkProcess::setSubresourceUnderTopFrameDomain);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetSubresourceUniqueRedirectTo::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetSubresourceUniqueRedirectTo>(connection, decoder, this, &NetworkProcess::setSubresourceUniqueRedirectTo);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetSubresourceUniqueRedirectFrom::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetSubresourceUniqueRedirectFrom>(connection, decoder, this, &NetworkProcess::setSubresourceUniqueRedirectFrom);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetTimeToLiveUserInteraction::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetTimeToLiveUserInteraction>(connection, decoder, this, &NetworkProcess::setTimeToLiveUserInteraction);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetTopFrameUniqueRedirectTo::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetTopFrameUniqueRedirectTo>(connection, decoder, this, &NetworkProcess::setTopFrameUniqueRedirectTo);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetTopFrameUniqueRedirectFrom::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetTopFrameUniqueRedirectFrom>(connection, decoder, this, &NetworkProcess::setTopFrameUniqueRedirectFrom);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::ResetCacheMaxAgeCapForPrevalentResources::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ResetCacheMaxAgeCapForPrevalentResources>(connection, decoder, this, &NetworkProcess::resetCacheMaxAgeCapForPrevalentResources);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::DidCommitCrossSiteLoadWithDataTransfer::name())
        return IPC::handleMessage<Messages::NetworkProcess::DidCommitCrossSiteLoadWithDataTransfer>(connection, decoder, this, &NetworkProcess::didCommitCrossSiteLoadWithDataTransfer);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetCrossSiteLoadWithLinkDecorationForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetCrossSiteLoadWithLinkDecorationForTesting>(connection, decoder, this, &NetworkProcess::setCrossSiteLoadWithLinkDecorationForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::ResetCrossSiteLoadsWithLinkDecorationForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ResetCrossSiteLoadsWithLinkDecorationForTesting>(connection, decoder, this, &NetworkProcess::resetCrossSiteLoadsWithLinkDecorationForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::DeleteCookiesForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::DeleteCookiesForTesting>(connection, decoder, this, &NetworkProcess::deleteCookiesForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::HasIsolatedSession::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::HasIsolatedSession>(connection, decoder, this, &NetworkProcess::hasIsolatedSession);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::CloseITPDatabase::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::CloseITPDatabase>(connection, decoder, this, &NetworkProcess::closeITPDatabase);
#endif
#if (ENABLE(INTELLIGENT_TRACKING_PREVENTION) && ENABLE(APP_BOUND_DOMAINS))
    if (decoder.messageName() == Messages::NetworkProcess::SetAppBoundDomainsForResourceLoadStatistics::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetAppBoundDomainsForResourceLoadStatistics>(connection, decoder, this, &NetworkProcess::setAppBoundDomainsForResourceLoadStatistics);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetShouldDowngradeReferrerForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetShouldDowngradeReferrerForTesting>(connection, decoder, this, &NetworkProcess::setShouldDowngradeReferrerForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetThirdPartyCookieBlockingMode::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetThirdPartyCookieBlockingMode>(connection, decoder, this, &NetworkProcess::setThirdPartyCookieBlockingMode);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetShouldEnbleSameSiteStrictEnforcementForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetShouldEnbleSameSiteStrictEnforcementForTesting>(connection, decoder, this, &NetworkProcess::setShouldEnbleSameSiteStrictEnforcementForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetFirstPartyWebsiteDataRemovalModeForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetFirstPartyWebsiteDataRemovalModeForTesting>(connection, decoder, this, &NetworkProcess::setFirstPartyWebsiteDataRemovalModeForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetToSameSiteStrictCookiesForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetToSameSiteStrictCookiesForTesting>(connection, decoder, this, &NetworkProcess::setToSameSiteStrictCookiesForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetFirstPartyHostCNAMEDomainForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetFirstPartyHostCNAMEDomainForTesting>(connection, decoder, this, &NetworkProcess::setFirstPartyHostCNAMEDomainForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcess::SetThirdPartyCNAMEDomainForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetThirdPartyCNAMEDomainForTesting>(connection, decoder, this, &NetworkProcess::setThirdPartyCNAMEDomainForTesting);
#endif
    if (decoder.messageName() == Messages::NetworkProcess::SetPrivateClickMeasurementDebugMode::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetPrivateClickMeasurementDebugMode>(connection, decoder, this, &NetworkProcess::setPrivateClickMeasurementDebugMode);
    if (decoder.messageName() == Messages::NetworkProcess::SetSessionIsControlledByAutomation::name())
        return IPC::handleMessage<Messages::NetworkProcess::SetSessionIsControlledByAutomation>(connection, decoder, this, &NetworkProcess::setSessionIsControlledByAutomation);
    if (decoder.messageName() == Messages::NetworkProcess::RegisterURLSchemeAsSecure::name())
        return IPC::handleMessage<Messages::NetworkProcess::RegisterURLSchemeAsSecure>(connection, decoder, this, &NetworkProcess::registerURLSchemeAsSecure);
    if (decoder.messageName() == Messages::NetworkProcess::RegisterURLSchemeAsBypassingContentSecurityPolicy::name())
        return IPC::handleMessage<Messages::NetworkProcess::RegisterURLSchemeAsBypassingContentSecurityPolicy>(connection, decoder, this, &NetworkProcess::registerURLSchemeAsBypassingContentSecurityPolicy);
    if (decoder.messageName() == Messages::NetworkProcess::RegisterURLSchemeAsLocal::name())
        return IPC::handleMessage<Messages::NetworkProcess::RegisterURLSchemeAsLocal>(connection, decoder, this, &NetworkProcess::registerURLSchemeAsLocal);
    if (decoder.messageName() == Messages::NetworkProcess::RegisterURLSchemeAsNoAccess::name())
        return IPC::handleMessage<Messages::NetworkProcess::RegisterURLSchemeAsNoAccess>(connection, decoder, this, &NetworkProcess::registerURLSchemeAsNoAccess);
    if (decoder.messageName() == Messages::NetworkProcess::SyncLocalStorage::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SyncLocalStorage>(connection, decoder, this, &NetworkProcess::syncLocalStorage);
    if (decoder.messageName() == Messages::NetworkProcess::StorePrivateClickMeasurement::name())
        return IPC::handleMessage<Messages::NetworkProcess::StorePrivateClickMeasurement>(connection, decoder, this, &NetworkProcess::storePrivateClickMeasurement);
    if (decoder.messageName() == Messages::NetworkProcess::DumpPrivateClickMeasurement::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::DumpPrivateClickMeasurement>(connection, decoder, this, &NetworkProcess::dumpPrivateClickMeasurement);
    if (decoder.messageName() == Messages::NetworkProcess::ClearPrivateClickMeasurement::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ClearPrivateClickMeasurement>(connection, decoder, this, &NetworkProcess::clearPrivateClickMeasurement);
    if (decoder.messageName() == Messages::NetworkProcess::SetPrivateClickMeasurementOverrideTimerForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPrivateClickMeasurementOverrideTimerForTesting>(connection, decoder, this, &NetworkProcess::setPrivateClickMeasurementOverrideTimerForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::MarkAttributedPrivateClickMeasurementsAsExpiredForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::MarkAttributedPrivateClickMeasurementsAsExpiredForTesting>(connection, decoder, this, &NetworkProcess::markAttributedPrivateClickMeasurementsAsExpiredForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::SetPrivateClickMeasurementEphemeralMeasurementForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPrivateClickMeasurementEphemeralMeasurementForTesting>(connection, decoder, this, &NetworkProcess::setPrivateClickMeasurementEphemeralMeasurementForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::SimulatePrivateClickMeasurementSessionRestart::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SimulatePrivateClickMeasurementSessionRestart>(connection, decoder, this, &NetworkProcess::simulatePrivateClickMeasurementSessionRestart);
    if (decoder.messageName() == Messages::NetworkProcess::SetPrivateClickMeasurementTokenPublicKeyURLForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPrivateClickMeasurementTokenPublicKeyURLForTesting>(connection, decoder, this, &NetworkProcess::setPrivateClickMeasurementTokenPublicKeyURLForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::SetPrivateClickMeasurementTokenSignatureURLForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPrivateClickMeasurementTokenSignatureURLForTesting>(connection, decoder, this, &NetworkProcess::setPrivateClickMeasurementTokenSignatureURLForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::SetPrivateClickMeasurementAttributionReportURLsForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPrivateClickMeasurementAttributionReportURLsForTesting>(connection, decoder, this, &NetworkProcess::setPrivateClickMeasurementAttributionReportURLsForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::MarkPrivateClickMeasurementsAsExpiredForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::MarkPrivateClickMeasurementsAsExpiredForTesting>(connection, decoder, this, &NetworkProcess::markPrivateClickMeasurementsAsExpiredForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::SetPCMFraudPreventionValuesForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPCMFraudPreventionValuesForTesting>(connection, decoder, this, &NetworkProcess::setPCMFraudPreventionValuesForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::SetPrivateClickMeasurementAppBundleIDForTesting::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::SetPrivateClickMeasurementAppBundleIDForTesting>(connection, decoder, this, &NetworkProcess::setPrivateClickMeasurementAppBundleIDForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::ClosePCMDatabase::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ClosePCMDatabase>(connection, decoder, this, &NetworkProcess::closePCMDatabase);
    if (decoder.messageName() == Messages::NetworkProcess::ResetQuota::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ResetQuota>(connection, decoder, this, &NetworkProcess::resetQuota);
    if (decoder.messageName() == Messages::NetworkProcess::ClearStorage::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ClearStorage>(connection, decoder, this, &NetworkProcess::clearStorage);
    if (decoder.messageName() == Messages::NetworkProcess::DidIncreaseQuota::name())
        return IPC::handleMessage<Messages::NetworkProcess::DidIncreaseQuota>(connection, decoder, this, &NetworkProcess::didIncreaseQuota);
#if ENABLE(APP_BOUND_DOMAINS)
    if (decoder.messageName() == Messages::NetworkProcess::HasAppBoundSession::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::HasAppBoundSession>(connection, decoder, this, &NetworkProcess::hasAppBoundSession);
#endif
#if ENABLE(APP_BOUND_DOMAINS)
    if (decoder.messageName() == Messages::NetworkProcess::ClearAppBoundSession::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ClearAppBoundSession>(connection, decoder, this, &NetworkProcess::clearAppBoundSession);
#endif
#if PLATFORM(IOS_FAMILY) && !PLATFORM(MACCATALYST)
    if (decoder.messageName() == Messages::NetworkProcess::DisableServiceWorkerEntitlement::name())
        return IPC::handleMessage<Messages::NetworkProcess::DisableServiceWorkerEntitlement>(connection, decoder, this, &NetworkProcess::disableServiceWorkerEntitlement);
#endif
#if PLATFORM(IOS_FAMILY) && !PLATFORM(MACCATALYST)
    if (decoder.messageName() == Messages::NetworkProcess::ClearServiceWorkerEntitlementOverride::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ClearServiceWorkerEntitlementOverride>(connection, decoder, this, &NetworkProcess::clearServiceWorkerEntitlementOverride);
#endif
    if (decoder.messageName() == Messages::NetworkProcess::UpdateBundleIdentifier::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::UpdateBundleIdentifier>(connection, decoder, this, &NetworkProcess::updateBundleIdentifier);
    if (decoder.messageName() == Messages::NetworkProcess::ClearBundleIdentifier::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ClearBundleIdentifier>(connection, decoder, this, &NetworkProcess::clearBundleIdentifier);
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::NetworkProcess::AppPrivacyReportTestingData::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::AppPrivacyReportTestingData>(connection, decoder, this, &NetworkProcess::appPrivacyReportTestingData);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::NetworkProcess::ClearAppPrivacyReportTestingData::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ClearAppPrivacyReportTestingData>(connection, decoder, this, &NetworkProcess::clearAppPrivacyReportTestingData);
#endif
    if (decoder.messageName() == Messages::NetworkProcess::AddWebPageNetworkParameters::name())
        return IPC::handleMessage<Messages::NetworkProcess::AddWebPageNetworkParameters>(connection, decoder, this, &NetworkProcess::addWebPageNetworkParameters);
    if (decoder.messageName() == Messages::NetworkProcess::RemoveWebPageNetworkParameters::name())
        return IPC::handleMessage<Messages::NetworkProcess::RemoveWebPageNetworkParameters>(connection, decoder, this, &NetworkProcess::removeWebPageNetworkParameters);
    if (decoder.messageName() == Messages::NetworkProcess::CountNonDefaultSessionSets::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::CountNonDefaultSessionSets>(connection, decoder, this, &NetworkProcess::countNonDefaultSessionSets);
#if ENABLE(CFPREFS_DIRECT_MODE)
    if (decoder.messageName() == Messages::NetworkProcess::NotifyPreferencesChanged::name())
        return IPC::handleMessage<Messages::NetworkProcess::NotifyPreferencesChanged>(connection, decoder, this, &NetworkProcess::notifyPreferencesChanged);
#endif
#if ENABLE(SERVICE_WORKER)
    if (decoder.messageName() == Messages::NetworkProcess::GetPendingPushMessages::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::GetPendingPushMessages>(connection, decoder, this, &NetworkProcess::getPendingPushMessages);
#endif
#if ENABLE(SERVICE_WORKER)
    if (decoder.messageName() == Messages::NetworkProcess::ProcessPushMessage::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::ProcessPushMessage>(connection, decoder, this, &NetworkProcess::processPushMessage);
#endif
    if (decoder.messageName() == Messages::NetworkProcess::DeletePushAndNotificationRegistration::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::DeletePushAndNotificationRegistration>(connection, decoder, this, &NetworkProcess::deletePushAndNotificationRegistration);
    if (decoder.messageName() == Messages::NetworkProcess::GetOriginsWithPushAndNotificationPermissions::name())
        return IPC::handleMessageAsync<Messages::NetworkProcess::GetOriginsWithPushAndNotificationPermissions>(connection, decoder, this, &NetworkProcess::getOriginsWithPushAndNotificationPermissions);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

bool NetworkProcess::didReceiveSyncNetworkProcessMessage(IPC::Connection& connection, IPC::Decoder& decoder, UniqueRef<IPC::Encoder>& replyEncoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::NetworkProcess::SetAllowsAnySSLCertificateForWebSocket::name())
        return IPC::handleMessageSynchronous<Messages::NetworkProcess::SetAllowsAnySSLCertificateForWebSocket>(connection, decoder, replyEncoder, this, &NetworkProcess::setAllowsAnySSLCertificateForWebSocket);
    if (decoder.messageName() == Messages::NetworkProcess::SetCacheModelSynchronouslyForTesting::name())
        return IPC::handleMessageSynchronous<Messages::NetworkProcess::SetCacheModelSynchronouslyForTesting>(connection, decoder, replyEncoder, this, &NetworkProcess::setCacheModelSynchronouslyForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::ProcessWillSuspendImminentlyForTestingSync::name())
        return IPC::handleMessageSynchronous<Messages::NetworkProcess::ProcessWillSuspendImminentlyForTestingSync>(connection, decoder, replyEncoder, this, &NetworkProcess::processWillSuspendImminentlyForTestingSync);
    if (decoder.messageName() == Messages::NetworkProcess::SetServiceWorkerFetchTimeoutForTesting::name())
        return IPC::handleMessageSynchronous<Messages::NetworkProcess::SetServiceWorkerFetchTimeoutForTesting>(connection, decoder, replyEncoder, this, &NetworkProcess::setServiceWorkerFetchTimeoutForTesting);
    if (decoder.messageName() == Messages::NetworkProcess::ResetServiceWorkerFetchTimeoutForTesting::name())
        return IPC::handleMessageSynchronous<Messages::NetworkProcess::ResetServiceWorkerFetchTimeoutForTesting>(connection, decoder, replyEncoder, this, &NetworkProcess::resetServiceWorkerFetchTimeoutForTesting);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
    UNUSED_PARAM(replyEncoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return false;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled synchronous message %s to %" PRIu64, description(decoder.messageName()), decoder.destinationID());
    return false;
}

} // namespace WebKit
