"""CacheMixin for IntelMQ

SPDX-FileCopyrightText: 2021 Sebastian Waldbauer
SPDX-License-Identifier: AGPL-3.0-or-later

CacheMixin is used for caching/storing data in redis.
"""

import json
from typing import Any, Optional
import redis
import intelmq.lib.utils as utils


class CacheMixin:
    """Provides caching possibilities for bots, see also https://docs.intelmq.org/latest/dev/bot-development/#mixins

    For key-value cache, use methods:
        cache_exists
        cache_get
        cache_set

    To store dict elements in a cache queue named after bot id, use methods:
        cache_lpush
        cache_rpop
        cache_llen
    """

    __redis: redis.Redis = None
    redis_cache_host: str = "127.0.0.1"
    redis_cache_port: int = 6379
    redis_cache_db: int = 9
    redis_cache_ttl: int = 15
    redis_cache_password: Optional[str] = None

    def __init__(self, **kwargs):
        if self.redis_cache_host.startswith("/"):
            kwargs = {"unix_socket_path": self.redis_cache_host}
        elif self.redis_cache_host.startswith("unix://"):
            kwargs = {"unix_socket_path": self.redis_cache_host.replace("unix://", "")}
        else:
            kwargs = {
                "host": self.redis_cache_host,
                "port": int(self.redis_cache_port),
                "socket_timeout": 5,
            }

        self.__redis = redis.Redis(
            db=self.redis_cache_db, password=self.redis_cache_password, **kwargs
        )
        super().__init__()

    def cache_exists(self, key: str):
        return self.__redis.exists(key)

    def cache_get(self, key: str):
        retval = self.__redis.get(key)
        if isinstance(retval, bytes):
            return utils.decode(retval)
        return retval

    def cache_set(self, key: str, value: Any, ttl: Optional[int] = None):
        if isinstance(value, str):
            value = utils.encode(value)
        # backward compatibility (Redis v2.2)
        self.__redis.set(key, value)
        if self.redis_cache_ttl:
            self.__redis.expire(key, self.redis_cache_ttl)

    def cache_lpush(self, key: str, value: Any) -> int:
        # Returns the length of the list after pushing
        return self.__redis.lpush(key, value)

    def cache_llen(self, key: str) -> int:
        return self.__redis.llen(key)

    def cache_rpop(self, key: str) -> Any:
        return self.__redis.rpop(key)

    def cache_flush(self):
        """
        Flushes the currently opened database by calling FLUSHDB.
        """
        self.__redis.flushdb()

    def cache_get_redis_instance(self):
        return self.__redis
