/**
 * PDF export utilities
 * Generates PDF reports with tables and charts using jsPDF
 * Copyright (c) 2025 Robert E. Lee <robert@unicornscan.org>
 */

import { jsPDF } from 'jspdf'
import autoTable from 'jspdf-autotable'
import type { Scan, IpReport, Host } from '@/types/database'
import { decodeTcpFlags, getProtocolName } from '@/types/database'
import { parseTimestamp, getHostIpAddr } from '@/lib/utils'
import type {
  ScanExportData,
  HostExportData,
  BulkExportData,
  MetadataDepth,
  ExportOptions,
  PDFSummaryStats,
} from './types'

// =============================================================================
// Constants
// =============================================================================

const COLORS = {
  primary: [25, 118, 210] as [number, number, number],    // Blue
  secondary: [156, 39, 176] as [number, number, number],  // Purple
  success: [76, 175, 80] as [number, number, number],     // Green
  warning: [255, 152, 0] as [number, number, number],     // Orange
  error: [244, 67, 54] as [number, number, number],       // Red
  text: [33, 33, 33] as [number, number, number],         // Dark gray
  muted: [117, 117, 117] as [number, number, number],     // Light gray
  white: [255, 255, 255] as [number, number, number],
}

const CHART_COLORS = [
  '#3b82f6', // Blue
  '#22c55e', // Green
  '#f59e0b', // Amber
  '#ef4444', // Red
  '#8b5cf6', // Purple
  '#06b6d4', // Cyan
  '#ec4899', // Pink
  '#84cc16', // Lime
  '#f97316', // Orange
  '#6366f1', // Indigo
]

const PAGE_MARGIN = 20
const HEADER_HEIGHT = 30

// =============================================================================
// PDF Document Creation
// =============================================================================

function createPDFDocument(orientation: 'portrait' | 'landscape'): jsPDF {
  const doc = new jsPDF({
    orientation,
    unit: 'mm',
    format: 'a4',
  })

  return doc
}

function addPageHeader(doc: jsPDF, title: string, pageNum: number): void {
  const pageWidth = doc.internal.pageSize.getWidth()

  // Title
  doc.setFontSize(14)
  doc.setTextColor(...COLORS.text)
  doc.setFont('helvetica', 'bold')
  doc.text(title, PAGE_MARGIN, 15)

  // Page number
  doc.setFontSize(9)
  doc.setFont('helvetica', 'normal')
  doc.setTextColor(...COLORS.muted)
  doc.text(`Page ${pageNum}`, pageWidth - PAGE_MARGIN, 15, { align: 'right' })

  // Separator line
  doc.setDrawColor(...COLORS.muted)
  doc.setLineWidth(0.5)
  doc.line(PAGE_MARGIN, 20, pageWidth - PAGE_MARGIN, 20)
}

function addPageFooter(doc: jsPDF): void {
  const pageWidth = doc.internal.pageSize.getWidth()
  const pageHeight = doc.internal.pageSize.getHeight()

  doc.setFontSize(8)
  doc.setTextColor(...COLORS.muted)
  doc.setFont('helvetica', 'normal')

  const timestamp = new Date().toLocaleString()
  doc.text(`Generated by Alicorn/unicornscan - ${timestamp}`, pageWidth / 2, pageHeight - 10, { align: 'center' })
}

// =============================================================================
// Chart Drawing
// =============================================================================

/**
 * Draw a bar chart on the PDF
 */
function drawBarChart(
  doc: jsPDF,
  x: number,
  y: number,
  width: number,
  height: number,
  data: { label: string; value: number }[],
  title: string
): number {
  const chartHeight = height - 25 // Space for title and labels
  const barWidth = (width - 20) / data.length - 4
  const maxValue = Math.max(...data.map((d) => d.value), 1)

  // Title
  doc.setFontSize(10)
  doc.setFont('helvetica', 'bold')
  doc.setTextColor(...COLORS.text)
  doc.text(title, x + width / 2, y, { align: 'center' })

  // Y-axis
  doc.setDrawColor(...COLORS.muted)
  doc.setLineWidth(0.3)
  doc.line(x + 15, y + 10, x + 15, y + 10 + chartHeight)

  // X-axis
  doc.line(x + 15, y + 10 + chartHeight, x + width - 5, y + 10 + chartHeight)

  // Bars
  data.forEach((item, i) => {
    const barHeight = (item.value / maxValue) * (chartHeight - 10)
    const barX = x + 20 + i * (barWidth + 4)
    const barY = y + 10 + chartHeight - barHeight

    // Bar fill
    const color = hexToRGB(CHART_COLORS[i % CHART_COLORS.length])
    doc.setFillColor(color.r, color.g, color.b)
    doc.rect(barX, barY, barWidth, barHeight, 'F')

    // Label
    doc.setFontSize(7)
    doc.setFont('helvetica', 'normal')
    doc.setTextColor(...COLORS.text)
    const labelText = truncateText(item.label, 8)
    doc.text(labelText, barX + barWidth / 2, y + 10 + chartHeight + 4, { align: 'center' })

    // Value on top of bar
    if (barHeight > 8) {
      doc.setTextColor(...COLORS.white)
      doc.text(String(item.value), barX + barWidth / 2, barY + barHeight - 2, { align: 'center' })
    }
  })

  return y + height
}

/**
 * Draw a pie chart on the PDF
 */
function drawPieChart(
  doc: jsPDF,
  x: number,
  y: number,
  radius: number,
  data: { label: string; value: number }[],
  title: string
): number {
  const total = data.reduce((sum, d) => sum + d.value, 0)
  if (total === 0) return y + radius * 2 + 30

  // Title
  doc.setFontSize(10)
  doc.setFont('helvetica', 'bold')
  doc.setTextColor(...COLORS.text)
  doc.text(title, x + radius, y, { align: 'center' })

  const centerX = x + radius
  const centerY = y + 15 + radius
  let currentAngle = -Math.PI / 2 // Start from top

  data.forEach((item, i) => {
    const sliceAngle = (item.value / total) * 2 * Math.PI
    const endAngle = currentAngle + sliceAngle

    // Draw slice using path
    const color = hexToRGB(CHART_COLORS[i % CHART_COLORS.length])
    doc.setFillColor(color.r, color.g, color.b)

    // Create pie slice using lines (jsPDF doesn't have native arc fill)
    const segments = 20
    const angleStep = sliceAngle / segments

    doc.setDrawColor(255, 255, 255)
    doc.setLineWidth(0.5)

    for (let j = 0; j < segments; j++) {
      const a1 = currentAngle + j * angleStep
      const a2 = currentAngle + (j + 1) * angleStep

      const x1 = centerX + Math.cos(a1) * radius
      const y1 = centerY + Math.sin(a1) * radius
      const x2 = centerX + Math.cos(a2) * radius
      const y2 = centerY + Math.sin(a2) * radius

      // Draw triangle for each segment
      doc.triangle(centerX, centerY, x1, y1, x2, y2, 'F')
    }

    currentAngle = endAngle
  })

  // Legend
  const legendX = x + radius * 2 + 10
  let legendY = y + 15

  data.forEach((item, i) => {
    const color = hexToRGB(CHART_COLORS[i % CHART_COLORS.length])
    doc.setFillColor(color.r, color.g, color.b)
    doc.rect(legendX, legendY - 3, 4, 4, 'F')

    doc.setFontSize(8)
    doc.setFont('helvetica', 'normal')
    doc.setTextColor(...COLORS.text)
    const percent = ((item.value / total) * 100).toFixed(1)
    doc.text(`${truncateText(item.label, 12)} (${percent}%)`, legendX + 6, legendY)
    legendY += 6
  })

  return y + radius * 2 + 30
}

// =============================================================================
// Summary Statistics
// =============================================================================

function calculateSummaryStats(reports: IpReport[]): PDFSummaryStats {
  const hosts = new Set(reports.map((r) => r.host_addr))
  const ports = new Set(reports.map((r) => r.sport))

  // Port frequency
  const portCounts: Record<number, number> = {}
  for (const report of reports) {
    portCounts[report.sport] = (portCounts[report.sport] || 0) + 1
  }

  const topPorts = Object.entries(portCounts)
    .sort(([, a], [, b]) => b - a)
    .slice(0, 10)
    .map(([port, count]) => ({ port: parseInt(port, 10), count }))

  // Protocol distribution
  const protoCounts: Record<string, number> = {}
  for (const report of reports) {
    const proto = getProtocolName(report.proto)
    protoCounts[proto] = (protoCounts[proto] || 0) + 1
  }

  const protocolDistribution = Object.entries(protoCounts)
    .map(([protocol, count]) => ({ protocol, count }))

  // OS distribution (from TTL inference)
  const osCounts: Record<string, number> = { Linux: 0, Windows: 0, Router: 0, Unknown: 0 }
  for (const report of reports) {
    if (report.ttl <= 64) osCounts.Linux++
    else if (report.ttl <= 128) osCounts.Windows++
    else if (report.ttl <= 255) osCounts.Router++
    else osCounts.Unknown++
  }

  const osDistribution = Object.entries(osCounts)
    .filter(([, count]) => count > 0)
    .map(([os, count]) => ({ os, count }))

  return {
    totalHosts: hosts.size,
    totalPorts: ports.size,
    uniquePorts: ports.size,
    topPorts,
    protocolDistribution,
    osDistribution,
  }
}

// =============================================================================
// Export Functions
// =============================================================================

/**
 * Export single scan to PDF with tables and charts
 */
export function exportScanToPDF(
  data: ScanExportData,
  options: ExportOptions
): Blob {
  const doc = createPDFDocument(options.pageOrientation)
  const pageWidth = doc.internal.pageSize.getWidth()
  let currentPage = 1
  let yPosition = HEADER_HEIGHT

  addPageHeader(doc, `Scan Report #${data.scan.scan_id}`, currentPage)

  // Scan metadata section
  yPosition = addScanMetadataSection(doc, data.scan, data.reports.length, yPosition)

  // Summary statistics
  if (options.includeSummaryStats) {
    const stats = calculateSummaryStats(data.reports)
    yPosition = addSummaryStatsSection(doc, stats, yPosition)
  }

  // Charts
  if (options.includeCharts && data.reports.length > 0) {
    const stats = calculateSummaryStats(data.reports)

    // Check if we need a new page
    if (yPosition > doc.internal.pageSize.getHeight() - 100) {
      doc.addPage()
      currentPage++
      addPageHeader(doc, `Scan Report #${data.scan.scan_id}`, currentPage)
      yPosition = HEADER_HEIGHT
    }

    yPosition += 10

    // Top ports bar chart
    if (stats.topPorts.length > 0) {
      yPosition = drawBarChart(
        doc,
        PAGE_MARGIN,
        yPosition,
        (pageWidth - PAGE_MARGIN * 2) / 2 - 5,
        60,
        stats.topPorts.slice(0, 8).map((p) => ({ label: String(p.port), value: p.count })),
        'Top Ports by Frequency'
      )
    }

    // Protocol distribution pie chart
    if (stats.protocolDistribution.length > 0) {
      yPosition = drawPieChart(
        doc,
        pageWidth / 2 + 5,
        yPosition - 60,
        25,
        stats.protocolDistribution.map((p) => ({ label: p.protocol, value: p.count })),
        'Protocol Distribution'
      )
    }

    yPosition += 10
  }

  // Port results table
  if (options.includeReports && data.reports.length > 0) {
    // Check if we need a new page
    if (yPosition > doc.internal.pageSize.getHeight() - 60) {
      doc.addPage()
      currentPage++
      addPageHeader(doc, `Scan Report #${data.scan.scan_id}`, currentPage)
      yPosition = HEADER_HEIGHT
    }

    yPosition = addReportsTable(doc, data.reports, options.metadataDepth, yPosition)
  }

  addPageFooter(doc)

  return doc.output('blob')
}

/**
 * Export single host to PDF
 */
export function exportHostToPDF(
  data: HostExportData,
  options: ExportOptions
): Blob {
  const doc = createPDFDocument(options.pageOrientation)
  let currentPage = 1
  let yPosition = HEADER_HEIGHT

  const hostIp = getHostIpAddr(data.host)
  addPageHeader(doc, `Host Report: ${hostIp}`, currentPage)

  // Host metadata section
  yPosition = addHostMetadataSection(doc, data.host, yPosition)

  // Port history table
  if (options.includeReports && data.reports.length > 0) {
    if (yPosition > doc.internal.pageSize.getHeight() - 60) {
      doc.addPage()
      currentPage++
      addPageHeader(doc, `Host Report: ${hostIp}`, currentPage)
      yPosition = HEADER_HEIGHT
    }

    yPosition = addReportsTable(doc, data.reports, options.metadataDepth, yPosition)
  }

  addPageFooter(doc)

  return doc.output('blob')
}

/**
 * Export multiple scans to PDF (bulk)
 */
export function exportBulkScansToPDF(
  data: BulkExportData,
  options: ExportOptions
): Blob {
  const doc = createPDFDocument(options.pageOrientation)
  const pageWidth = doc.internal.pageSize.getWidth()
  const currentPage = 1
  let yPosition = HEADER_HEIGHT

  addPageHeader(doc, 'Bulk Scan Report', currentPage)

  // Summary section
  doc.setFontSize(12)
  doc.setFont('helvetica', 'bold')
  doc.text(`Total Scans: ${data.scans.length}`, PAGE_MARGIN, yPosition)
  yPosition += 8

  const allReports = data.scans.flatMap((s) => s.reports)
  const totalHosts = new Set(allReports.map((r) => r.host_addr)).size
  const totalPorts = new Set(allReports.map((r) => `${r.sport}/${r.proto}`)).size

  doc.setFontSize(10)
  doc.setFont('helvetica', 'normal')
  doc.text(`Total Unique Hosts: ${totalHosts}`, PAGE_MARGIN, yPosition)
  yPosition += 6
  doc.text(`Total Unique Ports: ${totalPorts}`, PAGE_MARGIN, yPosition)
  yPosition += 6
  doc.text(`Generated: ${new Date(data.timestamp).toLocaleString()}`, PAGE_MARGIN, yPosition)
  yPosition += 15

  // Aggregate charts
  if (options.includeCharts && allReports.length > 0) {
    const stats = calculateSummaryStats(allReports)

    if (stats.topPorts.length > 0) {
      yPosition = drawBarChart(
        doc,
        PAGE_MARGIN,
        yPosition,
        (pageWidth - PAGE_MARGIN * 2) / 2 - 5,
        60,
        stats.topPorts.slice(0, 8).map((p) => ({ label: String(p.port), value: p.count })),
        'Top Ports (All Scans)'
      )
    }

    yPosition += 10
  }

  // Scans summary table
  const scansTableData = data.scans.map((s) => {
    const hostCount = new Set(s.reports.map((r) => r.host_addr)).size
    return [
      s.scan.scan_id,
      new Date(s.scan.s_time * 1000).toLocaleDateString(),
      s.scan.target_str ?? '—',
      hostCount,
      s.reports.length,
      s.scan.profile,
    ]
  })

  autoTable(doc, {
    startY: yPosition,
    head: [['ID', 'Date', 'Target', 'Hosts', 'Ports', 'Profile']],
    body: scansTableData,
    styles: { fontSize: 8 },
    headStyles: { fillColor: COLORS.primary },
    margin: { left: PAGE_MARGIN, right: PAGE_MARGIN },
  })

  addPageFooter(doc)

  return doc.output('blob')
}

/**
 * Export hosts list to PDF
 */
export function exportHostsListToPDF(
  hosts: Host[],
  options: ExportOptions
): Blob {
  const doc = createPDFDocument(options.pageOrientation)
  let yPosition = HEADER_HEIGHT

  addPageHeader(doc, 'Host Inventory Report', 1)

  // Summary (port_count = responding ports, not "open" in nmap sense)
  const getPortCount = (h: Host) => h.port_count ?? 0
  doc.setFontSize(10)
  doc.setFont('helvetica', 'normal')
  doc.text(`Total Hosts: ${hosts.length}`, PAGE_MARGIN, yPosition)
  yPosition += 6
  doc.text(`Hosts with Responses: ${hosts.filter((h) => getPortCount(h) > 0).length}`, PAGE_MARGIN, yPosition)
  yPosition += 15

  // Hosts table
  const hostsTableData = hosts.map((h) => [
    h.ip_addr ?? h.host_addr,
    h.hostname || '-',
    getPortCount(h),
    h.scan_count,
    new Date(parseTimestamp(h.last_seen) * 1000).toLocaleDateString(),
    h.os_guess || '-',
  ])

  autoTable(doc, {
    startY: yPosition,
    head: [['IP Address', 'Hostname', 'Responses', 'Scan Count', 'Last Seen', 'OS']],
    body: hostsTableData,
    styles: { fontSize: 8 },
    headStyles: { fillColor: COLORS.primary },
    margin: { left: PAGE_MARGIN, right: PAGE_MARGIN },
  })

  addPageFooter(doc)

  return doc.output('blob')
}

// =============================================================================
// Helper Functions
// =============================================================================

function addScanMetadataSection(doc: jsPDF, scan: Scan, reportCount: number, startY: number): number {
  let y = startY

  doc.setFontSize(12)
  doc.setFont('helvetica', 'bold')
  doc.setTextColor(...COLORS.text)
  doc.text('Scan Details', PAGE_MARGIN, y)
  y += 8

  doc.setFontSize(9)
  doc.setFont('helvetica', 'normal')

  const metadata = [
    ['Target:', scan.target_str ?? '—'],
    ['Port Range:', scan.port_str ?? '—'],
    ['Start Time:', new Date(scan.s_time * 1000).toLocaleString()],
    ['Duration:', formatDuration(scan.e_time - scan.s_time)],
    ['Profile:', scan.profile],
    ['Mode:', scan.mode_str ?? '—'],
    ['PPS:', (scan.pps ?? 0).toLocaleString()],
    ['Responses:', reportCount.toLocaleString()],
  ]

  for (const [label, value] of metadata) {
    doc.setFont('helvetica', 'bold')
    doc.text(label, PAGE_MARGIN, y)
    doc.setFont('helvetica', 'normal')
    doc.text(String(value), PAGE_MARGIN + 30, y)
    y += 5
  }

  return y + 10
}

function addHostMetadataSection(doc: jsPDF, host: Host, startY: number): number {
  let y = startY

  doc.setFontSize(12)
  doc.setFont('helvetica', 'bold')
  doc.setTextColor(...COLORS.text)
  doc.text('Host Details', PAGE_MARGIN, y)
  y += 8

  doc.setFontSize(9)
  doc.setFont('helvetica', 'normal')

  const portCount = host.port_count ?? 0
  const metadata = [
    ['IP Address:', host.ip_addr ?? host.host_addr],
    ['Hostname:', host.hostname ?? '-'],
    ['MAC Address:', host.mac_addr ?? '-'],
    ['OS Guess:', host.os_guess ?? '-'],
    ['First Seen:', new Date(parseTimestamp(host.first_seen) * 1000).toLocaleString()],
    ['Last Seen:', new Date(parseTimestamp(host.last_seen) * 1000).toLocaleString()],
    ['Scan Count:', host.scan_count.toString()],
    ['Responding Ports:', portCount.toString()],
  ]

  for (const [label, value] of metadata) {
    doc.setFont('helvetica', 'bold')
    doc.text(label, PAGE_MARGIN, y)
    doc.setFont('helvetica', 'normal')
    doc.text(String(value), PAGE_MARGIN + 30, y)
    y += 5
  }

  return y + 10
}

function addSummaryStatsSection(doc: jsPDF, stats: PDFSummaryStats, startY: number): number {
  let y = startY

  doc.setFontSize(11)
  doc.setFont('helvetica', 'bold')
  doc.text('Summary Statistics', PAGE_MARGIN, y)
  y += 8

  doc.setFontSize(9)
  doc.setFont('helvetica', 'normal')

  const summaryData = [
    ['Total Hosts:', stats.totalHosts.toString()],
    ['Unique Ports:', stats.uniquePorts.toString()],
  ]

  for (const [label, value] of summaryData) {
    doc.setFont('helvetica', 'bold')
    doc.text(label, PAGE_MARGIN, y)
    doc.setFont('helvetica', 'normal')
    doc.text(value, PAGE_MARGIN + 30, y)
    y += 5
  }

  return y + 10
}

function addReportsTable(
  doc: jsPDF,
  reports: IpReport[],
  depth: MetadataDepth,
  startY: number
): number {
  const tableData = reports.map((r) => {
    const base = [
      r.host_addr,
      r.sport,
      getProtocolName(r.proto),
    ]

    if (depth === 'basic') return base

    base.push(
      r.ttl,
      decodeTcpFlags(r.type).join(','),  // TCP flags are in type field
      new Date(r.tstamp * 1000).toLocaleTimeString()
    )

    if (depth === 'full') {
      base.push(r.window_size, r.sport)
    }

    return base
  })

  const headers = depth === 'basic'
    ? ['Host IP', 'Port', 'Protocol']
    : depth === 'standard'
      ? ['Host IP', 'Port', 'Protocol', 'TTL', 'Flags', 'Time']
      : ['Host IP', 'Port', 'Protocol', 'TTL', 'Flags', 'Time', 'Window', 'Src Port']

  autoTable(doc, {
    startY,
    head: [headers],
    body: tableData,
    styles: { fontSize: 8 },
    headStyles: { fillColor: COLORS.primary },
    margin: { left: PAGE_MARGIN, right: PAGE_MARGIN },
    didDrawPage: () => {
      addPageFooter(doc)
    },
  })

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  return (doc as any).lastAutoTable.finalY + 10
}

function hexToRGB(hex: string): { r: number; g: number; b: number } {
  const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex)
  return result
    ? {
      r: parseInt(result[1], 16),
      g: parseInt(result[2], 16),
      b: parseInt(result[3], 16),
    }
    : { r: 0, g: 0, b: 0 }
}

function truncateText(text: string, maxLength: number): string {
  if (text.length <= maxLength) return text
  return text.substring(0, maxLength - 1) + '…'
}

function formatDuration(seconds: number): string {
  if (seconds < 60) return `${seconds}s`
  if (seconds < 3600) {
    const mins = Math.floor(seconds / 60)
    const secs = seconds % 60
    return secs > 0 ? `${mins}m ${secs}s` : `${mins}m`
  }
  const hours = Math.floor(seconds / 3600)
  const mins = Math.floor((seconds % 3600) / 60)
  return mins > 0 ? `${hours}h ${mins}m` : `${hours}h`
}
