import vivisect
import envi.memory as e_mem
from vivisect.parsers.elf import archcalls

MS_MAPS = [
    (0x880000, 1, e_mem.MM_READ, b'\x00'),
    (0x884ec0, 32, e_mem.MM_RWX, b'\xb7D$*f\x89C\x0c\x0f\xb7D$,f\x89C\x0eH\x83\xc40[\xc3\x90\x90\x90\x90\x90\x90\x90\x90\x90'),
    (0x884ee0, 112, e_mem.MM_RWX, b'H\x83\xecHH\x85\xc9\x0f\x84{\xbe\x01\x00\x83\xf9\xf4\x0f\x83\xb97\x01\x00H\x85\xc9\x0f\x84i\xbe\x01\x00L\x8dD$PH\x8dT$0A\xb9\x08\x00\x00\x00\xc7D$ \x04\x00\x00\x00\xff\x15\xcbS\x04\x00\x85\xc0\x0f\x88\xa77\x01\x00H\x8bD$P\x83\xf8\x07\x0f\x85\xff\xc2\x00\x00\xb8\x01\x00\x00\x00H\x83\xc4H\xc3\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90'),
    (0x891232, 68, e_mem.MM_RWX, b'\x8dH\xfe\x83\xf9)\x0f\x87\x89\xfb\x00\x00H\x8d\x15\xbb\xed\xfe\xff\x0f\xb6\x84\n@\xc5\x01\x00\x8b\x8c\x820\xc5\x01\x00H\x03\xca\xff\xe1\xb8\x03\x00\x00\x00H\x83\xc4H\xc3\x8b\xc1%\x00\x00\x00\xc0=\x00\x00\x00\x80\x0f\x84\xe8\xfa\x00\x00\xe8'),
    (0x8986af, 62, e_mem.MM_RWX, b'\x0f\x84\xf8\x86\x00\x00\x83\xf9\xf5\x0f\x84\xd5\x86\x00\x00\x83\xf9\xf6\x0f\x85/\xc8\xfe\xff\xe9\xad\x86\x00\x00\x8b\xc8\xe8=\x90\xfe\xff3\xc0H\x83\xc4H\xc3H\x8d\r?\x90\x04\x00\xff\x15\x81\x1b\x03\x00\xe84\x04\xff\xffH'),
    (0x899db9, 29, e_mem.MM_RWX, b'\xb8\x02\x00\x00\x00H\x83\xc4H\xc3\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90\x90H\x83\xec(D\x0f'),

    (0x89c540, 41, e_mem.MM_RWX, b'\x00\x00\x00\x00\x00\x00\x00\x03\x03\x01\x03\x03\x03\x01\x03\x02\x03\x03\x03\x01\x01\x03\x01\x03\x03\x01\x01\x01\x03\x03\x03\x03\x03\x03\x00\x03\x03\x03\x03\x03\x03'),
    (0x89c634, 24, e_mem.MM_RWX, b'\x89\xf2\x00\x00\x89\xf2\x00\x00\x89\xf2\x00\x00\x89\xf2\x00\x00W\x98\x00\x00\x0f\xf9\x00\x00'),
    (0x8a0d68, 129, e_mem.MM_RWX, b'\xb9\x08\x00\x00\xc0\xe8\x9e\t\xfe\xff3\xc0H\x83\xc4H\xc3eH\x8b\x04%0\x00\x00\x00H\x8bH`H\x8bA H\x8bH \xe9cA\xfe\xffeH\x8b\x04%0\x00\x00\x00H\x8bH`H\x8bA H\x8bH(\xe9IA\xfe\xffeH\x8b\x04%0\x00\x00\x00H\x8bH`H\x8bA H\x8bH0\xe9/A\xfe\xff3\xc9\xff\x15\xd1\x92\x02\x003\xc0H\x83\xc4H\xc3eH\x8b\x04%0\x00\x00\x00L\x8b\xc73\xd2H\x8bH`H'),
    (0x8ca0a0, 28, e_mem.MM_RWX, b'@ \xeax\x00\x00\x00\x00@\xff\xe9x\x00\x00\x00\x00p\x1e\xeax\x00\x00\x00\x00p\x12\xeax'),
    (0x8ca2e8, 28, e_mem.MM_RWX, b'@\x03\xeax\x00\x00\x00\x00\x10\xff\xe9x\x00\x00\x00\x000\xff\xe9x\x00\x00\x00\x00`\x03\xeax'),

    (0x89c530, 64, e_mem.MM_READ, b'3O\x00\x00\xb9\x9d\x01\x00Y\x12\x01\x00\xc7\r\x02\x00\x00\x00\x00\x00\x00\x00\x00\x03\x03\x01\x03\x03\x03\x01\x03\x02\x03\x03\x03\x01\x01\x03\x01\x03\x03\x01\x01\x01\x03\x03\x03\x03\x03\x03\x00\x03\x03\x03\x03\x03\x03\x01\x90\x90\x9cz\x01\x00'), # table
]

# 32bit libc-2.13.so.  switch jmp at 0x0205af11
LIBC_MAPS = [\
    (0x205ab40, 0x107, e_mem.MM_RWX, b'U\x89\xe5WVS\x83\xec\x14\x8bM\x08\xe8\x1e\xc1\xfb\xff\x81\xc3\xa3\xe4\x0f\x00\x8bu\x0c\x85\xc9\x0f\x84\x9c\x00\x00\x00\x8b\x93\xf8\xfe\xff\xff\x85\xd2tt\x8bE\x08\x8b@h\x85\xc0~j\x89e\xf0\x894$\xe8\xff\x8c\x01\x00\x8dx\x01\x8d\x04\xbd\x10\x00\x00\x00)\xc4\x8dD$\x1f\x83\xe0\xf0\x85\xfft%\x0f\xb6\x0e\x84\xc9xo1\xd2\xeb\r\x90\x8dt&\x00\x0f\xb6\x0c\x16\x84\xc9x^\x0f\xbe\xc9\x89\x0c\x90\x83\xc2\x019\xd7w\xeb\x8dU\x10\x89T$\x08\x89D$\x04\x8bE\x08\x89\x04$\xe8mN\xff\xff\x8be\xf0\x8de\xf4[^_]\xc3f\x90\x8dU\x10\x89t$\x04\x89T$\x08\x8bE\x08\x89\x04$\xe8\x1a#\xfe\xff\x8de\xf4[^_]\xc3\x8b\x83H\xff\xff\xff\x8b\x00\x89E\x08\xe9T\xff\xff\xff\x8d\x836\xa1\xfd\xff\x89D$\x0c\x8d\x83\x11\xa1\xfd\xff\x89D$\x04\x8d\x83\x1c\xa1\xfd\xff\xc7D$\x08/\x00\x00\x00\x89\x04$\xe8\x98\x8a\xfc\xff\x90\x90\x90\x90\x90\x90\x90\x90U\x89\xe5WVS\x83'),
    ]


WALKER_MAPS = [\
    (0x203f150, 0xfc, e_mem.MM_RWX, b'UH\x89\xe5H\x83\xec\x10H\x89}\xf8H\x8b}\xf8\xe8{\xfe\xff\xffH\x83\xc4\x10]\xc3\x0f\x1fD\x00\x00UH\x89\xe5H\x83\xec\x10H\x89}\xf8H\x8b}\xf8\xe8[\xfe\xff\xffH\x83\xc4\x10]\xc3\x0f\x1fD\x00\x00UH\x89\xe5H\x83\xec H\x89}\xf0H\x89u\xe8H\x8b}\xf0\xe87\xfe\xff\xffH\x8b}\xe8H\x89E\xe0\xe8:\xff\xff\xff\x8a\x08H\x8bE\xe0\x88\x08H\x83\xc4 ]\xc3f.\x0f\x1f\x84\x00\x00\x00\x00\x00f\x90UH\x89\xe5H\x83\xec H\x89}\xf0H\x8b}\xf0\xe8\x0b\x00\x00\x00H\x89E\xe8H\x83\xc4 ]\xc3\x90UH\x89\xe5H\x83\xec\x10H\x89}\xf8H\x8b}\xf8\xe8\xdb\xfd\xff\xffH\x83\xc4\x10]\xc3\x0f\x1fD\x00\x00UH\x89\xe5H\x83\xec\x10H\x89}\xf8H\x89u\xf0H\x8b}\xf8H\x8bu\xf0\xe8\xc3\x00\x00\x00H\x83\xc4\x10]\xc3f.\x0f\x1f\x84\x00\x00\x00\x00\x00\x0f\x1f\x00UH\x89\xe5H\x83\xec0H\x89u\xf8'),
    (0x203fdb0, 0x38c, e_mem.MM_RWX, b'UH\x89\xe5H\x83\xec@H\x89}\xf8H\x89u\xf0\x89U\xec\x8bU\xec\x89\xd6H\x89\xf7H\x83\xef\x03H\x89u\xd8H\x89}\xd0\x0f\x87i\x00\x00\x00H\x8d\x05\xd8p\x03\x00H\x8bM\xd8Hc\x14\x88H\x01\xc2\xff\xe2H\x8b}\xf8\xe8V\xf3\xff\xffH\x8d=O\xdb%\x00H\x898\xe9=\x00\x00\x00H\x8b}\xf0\xe8\x8e\x00\x00\x00H\x8b}\xf8H\x89E\xc8\xe8\xc1\x01\x00\x00H\x8b}\xc8H\x898\xe9\x1b\x00\x00\x00H\x8b}\xf8H\x8bu\xf0\xe8\xc8\x01\x00\x00\xe9\t\x00\x00\x00H\x8b}\xf8\xe8\xfa\x01\x00\x001\xc0\x88\xc1\x80\xe1\x01\x0f\xb6\xc1H\x83\xc4@]\xc3f.\x0f\x1f\x84\x00\x00\x00\x00\x00UH\x89\xe5H\x83\xec H\x89}\xf0H\x89u\xe8H\x8b}\xf0\xe8g\xf1\xff\xffH\x8b}\xe8H\x89E\xe0\xe8\xda\xfe\xff\xffH\x8b\x00H\x8bu\xe0H\x89\x06H\x83\xc4 ]\xc3f.\x0f\x1f\x84\x00\x00\x00\x00\x00UH\x89\xe5H\x83\xec\x10H\x89}\xf8H\x8b}\xf8\xe8\xdb\x00\x00\x00H\x89\xc7\xe8\xc3\x00\x00\x00H\x89E\xf0H\x8bE\xf0H\x83\xc4\x10]\xc3\x0f\x1fD\x00\x00UH\x89\xe5H\x83\xec0@\x88\xf0H\x89}\xf8\x88E\xf7H\x8b}\xf8\x80={\xea%\x00\x00H\x89}\xd8\x0f\x85=\x00\x00\x00H\x8d=j\xea%\x00\xe8\x1d\x90\xfc\xff\x83\xf8\x00\x0f\x84(\x00\x00\x001\xf6H\x8b}\xd8\xe8\x99\x00\x00\x00\x88E\xd7\xe9\x00\x00\x00\x00\x8aE\xd7\x88\x058\xea%\x00H\x8d=9\xea%\x00\xe8\x8c\x8a\xfc\xffH\x8bE\xd8H\x89\xc7\x0f\xbeu\xf7\xe8l\x00\x00\x00\x0f\xbe\xf0\x0f\xbe\r\x12\xea%\x009\xce\x0f\x95\xc0$\x01\x0f\xb6\xc0H\x83\xc40]\xc3\x89\xd1H\x89E\xe8\x89M\xe4H\x8d=\xfa\xe9%\x00\xe8-\x8a\xfc\xffH\x8b}\xe8\xe8d\x8f\xfc\xff\x0f\x1f@\x00UH\x89\xe5H\x89}\xf8H\x8bE\xf8]\xc3f\x90UH\x89\xe5H\x83\xec\x10H\x89}\xf8H\x8b}\xf8\xe8k\xf1\xff\xffH\x83\xc4\x10]\xc3\x0f\x1fD\x00\x00UH\x89\xe5H\x83\xec\x10@\x88\xf0H\x89}\xf8\x88E\xf7H\x8b}\xf8H\x8b?\x0f\xbeu\xf7\xe8\x1e\xfb\xff\xff\x0f\xbe\xc0H\x83\xc4\x10]\xc3\x0f\x1fD\x00\x00UH\x89\xe5H\x83\xec\x10H\x89}\xf8H\x8b}\xf8\xe8\xeb\xef\xff\xffH\x83\xc4\x10]\xc3\x0f\x1fD\x00\x00UH\x89\xe5H\x83\xec H\x89}\xf0H\x89u\xe8H\x8b}\xf0\xe8\xc7\xef\xff\xffH\x8b}\xe8H\x89E\xe0\xe8j\xff\xff\xffH\x8b\x00H\x8bu\xe0H\x89\x06H\x83\xc4 ]\xc3f.\x0f\x1f\x84\x00\x00\x00\x00\x00UH\x89\xe5H\x83\xec H\x89}\xf0H\x8b}\xf0\xe8\x0b\x00\x00\x00H\x89E\xe8H\x83\xc4 ]\xc3\x90UH\x89\xe5H\x83\xec\x10H\x89}\xf8H\x8b}\xf8\xe8k\xef\xff\xffH\x83\xc4\x10]\xc3\x0f\x1fD\x00\x00UH\x89\xe5H\x83\xec\x10H\x89}\xf8H\x89u\xf0H\x8b}\xf8H\x8bu\xf0\xe8\x93\xed\xff\xffH\x83\xc4\x10]\xc3f.\x0f\x1f\x84\x00\x00\x00\x00\x00\x0f\x1f\x00UH\x89\xe5H\x83\xec0H\x89}\xf0H\x8b}\xf0H\x89}\xd8\xe8\xb7\xab\xff\xffH\x8d}\xf8\xe8~\x00\x00\x00\x88E\xd7\xe9\x00\x00\x00\x00\x8aE\xd7\xa8\x01\x0f\x85\x05\x00\x00\x00\xe9S\x00\x00\x00H\x8d}\xf8\xe8\xbd\x00\x00\x00H\x8b}\xd8H\x89\xc6\xe8q\x00\x00\x00\xe9\x00\x00\x00\x00H\x8d\x05\xb5\x00\x00\x00H\x8bM\xd8H\x89A\x18H\x8d\x05\xe6\x00\x00\x00H\x89A\x10\xe9\x1a\x00\x00\x00\x89\xd1H\x89E\xe8\x89M\xe4H\x8bE\xd8H\x89\xc7\xe8\xa8\x8d\xff\xff\xe9\x06\x00\x00'),
    (0x2076ebc, 0xfc, e_mem.MM_RWX, b'5\x8f\xfc\xffM\x8f\xfc\xffo\x8f\xfc\xff\x81\x8f\xfc\xffu\x93\xfc\xff\x8d\x93\xfc\xff\xaf\x93\xfc\xff\xc1\x93\xfc\xff\x95\x97\xfc\xff\xad\x97\xfc\xff\xcf\x97\xfc\xff\xe1\x97\xfc\xff\xc5\x9b\xfc\xff\xdd\x9b\xfc\xff\xff\x9b\xfc\xff\x11\x9c\xfc\xff\xf5\x9f\xfc\xff\r\xa0\xfc\xff/\xa0\xfc\xffA\xa0\xfc\xffE\xa4\xfc\xff]\xa4\xfc\xff\x7f\xa4\xfc\xff\x91\xa4\xfc\xff\x15\xa8\xfc\xff-\xa8\xfc\xffO\xa8\xfc\xffa\xa8\xfc\xff\xe5\xab\xfc\xff\xfd\xab\xfc\xff\x1f\xac\xfc\xff1\xac\xfc\xff\xb5\xaf\xfc\xff\xcd\xaf\xfc\xff\xef\xaf\xfc\xff\x01\xb0\xfc\xff\x9b\xfa\xfc\xff\\\xfa\xfc\xff\x1d\xfa\xfc\xff\xde\xf9\xfc\xff\x15\x0b\xfd\xff-\x0b\xfd\xffO\x0b\xfd\xffa\x0b\xfd\xff55\xfd\xffM5\xfd\xffo5\xfd\xff\x815\xfd\xff\x85J\xfd\xff\x9dJ\xfd\xff\xbfJ\xfd\xff\xd1J\xfd\xff\x15[\xfd\xff-[\xfd\xffO[\xfd\xffa[\xfd\xff                            '),
    (0x229d950, 0xfc, e_mem.MM_RWX, b'\x00\x00\x00\x00\x00\x00\x00\x00 \x81\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00p\x81\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xc0\x81\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x82\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00`\x82\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xb0\x82\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x83\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00P\x83\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xa0\x83\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xf0\x83\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x84\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x90\x84\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x84\x07\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\xdb)\x02\x00\x00\x00\x00\x00z\x05\x02\x00\x00\x00\x00\x80z\x05\x02\x00\x00\x00\x00\xd0\x82\x05\x02\x00\x00\x00\x00\xf0\x82\x05\x02'),
]


def applyMaps(vw, maps=MS_MAPS, mapbase=0x400000, bufferpages=2):
    bufferpgsz = bufferpages * 4096
    vw.addMemoryMap(mapbase - bufferpgsz, e_mem.MM_RWX, 'testswitches', '@' * bufferpgsz)

    mapdelta = mapbase - maps[0][0]
    

    vw.addFile('testswitches', mapbase, "@@@@@@@@")
    for mapva, sz, perms, mem in maps:
        mapva += mapdelta
        vw.addMemoryMap(mapva, perms, 'testswitches', mem)
        vw.addSegment(mapva, len(mem), 'switch_code_%x' % mapva, 'testswitches')


def genMsSwitchWorkspace(maps=MS_MAPS, mapbase=0x400000, bufferpages=2, arch='amd64'):
    vw = vivisect.VivWorkspace()
    vw.setMeta('Architecture', arch)
    vw.setMeta('Platform', 'windows')
    vw.setMeta('Format', 'pe')
    vw._snapInAnalysisModules()

    applyMaps(vw, maps, mapbase, bufferpages)
    return vw

def genLinuxSwitchWorkspace(maps=LIBC_MAPS, mapbase=0x400000, bufferpages=2, arch='i386'):
    vw = vivisect.VivWorkspace()
    vw.setMeta('Architecture', arch)
    vw.setMeta('Platform', 'linux')
    vw.setMeta('Format', 'elf')
    vw.setMeta('DefaultCall', archcalls.get(arch,'unknown'))
    vw._snapInAnalysisModules()

    applyMaps(vw, maps, mapbase, bufferpages)
    return vw


cbs_ms_0 = [
    (0x404ee0, 0xd , 0x404ee0),
    (0x404eed, 0x9 , 0x404ee0),
    (0x404ef6, 0x9 , 0x404ee0),
    (0x404eff, 0x26, 0x404ee0),
    (0x404f25, 0xe , 0x404ee0),
    (0x404f33, 0xa , 0x404ee0),
    (0x411232, 0xc , 0x404ee0),
    (0x41123e, 0x1b, 0x404ee0),
    (0x411259, 0xa , 0x404ee0),
    (0x4186af, 0x6 , 0x404ee0),
    (0x4186b5, 0x9 , 0x404ee0),
    (0x4186be, 0x9 , 0x404ee0),
    (0x4186c7, 0x5 , 0x404ee0),
    (0x4186cc, 0xe , 0x404ee0),
    (0x419db9, 0xa , 0x404ee0),
    (0x420d68, 0x11, 0x404ee0),
    (0x420d79, 0x1a, 0x404ee0),
    (0x420d93, 0x1a, 0x404ee0),
    (0x420dad, 0x1a, 0x404ee0),
    (0x420dc7, 0xf , 0x404ee0)]

cbs_libc_0 = [
    (5242880, 34, 5242880),
    (5242914, 10, 5242880),
    (5242924, 10, 5242880),
    (5242934, 34, 5242880),
    (5242968, 7, 5242880),
    (5242975, 4, 5242880),
    (5242984, 8, 5242880),
    (5242992, 13, 5242880),
    (5243005, 33, 5242880),
    (5243040, 30, 5242880),
    (5243070, 16, 5242880),
    (5243086, 56, 5242880)]


cbs_walker_0 = set(
    [(0x600000, 0x2d, 0x600000),
     (0x60002d, 0x14, 0x600000),
     (0x600096, 0x10, 0x600000),
     (0x600041, 0x18, 0x600000),
     (0x600059, 0x22, 0x600000),
     (0x60007b, 0x12, 0x600000),
     (0x60008d, 0x9, 0x600000)]
    )

cbs_amd64_ls_40407e = [
    (0x402755, 5, 0x404223),
    (0x40309f, 20, 0x404223),
    (0x4030b3, 13, 0x404223),
    (0x4030c0, 13, 0x404223),
    (0x4030cd, 11, 0x404223),
    (0x4030d8, 9, 0x404223),
    (0x4030e1, 7, 0x404223),
    (0x4030e8, 2, 0x404223),
    (0x4030ea, 19, 0x404223),
    (0x4030fd, 59, 0x404223),
    (0x403138, 7, 0x404223),
    (0x40313f, 33, 0x404223),
    (0x403160, 14, 0x404223),
    (0x40316e, 22, 0x404223),
    (0x403184, 12, 0x404223),
    (0x403190, 26, 0x404223),
    (0x4031aa, 10, 0x404223),
    (0x4031b4, 59, 0x404223),
    (0x4031ef, 14, 0x404223),
    (0x4031fd, 24, 0x404223),
    (0x403215, 15, 0x404223),
    (0x403224, 8, 0x404223),
    (0x40322c, 70, 0x404223),
    (0x403272, 9, 0x404223),
    (0x40327b, 27, 0x404223),
    (0x403296, 34, 0x404223),
    (0x4032b8, 9, 0x404223),
    (0x4032c1, 13, 0x404223),
    (0x4032ce, 9, 0x404223),
    (0x4032d7, 21, 0x404223),
    (0x4032ec, 11, 0x404223),
    (0x4032f7, 8, 0x404223),
    (0x4032ff, 20, 0x404223),
    (0x403313, 47, 0x404223),
    (0x403342, 24, 0x404223),
    (0x40335a, 18, 0x404223),
    (0x40336c, 35, 0x404223),
    (0x40338f, 17, 0x404223),
    (0x4033a0, 30, 0x404223),
    (0x4033be, 18, 0x404223),
    (0x4033d0, 15, 0x404223),
    (0x4033df, 4, 0x404223),
    (0x4033e3, 12, 0x404223),
    (0x4033ef, 1, 0x404223),
    (0x4033f0, 20, 0x404223),
    (0x403404, 9, 0x404223),
    (0x40340d, 12, 0x404223),
    (0x403419, 7, 0x404223),
    (0x403420, 24, 0x404223),
    (0x403438, 9, 0x404223),
    (0x403441, 18, 0x404223),
    (0x403453, 10, 0x404223),
    (0x40345d, 29, 0x404223),
    (0x40347a, 13, 0x404223),
    (0x403487, 13, 0x404223),
    (0x403494, 13, 0x404223),
    (0x4034a1, 25, 0x404223),
    (0x403526, 8, 0x404223),
    (0x403530, 18, 0x404223),
    (0x403542, 4, 0x404223),
    (0x403546, 13, 0x404223),
    (0x403553, 11, 0x404223),
    (0x40355e, 52, 0x404223),
    (0x403592, 6, 0x404223),
    (0x403598, 22, 0x404223),
    (0x4035ae, 26, 0x404223),
    (0x4035c8, 5, 0x404223),
    (0x403624, 44, 0x404223),
    (0x403650, 30, 0x404223),
    (0x40366e, 23, 0x404223),
    (0x403685, 9, 0x404223),
    (0x40368e, 13, 0x404223),
    (0x40369b, 18, 0x404223),
    (0x4036ad, 14, 0x404223),
    (0x4036bb, 13, 0x404223),
    (0x4036c8, 14, 0x404223),
    (0x4036d6, 13, 0x404223),
    (0x4036e3, 18, 0x404223),
    (0x4036f5, 13, 0x404223),
    (0x403702, 30, 0x404223),
    (0x403720, 13, 0x404223),
    (0x40372d, 10, 0x404223),
    (0x403737, 37, 0x404223),
    (0x40375c, 24, 0x404223),
    (0x403774, 33, 0x404223),
    (0x403795, 10, 0x404223),
    (0x40379f, 5, 0x404223),
    (0x4037fa, 14, 0x404223),
    (0x403808, 9, 0x404223),
    (0x403811, 15, 0x404223),
    (0x403820, 11, 0x404223),
    (0x403830, 9, 0x404223),
    (0x403839, 43, 0x404223),
    (0x403864, 16, 0x404223),
    (0x403874, 18, 0x404223),
    (0x403886, 12, 0x404223),
    (0x403892, 13, 0x404223),
    (0x40389f, 13, 0x404223),
    (0x4038ac, 71, 0x404223),
    (0x4038f3, 11, 0x404223),
    (0x4038fe, 96, 0x404223),
    (0x40395e, 11, 0x404223),
    (0x403969, 8, 0x404223),
    (0x403971, 10, 0x404223),
    (0x40397b, 5, 0x404223),
    (0x403980, 68, 0x404223),
    (0x4039c4, 13, 0x404223),
    (0x4039d1, 9, 0x404223),
    (0x4039da, 14, 0x404223),
    (0x4039e8, 5, 0x404223),
    (0x4039ed, 16, 0x404223),
    (0x4039fd, 14, 0x404223),
    (0x403a0b, 22, 0x404223),
    (0x403a21, 11, 0x404223),
    (0x403a30, 8, 0x404223),
    (0x403a38, 10, 0x404223),
    (0x403a42, 8, 0x404223),
    (0x403a4a, 5, 0x404223),
    (0x403a4f, 6, 0x404223),
    (0x403a55, 45, 0x404223),
    (0x403a82, 17, 0x404223),
    (0x403a93, 37, 0x404223),
    (0x403ab8, 21, 0x404223),
    (0x403acd, 26, 0x404223),
    (0x403ae7, 20, 0x404223),
    (0x403afb, 64, 0x404223),
    (0x403b3b, 62, 0x404223),
    (0x403b79, 40, 0x404223),
    (0x403ba1, 48, 0x404223),
    (0x403bd1, 20, 0x404223),
    (0x403be5, 23, 0x404223),
    (0x403bfc, 76, 0x404223),
    (0x403c48, 27, 0x404223),
    (0x403c63, 27, 0x404223),
    (0x403c7e, 27, 0x404223),
    (0x403c99, 27, 0x404223),
    (0x403d18, 37, 0x404223),
    (0x403d3d, 13, 0x404223),
    (0x403d4a, 13, 0x404223),
    (0x403d57, 13, 0x404223),
    (0x403d64, 25, 0x404223),
    (0x403d7d, 13, 0x404223),
    (0x403d8a, 13, 0x404223),
    (0x403d97, 13, 0x404223),
    (0x403da4, 5, 0x404223),
    (0x403da9, 8, 0x404223),
    (0x403db1, 13, 0x404223),
    (0x403dbe, 4, 0x404223),
    (0x403dc2, 10, 0x404223),
    (0x403dcc, 5, 0x404223),
    (0x403dd1, 9, 0x404223),
    (0x403dda, 69, 0x404223),
    (0x403e1f, 12, 0x404223),
    (0x403e2b, 10, 0x404223),
    (0x403e35, 25, 0x404223),
    (0x403e4e, 12, 0x404223),
    (0x403e5a, 25, 0x404223),
    (0x403e73, 12, 0x404223),
    (0x403e7f, 11, 0x404223),
    (0x403e8a, 9, 0x404223),
    (0x403e93, 9, 0x404223),
    (0x403e9c, 26, 0x404223),
    (0x403eb6, 19, 0x404223),
    (0x403ec9, 30, 0x404223),
    (0x403ee7, 49, 0x404223),
    (0x403f18, 50, 0x404223),
    (0x403f4a, 9, 0x404223),
    (0x403f53, 9, 0x404223),
    (0x403f5c, 6, 0x404223),
    (0x403f62, 11, 0x404223),
    (0x403f6d, 36, 0x404223),
    (0x403f91, 28, 0x404223),
    (0x403fad, 22, 0x404223),
    (0x403fc3, 48, 0x404223),
    (0x403ff3, 4, 0x404223),
    (0x403ff7, 19, 0x404223),
    (0x40400a, 55, 0x404223),
    (0x404041, 49, 0x404223),
    (0x404072, 12, 0x404223),
    (0x40407e, 7, 0x404223),
    (0x4040b8, 13, 0x404223),
    (0x4040c5, 30, 0x404223),
    (0x4040e3, 9, 0x404223),
    (0x4040ec, 11, 0x404223),
    (0x4040f7, 12, 0x404223),
    (0x404103, 18, 0x404223),
    (0x404115, 14, 0x404223),
    (0x404123, 26, 0x404223),
    (0x40413d, 21, 0x404223),
    (0x404152, 11, 0x404223),
    (0x40415d, 25, 0x404223),
    (0x404176, 22, 0x404223),
    (0x4041b5, 15, 0x404223),
    (0x4041c4, 5, 0x404223),
    (0x4041c9, 30, 0x404223),
    (0x404203, 15, 0x404223),
    (0x404212, 17, 0x404223),
    (0x404223, 50, 0x404223),
    (0x404255, 5, 0x404223),
    (0x40425a, 14, 0x404223),
    (0x4042c3, 7, 0x404223),
    (0x4042ca, 14, 0x404223),
    (0x4042d8, 30, 0x404223),
    (0x4042f6, 12, 0x404223),
    (0x4044ab, 18, 0x404223),
    (0x4044bd, 57, 0x404223),
    (0x4044f6, 27, 0x404223),
    (0x404511, 30, 0x404223),
    (0x40452f, 14, 0x404223),
    (0x40453d, 16, 0x404223),
    (0x40454d, 5, 0x404223),
    (0x404552, 25, 0x404223),
    (0x40456b, 15, 0x404223),
    (0x40457a, 15, 0x404223),
    (0x404589, 27, 0x404223),
]

cbs_amd64_ls_402980 = [
    (0x402690, 187, 0x402690),
    (0x40274b, 5, 0x402690),
    (0x402750, 5, 0x402690),
    (0x402755, 5, 0x402690),
    (0x40275a, 18, 0x402690),
    (0x40276c, 19, 0x402690),
    (0x40277f, 22, 0x402690),
    (0x402795, 139, 0x402690),
    (0x402820, 31, 0x402690),
    (0x40283f, 16, 0x402690),
    (0x40284f, 40, 0x402690),
    (0x402877, 14, 0x402690),
    (0x402885, 29, 0x402690),
    (0x4028a2, 9, 0x402690),
    (0x4028ab, 30, 0x402690),
    (0x4028c9, 13, 0x402690),
    (0x4028d6, 10, 0x402690),
    (0x4028e0, 29, 0x402690),
    (0x4028fd, 34, 0x402690),
    (0x40291f, 15, 0x402690),
    (0x402968, 12, 0x402690),
    (0x402974, 10, 0x402690),
    (0x402980, 7, 0x402690),
    (0x402fb0, 27, 0x402690),
    (0x402fcb, 22, 0x402690),
    (0x402fe1, 38, 0x402690),
    (0x403007, 11, 0x402690),
    (0x403012, 19, 0x402690),
    (0x403025, 34, 0x402690),
    (0x403047, 41, 0x402690),
    (0x403070, 9, 0x402690),
    (0x403079, 7, 0x402690),
    (0x403080, 18, 0x402690),
    (0x403092, 13, 0x402690),
    (0x40309f, 20, 0x402690),
    (0x4030b3, 13, 0x402690),
    (0x4030c0, 13, 0x402690),
    (0x4030cd, 11, 0x402690),
    (0x4030d8, 9, 0x402690),
    (0x4030e1, 7, 0x402690),
    (0x4030e8, 2, 0x402690),
    (0x4030ea, 19, 0x402690),
    (0x4030fd, 59, 0x402690),
    (0x403138, 7, 0x402690),
    (0x40313f, 33, 0x402690),
    (0x403160, 14, 0x402690),
    (0x40316e, 22, 0x402690),
    (0x403184, 12, 0x402690),
    (0x403190, 26, 0x402690),
    (0x4031aa, 10, 0x402690),
    (0x4031b4, 59, 0x402690),
    (0x4031ef, 14, 0x402690),
    (0x4031fd, 24, 0x402690),
    (0x403215, 15, 0x402690),
    (0x403224, 8, 0x402690),
    (0x40322c, 70, 0x402690),
    (0x403272, 9, 0x402690),
    (0x40327b, 27, 0x402690),
    (0x403296, 34, 0x402690),
    (0x4032b8, 9, 0x402690),
    (0x4032c1, 13, 0x402690),
    (0x4032ce, 9, 0x402690),
    (0x4032d7, 21, 0x402690),
    (0x4032ec, 11, 0x402690),
    (0x4032f7, 8, 0x402690),
    (0x4032ff, 20, 0x402690),
    (0x403313, 47, 0x402690),
    (0x403342, 24, 0x402690),
    (0x40335a, 18, 0x402690),
    (0x40336c, 35, 0x402690),
    (0x40338f, 17, 0x402690),
    (0x4033a0, 30, 0x402690),
    (0x4033be, 18, 0x402690),
    (0x4033d0, 15, 0x402690),
    (0x4033df, 4, 0x402690),
    (0x4033e3, 12, 0x402690),
    (0x4033ef, 1, 0x402690),
    (0x4033f0, 20, 0x402690),
    (0x403404, 9, 0x402690),
    (0x40340d, 12, 0x402690),
    (0x403419, 7, 0x402690),
    (0x403420, 24, 0x402690),
    (0x403438, 9, 0x402690),
    (0x403441, 18, 0x402690),
    (0x403453, 10, 0x402690),
    (0x40345d, 29, 0x402690),
    (0x40347a, 13, 0x402690),
    (0x403487, 13, 0x402690),
    (0x403494, 13, 0x402690),
    (0x4034a1, 25, 0x402690),
    (0x4034ba, 34, 0x402690),
    (0x4034dc, 17, 0x402690),
    (0x4034ed, 12, 0x402690),
    (0x4034f9, 15, 0x402690),
    (0x403508, 15, 0x402690),
    (0x403517, 15, 0x402690),
    (0x403526, 8, 0x402690),
    (0x403530, 18, 0x402690),
    (0x403542, 4, 0x402690),
    (0x403546, 13, 0x402690),
    (0x403553, 11, 0x402690),
    (0x40355e, 52, 0x402690),
    (0x403592, 6, 0x402690),
    (0x403598, 22, 0x402690),
    (0x4035ae, 26, 0x402690),
    (0x4035c8, 5, 0x402690),
    (0x4035cd, 9, 0x402690),
    (0x4035d6, 13, 0x402690),
    (0x4035e3, 15, 0x402690),
    (0x4035f2, 50, 0x402690),
    (0x403624, 44, 0x402690),
    (0x403650, 30, 0x402690),
    (0x40366e, 23, 0x402690),
    (0x403685, 9, 0x402690),
    (0x40368e, 13, 0x402690),
    (0x40369b, 18, 0x402690),
    (0x4036ad, 14, 0x402690),
    (0x4036bb, 13, 0x402690),
    (0x4036c8, 14, 0x402690),
    (0x4036d6, 13, 0x402690),
    (0x4036e3, 18, 0x402690),
    (0x4036f5, 13, 0x402690),
    (0x403702, 30, 0x402690),
    (0x403720, 13, 0x402690),
    (0x40372d, 10, 0x402690),
    (0x403737, 37, 0x402690),
    (0x40375c, 24, 0x402690),
    (0x403774, 33, 0x402690),
    (0x403795, 10, 0x402690),
    (0x40379f, 5, 0x402690),
    (0x4037fa, 14, 0x402690),
    (0x403808, 9, 0x402690),
    (0x403811, 15, 0x402690),
    (0x403820, 11, 0x402690),
    (0x403830, 9, 0x402690),
    (0x403839, 43, 0x402690),
    (0x403864, 16, 0x402690),
    (0x403874, 18, 0x402690),
    (0x403886, 12, 0x402690),
    (0x403892, 13, 0x402690),
    (0x40389f, 13, 0x402690),
    (0x4038ac, 71, 0x402690),
    (0x4038f3, 11, 0x402690),
    (0x4038fe, 96, 0x402690),
    (0x40395e, 11, 0x402690),
    (0x403969, 8, 0x402690),
    (0x403971, 10, 0x402690),
    (0x40397b, 5, 0x402690),
    (0x403980, 68, 0x402690),
    (0x4039c4, 13, 0x402690),
    (0x4039d1, 9, 0x402690),
    (0x4039da, 14, 0x402690),
    (0x4039e8, 5, 0x402690),
    (0x4039ed, 16, 0x402690),
    (0x4039fd, 14, 0x402690),
    (0x403a0b, 22, 0x402690),
    (0x403a21, 11, 0x402690),
    (0x403a30, 8, 0x402690),
    (0x403a38, 10, 0x402690),
    (0x403a42, 8, 0x402690),
    (0x403a4a, 5, 0x402690),
    (0x403a4f, 6, 0x402690),
    (0x403a55, 45, 0x402690),
    (0x403a82, 17, 0x402690),
    (0x403a93, 37, 0x402690),
    (0x403ab8, 21, 0x402690),
    (0x403acd, 26, 0x402690),
    (0x403ae7, 20, 0x402690),
    (0x403afb, 64, 0x402690),
    (0x403b3b, 62, 0x402690),
    (0x403b79, 40, 0x402690),
    (0x403ba1, 48, 0x402690),
    (0x403bd1, 20, 0x402690),
    (0x403be5, 23, 0x402690),
    (0x403bfc, 76, 0x402690),
    (0x403c48, 27, 0x402690),
    (0x403c63, 27, 0x402690),
    (0x403c7e, 27, 0x402690),
    (0x403c99, 27, 0x402690),
    (0x403cb4, 50, 0x402690),
    (0x403ce6, 50, 0x402690),
    (0x403d18, 37, 0x402690),
    (0x403d3d, 13, 0x402690),
    (0x403d4a, 13, 0x402690),
    (0x403d57, 13, 0x402690),
    (0x403d64, 25, 0x402690),
    (0x403d7d, 13, 0x402690),
    (0x403d8a, 13, 0x402690),
    (0x403d97, 13, 0x402690),
    (0x403da4, 5, 0x402690),
    (0x403da9, 8, 0x402690),
    (0x403db1, 13, 0x402690),
    (0x403dbe, 4, 0x402690),
    (0x403dc2, 10, 0x402690),
    (0x403dcc, 5, 0x402690),
    (0x403dd1, 9, 0x402690),
    (0x403dda, 69, 0x402690),
    (0x403e1f, 12, 0x402690),
    (0x403e2b, 10, 0x402690),
    (0x403e35, 25, 0x402690),
    (0x403e4e, 12, 0x402690),
    (0x403e5a, 25, 0x402690),
    (0x403e73, 12, 0x402690),
    (0x403e7f, 11, 0x402690),
    (0x403e8a, 9, 0x402690),
    (0x403e93, 9, 0x402690),
    (0x403e9c, 26, 0x402690),
    (0x403eb6, 19, 0x402690),
    (0x403ec9, 30, 0x402690),
    (0x403ee7, 49, 0x402690),
    (0x403f18, 50, 0x402690),
    (0x403f4a, 9, 0x402690),
    (0x403f53, 9, 0x402690),
    (0x403f5c, 6, 0x402690),
    (0x403f62, 11, 0x402690),
    (0x403f6d, 36, 0x402690),
    (0x403f91, 28, 0x402690),
    (0x403fad, 22, 0x402690),
    (0x403fc3, 48, 0x402690),
    (0x403ff3, 4, 0x402690),
    (0x403ff7, 19, 0x402690),
    (0x40400a, 55, 0x402690),
    (0x404041, 61, 0x402690),
    (0x40407e, 7, 0x402690),
    (0x404085, 27, 0x402690),
    (0x4040a0, 19, 0x402690),
    (0x4040b3, 5, 0x402690),
    (0x4040b8, 13, 0x402690),
    (0x4040c5, 30, 0x402690),
    (0x4040e3, 9, 0x402690),
    (0x4040ec, 11, 0x402690),
    (0x4040f7, 12, 0x402690),
    (0x404103, 18, 0x402690),
    (0x404115, 14, 0x402690),
    (0x404123, 26, 0x402690),
    (0x40413d, 21, 0x402690),
    (0x404152, 11, 0x402690),
    (0x40415d, 25, 0x402690),
    (0x404176, 22, 0x402690),
    (0x4041b5, 15, 0x402690),
    (0x4041c4, 5, 0x402690),
    (0x4041c9, 30, 0x402690),
    (0x404203, 15, 0x402690),
    (0x404212, 17, 0x402690),
    (0x4044ab, 18, 0x402690),
    (0x4044bd, 57, 0x402690),
    (0x4044f6, 27, 0x402690),
    (0x404511, 30, 0x402690),
    (0x40452f, 14, 0x402690),
    (0x40453d, 16, 0x402690),
    (0x40454d, 5, 0x402690),
    (0x404552, 25, 0x402690),
    (0x40456b, 15, 0x402690),
    (0x40457a, 15, 0x402690),
    (0x404589, 27, 0x402690),
    (0x40292e, 10, 0x402690),
    (0x402938, 48, 0x402690),
    (0x402987, 7, 0x402690),
    (0x40298e, 12, 0x402690),
    (0x40299a, 37, 0x402690),
    (0x4029bf, 12, 0x402690),
    (0x4029cb, 37, 0x402690),
    (0x4029f0, 22, 0x402690),
    (0x402a06, 33, 0x402690),
    (0x402a27, 25, 0x402690),
    (0x402a40, 12, 0x402690),
    (0x402a4c, 15, 0x402690),
    (0x402a5b, 17, 0x402690),
    (0x402a6c, 15, 0x402690),
    (0x402a7b, 12, 0x402690),
    (0x402a87, 21, 0x402690),
    (0x402a9c, 21, 0x402690),
    (0x402ab1, 35, 0x402690),
    (0x402ad4, 20, 0x402690),
    (0x402ae8, 21, 0x402690),
    (0x402afd, 12, 0x402690),
    (0x402b09, 17, 0x402690),
    (0x402b1a, 14, 0x402690),
    (0x402b28, 15, 0x402690),
    (0x402b37, 17, 0x402690),
    (0x402b48, 15, 0x402690),
    (0x402b57, 12, 0x402690),
    (0x402b63, 15, 0x402690),
    (0x402b72, 12, 0x402690),
    (0x402b7e, 15, 0x402690),
    (0x402b8d, 25, 0x402690),
    (0x402ba6, 13, 0x402690),
    (0x402bb3, 15, 0x402690),
    (0x402bc2, 13, 0x402690),
    (0x402bcf, 15, 0x402690),
    (0x402c3b, 12, 0x402690),
    (0x402c47, 64, 0x402690),
    (0x402c87, 37, 0x402690),
    (0x402cac, 58, 0x402690),
    (0x402ce6, 12, 0x402690),
    (0x402cf2, 59, 0x402690),
    (0x402d2d, 58, 0x402690),
    (0x402d67, 43, 0x402690),
    (0x402d92, 12, 0x402690),
    (0x402d9e, 21, 0x402690),
    (0x402db3, 58, 0x402690),
    (0x402ded, 15, 0x402690),
    (0x402dfc, 15, 0x402690),
    (0x402e0b, 16, 0x402690),
    (0x402e1b, 49, 0x402690),
    (0x402e4c, 9, 0x402690),
    (0x402e55, 12, 0x402690),
    (0x402e61, 30, 0x402690),
    (0x402e7f, 19, 0x402690),
    (0x402e92, 12, 0x402690),
    (0x402e9e, 15, 0x402690),
    (0x402ead, 31, 0x402690),
    (0x402ecc, 11, 0x402690),
    (0x402ed7, 52, 0x402690),
    (0x402f0b, 20, 0x402690),
    (0x402f1f, 21, 0x402690),
    (0x402f34, 15, 0x402690),
    (0x402f43, 21, 0x402690),
    (0x402f58, 12, 0x402690),
    (0x402f64, 12, 0x402690),
    (0x402f70, 12, 0x402690),
    (0x402f7c, 15, 0x402690),
    (0x402f8b, 22, 0x402690),
    (0x402fa1, 15, 0x402690),
    (0x4037a4, 57, 0x402690),
    (0x4037dd, 29, 0x402690),
    (0x40418c, 18, 0x402690),
    (0x40419e, 23, 0x402690),
    (0x4041e7, 28, 0x402690),
]

cbs_amd64_ls_4048a4 = [
    (0x404740, 0x25, 0x404740),
    (0x404765, 0x5,  0x404740),
    (0x40476a, 0x6,  0x404740),
    (0x404770, 0x9,  0x404740),
    (0x404779, 0x9,  0x404740),
    (0x404782, 0x6,  0x404740),
    (0x404788, 0x9,  0x404740),
    (0x404791, 0x9,  0x404740),
    (0x40479a, 0x16, 0x404740),
    (0x4047b0, 0x9,  0x404740),
    (0x4047b9, 0x2,  0x404740),
    (0x4047bb, 0x11, 0x404740),
    (0x4047d0, 0x8,  0x404740),
    (0x4047d8, 0xe,  0x404740),
    (0x4047e6, 0xb,  0x404740),
    (0x4047f1, 0x9,  0x404740),
    (0x4047fa, 0x8,  0x404740),
    (0x404802, 0x7,  0x404740),
    (0x404810, 0x5,  0x404740),
    (0x404815, 0x9,  0x404740),
    (0x40481e, 0xe,  0x404740),
    (0x40482c, 0x5,  0x404740),
    (0x404838, 0xf,  0x404740),
    (0x404847, 0xc,  0x404740),
    (0x404853, 0xd,  0x404740),
    (0x404860, 0xd,  0x404740),
    (0x404870, 0x4,  0x404740),
    (0x404874, 0x9,  0x404740),
    (0x40489a, 0x2,  0x404740),
    (0x4048a0, 0xc,  0x404740),
    (0x4048b0, 0x3,  0x404740),
    (0x4048b3, 0x14, 0x404740),
    (0x4048d0, 0x4,  0x404740),
    (0x4048d4, 0x5,  0x404740),
    (0x4048d9, 0xb,  0x404740),
    (0x4048e8, 0xd,  0x404740),
    (0x4048f5, 0xd,  0x404740),
    (0x404902, 0x1d, 0x404740),
    (0x40491f, 0x5,  0x404740),
    (0x404928, 0x5,  0x404740),
    (0x40492d, 0x5,  0x404740),
    (0x4049c0, 0x1d, 0x404740),
    (0x4049dd, 0x5,  0x404740),
    (0x4049e8, 0x11, 0x404740),
    (0x40487d, 0x3,  0x404740),
    (0x404880, 0xd,  0x404740),
    (0x40488d, 0xd,  0x404740),
    (0x404932, 0xa,  0x404740),
    (0x40493c, 0xa,  0x404740),
    (0x404946, 0xa,  0x404740),
    (0x404950, 0xd,  0x404740),
    (0x40495d, 0xa,  0x404740),
    (0x404967, 0xa,  0x404740),
    (0x404971, 0xa,  0x404740),
    (0x40497b, 0xa,  0x404740),
    (0x404985, 0xa,  0x404740),
    (0x40498f, 0xa,  0x404740),
    (0x404999, 0xa,  0x404740),
    (0x4049a3, 0xa,  0x404740),
    (0x4049ad, 0xc,  0x404740)
]

cbs_amd64_ls_40d57f = [
    (0x40d400, 0xc, 0x40d840),
    (0x40d40c, 0x14, 0x40d840),
    (0x40d420, 0x38, 0x40d840),
    (0x40d458, 0x9, 0x40d840),
    (0x40d461, 0x6, 0x40d840),
    (0x40d467, 0xe, 0x40d840),
    (0x40d475, 0xf, 0x40d840),
    (0x40d484, 0x3f, 0x40d840),
    (0x40d4c3, 0x9, 0x40d840),
    (0x40d4cc, 0xe, 0x40d840),
    (0x40d4da, 0x6, 0x40d840),
    (0x40d4e0, 0x8, 0x40d840),
    (0x40d4e8, 0x8, 0x40d840),
    (0x40d4f0, 0x1d, 0x40d840),
    (0x40d50d, 0x5, 0x40d840),
    (0x40d512, 0x9, 0x40d840),
    (0x40d51b, 0x7, 0x40d840),
    (0x40d522, 0x9, 0x40d840),
    (0x40d52b, 0x4, 0x40d840),
    (0x40d52f, 0x4, 0x40d840),
    (0x40d533, 0x5, 0x40d840),
    (0x40d538, 0x9, 0x40d840),
    (0x40d541, 0x4, 0x40d840),
    (0x40d545, 0x10, 0x40d840),
    (0x40d555, 0x8, 0x40d840),
    (0x40d55d, 0x9, 0x40d840),
    (0x40d566, 0x15, 0x40d840),
    (0x40d57b, 0xb, 0x40d840),
    (0x40d820, 0x9, 0x40d840),
    (0x40d829, 0x9, 0x40d840),
    (0x40d832, 0x5, 0x40d840),
    (0x40d840, 0x11, 0x40d840),
    (0x40d851, 0x4, 0x40d840),
    (0x40d855, 0x5, 0x40d840),
    (0x40d85a, 0x7, 0x40d840),
    (0x40d861, 0x5, 0x40d840),
    (0x40d866, 0xa, 0x40d840),
    (0x40d870, 0x3, 0x40d840),
    (0x40d873, 0x49, 0x40d840),
    (0x40d8bc, 0x1a, 0x40d840),
    (0x40d8d6, 0x12, 0x40d840),
    (0x40d8f0, 0x5, 0x40d840),
    (0x40d8f5, 0x7, 0x40d840),
    (0x40d8fc, 0x9, 0x40d840),
    (0x40d905, 0x8, 0x40d840),
    (0x40d910, 0x7, 0x40d840),
    (0x40d920, 0xc, 0x40d840),
    (0x40d92c, 0x5, 0x40d840),
    (0x40d931, 0x5, 0x40d840),
    (0x40d940, 0x7, 0x40d840),
    (0x40d950, 0x7, 0x40d840),
    (0x40d960, 0x9, 0x40d840),
    (0x40d969, 0x9, 0x40d840),
    (0x40d972, 0x5, 0x40d840),
    (0x40d977, 0x9, 0x40d840),
    (0x40d980, 0x17, 0x40d840),
    (0x40d997, 0xf, 0x40d840),
    (0x40d9a6, 0x9, 0x40d840),
    (0x40d9af, 0x5, 0x40d840),
    (0x40d9b4, 0x9, 0x40d840),
    (0x40d9bd, 0x9, 0x40d840),
    (0x40d9c6, 0xa, 0x40d840),
    (0x40d9d0, 0xf, 0x40d840),
    (0x40da68, 0xa, 0x40d840),
    (0x40da78, 0xa, 0x40d840),
    (0x40da88, 0x10, 0x40d840),
    (0x40da98, 0x8, 0x40d840),
    (0x40daa0, 0x7, 0x40d840),
    (0x40daa7, 0x9, 0x40d840),
    (0x40dab0, 0x8, 0x40d840),
    (0x40dab8, 0x5, 0x40d840),
    (0x40dabd, 0xc, 0x40d840),
    (0x40dac9, 0xf, 0x40d840),
    (0x40dad8, 0x5, 0x40d840),
    (0x40dadd, 0x3, 0x40d840),
    (0x40dae0, 0xc, 0x40d840),
    (0x40daec, 0x9, 0x40d840),
    (0x40daf5, 0xe, 0x40d840),
    (0x40db08, 0x16, 0x40d840),
    (0x40db1e, 0x44, 0x40d840),
    (0x40db62, 0x32, 0x40d840),
    (0x40db94, 0xa, 0x40d840),
    (0x40db9e, 0xa, 0x40d840),
    (0x40dba8, 0x5, 0x40d840),
    (0x40dbad, 0xb, 0x40d840),
    (0x40dbb8, 0x43, 0x40d840),
    (0x40dbfb, 0x31, 0x40d840),
    (0x40dc2c, 0xc, 0x40d840),
    (0x40dc38, 0x14, 0x40d840),
    (0x40dc50, 0x5, 0x40d840),
    (0x40dc55, 0x9, 0x40d840),
    (0x40dc5e, 0x5, 0x40d840),
    (0x40dc63, 0x4, 0x40d840),
    (0x40dc67, 0x9, 0x40d840),
    (0x40dc70, 0x10, 0x40d840),
    (0x40dc80, 0x9, 0x40d840),
    (0x40dc89, 0x13, 0x40d840),
    (0x40dc9c, 0xc, 0x40d840),
    (0x40dca8, 0xd, 0x40d840),
    (0x40dcb5, 0x5, 0x40d840),
    (0x40dcba, 0x4, 0x40d840),
    (0x40dcbe, 0x9, 0x40d840),
    (0x40dcc7, 0x5, 0x40d840),
    (0x40dccc, 0x5, 0x40d840),
    (0x40dcd1, 0x5, 0x40d840),
    (0x40dcd6, 0x4, 0x40d840),
    (0x40dcda, 0x9, 0x40d840),
    (0x40dce8, 0xf, 0x40d840),
    (0x40dcf7, 0x9, 0x40d840),
    (0x40dd00, 0xd, 0x40d840),
    (0x40dd10, 0xa, 0x40d840),
    (0x40dd1a, 0xe, 0x40d840),
    (0x40dd28, 0xc, 0x40d840),
    (0x40dd34, 0x18, 0x40d840),
    (0x40dd4c, 0x13, 0x40d840),
    (0x40dd60, 0x9, 0x40d840),
    (0x40dd69, 0x5, 0x40d840),
    (0x40dd6e, 0x30, 0x40d840),
    (0x40de19, 0x3b, 0x40d840),
    (0x40de54, 0x4d, 0x40d840),
    (0x40dea1, 0x9, 0x40d840),
    (0x40deaa, 0x7, 0x40d840),
    (0x40deb8, 0x7, 0x40d840),
    (0x40debf, 0xd, 0x40d840),
    (0x40decc, 0xf, 0x40d840),
    (0x40df16, 0x5, 0x40d840),
]

cbs_amd64_ls_410300 = [
    (0x4101a0, 0x37, 0x4101a0),
    (0x4101d7, 0x1a, 0x4101a0),
    (0x4101f1, 0x54, 0x4101a0),
    (0x410300, 0x8, 0x4101a0),
    (0x410660, 0x1c, 0x4101a0),
    (0x402600, 0x6, 0x4101a0),
    (0x410245, 0x32, 0x4101a0),
    (0x410277, 0x5c, 0x4101a0),
    (0x4102d3, 0x29, 0x4101a0),
    (0x410310, 0x37, 0x4101a0),
    (0x410350, 0x5, 0x4101a0),
    (0x410358, 0x51, 0x4101a0),
    (0x4103b0, 0x58, 0x4101a0),
    (0x410410, 0x5f, 0x4101a0),
    (0x410470, 0x44, 0x4101a0),
    (0x4104b8, 0x4d, 0x4101a0),
    (0x410508, 0x60, 0x4101a0),
    (0x410570, 0x70, 0x4101a0),
    (0x4105e0, 0x80, 0x4101a0),
]

cbs_i386_static_805edc0 = [
    (0x805edc0, 0x20, 0x805edc0),
    (0x805ede0, 0x12, 0x805edc0),
    (0x805ee90, 0x11, 0x805edc0),
    (0x805eea1, 0x1f, 0x805edc0),
    (0x805eec0, 0x8, 0x805edc0),
    (0x805eec8, 0x12, 0x805edc0),
    (0x805eee0, 0x21, 0x805edc0),
    (0x805ef01, 0x6, 0x805edc0),
    (0x805ef07, 0x35, 0x805edc0),
    (0x805ef3c, 0x35, 0x805edc0),
    (0x805ef71, 0x19, 0x805edc0),
    (0x805ef90, 0xe, 0x805edc0),
    (0x805ef9e, 0x5, 0x805edc0),
    (0x805efa3, 0xb, 0x805edc0),
    (0x805efae, 0x2, 0x805edc0),
    (0x805efb0, 0x7, 0x805edc0),
    (0x805ee00, 0x3, 0x805edc0),
    (0x805ee03, 0x3, 0x805edc0),
    (0x805ee06, 0x3, 0x805edc0),
    (0x805ee09, 0x3, 0x805edc0),
    (0x805ee0c, 0x3, 0x805edc0),
    (0x805ee0f, 0x3, 0x805edc0),
    (0x805ee12, 0x3, 0x805edc0),
    (0x805ee15, 0x6, 0x805edc0),
    (0x805ee20, 0x3, 0x805edc0),
    (0x805ee23, 0x3, 0x805edc0),
    (0x805ee26, 0x3, 0x805edc0),
    (0x805ee29, 0x3, 0x805edc0),
    (0x805ee2c, 0x3, 0x805edc0),
    (0x805ee2f, 0x3, 0x805edc0),
    (0x805ee32, 0x3, 0x805edc0),
    (0x805ee35, 0x9, 0x805edc0),
    (0x805ee40, 0x3, 0x805edc0),
    (0x805ee43, 0x3, 0x805edc0),
    (0x805ee46, 0x3, 0x805edc0),
    (0x805ee49, 0x3, 0x805edc0),
    (0x805ee4c, 0x3, 0x805edc0),
    (0x805ee4f, 0x3, 0x805edc0),
    (0x805ee52, 0x3, 0x805edc0),
    (0x805ee55, 0xa, 0x805edc0),
    (0x805ee60, 0x3, 0x805edc0),
    (0x805ee63, 0x3, 0x805edc0),
    (0x805ee66, 0x3, 0x805edc0),
    (0x805ee69, 0x3, 0x805edc0),
    (0x805ee6c, 0x3, 0x805edc0),
    (0x805ee6f, 0x3, 0x805edc0),
    (0x805ee72, 0x3, 0x805edc0),
    (0x805ee75, 0xd, 0x805edc0),
    (0x805efc0, 0x5, 0x805edc0),
    (0x805efc5, 0x5, 0x805edc0),
    (0x805efca, 0x5, 0x805edc0),
    (0x805efcf, 0x5, 0x805edc0),
    (0x805efd4, 0x5, 0x805edc0),
    (0x805efd9, 0x10, 0x805edc0),
    (0x805eff0, 0x5, 0x805edc0),
    (0x805eff5, 0x5, 0x805edc0),
    (0x805effa, 0x5, 0x805edc0),
    (0x805efff, 0x5, 0x805edc0),
    (0x805f004, 0x5, 0x805edc0),
    (0x805f009, 0x13, 0x805edc0),
    (0x805f020, 0x5, 0x805edc0),
    (0x805f025, 0x5, 0x805edc0),
    (0x805f02a, 0x5, 0x805edc0),
    (0x805f02f, 0x5, 0x805edc0),
    (0x805f034, 0x5, 0x805edc0),
    (0x805f039, 0x14, 0x805edc0),
    (0x805f050, 0x5, 0x805edc0),
    (0x805f055, 0x5, 0x805edc0),
    (0x805f05a, 0x5, 0x805edc0),
    (0x805f05f, 0x5, 0x805edc0),
    (0x805f064, 0x5, 0x805edc0),
    (0x805f069, 0x17, 0x805edc0),
    (0x805f080, 0x5, 0x805edc0),
    (0x805f085, 0x5, 0x805edc0),
    (0x805f08a, 0x5, 0x805edc0),
    (0x805f08f, 0x5, 0x805edc0),
    (0x805f094, 0x5, 0x805edc0),
    (0x805f099, 0x13, 0x805edc0),
    (0x805f0b0, 0x5, 0x805edc0),
    (0x805f0b5, 0x5, 0x805edc0),
    (0x805f0ba, 0x5, 0x805edc0),
    (0x805f0bf, 0x5, 0x805edc0),
    (0x805f0c4, 0x5, 0x805edc0),
    (0x805f0c9, 0x16, 0x805edc0),
    (0x805f0e0, 0x5, 0x805edc0),
    (0x805f0e5, 0x5, 0x805edc0),
    (0x805f0ea, 0x5, 0x805edc0),
    (0x805f0ef, 0x5, 0x805edc0),
    (0x805f0f4, 0x5, 0x805edc0),
    (0x805f0f9, 0x17, 0x805edc0),
    (0x805f110, 0x5, 0x805edc0),
    (0x805f115, 0x5, 0x805edc0),
    (0x805f11a, 0x5, 0x805edc0),
    (0x805f11f, 0x5, 0x805edc0),
    (0x805f124, 0x5, 0x805edc0),
    (0x805f129, 0x1a, 0x805edc0),
    (0x805f150, 0x5, 0x805edc0),
    (0x805f155, 0x5, 0x805edc0),
    (0x805f15a, 0x5, 0x805edc0),
    (0x805f15f, 0x5, 0x805edc0),
    (0x805f164, 0x5, 0x805edc0),
    (0x805f169, 0x15, 0x805edc0),
    (0x805f180, 0x5, 0x805edc0),
    (0x805f185, 0x5, 0x805edc0),
    (0x805f18a, 0x5, 0x805edc0),
    (0x805f18f, 0x5, 0x805edc0),
    (0x805f194, 0x5, 0x805edc0),
    (0x805f199, 0x18, 0x805edc0),
    (0x805f1c0, 0x5, 0x805edc0),
    (0x805f1c5, 0x5, 0x805edc0),
    (0x805f1ca, 0x5, 0x805edc0),
    (0x805f1cf, 0x5, 0x805edc0),
    (0x805f1d4, 0x5, 0x805edc0),
    (0x805f1d9, 0x19, 0x805edc0),
    (0x805f200, 0x5, 0x805edc0),
    (0x805f205, 0x5, 0x805edc0),
    (0x805f20a, 0x5, 0x805edc0),
    (0x805f20f, 0x5, 0x805edc0),
    (0x805f214, 0x5, 0x805edc0),
    (0x805f219, 0x1c, 0x805edc0),
    (0x805f240, 0x5, 0x805edc0),
    (0x805f245, 0x5, 0x805edc0),
    (0x805f24a, 0x5, 0x805edc0),
    (0x805f24f, 0x5, 0x805edc0),
    (0x805f254, 0x5, 0x805edc0),
    (0x805f259, 0x18, 0x805edc0),
    (0x805f280, 0x5, 0x805edc0),
    (0x805f285, 0x5, 0x805edc0),
    (0x805f28a, 0x5, 0x805edc0),
    (0x805f28f, 0x5, 0x805edc0),
    (0x805f294, 0x5, 0x805edc0),
    (0x805f299, 0x1b, 0x805edc0),
    (0x805f2c0, 0x5, 0x805edc0),
    (0x805f2c5, 0x5, 0x805edc0),
    (0x805f2ca, 0x5, 0x805edc0),
    (0x805f2cf, 0x5, 0x805edc0),
    (0x805f2d4, 0x5, 0x805edc0),
    (0x805f2d9, 0x1c, 0x805edc0),
    (0x805f300, 0x5, 0x805edc0),
    (0x805f305, 0x5, 0x805edc0),
    (0x805f30a, 0x5, 0x805edc0),
    (0x805f30f, 0x5, 0x805edc0),
    (0x805f314, 0x5, 0x805edc0),
    (0x805f319, 0x1f, 0x805edc0),
]

cbs_i386_static_80621a0 = [
    (0x80621a0, 0x15, 0x80621a0),
    (0x80621b5, 0xd, 0x80621a0),
    (0x80621c2, 0x6, 0x80621a0),
    (0x80621c8, 0x14, 0x80621a0),
    (0x80621e0, 0x6, 0x80621a0),
    (0x80621e6, 0x8, 0x80621a0),
    (0x80621ee, 0x5, 0x80621a0),
    (0x80621f3, 0x8, 0x80621a0),
    (0x80621fb, 0x5, 0x80621a0),
    (0x8062200, 0x8, 0x80621a0),
    (0x8062208, 0x5, 0x80621a0),
    (0x806220d, 0x8, 0x80621a0),
    (0x8062215, 0x5, 0x80621a0),
    (0x806221a, 0x8, 0x80621a0),
    (0x8062222, 0x5, 0x80621a0),
    (0x8062227, 0x8, 0x80621a0),
    (0x806222f, 0x8, 0x80621a0),
    (0x8062237, 0x2, 0x80621a0),
    (0x8062239, 0x1, 0x80621a0),
    (0x8062240, 0x4, 0x80621a0),
    (0x8062250, 0x2, 0x80621a0),
    (0x8062252, 0x9, 0x80621a0),
    (0x8062260, 0x3, 0x80621a0),
    (0x8062270, 0xb, 0x80621a0),
    (0x806227b, 0x13, 0x80621a0),
    (0x806228e, 0x15, 0x80621a0),
    (0x80622a3, 0x15, 0x80621a0),
    (0x80622b8, 0x15, 0x80621a0),
    (0x80622cd, 0x8, 0x80621a0),
    (0x80622d5, 0x16, 0x80621a0),
    (0x80622f0, 0x3, 0x80621a0),
    (0x80622f3, 0x3, 0x80621a0),
    (0x80622f6, 0x3, 0x80621a0),
    (0x80622f9, 0x15, 0x80621a0),
    (0x806230e, 0xe, 0x80621a0),
    (0x806231c, 0xe, 0x80621a0),
    (0x806232a, 0x13, 0x80621a0),
    (0x806233d, 0x2, 0x80621a0),
    (0x80623a6, 0xe, 0x80621a0),
    (0x80623b4, 0x12, 0x80621a0),
    (0x80623c6, 0x2, 0x80621a0),
    (0x8062436, 0xe, 0x80621a0),
    (0x8062444, 0x10, 0x80621a0),
    (0x8062454, 0xd, 0x80621a0),
    (0x8062461, 0x2, 0x80621a0),
    (0x80624d6, 0xe, 0x80621a0),
    (0x80624e4, 0x10, 0x80621a0),
    (0x80624f4, 0x9, 0x80621a0),
    (0x80624fd, 0x12, 0x80621a0),
    (0x806250f, 0x2, 0x80621a0),
    (0x8062d00, 0x4, 0x80621a0),
    (0x8062d04, 0x5, 0x80621a0),
    (0x8062d09, 0xa, 0x80621a0),
    (0x8062d13, 0x3, 0x80621a0),
    (0x8062d16, 0x8, 0x80621a0),
    (0x8062d1e, 0x2, 0x80621a0),
    (0x8062d20, 0x1, 0x80621a0),
]


cbs_i386_static_808dc28 = [
    (0x808dac8, 0x18, 0x808dc28),
    (0x808dae0, 0xb, 0x808dc28),
    (0x808daeb, 0x16, 0x808dc28),
    (0x808db01, 0xb, 0x808dc28),
    (0x808dc28, 0xf, 0x808dc28),
    (0x808dc40, 0x17, 0x808dc28),
    (0x808dc57, 0x9, 0x808dc28),
    (0x808dc60, 0x8, 0x808dc28),
    (0x808dc68, 0x15, 0x808dc28),
    (0x808dc80, 0xd, 0x808dc28),
    (0x808dc8d, 0x9, 0x808dc28),
    (0x808dca0, 0x7, 0x808dc28),
    (0x808dca7, 0x9, 0x808dc28),
    (0x808dcb0, 0x20, 0x808dc28),
    (0x808dcd0, 0x10, 0x808dc28),
    (0x808dce0, 0x5, 0x808dc28),
    (0x808dce5, 0x10, 0x808dc28),
    (0x808dcf8, 0x9, 0x808dc28),
    (0x808dd08, 0x9, 0x808dc28),
    (0x808dd18, 0x10, 0x808dc28),
    (0x808dd28, 0x11, 0x808dc28),
    (0x808dd40, 0xc, 0x808dc28),
    (0x808dd70, 0xc, 0x808dc28),
    (0x808dd7c, 0x11, 0x808dc28),
    (0x808dd8d, 0x20, 0x808dc28),
    (0x808ddad, 0xa, 0x808dc28),
    (0x808ddb7, 0x8, 0x808dc28),
    (0x808ddbf, 0x5, 0x808dc28),
    (0x808de70, 0x5, 0x808dc28),
    (0x808de75, 0x9, 0x808dc28),
    (0x808de7e, 0xc, 0x808dc28),
    (0x808de8a, 0xc, 0x808dc28),
    (0x808df08, 0x11, 0x808dc28),
]


cbs_i386_static_80a6920 = [
    (0x80a6920, 0x18, 0x80a6920),
    (0x80a6938, 0x10, 0x80a6920),
    (0x80a6948, 0xe, 0x80a6920),
    (0x80a6990, 0x1b, 0x80a6920),
    (0x80a6a6b, 0x5, 0x80a6920),
    (0x80a6960, 0x5, 0x80a6920),
    (0x80a6965, 0x4, 0x80a6920),
    (0x80a6969, 0x11, 0x80a6920),
    (0x80a697a, 0x2, 0x80a6920),
    (0x80a697c, 0x10, 0x80a6920),
    (0x80a69b0, 0x7, 0x80a6920),
    (0x80a69c0, 0x8, 0x80a6920),
    (0x80a69d0, 0x10, 0x80a6920),
    (0x80a69e0, 0x17, 0x80a6920),
    (0x80a69f7, 0x13, 0x80a6920),
    (0x80a6a0a, 0xb, 0x80a6920),
    (0x80a6a15, 0xe, 0x80a6920),
    (0x80a6a28, 0x10, 0x80a6920),
    (0x80a6a38, 0x17, 0x80a6920),
    (0x80a6a4f, 0xb, 0x80a6920),
    (0x80a6a60, 0xb, 0x80a6920),
]

cbs_i386_ld_2002160 = [
    (0x2001dd0, 74, 0x2001dd0),
    (0x2001e1a, 10, 0x2001dd0),
    (0x2001e28, 13, 0x2001dd0),
    (0x2001e35, 14, 0x2001dd0),
    (0x2001e43, 10, 0x2001dd0),
    (0x2001e4d, 5, 0x2001dd0),
    (0x2001e52, 7, 0x2001dd0),
    (0x2001e59, 10, 0x2001dd0),
    (0x2001e63, 3, 0x2001dd0),
    (0x2001e66, 4, 0x2001dd0),
    (0x2001e6a, 10, 0x2001dd0),
    (0x2001e74, 3, 0x2001dd0),
    (0x2001e77, 10, 0x2001dd0),
    (0x2001e81, 3, 0x2001dd0),
    (0x2001e84, 10, 0x2001dd0),
    (0x2001e8e, 3, 0x2001dd0),
    (0x2001e91, 10, 0x2001dd0),
    (0x2001e9b, 3, 0x2001dd0),
    (0x2001e9e, 10, 0x2001dd0),
    (0x2001ea8, 3, 0x2001dd0),
    (0x2001eab, 10, 0x2001dd0),
    (0x2001eb5, 3, 0x2001dd0),
    (0x2001eb8, 10, 0x2001dd0),
    (0x2001ec2, 3, 0x2001dd0),
    (0x2001ec5, 10, 0x2001dd0),
    (0x2001ecf, 3, 0x2001dd0),
    (0x2001ed2, 10, 0x2001dd0),
    (0x2001edc, 10, 0x2001dd0),
    (0x2001ee6, 10, 0x2001dd0),
    (0x2001ef0, 16, 0x2001dd0),
    (0x2001f00, 10, 0x2001dd0),
    (0x2001f0a, 13, 0x2001dd0),
    (0x2001f17, 10, 0x2001dd0),
    (0x2001f21, 13, 0x2001dd0),
    (0x2001f2e, 14, 0x2001dd0),
    (0x2001f3c, 14, 0x2001dd0),
    (0x2001f4a, 8, 0x2001dd0),
    (0x2001f52, 14, 0x2001dd0),
    (0x2001f60, 101, 0x2001dd0),
    #(0x2001fd7, 10, 0x2001dd0),
    (0x2001fe8, 12, 0x2001dd0),
    (0x2001ff4, 19, 0x2001dd0),
    (0x2002010, 16, 0x2001dd0),
    (0x2002020, 19, 0x2001dd0),
    (0x2002038, 8, 0x2001dd0),
    (0x2002040, 35, 0x2001dd0),
    (0x2002063, 9, 0x2001dd0),
    (0x200206c, 4, 0x2001dd0),
    (0x2002070, 44, 0x2001dd0),
    (0x200209c, 20, 0x2001dd0),
    (0x20020b0, 14, 0x2001dd0),
    (0x20020be, 9, 0x2001dd0),
    (0x20020c7, 3, 0x2001dd0),
    (0x20020ca, 14, 0x2001dd0),
    (0x20020d8, 21, 0x2001dd0),
    (0x20020ed, 11, 0x2001dd0),
    (0x20020f8, 34, 0x2001dd0),
    (0x200211a, 33, 0x2001dd0),
    (0x200213b, 17, 0x2001dd0),
    (0x200214c, 11, 0x2001dd0),
    (0x2002157, 12, 0x2001dd0),
    (0x2002168, 16, 0x2001dd0),
    (0x2002178, 21, 0x2001dd0),
    (0x20021a0, 12, 0x2001dd0),
    (0x20021ac, 5, 0x2001dd0),
    (0x2002240, 8, 0x2001dd0),
    (0x2002248, 19, 0x2001dd0),
    (0x2002260, 11, 0x2001dd0),
    (0x200226b, 19, 0x2001dd0),
    (0x2002280, 28, 0x2001dd0),
    #(0x200229c, 33, 0x2001dd0),
    (0x20022bd, 31, 0x2001dd0),
    (0x20022dc, 31, 0x2001dd0),
    (0x20022fb, 28, 0x2001dd0),
    (0x2002317, 31, 0x2001dd0),
    (0x2002336, 31, 0x2001dd0),
    (0x2002355, 31, 0x2001dd0),
    (0x2002374, 31, 0x2001dd0),
    (0x2002393, 28, 0x2001dd0),
    (0x2002190, 16, 0x2001dd0),
    (0x20021b8, 17, 0x2001dd0),
    (0x20021c9, 5, 0x2001dd0),
    (0x20021d0, 32, 0x2001dd0),
    (0x20021f0, 5, 0x2001dd0),
    (0x20021f8, 25, 0x2001dd0),
    (0x2002211, 5, 0x2001dd0),
    (0x2002220, 18, 0x2001dd0),
    (0x2002232, 5, 0x2001dd0),
]

cbs_amd64_chown_2004050 = [
    (0x2004050, 128, 0x2004050),
    (0x20040d0, 19, 0x2004050),
    (0x200527f, 5, 0x2004050),
    (0x20040e3, 32, 0x2004050),
    (0x2004103, 32, 0x2004050),
    (0x2004123, 21, 0x2004050),
    (0x2004138, 13, 0x2004050),
    (0x2004145, 13, 0x2004050),
    (0x2004152, 9, 0x2004050),
    (0x200415b, 30, 0x2004050),
    (0x2004179, 14, 0x2004050),
    (0x2004187, 11, 0x2004050),
    (0x2004192, 6, 0x2004050),
    (0x2004198, 40, 0x2004050),
    (0x20041c0, 14, 0x2004050),
    (0x20041ce, 57, 0x2004050),
    (0x2004207, 17, 0x2004050),
    (0x2004218, 8, 0x2004050),
    (0x2004220, 13, 0x2004050),
    (0x200422d, 19, 0x2004050),
    (0x2004278, 11, 0x2004050),
    (0x2004283, 8, 0x2004050),
    (0x200428b, 10, 0x2004050),
    (0x2004295, 20, 0x2004050),
    (0x20042a9, 9, 0x2004050),
    (0x20042b2, 10, 0x2004050),
    (0x20042bc, 4, 0x2004050),
    (0x20042c0, 5, 0x2004050),
    (0x20042c5, 5, 0x2004050),
    (0x20042ca, 9, 0x2004050),
    (0x20042d3, 6, 0x2004050),
    (0x20042d9, 7, 0x2004050),
    (0x20042e0, 5, 0x2004050),
    (0x20042e5, 4, 0x2004050),
    (0x20042e9, 30, 0x2004050),
    (0x2004392, 11, 0x2004050),
    (0x200439d, 5, 0x2004050),
    (0x20043f3, 21, 0x2004050),
    (0x2004408, 10, 0x2004050),
    (0x2004412, 5, 0x2004050),
    (0x2004417, 5, 0x2004050),
    (0x200441c, 9, 0x2004050),
    (0x2004425, 6, 0x2004050),
    (0x200442b, 9, 0x2004050),
    (0x2004434, 6, 0x2004050),
    (0x200443a, 7, 0x2004050),
    (0x2004441, 5, 0x2004050),
    (0x2004446, 5, 0x2004050),
    (0x200444b, 13, 0x2004050),
    (0x2004540, 8, 0x2004050),
    (0x2004548, 12, 0x2004050),
    (0x2004554, 67, 0x2004050),
    (0x2004597, 8, 0x2004050),
    (0x200459f, 13, 0x2004050),
    (0x20045b0, 9, 0x2004050),
    (0x20045b9, 19, 0x2004050),
    (0x2004780, 8, 0x2004050),
    (0x20047b0, 37, 0x2004050),
    (0x20047d5, 13, 0x2004050),
    (0x20047e2, 11, 0x2004050),
    (0x20047ed, 11, 0x2004050),
    (0x20047f8, 23, 0x2004050),
    (0x200480f, 5, 0x2004050),
    (0x2004814, 53, 0x2004050),
    (0x20048f8, 31, 0x2004050),
    (0x2004917, 45, 0x2004050),
    (0x2004944, 63, 0x2004050),
    (0x2004983, 48, 0x2004050),
    (0x20049b3, 10, 0x2004050),
    (0x20049bd, 10, 0x2004050),
    (0x20049c7, 18, 0x2004050),
    (0x20049d9, 40, 0x2004050),
    (0x2004a01, 53, 0x2004050),
    (0x2004a36, 10, 0x2004050),
    (0x2004a40, 24, 0x2004050),
    (0x2004a60, 17, 0x2004050),
    (0x2004a71, 11, 0x2004050),
    (0x2004a7c, 5, 0x2004050),
    (0x2004a81, 5, 0x2004050),
    (0x2004a86, 9, 0x2004050),
    (0x2004a8f, 6, 0x2004050),
    (0x2004a95, 9, 0x2004050),
    (0x2004a9e, 6, 0x2004050),
    (0x2004aa4, 7, 0x2004050),
    (0x2004aab, 5, 0x2004050),
    (0x2004ab0, 5, 0x2004050),
    (0x2004ab5, 9, 0x2004050),
    (0x2004abe, 13, 0x2004050),
    (0x2004acb, 9, 0x2004050),
    (0x2004ad4, 16, 0x2004050),
    (0x2004ae4, 23, 0x2004050),
    (0x2004afb, 2, 0x2004050),
    (0x2004afd, 5, 0x2004050),
    (0x2004b02, 4, 0x2004050),
    (0x2004b06, 8, 0x2004050),
    (0x2004b0e, 8, 0x2004050),
    (0x2004b16, 13, 0x2004050),
    (0x2004b23, 5, 0x2004050),
    (0x2004b28, 5, 0x2004050),
    (0x2004b2d, 4, 0x2004050),
    (0x2004b31, 13, 0x2004050),
    (0x2004b3e, 8, 0x2004050),
    (0x2004b46, 5, 0x2004050),
    (0x2004b4b, 5, 0x2004050),
    (0x2004b50, 9, 0x2004050),
    (0x2004b59, 6, 0x2004050),
    (0x2004b5f, 12, 0x2004050),
    (0x2004b70, 10, 0x2004050),
    (0x2004b7a, 20, 0x2004050),
    (0x2004b90, 13, 0x2004050),
    (0x2004b9d, 11, 0x2004050),
    (0x2004ba8, 26, 0x2004050),
    (0x2004bc2, 23, 0x2004050),
    (0x2004bd9, 7, 0x2004050),
    (0x2004be0, 16, 0x2004050),
    (0x2004bf0, 57, 0x2004050),
    (0x2004c29, 26, 0x2004050),
    (0x2004c43, 18, 0x2004050),
    (0x2004c58, 8, 0x2004050),
    (0x2004cb4, 8, 0x2004050),
    (0x2004cbc, 5, 0x2004050),
    (0x2004cc1, 9, 0x2004050),
    (0x2004cca, 67, 0x2004050),
    (0x2004d10, 8, 0x2004050),
    (0x2004d18, 9, 0x2004050),
    (0x2004d21, 26, 0x2004050),
    (0x2004d3b, 42, 0x2004050),
    (0x2004d65, 69, 0x2004050),
    (0x2004daa, 58, 0x2004050),
    (0x2004de4, 7, 0x2004050),
    (0x2004deb, 43, 0x2004050),
    (0x2004e16, 43, 0x2004050),
    (0x2004e41, 11, 0x2004050),
    (0x2004e4c, 62, 0x2004050),
    (0x2004e8a, 58, 0x2004050),
    (0x2004ec4, 37, 0x2004050),
    (0x2004f20, 17, 0x2004050),
    (0x2004f31, 13, 0x2004050),
    (0x2004f3e, 8, 0x2004050),
    (0x2004f46, 27, 0x2004050),
    (0x2004f61, 65, 0x2004050),
    (0x2004fa2, 5, 0x2004050),
    (0x2004fa7, 2, 0x2004050),
    (0x2004fb0, 7, 0x2004050),
    (0x2004fb7, 14, 0x2004050),
    (0x2004fc5, 3, 0x2004050),
    (0x2004fc8, 13, 0x2004050),
    (0x2004fd5, 56, 0x2004050),
    (0x200500d, 16, 0x2004050),
    (0x200501d, 3, 0x2004050),
    (0x2005020, 5, 0x2004050),
    (0x2005025, 4, 0x2004050),
    (0x2005029, 13, 0x2004050),
    (0x2005036, 5, 0x2004050),
    (0x200503b, 7, 0x2004050),
    (0x20050b8, 8, 0x2004050),
    (0x20050c0, 61, 0x2004050),
    (0x200510d, 61, 0x2004050),
    (0x200514a, 2, 0x2004050),
    (0x200514c, 10, 0x2004050),
    (0x2005156, 4, 0x2004050),
    (0x200515a, 10, 0x2004050),
    (0x2005164, 6, 0x2004050),
    (0x200516a, 5, 0x2004050),
    (0x200516f, 4, 0x2004050),
    (0x2005173, 12, 0x2004050),
    (0x200517f, 3, 0x2004050),
    (0x2005182, 9, 0x2004050),
    (0x200518b, 10, 0x2004050),
    (0x2005195, 25, 0x2004050),
    (0x20051ae, 22, 0x2004050),
    (0x20051c4, 22, 0x2004050),
    (0x20051ed, 63, 0x2004050),
    (0x200522c, 10, 0x2004050),
    (0x2005236, 13, 0x2004050),
    (0x2005243, 5, 0x2004050),
    (0x2005248, 10, 0x2004050),
    (0x200525a, 24, 0x2004050),
    (0x2004243, 13, 0x2004050),
    (0x2004250, 12, 0x2004050),
    (0x200425c, 16, 0x2004050),
    (0x200426c, 12, 0x2004050),
    (0x2004313, 13, 0x2004050),
    (0x2004320, 9, 0x2004050),
    (0x2004329, 21, 0x2004050),
    (0x2004343, 11, 0x2004050),
    (0x200434e, 19, 0x2004050),
    (0x2004368, 10, 0x2004050),
    (0x2004372, 18, 0x2004050),
    (0x2004384, 7, 0x2004050),
    (0x200438b, 7, 0x2004050),
    (0x20043a8, 12, 0x2004050),
    (0x20043b8, 11, 0x2004050),
    (0x20043c3, 11, 0x2004050),
    (0x20043ce, 11, 0x2004050),
    (0x20043d9, 12, 0x2004050),
    (0x20043e5, 11, 0x2004050),
    (0x20043f0, 3, 0x2004050),
    (0x2004460, 11, 0x2004050),
    (0x200446b, 13, 0x2004050),
    (0x2004478, 9, 0x2004050),
    (0x2004488, 15, 0x2004050),
    (0x20044a0, 15, 0x2004050),
    (0x20044b0, 8, 0x2004050),
    (0x20044b8, 9, 0x2004050),
    (0x20044c1, 13, 0x2004050),
    (0x20044ce, 16, 0x2004050),
    (0x20044e0, 13, 0x2004050),
    (0x20044f0, 15, 0x2004050),
    (0x2004500, 15, 0x2004050),
    (0x2004510, 15, 0x2004050),
    (0x2004520, 15, 0x2004050),
    (0x2004530, 15, 0x2004050),
    (0x20045e0, 15, 0x2004050),
    (0x2004633, 19, 0x2004050),
    (0x2004646, 13, 0x2004050),
    (0x2004653, 5, 0x2004050),
    (0x2004658, 5, 0x2004050),
    (0x200465d, 9, 0x2004050),
    (0x2004666, 6, 0x2004050),
    (0x200466c, 13, 0x2004050),
    (0x2004679, 7, 0x2004050),
    (0x2004680, 5, 0x2004050),
    (0x2004685, 5, 0x2004050),
    (0x200468a, 15, 0x2004050),
    (0x2004699, 9, 0x2004050),
    (0x20046a2, 25, 0x2004050),
    (0x20046bb, 16, 0x2004050),
    (0x2004790, 11, 0x2004050),
    (0x200479b, 20, 0x2004050),
    (0x2004850, 11, 0x2004050),
    (0x200485b, 9, 0x2004050),
    (0x2004864, 14, 0x2004050),
    (0x2004872, 32, 0x2004050),
    (0x2004898, 7, 0x2004050),
    (0x200489f, 9, 0x2004050),
    (0x20048a8, 16, 0x2004050),
    (0x20048b8, 8, 0x2004050),
    (0x20048c0, 17, 0x2004050),
    (0x20048d8, 11, 0x2004050),
    (0x20048e3, 15, 0x2004050),
    (0x2004c60, 5, 0x2004050),
    (0x2004c65, 5, 0x2004050),
    (0x2004c6a, 9, 0x2004050),
    (0x2004c73, 6, 0x2004050),
    (0x2004c79, 13, 0x2004050),
    (0x2004c86, 19, 0x2004050),
    (0x2004c99, 16, 0x2004050),
    (0x2004ca9, 11, 0x2004050),
    (0x2004ee9, 5, 0x2004050),
    (0x2004eee, 5, 0x2004050),
    (0x2004ef3, 9, 0x2004050),
    (0x2004efc, 6, 0x2004050),
    (0x2004f02, 20, 0x2004050),
    (0x2005042, 13, 0x2004050),
    (0x200504f, 20, 0x2004050),
    (0x2005063, 11, 0x2004050),
    (0x200506e, 5, 0x2004050),
    (0x2005073, 5, 0x2004050),
    (0x2005078, 9, 0x2004050),
    (0x2005081, 6, 0x2004050),
    (0x2005087, 9, 0x2004050),
    (0x2005090, 6, 0x2004050),
    (0x2005096, 9, 0x2004050),
    (0x200509f, 6, 0x2004050),
    (0x20050a5, 19, 0x2004050),
    (0x2005100, 13, 0x2004050),
    (0x20051da, 19, 0x2004050),
    (0x2005252, 8, 0x2004050),
    (0x2005272, 13, 0x2004050),
    (0x2004240, 3, 0x2004050),
    (0x2004310, 3, 0x2004050),
    (0x2004340, 3, 0x2004050),
    (0x20045d0, 14, 0x2004050),
    (0x20045de, 2, 0x2004050),
    (0x20045f0, 18, 0x2004050),
    (0x2004608, 18, 0x2004050),
    (0x2004620, 10, 0x2004050),
    (0x2004630, 3, 0x2004050),
    (0x20046d0, 21, 0x2004050),
    (0x20046e8, 16, 0x2004050),
    (0x2004700, 19, 0x2004050),
    (0x2004718, 24, 0x2004050),
    (0x2004730, 18, 0x2004050),
    (0x2004748, 10, 0x2004050),
    (0x2004758, 18, 0x2004050),
    (0x2004770, 10, 0x2004050),
]

ld_switches = [0x200704d,
               0x200d274,
               0x200d62d,
               0x200d6e0,
               0x200d8ac,
               0x200dd56,
               0x200e0e1,
               0x200e19c,
               0x200e371,
               0x201b438,
               0x201dea8,
               0x201e0fe,
               0x2002160,
               0x2002d80,
               0x201a579,
               #0x201ad11,  # lost this one to performance timing
               ]


