/* This file is part of the Chakra project

   Copyright (C) 2012 Lukas Appelhans <boom1992@chakra-project.org>
   Copyright (C) 2011 Lisa Vitolo <shainer@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/
#ifndef POLKITSQLITECONNECTION_H
#define POLKITSQLITECONNECTION_H

#include <QVariantMap>
#include <QString>
#include <SQLiteConnection.h>

/**
 * @class SQLiteConnection
 * @brief manages a connection with a database using the sqlite C library
 *
 * This class is not thread-safe.
 */
class PolKitSQLiteConnection : public AbstractSQLiteConnection
{
    typedef QVariant Value;
    typedef QVariantMap Row;

public:
    /**
     * Constructs an object and establishes the connection
     *
     * @param name the database filename
     * @param readonly true if read-only access, false for read-write
     */
    PolKitSQLiteConnection(QString const& name, bool readonly);

    /**
     * Constructs an object without establishing any connection
     */
    PolKitSQLiteConnection();

    ~PolKitSQLiteConnection();

    /**
     * Establishes a new connection with a database
     *
     * @param name the database filename
     * @param readonly true if read-only access, false for read-write
     */
    void connectToDB(QString const&, bool) override;

    /**
     * Executes a query on the database
     *
     * @param q the query string
     * @returns a SQLiteResource object with the result, if present
     */
    SQLiteResource query(const QString &q) override;

    /**
     * Executes multiple queries on the database
     *
     * @param q the query
     * @param bindings the bindings we're going to use
     *                 this is pair of values, the key is a string determining the binding name, the value is a stringlist which contains all the values
     */
    void query(const QString &q, const QVariantMap &bindings) override;

    /**
     * If the table interested by the last query as an autoincremented primary key,
     * gets the last value assigned to the key
     */
    qint64 getLastRowId() override;

    /**
     * Creates a "macro" (fake name) that maps into an integer, a QString, or a QByteArray
     * The next query can contain the defined macros in place of the actual values.
     * @note All bindings stay valid only for the subsequent call of query().
     */
    void bind(QString const&, int) override;
    void bind(QString const&, QString const&) override;
    void bind(QString const&, QByteArray const&) override;

private:
    // Disable copy constructor and default constructor
    PolKitSQLiteConnection(PolKitSQLiteConnection const&);

    class Private;
    Private *d;
};

#endif
