/*
   This file is part of the Chakra project

   Copyright (C) 2010 Dario Freddi <drf@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/

#ifndef AKABEI_AKABEIDELTA_H
#define AKABEI_AKABEIDELTA_H

#include <QObject>

namespace Akabei {

class DeltaPrivate;

/**
 * \class Delta  akabeidelta.h "akabeidelta.h"
 *
 * \brief This class describes a binary patch we use to upgrade from one package version to another.
 *
 * Akabei Deltas can either be loaded from the \c Database or from a local file (\c Backend). If both the
 * path to the .delta.tar.xz-file and the original package are set, it's just a matter of running the command
 * generated by \c xdeltaApplyCommand.
 * @see setPathToDelta()
 * @see setPathToSource()
 * @see xdeltaApplyCommand
 *
 * This class is not thread-safe.
 */
class Delta {
    Q_DISABLE_COPY(Delta)
    Q_DECLARE_PRIVATE(Delta)
    public:
        virtual ~Delta();

        /**
         * @returns the correct md5sum of the delta
         */
        QByteArray md5sum() const;

        /**
         * @returns the name of the package the delta needs to be applied on
         */
        QString targetName() const;
        /**
         * @returns the version of the package which the delta needs to be applied on
         */
        QByteArray versionFrom() const;
        /**
         * @returns the version of the package which is generated after the delta got applied to
         * a package with \c versionFrom
         */
        QByteArray versionTo() const;

        /**
         * @returns the filename of the .delta.tar.xz-file on disk
         */
        QString filename() const;

        /**
         * Checks whether the md5sum given in the .DELTAINFO and the real md5sum match
         * @returns true when both match
         */
        bool checkMD5Sum();

        /**
         * @returns the command used for generating the package with version \c versionTo
         * @param pathToDestination the path to the package with \c versionTo, not existent at this point
         */
        QString xdeltaApplyCommand(const QString &pathToDestination) const;

        /**
         * @returns the path to the .delta.tar.xz-file
         */
        QString pathToDelta() const;
        /**
         * Set the path to the .delta.tar.xz-file
         * @param path the path of the file
         */
        void setPathToDelta(const QString &path);

        /**
         * @returns the path to the package with version \c versionFrom
         */
        QString pathToSource() const;
        /**
         * Sets the path to the package with version \c versionFrom
         * @param path the path to the package
         */
        void setPathToSource(const QString &path);

    private:
        Delta(const QString &targetName, const QByteArray &versionFrom, const QByteArray &versionTo);

        DeltaPrivate * const d_ptr;

        friend class Backend;
        friend class Database;
        friend class DatabasePrivate;
};

}

#endif // AKABEI_AKABEIDELTA_H
