/*
 * Logging, through polkit when required

   Copyright (C) 2011 Lisa Vitolo <shainer@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/

#ifndef AKABEILOG_H
#define AKABEILOG_H

#include <QString>
#include <iostream>

/*
 * This is a globally accessible variable used to signal the logging system when the line ends, so
 * it can be flushed in the log file. It must be concatenated at the end of the line.
 * 
 * Avoids having to invoke services over DBus in the AkabeiLogLine destructor, which isn't recommended.
 */
namespace Akabei
{
    extern const void* endlog;
}

typedef enum __e_AkabeiLogLevel {
    LogNormal, LogWarning, LogCritical
} AkabeiLogLevel;


/**
 * @class AkabeiLogLine
 * @brief Handles logging
 */
class AkabeiLogLine
{
public:
    AkabeiLogLine(AkabeiLogLevel);
    ~AkabeiLogLine();
    
    /*
     * Logging of different data types. You can concatenate more variables in the same logging, like
     * log() << "message" << intVar << "message2" << doubleVar;
     */
    AkabeiLogLine& operator<<(const char *);
    AkabeiLogLine& operator<<(QChar);
    AkabeiLogLine& operator<<(bool);
    AkabeiLogLine& operator<<(char);
    AkabeiLogLine& operator<<(int);
    AkabeiLogLine& operator<<(unsigned int);
    AkabeiLogLine& operator<<(short);
    AkabeiLogLine& operator<<(unsigned short);
    AkabeiLogLine& operator<<(long);
    AkabeiLogLine& operator<<(unsigned long);
    AkabeiLogLine& operator<<(qint64);
    AkabeiLogLine& operator<<(quint64);
    AkabeiLogLine& operator<<(float);
    AkabeiLogLine& operator<<(double);
    AkabeiLogLine& operator<<(const QString &);
    AkabeiLogLine& operator<<(const QByteArray &);
    AkabeiLogLine& operator<<(const void *);

    /**
     * Initializes the logging system. It must be called before logging anything.
     * 
     * @param syslog whether we'll log through syslog
     * @param filename file which we log on (required only when syslog is false).
     */
    static void initialize(bool syslog, const QString &filename = QString());
    
private:
    class Private;
    Private * d;
};

/**
 * Starts a warning log message.
 */
AkabeiLogLine logWarning();

/**
 * Starts a critical log message.
 */
AkabeiLogLine logCritical();

/**
 * Starts a normal log message.
 */
AkabeiLogLine log();

QString readLoggedActions(const QString &filter);

#endif
