/*
 * Libarchive interface: operation functors.

   Copyright (C) 2011 Lisa Vitolo <shainer@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/

#ifndef _ARCHIVEOPERATIONS_H
#define _ARCHIVEOPERATIONS_H

#include <akabeioperation.h>
#include <QMap>
#include <QStringList>
#include <QVariantMap>

class ArchiveHandler;
class PolKitArchiveHandler;
class ArchiveEntry;

namespace Akabei {

/**
 * @class EntryHandlerFunctor
 * @brief represents a generic operation to perform on archive entries.
 */
class EntryHandlerFunctor
{
public:
    EntryHandlerFunctor(Akabei::Operation *, ArchiveHandler &);
    virtual bool operator() (const ArchiveEntry &) =0;

protected:
    Akabei::Operation *m_caller;
    ArchiveHandler &m_handle;
};

/**
 * @class InstallFunctor
 * @extends EntryHandlerFunctor
 * @brief performs the operations needed on archive entries during an installation.
 */
class InstallFunctor : public EntryHandlerFunctor
{
public:
    InstallFunctor(QMap<QString, QString> &, const QStringList &, Akabei::Operation *, ArchiveHandler &);
    bool operator() (const ArchiveEntry &) override;

private:
    QMap<QString, QString> &backupFiles;
    const QStringList &ultimatelyOwned;
    qint64 partialSize;
};

/**
 * @class ReinstallUpgradeFunctor
 * @extends EntryHandlerFunctor
 * @brief perform the operations needed on archive entries during a reinstallation or an upgrade.
 */
class ReinstallUpgradeFunctor : public EntryHandlerFunctor
{
public:
    ReinstallUpgradeFunctor(QMap<QString, QString> &, const QStringList &, Akabei::Operation *, ArchiveHandler &);
    bool operator() (const ArchiveEntry &) override;

private:
    QMap<QString, QString> &backupFiles;
    const QStringList &ultimatelyOwned;
    qint64 partialSize;
};

/**
 * @class ExtractAllFunctor
 * @extends EntryHandlerFunctor
 * @brief extracts all the entries on filesystem.
 */
class ExtractAllFunctor : public EntryHandlerFunctor
{
public:
    ExtractAllFunctor(QString, bool, Akabei::Operation *, ArchiveHandler &);
    bool operator() (const ArchiveEntry &) override;

private:
    QString pathPrefix;
    bool overwrite;
};

/**
 * @class Md5Functor
 * @extends EntryHandlerFunctor
 * @brief computes the MD5 hash on a subset of the archive entries.
 */
class Md5Functor : public EntryHandlerFunctor
{
public:
    Md5Functor(QStringList, Akabei::Operation *, ArchiveHandler &);
    bool operator() (const ArchiveEntry &) override;

    QList<QByteArray> result();
    QStringList skipped();

private:
    QStringList m_toCompute;
    QList<QByteArray> m_result;
};
}

#endif