// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.

/**
 * Data that is used for testing in a web preview where data from the JavaFX is unavailable.
 *
 * TODO: should be replaced with the appropriate placeholder in production.
 *
 */
const TEST_NOTEBOOK = {
      "cells": [
        {
          "cell_type": "markdown",
          "source": "\n- [x] Mercury\n- [x] Venus\n- [x] Earth (Orbit/Moon)\n\n",
          "metadata": {
            "pycharm": {
              "metadata": false
            }
          }
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "\n",
            "First, we will show how the brain can decode a likelihood over stimulus from neural activity. Then we will ask how the brain can compute joint likelihoods.\n",
            "### How can the brain decode $p(\\mathbf{r_\\mathrm{V}}|s)$?\n",
            "\n",
            "\\begin{align}\n",
            "L(s) &= p(\\mathbf{r_\\mathrm{V}}\\ |\\ s) \\tag{1} \\\\ \n",
            "&= \\prod_{i=0}^N \\frac{e^{-g_\\mathrm{V}\\ f_i(s)}\\ g_\\mathrm{V}\\ f_i(s)^{r_{\\mathrm{V}i}}}{r_{\\mathrm{V}i}!} \\tag{2} \\\\\n",
            "&\\propto \\prod_{i=0}^N e^{-g_\\mathrm{V}\\ f_i(s)}\\ f_i(s)^{r_{\\mathrm{V}i}} \\tag{3} \\\\\n",
            "&= e^{-g_\\mathrm{V}\\sum_{i=0}^N f_i(s)} \\prod_{i=0}^N f_i(s)^{r_{\\mathrm{V}i}}\\tag{4} \\\\ \n",
            "&= e^{-g_\\mathrm{V}k} \\prod_{i=0}^N f_i(s)^{r_{\\mathrm{V}i}} \\tag{5} \\\\\n",
            "&\\propto \\prod_{i=0}^N f_i(s)^{r_{\\mathrm{V}i}} \\tag{6} \\\\\n",
            "\\end{align}\n",
            "\n",
            "\\begin{align}\n",
            "\\sqrt{37} & = \\sqrt{\\frac{73^2-1}{12^2}} \\\\\n",
            " & = \\sqrt{\\frac{73^2}{12^2}\\cdot\\frac{73^2-1}{73^2}} \\\\ \n",
            " & = \\sqrt{\\frac{73^2}{12^2}}\\sqrt{\\frac{73^2-1}{73^2}} \\\\\n",
            " & = \\frac{73}{12}\\sqrt{1 - \\frac{1}{73^2}} \\\\ \n",
            " & \\approx \\frac{73}{12}\\left(1 - \\frac{1}{2\\cdot73^2}\\right)\n",
            "\\end{align}\n"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": "In Python, objects can declare their textual representation using the `__repr__` method.  IPython expands on this idea and allows objects to declare other, rich representations including:\n\n* HTML\n* JSON\n* PNG\n* JPEG\n* SVG\n* LaTeX\n\nA single object can declare some or all of these representations; all are handled by IPython\u0027s *display system*. This Notebook shows how you can use this display system to incorporate a broad range of content into your Notebooks.\n"
        },
        {
          "cell_type": "markdown",
          "source": [
            "## References\nReferences of different types should be processed differently, e.g.:\n",
            "* External references should be opened in default browser: http://www.example.com\n",
            "* References to the same sections of the file should lead to scrolling: <a href='#Basic display imports'>Basic display imports</a>\n",
            "* References with nested elements should work well: [`line_profiler`](https://github.com/rkern/line_profiler)\n"
          ],
          "metadata": {
            "pycharm": {
              "metadata": false,
              "name": "#%% m"
            }
          }
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": "## Basic display imports"
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "The `display` function is a general purpose tool for displaying different representations of objects. Think of it as `print` for these rich representations."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 1,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "ename": "ZeroDivisionError",
              "evalue": "integer division or modulo by zero",
              "output_type": "error",
              "traceback": [
                "\u001b[1;31m\u001b[0m",
                "\u001b[1;31mZeroDivisionError\u001b[0mTraceback (most recent call last)",
                "\u001b[1;32m\u003cipython-input-3-18db0d879ab0\u003e\u001b[0m in \u001b[0;36m\u003cmodule\u003e\u001b[1;34m()\u001b[0m\n\u001b[0;32m      1\u001b[0m \u001b[1;33m\u001b[0m\u001b[0m\n\u001b[1;32m----\u003e 2\u001b[1;33m \u001b[0ma\u001b[0m \u001b[1;33m\u003d\u001b[0m \u001b[1;36m1\u001b[0m \u001b[1;33m/\u001b[0m \u001b[1;36m0\u001b[0m\u001b[1;33m\u001b[0m\u001b[0m\n\u001b[0m\u001b[0;32m      3\u001b[0m \u001b[1;33m\u001b[0m\u001b[0m\n\u001b[0;32m      4\u001b[0m \u001b[1;31m#print(a)\u001b[0m\u001b[1;33m\u001b[0m\u001b[1;33m\u001b[0m\u001b[0m\n\u001b[0;32m      5\u001b[0m \u001b[1;33m\u001b[0m\u001b[0m\n",
                "\u001b[1;31mZeroDivisionError\u001b[0m: integer division or modulo by zero"
              ]
            }
          ],
          "source": [
            "from IPython.display import display"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "A few points:\n",
            "\n",
            "* Calling `display` on an object will send **all** possible representations to the Notebook.\n",
            "* These representations are stored in the Notebook document.\n",
            "* In general the Notebook will use the richest available representation.\n",
            "\n",
            "If you want to display a particular representation, there are specific functions for that:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 2,
          "metadata": {
            "collapsed": false
          },
          "outputs": [],
          "source": [
            "from IPython.display import (\n",
            "    display_pretty, display_html, display_jpeg,\n",
            "    display_png, display_json, display_latex, display_svg\n",
            ")"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## Images"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "To work with images (JPEG, PNG) use the `Image` class."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 3,
          "metadata": {
            "collapsed": false
          },
          "outputs": [],
          "source": [
            "from IPython.display import Image"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 4,
          "metadata": {
            "collapsed": false
          },
          "outputs": [],
          "source": [
            "i \u003d Image(filename\u003d\u0027../images/ipython_logo.png\u0027)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Returning an `Image` object from an expression will automatically display it:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 5,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "image/png": "iVBORw0KGgoAAAANSUhEUgAAAggAAABDCAYAAAD5/P3lAAAABHNCSVQICAgIfAhkiAAAAAlwSFlz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\nHMRRca/E5hVINNIVwI2B56z6/3ExLRI31pXNAAAAAElFTkSuQmCC\n",
                "text/plain": [
                  "\u003cIPython.core.display.Image object\u003e"
                ]
              },
              "execution_count": 5,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "i"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Or you can pass an object with a rich representation to `display`:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 6,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "image/png": "iVBORw0KGgoAAAANSUhEUgAAAggAAABDCAYAAAD5/P3lAAAABHNCSVQICAgIfAhkiAAAAAlwSFlz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\nHMRRca/E5hVINNIVwI2B56z6/3ExLRI31pXNAAAAAElFTkSuQmCC\n",
                "text/plain": [
                  "\u003cIPython.core.display.Image object\u003e"
                ]
              },
              "metadata": {},
              "output_type": "display_data"
            }
          ],
          "source": [
            "display(i)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "An image can also be displayed from raw data or a URL."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 7,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\u003cimg src\u003d\"http://python.org/images/python-logo.gif\"/\u003e"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.Image object\u003e"
                ]
              },
              "execution_count": 7,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "Image(url\u003d\u0027http://python.org/images/python-logo.gif\u0027)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "SVG images are also supported out of the box."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 8,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "image/svg+xml": [
                  "\u003csvg height\u003d\"115.02pt\" id\u003d\"svg2\" inkscape:version\u003d\"0.43\" sodipodi:docbase\u003d\"/home/sdeibel\" sodipodi:docname\u003d\"logo-python-generic.svg\" sodipodi:version\u003d\"0.32\" version\u003d\"1.0\" width\u003d\"388.84pt\" xmlns\u003d\"http://www.w3.org/2000/svg\" xmlns:cc\u003d\"http://web.resource.org/cc/\" xmlns:dc\u003d\"http://purl.org/dc/elements/1.1/\" xmlns:inkscape\u003d\"http://www.inkscape.org/namespaces/inkscape\" xmlns:rdf\u003d\"http://www.w3.org/1999/02/22-rdf-syntax-ns#\" xmlns:sodipodi\u003d\"http://inkscape.sourceforge.net/DTD/sodipodi-0.dtd\" xmlns:svg\u003d\"http://www.w3.org/2000/svg\" xmlns:xlink\u003d\"http://www.w3.org/1999/xlink\"\u003e\n",
                  "  \u003cmetadata id\u003d\"metadata2193\"\u003e\n",
                  "    \u003crdf:RDF\u003e\n",
                  "      \u003ccc:Work rdf:about\u003d\"\"\u003e\n",
                  "        \u003cdc:format\u003eimage/svg+xml\u003c/dc:format\u003e\n",
                  "        \u003cdc:type rdf:resource\u003d\"http://purl.org/dc/dcmitype/StillImage\"/\u003e\n",
                  "      \u003c/cc:Work\u003e\n",
                  "    \u003c/rdf:RDF\u003e\n",
                  "  \u003c/metadata\u003e\n",
                  "  \u003csodipodi:namedview bordercolor\u003d\"#666666\" borderopacity\u003d\"1.0\" id\u003d\"base\" inkscape:current-layer\u003d\"svg2\" inkscape:cx\u003d\"243.02499\" inkscape:cy\u003d\"71.887497\" inkscape:pageopacity\u003d\"0.0\" inkscape:pageshadow\u003d\"2\" inkscape:window-height\u003d\"543\" inkscape:window-width\u003d\"791\" inkscape:window-x\u003d\"0\" inkscape:window-y\u003d\"0\" inkscape:zoom\u003d\"1.4340089\" pagecolor\u003d\"#ffffff\"/\u003e\n",
                  "  \u003cdefs id\u003d\"defs4\"\u003e\n",
                  "    \u003clinearGradient id\u003d\"linearGradient2795\"\u003e\n",
                  "      \u003cstop id\u003d\"stop2797\" offset\u003d\"0\" style\u003d\"stop-color:#b8b8b8;stop-opacity:0.49803922\"/\u003e\n",
                  "      \u003cstop id\u003d\"stop2799\" offset\u003d\"1\" style\u003d\"stop-color:#7f7f7f;stop-opacity:0\"/\u003e\n",
                  "    \u003c/linearGradient\u003e\n",
                  "    \u003clinearGradient id\u003d\"linearGradient2787\"\u003e\n",
                  "      \u003cstop id\u003d\"stop2789\" offset\u003d\"0\" style\u003d\"stop-color:#7f7f7f;stop-opacity:0.5\"/\u003e\n",
                  "      \u003cstop id\u003d\"stop2791\" offset\u003d\"1\" style\u003d\"stop-color:#7f7f7f;stop-opacity:0\"/\u003e\n",
                  "    \u003c/linearGradient\u003e\n",
                  "    \u003clinearGradient id\u003d\"linearGradient3676\"\u003e\n",
                  "      \u003cstop id\u003d\"stop3678\" offset\u003d\"0\" style\u003d\"stop-color:#b2b2b2;stop-opacity:0.5\"/\u003e\n",
                  "      \u003cstop id\u003d\"stop3680\" offset\u003d\"1\" style\u003d\"stop-color:#b3b3b3;stop-opacity:0\"/\u003e\n",
                  "    \u003c/linearGradient\u003e\n",
                  "    \u003clinearGradient id\u003d\"linearGradient3236\"\u003e\n",
                  "      \u003cstop id\u003d\"stop3244\" offset\u003d\"0\" style\u003d\"stop-color:#f4f4f4;stop-opacity:1\"/\u003e\n",
                  "      \u003cstop id\u003d\"stop3240\" offset\u003d\"1\" style\u003d\"stop-color:#ffffff;stop-opacity:1\"/\u003e\n",
                  "    \u003c/linearGradient\u003e\n",
                  "    \u003clinearGradient id\u003d\"linearGradient4671\"\u003e\n",
                  "      \u003cstop id\u003d\"stop4673\" offset\u003d\"0\" style\u003d\"stop-color:#ffd43b;stop-opacity:1\"/\u003e\n",
                  "      \u003cstop id\u003d\"stop4675\" offset\u003d\"1\" style\u003d\"stop-color:#ffe873;stop-opacity:1\"/\u003e\n",
                  "    \u003c/linearGradient\u003e\n",
                  "    \u003clinearGradient id\u003d\"linearGradient4689\"\u003e\n",
                  "      \u003cstop id\u003d\"stop4691\" offset\u003d\"0\" style\u003d\"stop-color:#5a9fd4;stop-opacity:1\"/\u003e\n",
                  "      \u003cstop id\u003d\"stop4693\" offset\u003d\"1\" style\u003d\"stop-color:#306998;stop-opacity:1\"/\u003e\n",
                  "    \u003c/linearGradient\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"translate(100.2702,99.61116)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient2987\" x1\u003d\"224.23996\" x2\u003d\"-65.308502\" xlink:href\u003d\"#linearGradient4671\" y1\u003d\"144.75717\" y2\u003d\"144.75717\"/\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"translate(100.2702,99.61116)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient2990\" x1\u003d\"172.94208\" x2\u003d\"26.670298\" xlink:href\u003d\"#linearGradient4689\" y1\u003d\"77.475983\" y2\u003d\"76.313133\"/\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"translate(100.2702,99.61116)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient2587\" x1\u003d\"172.94208\" x2\u003d\"26.670298\" xlink:href\u003d\"#linearGradient4689\" y1\u003d\"77.475983\" y2\u003d\"76.313133\"/\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"translate(100.2702,99.61116)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient2589\" x1\u003d\"224.23996\" x2\u003d\"-65.308502\" xlink:href\u003d\"#linearGradient4671\" y1\u003d\"144.75717\" y2\u003d\"144.75717\"/\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"translate(100.2702,99.61116)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient2248\" x1\u003d\"172.94208\" x2\u003d\"26.670298\" xlink:href\u003d\"#linearGradient4689\" y1\u003d\"77.475983\" y2\u003d\"76.313133\"/\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"translate(100.2702,99.61116)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient2250\" x1\u003d\"224.23996\" x2\u003d\"-65.308502\" xlink:href\u003d\"#linearGradient4671\" y1\u003d\"144.75717\" y2\u003d\"144.75717\"/\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"matrix(0.562541,0,0,0.567972,-11.5974,-7.60954)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient2255\" x1\u003d\"224.23996\" x2\u003d\"-65.308502\" xlink:href\u003d\"#linearGradient4671\" y1\u003d\"144.75717\" y2\u003d\"144.75717\"/\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"matrix(0.562541,0,0,0.567972,-11.5974,-7.60954)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient2258\" x1\u003d\"172.94208\" x2\u003d\"26.670298\" xlink:href\u003d\"#linearGradient4689\" y1\u003d\"76.176224\" y2\u003d\"76.313133\"/\u003e\n",
                  "    \u003cradialGradient cx\u003d\"61.518883\" cy\u003d\"132.28575\" fx\u003d\"61.518883\" fy\u003d\"132.28575\" gradientTransform\u003d\"matrix(1,0,0,0.177966,0,108.7434)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"radialGradient2801\" r\u003d\"29.036913\" xlink:href\u003d\"#linearGradient2795\"/\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"matrix(0.562541,0,0,0.567972,-9.399749,-5.305317)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient1475\" x1\u003d\"150.96111\" x2\u003d\"112.03144\" xlink:href\u003d\"#linearGradient4671\" y1\u003d\"192.35176\" y2\u003d\"137.27299\"/\u003e\n",
                  "    \u003clinearGradient gradientTransform\u003d\"matrix(0.562541,0,0,0.567972,-9.399749,-5.305317)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"linearGradient1478\" x1\u003d\"26.648937\" x2\u003d\"135.66525\" xlink:href\u003d\"#linearGradient4689\" y1\u003d\"20.603781\" y2\u003d\"114.39767\"/\u003e\n",
                  "    \u003cradialGradient cx\u003d\"61.518883\" cy\u003d\"132.28575\" fx\u003d\"61.518883\" fy\u003d\"132.28575\" gradientTransform\u003d\"matrix(2.382716e-8,-0.296405,1.43676,4.683673e-7,-128.544,150.5202)\" gradientUnits\u003d\"userSpaceOnUse\" id\u003d\"radialGradient1480\" r\u003d\"29.036913\" xlink:href\u003d\"#linearGradient2795\"/\u003e\n",
                  "  \u003c/defs\u003e\n",
                  "  \u003cg id\u003d\"g2303\"\u003e\n",
                  "    \u003cpath d\u003d\"M 184.61344,61.929363 C 184.61344,47.367213 180.46118,39.891193 172.15666,39.481813 C 168.85239,39.325863 165.62611,39.852203 162.48754,41.070593 C 159.98254,41.967323 158.2963,42.854313 157.40931,43.751043 L 157.40931,78.509163 C 162.72147,81.842673 167.43907,83.392453 171.55234,83.148783 C 180.25649,82.573703 184.61344,75.507063 184.61344,61.929363 z M 194.85763,62.533683 C 194.85763,69.931723 193.12265,76.072393 189.63319,80.955683 C 185.7441,86.482283 180.35396,89.328433 173.46277,89.484393 C 168.26757,89.650093 162.91642,88.022323 157.40931,84.610843 L 157.40931,116.20116 L 148.50047,113.02361 L 148.50047,42.903043 C 149.96253,41.109583 151.84372,39.569543 154.12454,38.263433 C 159.42696,35.173603 165.86978,33.584823 173.45302,33.506853 L 173.57973,33.633563 C 180.50991,33.545833 185.85132,36.391993 189.60395,42.162263 C 193.10315,47.454933 194.85763,54.238913 194.85763,62.533683 z \" id\u003d\"path46\" style\u003d\"fill:#646464;fill-opacity:1\"/\u003e\n",
                  "    \u003cpath d\u003d\"M 249.30487,83.265743 C 249.30487,93.188283 248.31067,100.05998 246.32227,103.88084 C 244.32411,107.7017 240.52275,110.75254 234.90842,113.02361 C 230.35653,114.81707 225.43425,115.79178 220.15133,115.95748 L 218.67952,110.34316 C 224.05016,109.61213 227.83204,108.88109 230.02513,108.15006 C 234.34309,106.688 237.30621,104.44617 238.93397,101.44406 C 240.24008,98.997543 240.88339,94.328693 240.88339,87.418003 L 240.88339,85.098203 C 234.79146,87.866373 228.40711,89.240713 221.73036,89.240713 C 217.34417,89.240713 213.47457,87.866373 210.14107,85.098203 C 206.39818,82.086343 204.52674,78.265483 204.52674,73.635623 L 204.52674,36.557693 L 213.43558,33.506853 L 213.43558,70.828453 C 213.43558,74.815013 214.7222,77.885353 217.29543,80.039463 C 219.86866,82.193563 223.20217,83.226753 227.2862,83.148783 C 231.37023,83.061053 235.74667,81.482023 240.39603,78.392203 L 240.39603,34.851953 L 249.30487,34.851953 L 249.30487,83.265743 z \" id\u003d\"path48\" style\u003d\"fill:#646464;fill-opacity:1\"/\u003e\n",
                  "    \u003cpath d\u003d\"M 284.08249,88.997033 C 283.02006,89.084753 282.04535,89.123743 281.14862,89.123743 C 276.10937,89.123743 272.18129,87.924853 269.37413,85.517323 C 266.57671,83.109793 265.17314,79.786033 265.17314,75.546053 L 265.17314,40.456523 L 259.07146,40.456523 L 259.07146,34.851953 L 265.17314,34.851953 L 265.17314,19.968143 L 274.07223,16.800333 L 274.07223,34.851953 L 284.08249,34.851953 L 284.08249,40.456523 L 274.07223,40.456523 L 274.07223,75.302373 C 274.07223,78.645623 274.96896,81.014163 276.76243,82.398253 C 278.30247,83.538663 280.74899,84.191723 284.08249,84.357423 L 284.08249,88.997033 z \" id\u003d\"path50\" style\u003d\"fill:#646464;fill-opacity:1\"/\u003e\n",
                  "    \u003cpath d\u003d\"M 338.02288,88.266003 L 329.11404,88.266003 L 329.11404,53.878273 C 329.11404,50.379063 328.29528,47.367213 326.66753,44.852463 C 324.78634,42.006313 322.17411,40.583233 318.82112,40.583233 C 314.73708,40.583233 309.6296,42.737343 303.4987,47.045563 L 303.4987,88.266003 L 294.58985,88.266003 L 294.58985,6.0687929 L 303.4987,3.2616329 L 303.4987,40.700203 C 309.191,36.557693 315.40963,34.481563 322.16436,34.481563 C 326.88196,34.481563 330.70282,36.070333 333.62694,39.238143 C 336.56082,42.405943 338.02288,46.353513 338.02288,51.071103 L 338.02288,88.266003 L 338.02288,88.266003 z \" id\u003d\"path52\" style\u003d\"fill:#646464;fill-opacity:1\"/\u003e\n",
                  "    \u003cpath d\u003d\"M 385.37424,60.525783 C 385.37424,54.930953 384.31182,50.310833 382.19669,46.655673 C 379.68195,42.201253 375.77337,39.852203 370.49044,39.608523 C 360.72386,40.173863 355.85032,47.172273 355.85032,60.584263 C 355.85032,66.734683 356.86401,71.871393 358.91089,75.994413 C 361.52312,81.248093 365.44145,83.840823 370.66589,83.753103 C 380.47146,83.675123 385.37424,75.935933 385.37424,60.525783 z M 395.13109,60.584263 C 395.13109,68.547643 393.09395,75.175663 389.02941,80.468333 C 384.5555,86.394563 378.37584,89.367423 370.49044,89.367423 C 362.67328,89.367423 356.58135,86.394563 352.18541,80.468333 C 348.19885,75.175663 346.21044,68.547643 346.21044,60.584263 C 346.21044,53.098503 348.36455,46.801883 352.67276,41.674913 C 357.22466,36.236033 363.20937,33.506853 370.6074,33.506853 C 378.00545,33.506853 384.02914,36.236033 388.66877,41.674913 C 392.97697,46.801883 395.13109,53.098503 395.13109,60.584263 z \" id\u003d\"path54\" style\u003d\"fill:#646464;fill-opacity:1\"/\u003e\n",
                  "    \u003cpath d\u003d\"M 446.20583,88.266003 L 437.29699,88.266003 L 437.29699,51.928853 C 437.29699,47.942293 436.0981,44.832973 433.70032,42.591133 C 431.30253,40.359053 428.10549,39.277123 424.11893,39.364853 C 419.8887,39.442833 415.86314,40.826913 412.04229,43.507363 L 412.04229,88.266003 L 403.13345,88.266003 L 403.13345,42.405943 C 408.26042,38.672813 412.97801,36.236033 417.28621,35.095623 C 421.35076,34.033193 424.93769,33.506853 428.02752,33.506853 C 430.14264,33.506853 432.13104,33.711543 434.00248,34.120913 C 437.50169,34.929923 440.34783,36.430973 442.54093,38.633823 C 444.98744,41.070593 446.20583,43.994723 446.20583,47.415943 L 446.20583,88.266003 z \" id\u003d\"path56\" style\u003d\"fill:#646464;fill-opacity:1\"/\u003e\n",
                  "    \u003cpath d\u003d\"M 60.510156,6.3979729 C 55.926503,6.4192712 51.549217,6.8101906 47.697656,7.4917229 C 36.35144,9.4962267 34.291407,13.691825 34.291406,21.429223 L 34.291406,31.647973 L 61.103906,31.647973 L 61.103906,35.054223 L 34.291406,35.054223 L 24.228906,35.054223 C 16.436447,35.054223 9.6131468,39.73794 7.4789058,48.647973 C 5.0170858,58.860939 4.9078907,65.233996 7.4789058,75.897973 C 9.3848341,83.835825 13.936449,89.491721 21.728906,89.491723 L 30.947656,89.491723 L 30.947656,77.241723 C 30.947656,68.391821 38.6048,60.585475 47.697656,60.585473 L 74.478906,60.585473 C 81.933857,60.585473 87.885159,54.447309 87.885156,46.960473 L 87.885156,21.429223 C 87.885156,14.162884 81.755176,8.7044455 74.478906,7.4917229 C 69.872919,6.7249976 65.093809,6.3766746 60.510156,6.3979729 z M 46.010156,14.616723 C 48.779703,14.616723 51.041406,16.915369 51.041406,19.741723 C 51.041404,22.558059 48.779703,24.835473 46.010156,24.835473 C 43.23068,24.835472 40.978906,22.558058 40.978906,19.741723 C 40.978905,16.91537 43.23068,14.616723 46.010156,14.616723 z \" id\u003d\"path1948\" style\u003d\"fill:url(#linearGradient1478);fill-opacity:1\"/\u003e\n",
                  "    \u003cpath d\u003d\"M 91.228906,35.054223 L 91.228906,46.960473 C 91.228906,56.191228 83.403011,63.960472 74.478906,63.960473 L 47.697656,63.960473 C 40.361823,63.960473 34.291407,70.238956 34.291406,77.585473 L 34.291406,103.11672 C 34.291406,110.38306 40.609994,114.65704 47.697656,116.74172 C 56.184987,119.23733 64.323893,119.68835 74.478906,116.74172 C 81.229061,114.78733 87.885159,110.85411 87.885156,103.11672 L 87.885156,92.897973 L 61.103906,92.897973 L 61.103906,89.491723 L 87.885156,89.491723 L 101.29141,89.491723 C 109.08387,89.491723 111.98766,84.056315 114.69765,75.897973 C 117.49698,67.499087 117.37787,59.422197 114.69765,48.647973 C 112.77187,40.890532 109.09378,35.054223 101.29141,35.054223 L 91.228906,35.054223 z M 76.166406,99.710473 C 78.945884,99.710476 81.197656,101.98789 81.197656,104.80422 C 81.197654,107.63057 78.945881,109.92922 76.166406,109.92922 C 73.396856,109.92922 71.135156,107.63057 71.135156,104.80422 C 71.135158,101.98789 73.396853,99.710473 76.166406,99.710473 z \" id\u003d\"path1950\" style\u003d\"fill:url(#linearGradient1475);fill-opacity:1\"/\u003e\n",
                  "    \u003cpath d\u003d\"M 463.5544,26.909383 L 465.11635,26.909383 L 465.11635,17.113143 L 468.81648,17.113143 L 468.81648,15.945483 L 459.85427,15.945483 L 459.85427,17.113143 L 463.5544,17.113143 L 463.5544,26.909383 M 470.20142,26.909383 L 471.53589,26.909383 L 471.53589,17.962353 L 474.4323,26.908259 L 475.91799,26.908259 L 478.93615,17.992683 L 478.93615,26.909383 L 480.39194,26.909383 L 480.39194,15.945483 L 478.46605,15.945483 L 475.16774,25.33834 L 472.35477,15.945483 L 470.20142,15.945483 L 470.20142,26.909383\" id\u003d\"text3004\" style\u003d\"font-size:15.16445827px;font-style:normal;font-weight:normal;line-height:125%;fill:#646464;fill-opacity:1;stroke:none;stroke-width:1px;stroke-linecap:butt;stroke-linejoin:miter;stroke-opacity:1;font-family:Bitstream Vera Sans\"/\u003e\n",
                  "    \u003cpath d\u003d\"M 110.46717 132.28575 A 48.948284 8.6066771 0 1 1  12.570599,132.28575 A 48.948284 8.6066771 0 1 1  110.46717 132.28575 z\" id\u003d\"path1894\" style\u003d\"opacity:0.44382019;fill:url(#radialGradient1480);fill-opacity:1;fill-rule:nonzero;stroke:none;stroke-width:20;stroke-miterlimit:4;stroke-dasharray:none;stroke-opacity:1\" transform\u003d\"matrix(0.73406,0,0,0.809524,16.24958,27.00935)\"/\u003e\n",
                  "  \u003c/g\u003e\n",
                  "\u003c/svg\u003e"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.SVG object\u003e"
                ]
              },
              "execution_count": 8,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "from IPython.display import SVG\n",
            "SVG(filename\u003d\u0027../images/python_logo.svg\u0027)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "### Embedded vs non-embedded Images"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "By default, image data is embedded in the notebook document so that the images can be viewed offline. However it is also possible to tell the `Image` class to only store a *link* to the image. Let\u0027s see how this works using a webcam at Berkeley."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 9,
          "metadata": {
            "collapsed": false
          },
          "outputs": [],
          "source": [
            "from IPython.display import Image\n",
            "img_url \u003d \u0027http://www.lawrencehallofscience.org/static/scienceview/scienceview.berkeley.edu/html/view/view_assets/images/newview.jpg\u0027\n",
            "\n",
            "# by default Image data are embedded\n",
            "Embed      \u003d Image(img_url)\n",
            "\n",
            "# if kwarg `url` is given, the embedding is assumed to be false\n",
            "SoftLinked \u003d Image(url\u003dimg_url)\n",
            "\n",
            "# In each case, embed can be specified explicitly with the `embed` kwarg\n",
            "# ForceEmbed \u003d Image(url\u003dimg_url, embed\u003dTrue)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Here is the embedded version. Note that this image was pulled from the webcam when this code cell was originally run and stored in the Notebook. Unless we rerun this cell, this is not today\u0027s image."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 10,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "image/jpeg": [
                  "/9j/4AAQSkZJRgABAQEAtAC0AAD//gAdQ29weXJpZ2h0IDIwMTUgVS5DLiBSZWdlbnRz/+Ej/kV4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\nsAcfzdqzkJH/2Q\u003d\u003d\n"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.Image object\u003e"
                ]
              },
              "metadata": {}
            }
          ],
          "source": [
            "Embed"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Here is today\u0027s image from same webcam at Berkeley, (refreshed every minutes, if you reload the notebook), visible only with an active internet connection, that should be different from the previous one. Notebooks saved with this kind of image will be smaller and always reflect the current version of the source, but the image won\u0027t display offline."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 11,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\u003cimg src\u003d\"http://www.lawrencehallofscience.org/static/scienceview/scienceview.berkeley.edu/html/view/view_assets/images/newview.jpg\"/\u003e"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.Image object\u003e"
                ]
              },
              "execution_count": 11,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "SoftLinked"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Of course, if you re-run this Notebook, the two images will be the same again."
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## HTML"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Python objects can declare HTML representations that will be displayed in the Notebook. If you have some HTML you want to display, simply use the `HTML` class."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 12,
          "metadata": {
            "collapsed": false
          },
          "outputs": [],
          "source": [
            "from IPython.display import HTML"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 13,
          "metadata": {
            "collapsed": false
          },
          "outputs": [],
          "source": [
            "s \u003d \"\"\"\u003ctable\u003e\n",
            "\u003ctr\u003e\n",
            "\u003cth\u003eHeader 1\u003c/th\u003e\n",
            "\u003cth\u003eHeader 2\u003c/th\u003e\n",
            "\u003c/tr\u003e\n",
            "\u003ctr\u003e\n",
            "\u003ctd\u003erow 1, cell 1\u003c/td\u003e\n",
            "\u003ctd\u003erow 1, cell 2\u003c/td\u003e\n",
            "\u003c/tr\u003e\n",
            "\u003ctr\u003e\n",
            "\u003ctd\u003erow 2, cell 1\u003c/td\u003e\n",
            "\u003ctd\u003erow 2, cell 2\u003c/td\u003e\n",
            "\u003c/tr\u003e\n",
            "\u003c/table\u003e\"\"\""
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 14,
          "metadata": {
            "collapsed": false
          },
          "outputs": [],
          "source": [
            "h \u003d HTML(s)"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 15,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\u003ctable\u003e\n",
                  "\u003ctr\u003e\n",
                  "\u003cth\u003eHeader 1\u003c/th\u003e\n",
                  "\u003cth\u003eHeader 2\u003c/th\u003e\n",
                  "\u003c/tr\u003e\n",
                  "\u003ctr\u003e\n",
                  "\u003ctd\u003erow 1, cell 1\u003c/td\u003e\n",
                  "\u003ctd\u003erow 1, cell 2\u003c/td\u003e\n",
                  "\u003c/tr\u003e\n",
                  "\u003ctr\u003e\n",
                  "\u003ctd\u003erow 2, cell 1\u003c/td\u003e\n",
                  "\u003ctd\u003erow 2, cell 2\u003c/td\u003e\n",
                  "\u003c/tr\u003e\n",
                  "\u003c/table\u003e"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.HTML object\u003e"
                ]
              },
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "display(h)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "You can also use the `%%html` cell magic to accomplish the same thing."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 16,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\u003ctable\u003e\n",
                  "\u003ctr\u003e\n",
                  "\u003cth\u003eHeader 1\u003c/th\u003e\n",
                  "\u003cth\u003eHeader 2\u003c/th\u003e\n",
                  "\u003c/tr\u003e\n",
                  "\u003ctr\u003e\n",
                  "\u003ctd\u003erow 1, cell 1\u003c/td\u003e\n",
                  "\u003ctd\u003erow 1, cell 2\u003c/td\u003e\n",
                  "\u003c/tr\u003e\n",
                  "\u003ctr\u003e\n",
                  "\u003ctd\u003erow 2, cell 1\u003c/td\u003e\n",
                  "\u003ctd\u003erow 2, cell 2\u003c/td\u003e\n",
                  "\u003c/tr\u003e\n",
                  "\u003c/table\u003e"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.HTML object\u003e"
                ]
              },
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "%%html\n",
            "\u003ctable\u003e\n",
            "\u003ctr\u003e\n",
            "\u003cth\u003eHeader 1\u003c/th\u003e\n",
            "\u003cth\u003eHeader 2\u003c/th\u003e\n",
            "\u003c/tr\u003e\n",
            "\u003ctr\u003e\n",
            "\u003ctd\u003erow 1, cell 1\u003c/td\u003e\n",
            "\u003ctd\u003erow 1, cell 2\u003c/td\u003e\n",
            "\u003c/tr\u003e\n",
            "\u003ctr\u003e\n",
            "\u003ctd\u003erow 2, cell 1\u003c/td\u003e\n",
            "\u003ctd\u003erow 2, cell 2\u003c/td\u003e\n",
            "\u003c/tr\u003e\n",
            "\u003c/table\u003e"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## JavaScript"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "The Notebook also enables objects to declare a JavaScript representation. At first, this may seem odd as  output is inherently visual and JavaScript is a programming language. However, this opens the door for rich output that leverages the full power of JavaScript and associated libraries such as [d3.js](http://d3js.org) for output."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 17,
          "metadata": {
            "collapsed": false
          },
          "outputs": [],
          "source": [
            "from IPython.display import Javascript"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Pass a string of JavaScript source code to the `JavaScript` object and then display it."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 18,
          "metadata": {
            "collapsed": false
          },
          "outputs": [],
          "source": [
            "js \u003d Javascript(\u0027alert(\"hi\")\u0027);"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 19,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/plain": [
                  "\u003cIPython.core.display.Javascript object\u003e"
                ]
              },
              "metadata": {},
              "output_type": "display_data"
            }
          ],
          "source": [
            "display(js)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "The same thing can be accomplished using the `%%javascript` cell magic:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 20,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/plain": [
                  "\u003cIPython.core.display.Javascript object\u003e"
                ]
              },
              "metadata": {},
              "output_type": "display_data"
            }
          ],
          "source": [
            "%%javascript\n",
            "\n",
            "alert(\"hi\");"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Here is a more complicated example that loads `d3.js` from a CDN, uses the `%%html` magic to load CSS styles onto the page and then runs ones of the `d3.js` examples."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 21,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/plain": [
                  "\u003cIPython.core.display.Javascript object\u003e"
                ]
              },
              "execution_count": 21,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "Javascript(\n",
            "    \"\"\"$.getScript(\u0027//cdnjs.cloudflare.com/ajax/libs/d3/3.2.2/d3.v3.min.js\u0027)\"\"\"\n",
            ")"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 22,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\u003cstyle type\u003d\"text/css\"\u003e\n",
                  "\n",
                  "circle {\n",
                  "  fill: rgb(31, 119, 180);\n",
                  "  fill-opacity: .25;\n",
                  "  stroke: rgb(31, 119, 180);\n",
                  "  stroke-width: 1px;\n",
                  "}\n",
                  "\n",
                  ".leaf circle {\n",
                  "  fill: #ff7f0e;\n",
                  "  fill-opacity: 1;\n",
                  "}\n",
                  "\n",
                  "text {\n",
                  "  font: 10px sans-serif;\n",
                  "}\n",
                  "\n",
                  "\u003c/style\u003e"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.HTML object\u003e"
                ]
              },
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "%%html\n",
            "\u003cstyle type\u003d\"text/css\"\u003e\n",
            "\n",
            "circle {\n",
            "  fill: rgb(31, 119, 180);\n",
            "  fill-opacity: .25;\n",
            "  stroke: rgb(31, 119, 180);\n",
            "  stroke-width: 1px;\n",
            "}\n",
            "\n",
            ".leaf circle {\n",
            "  fill: #ff7f0e;\n",
            "  fill-opacity: 1;\n",
            "}\n",
            "\n",
            "text {\n",
            "  font: 10px sans-serif;\n",
            "}\n",
            "\n",
            "\u003c/style\u003e"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 23,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/plain": [
                  "\u003cIPython.core.display.Javascript object\u003e"
                ]
              },
              "metadata": {},
              "output_type": "display_data"
            }
          ],
          "source": [
            "%%javascript\n",
            "\n",
            "// element is the jQuery element we will append to\n",
            "var e \u003d element.get(0);\n",
            "    \n",
            "var diameter \u003d 600,\n",
            "    format \u003d d3.format(\",d\");\n",
            "\n",
            "var pack \u003d d3.layout.pack()\n",
            "    .size([diameter - 4, diameter - 4])\n",
            "    .value(function(d) { return d.size; });\n",
            "\n",
            "var svg \u003d d3.select(e).append(\"svg\")\n",
            "    .attr(\"width\", diameter)\n",
            "    .attr(\"height\", diameter)\n",
            "  .append(\"g\")\n",
            "    .attr(\"transform\", \"translate(2,2)\");\n",
            "\n",
            "d3.json(\"data/flare.json\", function(error, root) {\n",
            "  var node \u003d svg.datum(root).selectAll(\".node\")\n",
            "      .data(pack.nodes)\n",
            "    .enter().append(\"g\")\n",
            "      .attr(\"class\", function(d) { return d.children ? \"node\" : \"leaf node\"; })\n",
            "      .attr(\"transform\", function(d) { return \"translate(\" + d.x + \",\" + d.y + \")\"; });\n",
            "\n",
            "  node.append(\"title\")\n",
            "      .text(function(d) { return d.name + (d.children ? \"\" : \": \" + format(d.size)); });\n",
            "\n",
            "  node.append(\"circle\")\n",
            "      .attr(\"r\", function(d) { return d.r; });\n",
            "\n",
            "  node.filter(function(d) { return !d.children; }).append(\"text\")\n",
            "      .attr(\"dy\", \".3em\")\n",
            "      .style(\"text-anchor\", \"middle\")\n",
            "      .text(function(d) { return d.name.substring(0, d.r / 3); });\n",
            "});\n",
            "\n",
            "d3.select(self.frameElement).style(\"height\", diameter + \"px\");"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## LaTeX"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "The IPython display system also has builtin support for the display of mathematical expressions typeset in LaTeX, which is rendered in the browser using [MathJax](http://mathjax.org)."
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "You can pass raw LaTeX test as a string to the `Math` object:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 24,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/latex": [
                  "$$F(k) \u003d \\int_{-\\infty}^{\\infty} f(x) e^{2\\pi i k} dx$$"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.Math object\u003e"
                ]
              },
              "execution_count": 24,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "from IPython.display import Math\n",
            "Math(r\u0027F(k) \u003d \\int_{-\\infty}^{\\infty} f(x) e^{2\\pi i k} dx\u0027)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "With the `Latex` class, you have to include the delimiters yourself.  This allows you to use other LaTeX modes such as `eqnarray`:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 25,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/latex": [
                  "\\begin{eqnarray}\n",
                  "\\nabla \\times \\vec{\\mathbf{B}} -\\, \\frac1c\\, \\frac{\\partial\\vec{\\mathbf{E}}}{\\partial t} \u0026 \u003d \\frac{4\\pi}{c}\\vec{\\mathbf{j}} \\\\\n",
                  "\\nabla \\cdot \\vec{\\mathbf{E}} \u0026 \u003d 4 \\pi \\rho \\\\\n",
                  "\\nabla \\times \\vec{\\mathbf{E}}\\, +\\, \\frac1c\\, \\frac{\\partial\\vec{\\mathbf{B}}}{\\partial t} \u0026 \u003d \\vec{\\mathbf{0}} \\\\\n",
                  "\\nabla \\cdot \\vec{\\mathbf{B}} \u0026 \u003d 0 \n",
                  "\\end{eqnarray}"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.Latex object\u003e"
                ]
              },
              "execution_count": 25,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "from IPython.display import Latex\n",
            "Latex(r\"\"\"\\begin{eqnarray}\n",
            "\\nabla \\times \\vec{\\mathbf{B}} -\\, \\frac1c\\, \\frac{\\partial\\vec{\\mathbf{E}}}{\\partial t} \u0026 \u003d \\frac{4\\pi}{c}\\vec{\\mathbf{j}} \\\\\n",
            "\\nabla \\cdot \\vec{\\mathbf{E}} \u0026 \u003d 4 \\pi \\rho \\\\\n",
            "\\nabla \\times \\vec{\\mathbf{E}}\\, +\\, \\frac1c\\, \\frac{\\partial\\vec{\\mathbf{B}}}{\\partial t} \u0026 \u003d \\vec{\\mathbf{0}} \\\\\n",
            "\\nabla \\cdot \\vec{\\mathbf{B}} \u0026 \u003d 0 \n",
            "\\end{eqnarray}\"\"\")"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Or you can enter LaTeX directly with the `%%latex` cell magic:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 26,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/latex": [
                  "\\begin{align}\n",
                  "\\nabla \\times \\vec{\\mathbf{B}} -\\, \\frac1c\\, \\frac{\\partial\\vec{\\mathbf{E}}}{\\partial t} \u0026 \u003d \\frac{4\\pi}{c}\\vec{\\mathbf{j}} \\\\\n",
                  "\\nabla \\cdot \\vec{\\mathbf{E}} \u0026 \u003d 4 \\pi \\rho \\\\\n",
                  "\\nabla \\times \\vec{\\mathbf{E}}\\, +\\, \\frac1c\\, \\frac{\\partial\\vec{\\mathbf{B}}}{\\partial t} \u0026 \u003d \\vec{\\mathbf{0}} \\\\\n",
                  "\\nabla \\cdot \\vec{\\mathbf{B}} \u0026 \u003d 0\n",
                  "\\end{align}"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.Latex object\u003e"
                ]
              },
              "metadata": {},
              "output_type": "display_data"
            }
          ],
          "source": [
            "%%latex\n",
            "\\begin{align}\n",
            "\\nabla \\times \\vec{\\mathbf{B}} -\\, \\frac1c\\, \\frac{\\partial\\vec{\\mathbf{E}}}{\\partial t} \u0026 \u003d \\frac{4\\pi}{c}\\vec{\\mathbf{j}} \\\\\n",
            "\\nabla \\cdot \\vec{\\mathbf{E}} \u0026 \u003d 4 \\pi \\rho \\\\\n",
            "\\nabla \\times \\vec{\\mathbf{E}}\\, +\\, \\frac1c\\, \\frac{\\partial\\vec{\\mathbf{B}}}{\\partial t} \u0026 \u003d \\vec{\\mathbf{0}} \\\\\n",
            "\\nabla \\cdot \\vec{\\mathbf{B}} \u0026 \u003d 0\n",
            "\\end{align}"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## Audio"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "IPython makes it easy to work with sounds interactively. The `Audio` display class allows you to create an audio control that is embedded in the Notebook. The interface is analogous to the interface of the `Image` display class. All audio formats supported by the browser can be used. Note that no single format is presently supported in all browsers."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 27,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\n",
                  "                \u003caudio controls\u003d\"controls\" \u003e\n",
                  "                    \u003csource src\u003d\"http://www.nch.com.au/acm/8k16bitpcm.wav\" type\u003d\"audio/x-wav\" /\u003e\n",
                  "                    Your browser does not support the audio element.\n",
                  "                \u003c/audio\u003e\n",
                  "              "
                ],
                "text/plain": [
                  "\u003cIPython.lib.display.Audio object\u003e"
                ]
              },
              "execution_count": 27,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "from IPython.display import Audio\n",
            "Audio(url\u003d\"http://www.nch.com.au/acm/8k16bitpcm.wav\")"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "A NumPy array can be auralized automatically. The `Audio` class normalizes and encodes the data and embeds the resulting audio in the Notebook.\n",
            "\n",
            "For instance, when two sine waves with almost the same frequency are superimposed a phenomena known as [beats](https://en.wikipedia.org/wiki/Beat_%28acoustics%29) occur. This can be auralised as follows:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 28,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\n",
                  "                \u003caudio controls\u003d\"controls\" \u003e\n",
                  "                    \u003csource src\u003d\"data:audio/wav;base64,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\u003d\" type\u003d\"audio/wav\" /\u003e\n",
                  "                    Your browser does not support the audio element.\n",
                  "                \u003c/audio\u003e\n",
                  "              "
                ],
                "text/plain": [
                  "\u003cIPython.lib.display.Audio object\u003e"
                ]
              },
              "execution_count": 28,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "import numpy as np\n",
            "max_time \u003d 3\n",
            "f1 \u003d 220.0\n",
            "f2 \u003d 224.0\n",
            "rate \u003d 8000.0\n",
            "L \u003d 3\n",
            "times \u003d np.linspace(0,L,rate*L)\n",
            "signal \u003d np.sin(2*np.pi*f1*times) + np.sin(2*np.pi*f2*times)\n",
            "\n",
            "Audio(data\u003dsignal, rate\u003drate)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## Video"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "More exotic objects can also be displayed, as long as their representation supports the IPython display protocol.  For example, videos hosted externally on YouTube are easy to load:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 29,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\n",
                  "        \u003ciframe\n",
                  "            width\u003d\"400\"\n",
                  "            height\u003d\"300\"\n",
                  "            src\u003d\"https://www.youtube.com/embed/sjfsUzECqK0\"\n",
                  "            frameborder\u003d\"0\"\n",
                  "            allowfullscreen\n",
                  "        \u003e\u003c/iframe\u003e\n",
                  "        "
                ],
                "text/plain": [
                  "\u003cIPython.lib.display.YouTubeVideo at 0x1058e5cf8\u003e"
                ]
              },
              "execution_count": 29,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "from IPython.display import YouTubeVideo\n",
            "YouTubeVideo(\u0027sjfsUzECqK0\u0027)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Using the nascent video capabilities of modern browsers, you may also be able to display local\n",
            "videos.  At the moment this doesn\u0027t work very well in all browsers, so it may or may not work for you;\n",
            "we will continue testing this and looking for ways to make it more robust.  \n",
            "\n",
            "The following cell loads a local file called  `animation.m4v`, encodes the raw video as base64 for http\n",
            "transport, and uses the HTML5 video tag to load it. On Chrome 15 it works correctly, displaying a control bar at the bottom with a play/pause button and a location slider."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 30,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\u003cvideo controls alt\u003d\"test\" src\u003d\"data:video/x-m4v;base64,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\u003d\"\u003e"
                ],
                "text/plain": [
                  "\u003cIPython.core.display.HTML object\u003e"
                ]
              },
              "execution_count": 30,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "from IPython.display import HTML\n",
            "from base64 import b64encode\n",
            "video \u003d open(\"../images/animation.m4v\", \"rb\").read()\n",
            "video_encoded \u003d b64encode(video).decode(\u0027ascii\u0027)\n",
            "video_tag \u003d \u0027\u003cvideo controls alt\u003d\"test\" src\u003d\"data:video/x-m4v;base64,{0}\"\u003e\u0027.format(video_encoded)\n",
            "HTML(data\u003dvideo_tag)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## External sites"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "You can even embed an entire page from another site in an iframe; for example this is today\u0027s Wikipedia\n",
            "page for mobile users:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 31,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\n",
                  "        \u003ciframe\n",
                  "            width\u003d\"100%\"\n",
                  "            height\u003d\"350\"\n",
                  "            src\u003d\"https://jupyter.org\"\n",
                  "            frameborder\u003d\"0\"\n",
                  "            allowfullscreen\n",
                  "        \u003e\u003c/iframe\u003e\n",
                  "        "
                ],
                "text/plain": [
                  "\u003cIPython.lib.display.IFrame at 0x1059ae7b8\u003e"
                ]
              },
              "execution_count": 31,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "from IPython.display import IFrame\n",
            "IFrame(\u0027https://jupyter.org\u0027, width\u003d\u0027100%\u0027, height\u003d350)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## Links to local files"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "IPython provides builtin display classes for generating links to local files. Create a link to a single file using the `FileLink` object:"
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 32,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "\u003ca href\u003d\u0027Cell Magics.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eCell Magics.ipynb\u003c/a\u003e\u003cbr\u003e"
                ],
                "text/plain": [
                  "/Users/minrk/dev/ip/mine/examples/IPython Kernel/Cell Magics.ipynb"
                ]
              },
              "execution_count": 32,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "from IPython.display import FileLink, FileLinks\n",
            "FileLink(\u0027Cell Magics.ipynb\u0027)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Alternatively, to generate links to all of the files in a directory, use the `FileLinks` object, passing `\u0027.\u0027` to indicate that we want links generated for the current working directory. Note that if there were other directories under the current directory, `FileLinks` would work in a recursive manner creating links to files in all sub-directories as well."
          ]
        },
        {
          "cell_type": "code",
          "execution_count": 33,
          "metadata": {
            "collapsed": false
          },
          "outputs": [
            {
              "data": {
                "text/html": [
                  "./\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Animations Using clear_output.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eAnimations Using clear_output.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Background Jobs.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eBackground Jobs.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Beyond Plain Python.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eBeyond Plain Python.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Capturing Output.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eCapturing Output.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Cell Magics.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eCell Magics.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Custom Display Logic.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eCustom Display Logic.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./example-demo.py\u0027 target\u003d\u0027_blank\u0027\u003eexample-demo.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./foo.py\u0027 target\u003d\u0027_blank\u0027\u003efoo.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Index.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eIndex.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./ipython-completion.bash\u0027 target\u003d\u0027_blank\u0027\u003eipython-completion.bash\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./ipython-get-history.py\u0027 target\u003d\u0027_blank\u0027\u003eipython-get-history.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./ipython-qtconsole.desktop\u0027 target\u003d\u0027_blank\u0027\u003eipython-qtconsole.desktop\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./ipython.desktop\u0027 target\u003d\u0027_blank\u0027\u003eipython.desktop\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./mod.py\u0027 target\u003d\u0027_blank\u0027\u003emod.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Plotting in the Notebook.ipynb\u0027 target\u003d\u0027_blank\u0027\u003ePlotting in the Notebook.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Raw Input in the Notebook.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eRaw Input in the Notebook.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Rich Output.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eRich Output.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Script Magics.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eScript Magics.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./SymPy.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eSymPy.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Terminal Usage.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eTerminal Usage.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./test.txt\u0027 target\u003d\u0027_blank\u0027\u003etest.txt\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Third Party Rich Output.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eThird Party Rich Output.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Trapezoid Rule.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eTrapezoid Rule.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./Working With External Code.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eWorking With External Code.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "./.ipynb_checkpoints/\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Animations Using clear_output-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eAnimations Using clear_output-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Background Jobs-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eBackground Jobs-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Beyond Plain Python-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eBeyond Plain Python-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Capturing Output-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eCapturing Output-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Cell Magics-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eCell Magics-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Custom Display Logic-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eCustom Display Logic-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Index-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eIndex-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Plotting in the Notebook-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003ePlotting in the Notebook-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Raw Input in the Notebook-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eRaw Input in the Notebook-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./.ipynb_checkpoints/Rich Output-checkpoint.ipynb\u0027 target\u003d\u0027_blank\u0027\u003eRich Output-checkpoint.ipynb\u003c/a\u003e\u003cbr\u003e\n",
                  "./__pycache__/\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./__pycache__/mod.cpython-34.pyc\u0027 target\u003d\u0027_blank\u0027\u003emod.cpython-34.pyc\u003c/a\u003e\u003cbr\u003e\n",
                  "./data/\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./data/flare.json\u0027 target\u003d\u0027_blank\u0027\u003eflare.json\u003c/a\u003e\u003cbr\u003e\n",
                  "./gui/\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./gui/gui-glut.py\u0027 target\u003d\u0027_blank\u0027\u003egui-glut.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./gui/gui-gtk.py\u0027 target\u003d\u0027_blank\u0027\u003egui-gtk.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./gui/gui-gtk3.py\u0027 target\u003d\u0027_blank\u0027\u003egui-gtk3.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./gui/gui-pyglet.py\u0027 target\u003d\u0027_blank\u0027\u003egui-pyglet.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./gui/gui-qt.py\u0027 target\u003d\u0027_blank\u0027\u003egui-qt.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./gui/gui-tk.py\u0027 target\u003d\u0027_blank\u0027\u003egui-tk.py\u003c/a\u003e\u003cbr\u003e\n",
                  "\u0026nbsp;\u0026nbsp;\u003ca href\u003d\u0027./gui/gui-wx.py\u0027 target\u003d\u0027_blank\u0027\u003egui-wx.py\u003c/a\u003e\u003cbr\u003e"
                ],
                "text/plain": [
                  "./\n",
                  "  Animations Using clear_output.ipynb\n",
                  "  Background Jobs.ipynb\n",
                  "  Beyond Plain Python.ipynb\n",
                  "  Capturing Output.ipynb\n",
                  "  Cell Magics.ipynb\n",
                  "  Custom Display Logic.ipynb\n",
                  "  example-demo.py\n",
                  "  foo.py\n",
                  "  Index.ipynb\n",
                  "  ipython-completion.bash\n",
                  "  ipython-get-history.py\n",
                  "  ipython-qtconsole.desktop\n",
                  "  ipython.desktop\n",
                  "  mod.py\n",
                  "  Plotting in the Notebook.ipynb\n",
                  "  Raw Input in the Notebook.ipynb\n",
                  "  Rich Output.ipynb\n",
                  "  Script Magics.ipynb\n",
                  "  SymPy.ipynb\n",
                  "  Terminal Usage.ipynb\n",
                  "  test.txt\n",
                  "  Third Party Rich Output.ipynb\n",
                  "  Trapezoid Rule.ipynb\n",
                  "  Working With External Code.ipynb\n",
                  "./.ipynb_checkpoints/\n",
                  "  Animations Using clear_output-checkpoint.ipynb\n",
                  "  Background Jobs-checkpoint.ipynb\n",
                  "  Beyond Plain Python-checkpoint.ipynb\n",
                  "  Capturing Output-checkpoint.ipynb\n",
                  "  Cell Magics-checkpoint.ipynb\n",
                  "  Custom Display Logic-checkpoint.ipynb\n",
                  "  Index-checkpoint.ipynb\n",
                  "  Plotting in the Notebook-checkpoint.ipynb\n",
                  "  Raw Input in the Notebook-checkpoint.ipynb\n",
                  "  Rich Output-checkpoint.ipynb\n",
                  "./__pycache__/\n",
                  "  mod.cpython-34.pyc\n",
                  "./data/\n",
                  "  flare.json\n",
                  "./gui/\n",
                  "  gui-glut.py\n",
                  "  gui-gtk.py\n",
                  "  gui-gtk3.py\n",
                  "  gui-pyglet.py\n",
                  "  gui-qt.py\n",
                  "  gui-tk.py\n",
                  "  gui-wx.py"
                ]
              },
              "execution_count": 33,
              "metadata": {},
              "output_type": "execute_result"
            }
          ],
          "source": [
            "FileLinks(\u0027.\u0027)"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## Rich output and security"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "The IPython Notebook allows arbitrary code execution in both the IPython kernel and in the browser, though HTML and JavaScript output. More importantly, because IPython has a JavaScript API for running code in the browser, HTML and JavaScript output can actually trigger code to be run in the kernel. This poses a significant security risk as it would allow IPython Notebooks to execute arbitrary code on your computers.\n",
            "\n",
            "To protect against these risks, the IPython Notebook has a security model that specifies how dangerous output is handled. Here is a short summary:\n",
            "\n",
            "* When you run code in the Notebook, all rich output is displayed.\n",
            "* When you open a notebook, rich output is only displayed if it doesn\u0027t contain security vulberabilities, ...\n",
            "* ... or if you have trusted a notebook, all rich output will run upon opening it.\n",
            "\n",
            "A full description of the IPython security model can be found on [this page](https://ipython.org/ipython-doc/dev/notebook/security.html)."
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "## Rich output and nbviewer"
          ]
        },
        {
          "cell_type": "markdown",
          "metadata": {},
          "source": [
            "Much of the power of the Notebook is that it enables users to share notebooks with each other using http://nbviewer.ipython.org, without installing IPython locally. As of IPython 2.0, notebooks rendered on nbviewer will display all output, including HTML and JavaScript. Furthermore, to provide a consistent JavaScript environment on the live Notebook and nbviewer, the following JavaScript libraries are loaded onto the nbviewer page, *before* the notebook and its output is displayed:\n",
            "\n",
            "* [jQuery](http://jquery.com/)\n",
            "* [RequireJS](http://requirejs.org/)\n",
            "\n",
            "Libraries such as [mpld3](http://mpld3.github.io/) use these capabilities to generate interactive visualizations that work on nbviewer."
          ]
        }
      ],
      "metadata": {
        "kernelspec": {
          "display_name": "Python 3",
          "language": "python",
          "name": "python3"
        },
        "language_info": {
          "codemirror_mode": {
            "name": "ipython",
            "version": 3
          },
          "file_extension": ".py",
          "mimetype": "text/x-python",
          "name": "python",
          "nbconvert_exporter": "python",
          "pygments_lexer": "ipython3",
          "version": "3.4.2"
        }
      },
      "nbformat": 4,
      "nbformat_minor": 0
    }

;