-- noinspection SqlResolveForFile


---- QueryCurrentSessionInfo ----
select sign(charindex('Azure', @@version))                     as is_azure,
       db_id()                                                 as db_id,
       db_name()                                               as db_name,
       schema_id()                                             as schema_id,
       schema_name()                                           as schema_name
;


---- QueryFingerprint ----
select cast(database_id as varchar(20)) + '.' + cast(cast(create_date as bigint) as varchar(20)) + '/' + cast(service_broker_guid as varchar(60)) + '@' + @@servername as fingerprint
from sys.databases
where #AZURE: name = db_name() #: database_id = db_id() #.
;


---- ListDatabases ----
select database_id as id, name, cast(databasepropertyex(name,'collation') as char) as collation_name
from sys.databases
order by case when #AZURE: name = db_name() #: database_id = db_id() #. then 1 else 0 end desc, database_id
;


---- ListSchemas ----
select schema_id as id, name
from #CAT.schemas
order by case when schema_id = schema_id() then 1 else 0 end desc, schema_id
;


---- CalculateCustomTypesCheckSums ----
with CT as ( select cast(1 + #ISTABLETYPE + is_assembly_type * 2 as tinyint) as type_kind,
                    abs(cast(checksum(user_type_id, name) as bigint)) as c_s
                    from #CAT.types
                    where schema_id = :schema_id
                      and is_user_defined >= 1 )
select type_kind, sum(c_s) as check_sum
from CT
group by type_kind
order by type_kind
;


---- RetrieveAliasTypes ----
with A as ( select user_type_id as type_id, system_type_id, name,
                   max_length, precision, scale, is_nullable
            from #CAT.types
            where schema_id = :schema_id
              and is_user_defined >= 1
              and not (is_assembly_type >= 1)
              and not (#ISTABLETYPE >= 1) )
select A.type_id, A.name,
       B.name as base_name,
       A.max_length, A.precision, A.scale, A.is_nullable
from A left outer join #CAT.types B on A.system_type_id = B.user_type_id
order by A.type_id
;


---- RetrieveTableTypes ----
with T as ( select user_type_id as type_id,
                   type_table_object_id as object_id,
                   name
            from #CAT.table_types
            where schema_id = :schema_id
              and is_user_defined = 1
              and #ISTABLETYPE = 1 )
select T.type_id, O.object_id, T.name,
       O.create_date, O.modify_date
from T join #CAT.all_objects O on T.object_id = O.object_id
order by create_date, type_id
;


---- ListExistentObjects ----
select object_id
from #CAT.all_objects
where schema_id = :schema_id
;


---- ListExistentIndices ----
select cast(I.object_id as bigint) * 100000 + (I.index_id % 30000) as Combi_Id
from #CAT.all_objects T,
     #CAT.indexes I
where T.schema_id = :schema_id
  and T.type in ('U','S','IT','V','TT')
  and T.object_id = I.object_id
  and I.type != 0
;


---- ListExistentFulltextIndices ----
select F.object_id
from #CAT.fulltext_indexes F,
     #CAT.all_objects T
where T.schema_id = :schema_id
  and T.object_id = F.object_id
;


---- RetrieveStates ----
select object_id, modify_date
from #CAT.all_objects
where object_id in (:[*object_ids])
;


---- RetrieveObjects ----
select object_id,
       name,
       rtrim(type) as type,
       create_date,
       modify_date,
       is_ms_shipped
from #CAT.all_objects
where type in ('U','S','IT','V','FN','IF','P','X','PC','TF','TT')
  and schema_id = :schema_id
  and modify_date >= :since
order by create_date
;


---- RetrieveColumns ----
with T as ( select object_id, create_date
            from #CAT.all_objects
            where schema_id = :schema_id
              and type in ('U','S','IT','V','TT','IF','FT','TF')
              and modify_date >= :since )
select T.create_date,
       C.object_id, C.column_id as column_position, C.name,
       Y.name as type_name,
       C.max_length, C.precision, C.scale, C.collation_name,
       C.is_nullable, C.is_computed,
       cast(I.seed_value as bigint) as identity_seed_value,
       cast(I.increment_value as bigint) as identity_increment_value,
       cast(I.last_value as bigint) as identity_last_value,
       D.name as default_constraint_name,
       D.is_system_named as default_constraint_name_is_surrogate,
       D.object_id as default_constraint_object_id,
       D.modify_date default_constraint_modify_date,
       coalesce(D.definition, X.definition) as default_expression,
       #V13: C.is_hidden #: 0 #. as is_hidden
from T join #CAT.all_columns C on T.object_id = C.object_id
       left outer join #CAT.types Y on C.system_type_id = Y.system_type_id and C.user_type_id = Y.user_type_id
       left outer join #CAT.identity_columns I on C.object_id = I.object_id and C.column_id = I.column_id
       left outer join #CAT.default_constraints D on C.object_id = D.parent_object_id and C.column_id = D.parent_column_id
       left outer join #CAT.computed_columns X on C.object_id = X.object_id and C.column_id = X.column_id
order by T.create_date, C.object_id, C.column_id
;


---- RetrieveTypeColumns ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              and is_user_defined = 1 )
select C.object_id, C.column_id as column_position, C.name,
       Y.name as type_name,
       C.max_length, C.precision, C.scale, C.collation_name,
       C.is_nullable, C.is_computed,
       cast(I.seed_value as bigint) as identity_seed_value,
       cast(I.increment_value as bigint) as identity_increment_value,
       cast(I.last_value as bigint) as identity_last_value,
       coalesce(D.definition, X.definition) as default_expression
from T join #CAT.all_columns C on T.object_id = C.object_id
       left outer join #CAT.types Y on C.system_type_id = Y.system_type_id and C.user_type_id = Y.user_type_id
       left outer join #CAT.identity_columns I on C.object_id = I.object_id and C.column_id = I.column_id
       left outer join #CAT.default_constraints D on C.object_id = D.parent_object_id and C.column_id = D.parent_column_id
       left outer join #CAT.computed_columns X on C.object_id = X.object_id and C.column_id = X.column_id
order by C.object_id, C.column_id
;


---- RetrieveIndices ----
select T.object_id,
       I.index_id % 30000 as position,
       I.name,
       I.type,
       #V10: I.filter_definition, #.
       I.is_unique,
       I.is_disabled
from #CAT.all_objects T,
     #CAT.indexes I
where T.schema_id = :schema_id
  and T.type in ('U','S','IT','V','TT')
  and T.modify_date >= :since
  and T.object_id = I.object_id
  and I.type != 0
order by T.create_date, I.object_id, I.index_id
;


---- RetrieveFulltextIndices ----
select F.object_id,
       F.unique_index_id
from #CAT.all_objects T,
     #CAT.fulltext_indexes F
where T.schema_id = :schema_id
  and T.modify_date >= :since
  and T.object_id = F.object_id
;


---- RetrieveTypeIndices ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              and is_user_defined = 1 )
select T.object_id,
       I.index_id % 30000 as position,
       I.name,
       I.type,
       #V10: I.filter_definition, #.
       I.is_unique,
       I.is_disabled
from T join #CAT.indexes I on T.object_id = I.object_id
where I.type != 0
order by I.object_id, I.index_id
;


---- RetrieveIndexColumns ----
select T.object_id,
       X.index_id % 30000 as index_position,
       C.name,
       X.is_descending_key as is_desc,
       X.is_included_column
from #CAT.all_objects T,
     #CAT.indexes I,
     #CAT.index_columns X,
     #CAT.all_columns C
where T.schema_id = :schema_id
  and T.type in ('U','S','IT','V','TT')
  and T.modify_date >= :since
  and T.object_id = I.object_id
  and I.type != 0
  and I.object_id = X.object_id
  and I.index_id = X.index_id
  and X.object_id = C.object_id
  and X.column_id = C.column_id
order by T.create_date, X.object_id, X.index_id, X.index_column_id
;


---- RetrieveFulltextIndexColumns ----
select T.object_id,
       C.name
from sys.all_objects T,
     sys.fulltext_index_columns X,
     sys.all_columns C
where T.schema_id = :schema_id
  and T.modify_date >= :since
  and T.object_id = X.object_id
  and X.object_id = C.object_id
  and X.column_id = C.column_id
;


---- RetrieveTypeIndexColumns ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              and is_user_defined = 1 )
select T.object_id,
       X.index_id % 30000 as index_position,
       C.name,
       X.is_descending_key as is_desc,
       X.is_included_column
from T,
     #CAT.indexes I,
     #CAT.index_columns X,
     #CAT.all_columns C
where T.object_id = I.object_id
  and I.type != 0
  and I.object_id = X.object_id
  and I.index_id = X.index_id
  and X.object_id = C.object_id
  and X.column_id = C.column_id
order by X.object_id, X.index_id, X.index_column_id
;



---- RetrieveKeys ----
select T.object_id as table_id,
       K.object_id as key_id,
       K.unique_index_id % 30000 as index_position,
       K.name,
       case when K.type = 'PK' then 1 else 0 end as is_primary,
       K.is_system_named,
       K.create_date,
       K.modify_date
from #CAT.all_objects T
     join #CAT.key_constraints K on T.object_id = K.parent_object_id and T.schema_id = K.schema_id
where T.schema_id = :schema_id
  and T.modify_date >= :since
order by T.create_date, T.object_id, is_primary desc, K.create_date, K.object_id
;


---- RetrieveTypeKeys ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              and is_user_defined = 1 )
select T.object_id as table_id,
       K.object_id as key_id,
       K.unique_index_id % 30000 as index_position,
       K.name,
       case when K.type = 'PK' then 1 else 0 end as is_primary,
       K.is_system_named,
       K.create_date,
       K.modify_date
from T join #CAT.key_constraints K on T.object_id = K.parent_object_id
order by T.object_id, is_primary desc, K.create_date, K.object_id
;


---- RetrieveForeignKeys ----
select T.object_id as table_id,
       F.object_id as constraint_id,
       F.name,
       F.is_system_named,
       F.referenced_object_id,
       X.name as referenced_object_name,
       X.schema_id as referenced_schema_id,
       F.key_index_id,
       F.is_disabled,
       F.update_referential_action,
       F.delete_referential_action,
       F.create_date,
       F.modify_date
from #CAT.all_objects T
     join #CAT.foreign_keys F on T.object_id = F.parent_object_id and T.schema_id = F.schema_id
     left join #CAT.objects X on F.referenced_object_id = X.object_id
where T.schema_id = :schema_id
  and T.modify_date >= :since
order by T.create_date, T.object_id
;


---- RetrieveForeignKeyColumns ----
select K.constraint_object_id as constraint_id,
       CD.name as domestic_column_name,
       CR.name as reference_column_name
from #CAT.all_objects T
     join #CAT.foreign_keys F on T.object_id = F.parent_object_id and T.schema_id = F.schema_id
     join #CAT.foreign_key_columns K on F.object_id = K.constraint_object_id
     left outer join #CAT.all_columns CD on K.parent_object_id = CD.object_id and K.parent_column_id = CD.column_id
     left outer join #CAT.all_columns CR on K.referenced_object_id = CR.object_id and K.referenced_column_id = CR.column_id
where T.schema_id = :schema_id
  and T.modify_date >= :since
  and CD.name is not null
  and CR.name is not null
order by T.create_date, T.object_id, K.constraint_object_id, K.constraint_column_id
;


---- RetrieveCheckConstraints ----
select T.object_id as table_id,
       X.object_id as constraint_id,
       X.name,
       X.is_system_named,
       C.name as column_name,
       X.definition,
       X.create_date,
       X.modify_date
from #CAT.all_objects T
     join #CAT.check_constraints X on T.object_id = X.parent_object_id and T.schema_id = X.schema_id
     left outer join #CAT.all_columns C on X.parent_object_id = C.object_id and X.parent_column_id = C.column_id
where T.schema_id = :schema_id
  and T.modify_date >= :since
order by T.create_date, T.object_id, X.parent_column_id, X.create_date, X.object_id
;


---- RetrieveTypeCheckConstraints ----
with T as ( select type_table_object_id as object_id
            from #CAT.table_types
            where schema_id = :schema_id
              and is_user_defined = 1 )
select T.object_id as table_id,
       X.object_id as constraint_id,
       X.name,
       X.is_system_named,
       C.name as column_name,
       X.definition,
       X.create_date,
       X.modify_date
from T
     join #CAT.check_constraints X on T.object_id = X.parent_object_id
     left outer join #CAT.all_columns C on X.parent_object_id = C.object_id and X.parent_column_id = C.column_id
order by T.object_id, X.parent_column_id, X.create_date, X.object_id
;


---- RetrieveArguments ----
select P.object_id,
       P.parameter_id % 30000 as position,
       P.name,
       T.name as type_name,
       P.max_length,
       P.precision,
       P.scale,
       P.is_output,
       P.is_cursor_ref,
       P.has_default_value
from #CAT.all_objects R
     join #CAT.all_parameters P on R.object_id = P.object_id
     left outer join #CAT.types T on P.system_type_id = T.system_type_id and P.user_type_id = T.user_type_id
where R.schema_id = :schema_id
  and R.type in ('FN','IF','TF','P','X','PC')
  and R.modify_date >= :since
order by R.create_date, P.object_id, P.parameter_id
;


---- RetrieveTriggers ----
select O.parent_object_id as table_id,
       O.object_id as trigger_id,
       O.name,
       O.create_date,
       O.modify_date,
       ( select sum(power(2,type-1))
         from #CAT.trigger_events
         where object_id = O.object_id
           and type between 1 and 3
         group by object_id ) as events,
       ( select sum(power(2,type-1))
         from #CAT.trigger_events
         where object_id = O.object_id
           and is_first = 1
           and type between 1 and 3
         group by object_id ) as first_events,
       ( select sum(power(2,type-1))
         from #CAT.trigger_events
         where object_id = O.object_id
           and is_last = 1
           and type between 1 and 3
         group by object_id ) as last_events,
       T.is_instead_of_trigger,
       T.is_not_for_replication,
       T.is_disabled,
       C.encrypted
from #CAT.objects O
     inner join #CAT.triggers T on O.object_id = T.object_id
     left outer join #CAT.syscomments C on O.object_id = C.id
where O.schema_id = :schema_id
  and O.modify_date >= :since
  and O.type in ('TA','TR')
;


---- RetrieveSynonyms ----
select object_id,
       name,
       create_date,
       modify_date,
       parsename(base_object_name, 3) as origin_db_name,
       parsename(base_object_name, 2) as origin_schema_name,
       parsename(base_object_name, 1) as origin_object_name,
       object_id(base_object_name)    as origin_id
from #CAT.synonyms
where schema_id = :schema_id
  and modify_date >= :since
;


---- CalculateDescriptionsChecksum ----
select sum(abs(cast(checksum(major_id, minor_id, cast(value as varchar(40))) as bigint))) as checksum
from #CAT.extended_properties X
     left join #CAT.all_objects O
        on O.object_id = X.major_id
       and O.type in ('U','S','IT','V','FN','IF','P','X','PC','TF','TT','C','F','D','PK','SN','SO','UQ','TR')
       and X.class <> 6
       and X.class <> 3
     left join #CAT.types T
        on T.user_type_id = X.major_id
       and X.class = 6
where coalesce(O.schema_id, T.schema_id, case when X.class = 3 then X.major_id else null end) = :schema_id
  and X.name = N'MS_Description'
  and X.value is not null
;


---- RetrieveDescriptions ----
select X.major_id object_id,
       case when X.class = 7 then -1 else 1 end*(X.minor_id % 30000) as position,
       cast(X.value as varchar(max)) as description
from #CAT.extended_properties X
     left join #CAT.all_objects O
        on O.object_id = X.major_id
       and O.type in ('U','S','IT','V','FN','IF','P','X','PC','TF','TT','C','F','D','PK','SN','SO','UQ','TR')
       and X.class <> 6
       and X.class <> 3
     left join #CAT.types T
        on T.user_type_id = X.major_id
       and X.class = 6
where (coalesce(O.schema_id, T.schema_id, case when X.class = 3 then X.major_id else null end) = :schema_id)
  and X.name = N'MS_Description'
  and X.value is not null
;


---- RetrieveSources ----
select O.parent_object_id as parent_id,
       M.object_id,
       case when O.type = 'TR' or O.type = 'TA' then 1 else 0 end as is_trigger,
       M.definition as text
from #CAT.sql_modules M
     inner join #CAT.objects O on M.object_id = O.object_id
where O.schema_id = :schema_id
  and O.modify_date >= :src_since
;


---- RetrieveSourcesWithSys ----
select O.parent_object_id as parent_id,
       M.object_id,
       case when O.type = 'TR' or O.type = 'TA' then 1 else 0 end as is_trigger,
       M.definition as text
from #CAT.all_sql_modules M
     inner join #CAT.all_objects O on M.object_id = O.object_id
where O.schema_id = :schema_id
  and O.modify_date >= :src_since
;

