/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#pragma once

#include <glib.h>

/**
 * SECTION: version
 * @section_id: version-macros
 * @title: Version related macros
 * @include: adbc-glib/adbc-glib.h
 *
 * ADBC GLib provides macros that can be used by C pre-processor.
 * They are useful to check version related things at compile time.
 */

/**
 * GADBC_VERSION_MAJOR:
 *
 * The major version.
 *
 * Since: 0.1.0
 */
#define GADBC_VERSION_MAJOR (1)

/**
 * GADBC_VERSION_MINOR:
 *
 * The minor version.
 *
 * Since: 0.1.0
 */
#define GADBC_VERSION_MINOR (10)

/**
 * GADBC_VERSION_MICRO:
 *
 * The micro version.
 *
 * Since: 0.1.0
 */
#define GADBC_VERSION_MICRO (0)

/**
 * GADBC_VERSION_TAG:
 *
 * The version tag. Normally, it's an empty string. It's "SNAPSHOT"
 * for snapshot version.
 *
 * Since: 1.7.0
 */
#define GADBC_VERSION_TAG   ""

/**
 * GADBC_VERSION_CHECK:
 * @major: A major version to check for.
 * @minor: A minor version to check for.
 * @micro: A micro version to check for.
 *
 * You can use this macro in C pre-processor.
 *
 * Returns: %TRUE if the compile time ADBC GLib version is the same as
 *   or newer than the passed version, %FALSE otherwise.
 *
 * Since: 0.1.0
 */
#define GADBC_VERSION_CHECK(major, minor, micro)       \
  (GADBC_VERSION_MAJOR > (major) ||                    \
   (GADBC_VERSION_MAJOR == (major) &&                  \
    GADBC_VERSION_MINOR > (minor)) ||                  \
   (GADBC_VERSION_MAJOR == (major) &&                  \
    GADBC_VERSION_MINOR == (minor) &&                  \
    GADBC_VERSION_MICRO >= (micro)))

/**
 * GADBC_DISABLE_DEPRECATION_WARNINGS:
 *
 * If this macro is defined, no deprecated warnings are produced.
 *
 * You must define this macro before including the
 * adbc-glib/adbc-glib.h header.
 *
 * Since: 0.1.0
 */

#ifdef GADBC_DISABLE_DEPRECATION_WARNINGS
#  define GADBC_DEPRECATED
#  define GADBC_DEPRECATED_FOR(function)
#  define GADBC_UNAVAILABLE(major, minor)
#else
#  define GADBC_DEPRECATED G_DEPRECATED
#  define GADBC_DEPRECATED_FOR(function) G_DEPRECATED_FOR(function)
#  define GADBC_UNAVAILABLE(major, minor) G_UNAVAILABLE(major, minor)
#endif

/**
 * GADBC_VERSION_1_10:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.10.0
 */
#define GADBC_VERSION_1_10 G_ENCODE_VERSION(1, 10)

/**
 * GADBC_VERSION_1_9:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.9.0
 */
#define GADBC_VERSION_1_9 G_ENCODE_VERSION(1, 9)

/**
 * GADBC_VERSION_1_8:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.8.0
 */
#define GADBC_VERSION_1_8 G_ENCODE_VERSION(1, 8)

/**
 * GADBC_VERSION_1_7:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.7.0
 */
#define GADBC_VERSION_1_7 G_ENCODE_VERSION(1, 7)

/**
 * GADBC_VERSION_1_6:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.6.0
 */
#define GADBC_VERSION_1_6 G_ENCODE_VERSION(1, 6)

/**
 * GADBC_VERSION_1_5:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.5.0
 */
#define GADBC_VERSION_1_5 G_ENCODE_VERSION(1, 5)

/**
 * GADBC_VERSION_1_4:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.4.0
 */
#define GADBC_VERSION_1_4 G_ENCODE_VERSION(1, 4)

/**
 * GADBC_VERSION_1_3:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.3.0
 */
#define GADBC_VERSION_1_3 G_ENCODE_VERSION(1, 3)

/**
 * GADBC_VERSION_1_2:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.2.0
 */
#define GADBC_VERSION_1_2 G_ENCODE_VERSION(1, 2)

/**
 * GADBC_VERSION_1_1:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.1.0
 */
#define GADBC_VERSION_1_1 G_ENCODE_VERSION(1, 1)

/**
 * GADBC_VERSION_1_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.0.0
 */
#define GADBC_VERSION_1_0 G_ENCODE_VERSION(1, 0)

/**
 * GADBC_VERSION_0_11:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.11.0
 */
#define GADBC_VERSION_0_11 G_ENCODE_VERSION(0, 11)

/**
 * GADBC_VERSION_0_10:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.10.0
 */
#define GADBC_VERSION_0_10 G_ENCODE_VERSION(0, 10)

/**
 * GADBC_VERSION_0_9:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.9.0
 */
#define GADBC_VERSION_0_9 G_ENCODE_VERSION(0, 9)

/**
 * GADBC_VERSION_0_8:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.8.0
 */
#define GADBC_VERSION_0_8 G_ENCODE_VERSION(0, 8)

/**
 * GADBC_VERSION_0_7:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.7.0
 */
#define GADBC_VERSION_0_7 G_ENCODE_VERSION(0, 7)

/**
 * GADBC_VERSION_0_6:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.6.0
 */
#define GADBC_VERSION_0_6 G_ENCODE_VERSION(0, 6)

/**
 * GADBC_VERSION_0_5:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.5.0
 */
#define GADBC_VERSION_0_5 G_ENCODE_VERSION(0, 5)

/**
 * GADBC_VERSION_0_4:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.4.0
 */
#define GADBC_VERSION_0_4 G_ENCODE_VERSION(0, 4)

/**
 * GADBC_VERSION_0_3:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.3.0
 */
#define GADBC_VERSION_0_3 G_ENCODE_VERSION(0, 3)

/**
 * GADBC_VERSION_0_2:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.2.0
 */
#define GADBC_VERSION_0_2 G_ENCODE_VERSION(0, 2)

/**
 * GADBC_VERSION_0_1:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.1.0
 */
#define GADBC_VERSION_0_1 G_ENCODE_VERSION(0, 1)

/**
 * GADBC_VERSION_MIN_REQUIRED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GADBC_VERSION_0_1.
 *
 * If you use any functions that is defined by newer version than
 * %GADBC_VERSION_MIN_REQUIRED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * adbc-glib/adbc-glib.h header.
 *
 * Since: 0.1.0
 */
#ifndef GADBC_VERSION_MIN_REQUIRED
#  define GADBC_VERSION_MIN_REQUIRED                           \
  G_ENCODE_VERSION(GADBC_VERSION_MAJOR, GADBC_VERSION_MINOR)
#endif

/**
 * GADBC_VERSION_MAX_ALLOWED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GADBC_VERSION_0_1.
 *
 * If you use any functions that is defined by newer version than
 * %GADBC_VERSION_MAX_ALLOWED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * adbc-glib/adbc-glib.h header.
 *
 * Since: 0.1.0
 */
#ifndef GADBC_VERSION_MAX_ALLOWED
#  define GADBC_VERSION_MAX_ALLOWED            \
  G_ENCODE_VERSION(GADBC_VERSION_MAJOR, GADBC_VERSION_MINOR)
#endif

#if (defined(_WIN32) || defined(__CYGWIN__)) && defined(_MSC_VER) &&   !defined(GADBC_STATIC_COMPILATION)
#  define GADBC_EXPORT __declspec(dllexport)
#  define GADBC_IMPORT __declspec(dllimport)
#else
#  define GADBC_EXPORT
#  define GADBC_IMPORT
#endif

#ifdef GADBC_COMPILATION
#  define GADBC_API GADBC_EXPORT
#else
#  define GADBC_API GADBC_IMPORT
#endif

#define GADBC_EXTERN GADBC_API extern

#define GADBC_AVAILABLE_IN_ALL GADBC_EXTERN

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_10
#  define GADBC_DEPRECATED_IN_1_10               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_10_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_10
#  define GADBC_DEPRECATED_IN_1_10_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_10
#  define GADBC_AVAILABLE_IN_1_10 GADBC_EXTERN GADBC_UNAVAILABLE(1, 10)
#else
#  define GADBC_AVAILABLE_IN_1_10 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_9
#  define GADBC_DEPRECATED_IN_1_9               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_9_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_9
#  define GADBC_DEPRECATED_IN_1_9_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_9
#  define GADBC_AVAILABLE_IN_1_9 GADBC_EXTERN GADBC_UNAVAILABLE(1, 9)
#else
#  define GADBC_AVAILABLE_IN_1_9 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_8
#  define GADBC_DEPRECATED_IN_1_8               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_8_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_8
#  define GADBC_DEPRECATED_IN_1_8_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_8
#  define GADBC_AVAILABLE_IN_1_8 GADBC_EXTERN GADBC_UNAVAILABLE(1, 8)
#else
#  define GADBC_AVAILABLE_IN_1_8 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_7
#  define GADBC_DEPRECATED_IN_1_7               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_7_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_7
#  define GADBC_DEPRECATED_IN_1_7_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_7
#  define GADBC_AVAILABLE_IN_1_7 GADBC_EXTERN GADBC_UNAVAILABLE(1, 7)
#else
#  define GADBC_AVAILABLE_IN_1_7 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_6
#  define GADBC_DEPRECATED_IN_1_6               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_6_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_6
#  define GADBC_DEPRECATED_IN_1_6_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_6
#  define GADBC_AVAILABLE_IN_1_6 GADBC_EXTERN GADBC_UNAVAILABLE(1, 6)
#else
#  define GADBC_AVAILABLE_IN_1_6 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_5
#  define GADBC_DEPRECATED_IN_1_5               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_5_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_5
#  define GADBC_DEPRECATED_IN_1_5_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_5
#  define GADBC_AVAILABLE_IN_1_5 GADBC_EXTERN GADBC_UNAVAILABLE(1, 5)
#else
#  define GADBC_AVAILABLE_IN_1_5 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_4
#  define GADBC_DEPRECATED_IN_1_4               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_4_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_4
#  define GADBC_DEPRECATED_IN_1_4_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_4
#  define GADBC_AVAILABLE_IN_1_4 GADBC_EXTERN GADBC_UNAVAILABLE(1, 4)
#else
#  define GADBC_AVAILABLE_IN_1_4 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_3
#  define GADBC_DEPRECATED_IN_1_3               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_3_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_3
#  define GADBC_DEPRECATED_IN_1_3_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_3
#  define GADBC_AVAILABLE_IN_1_3 GADBC_EXTERN GADBC_UNAVAILABLE(1, 3)
#else
#  define GADBC_AVAILABLE_IN_1_3 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_2
#  define GADBC_DEPRECATED_IN_1_2               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_2_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_2
#  define GADBC_DEPRECATED_IN_1_2_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_2
#  define GADBC_AVAILABLE_IN_1_2 GADBC_EXTERN GADBC_UNAVAILABLE(1, 2)
#else
#  define GADBC_AVAILABLE_IN_1_2 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_1
#  define GADBC_DEPRECATED_IN_1_1               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_1_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_1
#  define GADBC_DEPRECATED_IN_1_1_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_1
#  define GADBC_AVAILABLE_IN_1_1 GADBC_EXTERN GADBC_UNAVAILABLE(1, 1)
#else
#  define GADBC_AVAILABLE_IN_1_1 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_1_0
#  define GADBC_DEPRECATED_IN_1_0               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_1_0_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_1_0
#  define GADBC_DEPRECATED_IN_1_0_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_1_0
#  define GADBC_AVAILABLE_IN_1_0 GADBC_EXTERN GADBC_UNAVAILABLE(1, 0)
#else
#  define GADBC_AVAILABLE_IN_1_0 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_11
#  define GADBC_DEPRECATED_IN_0_11               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_11_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_11
#  define GADBC_DEPRECATED_IN_0_11_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_11
#  define GADBC_AVAILABLE_IN_0_11 GADBC_EXTERN GADBC_UNAVAILABLE(0, 11)
#else
#  define GADBC_AVAILABLE_IN_0_11 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_10
#  define GADBC_DEPRECATED_IN_0_10               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_10_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_10
#  define GADBC_DEPRECATED_IN_0_10_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_10
#  define GADBC_AVAILABLE_IN_0_10 GADBC_EXTERN GADBC_UNAVAILABLE(0, 10)
#else
#  define GADBC_AVAILABLE_IN_0_10 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_9
#  define GADBC_DEPRECATED_IN_0_9               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_9_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_9
#  define GADBC_DEPRECATED_IN_0_9_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_9
#  define GADBC_AVAILABLE_IN_0_9 GADBC_EXTERN GADBC_UNAVAILABLE(0, 9)
#else
#  define GADBC_AVAILABLE_IN_0_9 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_8
#  define GADBC_DEPRECATED_IN_0_8               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_8_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_8
#  define GADBC_DEPRECATED_IN_0_8_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_8
#  define GADBC_AVAILABLE_IN_0_8 GADBC_EXTERN GADBC_UNAVAILABLE(0, 8)
#else
#  define GADBC_AVAILABLE_IN_0_8 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_7
#  define GADBC_DEPRECATED_IN_0_7               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_7_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_7
#  define GADBC_DEPRECATED_IN_0_7_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_7
#  define GADBC_AVAILABLE_IN_0_7 GADBC_EXTERN GADBC_UNAVAILABLE(0, 7)
#else
#  define GADBC_AVAILABLE_IN_0_7 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_6
#  define GADBC_DEPRECATED_IN_0_6               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_6_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_6
#  define GADBC_DEPRECATED_IN_0_6_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_6
#  define GADBC_AVAILABLE_IN_0_6 GADBC_EXTERN GADBC_UNAVAILABLE(0, 6)
#else
#  define GADBC_AVAILABLE_IN_0_6 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_5
#  define GADBC_DEPRECATED_IN_0_5               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_5_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_5
#  define GADBC_DEPRECATED_IN_0_5_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_5
#  define GADBC_AVAILABLE_IN_0_5 GADBC_EXTERN GADBC_UNAVAILABLE(0, 5)
#else
#  define GADBC_AVAILABLE_IN_0_5 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_4
#  define GADBC_DEPRECATED_IN_0_4               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_4_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_4
#  define GADBC_DEPRECATED_IN_0_4_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_4
#  define GADBC_AVAILABLE_IN_0_4 GADBC_EXTERN GADBC_UNAVAILABLE(0, 4)
#else
#  define GADBC_AVAILABLE_IN_0_4 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_3
#  define GADBC_DEPRECATED_IN_0_3               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_3_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_3
#  define GADBC_DEPRECATED_IN_0_3_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_3
#  define GADBC_AVAILABLE_IN_0_3 GADBC_EXTERN GADBC_UNAVAILABLE(0, 3)
#else
#  define GADBC_AVAILABLE_IN_0_3 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_2
#  define GADBC_DEPRECATED_IN_0_2               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_2_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_2
#  define GADBC_DEPRECATED_IN_0_2_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_2
#  define GADBC_AVAILABLE_IN_0_2 GADBC_EXTERN GADBC_UNAVAILABLE(0, 2)
#else
#  define GADBC_AVAILABLE_IN_0_2 GADBC_EXTERN
#endif

#if GADBC_VERSION_MIN_REQUIRED >= GADBC_VERSION_0_1
#  define GADBC_DEPRECATED_IN_0_1               GADBC_DEPRECATED
#  define GADBC_DEPRECATED_IN_0_1_FOR(function) GADBC_DEPRECATED_FOR(function)
#else
#  define GADBC_DEPRECATED_IN_0_1
#  define GADBC_DEPRECATED_IN_0_1_FOR(function)
#endif

#if GADBC_VERSION_MAX_ALLOWED < GADBC_VERSION_0_1
#  define GADBC_AVAILABLE_IN_0_1 GADBC_EXTERN GADBC_UNAVAILABLE(0, 1)
#else
#  define GADBC_AVAILABLE_IN_0_1 GADBC_EXTERN
#endif
