// Created on: 1995-09-18
// Created by: Bruno DUMORTIER
// Copyright (c) 1995-1999 Matra Datavision
// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#ifndef _BRepOffsetAPI_MakeOffset_HeaderFile
#define _BRepOffsetAPI_MakeOffset_HeaderFile

#include <Standard.hxx>
#include <Standard_DefineAlloc.hxx>

#include <TopoDS_Face.hxx>
#include <TopTools_ListOfShape.hxx>
#include <BRepFill_ListOfOffsetWire.hxx>
#include <BRepBuilderAPI_MakeShape.hxx>
class TopoDS_Wire;
class TopoDS_Shape;


//! Describes algorithms for offsetting wires from a set of
//! wires contained in a planar face.
//! A MakeOffset object provides a framework for:
//! - defining the construction of an offset,
//! - implementing the construction algorithm, and
//! - consulting the result.
class Standard_EXPORT BRepOffsetAPI_MakeOffset  : public BRepBuilderAPI_MakeShape
{
public:

  DEFINE_STANDARD_ALLOC

  
  //! Constructs an algorithm for creating an empty offset
  Standard_EXPORT BRepOffsetAPI_MakeOffset();
  
  //! Constructs an algorithm for creating an algorithm
  //! to build parallels to the spine Spine
  Standard_EXPORT BRepOffsetAPI_MakeOffset(const TopoDS_Face& Spine, const GeomAbs_JoinType Join = GeomAbs_Arc, const Standard_Boolean IsOpenResult = Standard_False);
  
  //! Initializes the algorithm to construct parallels to the spine Spine.
  //! Join defines the type of parallel generated by the
  //! salient vertices of the spine.
  //! The default type is GeomAbs_Arc where the vertices generate
  //! sections of a circle.
  //! If join type is GeomAbs_Intersection, the edges that
  //! intersect in a salient vertex generate the edges
  //! prolonged until intersection.
  Standard_EXPORT void Init (const TopoDS_Face& Spine, const GeomAbs_JoinType Join = GeomAbs_Arc, const Standard_Boolean IsOpenResult = Standard_False);
  
  Standard_EXPORT BRepOffsetAPI_MakeOffset(const TopoDS_Wire& Spine, const GeomAbs_JoinType Join = GeomAbs_Arc, const Standard_Boolean IsOpenResult = Standard_False);
  
  //! Initialize the evaluation of Offsetting.
  Standard_EXPORT void Init (const GeomAbs_JoinType Join = GeomAbs_Arc, const Standard_Boolean IsOpenResult = Standard_False);
  
  //! Set approximation flag
  //! for convertion input contours into ones consisting of
  //! 2D circular arcs and 2D linear segments only.
  Standard_EXPORT void SetApprox (const Standard_Boolean ToApprox);
  
  //! Initializes the algorithm to construct parallels to the wire Spine.
  Standard_EXPORT void AddWire (const TopoDS_Wire& Spine);
  
  //! Computes a parallel to the spine at distance Offset and
  //! at an altitude Alt from the plane of the spine in relation
  //! to the normal to the spine.
  //! Exceptions: StdFail_NotDone if the offset is not built.
  Standard_EXPORT void Perform (const Standard_Real Offset, const Standard_Real Alt = 0.0);
  
  //! Builds the resulting shape (redefined from MakeShape).
  Standard_EXPORT virtual void Build(const Message_ProgressRange& theRange = Message_ProgressRange()) Standard_OVERRIDE;
  
  //! returns a list of the created shapes
  //! from the shape <S>.
  Standard_EXPORT virtual const TopTools_ListOfShape& Generated (const TopoDS_Shape& S) Standard_OVERRIDE;

  //! Converts each wire of the face into contour consisting only of
  //! arcs and segments. New 3D curves are built too.
  Standard_EXPORT static TopoDS_Face ConvertFace (const TopoDS_Face& theFace,
                                                  const Standard_Real theAngleTolerance);


protected:





private:



  Standard_Boolean myIsInitialized;
  Standard_Boolean myLastIsLeft;
  GeomAbs_JoinType myJoin;
  Standard_Boolean myIsOpenResult;
  Standard_Boolean myIsToApprox;
  TopoDS_Face myFace;
  TopTools_ListOfShape myWires;
  BRepFill_ListOfOffsetWire myLeft;
  BRepFill_ListOfOffsetWire myRight;


};







#endif // _BRepOffsetAPI_MakeOffset_HeaderFile
