%feature("docstring") OT::GeneralizedParetoFactory
"Generalized Pareto factory.

See also
--------
DistributionFactory, GeneralizedPareto

Notes
-----
Several estimators are available to build a :class:`~openturns.GeneralizedPareto` distribution
from a scalar sample (see [matthys2003]_ for the theory).

The chosen algorithm depends on the size of the sample compared to the
:class:`~openturns.ResourceMap` key `GeneralizedParetoFactory-SmallSize`:

- When the size of the sample is small the probability weighted method of moments is used,
  with a fallback on the exponential regression method
- When the size of the sample is large the exponential regression method is used,
  with a fallback on the probability weighted moment method

**Moments based estimator:**

Lets denote:

- :math:`\\displaystyle \\overline{x}_n = \\frac{1}{n} \\sum_{i=1}^n x_i` the empirical
  mean of the sample, 
- :math:`\\displaystyle s_n^2 = \\frac{1}{n-1} \\sum_{i=1}^n (x_i - \\overline{x}_n)^2`
  its empirical variance.

Then we estimate :math:`(\\hat{\\sigma}_n, \\hat{\\xi}_n, \\hat{u}_n)` using:

.. math::
    :nowrap:
    :label: gpd_moment_estimator

    \\begin{eqnarray*}
        \\hat{u}_n &= x_{(1)} - \\frac{x_{(1)}}{2 + n} \\\\
        \\hat{\\xi}_n &= -\\dfrac{1}{2}\\left(\\dfrac{(\\overline{x}_n - \\hat{u}_n)^2}{s_n^2}-1\\right) \\\\
        \\hat{\\sigma}_n &= \\dfrac{(\\overline{x}_n- \\hat{u}_n)}{2}\\left(\\dfrac{(\\overline{x}_n- \\hat{u}_n)^2}{s_n^2}+1\\right)
    \\end{eqnarray*}

This estimator is well-defined only if :math:`\\hat{\\xi}>-1/4`, otherwise the second moment does not exist.

**Probability weighted moments based estimator:**

Lets denote:

- :math:`\\left(x_{(i)}\\right)_{i\\in\\{1,\\dots,n\\}}` the sample sorted in ascending order
- :math:`m=\\dfrac{1}{n}\\sum_{i=1}^n\\left(1-\\dfrac{i-7/20}{n}\\right)x_{(i)}`
- :math:`\\rho=\\dfrac{m}{\\overline{x}_n}`

Then we estimate :math:`(\\hat{\\sigma}, \\hat{\\xi}, \\hat{u})` using:

.. math::
    :nowrap:
    :label: gpd_probability_weighted_moment_estimator

    \\begin{eqnarray*}
        \\hat{u}_n &= x_{(1)} - \\frac{x_{(1)}}{2 + n}\\\\
        \\hat{\\xi}_n &= \\dfrac{1-4\\rho}{1-2\\rho} \\\\
        \\hat{\\sigma}_n &= \\dfrac{2(\\overline{x}_n- \\hat{u}_n)}{1-2\\rho}
    \\end{eqnarray*}

This estimator is well-defined only if :math:`\\hat{\\xi}_n>-1/2`, otherwise the first moment does not exist.


**Maximum likelihood based estimator:**

These estimators are not yet implemented.

For a given :math:`u < x_{(1,n)}`, we get :math:`(\\hat{\\sigma}_n(u),  \\hat{\\xi}_n(u)` by maximizing the likelihood of the sample :math:`\\ell(\\sigma(u),  \\xi(u),u)`:

.. math::

    (\\hat{\\sigma}_n(u), \\hat{\\xi}_n(u)) = \\argmax_{\\sigma, \\xi}   \\ell(\\sigma(u), \\xi(u)|  x_1, \\dots, x_n, u) 

The threshold :math:`u` is obtained by maximizing the optimal likelihood :math:`\\ell(\\hat{\\sigma}_n(u),  \\hat{\\xi}_n(u), u)`

.. math::
    
    \\hat{u}_n = \\argmax_{u} \\ell(\\hat{\\sigma}_n(u),  \\hat{\\xi}_n(u),u) \\text{ under the constraint } u < x_{(1,n)}


The initial point of the optimisation problem is :math:`u_0 = x_{(1,n)} - |x_{(1,n)}|/(2+n)`.


**Exponential regression based estimator:**

Lets denote:

- :math:`y_{i}=i\\log\\left(\\dfrac{x_{(n-i)}-x_{(1)}}{x_{(n-i-1)}-x_{(1)}}\\right)` for :math:`i\\in\\{1,n-3\\}`

Then we estimate :math:`(\\hat{\\sigma}, \\hat{\\xi}, \\hat{u})`
using:

.. math::
    :label: gpd_exponential_estimator

    \\hat{\\xi} &= \\xi^* \\\\
    \\hat{\\sigma} &= \\dfrac{2(\\overline{x}_n- \\hat{u}_n)}{1-2\\rho} \\\\
    \\hat{u} &= x_{(1)} - \\frac{x_{(1)}}{2 + n}

Where :math:`\\xi^*` maximizes:

.. math::
    :label: gpd_xi_relation
    
    \\sum_{i=1}^{n-2}\\log\\left(\\dfrac{1-(j/n)^{\\xi}}{\\xi}\\right)-\\dfrac{1-(j/n)^{\\xi}}{\\xi}y_i

under the constraint :math:`-1 \\leq \\xi \\leq 1`.

The following :class:`~openturns.ResourceMap` entries can be used to tweak
the parameters of the optimization solver involved in the different estimators:

- `GeneralizedParetoFactory-DefaultOptimizationAlgorithm`
- `GeneralizedParetoFactory-MaximumEvaluationNumber`
- `GeneralizedParetoFactory-MaximumAbsoluteError`
- `GeneralizedParetoFactory-MaximumRelativeError`
- `GeneralizedParetoFactory-MaximumObjectiveError`
- `GeneralizedParetoFactory-MaximumConstraintError`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedParetoFactory::setOptimizationAlgorithm
"Accessor to the solver.

Parameters
----------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedParetoFactory::getOptimizationAlgorithm
"Accessor to the solver.

Returns
-------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedParetoFactory::build
"Build the distribution.

**Available usages**:

    build()

    build(*sample*)

    build(*param*)

Parameters
----------
sample : 2-d sequence of float, of dimension 1
    The sample from which the distribution parameters are estimated.
param : sequence of float
   The parameters of the :class:`~openturns.GeneralizedPareto`.

Returns
-------
dist : :class:`~openturns.Distribution`
    The built distribution.

Notes
-----
In the first usage, the default :class:`~openturns.GeneralizedPareto` distribution is built.

In the second usage, the parameters are evaluated according the following strategy:

- If the sample size is less or equal to `GeneralizedParetoFactory-SmallSize` from :class:`~openturns.ResourceMap`, then the method of probability weighted moments is used. If it fails, the method of exponential regression is used.
- Otherwise, the first method tried is the method of exponential regression, then the method of probability weighted moments if the first one fails.

In the third usage, a :class:`~openturns.GeneralizedPareto` distribution corresponding to the given parameters is built."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedParetoFactory::buildAsGeneralizedPareto
"Build the distribution as a GeneralizedPareto type.

**Available usages**:

    build()

    build(*sample*)

    build(*param*)

Parameters
----------
sample : 2-d sequence of float, of dimension 1
    The sample from which the distribution parameters are estimated.
param : sequence of float,
    A vector of parameters of the :class:`~openturns.GeneralizedPareto`.

Returns
-------
dist : :class:`~openturns.GeneralizedPareto`
    The estimated distribution as a GeneralizedPareto.
    
    In the first usage, the default GeneralizedPareto distribution is built."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedParetoFactory::buildMethodOfExponentialRegression
"Build the distribution based on the exponential regression estimator.

Parameters
----------
sample : 2-d sequence of float, of dimension 1
    The sample from which the distribution parameters are estimated."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedParetoFactory::buildMethodOfProbabilityWeightedMoments
"Build the distribution based on the probability weighted moments estimator.

Parameters
----------
sample : 2-d sequence of float, of dimension 1
    The sample from which the distribution parameters are estimated."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedParetoFactory::buildMethodOfMoments
"Build the distribution based on the method of moments estimator.

Parameters
----------
sample : 2-d sequence of float, of dimension 1
    The sample from which the distribution parameters are estimated."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralizedParetoFactory::drawMeanResidualLife
"Draw the mean residual life plot.

Let :math:`X` a random variable defined so the excess of a threshold :math:`u_s`
follow a Generalized Pareto distribution :math:`GPD(\\xi, \\sigma_s)`.
The mean of excesses of :math:`X` for :math:`u>u_s` is

.. math::

    M_n(u) = \\Expect{X-u|X>u} = \\frac{\\sigma_s+\\xi_u}{1-\\xi}

So for all :math:`u>u_s` M_n(u) is linear wrt. :math:`u`.
The threshold :math:`u_s` is the smallest value of :math:`u` for which the curve is linear.

The quantity :math:`M_n(u)` is estimater par the empirical estimator of the mean:

.. math::

    m_n(u) = \\frac{1}{n} \\sum_{i=1}^n (X_i - u) 1_{X_i \\ge u} = \\frac{1}{n} \\sum_{i=1}^n X_i 1_{X_i \\ge u} - u

The quantity :math:`M_n(u)` is asymptotically Gaussian with mean :math:`m_n(u)`
and variance :math:`m_n(u) (1 - m_n(u))/n`

Parameters
----------
sample : 2-d sequence of float, of dimension 1
    The sample from which the distribution parameters are estimated.

Returns
-------
graph : :class:`~openturns.Graph`
    The graph of :math:`m_n(u)` and its confidence interval.
    
Notes
-----
The confidence level can be set using the :class:`~openturns.ResourceMap` key
`GeneralizedParetoFactory-MeanResidualLifeConfidenceLevel`
The number of threshold points in the graph can be set with the key
`GeneralizedParetoFactory-MeanResidualLifePointNumber`.
"
