%feature("docstring") OT::Student
"Student distribution.

Its probability density function is defined as:

.. math::

    f_X(x) = \\frac{\\Gamma \\left(\\frac{\\nu + d}{2}\\right)}
                  {(\\pi \\nu) ^ {\\frac{d}{2}} \\Gamma \\left(\\frac{\\nu}{2}\\right)}
             \\frac{\\left| \\mathrm{det}(\\mat{R}) \\right| ^ {-1/2}}
                  {\\prod_{k = 1} ^ d \\sigma_k}
             \\left(1 + \\frac{\\vect{z} ^ t \\mat{R} ^ {-1} \\vect{z}}{\\nu}
             \\right) ^ {-\\frac{\\nu + d}{2}}, \\quad x \\in \\Rset

with :math:`\\nu > 0, d \\geq 1, \\mat{R} \\in  \\cM_{d \\times d}(\\Rset),
\\sigma_k > 0, k=1, \\dots, d` , and
where :math:`\\vect{z}=\\mat{\\Delta}^{-1}\\left(\\vect{x}-\\vect{\\mu}\\right)`, 
:math:`\\mat{\\Delta} = \\mat{\\mathrm{diag}}(\\vect{\\sigma})` and :math:`\\Gamma`
denotes Euler's Gamma function :meth:`~openturns.SpecFunc.Gamma`.

with :math:`\\mat{\\Sigma} = \\diag(\\vect{\\sigma}) \\mat{R} \\diag(\\vect{\\sigma})`.
Both :math:`\\mat{R}` and :math:`\\mat{\\Sigma}` are symmetric positive definite.

The first moments of the distribution are:

.. math::
    :nowrap:

    \\begin{eqnarray*}
        \\vect{\\Expect{X}} & = & \\vect{\\mu} \\quad \\text{ if }\\nu>1\\\\
        \\mat{\\Cov{X}} & = & \\displaystyle \\frac{\\nu}{\\nu-2}
                            \\Tr{\\mat{\\Delta}}\\,\\mat{R}\\,\\mat{\\Delta}
                          \\quad \\text{ if }\\nu>2
    \\end{eqnarray*}

Available constructors:
    Student(*nu, d*)

    Student(*nu, mu, sigma*)

    Student(*nu, mu, sigma, R*)

    Student(*nu, mu, Sigma*)

Parameters
----------
d : int, :math:`d \\geq 1`
    Dimension.

    Default value is 1.

nu : float, :math:`\\nu > 0`
    Generalized number of degrees of freedom.

    Default value is 3.0.

mu : float :math:`\\mu`, sequence of float, :math:`\\vect{\\mu} \\in \\Rset^d`
    Location parameter.

    Default value  is 0.0.

sigma : float :math:`\\sigma`, sequence of float, :math:`\\vect{\\sigma} \\in \\Rset^{*d}_+`
    Scale parameter.

    Default value  is 1.0.

R : :class:`~openturns.CorrelationMatrix` :math:`\\mat{R} \\in  \\cM_{d \\times d}(\\Rset)`
    Shape matrix.

    Default value is the identity matrix.

Sigma : :class:`~openturns.CovarianceMatrix` :math:`\\mat{\\Sigma} \\in  \\cM_{d \\times d}(\\Rset)`
    Covariance matrix.

Notes
-----
The CDF is computed using several algorithms, depending on the dimension of the distribution:

- In dimension 1, it is done using boost algorithm or the incomplete
  regularized beta function if boost is not available;
- In dimension 2, it is done using an integration in dimension 1 of a
  dedicated kernel;
- In dimension 3 to the value given by the entry *Student-SmallDimension* of
  :class:`~openturns.ResourceMap`, it is done using the generic algorithm for
  unimodal distributions. This algorithm is controlled by a positive scale
  given by the entry *Student-CDFScaleFactor* of
  :class:`~openturns.ResourceMap`.
- In dimension greater than the entry *Student-SmallDimension* of
  :class:`~openturns.ResourceMap`, it is done using a Monte Carlo estimation
  controlled by the *Student-MinimumCDFEpsilon*,
  *Student-MinimumNumberOfPoints* and *Student-MaximumNumberOfPoints* entries
  of :class:`~openturns.ResourceMap`.


Examples
--------
Create a distribution:

>>> import openturns as ot
>>> distribution = ot.Student(3.0, -0.5, 2.0)

Draw a sample:

>>> sample = distribution.getSample(5)

Create a multivariate Student distribution with constant :math:`\\sigma` and identity correlation matrix:

>>> dimension = 5
>>> nu = 3.0
>>> mu = [0.0] * dimension
>>> sigma = [0.1] * dimension
>>> R = ot.CorrelationMatrix(dimension)
>>> distribution = ot.Student(nu, mu, sigma, R)
>>> print('distribution=', distribution)
distribution= Student(nu = 3, mu = [0,0,0,0,0], sigma = [0.1,0.1,0.1,0.1,0.1], R = 5x5
[[ 1 0 0 0 0 ]
 [ 0 1 0 0 0 ]
 [ 0 0 1 0 0 ]
 [ 0 0 0 1 0 ]
 [ 0 0 0 0 1 ]])

Create a multivariate Student distribution with a different :math:`\\sigma` parameter and a different correlation matrix:

>>> sigma = [1.5, 0.4, 1.0, 2.5, 1.0]
>>> R = ot.CorrelationMatrix(dimension)
>>> R[0, 2] = -0.5
>>> R[3, 4] = -0.2
>>> distribution = ot.Student(nu, mu, sigma, R)
>>> print('distribution=', distribution)
distribution= Student(nu = 3, mu = [0,0,0,0,0], sigma = [1.5,0.4,1,2.5,1], R = 5x5
[[  1    0   -0.5  0    0   ]
 [  0    1    0    0    0   ]
 [ -0.5  0    1    0    0   ]
 [  0    0    0    1   -0.2 ]
 [  0    0    0   -0.2  1   ]])"

// ---------------------------------------------------------------------

%feature("docstring") OT::Student::getNu
"Accessor to the generalized number of degrees of freedom.

Returns
-------
nu : float
    The :math:`\\nu` parameter."

// ---------------------------------------------------------------------

%feature("docstring") OT::Student::setNu
"Accessor to the generalized number of degrees of freedom.

Parameters
----------
nu : float, :math:`\\nu > 0`
    The :math:`\\nu` parameter."

// ---------------------------------------------------------------------

%feature("docstring") OT::Student::getMu
"Accessor to the :math:`\\mu` parameter.

Returns
-------
mu : float,
    The :math:`\\mu` parameter."


// ---------------------------------------------------------------------

%feature("docstring") OT::Student::setMu
"Accessor to the :math:`\\mu` parameter.

Parameters
----------
mu : float,
    The :math:`\\mu` parameter."


