// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
/**
 * @class   vtkPLinearExtrusionFilter
 * @brief   Subclass that handles piece invariance.
 *
 * vtkPLinearExtrusionFilter is a parallel version of vtkLinearExtrusionFilter.
 *
 * @sa
 * vtkLinearExtrusionFilter
 */

#ifndef vtkPLinearExtrusionFilter_h
#define vtkPLinearExtrusionFilter_h

#include "vtkFiltersParallelModule.h" // For export macro
#include "vtkLinearExtrusionFilter.h"

VTK_ABI_NAMESPACE_BEGIN
class VTKFILTERSPARALLEL_EXPORT vtkPLinearExtrusionFilter : public vtkLinearExtrusionFilter
{
public:
  vtkTypeMacro(vtkPLinearExtrusionFilter, vtkLinearExtrusionFilter);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   * Create an object with PieceInvariant off.
   */
  static vtkPLinearExtrusionFilter* New();

  // To get piece invariance, this filter has to request an
  // extra ghost level.  Since piece invariance is not very
  // important for this filter,  it is optional.  Without invariance,
  // Internal surfaces will be generated.  These surface
  // Are hidden by the normal surface generated by this filter.
  // By default, PieceInvariance is off.
  vtkSetMacro(PieceInvariant, vtkTypeBool);
  vtkGetMacro(PieceInvariant, vtkTypeBool);
  vtkBooleanMacro(PieceInvariant, vtkTypeBool);

protected:
  vtkPLinearExtrusionFilter();
  ~vtkPLinearExtrusionFilter() override = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;
  int RequestUpdateExtent(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  vtkTypeBool PieceInvariant;

private:
  vtkPLinearExtrusionFilter(const vtkPLinearExtrusionFilter&) = delete;
  void operator=(const vtkPLinearExtrusionFilter&) = delete;
};

VTK_ABI_NAMESPACE_END
#endif
