%feature("docstring") OT::SquareMatrix
"Real square matrix.

Parameters
----------
size : int, :math:`n > 0`, optional
    Matrix size.
    Default is 1.
values : sequence of float with size :math:`n^2`, optional
    Values. OpenTURNS uses **column-major** ordering (like Fortran) for
    reshaping the flat list of values.
    Default creates a zero matrix.

Examples
--------
Create a matrix

>>> import openturns as ot
>>> M = ot.SquareMatrix(2, range(2 * 2))
>>> print(M)
[[ 0 2 ]
 [ 1 3 ]]

Get or set terms

>>> print(M[0, 0])
0.0
>>> M[0, 0] = 1.0
>>> print(M[0, 0])
1.0
>>> print(M[:, 0])
[[ 1 ]
 [ 1 ]]

Create a matrix from a **square** Numpy 2d-array (or matrix, or
2d-list)...

>>> import numpy as np
>>> np_2d_array = np.array([[1.0, 2.0], [3.0, 4.0]])
>>> ot_matrix = ot.SquareMatrix(np_2d_array)

and back

>>> np_matrix = np.matrix(ot_matrix)

Basic linear algebra operations (provided the dimensions are compatible)

>>> A = ot.Matrix([[1.0, 2.0], [3.0, 4.0], [5.0, 6.0]])
>>> B = ot.SquareMatrix(np.eye(2))
>>> C = ot.Matrix(3, 2, [1.0] * 3 * 2)
>>> print(A * B - C)
[[ 0 1 ]
 [ 2 3 ]
 [ 4 5 ]]
>>> A = ot.SquareMatrix([[1.0, 2.0], [3.0, 4.0]])
>>> print(A ** 2)
[[  7 10 ]
 [ 15 22 ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeDeterminant
"Compute the determinant.

Returns
-------
determinant : float
    The square matrix determinant.

Examples
--------
>>> import openturns as ot
>>> A = ot.SquareMatrix([[1.0, 2.0], [3.0, 4.0]])
>>> A.computeDeterminant()
-2.0"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeDeterminantInPlace
"Compute the determinant in place.

Similar to :meth:`computeDeterminant` but modifies the matrix in place to avoid copy.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeEigenValues
"Compute eigenvalues.

Returns
-------
eigenvalues : :class:`~openturns.ComplexCollection`
    Eigenvalues.

See Also
--------
computeEV

Examples
--------
>>> import openturns as ot
>>> M = ot.SquareMatrix([[1.0, 2.0], [3.0, 4.0]])
>>> M.computeEigenValues()
[(-0.372281,0),(5.37228,0)]"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeEigenValuesInPlace
"Compute eigenvalues in place.

Similar to :meth:`computeEigenValues` but the matrix is modified in place to avoid copy.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeEV
"Compute the eigenvalues decomposition (EVD).

The eigenvalues decomposition of a square matrix :math:`\\mat{M}` with
size :math:`n` reads:

.. math::

    \\mat{M} = \\mat{\\Phi} \\mat{\\Lambda} \\mat{\\Phi}^{-1}

where :math:`\\mat{\\Lambda}` is an :math:`n \\times n` diagonal matrix and
:math:`\\mat{\\Phi}` is an :math:`n \\times n` orthogonal matrix.

Returns
-------
eigen_values : :class:`~openturns.ComplexCollection`
    The vector of eigenvalues with size :math:`n` that form the diagonal of
    the :math:`n \\times n` matrix :math:`\\mat{\\Lambda}` of the EVD.
Phi : :class:`~openturns.SquareComplexMatrix`
    The left matrix of the EVD.

Notes
-----
This uses LAPACK'S `DGEEV <http://www.netlib.org/lapack/lapack-3.1.1/html/dgeev.f.html>`_.

Examples
--------
>>> import openturns as ot
>>> import numpy as np
>>> M = ot.SquareMatrix([[1.0, 2.0], [3.0, 4.0]])
>>> eigen_values, Phi = M.computeEV()
>>> Lambda = ot.SquareComplexMatrix(M.getDimension())
>>> for i in range(eigen_values.getSize()):
...     Lambda[i, i] = eigen_values[i]
>>> # from scipy.linalg import inv # SquareComplexMatrix does not implement solveLinearSystem
>>> # Phi, Lambda = np.matrix(Phi), np.matrix(Lambda)
>>> # np.testing.assert_array_almost_equal(Phi * Lambda * inv(Phi), M)"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeEVInPlace
"Compute the eigenvalues decomposition (EVD) in place.

Similar to :meth:`computeEVInPlace` but the matrix is modified in place to avoid copy.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeLargestEigenValueModule
"Compute the largest eigenvalue module.

Parameters
----------
maximumIterations : int, optional
    The maximum number of power iterations to perform to get the approximation.
    Default is given by the 'Matrix-LargestEigenValueIterations' key in
    the :class:`~openturns.ResourceMap`.
epsilon : float, optional
    The target relative error. Default is given by the
    'Matrix-LargestEigenValueRelativeError' key in the
    :class:`~openturns.ResourceMap`.

Returns
-------
largestEigenvalueModule : float
    The largest eigenvalue module.

See Also
--------
computeEV, computeEigenValues

Examples
--------
>>> import openturns as ot
>>> M = ot.SquareMatrix([[1.0, 2.0], [3.0, 4.0]])
>>> M.computeLargestEigenValueModule()
5.3722..."

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeLogAbsoluteDeterminant
"Compute the logarithm of the absolute value of the determinant.

Returns
-------
determinant : float
    The logarithm of the absolute value of the square matrix determinant.
sign : float
    The sign of the determinant.

Examples
--------
>>> import openturns as ot
>>> A = ot.SquareMatrix([[1.0, 2.0], [3.0, 4.0]])
>>> A.computeLogAbsoluteDeterminant()
[0.693147..., -1.0]"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeLogAbsoluteDeterminantInPlace
"Compute the determinant in place.

Similar to :meth:`computeLogAbsoluteDeterminant` but modifies the matrix in place to avoid copy.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::getDimension
"Accessor to the dimension (the number of rows).

Returns
-------
dimension : int"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::isDiagonal
"Test whether the matrix is diagonal or not.

Returns
-------
test : bool
    Answer."

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::solveLinearSystem
"Solve a square linear system whose the present matrix is the operator.

Parameters
----------
rhs : sequence of float or :class:`~openturns.Matrix` with :math:`n_r` values or rows, respectively
    The right hand side member of the linear system.

Returns
-------
solution : :class:`~openturns.Point` or :class:`~openturns.Matrix`
    The solution of the square linear system.

Notes
-----
This will handle both matrices and vectors. Note that you'd better type
explicitly the matrix if it has some properties that could simplify the
resolution (see :class:`~openturns.TriangularMatrix`).

This uses LAPACK'S `DGESV <http://www.netlib.org/lapack/lapack-3.1.1/html/dgesv.f.html>`_
for matrices and `DGELSY <http://www.netlib.org/lapack/lapack-3.1.1/html/dgelsy.f.html>`_
for vectors.

Examples
--------
>>> import openturns as ot
>>> import numpy as np
>>> M = ot.SquareMatrix([[1.0, 2.0], [3.0, 4.0]])
>>> b = [1.0] * 2
>>> x = M.solveLinearSystem(b)
>>> np.testing.assert_array_almost_equal(M * x, b)"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::transpose
"Transpose the matrix.

Returns
-------
MT : :class:`~openturns.SquareMatrix`
    The transposed matrix.

Examples
--------
>>> import openturns as ot
>>> M = ot.SquareMatrix([[1.0, 2.0], [3.0, 4.0]])
>>> print(M)
[[ 1 2 ]
 [ 3 4 ]]
>>> print(M.transpose())
[[ 1 3 ]
 [ 2 4 ]]"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::computeTrace
"Compute the trace of the matrix.

Returns
-------
trace : float
    The trace of the matrix.

Examples
--------
>>> import openturns as ot
>>> M = ot.SquareMatrix([[1.0, 2.0], [3.0, 4.0]])
>>> M.computeTrace()
5.0"

// ---------------------------------------------------------------------

%feature("docstring") OT::SquareMatrix::inverse
"Compute the inverse of the matrix.

Returns
-------
inverseMatrix : :class:`~openturns.SquareMatrix`
    The inverse of the matrix.

Examples
--------
>>> import openturns as ot
>>> M = ot.SquareMatrix([[1.0, 2.0, 3.0], [3.0, 2.0, 1.0], [2.0, 1.0, 3.0]])
>>> print(12.0 * M.inverse())
[[ -5  3  4 ]
 [  7  3 -8 ]
 [  1 -3  4 ]]"
