import { jsx as _jsx, jsxs as _jsxs } from "react/jsx-runtime";
import { useState } from 'react';
import { Box, Text } from 'ink';
import { Colors } from '../colors.js';
import { RadioButtonSelect } from './shared/RadioButtonSelect.js';
import { SettingScope } from '../../config/settings.js';
import { AuthType } from '@google/gemini-cli-core';
import { validateAuthMethod } from '../../config/auth.js';
import { useKeypress } from '../hooks/useKeypress.js';
function parseDefaultAuthType(defaultAuthType) {
    if (defaultAuthType &&
        Object.values(AuthType).includes(defaultAuthType)) {
        return defaultAuthType;
    }
    return null;
}
export function AuthDialog({ onSelect, settings, initialErrorMessage, }) {
    const [errorMessage, setErrorMessage] = useState(() => {
        if (initialErrorMessage) {
            return initialErrorMessage;
        }
        const defaultAuthType = parseDefaultAuthType(process.env['GEMINI_DEFAULT_AUTH_TYPE']);
        if (process.env['GEMINI_DEFAULT_AUTH_TYPE'] && defaultAuthType === null) {
            return (`Invalid value for GEMINI_DEFAULT_AUTH_TYPE: "${process.env['GEMINI_DEFAULT_AUTH_TYPE']}". ` +
                `Valid values are: ${Object.values(AuthType).join(', ')}.`);
        }
        if (process.env['GEMINI_API_KEY'] &&
            (!defaultAuthType || defaultAuthType === AuthType.USE_GEMINI)) {
            return 'Existing API key detected (GEMINI_API_KEY). Select "Gemini API Key" option to use it.';
        }
        return null;
    });
    let items = [
        {
            label: 'Login with Google',
            value: AuthType.LOGIN_WITH_GOOGLE,
        },
        ...(process.env['CLOUD_SHELL'] === 'true'
            ? [
                {
                    label: 'Use Cloud Shell user credentials',
                    value: AuthType.CLOUD_SHELL,
                },
            ]
            : []),
        {
            label: 'Use Gemini API Key',
            value: AuthType.USE_GEMINI,
        },
        { label: 'Vertex AI', value: AuthType.USE_VERTEX_AI },
    ];
    if (settings.merged.security?.auth?.enforcedType) {
        items = items.filter((item) => item.value === settings.merged.security?.auth?.enforcedType);
    }
    let initialAuthIndex = items.findIndex((item) => {
        if (settings.merged.security?.auth?.selectedType) {
            return item.value === settings.merged.security.auth.selectedType;
        }
        const defaultAuthType = parseDefaultAuthType(process.env['GEMINI_DEFAULT_AUTH_TYPE']);
        if (defaultAuthType) {
            return item.value === defaultAuthType;
        }
        if (process.env['GEMINI_API_KEY']) {
            return item.value === AuthType.USE_GEMINI;
        }
        return item.value === AuthType.LOGIN_WITH_GOOGLE;
    });
    if (settings.merged.security?.auth?.enforcedType) {
        initialAuthIndex = 0;
    }
    const handleAuthSelect = (authMethod) => {
        const error = validateAuthMethod(authMethod);
        if (error) {
            setErrorMessage(error);
        }
        else {
            setErrorMessage(null);
            onSelect(authMethod, SettingScope.User);
        }
    };
    useKeypress((key) => {
        if (key.name === 'escape') {
            // Prevent exit if there is an error message.
            // This means they user is not authenticated yet.
            if (errorMessage) {
                return;
            }
            if (settings.merged.security?.auth?.selectedType === undefined) {
                // Prevent exiting if no auth method is set
                setErrorMessage('You must select an auth method to proceed. Press Ctrl+C twice to exit.');
                return;
            }
            onSelect(undefined, SettingScope.User);
        }
    }, { isActive: true });
    return (_jsxs(Box, { borderStyle: "round", borderColor: Colors.Gray, flexDirection: "column", padding: 1, width: "100%", children: [_jsx(Text, { bold: true, children: "Get started" }), _jsx(Box, { marginTop: 1, children: _jsx(Text, { children: "How would you like to authenticate for this project?" }) }), _jsx(Box, { marginTop: 1, children: _jsx(RadioButtonSelect, { items: items, initialIndex: initialAuthIndex, onSelect: handleAuthSelect }) }), errorMessage && (_jsx(Box, { marginTop: 1, children: _jsx(Text, { color: Colors.AccentRed, children: errorMessage }) })), _jsx(Box, { marginTop: 1, children: _jsx(Text, { color: Colors.Gray, children: "(Use Enter to select)" }) }), _jsx(Box, { marginTop: 1, children: _jsx(Text, { children: "Terms of Services and Privacy Notice for Gemini CLI" }) }), _jsx(Box, { marginTop: 1, children: _jsx(Text, { color: Colors.AccentBlue, children: 'https://github.com/google-gemini/gemini-cli/blob/main/docs/tos-privacy.md' }) })] }));
}
//# sourceMappingURL=AuthDialog.js.map