/* *****************************************************************
    MESQUITE -- The Mesh Quality Improvement Toolkit

    Copyright 2004 Sandia Corporation and Argonne National
    Laboratory.  Under the terms of Contract DE-AC04-94AL85000
    with Sandia Corporation, the U.S. Government retains certain
    rights in this software.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    (lgpl.txt) along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    diachin2@llnl.gov, djmelan@sandia.gov, mbrewer@sandia.gov,
    pknupp@sandia.gov, tleurent@mcs.anl.gov, tmunson@mcs.anl.gov

  ***************************************************************** */
// -*- Mode : c++; tab-width: 2; c-tab-always-indent: t; indent-tabs-mode: nil; c-basic-offset: 2
// -*-
//

#include <cstdio>
static bool create_exodus_file( const char* filename );

#undef DEBUG

#ifndef DEBUG
#ifndef MSQ_USING_EXODUS
#error Attempt to build test of disabled ExodusII support
#endif

#include "Mesquite.hpp"
#include "MeshImpl.hpp"
#include "MsqVertex.hpp"
#include "MsqError.hpp"

using MBMesquite::arrptr;

#include "cppunit/extensions/HelperMacros.h"

#include <iostream>
#include <algorithm>

#ifndef _WIN32
#include <unistd.h>
#endif

class ExodusTest : public CppUnit::TestFixture
{
  private:
    CPPUNIT_TEST_SUITE( ExodusTest );
    CPPUNIT_TEST( test_read );
    CPPUNIT_TEST( test_write );
    CPPUNIT_TEST_SUITE_END();

  public:
    /* Automatically called by CppUnit before each test function. */
    void setUp() {}

    // Automatically called by CppUnit after each test function.
    void tearDown() {}

    ExodusTest() {}

    void test_read();

    void test_write();

    void check_mesh( const char* filaname );
};

void ExodusTest::test_read()
{
    // Create a mesh file to read
    char filename[] = "MsqExoTestTemp.g";
    bool havefile   = create_exodus_file( filename );
    CPPUNIT_ASSERT( havefile );

    check_mesh( filename );
}

void ExodusTest::test_write()
{
    MBMesquite::MeshImpl* mMesh;
    MBMesquite::MsqPrintError err( std::cout );

    // Create a mesh file
    char filename[] = "MsqExoTestTemp.g";
    bool havefile   = create_exodus_file( filename );
    CPPUNIT_ASSERT( havefile );

    // Read in test file
    mMesh = new MBMesquite::MeshImpl;
    mMesh->read_exodus( filename, err );
    remove( filename );
    CPPUNIT_ASSERT( !err );

    // Write the test file back out
    mMesh->write_exodus( filename, err );
    if( err ) remove( filename );
    CPPUNIT_ASSERT( !err );
    delete mMesh;

    // Read back in the file we just wrote and check it
    check_mesh( filename );
}

void ExodusTest::check_mesh( const char* filename )
{
    MBMesquite::MeshImpl* mMesh;
    MBMesquite::MsqPrintError err( std::cout );
    int i;
    const unsigned NUM_HEXES = 115;
    const unsigned NUM_QUADS = 0;
    const unsigned NUM_NODES = 216;

    // Read a Exodus Mesh file
    mMesh = new MBMesquite::MeshImpl;
    mMesh->read_exodus( filename, err );
    remove( filename );
    CPPUNIT_ASSERT( !err );

    // Check overall counts
    std::vector< MBMesquite::Mesh::ElementHandle > elem_handle_vect;
    mMesh->get_all_elements( elem_handle_vect, err );
    CPPUNIT_ASSERT( !err );
    CPPUNIT_ASSERT( elem_handle_vect.size() == NUM_QUADS + NUM_HEXES );
    std::vector< MBMesquite::Mesh::VertexHandle > vert_handle_vect;
    std::vector< size_t > offset_vect;
    mMesh->elements_get_attached_vertices( arrptr( elem_handle_vect ), elem_handle_vect.size(), vert_handle_vect,
                                           offset_vect, err );
    CPPUNIT_ASSERT( !err );
    CPPUNIT_ASSERT( vert_handle_vect.size() == 8 * NUM_HEXES + 4 * NUM_QUADS );
    std::sort( vert_handle_vect.begin(), vert_handle_vect.end() );
    std::vector< MBMesquite::Mesh::VertexHandle >::iterator new_end =
        std::unique( vert_handle_vect.begin(), vert_handle_vect.end() );
    vert_handle_vect.resize( new_end - vert_handle_vect.begin() );
    CPPUNIT_ASSERT( vert_handle_vect.size() == NUM_NODES );

    // Array of names of expected element types
    const char* names[Mesquite::MIXED];
    memset( names, 0, sizeof( names ) );
    names[Mesquite::QUADRILATERAL] = "quad";
    names[Mesquite::TRIANGLE]      = "tri";
    names[Mesquite::TETRAHEDRON]   = "tet";
    names[Mesquite::HEXAHEDRON]    = "hex";
    names[Mesquite::PRISM]         = "wedge";
    names[Mesquite::PYRAMID]       = "pyr";

    // Count elements by type
    unsigned counts[Mesquite::MIXED];
    memset( counts, 0, sizeof( counts ) );
    MBMesquite::ElementIterator* iter = mMesh->element_iterator( err );
    CPPUNIT_ASSERT( !err );
    while( !iter->is_at_end() )
    {
        MBMesquite::EntityTopology type;
        MBMesquite::Mesh::ElementHandle handle = iter->operator*();
        mMesh->elements_get_topologies( &handle, &type, 1, err );
        CPPUNIT_ASSERT( !err );
        CPPUNIT_ASSERT( type < Mesquite::MIXED && names[type] != NULL );
        ++counts[type];
        iter->operator++();
    }

    // Print counts
    printf( "TYPE   COUNT\n-----  -----\n" );
    for( i = 0; i < MBMesquite::MIXED; ++i )
        if( counts[i] ) printf( "%5s  %5d\n", names[i], counts[i] );

    CPPUNIT_ASSERT( counts[Mesquite::TRIANGLE] == 0 );
    CPPUNIT_ASSERT( counts[Mesquite::QUADRILATERAL] == NUM_QUADS );
    CPPUNIT_ASSERT( counts[Mesquite::TETRAHEDRON] == 0 );
    CPPUNIT_ASSERT( counts[Mesquite::HEXAHEDRON] = NUM_HEXES );
    CPPUNIT_ASSERT( counts[Mesquite::PRISM] == 0 );
    CPPUNIT_ASSERT( counts[Mesquite::PYRAMID] == 0 );

    // Check a few hexes and nodes for correctness
    const unsigned num_to_check = 6;
    unsigned j;
    char buffer[64];

    // Check connectivity of first six hexes
    static const unsigned expected_hex_connectivity[num_to_check][8] = {
        { 64, 63, 168, 167, 4, 5, 28, 27 }, { 63, 62, 132, 133, 5, 6, 24, 25 },   { 62, 169, 131, 132, 6, 29, 23, 24 },
        { 62, 46, 170, 169, 6, 1, 30, 29 }, { 46, 152, 153, 170, 1, 10, 11, 30 }, { 46, 66, 171, 152, 1, 2, 31, 10 }
    };
    for( j = 0; j < num_to_check; ++j )
    {
        sprintf( buffer, "bad hex: %d\n", j );
        vert_handle_vect.clear();
        offset_vect.clear();
        MBMesquite::Mesh::ElementHandle handle = (MBMesquite::Mesh::ElementHandle)j;
        mMesh->elements_get_attached_vertices( &handle, 1, vert_handle_vect, offset_vect, err );
        CPPUNIT_ASSERT( !err );
        CPPUNIT_ASSERT( vert_handle_vect.size() == 8 );
        for( i = 0; i < 8; ++i )
        {
            size_t mesh = (size_t)vert_handle_vect[i] + 1;
            size_t exp  = expected_hex_connectivity[j][i];
            CPPUNIT_ASSERT_MESSAGE( buffer, mesh == exp );
        }
    }

    // Check locations of first six vertices
    static const double expected_node_coords[num_to_check][3] = {
        { 2.000000, 0.000000, 5.000000 },   { 1.000000, -1.732051, 5.000000 }, { -1.000000, -1.732051, 5.000000 },
        { -2.000000, -0.000000, 5.000000 }, { -1.000000, 1.732051, 5.000000 }, { 1.000000, 1.732051, 5.000000 }
    };
    MBMesquite::MsqVertex vert;
    for( j = 0; j < num_to_check; ++j )
    {
        sprintf( buffer, "bad node: %d\n", j );
        MBMesquite::Mesh::VertexHandle handle = (MBMesquite::Mesh::VertexHandle)j;
        mMesh->vertices_get_coordinates( &handle, &vert, 1, err );
        CPPUNIT_ASSERT( !err );
        for( i = 0; i < 3; ++i )
        {
            CPPUNIT_ASSERT_MESSAGE( buffer, fabs( expected_node_coords[j][i] - vert[i] ) < 1e-6 );
        }
    }
}

CPPUNIT_TEST_SUITE_NAMED_REGISTRATION( ExodusTest, "ExodusTest" );
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION( ExodusTest, "Unit" );

#else /* #ifndef DEBUG */

int main( int argc, char* argv[] )
{
    const char* filename = "dump.g";
    if( argc > 1 ) filename = argv[1];

    bool result = create_exodus_file( filename );
    if( result )
        printf( "Wrote \"%s\"\n", filename );
    else
        printf( "Failed to write \"%s\"\n", filename );

    return !result;
}

#endif /* #ifndef DEBUG */

bool create_exodus_file( const char* filename )
{
    // This is the binary data composing an ExodusII mesh.
    // This mesh was generated using the following commands
    // in Cubit 9.2b Build 113:
    //  CUBIT> brick x 10
    //  CUBIT> cylinder radius 2 z 10
    //  CUBIT> subtract 2 from 1
    //  CUBIT> volume 1 size 1 scheme auto
    //  CUBIT> mesh volume 1
    //  CUBIT> export mesh 'mesh.g'
    // To extract this data into a file for debugging, do
    //  $ CC -DDEBUG ExodusTest.cpp -o exodump
    //  $ ./exodump mesh.g
    static const unsigned char binary_data[] = {
        0x43, 0x44, 0x46, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00,
        0x00, 0x0a, 0x6c, 0x65, 0x6e, 0x5f, 0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x21,
        0x00, 0x00, 0x00, 0x08, 0x6c, 0x65, 0x6e, 0x5f, 0x6c, 0x69, 0x6e, 0x65, 0x00, 0x00, 0x00, 0x51, 0x00, 0x00,
        0x00, 0x04, 0x66, 0x6f, 0x75, 0x72, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x09, 0x74, 0x69, 0x6d, 0x65,
        0x5f, 0x73, 0x74, 0x65, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x6e, 0x75,
        0x6d, 0x5f, 0x64, 0x69, 0x6d, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x09, 0x6e, 0x75, 0x6d, 0x5f,
        0x6e, 0x6f, 0x64, 0x65, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x08, 0x6e, 0x75,
        0x6d, 0x5f, 0x65, 0x6c, 0x65, 0x6d, 0x00, 0x00, 0x00, 0x73, 0x00, 0x00, 0x00, 0x0a, 0x6e, 0x75, 0x6d, 0x5f,
        0x65, 0x6c, 0x5f, 0x62, 0x6c, 0x6b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0e, 0x6e, 0x75,
        0x6d, 0x5f, 0x65, 0x6c, 0x5f, 0x69, 0x6e, 0x5f, 0x62, 0x6c, 0x6b, 0x31, 0x00, 0x00, 0x00, 0x00, 0x00, 0x73,
        0x00, 0x00, 0x00, 0x0f, 0x6e, 0x75, 0x6d, 0x5f, 0x6e, 0x6f, 0x64, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x65, 0x6c,
        0x31, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x0a, 0x6e, 0x75, 0x6d, 0x5f, 0x71, 0x61, 0x5f, 0x72,
        0x65, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
        0x00, 0x0b, 0x61, 0x70, 0x69, 0x5f, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x00, 0x00, 0x00, 0x05,
        0x00, 0x00, 0x00, 0x01, 0x40, 0x80, 0x51, 0xec, 0x00, 0x00, 0x00, 0x07, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
        0x6e, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x40, 0x40, 0xa3, 0xd7, 0x00, 0x00, 0x00, 0x18,
        0x66, 0x6c, 0x6f, 0x61, 0x74, 0x69, 0x6e, 0x67, 0x5f, 0x70, 0x6f, 0x69, 0x6e, 0x74, 0x5f, 0x77, 0x6f, 0x72,
        0x64, 0x5f, 0x73, 0x69, 0x7a, 0x65, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08,
        0x00, 0x00, 0x00, 0x09, 0x66, 0x69, 0x6c, 0x65, 0x5f, 0x73, 0x69, 0x7a, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x74, 0x69, 0x74, 0x6c,
        0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x35, 0x63, 0x75, 0x62, 0x69, 0x74, 0x28,
        0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x2d, 0x77, 0x69, 0x74, 0x68, 0x2d, 0x68, 0x6f, 0x6c, 0x65, 0x2d, 0x68, 0x65,
        0x78, 0x65, 0x73, 0x2e, 0x67, 0x29, 0x3a, 0x20, 0x31, 0x31, 0x2f, 0x32, 0x34, 0x2f, 0x32, 0x30, 0x30, 0x34,
        0x3a, 0x20, 0x31, 0x32, 0x3a, 0x33, 0x34, 0x3a, 0x32, 0x35, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b,
        0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0a, 0x74, 0x69, 0x6d, 0x65, 0x5f, 0x77, 0x68, 0x6f, 0x6c, 0x65,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x09, 0x65, 0x62,
        0x5f, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x00, 0x08, 0x65, 0x62, 0x5f, 0x70, 0x72, 0x6f, 0x70, 0x31, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x6e, 0x61,
        0x6d, 0x65, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x49, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x10, 0x04, 0x00, 0x00, 0x00, 0x08, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63,
        0x74, 0x31, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x0c,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x09, 0x65, 0x6c, 0x65, 0x6d, 0x5f, 0x74, 0x79, 0x70, 0x65, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x05, 0x48, 0x45, 0x58, 0x38, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x0e, 0x60, 0x00, 0x00, 0x10, 0x08, 0x00, 0x00, 0x00, 0x05, 0x63, 0x6f,
        0x6f, 0x72, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x14, 0x40, 0x00, 0x00,
        0x1e, 0x68, 0x00, 0x00, 0x00, 0x0a, 0x71, 0x61, 0x5f, 0x72, 0x65, 0x63, 0x6f, 0x72, 0x64, 0x73, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x32, 0xa8,
        0x00, 0x00, 0x00, 0x0a, 0x63, 0x6f, 0x6f, 0x72, 0x5f, 0x6e, 0x61, 0x6d, 0x65, 0x73, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x33, 0x2c, 0x00, 0x00, 0x00, 0x08, 0x65, 0x6c,
        0x65, 0x6d, 0x5f, 0x6d, 0x61, 0x70, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0xcc, 0x00, 0x00, 0x33, 0x90, 0x00, 0x00,
        0x00, 0x0c, 0x65, 0x6c, 0x65, 0x6d, 0x5f, 0x6e, 0x75, 0x6d, 0x5f, 0x6d, 0x61, 0x70, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
        0x01, 0xcc, 0x00, 0x00, 0x35, 0x5c, 0x00, 0x00, 0x00, 0x0c, 0x6e, 0x6f, 0x64, 0x65, 0x5f, 0x6e, 0x75, 0x6d,
        0x5f, 0x6d, 0x61, 0x70, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x03, 0x60, 0x00, 0x00, 0x37, 0x28, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x3f,
        0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x85, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
        0x00, 0x06, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0xa9,
        0x00, 0x00, 0x00, 0x83, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00,
        0x00, 0x17, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0xaa,
        0x00, 0x00, 0x00, 0xa9, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x00,
        0x00, 0x1d, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00, 0x99, 0x00, 0x00, 0x00, 0xaa,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x00,
        0x00, 0x2e, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0xab, 0x00, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00,
        0x00, 0xac, 0x00, 0x00, 0x00, 0xb0, 0x00, 0x00, 0x00, 0xab, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x20,
        0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00,
        0x00, 0xad, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x21,
        0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00,
        0x00, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0x6b, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00,
        0x00, 0x1b, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00, 0x85,
        0x00, 0x00, 0x00, 0x86, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00,
        0x00, 0x1a, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x9a, 0x00, 0x00, 0x00, 0x7b,
        0x00, 0x00, 0x00, 0x83, 0x00, 0x00, 0x00, 0xa9, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
        0x00, 0x17, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0xa9, 0x00, 0x00, 0x00, 0xaa, 0x00, 0x00, 0x00, 0x99,
        0x00, 0x00, 0x00, 0x9a, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00,
        0x00, 0x0c, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x86, 0x00, 0x00, 0x00, 0x62, 0x00, 0x00, 0x00, 0x6a,
        0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00,
        0x00, 0x6a, 0x00, 0x00, 0x00, 0x6b, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x0f,
        0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00,
        0x00, 0x51, 0x00, 0x00, 0x00, 0xaf, 0x00, 0x00, 0x00, 0x6d, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x13,
        0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00, 0x6d, 0x00, 0x00,
        0x00, 0xaf, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x23,
        0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00, 0xab, 0x00, 0x00, 0x00, 0xb0, 0x00, 0x00,
        0x00, 0x97, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x09,
        0x00, 0x00, 0x00, 0x4d, 0x00, 0x00, 0x00, 0x97, 0x00, 0x00, 0x00, 0xb0, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00,
        0x00, 0x07, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x54,
        0x00, 0x00, 0x00, 0xb0, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x53, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00,
        0x00, 0x24, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0x53,
        0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0xad, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00,
        0x00, 0x20, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0xaf, 0x00, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x52,
        0x00, 0x00, 0x00, 0xad, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00,
        0x00, 0x21, 0x00, 0x00, 0x00, 0xad, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00, 0x00, 0xaf,
        0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00,
        0x00, 0x3b, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0xb2, 0x00, 0x00, 0x00, 0xb1, 0x00, 0x00, 0x00, 0x40,
        0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00,
        0x00, 0x39, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x89, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x3e,
        0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x85, 0x00, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0xb3, 0x00, 0x00,
        0x00, 0x87, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0xa9, 0x00, 0x00, 0x00, 0x83,
        0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00,
        0x00, 0xb3, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0xaa, 0x00, 0x00, 0x00, 0xa9,
        0x00, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x9d, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00,
        0x00, 0x2e, 0x00, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00, 0x99, 0x00, 0x00, 0x00, 0xaa, 0x00, 0x00, 0x00, 0x2d,
        0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00, 0xb5, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00,
        0x00, 0x42, 0x00, 0x00, 0x00, 0xab, 0x00, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00, 0xb6,
        0x00, 0x00, 0x00, 0xba, 0x00, 0x00, 0x00, 0xb5, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00,
        0x00, 0xb0, 0x00, 0x00, 0x00, 0xab, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0xb7,
        0x00, 0x00, 0x00, 0xb6, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0xad, 0x00, 0x00,
        0x00, 0xac, 0x00, 0x00, 0x00, 0x3b, 0x00, 0x00, 0x00, 0xb1, 0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0x3c,
        0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00,
        0x00, 0xb1, 0x00, 0x00, 0x00, 0x6f, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0xa7,
        0x00, 0x00, 0x00, 0x6b, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00, 0x00, 0x89, 0x00, 0x00,
        0x00, 0x8a, 0x00, 0x00, 0x00, 0xb2, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x85, 0x00, 0x00, 0x00, 0x86,
        0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x9e, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00,
        0x00, 0x87, 0x00, 0x00, 0x00, 0xb3, 0x00, 0x00, 0x00, 0x9a, 0x00, 0x00, 0x00, 0x7b, 0x00, 0x00, 0x00, 0x83,
        0x00, 0x00, 0x00, 0xa9, 0x00, 0x00, 0x00, 0xb3, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0x9d, 0x00, 0x00,
        0x00, 0x9e, 0x00, 0x00, 0x00, 0xa9, 0x00, 0x00, 0x00, 0xaa, 0x00, 0x00, 0x00, 0x99, 0x00, 0x00, 0x00, 0x9a,
        0x00, 0x00, 0x00, 0xb2, 0x00, 0x00, 0x00, 0x8a, 0x00, 0x00, 0x00, 0x63, 0x00, 0x00, 0x00, 0x6e, 0x00, 0x00,
        0x00, 0xa8, 0x00, 0x00, 0x00, 0x86, 0x00, 0x00, 0x00, 0x62, 0x00, 0x00, 0x00, 0x6a, 0x00, 0x00, 0x00, 0x6e,
        0x00, 0x00, 0x00, 0x6f, 0x00, 0x00, 0x00, 0xb1, 0x00, 0x00, 0x00, 0xb2, 0x00, 0x00, 0x00, 0x6a, 0x00, 0x00,
        0x00, 0x6b, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x55,
        0x00, 0x00, 0x00, 0xb9, 0x00, 0x00, 0x00, 0x71, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x51, 0x00, 0x00,
        0x00, 0xaf, 0x00, 0x00, 0x00, 0x6d, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x71, 0x00, 0x00, 0x00, 0xb9,
        0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00, 0x6d, 0x00, 0x00, 0x00, 0xaf, 0x00, 0x00,
        0x00, 0xae, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0xb5, 0x00, 0x00, 0x00, 0xba, 0x00, 0x00, 0x00, 0x9b,
        0x00, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00, 0xab, 0x00, 0x00, 0x00, 0xb0, 0x00, 0x00, 0x00, 0x97, 0x00, 0x00,
        0x00, 0x4e, 0x00, 0x00, 0x00, 0x9b, 0x00, 0x00, 0x00, 0xba, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x4d,
        0x00, 0x00, 0x00, 0x97, 0x00, 0x00, 0x00, 0xb0, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00,
        0x00, 0xba, 0x00, 0x00, 0x00, 0xb6, 0x00, 0x00, 0x00, 0x57, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0xb0,
        0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0x53, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0x57, 0x00, 0x00,
        0x00, 0xb6, 0x00, 0x00, 0x00, 0xb7, 0x00, 0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0x53, 0x00, 0x00, 0x00, 0xac,
        0x00, 0x00, 0x00, 0xad, 0x00, 0x00, 0x00, 0xb9, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00,
        0x00, 0xb7, 0x00, 0x00, 0x00, 0xaf, 0x00, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0xad,
        0x00, 0x00, 0x00, 0xb7, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0xb9, 0x00, 0x00,
        0x00, 0xad, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00, 0x00, 0xaf, 0x00, 0x00, 0x00, 0x36,
        0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0xbc, 0x00, 0x00, 0x00, 0xbb, 0x00, 0x00, 0x00, 0x3b, 0x00, 0x00,
        0x00, 0x3a, 0x00, 0x00, 0x00, 0xb2, 0x00, 0x00, 0x00, 0xb1, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x34,
        0x00, 0x00, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x39, 0x00, 0x00,
        0x00, 0x88, 0x00, 0x00, 0x00, 0x89, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0xbd, 0x00, 0x00, 0x00, 0x8b,
        0x00, 0x00, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0xb3, 0x00, 0x00, 0x00, 0x87, 0x00, 0x00,
        0x00, 0x88, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00, 0xbd,
        0x00, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0xb3, 0x00, 0x00,
        0x00, 0x2c, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0xa1, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00, 0x2d,
        0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x9d, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00,
        0x00, 0x38, 0x00, 0x00, 0x00, 0xbf, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0x3d,
        0x00, 0x00, 0x00, 0xb5, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00,
        0x00, 0xc4, 0x00, 0x00, 0x00, 0xbf, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00, 0xb6, 0x00, 0x00, 0x00, 0xba,
        0x00, 0x00, 0x00, 0xb5, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0xc1, 0x00, 0x00,
        0x00, 0xc0, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0xb7, 0x00, 0x00, 0x00, 0xb6,
        0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0xbb, 0x00, 0x00, 0x00, 0xc2, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00,
        0x00, 0x3b, 0x00, 0x00, 0x00, 0xb1, 0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0xbb,
        0x00, 0x00, 0x00, 0x73, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0xc2, 0x00, 0x00, 0x00, 0xb1, 0x00, 0x00,
        0x00, 0x6f, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x8e,
        0x00, 0x00, 0x00, 0xbc, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x89, 0x00, 0x00, 0x00, 0x8a, 0x00, 0x00,
        0x00, 0xb2, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0xa2, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00, 0x8b,
        0x00, 0x00, 0x00, 0xbd, 0x00, 0x00, 0x00, 0x9e, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x87, 0x00, 0x00,
        0x00, 0xb3, 0x00, 0x00, 0x00, 0xbd, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00, 0xa1, 0x00, 0x00, 0x00, 0xa2,
        0x00, 0x00, 0x00, 0xb3, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0x9d, 0x00, 0x00, 0x00, 0x9e, 0x00, 0x00,
        0x00, 0xbc, 0x00, 0x00, 0x00, 0x8e, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x72, 0x00, 0x00, 0x00, 0xb2,
        0x00, 0x00, 0x00, 0x8a, 0x00, 0x00, 0x00, 0x63, 0x00, 0x00, 0x00, 0x6e, 0x00, 0x00, 0x00, 0x72, 0x00, 0x00,
        0x00, 0x73, 0x00, 0x00, 0x00, 0xbb, 0x00, 0x00, 0x00, 0xbc, 0x00, 0x00, 0x00, 0x6e, 0x00, 0x00, 0x00, 0x6f,
        0x00, 0x00, 0x00, 0xb1, 0x00, 0x00, 0x00, 0xb2, 0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x59, 0x00, 0x00,
        0x00, 0xc3, 0x00, 0x00, 0x00, 0x75, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0xb9,
        0x00, 0x00, 0x00, 0x71, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x75, 0x00, 0x00, 0x00, 0xc3, 0x00, 0x00,
        0x00, 0xc2, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x71, 0x00, 0x00, 0x00, 0xb9, 0x00, 0x00, 0x00, 0xb8,
        0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0xbf, 0x00, 0x00, 0x00, 0xc4, 0x00, 0x00, 0x00, 0x9f, 0x00, 0x00,
        0x00, 0x9c, 0x00, 0x00, 0x00, 0xb5, 0x00, 0x00, 0x00, 0xba, 0x00, 0x00, 0x00, 0x9b, 0x00, 0x00, 0x00, 0x4f,
        0x00, 0x00, 0x00, 0x9f, 0x00, 0x00, 0x00, 0xc4, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x4e, 0x00, 0x00,
        0x00, 0x9b, 0x00, 0x00, 0x00, 0xba, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0xc4,
        0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x5b, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0xba, 0x00, 0x00,
        0x00, 0xb6, 0x00, 0x00, 0x00, 0x57, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00, 0x5b, 0x00, 0x00, 0x00, 0xc0,
        0x00, 0x00, 0x00, 0xc1, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0x57, 0x00, 0x00, 0x00, 0xb6, 0x00, 0x00,
        0x00, 0xb7, 0x00, 0x00, 0x00, 0xc3, 0x00, 0x00, 0x00, 0x59, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00, 0xc1,
        0x00, 0x00, 0x00, 0xb9, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0xb7, 0x00, 0x00,
        0x00, 0xc1, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0xc2, 0x00, 0x00, 0x00, 0xc3, 0x00, 0x00, 0x00, 0xb7,
        0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0xb9, 0x00, 0x00, 0x00, 0x31, 0x00, 0x00,
        0x00, 0x30, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00, 0x00, 0xc5, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x35,
        0x00, 0x00, 0x00, 0xbc, 0x00, 0x00, 0x00, 0xbb, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00,
        0x00, 0x90, 0x00, 0x00, 0x00, 0x91, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x8c,
        0x00, 0x00, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00, 0xc7, 0x00, 0x00, 0x00, 0x8f, 0x00, 0x00,
        0x00, 0x90, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0xbd, 0x00, 0x00, 0x00, 0x8b, 0x00, 0x00, 0x00, 0x8c,
        0x00, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0xc7, 0x00, 0x00,
        0x00, 0x34, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00, 0xbd, 0x00, 0x00, 0x00, 0x2b,
        0x00, 0x00, 0x00, 0xa4, 0x00, 0x00, 0x00, 0xa5, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00,
        0x00, 0xa0, 0x00, 0x00, 0x00, 0xa1, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x33,
        0x00, 0x00, 0x00, 0xc9, 0x00, 0x00, 0x00, 0xa4, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00,
        0x00, 0xbf, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0xca, 0x00, 0x00, 0x00, 0xce,
        0x00, 0x00, 0x00, 0xc9, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0xc4, 0x00, 0x00,
        0x00, 0xbf, 0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00, 0xca,
        0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0xc1, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00,
        0x00, 0x31, 0x00, 0x00, 0x00, 0xc5, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x36,
        0x00, 0x00, 0x00, 0xbb, 0x00, 0x00, 0x00, 0xc2, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0xc5, 0x00, 0x00,
        0x00, 0x77, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0xbb, 0x00, 0x00, 0x00, 0x73,
        0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0xc2, 0x00, 0x00, 0x00, 0x91, 0x00, 0x00, 0x00, 0x92, 0x00, 0x00,
        0x00, 0xc6, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x8e, 0x00, 0x00, 0x00, 0xbc,
        0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0xa6, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x8f, 0x00, 0x00,
        0x00, 0xc7, 0x00, 0x00, 0x00, 0xa2, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00, 0x8b, 0x00, 0x00, 0x00, 0xbd,
        0x00, 0x00, 0x00, 0xc7, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0xa5, 0x00, 0x00, 0x00, 0xa6, 0x00, 0x00,
        0x00, 0xbd, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00, 0xa1, 0x00, 0x00, 0x00, 0xa2, 0x00, 0x00, 0x00, 0xc6,
        0x00, 0x00, 0x00, 0x92, 0x00, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x76, 0x00, 0x00, 0x00, 0xbc, 0x00, 0x00,
        0x00, 0x8e, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x72, 0x00, 0x00, 0x00, 0x76, 0x00, 0x00, 0x00, 0x77,
        0x00, 0x00, 0x00, 0xc5, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00, 0x00, 0x72, 0x00, 0x00, 0x00, 0x73, 0x00, 0x00,
        0x00, 0xbb, 0x00, 0x00, 0x00, 0xbc, 0x00, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x5d, 0x00, 0x00, 0x00, 0xcd,
        0x00, 0x00, 0x00, 0x79, 0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x59, 0x00, 0x00, 0x00, 0xc3, 0x00, 0x00,
        0x00, 0x75, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00, 0x79, 0x00, 0x00, 0x00, 0xcd, 0x00, 0x00, 0x00, 0xcc,
        0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x75, 0x00, 0x00, 0x00, 0xc3, 0x00, 0x00, 0x00, 0xc2, 0x00, 0x00,
        0x00, 0xa4, 0x00, 0x00, 0x00, 0xc9, 0x00, 0x00, 0x00, 0xce, 0x00, 0x00, 0x00, 0xa3, 0x00, 0x00, 0x00, 0xa0,
        0x00, 0x00, 0x00, 0xbf, 0x00, 0x00, 0x00, 0xc4, 0x00, 0x00, 0x00, 0x9f, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00,
        0x00, 0xa3, 0x00, 0x00, 0x00, 0xce, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00, 0x9f,
        0x00, 0x00, 0x00, 0xc4, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xce, 0x00, 0x00,
        0x00, 0xca, 0x00, 0x00, 0x00, 0x5f, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0xc4, 0x00, 0x00, 0x00, 0xc0,
        0x00, 0x00, 0x00, 0x5b, 0x00, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00, 0x5f, 0x00, 0x00, 0x00, 0xca, 0x00, 0x00,
        0x00, 0xcb, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00, 0x5b, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0xc1,
        0x00, 0x00, 0x00, 0xcd, 0x00, 0x00, 0x00, 0x5d, 0x00, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00, 0xcb, 0x00, 0x00,
        0x00, 0xc3, 0x00, 0x00, 0x00, 0x59, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00, 0xc1, 0x00, 0x00, 0x00, 0xcb,
        0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0xcd, 0x00, 0x00, 0x00, 0xc1, 0x00, 0x00,
        0x00, 0x37, 0x00, 0x00, 0x00, 0xc2, 0x00, 0x00, 0x00, 0xc3, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x27,
        0x00, 0x00, 0x00, 0xd0, 0x00, 0x00, 0x00, 0xcf, 0x00, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00,
        0x00, 0xc6, 0x00, 0x00, 0x00, 0xc5, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0x26, 0x00, 0x00, 0x00, 0x81,
        0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00,
        0x00, 0x91, 0x00, 0x00, 0x00, 0x26, 0x00, 0x00, 0x00, 0xd1, 0x00, 0x00, 0x00, 0x82, 0x00, 0x00, 0x00, 0x81,
        0x00, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00, 0xc7, 0x00, 0x00, 0x00, 0x8f, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00,
        0x00, 0x26, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00, 0xd2, 0x00, 0x00, 0x00, 0xd1, 0x00, 0x00, 0x00, 0x2f,
        0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0xc7, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00,
        0x00, 0x95, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0xd2, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0xa4,
        0x00, 0x00, 0x00, 0xa5, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00,
        0x00, 0xd3, 0x00, 0x00, 0x00, 0x95, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0xc9,
        0x00, 0x00, 0x00, 0xa4, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0xd4, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00,
        0x00, 0xd3, 0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0xca, 0x00, 0x00, 0x00, 0xce, 0x00, 0x00, 0x00, 0xc9,
        0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x29, 0x00, 0x00, 0x00, 0xd5, 0x00, 0x00, 0x00, 0xd4, 0x00, 0x00,
        0x00, 0x33, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00, 0xca, 0x00, 0x00, 0x00, 0x28,
        0x00, 0x00, 0x00, 0xcf, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00, 0x00, 0x29, 0x00, 0x00, 0x00, 0x31, 0x00, 0x00,
        0x00, 0xc5, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0xcf, 0x00, 0x00, 0x00, 0x68,
        0x00, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00, 0x00, 0xc5, 0x00, 0x00, 0x00, 0x77, 0x00, 0x00,
        0x00, 0x78, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x00, 0xd0,
        0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0x91, 0x00, 0x00, 0x00, 0x92, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00,
        0x00, 0x30, 0x00, 0x00, 0x00, 0x93, 0x00, 0x00, 0x00, 0x7a, 0x00, 0x00, 0x00, 0x82, 0x00, 0x00, 0x00, 0xd1,
        0x00, 0x00, 0x00, 0xa6, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x8f, 0x00, 0x00, 0x00, 0xc7, 0x00, 0x00,
        0x00, 0xd1, 0x00, 0x00, 0x00, 0xd2, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x93, 0x00, 0x00, 0x00, 0xc7,
        0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0xa5, 0x00, 0x00, 0x00, 0xa6, 0x00, 0x00, 0x00, 0xd0, 0x00, 0x00,
        0x00, 0x7f, 0x00, 0x00, 0x00, 0x61, 0x00, 0x00, 0x00, 0x69, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00, 0x00, 0x92,
        0x00, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x76, 0x00, 0x00, 0x00, 0x69, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00,
        0x00, 0xcf, 0x00, 0x00, 0x00, 0xd0, 0x00, 0x00, 0x00, 0x76, 0x00, 0x00, 0x00, 0x77, 0x00, 0x00, 0x00, 0xc5,
        0x00, 0x00, 0x00, 0xc6, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0xd7, 0x00, 0x00,
        0x00, 0x66, 0x00, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x5d, 0x00, 0x00, 0x00, 0xcd, 0x00, 0x00, 0x00, 0x79,
        0x00, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x66, 0x00, 0x00, 0x00, 0xd7, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00,
        0x00, 0x78, 0x00, 0x00, 0x00, 0x79, 0x00, 0x00, 0x00, 0xcd, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0x95,
        0x00, 0x00, 0x00, 0xd3, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00, 0xa4, 0x00, 0x00,
        0x00, 0xc9, 0x00, 0x00, 0x00, 0xce, 0x00, 0x00, 0x00, 0xa3, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x96,
        0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0xa3, 0x00, 0x00,
        0x00, 0xce, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0xd4,
        0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xce, 0x00, 0x00, 0x00, 0xca, 0x00, 0x00,
        0x00, 0x5f, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0xd4, 0x00, 0x00, 0x00, 0xd5,
        0x00, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00, 0x5f, 0x00, 0x00, 0x00, 0xca, 0x00, 0x00, 0x00, 0xcb, 0x00, 0x00,
        0x00, 0xd7, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x00, 0x00, 0xd5, 0x00, 0x00, 0x00, 0xcd,
        0x00, 0x00, 0x00, 0x5d, 0x00, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00, 0xd5, 0x00, 0x00,
        0x00, 0x29, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00, 0x00, 0xd7, 0x00, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00, 0x32,
        0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0xcd, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xc0, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x0a, 0xb9, 0x00, 0xe6, 0x16, 0x57, 0x22, 0xc0, 0x08, 0x67, 0xed, 0xd7, 0x8c, 0xf3, 0x2a, 0x40, 0x08,
        0x93, 0x0c, 0xe4, 0x23, 0x2e, 0x2d, 0x40, 0x0a, 0xdb, 0xae, 0xf6, 0xb6, 0x64, 0xb9, 0x40, 0x08, 0x01, 0x02,
        0x45, 0x0d, 0x10, 0xc3, 0x3f, 0xf0, 0x03, 0x57, 0x24, 0xfe, 0x03, 0xda, 0xbf, 0xf0, 0x16, 0x60, 0xea, 0xcd,
        0xf8, 0x3a, 0xc0, 0x09, 0x11, 0xe3, 0x63, 0x56, 0x24, 0x87, 0xc0, 0x08, 0x4d, 0xb1, 0x06, 0xba, 0xa2, 0x60,
        0x40, 0x08, 0x00, 0x9e, 0xd7, 0xe4, 0x5a, 0x35, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xc0, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x3f, 0xef, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x3f, 0xef,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x02, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02,
        0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xef,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xef,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x03, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02,
        0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x40, 0x07,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x40, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x02, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x02, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x0a, 0xb9, 0x00, 0xe6, 0x16, 0x57, 0x22, 0xc0, 0x08, 0x67, 0xed, 0xd7, 0x8c,
        0xf3, 0x2a, 0x40, 0x08, 0x93, 0x0c, 0xe4, 0x23, 0x2e, 0x2d, 0x40, 0x0a, 0xdb, 0xae, 0xf6, 0xb6, 0x64, 0xb9,
        0x40, 0x08, 0x01, 0x02, 0x45, 0x0d, 0x10, 0xc3, 0x3f, 0xf0, 0x03, 0x57, 0x24, 0xfe, 0x03, 0xda, 0xbf, 0xf0,
        0x16, 0x60, 0xea, 0xcd, 0xf8, 0x3a, 0xc0, 0x09, 0x11, 0xe3, 0x63, 0x56, 0x24, 0x87, 0xc0, 0x08, 0x4d, 0xb1,
        0x06, 0xba, 0xa2, 0x60, 0x40, 0x08, 0x00, 0x9e, 0xd7, 0xe4, 0x5a, 0x35, 0xc0, 0x0a, 0xb9, 0x00, 0xe6, 0x16,
        0x57, 0x22, 0xc0, 0x08, 0x67, 0xed, 0xd7, 0x8c, 0xf3, 0x2a, 0x40, 0x08, 0x93, 0x0c, 0xe4, 0x23, 0x2e, 0x2d,
        0x40, 0x0a, 0xdb, 0xae, 0xf6, 0xb6, 0x64, 0xb9, 0x40, 0x08, 0x01, 0x02, 0x45, 0x0d, 0x10, 0xc3, 0x3f, 0xf0,
        0x03, 0x57, 0x24, 0xfe, 0x03, 0xdb, 0xbf, 0xf0, 0x16, 0x60, 0xea, 0xcd, 0xf8, 0x39, 0xc0, 0x09, 0x11, 0xe3,
        0x63, 0x56, 0x24, 0x87, 0xc0, 0x08, 0x4d, 0xb1, 0x06, 0xba, 0xa2, 0x60, 0x40, 0x08, 0x00, 0x9e, 0xd7, 0xe4,
        0x5a, 0x35, 0xc0, 0x0a, 0xb9, 0x00, 0xe6, 0x16, 0x57, 0x22, 0xc0, 0x08, 0x67, 0xed, 0xd7, 0x8c, 0xf3, 0x2a,
        0x40, 0x08, 0x93, 0x0c, 0xe4, 0x23, 0x2e, 0x2d, 0x40, 0x0a, 0xdb, 0xae, 0xf6, 0xb6, 0x64, 0xb9, 0x40, 0x08,
        0x01, 0x02, 0x45, 0x0d, 0x10, 0xc3, 0x3f, 0xf0, 0x03, 0x57, 0x24, 0xfe, 0x03, 0xda, 0xbf, 0xf0, 0x16, 0x60,
        0xea, 0xcd, 0xf8, 0x3a, 0xc0, 0x09, 0x11, 0xe3, 0x63, 0x56, 0x24, 0x87, 0xc0, 0x08, 0x4d, 0xb1, 0x06, 0xba,
        0xa2, 0x60, 0x40, 0x08, 0x00, 0x9e, 0xd7, 0xe4, 0x5a, 0x34, 0xc0, 0x0a, 0xb9, 0x00, 0xe6, 0x16, 0x57, 0x22,
        0xc0, 0x08, 0x67, 0xed, 0xd7, 0x8c, 0xf3, 0x2a, 0x40, 0x08, 0x93, 0x0c, 0xe4, 0x23, 0x2e, 0x2d, 0x40, 0x0a,
        0xdb, 0xae, 0xf6, 0xb6, 0x64, 0xb9, 0x40, 0x08, 0x01, 0x02, 0x45, 0x0d, 0x10, 0xc3, 0x3f, 0xf0, 0x03, 0x57,
        0x24, 0xfe, 0x03, 0xda, 0xbf, 0xf0, 0x16, 0x60, 0xea, 0xcd, 0xf8, 0x3a, 0xc0, 0x09, 0x11, 0xe3, 0x63, 0x56,
        0x24, 0x87, 0xc0, 0x08, 0x4d, 0xb1, 0x06, 0xba, 0xa2, 0x60, 0x40, 0x08, 0x00, 0x9e, 0xd7, 0xe4, 0x5a, 0x34,
        0xc0, 0x0a, 0xb9, 0x00, 0xe6, 0x16, 0x57, 0x22, 0xc0, 0x08, 0x67, 0xed, 0xd7, 0x8c, 0xf3, 0x2a, 0x40, 0x08,
        0x93, 0x0c, 0xe4, 0x23, 0x2e, 0x2d, 0x40, 0x0a, 0xdb, 0xae, 0xf6, 0xb6, 0x64, 0xb9, 0x40, 0x08, 0x01, 0x02,
        0x45, 0x0d, 0x10, 0xc3, 0x3f, 0xf0, 0x03, 0x57, 0x24, 0xfe, 0x03, 0xda, 0xbf, 0xf0, 0x16, 0x60, 0xea, 0xcd,
        0xf8, 0x3a, 0xc0, 0x09, 0x11, 0xe3, 0x63, 0x56, 0x24, 0x87, 0xc0, 0x08, 0x4d, 0xb1, 0x06, 0xba, 0xa2, 0x60,
        0x40, 0x08, 0x00, 0x9e, 0xd7, 0xe4, 0x5a, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xfb,
        0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xaa, 0xbf, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xaa, 0xbc, 0xb1, 0xa6, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x3f, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xa8, 0x3f, 0xfb, 0xb6, 0x7a, 0xe8, 0x58,
        0x4c, 0xaa, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x3f, 0xe0, 0xb3, 0x17, 0xb3, 0x8e, 0x9a, 0x9a, 0x40, 0x04, 0xa1, 0x15, 0xcf, 0x9d, 0xa3, 0x56, 0x40, 0x06,
        0x8e, 0x9f, 0x90, 0x1b, 0xac, 0xc3, 0x3f, 0xf4, 0x5f, 0x15, 0x0a, 0xbd, 0x1d, 0xd7, 0xc0, 0x00, 0x5f, 0xf7,
        0xd7, 0xbd, 0xfe, 0x90, 0xc0, 0x0a, 0xf1, 0x0d, 0x18, 0x85, 0xfb, 0xd0, 0xc0, 0x0a, 0xa8, 0xa6, 0x19, 0x12,
        0x5e, 0xdc, 0xbf, 0xf5, 0xe6, 0x13, 0x67, 0x45, 0xd0, 0x8c, 0xc0, 0x09, 0x7f, 0x08, 0xc2, 0x31, 0xf5, 0x25,
        0xc0, 0x0b, 0x4c, 0x51, 0xef, 0x67, 0x04, 0x17, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xfb,
        0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xaa, 0x3f, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xaa, 0x3c, 0xb1, 0xa6, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xbf, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xa8, 0xbf, 0xfb, 0xb6, 0x7a, 0xe8, 0x58,
        0x4c, 0xaa, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xfb,
        0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xaa, 0x3f, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xab, 0x3c, 0xa5, 0x2d, 0x99,
        0x99, 0x99, 0x99, 0x9a, 0xbf, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xa8, 0xbf, 0xfb, 0xb6, 0x7a, 0xe8, 0x58,
        0x4c, 0xaa, 0x3f, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xaa, 0x3f, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xab,
        0x3c, 0x8c, 0x3c, 0xcc, 0xcc, 0xcc, 0xcc, 0xca, 0xbf, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xa9, 0xbf, 0xfb,
        0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xab, 0x3f, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xab, 0x3f, 0xfb, 0xb6, 0x7a,
        0xe8, 0x58, 0x4c, 0xa9, 0xbc, 0x8c, 0x3c, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xbf, 0xfb, 0xb6, 0x7a, 0xe8, 0x58,
        0x4c, 0xab, 0xbf, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xaa, 0x3f, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xaa,
        0x3f, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xa9, 0xbc, 0xa5, 0x2d, 0x99, 0x99, 0x99, 0x99, 0x9a, 0xbf, 0xfb,
        0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xab, 0xbf, 0xfb, 0xb6, 0x7a, 0xe8, 0x58, 0x4c, 0xaa, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x03, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x02, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02,
        0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x3f, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x03, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x02, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02,
        0x40, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x40, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x3f, 0xe0, 0xb3, 0x17, 0xb3, 0x8e, 0x9a, 0x9a, 0x40, 0x04, 0xa1, 0x15, 0xcf, 0x9d,
        0xa3, 0x56, 0x40, 0x06, 0x8e, 0x9f, 0x90, 0x1b, 0xac, 0xc3, 0x3f, 0xf4, 0x5f, 0x15, 0x0a, 0xbd, 0x1d, 0xd7,
        0xc0, 0x00, 0x5f, 0xf7, 0xd7, 0xbd, 0xfe, 0x90, 0xc0, 0x0a, 0xf1, 0x0d, 0x18, 0x85, 0xfb, 0xd0, 0xc0, 0x0a,
        0xa8, 0xa6, 0x19, 0x12, 0x5e, 0xdc, 0xbf, 0xf5, 0xe6, 0x13, 0x67, 0x45, 0xd0, 0x8c, 0xc0, 0x09, 0x7f, 0x08,
        0xc2, 0x31, 0xf5, 0x25, 0xc0, 0x0b, 0x4c, 0x51, 0xef, 0x67, 0x04, 0x17, 0x3f, 0xe0, 0xb3, 0x17, 0xb3, 0x8e,
        0x9a, 0x98, 0x40, 0x04, 0xa1, 0x15, 0xcf, 0x9d, 0xa3, 0x56, 0x40, 0x06, 0x8e, 0x9f, 0x90, 0x1b, 0xac, 0xc3,
        0x3f, 0xf4, 0x5f, 0x15, 0x0a, 0xbd, 0x1d, 0xd8, 0xc0, 0x00, 0x5f, 0xf7, 0xd7, 0xbd, 0xfe, 0x90, 0xc0, 0x0a,
        0xf1, 0x0d, 0x18, 0x85, 0xfb, 0xd0, 0xc0, 0x0a, 0xa8, 0xa6, 0x19, 0x12, 0x5e, 0xdc, 0xbf, 0xf5, 0xe6, 0x13,
        0x67, 0x45, 0xd0, 0x8d, 0xc0, 0x09, 0x7f, 0x08, 0xc2, 0x31, 0xf5, 0x25, 0xc0, 0x0b, 0x4c, 0x51, 0xef, 0x67,
        0x04, 0x17, 0x3f, 0xe0, 0xb3, 0x17, 0xb3, 0x8e, 0x9a, 0x9a, 0x40, 0x04, 0xa1, 0x15, 0xcf, 0x9d, 0xa3, 0x56,
        0x40, 0x06, 0x8e, 0x9f, 0x90, 0x1b, 0xac, 0xc3, 0x3f, 0xf4, 0x5f, 0x15, 0x0a, 0xbd, 0x1d, 0xd7, 0xc0, 0x00,
        0x5f, 0xf7, 0xd7, 0xbd, 0xfe, 0x90, 0xc0, 0x0a, 0xf1, 0x0d, 0x18, 0x85, 0xfb, 0xd0, 0xc0, 0x0a, 0xa8, 0xa6,
        0x19, 0x12, 0x5e, 0xdc, 0xbf, 0xf5, 0xe6, 0x13, 0x67, 0x45, 0xd0, 0x8c, 0xc0, 0x09, 0x7f, 0x08, 0xc2, 0x31,
        0xf5, 0x25, 0xc0, 0x0b, 0x4c, 0x51, 0xef, 0x67, 0x04, 0x17, 0x3f, 0xe0, 0xb3, 0x17, 0xb3, 0x8e, 0x9a, 0x9a,
        0x40, 0x04, 0xa1, 0x15, 0xcf, 0x9d, 0xa3, 0x56, 0x40, 0x06, 0x8e, 0x9f, 0x90, 0x1b, 0xac, 0xc3, 0x3f, 0xf4,
        0x5f, 0x15, 0x0a, 0xbd, 0x1d, 0xd7, 0xc0, 0x00, 0x5f, 0xf7, 0xd7, 0xbd, 0xfe, 0x90, 0xc0, 0x0a, 0xf1, 0x0d,
        0x18, 0x85, 0xfb, 0xd0, 0xc0, 0x0a, 0xa8, 0xa6, 0x19, 0x12, 0x5e, 0xdc, 0xbf, 0xf5, 0xe6, 0x13, 0x67, 0x45,
        0xd0, 0x8c, 0xc0, 0x09, 0x7f, 0x08, 0xc2, 0x31, 0xf5, 0x25, 0xc0, 0x0b, 0x4c, 0x51, 0xef, 0x67, 0x04, 0x17,
        0x3f, 0xe0, 0xb3, 0x17, 0xb3, 0x8e, 0x9a, 0x9a, 0x40, 0x04, 0xa1, 0x15, 0xcf, 0x9d, 0xa3, 0x56, 0x40, 0x06,
        0x8e, 0x9f, 0x90, 0x1b, 0xac, 0xc3, 0x3f, 0xf4, 0x5f, 0x15, 0x0a, 0xbd, 0x1d, 0xd7, 0xc0, 0x00, 0x5f, 0xf7,
        0xd7, 0xbd, 0xfe, 0x90, 0xc0, 0x0a, 0xf1, 0x0d, 0x18, 0x85, 0xfb, 0xd0, 0xc0, 0x0a, 0xa8, 0xa6, 0x19, 0x12,
        0x5e, 0xdc, 0xbf, 0xf5, 0xe6, 0x13, 0x67, 0x45, 0xd0, 0x8c, 0xc0, 0x09, 0x7f, 0x08, 0xc2, 0x31, 0xf5, 0x25,
        0xc0, 0x0b, 0x4c, 0x51, 0xef, 0x67, 0x04, 0x17, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xbf, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x40, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x07,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x07, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xfe, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x07,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x07, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xbf, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x07,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x07,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x07, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x3f, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x3f, 0xef,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x3f, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xef, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xfe, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd,
        0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xf0,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbf, 0xf0, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0xef, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xbf, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc0, 0x08, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x07, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x55, 0x42, 0x49, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x39, 0x2e, 0x32, 0x62, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x31, 0x31, 0x2f, 0x32, 0x34, 0x2f, 0x32, 0x30, 0x30, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x31,
        0x32, 0x3a, 0x33, 0x34, 0x3a, 0x32, 0x35, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00,
        0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x0c,
        0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
        0x00, 0x11, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x15,
        0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00,
        0x00, 0x1a, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x1e,
        0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00,
        0x00, 0x23, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00, 0x26, 0x00, 0x00, 0x00, 0x27,
        0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x29, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00,
        0x00, 0x2c, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00, 0x30,
        0x00, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00,
        0x00, 0x35, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x39,
        0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x3b, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00,
        0x00, 0x3e, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x42,
        0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x46, 0x00, 0x00,
        0x00, 0x47, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x4b,
        0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x4d, 0x00, 0x00, 0x00, 0x4e, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00,
        0x00, 0x50, 0x00, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0x53, 0x00, 0x00, 0x00, 0x54,
        0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0x57, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00,
        0x00, 0x59, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00, 0x5b, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x5d,
        0x00, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00, 0x5f, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x61, 0x00, 0x00,
        0x00, 0x62, 0x00, 0x00, 0x00, 0x63, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x66,
        0x00, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x69, 0x00, 0x00, 0x00, 0x6a, 0x00, 0x00,
        0x00, 0x6b, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00, 0x6d, 0x00, 0x00, 0x00, 0x6e, 0x00, 0x00, 0x00, 0x6f,
        0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x71, 0x00, 0x00, 0x00, 0x72, 0x00, 0x00, 0x00, 0x73, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05,
        0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00,
        0x00, 0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x0e,
        0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00,
        0x00, 0x13, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x17,
        0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00,
        0x00, 0x1c, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x20,
        0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
        0x00, 0x25, 0x00, 0x00, 0x00, 0x26, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x29,
        0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x00,
        0x00, 0x2e, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0x32,
        0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00,
        0x00, 0x37, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x3b,
        0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x00,
        0x00, 0x40, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00, 0x44,
        0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
        0x00, 0x49, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x4d,
        0x00, 0x00, 0x00, 0x4e, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x51, 0x00, 0x00,
        0x00, 0x52, 0x00, 0x00, 0x00, 0x53, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x56,
        0x00, 0x00, 0x00, 0x57, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x59, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00,
        0x00, 0x5b, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x5d, 0x00, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00, 0x5f,
        0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x61, 0x00, 0x00, 0x00, 0x62, 0x00, 0x00, 0x00, 0x63, 0x00, 0x00,
        0x00, 0x64, 0x00, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x66, 0x00, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x68,
        0x00, 0x00, 0x00, 0x69, 0x00, 0x00, 0x00, 0x6a, 0x00, 0x00, 0x00, 0x6b, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00,
        0x00, 0x6d, 0x00, 0x00, 0x00, 0x6e, 0x00, 0x00, 0x00, 0x6f, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x71,
        0x00, 0x00, 0x00, 0x72, 0x00, 0x00, 0x00, 0x73, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
        0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x07,
        0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00,
        0x00, 0x0c, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x10,
        0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00,
        0x00, 0x15, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x19,
        0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00,
        0x00, 0x1e, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x22,
        0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00, 0x26, 0x00, 0x00,
        0x00, 0x27, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x29, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x2b,
        0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00,
        0x00, 0x30, 0x00, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x34,
        0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00,
        0x00, 0x39, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x3b, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x3d,
        0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00,
        0x00, 0x42, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x46,
        0x00, 0x00, 0x00, 0x47, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00,
        0x00, 0x4b, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x4d, 0x00, 0x00, 0x00, 0x4e, 0x00, 0x00, 0x00, 0x4f,
        0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x52, 0x00, 0x00, 0x00, 0x53, 0x00, 0x00,
        0x00, 0x54, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0x57, 0x00, 0x00, 0x00, 0x58,
        0x00, 0x00, 0x00, 0x59, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00, 0x5b, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00,
        0x00, 0x5d, 0x00, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00, 0x5f, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x61,
        0x00, 0x00, 0x00, 0x62, 0x00, 0x00, 0x00, 0x63, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x65, 0x00, 0x00,
        0x00, 0x66, 0x00, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x69, 0x00, 0x00, 0x00, 0x6a,
        0x00, 0x00, 0x00, 0x6b, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00, 0x6d, 0x00, 0x00, 0x00, 0x6e, 0x00, 0x00,
        0x00, 0x6f, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x71, 0x00, 0x00, 0x00, 0x72, 0x00, 0x00, 0x00, 0x73,
        0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x75, 0x00, 0x00, 0x00, 0x76, 0x00, 0x00, 0x00, 0x77, 0x00, 0x00,
        0x00, 0x78, 0x00, 0x00, 0x00, 0x79, 0x00, 0x00, 0x00, 0x7a, 0x00, 0x00, 0x00, 0x7b, 0x00, 0x00, 0x00, 0x7c,
        0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00,
        0x00, 0x81, 0x00, 0x00, 0x00, 0x82, 0x00, 0x00, 0x00, 0x83, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x85,
        0x00, 0x00, 0x00, 0x86, 0x00, 0x00, 0x00, 0x87, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x89, 0x00, 0x00,
        0x00, 0x8a, 0x00, 0x00, 0x00, 0x8b, 0x00, 0x00, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x8d, 0x00, 0x00, 0x00, 0x8e,
        0x00, 0x00, 0x00, 0x8f, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00, 0x00, 0x91, 0x00, 0x00, 0x00, 0x92, 0x00, 0x00,
        0x00, 0x93, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x95, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00, 0x97,
        0x00, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00, 0x99, 0x00, 0x00, 0x00, 0x9a, 0x00, 0x00, 0x00, 0x9b, 0x00, 0x00,
        0x00, 0x9c, 0x00, 0x00, 0x00, 0x9d, 0x00, 0x00, 0x00, 0x9e, 0x00, 0x00, 0x00, 0x9f, 0x00, 0x00, 0x00, 0xa0,
        0x00, 0x00, 0x00, 0xa1, 0x00, 0x00, 0x00, 0xa2, 0x00, 0x00, 0x00, 0xa3, 0x00, 0x00, 0x00, 0xa4, 0x00, 0x00,
        0x00, 0xa5, 0x00, 0x00, 0x00, 0xa6, 0x00, 0x00, 0x00, 0xa7, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0xa9,
        0x00, 0x00, 0x00, 0xaa, 0x00, 0x00, 0x00, 0xab, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0xad, 0x00, 0x00,
        0x00, 0xae, 0x00, 0x00, 0x00, 0xaf, 0x00, 0x00, 0x00, 0xb0, 0x00, 0x00, 0x00, 0xb1, 0x00, 0x00, 0x00, 0xb2,
        0x00, 0x00, 0x00, 0xb3, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0xb5, 0x00, 0x00, 0x00, 0xb6, 0x00, 0x00,
        0x00, 0xb7, 0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00, 0xb9, 0x00, 0x00, 0x00, 0xba, 0x00, 0x00, 0x00, 0xbb,
        0x00, 0x00, 0x00, 0xbc, 0x00, 0x00, 0x00, 0xbd, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00, 0xbf, 0x00, 0x00,
        0x00, 0xc0, 0x00, 0x00, 0x00, 0xc1, 0x00, 0x00, 0x00, 0xc2, 0x00, 0x00, 0x00, 0xc3, 0x00, 0x00, 0x00, 0xc4,
        0x00, 0x00, 0x00, 0xc5, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00, 0x00, 0xc7, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00,
        0x00, 0xc9, 0x00, 0x00, 0x00, 0xca, 0x00, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0xcd,
        0x00, 0x00, 0x00, 0xce, 0x00, 0x00, 0x00, 0xcf, 0x00, 0x00, 0x00, 0xd0, 0x00, 0x00, 0x00, 0xd1, 0x00, 0x00,
        0x00, 0xd2, 0x00, 0x00, 0x00, 0xd3, 0x00, 0x00, 0x00, 0xd4, 0x00, 0x00, 0x00, 0xd5, 0x00, 0x00, 0x00, 0xd6,
        0x00, 0x00, 0x00, 0xd7, 0x00, 0x00, 0x00, 0xd8
    };

    FILE* file = fopen( filename, "w" );
    if( !file ) return false;

    size_t rval = fwrite( binary_data, sizeof( binary_data ), 1, file );
    fclose( file );
    return rval == 1;
}
