/*
 * Copyright (c) 2008-2024 Jonathan Schleifer <js@nil.im>
 *
 * All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3.0 only,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3.0 along with this program. If not, see
 * <https://www.gnu.org/licenses/>.
 */

#include "config.h"

#import "OHLeftJoyCon.h"
#import "OHLeftJoyCon+Private.h"
#import "OFDictionary.h"
#import "OHGameControllerAxis.h"
#import "OHGameControllerButton.h"
#import "OHGameControllerDirectionalPad.h"

#if defined(OF_LINUX) && defined(OF_HAVE_FILES)
# include <linux/input.h>
# import "evdev_compat.h"
#endif

static OFString *const buttonNames[] = {
	@"L", @"ZL", @"Left Thumbstick", @"-", @"Capture", @"SL", @"SR"
};
static const size_t numButtons = sizeof(buttonNames) / sizeof(*buttonNames);

@implementation OHLeftJoyCon
@synthesize buttons = _buttons, directionalPads = _directionalPads;

- (instancetype)init
{
	self = [super init];

	@try {
		void *pool = objc_autoreleasePoolPush();
		OFMutableDictionary *buttons =
		    [OFMutableDictionary dictionaryWithCapacity: numButtons];
		OFMutableDictionary *directionalPads;
		OHGameControllerAxis *xAxis, *yAxis;
		OHGameControllerDirectionalPad *directionalPad;
		OHGameControllerButton *up, *down, *left, *right;

		for (size_t i = 0; i < numButtons; i++) {
			OHGameControllerButton *button =
			    [[[OHGameControllerButton alloc]
			    initWithName: buttonNames[i]
				  analog: false] autorelease];
			[buttons setObject: button forKey: buttonNames[i]];
		}
		[buttons makeImmutable];
		_buttons = [buttons retain];

		directionalPads =
		    [OFMutableDictionary dictionaryWithCapacity: 2];

		xAxis = [[[OHGameControllerAxis alloc]
		    initWithName: @"X"
			  analog: true] autorelease];
		yAxis = [[[OHGameControllerAxis alloc]
		    initWithName: @"Y"
			  analog: true] autorelease];
		directionalPad = [[[OHGameControllerDirectionalPad alloc]
		    initWithName: @"Left Thumbstick"
			   xAxis: xAxis
			   yAxis: yAxis
			  analog: true] autorelease];
		[directionalPads setObject: directionalPad
				    forKey: @"Left Thumbstick"];

		up = [[[OHGameControllerButton alloc]
		    initWithName: @"D-Pad Up"
			  analog: false] autorelease];
		down = [[[OHGameControllerButton alloc]
		    initWithName: @"D-Pad Down"
			  analog: false] autorelease];
		left = [[[OHGameControllerButton alloc]
		    initWithName: @"D-Pad Left"
			  analog: false] autorelease];
		right = [[[OHGameControllerButton alloc]
		    initWithName: @"D-Pad Right"
			  analog: false] autorelease];
		directionalPad = [[[OHGameControllerDirectionalPad alloc]
		    initWithName: @"D-Pad"
			      up: up
			    down: down
			    left: left
			   right: right
			  analog: false] autorelease];
		[directionalPads setObject: directionalPad forKey: @"D-Pad"];

		[directionalPads makeImmutable];
		_directionalPads = [directionalPads retain];

		objc_autoreleasePoolPop(pool);
	} @catch (id e) {
		[self release];
		@throw e;
	}

	return self;
}

- (void)dealloc
{
	[_buttons release];
	[_directionalPads release];

	[super dealloc];
}

- (OFDictionary OF_GENERIC(OFString *, OHGameControllerAxis *) *)axes
{
	return [OFDictionary dictionary];
}

#if defined(OF_LINUX) && defined(OF_HAVE_FILES)
- (OHGameControllerButton *)oh_buttonForEvdevButton: (uint16_t)button
{
	switch (button) {
	case BTN_DPAD_UP:
		return [[_directionalPads objectForKey: @"D-Pad"] up];
	case BTN_DPAD_DOWN:
		return [[_directionalPads objectForKey: @"D-Pad"] down];
	case BTN_DPAD_LEFT:
		return [[_directionalPads objectForKey: @"D-Pad"] left];
	case BTN_DPAD_RIGHT:
		return [[_directionalPads objectForKey: @"D-Pad"] right];
	case BTN_TL:
		return [_buttons objectForKey: @"L"];
	case BTN_TL2:
		return [_buttons objectForKey: @"ZL"];
	case BTN_THUMBL:
		return [_buttons objectForKey: @"Left Thumbstick"];
	case BTN_SELECT:
		return [_buttons objectForKey: @"-"];
	case BTN_Z:
		return [_buttons objectForKey: @"Capture"];
	case BTN_TR:
		return [_buttons objectForKey: @"SL"];
	case BTN_TR2:
		return [_buttons objectForKey: @"SR"];
	}

	return nil;
}

- (OHGameControllerAxis *)oh_axisForEvdevAxis: (uint16_t)axis
{
	switch (axis) {
	case ABS_X:
		return [[_directionalPads
		    objectForKey: @"Left Thumbstick"] xAxis];
	case ABS_Y:
		return [[_directionalPads
		    objectForKey: @"Left Thumbstick"] yAxis];
	}

	return nil;
}
#endif
@end
