Configuring oqsprovider
=======================

This document lists all configuration options for `oqsprovider`.

## Pre-build configuration

Significant parts of this code are generated by the script `oqs-template/generate.py`.
This script permits integration and activation of all quantum safe
algorithms made available by [liboqs](https://github.com/open-quantum-safe/liboqs).
The default configuration is as defined in the [algorithm configuration file](oqs-template/generate.yml).

Any PR changing this default must include all files changed by `generate.py`.

## Build install options

### OPENSSL_ROOT_DIR

Defines a non-standard location for an OpenSSL(v3) installation via `cmake` define.
By default this value is unset, requiring presence of an OpenSSL3 installation
in a standard OS deployment location.

### CMAKE_BUILD_TYPE

By setting this `cmake` configuration option to "Release" all debug output is disabled.
This is the default setting.

In case of any problem, setting this value to "Debug" is _highly_ recommended to
activate further warning messages. In particular, when "Debug" has been set, distinct
[debugging capabilities](https://github.com/open-quantum-safe/oqs-provider/wiki/Debugging)
are activated and additional setup warnings are output.

### CMAKE_PARAMS

This environment variable lets you specify additional flags to pass to `cmake` explicitly when using the `fullbuild.sh` script.

For example, in order to point `cmake` to a specific library, you might run:

```
env CMAKE_PARAMS="-DOPENSSL_CRYPTO_LIBRARY=/opt/lib64/libcrypto.so" bash scripts/fullbuild.sh
```

### liboqs_DIR

This environment variable must be set to the location of the `liboqs` installation to be
utilized in the build.
By default, this is un-set, requiring installation of `liboqs` in a standard
location for the OS.
This uses the [`find_package`](https://cmake.org/cmake/help/latest/command/find_package.html)
command in `cmake`, which checks for local builds of a package at `<PackageName>_DIR`

### NOPUBKEY_IN_PRIVKEY

By setting this to "ON", it can be specified to omit explicitly serializing
the public key in a `privateKey` structure, e.g., for interoperability testing.
The default value is `OFF`.

### OQS_KEM_ENCODERS

By setting this to "ON", `oqsprovider` is configured to provide encoders
and decoders for KEM algorithms both for public and private key file formats.
This increases the size of the provider but enables further use cases.
The underlying OIDs are chosen at random and should not be relied on for
future use. For purposes of interoperability testing the chosen OIDs can
always --at runtime-- be set by [environment variables](https://github.com/open-quantum-safe/oqs-provider/blob/main/ALGORITHMS.md#oids) to arbitrary values
The default value therefore is `OFF`.

### OQS_PROVIDER_BUILD_STATIC

By setting `-DOQS_PROVIDER_BUILD_STATIC=ON` at compile-time, oqs-provider can be
compiled as a static library (`oqs-provider.a`).
When built as a static library, the name of the provider entrypoint is `oqs_provider_init`.
The provider can be added using the [`OSSL_PROVIDER_add_builtin`](https://www.openssl.org/docs/man3.1/man3/OSSL_PROVIDER_add_builtin.html)
function:

```c
#include <openssl/provider.h>

// Entrypoint.
extern OSSL_provider_init_fn oqs_provider_init;

void load_oqs_provider(OSSL_LIB_CTX *libctx) {
  int err;

  if (OSSL_PROVIDER_add_builtin(libctx, "oqsprovider", oqs_provider_init) == 1) {
    if (OSSL_PROVIDER_load(libctx, "oqsprovider") == 1) {
      fputs("successfully loaded `oqsprovider`.", stderr);
    } else {
      fputs("failed to load `oqsprovider`", stderr);
    }
  } else {
    fputs("failed to add the builtin provider `oqsprovider`", stderr);
  }
}
```

> **Warning**
> `OQS_PROVIDER_BUILD_STATIC` and `BUILD_SHARED_LIBS` are mutually exclusive.

See [`examples/static_oqsprovider.c`](examples/static_oqsprovider.c) for a complete
example of how to load oqsprovider using `OSSL_PROVIDER_add_builtin`.

### BUILD_TESTING

By setting this to "OFF", no tests or examples will be compiled.

## Convenience build script options

For anyone interested in building the complete software stack
(`openssl`(v3), `liboqs` and `oqs-provider`) the files `fullbuild.sh`
and `runtests.sh` in the `scripts` directory cater for that.
These files can be configured via the following environment variables:

### OPENSSL_INSTALL

Directory of an existing, non-standard OpenSSL binary distribution.

### OPENSSL_BRANCH

Tag of a specific OpenSSL release to be built and used in testing.
Care is advised setting this to values lower than "3.0.9" due to many known
code deficiencies related to providers in such old OpenSSL branches.

### LIBOQS_BRANCH

This defines the branch of `liboqs` against which `oqs-provider` is built.
This can be used, for example, to facilitate a release of `oqsprovider`
to track an old/stable `liboqs` release. If this variable is not set, the
"main" branch is built.

### liboqs_DIR

If this environment variable is set, `liboqs` is not being built but
used from the directory specified in this variable: Both `include`
and `lib` directories must be present in that location.
By not setting this variable, `liboqs` is build from source.

### MAKE_PARAMS

This environment variable permits passing parameters to the `make`
command used to build `openssl`, e.g., "-j 8" to activate 8-fold
parallel builds to reduce the compilation time on a suitable multicore
machine.

### OQSPROV_CMAKE_PARAMS

This environment variable permits passing parameters to the `cmake`
command used to build `oqsprovider`.

### OQS_SKIP_TESTS

By setting this tests environment variable, testing of specific
algorithm families as listed [here](https://github.com/open-quantum-safe/openssl#supported-algorithms)
can be disabled in testing.
By default this variable is unset.

Example use:

    OQS_SKIP_TESTS="sphincs" ./scripts/runtests.sh

excludes all algorithms of the "Sphincs" family (speeding up testing significantly).

*Note*: By default, interoperability testing with oqs-openssl111 is no longer
performed by default but can be manually enabled in the script `scripts/runtests.sh`.

### OPENSSL_CONF

This test environment variable can be used to instruct `openssl` to use a
configuration file from a non-standard location. Setting this value also
disables the automation logic built into `runtests.sh`, thus requiring
knowledge of `openssl` operations when setting it.
By default this variable is unset.

## LIBOQS configuration options

These are [documented in full here](https://github.com/open-quantum-safe/liboqs/wiki/Customizing-liboqs).
One option is of particular context here, particularly if building
`oqs-provider` static, i.e., as a standalone binary not requiring
presence of `liboqs` during deployment:

### OQS_ALGS_ENABLED

In order to reduce the size of the oqsprovider, it is possible to limit the number
of algorithms supported, e.g., to the set of NIST standardized algorithms. This is
facilitated by setting the `liboqs` build option `-DOQS_ALGS_ENABLED=STD` when building
`liboqs`. The list of algorithms supported by `oqs-provider` is defined by
the contents of the file `generate.yml` documented in the [pre-build configuration](#pre-build-configuration).

### OQS_LIBJADE_BUILD

This environment variable lets you specify the build option `-DOQS_LIBJADE_BUILD` for building liboqs when using the `fullbuild.sh` script. If this environment variable is not set `fullbuild.sh` defaults to building liboqs with `-DOQS_LIBJADE_BUILD=ON`.

When building liboqs with `-DOQS_LIBJADE_BUILD=ON`, default implementations of post-quantum algorithms are replaced with formally verified implementations sourced from [libjade](https://github.com/formosa-crypto/libjade) (if liboqs provides such an implementation.) Please refer to [liboqs documentation](https://github.com/open-quantum-safe/liboqs/blob/main/CONFIGURE.md#oqs_libjade_build) to see which algorithms have formally verified implementations and learn more about the `-DOQS_LIBJADE_BUILD` build option.

For example, in order to build liboqs with `-DOQS_LIBJADE_BUILD=OFF`, you might run:

```
env OQS_LIBJADE_BUILD="OFF" bash scripts/fullbuild.sh
```

Alternatively, the string `-DOQS_LIBJADE_BUILD=` along with its value may be passed to the `fullbuild.sh` script via the [`CMAKE_PARAMS`](#cmake_params) environment variable.

## Runtime options

The `openssl` [property selection mechanism](https://www.openssl.org/docs/manmaster/man7/property.html)
can be utilized to make run-time algorithm selections.

### oqsprovider.security_bits

It is possible to select only algorithms of a specific bit strength by using
the openssl property selection mechanism on the key "oqsprovider.security_bits",
e.g., as such: `openssl list -kem-algorithms -propquery oqsprovider.security_bits=256`.
The bit strength of hybrid algorithms is always defined by the bit strength
of the classic algorithm.


