/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#include "./follower-stat.hpp"

namespace sh4lt::monitor {

using namespace std::chrono;

const size_t FollowerStat::kMinUpdateTimeInMSec;

FollowerStat::FollowerStat(logger::Logger::ptr log, fs::path path)
    : path_(std::move(path)),
      bps_fps_update_(system_clock::now()),
      follower_(
          path_.string(),
          [&](void* data, size_t size, const sh4lt_time_info_t* time_info) {
            on_data(data, size, time_info);
          },
          [&](const ShType& shtype) { on_shtype(shtype); },
          [&]() { on_disconnected(); },
          std::move(log)) {}

void FollowerStat::on_data(void* /*data*/, size_t size, const sh4lt_time_info_t* info) {
  const std::lock_guard<std::mutex> lock(mutex_);
  total_size_ += size;
  size_since_last_update_ += size;
  size_ = size;
  frame_count_since_last_update_ += 1;
  buffer_number_ = info->buffer_number;
  buffer_date_ = info->buffer_date;
  if (-1 != info->buffer_duration) {
    buffer_duration_ = info->buffer_duration;
  }
  system_clock_date_ = info->system_clock_date;
}

void FollowerStat::on_shtype(const ShType& shtype) {
  const std::lock_guard<std::mutex> lock(mutex_);
  connected_ = true;
  if (shtype_) ++re_up_;
  shtype_ = shtype;
}

void FollowerStat::on_disconnected() {
  const std::lock_guard<std::mutex> lock(mutex_);
  connected_ = false;
  size_ = 0;
  buffer_duration_ = 0;
}

auto FollowerStat::shtype() const -> ShType {
  const std::lock_guard<std::mutex> lock(mutex_);
  return shtype_;
}

auto FollowerStat::group() const -> std::string {
  const std::lock_guard<std::mutex> lock(mutex_);
  return shtype_.group();
}

auto FollowerStat::label() const -> std::string {
  const std::lock_guard<std::mutex> lock(mutex_);
  return shtype_.label();
}

auto FollowerStat::id() const -> std::size_t {
  const std::lock_guard<std::mutex> lock(mutex_);
  return shtype_.id();
}

auto FollowerStat::format_date(int64_t rawdate) -> std::string {
  std::ostringstream timecode;
  auto date = nanoseconds(rawdate);
  auto numdays = duration_cast<days>(date).count();
  if (0 != numdays) timecode << duration_cast<days>(date).count() << "_";
  timecode << std::setfill('0') << std::setprecision(0) << std::setw(2)
           << duration_cast<hours>(date).count() % 24 << ":";
  timecode << std::setfill('0') << std::setprecision(0) << std::setw(2)
           << duration_cast<minutes>(date).count() % 60 << ":";
  timecode << std::setfill('0') << std::setprecision(0) << std::setw(2)
           << duration_cast<seconds>(date).count() % 60 << ".";
  timecode << std::setfill('0') << std::setprecision(0) << std::setw(3)
           << duration_cast<milliseconds>(date).count() % 1000;
  return timecode.str();
}

void FollowerStat::update_fps_bps() {
  auto now = system_clock::now();
  auto delta = now - bps_fps_update_;
  //  do not update if less than 1 sec of data has been collected
  if (delta < milliseconds(kMinUpdateTimeInMSec)) return;
  bps_ = 8 * size_since_last_update_ / (duration_cast<milliseconds>(delta).count() / 1000.f);
  fps_ = frame_count_since_last_update_ /
         static_cast<float>(duration_cast<milliseconds>(delta).count() / 1000.f);
  size_since_last_update_ = 0;
  frame_count_since_last_update_ = 0;
  bps_fps_update_ = now;
}

auto FollowerStat::stats() -> InfoTree::ptr {
  auto info = InfoTree::make();
  auto shtype = shtype_.as_info_tree();
  {
    const std::lock_guard<std::mutex> lock(mutex_);
    update_fps_bps();
    info->graft("shtype", shtype);
    info->vgraft("connected", connected_);
    info->vgraft("stats.size_kB", size_ / 1000.f);
    info->vgraft("stats.total_size", total_size_);
    info->vgraft("stats.buffer_number", buffer_number_);
    info->vgraft("stats.buffer_date", buffer_date_);
    info->vgraft("stats.dur_ms", buffer_duration_ / 1000000.f);
    info->vgraft("stats.system_clock_date", system_clock_date_);
    info->vgraft("stats.timecode", format_date(buffer_date_));
    info->vgraft("stats.Mbps", bps_ / 1000000.f);
    info->vgraft("stats.reup", re_up_);
    info->vgraft("stats.fps", fps_);
  }
  return info;
}

auto FollowerStat::path() const -> fs::path { return path_; }

}  // namespace sh4lt::monitor
