import pandas as pd
import parsec_dag as pdag
import argparse
import sys

parser = argparse.ArgumentParser(
    description='Example of combining DOT information with HDF5 trace to explore the DAG execution',
    epilog="""This program selects a single PAYLOAD_RECV event from the HDF5 trace,
and finds the origin task and the dependent tasks on the same node that
executed the PAYLOAD_RECV event. It then displays the dates of each of
these events.""")
parser.add_argument('--dot', nargs='+', required=True, help='List of DOT files')
parser.add_argument('--h5', action='store', required=True, help="HDF5 file of a PaRSEC trace")
args = parser.parse_args(sys.argv[1:])

# First, we create the DAG object from a collection of DOT files
# The DOT files have been generated by running the PaRSEC-enabled
# program, with PARSEC_PROF_GRAPHER enabled during the PaRSEC compilation
# and passing to the runtime the option -. with a filename
#     e.g.: mpirun -np 4 ./testing_dpotrf -N 4000 -- -. dpotrf
# This generated dpotrf-0.dot, dpotrf-1.dot, dpotrf-2.dot and dpotrf-3.dot
# that we load here. This operation can take a few seconds to minutes
# depending on the size of the DOT files (e.g. it takes about 30s
# to load a 183MB DOT file on a 32-core Intel Xeo X7550)
dag = pdag.ParsecDAG()
dag.load_parsec_dot_files(args.dot)

# Then, we load the HDF5 trace (obtained by passing the MCA parameter
# profile_filename, then converting the *.prof-* files into a HDF5
# using profile2h5
trace = pd.HDFStore(args.h5)

# In the trace there are events, some from the runtime system (e.g.
# network-related events), and some from the user tasks of the
# PaRSEC-enabled program.

# To increase efficiency, we gather only the user-task events in a
# separate dataframe.
# Unfortunately, there is no perfect way of finding the first user-task
# type in the dictionary. We use the heuristic that everything after PUT_CB
# is a user-defined task.
tasks = trace.events[trace.events.type >= trace.event_types['PUT_CB']+1]
# And then, we re-index it on the 3-tuple (taskpool_id, type, id)
tasks.set_index(['taskpool_id', 'type', 'id'], inplace=True)

# For the example, we select a single network event: the first
# payload receive.
network_event = trace.events[ trace.events.type == trace.event_types['MPI_DATA_PLD_RCV'] ].iloc[0]

# Now, we can find the corresponding task in the events trace by
# just providing the index
task_event = tasks.loc[(int(network_event.tpid), int(network_event.tcid), int(network_event.tid))]

# We can also find the task in the DAG of tasks
task_node = dag.node_from_id(network_event.tpid, network_event.tcid, network_event.tid)
# And then we can compute its successors in the DAG
slist = dag.successors_from_id(network_event.tpid, network_event.tcid, network_event.tid)

# Small caveat: tasks found from the DAG are dicts, not python classes, so one
# needs to access the fields with the ['field'] notation, and not with the .field
# notation
print "Task %s( %s ) ran on node %d from %g (s) to %g (s)" %\
    (task_node['label'], task_node['param'], task_event.node_id, task_event.begin/1e9, task_event.end/1e9)
print "It generated a payload receive event on node %d from %g (s) to %g (s)" %\
    (network_event.node_id, network_event.begin/1e9, network_event.end/1e9)

# Let's iterate over the successors
for n in slist.keys():
    # Lookup the successor in the DAG to have access to its fields
    s = dag.node_from_name(n)
    # Lookup that task in the set of traced events. We use the indexing
    # again to speedup the process
    succ_event = tasks.loc[(s['tpid'], s['tcid'], s['tid'])]
    # It's possible that this task task class id not execute on the same node
    # as the payload receive event. For example, the origin task could
    # have broadcast the data to many other tasks. So, as we are only
    # interested in the tasks that depend on this payload receive event,
    # we ignore the ones that ran on another node.
    if network_event.node_id == succ_event.node_id:
        # Lookup that task in the DAG to get access to its fields
        task_node = dag.node_from_name(n)
        print "Then, its successor task %s ( %s ) ran on node %d from %g (s) to %g (s)" %\
          ( task_node['label'], task_node['param'], succ_event.node_id, succ_event.begin/1e9, succ_event.end/1e9)

