---
title: Release notes of the 4.1 version of `TFEL`, `MFront` and `MTest`
author: Thomas Helfer
date: 2022
lang: en-EN
numbersections: true
documentclass: article
from: markdown+tex_math_single_backslash
geometry:
  - margin=2cm
papersize: a4
link-citations: true
colorlinks: true
figPrefixTemplate: "$$i$$"
tblPrefixTemplate: "$$i$$"
secPrefixTemplate: "$$i$$"
eqnPrefixTemplate: "($$i$$)"
---

\newcommand{\tns}[1]{{\underset{\tilde{}}{\mathbf{#1}}}}
\newcommand{\tenseur}[1]{\underline{#1}}
\newcommand{\tenseurq}[1]{\underline{\mathbf{#1}}}
\newcommand{\tsigma}{\underline{\sigma}}
\newcommand{\sigmaeq}{\sigma_{\mathrm{eq}}}
\newcommand{\paren}[1]{{\left(#1\right)}}
\newcommand{\Frac}[2]{{{\displaystyle \frac{\displaystyle #1}{\displaystyle #2}}}}
\newcommand{\deriv}[2]{{\displaystyle \frac{\displaystyle \partial #1}{\displaystyle \partial #2}}}
\newcommand{\bts}[1]{{\left.#1\right|_{t}}}
\newcommand{\ets}[1]{{\left.#1\right|_{t+\Delta\,t}}}

The page describes the new functionalities of Version 4.1 of the
`TFEL` project.

# Known incompatiblities

## New restrictions in user defined variable names

The user defined variables shall not start with `mfront_` which is used
internally.

## New reserved keywords in `MFront`

As described in Section @sec:tfel_4.1:mfront:initialize_functions,
behaviours can now declare initialize functions. Each initialize
function has a dedicated name. If `@initialize_function@` be the name of
a initialize function, `MFront` will automatically register
`execute@initialize_function@IntializeFunction` as a member name.

As described in Section @sec:tfel_4.1:mfront:postprocessings, behaviours
can now declare post-processings. Each post-processing has a dedicated
name. If `@postprocessing@` be the name of a post-processing, `MFront`
will automatically register `execute@postprocessing@PostProcessing` as a
member name.

# `TFEL/Config` improvements

## The `TFEL_HOST`, `TFEL_DEVICE` and `TFEL_HOST_DEVICE` macros {#sec:tfel_4.1:tfel_config:device_specification}

The `TFEL_HOST` decorator can be used to mark functions that shall be
used on the host.

The `TFEL_DEVICE` decorator can be used to mark functions that shall be
used on the device.

The `TFEL_HOST_DEVICE` decorator can be used to mark functions that
shall be used on the host and on the device.

# `TFEL/Exceptions` improvements

## Disabling `reportContractViolation` on GPUs {#sec:tfel_4.1:tfel_exception:reportContractViolation_gpu}

The `TFEL_NO_REPORT_CONTRACT_VIOLATION` macro replaces the standard
implementation of the `reportContractViolation` by a dummy version which
does nothing.

This macros is automatically defined with `CUDA` and `SYCL`.

# `TFEL/System` improvements

## Improvements to the `ExternalLibraryManager` class

### Retrieving the unit system of an entry point {#sec:tfel_4.1:system:elm:unit_system}

The `getUnitSystem` allow to retrieve the unit system associated with an
entry point (material property, behaviour, model).

### Retrieving initialize functions generated by the `generic` interface {#sec:tfel_4.1:system:elm:initialize_functions}

The `generic` interface generates a function dedicated for each
initialize function and each modelling hypothesis supported (see Section
@sec:tfel_4.1:mfront:initialize_functions for details about
initialize functions).

The `ExternalLibraryManager` class now exposes the following methods:

- `getGenericBehaviourInitializeFunctions`, which returns the
  initialize functions associated with a behaviour.
- `getGenericBehaviourInitializeFunction`, which returns the
  function implementing a initialize function.
- `getGenericBehaviourInitializeFunctionOutputs`, which returns the
  outputs of a initialize function function.
- `getGenericBehaviourInitializeFunctionOutputs`, which returns the
  types of the outputs of a initialize function function.

### Retrieving post-processing functions generated by the `generic` interface {#sec:tfel_4.1:system:elm:postprocessings}

The `generic` interface generates a function dedicated for each
post-processing and each modelling hypothesis supported (see Section
@sec:tfel_4.1:mfront:postprocessings for details about
post-processings).

The `ExternalLibraryManager` class now exposes the following methods:

- `getGenericBehaviourPostProcessingFunctions`, which returns the
  post-processings associated with a behaviour.
- `getGenericBehaviourPostProcessingFunction`, which returns the
  function implementing a post-processing.
- `getGenericBehaviourPostProcessingFunctionOutputs`, which returns the
  outputs of a post-processing function.
- `getGenericBehaviourPostProcessingFunctionOutputs`, which returns the
  types of the outputs of a post-processing function.

### The `hasTemperatureBeenRemovedFromExternalStateVariables` method

In previous versions of `MFront`, the temperature was automatically
defined as the first external state variable by domain specific
languages handling behaviours.

By conventions, for consistency with behaviour interfaces derived from
the `Abaqus`' `UMAT` interface, the temperature was removed from the
list of external state variables exported by the behaviour. This list
can be retrieved using the `getUMATExternalStateVariablesNames` method
of the `ExternalLibraryManager` class.

Following Issue #50 (see Sections
@sec:tfel:4.1:mfront:global_options:temperature_as_first_external_state_variable
and @sec:tfel:4.1:issue:50), this automatic declaration is now optional.

For backward compatibility, the `getUMATExternalStateVariablesNames` method
still return the list of external state variables without the
temperature. To know if the temperature was removed, the user must call
the `hasTemperatureBeenRemovedFromExternalStateVariables` method.

In pratice, if the `hasTemperatureBeenRemovedFromExternalStateVariables`
method returns `true`, the full list of external state variables is
given by the temperature followed by the list of external state
variables returned by the `getUMATExternalStateVariablesNames` method.

### The `getLaw` method {#sec:tfel_4.1:system:elm:getLaw}

The `getLaw` method returns the law name associated with a material
property as defined by the `@Law` keyword in `MFront`.

### The `getAuthor` method {#sec:tfel_4.1:system:elm:getAuthor}

The `getAuthor` method returns the author name associated with an entry
point as defined by the `@Author` keyword in `MFront`. If this author
name is not defined, an empty string is returned.

### The `getDate` method {#sec:tfel_4.1:system:elm:getDate}

The `getDate` method returns the date associated with an entry point as
defined by the `@Date` keyword in `MFront`. If this date is not defined,
an empty string is returned.

### The `getDescription` method {#sec:tfel_4.1:system:elm:getDescription}

The `getDescription` method returns the description associated with an
entry point as defined by the `@Description` keyword in `MFront`. If
this description is not defined, an empty string is returned.

### The `getMaterialPropertyOutput` method {#sec:tfel_4.1:system:elm:getMaterialPropertyOutput}

The `getMaterialPropertyOutput` method returns the external name
associated with the output of a material property.

## New class `ExternalMaterialKnowledgeDescription`

The `ExternalMaterialKnowledgeDescription` gathers information exported
by `MFront` about an entry point (material property or behaviour). This
class has been introduced as a base class for the
`ExternalMaterialPropertyDescription` and `ExternalBehaviourDescription`
classes.

This class exposes the following data members:

- `library`: the name of the library
- `entry_point`: the name of the entry point
- `unit_sytem`: the unit sytem using by an entry point. This can be
  empty.
- `tfel_version`: the version of TFEL used to generate the entry point
- `build_id`: the build identifier (if any)
- `source`: the MFront source file
- `mfront_interface`: the name of the interface used to generate the
  entry point.
- `material`: the name of the material, as introduced by
  the `@Material` keyword in `MFront`.
- `author`: the author of the implementation, as introduced by
  the `@Author` keyword in `MFront`.
- `date`: the date of the implementation, as introduced by
  the `@Date` keyword in `MFront`.
- `description`: the description of the entry point, as introduced by
  the `@Description` keyword in `MFront`.

## Improvements to the `ExternalMaterialPropertyDescription` class

The `ExternalMaterialPropertyDescription` class now inherits from the
`ExternalMaterialKnowledgeDescription` class.

### New data members

The `ExternalMaterialPropertyDescription` class exposes the following
new data members:

- `law`: the name of the law defining the material property as
  introduced by the `@Law` keyword.
- `output`: the external name of the ouput of the the material property.

### New member functions

#### Retrieving the default value of a parameter {#sec:tfel_4.1:system:empd:getParameterDefaultValue}

The `getParameterDefaultValue` returns the default value of a parameter.

#### Retrieving information about bounds of a variable {#sec:tfel_4.1:system:empd:bounds}

- The `hasBounds` method indicates if the given variable has bounds (either lower or upper bounds).
- The `hasLowerBound` method indicates if the given variable has a lower bound.
- The `hasUpperBound` method indicates if the given variable has a upper bound.
- The `getLowerBound` method returns the lower bound of the given variable.
- The `getUpperBound` method returns the upper bound of the given variable.
- The `hasPhysicalBounds` method indicates if the given variable has physical bounds (either lower or upper physical bounds).
- The `hasLowerPhysicalBound` method indicates if the given variable has a lower physical bound.
- The `hasUpperPhysicalBound` method indicates if the given variable has a upper physical bound.
- The `getLowerPhysicalBound` method returns the lower physical bound of the given variable.
- The `getUpperPhysicalBound` method returns the upper physical bound of the given variable.


## Improvements to the `ExternalBehaviourDescription` class

The `ExternalBehaviourDescription` class now inherits from the
`ExternalMaterialKnowledgeDescription` class.

### The `hasTemperatureBeenRemovedFromExternalStateVariables` data member

The `ExternalBehaviourDescription` class now have
`hasTemperatureBeenRemovedFromExternalStateVariables` boolean public
data member which states if the temperature was removed from the list of
external state variables.

# `TFEL/Math` improvements

## Generalized usage of the `constexpr` keyword

## Port to GPUs

## Miscellaneous improvements

### Linear interpolation {#sec:tfel_4.1:tfel_math:issue_264}

The `computeLinearInterpolation` allows to compute the linear
inteporlation of a set of data by the arrays of abscissae and
values respectively.

The `computeLinearInterpolationAndDerivative` returns the a pair
containing the value resulting for the linear interpolation and the
derivative.

The first template argument of those functions is a boolean stating if
extrapolation must be performed.

#### Notes

1. Both functions assumes that the abscissae are ordered.
2. Both functions are `constexpr`
3. Both functions are compatible with quantitites.
4. Both functions can use arrays and tensorial objects as values.

#### Example of usage

~~~~{.cxx}
constexpr std::array<time, 3u> abscissae{time{0}, time{1}, time{2}};
constexpr std::array<stress, 3u> values{stress{1}, stress{2}, stress{4}};
constexpr auto v = tfel::math::computeLinearInterpolation<true>(abscissae, values,
                                                                time{-1});
~~~~

### Interpolation with cubic spline {#sec:tfel_4.1:tfel_math:issue_266}

The `CubicSplineCollocationPoint` structure describes a collocation
point used to build an interpolation by a cubic spline.

Cubic spline interpolation requires a set of such collocation points.
For example, the `CubicSpline` class can build a vector of collocation
point from data. The set of collocation points can also be precomputed.

The `computeCubicSplineInterpolation` allows to perform the
interpolation using a set of collocation points. The
`computeCubicSplineInterpolationAndDerivative` computes the interpolated
value and the derivative at the cubic spline.

The first template argument of those functions is a boolean stating if
extrapolation must be performed.

#### Notes

1. Both functions assumes that the abscissae are ordered.
2. Both functions are `constexpr`
3. Both functions are compatible with quantitites.
4. Both functions can use arrays and tensorial objects as values.

#### Example of usage

~~~~{.cxx}
using CollocationPoint =
    tfel::math::CubicSplineCollocationPoint<time, stress>;
constexpr auto pts = std::array<CollocationPoint, 3u>{
    CollocationPoint{time{0}, stress{1}, stressrate{0.75}},  //
    CollocationPoint{time{1}, stress{2}, stressrate{1.5}},   //
    CollocationPoint{time{2}, stress{4}, stressrate{2.25}}};
constexpr auto f_values = std::array<stress, 4u>{
    tfel::math::computeCubicSplineInterpolation<true>(pts, time{-1}),
    tfel::math::computeCubicSplineInterpolation<true>(pts, time{0.4}),
    tfel::math::computeCubicSplineInterpolation<true>(pts, time{1.2}),
    tfel::math::computeCubicSplineInterpolation<true>(pts, time{3}),
~~~~

### Extension of the `derivative_type` metafunction to higher order derivatives 

The `derivative_type` metafunction now accepts multiple arguments
defining the variable types with respect to which the function type is
derived.

#### Example

The declarations:

~~~~{.cxx}
// declaration valid in Version 4.0 and 4.1
auto d2E_dT2 = derivative_type<derivative_type<stress, temperature>, temperature>{};
~~~~

and

~~~~{.cxx}
// declaration valid in Version 4.1
auto d2E_dT2 = derivative_type<stress, temperature, temperature>{};
~~~~

are now equivalent.

### Add support to clamp values of arrays {#sec:tfel_4.1:tfel_math:issue_256}

The `clamp` method is available in all mutable arrays in `TFEL/Math`,
including all tensorial objects.

#### Example of usage

~~~~{.cxx}
constexpr auto a = [] {
  auto values = tfel::math::fsarray<3u, int>{-14, 12, -3};
  values.clamp(-4, 4);
  return values;
}();
static_assert(a[0] == -4);
static_assert(a[1] == 4);
static_assert(a[2] == -3);
~~~~

# `TFEL/Math/Parser` improvements

## Improved differentiation

The differentiation of formula defined by the `Evaluator` class has been
improved in several manner.

For example, the derivation of the formula `2*sin(x)` would lead in
previous versions in the following ev

# `TFEL/Material` improvements

## Generalized usage of the `constexpr` keyword {#sec:tfel_4.1:tfel_material:constexpr}

## Port to GPUs {#sec:tfel_4.1:tfel_material:gpu}

## Computation of the inverse of the Langevin function {#sec:tfel_4.1:tfel_material:inverse_langevin_function}

The inverse Langevin function is used in many statistically based
network behaviours describing rubber-like materials.

The Langevin function \(\mathcal{L}\) is defined as follows:
\[
\mathcal{L}\paren{x}=\Frac{1}{\coth\paren{x}}-\Frac{1}{x}
\]

The complexity of the inverse Langevin function
\(\mathcal{L}^{-1}\paren{x}\) motivated the development of various
approximations
[@cohen_pade_1991;@bergstrom_constitutive_1998;@jedynak_approximation_2015].

![Comparison of various approximations of the inverse Langenvin
function](img/InverseLangevinFunction.svg){#fig:tfel_4.1:tfel_material:inverse_langevin_function
width=95%}

Figure @fig:tfel_4.1:tfel_material:inverse_langevin_function compares
those approximations. The approximations of Bergström and Boyce
[@bergstrom_constitutive_1998] and Jedynak [@jedynak_approximation_2015]
are undistinguishable. See Jedynak for a quantitative discussion of the
error generated by those approximations [@jedynak_approximation_2015].
It is worth noting that all those approximations mostly differs near the
pole of inverse Langevin function \(\mathcal{L}^{-1}\paren{x}\).

The `InverseLangevinFunctionApproximations` enumeration lists the
approximations that have been implemented and that can be evaluated in a
`constexpr` context:

- The value `COHEN_1991` is associated with the approximation proposed by Cohen [@cohen_pade_1991]:
  \[
  \mathcal{L}^{-1}\paren{x} \approx
  y\Frac{3-y^{2}}{1-y^{2}}.
  \]
- The value `JEDYNAK_2015` is associated with the approximation proposed
  by Jedynak [@jedynak_approximation_2015]:
  \[
  \mathcal{L}^{-1}\paren{x} \approx
  y \, \Frac{c_{0} + c_{1}\,y + c_{2}\,y^{2}}{1 + d_{1}\,y + d_{2}\,y^{2}}
  \]
- `KUHN_GRUN_1942` or `MORCH_2022` are associated with a taylor expansion of
  \(\mathcal{L}^{-1}\paren{x}\) at \(0\):
  \[
  \mathcal{L}^{-1}\paren{x} \approx
  y\,P\paren{y^{2}} \quad\text{with}\quad P\paren{y^{2}}=\sum_{i=0}^{9}c_{i}\,\paren{y^{2}}^{i}
  \]
  where \(P\) is a \(9\)th order polynomial. Hence, the
  Taylor expression is of order \(19\).

The `computeApproximateInverseLangevinFunction` computes one
approximation of the inverse Langevin function and the
`computeApproximateInverseLangevinFunctionAndDerivative` function
computes ne approximation of the inverse Langevin function and its
derivative. These functions have two template parameters: the
approximation selected and the numeric type to be used. By default, the
`JEDYNAK_2015` approximation is used and the numeric type can be deduced
from the type of the argument.

The approximation proposed by Bergström and Boyce
[@bergstrom_constitutive_1998] is given by the following function:
\[
\mathcal{L}^{-1}\paren{x}
\approx
\left\{
\begin{aligned}
c_{1} \tan\paren{c_{2} \, x} + c_{3} \, x &\quad\text{if}\quad \left|x\right| \leq c_{0}\\
\Frac{1}{\mathop{sign}\paren{x}-x}&\quad\text{if}\quad \left|x\right| > c_{0}
\end{aligned}
\right.
\]

The `computeBergstromBoyce1998ApproximateInverseLangevinFunction`
function computes this approximation and the
`computeBergstromBoyce1998ApproximateInverseLangevinFunctionAndDerivative`
function computes this function and its derivative. These functions
can't be declared `constexpr` because of tangent function is not
`constexpr`. These functions have one template parameter, the numeric
type to be used. This template parameter can be automatically deduced
from the type of the argument.

## Example of usage

~~~~{.cxx}
using ApproximationFunctions = InverseLangevinFunctionApproximations;
// compute Cohen's approximation of the inverse Langevin function
const auto v = computeApproximateInverseLangevinFunction<
            ApproximationFunctions::COHEN_1991>(y)
// compute Jedynak's approximation of the inverse Langevin function and its derivative
const auto [f, df] = computeApproximateInverseLangevinFunctionAndDerivative(y)
~~~~

# `TFEL/Glossary` improvements

## Physical bounds

Most entries of the glossary now declare physical bounds. The physical
bounds are associated with an unit system. Currently, only the Internal
System of units (SI) is supported.

# `MFront` improvements

## Selecting an unit system {#sec:mfront:issue_239}

The `@UnitSystem` allows to select an unit system. Currently, only the
Internal System of units (SI) is supported.

### Example of usage

~~~~{.cxx}
@UnitSystem SI;
~~~~

### Allow failure of the `@InitLocalVariables` code block {#sec:mfront:issue_295}

The user can now abort the behaviour integration by returning `false` or
`FAILURE` from the `@InitLocalVariables` code block.

### Automatic definition of physical bounds of variable associated with a glossary entry

If an unit system is declared and that a variable is associated with a
glossary entry and that no physical bounds have been declared for this
variable, `MFront` will automatically declare the physical bounds from
the glossary entry.

Consider the following example:

~~~~{.cxx}
@DSL MaterialLaw;
@Law TestUnitSystem;
@UnitSystem SI;

@Output stress E;
E.setGlossaryName("YoungModulus");

@StateVariable temperature T;
T.setGlossaryName("Temperature");

@Function{
  constexpr auto Ta = temperature{3000};
  E = 150e9 * exp(T / Ta);
};
~~~~

`MFront` will automatically associate the following physical bounds to
the temperature `T`:

~~~~{.bash}
$ mfront-query --has-physical-bounds=Temperature TestUnitSystem.mfront 
true
$ mfront-query --physical-bounds-value=Temperature TestUnitSystem.mfront 
[0:*[
~~~~

## Options to domain specific languages

Domain specific language can have options which can modify their
default behaviour.

### Treat parameters as static variables {#sec:tfel:4.1:mfront:global_options:parameters_as_static_variables}

The boolean option `parameters_as_static_variables` modifies the way
parameters are treated. This option can be declared as follows:

~~~~{.cxx}
@DSL Default{parameters_as_static_variables : true};
~~~~

By default, `MFront` behaves as if this option was set to `false`.

If `true`, the parameters will be treated as static variables. In
particular, the values of the parameters can not be changed as runtime.
From the solver point of view, the behaviour does not declare any
parameter.

If the current domain specific language calls other domain specific
languages (for example, a behaviour calling an external material
property), the value of this option is automatically passed to those
domain specific languages (unless superceeded by global options, as
detailled in Sections @sec:tfel_4.1:mfront:global_dsl_options and
@sec:tfel_4.1:mfront:global_dsl_options:external_file).

### Specifying a build identifier {#sec:tfel:4.1:mfront:global_options:build_identifier}

In previous versions, a build identifier could be specified using the
`TFEL_BUILD_ID` environment variable.

The `build_identifier` option is another way of specifying the build
identifier, as follows:

~~~~{.cxx}
@DSL IsotropicPlasticMisesFlow{
  build_identifier : "Cyrano-3.4"
};
~~~~

However, the `build_identifier` is not meant to be directly specified in
the `MFront` source file. It shall rather be defined on the command line
(see Section @sec:tfel_4.1:mfront:global_dsl_options).

### Automatic declaration of the temperature as the first external state variable for behaviours {#sec:tfel:4.1:mfront:global_options:temperature_as_first_external_state_variable}

In previous versions of `MFront`, the temperature was automatically
defined as the first external state variable by domain specific
languages handling behaviours, as this is required by most behaviour
interfaces derived from `Abaqus`' `UMAT` interface.

This automatic declaration can now be disabled using the
`automatic_declaration_of_the_temperature_as_first_external_state_variable`
boolean option, as follows:

~~~~{.cxx}
@DSL IsotropicPlasticMisesFlow{
  automatic_declaration_of_the_temperature_as_first_external_state_variable : false
};
~~~~

By default, `MFront` behaves as if this option was set to `true`.
Currently, only the generic interface supports behaviours which do not
declare the temperature a the first external state variable.

#### Special case of specialized domain specific languages

`IsotropicMisesCreep`, `IsotropicMisesPlasticFlow`,
`IsotropicStrainHardeningMisesCreep` and `MultipleIsotropicMisesFlows`
domain specific languages used to automatically declare the temperature
at the middle of the time step in a local variable named `T_`. This
declaration is disabled if the
`automatic_declaration_of_the_temperature_as_first_external_state_variable`
option is set to `false`.

### Disabling initialization of parameters from text file {#sec:tfel:4.1:mfront:global_options:parameters_initialization_from_file}

By default, parameters may be initialized from a text file in the
current directory, if this text file exists. This feature is implemented
and enabled by default for behaviours. For material properties, this
feature is implemented by some interfaces (`Cast3M`, `Cyrano`, `Octave`,
`Python`).

This behaviour can now be changed by using the
`parameters_initialization_from_file` boolean option.

### Specifying the default out of bounds policy {#sec:tfel:4.1:mfront:global_options:default_out_of_bounds_policy}

In previous versions, the default out of bound policy was
`tfel::material::None`, which meant that nothing is done when a variable
is out of its bound.

This behaviour can now be changed by using the
`default_out_of_bounds_policy` string option which can take the values
`None`, `Warning` or `Strict`.

### Disabling runtime change of the out of bounds policy {#sec:tfel:4.1:mfront:global_options:out_of_bounds_policy_runtime_modification}

The `out_of_bounds_policy_runtime_modification` boolean option states if
the out of bounds policy can be changed at runtime. By default, this
option is `true`.

If `true`, many interfaces (`Cast3M`, `Python`, `Java`, etc..) uses
environment variables to modify the out of bounds policy.

### Specifying overriding parameters {#sec:tfel:4.1:mfront:global_options:overriding_parameters}

The `overriding_parameters` option allows to specify overriding
parameters. This parameters must be a map associating variables names
and default values of the overriding parameters.

#### Example of usage {.unnumbered}

The following codes allows to turn the temperature, defined by default
as an external state variable, into a parameter:

~~~~{.bash}
$ mfront --obuild --interface=generic                              \
         --behaviour-dsl-option='overriding_parameters:{T:293.15}' \
         Plasticity.mfront 
~~~~

Note that the temperature increment is implicitly overriden but a
parameter whose default value is null.

### Specifying the modelling hypotheses to be treated {#sec:tfel:4.1:mfront:global_options:modelling_hypothesis}

The `modelling_hypothesis` and `modelling_hypotheses` options can be
used to specify the modelling hypotheses to be treated. This option is
specific to behaviours.

The hypotheses must be a subset of the hypotheses supported be the
behaviour.

~~~~{.bash}
$ mfront --obuild --interface=generic                      \
   --behaviour-dsl-option=modelling_hypothesis:PlaneStrain \
   Plasticity.mfront
~~~~

### Defining global DSL options from the command line {#sec:tfel_4.1:mfront:global_dsl_options}

Options passed to domain specific languages can be defined globally
using one of the following command line arguments:

- `--dsl-option`, which allows to define an option for all domain
  specific languages.
- `--material-property-dsl-option`, which allows to define an option for
  all domain specific languages related to material properties.
- `--behaviour-dsl-option`, which allows to define an option for all
  domain specific languages related to behaviours.
- `--model-dsl-option`, which allows to define an option for all domain
  specific languages related to models.

The options defined by command line arguments are merged with the
options defined inside the `MFront` file. In case of conflicts, an
option defined on the command-line overwrites the option defined in the
`MFront` file.

#### Example of usage {.unnumbered}

~~~~{.cxx}
$ mfront --obuild --interface=generic                          \
    --behaviour-dsl-option=parameters_as_static_variables:true \
    Plasticity.mfront
~~~~

### Defining global DSL options from an external file {#sec:tfel_4.1:mfront:global_dsl_options:external_file}

Options passed to domain specific languages can be defined globally
using an external file in a JSON-like format using one of the following
command line arguments: `--dsl-options-file`,
`--material-property-dsl-options-file`, `--behaviour-dsl-options-file`
or `--model-dsl-options-file`.

The options defined by command line arguments are merged with the
options defined inside the `MFront` file. In case of conflicts, an
option defined on the command-line overwrites the option defined in the
`MFront` file.

#### Example of usage {.unnumbered}

~~~~{.cxx}
$ mfront --obuild --interface=generic    \
    --behaviour-dsls-options-file=options.json \
    Plasticity.mfront
~~~~

where the `options.json` file may look like:

~~~~{.json}
overriding_parameters : {T : 293.15, dT : 0},
parameters_as_static_variables : true
~~~~

### Retrieving the list of options associated with a domain specific language {#sec:tfel_4.1:mfront:list_dsl_options}

The list of options associated with a domain specific language can be
retrieved using the `--list-dsl-options` command line argument as
follows:

~~~~{.bash}
$ mfront --list-dsl-options=RungeKutta
- parameters_as_static_variables: boolean stating if the parameter shall be treated as static variables.
- automatic_declaration_of_the_temperature_as_first_external_state_variable: boolean stating if the temperature shall be automatically declared as an external state variable.
~~~~

## Initialize functions for behaviours {#sec:tfel_4.1:mfront:initialize_functions}

The `@InitializeFunction` keyword introduces a code block that can be
used to initialize internal state variables at the very beginning of the
computation. Initalize functions may have user so-called
*initialize funtion variables*.

In this version, only the `generic` interface generates functions
associated with initialize functions (See Section
@sec:tfel_4.1:system:elm:initialize_functions to see how to retrieve
the initialize functions generated by the `generic` interfaces).

Because initialize functions are called before any behaviour
integration, special care to the meaning of the variables must be taken:

- The gradients and external state variable will have their values at
  the beginning of the time step and their increments will be null.
- The thermodynamic forces will have their values at the beginning of
  the time step.
- The state variables and auxiliary state variables will have undefined
  values. The associated increments, if defined, will be null.

Concerning material properties, they have their values at the beginning
of the time step.

> **About initialisation of local variables**
>
> The code block defined by the `@InitLocalVariables` code block
> shall be called before the execution of an initialize function.

### Example of usage

The following code defines an initializer function which initializes the
elastic strain from the value of stress:

~~~~{.cxx}
@InitializeFunction ElasticStrainFromInitialStress{
  const auto K = 2 / (3 * (1 - 2 * nu));
  const auto pr = trace(sig) / 3;
  const auto s = deviator(sig);
  eel = eval((pr / K) * Stensor::Id() + s / mu);
}
~~~~

### About initialize function variables

Initialize function variables are introduced by the
`@InitializeFunctionVariable` keyword.

Initialize function variables are only defined in initialize functions,
and can't be used in the other code blocks.

Contrary most variables (internal state variables, external state
variables, etc.), initialize function variables can be defined after the
first code block. However, care must be taken to declare initialize
function variables **before** their use in an initialize function.

Note that an initialize function variable can be used in differents
initialize function.

## Post-processings of behaviours {#sec:tfel_4.1:mfront:postprocessings}

The `@PostProcessing` keyword introduces a code block that can be used
to perform computations independently of the behaviour integration. The
outputs of post-processings are stored in so-called *post-processing
variables*.

Post-processings are typically meant to be called at the end of a time
step, when the equilibrium has been reached.

In this version, only the `generic` interface generates functions
associated with post-processings (See Section
@sec:tfel_4.1:system:elm:postprocessings to see how to retrieve the
post-processing functions generated by the `generic` interfaces).

Because post-processings are called independently of the behaviour
integration step, special care to the meaning of the variables must be
taken:

- The thermodynamic forces will have their values at the end of the time
  step.
- The state variables and auxiliary state variables will have their
  values at the end of the time step. The associated increments, if
  defined, will be null.

The values of the thermodynamic forces, state variables, auxiliary state
variables at the beginning of the time step are available in a special
data structure named `initial_state`.

Concerning material properties, they have their values at the end of the
time step as usual.

For the gradients and external state variables have their values at the
end of the time step. Their values at the beginning of the time step are
avaiable in the `initial_state` data structure. Their increments have
their usual values.

> **About initialisation of local variables**
>
> The code block defined by the `@InitLocalVariables` code block
> shall be called before the execution of the post-processing.
> However, this code block will be called with the thermodynamic forces, 
> state variables, auxiliary state variables at the end of the time step.

### Example of usage

The following code defines a post-processing computing the principal
strain at the end of the time step:

~~~~{.cxx}
//! principal strains
@PostProcessingVariable tvector<3u,strain> εᵖ;
εᵖ.setEntryName("PrincipalStrain");
//! compute the principal strain
@PostProcessing PrincipalStrain {
  εᵖ = eto.computeEigenValues();
}
~~~~

### About post-processing variables

Post-processing variables are introduced by the
`@PostProcessingVariable` keyword.

Post-processing variables are only defined in post-processings, and
can't be used in the other code blocks.

Contrary most variables (internal state variables, external state
variables, etc.), post-processing variables can be defined after the
first code block. However, care must be taken to declare post-processing
variables **before** their use in a post-processing.

Note that a post-processing variable can be used in differents
post-processings. Typically, one may compute the principal strains in a
dedicated post-processing and in a post-processing computing the
principal strains and the strain eigen vectors.

## Add a `madnex` file to the search paths {#sec:tfel_4.1:mfront:madnex_search_paths}

`MFront` files may depend on other `MFront` files. A `madnex` file can
be used resolve those dependencies using the `--madnex-search-path`
command line argument.

### A first example

Let us consider a `madnex` file containing the `YoungModulusTest`
material property associated with no material and a file `Test.mfront`
requiring to have access to this material property. For example, the
`Test.mfront` may contain an instruction such as:

~~~~{.cxx}
@MaterialLaw "YoungModulusTest";
~~~~

The `Test.mfront` file can be compiled as follows:

~~~~{.cxx}
$ mfront --obuild --interface=generic                 `
         --madnex-search-path=MaterialProperties.mdnx `
         Test.mfront
~~~~

### `madnex` search paths

Options to the `--madnex-search-path` can be decomposed as

`<file_path>:<material_knowledge_type>:<material_identifier>`

where `<material_knowledge_type>` and `<material_identifier>` are
optionals, as in the previous example.

#### Selecting a specific type of material kowledge

`material_knowledge_type` may have one of the following values
`material_property`, `behaviours` and `models`.

If `material_knowledge_type` is not specified, materials properties,
behaviours and models are all considered (in that order).

#### About the material identifier

The `<material_identifier>` is interpreted as a regular expression.

The regular expression that selects material knowledge associated with
all materials is `.+`, but this will exclude material knowledge
associated with no material.

The regular expression `.*` will select material knowledge associated to
all material *and* material knowledge associated with no material. This
is what happen if not `material_identifier` is specified.

The special material identifier `<none>` selects only material knowledge
associated with no material.

#### Examples of madnex search paths

- `MaterialProperties.mdnx`: appends all the materials properties,
  behaviours, models associated to any materials or to no material to
  the search path.
- `MaterialProperties.mdnx:material_property`: appends all the
  materials properties, associated to any materials or to no material to
  the search path.
- `MaterialProperties.mdnx:material_property:Zircaloy4`: appends all
  the materials properties, associated to `Zircaloy4` to the search
  path.

### How `madnex` files are searched

The `madnex` files specified in `madnex` search paths are first search
in the current directory, and then in the directories specified by the
`--search-path` command line arguments.

## Automatic declaration of a `madnex` input file as a `madnex` search path{#sec:tfel_4.1:mfront:automatic_madnex_file_search_path}

When `MFront` uses a `madnex` input file, this file is automatically
added to the `madnex` search path.

### Example of usage

Let us consider a file a `madnex` file containing a behaviour `Test` and
a material property `YoungModulusTest` which is used by the `Test`
behaviour, then the following instructions work as expected:

~~~~{.bash}
$ mfront --obuild --interface=aster --behaviour=Test Example.mdnx 
Treating target : all
The following libraries have been built :
- libAsterBehaviour.so :  astertest
$ mfront-query --list-dependencies --behaviour=Test Example.mdnx 
madnex:Example.mdnx:MaterialProperty::YoungModulusTest 
~~~~

## Improvements to the `StandardElastoViscoPlasticity` brick

### User defined isotropic hardening rule

The `UserDefined` isotropic hardening rule allows the user to specify
the radius of the yield surface as a function of the equivalent plastic
strain `p`.

This function shall be given by a string option named `R` and must
depend on `p`. The function may also depend on other variables. Let `A`
be such a variable. The `UserDefined` isotropic hardening rule will look
if an option named `A` has been given:

- If this option exists, it will be interpreted as a material
  coefficient as usal and this option can be a number, a formula or the
  name of an external `MFront` file.
- If this option does not exist, a suitable variable will be search in
  the variables defined in the behaviour (static variables, parameters,
  material properties, etc...).

If required, the derivative of `R` with respect to `p` can be
provided through the option `dR_dp`. The derivative `dR_dp` can depend
on the variable `R`.

If this derivative is not provided, automatic differentiation will be
used. The user shall be warned that the automatic differentiation
provided by the `tfel::math::Evaluator` class may result in inefficient
code.

#### Example of usage {.unnumbered}

~~~~{.cxx}
@Parameter stress R0 = 200e6;
@Parameter stress Hy = 40e6;
@Parameter real b = 100;

@Brick StandardElastoViscoPlasticity{
  stress_potential : "Hooke" {young_modulus : 150e9, poisson_ratio : 0.3},
  inelastic_flow : "Plastic" {
    criterion : "Mises",
    isotropic_hardening : "UserDefined" {
      R : "R0 + Hy * (1 - exp(-b * p))",     // Yield radius
      dR_dp : "b * (R0 + Hy - R)"
    }
  }
};
~~~~

### New inelastic flow `UserDefinedViscoplasticity`

The `UserDefinedViscoplasticity` inelastic flow allows the user to
specify the viscoplastic strain rate `vp` as a function of `f` and `p`
where:

- `f` is the positive part of the
  \(\phi\paren{\tsigma-\sum_{i}\tenseur{X}_{i}}-\sum_{i}R_{i}\paren{p}\)
  where \(\phi\) is the stress criterion.
- `p` is the equivalent viscoplastic strain.

This function shall be given by a string option named `vp`. This
function must depend on `f`. Dependance to `p` is optional.

The function may also depend on other variables. Let `A` be such a
variable. The `UserDefinedViscoplasticity` flow will look if an option
named `A` has been given to the flow:


- If this option exists, it will be interpreted as a material
  coefficient as usal and this option can be a number, a formula or the
  name of an external `MFront` file.
- If this option does not exist, a suitable variable will be search in
  the variables defined in the behaviour (static variables, parameters,
  material properties, etc...).

If required, the derivatives of `vp` with respect to `f` and `p` can be
provided through the options `dvp_df` and `dvp_dp`. The derivatives
`dvp_df` and `dvp_dp` can depend on two additional variables, `vp` and
`seps`, which denotes the viscoplastic strain rate and a stress
threshold.

If those derivatives are not provided, automatic differentiation will be
used. The user shall be warned that the automatic differentiation
provided by the `tfel::math::Evaluator` class may result in inefficient
code.

#### Example of usage {.unnumbered}

~~~~{.cxx}
@Parameter temperature Ta = 600;
@Parameter strain p0 = 1e-8;

@Brick StandardElastoViscoPlasticity{
  stress_potential : "Hooke" {young_modulus : 150e9, poisson_ratio : 0.3},
  inelastic_flow : "UserDefinedViscoplasticity" {
    criterion : "Mises",
    E : 8.2,
    A : "8e-67 * exp(- T / Ta)",
    m : 0.32,
    vp : "A * (f ** E) / ((p + p0) ** m)",
    dvp_df : "E * vp / (max(f, seps))"
    // dvp_dp is evaluated by automatic differentiation (which is not recommended)
  }
};
~~~~

### Isotropic harderning rule based defined by points

The `Data` isotropic hardening rule allows the user to define an
isotropic hardening rule using a curve defined by a set of pairs of
equivalent strain and equivalent stress.

This isotropic hardening rule can be parametrised using three entries:

- `values`: which must a dictionnary giving the value of the yield
  surface radius as a function of the equivalent plastic strain.
- `interpolation`: which allows to select the interpolation type.
  Possible values are `linear` (default choice) and `cubic_spline`.
- `extrapolation`: which allows to select the extrapolation type.
  Possible values are `bound_to_last_value` (or `constant`) and
  `extrapolation` (default choice).

#### Example of usage

~~~~{.cxx}
@Brick StandardElastoViscoPlasticity{
  stress_potential : "Hooke" {young_modulus : 150e9, poisson_ratio : 0.3},
  inelastic_flow : "Plastic" {
    criterion : "Mises",
    isotropic_hardening : "Data" {
      values : {0 : 150e6, 1e-3 : 200e6, 2e-3 : 400e6},
      interpolation : "linear"
    }
  }
};
~~~~

### Delobelle-Robinet-Schaffler kinematic hardening rule

The Delobelle-Robinet-Schaffler (DRS) kinematic hardening rule has been
introduced to describe orthotropic viscoplasticity of Zircaloy alloys
[@delobelle_model_1996;@schaffler_thermomechanical_1999]. It describes
both dynamic and static recovery by the following evolution law:
\[
\tenseur{\dot{a}}=
\dot{p}\,\tenseurq{E}_{c}\,\colon\,\tenseur{n}
-D\,\dot{p}\,\tenseurq{R}_{d}\,\colon\,\tenseur{a}
-f\,\paren{\Frac{a_{\mathrm{eq}}}{a_{0}}}^{m}\,\deriv{a_{\mathrm{eq}}}{\tenseur{a}}
\]
with \(a_{\mathrm{eq}}=\sqrt{\tenseur{a}\,\colon\,\tenseurq{R}_{s}\,\colon\,\tenseur{a}}\) and 
\(\deriv{a_{\mathrm{eq}}}{\tenseur{a}}=\Frac{\tenseurq{R}_{s}\,\colon\,\tenseur{a}}{a_{\mathrm{eq}}}\)

The three fourth order tensors \(\tenseurq{E}_{c}\),
\(\tenseurq{R}_{d}\) and \(\tenseurq{R}_{s}\) are assumed to have the
same structure as the Hill tensors and are defined by \(6\) components
(see [this page](tensors.html) for details).

The `f` and `a0` parameters are optional and defaults to \(1\).


#### Example

~~~~{.cxx}
    kinematic_hardening : "DRS" {
      C : 150.e9,  // kinematic moduli
      D : 1e2,    // back-strain callback coefficient
      f : 10,
      m : 5,
      Ec : {0.33, 0.33, 0.33, 1, 1, 1},
      Rs : {0.33, 0.63, 0.33, 1, 1, 1},
      Rd : {0.33, 0.33, 0.33, 1, 1, 1}  //
    },
~~~~

## Improvements to the `RungeKutta` domain specific language

### Improvements to the `rkCastem` algorithm{#sec:tfel_4.1:mfront:rkCastem}

The `rkCastem` algorithm compares the values of the stress computed by a
predictor step and corrector step. The error between those values must
be normalised. In previous versions, the existence of a variable named
`young` was assumed.

The `@StressErrorNormalizationFactor` allows to specify this
normalization factor more explicitely and consistenly.

For backward compatibility with previous versions, the `young` variable
will be used, if an appropriate variable is defined and if the
`@StressErrorNormalizationFactor` keyword was not used.

Note that if the `@StressErrorNormalizationFactor` keyword is not used
and that the `young` variable is not defined, `MFront` will look:

1. if the stiffnes matrix is defined and take its first component.
1. if a material property, a parameter or a local variable is associated
  with the `YoungModulus` glossary name.

## New domain specific language `ImplicitCZMDSL`

The domain specific language `ImplicitCZMDSL` allows to implement a
cohesive zone model using an implicit scheme.

## Support for for point-wise models in the `Cast3M` interface {#sec:tfel:4.1:mfront:castem_model_interface}

The `Cast3M` interface has been extended to support point-wise models,
i.e. generic behaviours without gradients, as generated by the
`DefaultModel` DSL, the `RungeKuttaModel` DSL and the `ImplicitModel`
DSL.

Proper support for models will land in `Cast3M` Version 2023. In the
meantime, the generated models can be tested with `MTest` (See Section
@sec:tfel:4.1:mtest:castem_model_interface).

## `Cast3M` interface for point-wise models implemented using the `Model` domain specific language {#sec:tfel:4.1:mfront:castem_model_interface2}

The `Model` domain specific language (DSL) is mostly superseeded by the
domain specific languages introduced in Version 3.4.3 (namely the
`DefaultModel`, `RungeKuttaModel` and `ImplicitModel` DSLs). However,
backward-compatibility still requires to maintain the `Model` domain
specific language.

The generated function has the same prototype as behaviours. From the
`Cast3M` point of view, point-wise models implemented by the `Model`
domain specific language are no different than the ones implemented by
the `DefaultModel`, `RungeKuttaModel` and `ImplicitModel` DSLs.

Proper support for models will land in `Cast3M` Version 2023. In the
meantime, the generated models can be tested with `MTest` (See Section
@sec:tfel:4.1:mtest:castem_model_interface).

## `generic` interface for material properties {#sec:tfel:4.1:mfront:generic_material_property_interface}

The `generic` interface for material properties generates functions
matching the following prototype:

~~~~{.cxx}
mfront_gmp_real (*)(mfront_gmp_OutputStatus* const,       // output status
                    const mfront_gmp_real* const,         // arguments
                    const mfront_gmp_size_type,           // number of arguments
                    const mfront_gmp_OutOfBoundsPolicy);  // out of bounds policy
~~~~

The `mfront_gmp_OutputStatus` structure and the
`mfront_gmp_OutOfBoundsPolicy` enumeration type are described in the
next paragraphs.

The arguments are passed using a continuous array. The number of
arguments is used to make some basic consistency checks. The list of
arguments can be retrieved using the `ExternalLibraryManager` class (See
Section @???).

If the material property declares parameters (and that the
`parameters_as_static_variables` DSL option was not specified), their
names and default values can be retrieved using the
`ExternalLibraryManager` class. The value of those parameters can also
be modified using the `ExternalLibraryManager` class (See Section @???).

> **Note: link with the `cyrano` interface for material properties**
>
> The `generic` interface for material properties is very similar to the
> `cyrano` interface. In pratice, both interfaces shares a common base
> class (called `GenericMaterialPropertyInterfaceBase`).

### The `mfront_gmp_OutputStatus` structure

The `mfront_gmp_OutputStatus` structure describes the output status of
the evaluation of a material property.

This data structure is defined as follows:

~~~~{.cxx}
/*!
 * \brief this structure summarizes the exit status of a function conforming to
 * one of the `generic` material property interface.
 */
typedef struct {
  /*!
   * \brief exit status
   *
   * The exit status is zero if the result has been correctly evaluated.
   *
   * If the exit status is 1, a result has been computed, but it must be used
   * with caution. This is typically used to report that one argument was out of
   * its bounds.
   *
   * All negative values indicates that the result is not usable. For a material
   * property, thereturned is `nan`.
   *
   * For a material property, a negative value has the following meaning:
   *
   * - If the exit status is -1, an argument was out of its physical bounds, or
   *   out of its bounds and a strict out of bounds policy is declared.
   * - If the exit status is -2, a C++ exception was thrown. If the exception
   *   was a child of `std::exception`, the content of the string returned by
   *   the `what` method is copyied in the `message` field. Otherwise, the
   *   message field contains the "unknown exception" string.
   * - If the exit status is -3, an error occured in the `C` library, i.e. the
   *   `errno` value was set to a non zero value during the computation.
   *   The value of `errno` corresponding to the error is stored to in the
   *   `c_error_number` field of this structure. The string returned by
   *   `strerrno` is returned. Note that the `errno` value is always reset to
   *   the  value it had before the call.
   * - If the exit status is -4, the computed value is invalid (either \nan`,
   *   `inf`, or `-inf`).
   * - If the exit status is -5, the number of arguments is invalid.
   */
  int status;
  //! \brief error number reported by the C library.
  int c_error_number;
  /*!
   * \brief bounds status
   * This status has the following meaning:
   * - zero means that no argument was outside its bounds or its physical
   * bounds.
   * - a negative values means that one argument went beyond its physical
   * bounds.
   *   The absolute value gives the rank of this argument (here the rank starts
   * at 1).
   * - a positive value means that one argument went beyond its bounds.
   *   The value gives the rank of this argument (here the rank starts at 1).
   */
  int bounds_status;
  //! \brief error message
  char msg[512];
} mfront_gmp_OutputStatus;  // end of struct mfront_gmp_OutputStatus
~~~~

### The `mfront_gmp_OutOfBoundsPolicy` enumeration type

The `mfront_gmp_OutOfBoundsPolicy` enumeration type is defined as follows:

~~~~{.cxx}
/*!
 * \brief available out of bounds policies
 */
typedef enum {
  GENERIC_MATERIALPROPERTY_NONE_POLICY,    /*!<
                                            * With this policy, nothing is done if
                                            * the arguments are    out of their
                                            * bounds    (checks are not even
                                            * performed).
                                            */
  GENERIC_MATERIALPROPERTY_WARNING_POLICY, /*!<
                                            * With this policy, checks on the
                                            * arguments are performed. If one
                                            * argument if out of its bounds,
                                            * this will be reported in the
                                            * output status and an
                                            * appropriate error message will be
                                            * reported. The computations are
                                            * however performed.
                                            */
  GENERIC_MATERIALPROPERTY_STRICT_POLICY   /*!<
                                            * With this policy, checks on the
                                            * arguments are   performed. If one
                                            * argument   if out of its bounds,
                                            * this   will be reported in the
                                            * output   status and an   appropriate
                                            * error   message will be reported.
                                            */
} mfront_gmp_OutOfBoundsPolicy;  // end of mfront_gmp_OutOfBoundsPolicy
~~~~

## `generic` interface for point-wise models implemented using the `Model` domain specific language {#sec:tfel:4.1:mfront:generic_model_interface}

The `Model` domain specific language (DSL) is mostly superseeded by the
domain specific languages introduced in Version 3.4.3 (namely the
`DefaultModel`, `RungeKuttaModel` and `ImplicitModel` DSLs). However,
backward-compatibility still requires to maintain the `Model` domain
specific language.

It is also interesting to use of the existing point-wise models in
solvers based on the `MFrontGenericInterfaceSupport`. This is the main
motivation of the development for `generic` interface for point-wise
models implemented using the `Model` domain specific language.

An interesting consequence of this development is those point-wise
models can now be tested in `MTest`.

### Example of usage

~~~~{.bash}
$ mfront --obuild --interface=generic UO2_Shrinkage_RAPHAEL2008.mfront
Treating target : all
The following libraries have been built :
- libUO2-generic.so :  UO2_Shrinkage_RAPHAEL2008_AxisymmetricalGeneralisedPlaneStrain UO2_Shrinkage_RAPHAEL2008_AxisymmetricalGeneralisedPlaneStress UO2_Shrinkage_RAPHAEL2008_Axisymmetrical UO2_Shrinkage_RAPHAEL2008_PlaneStress UO2_Shrinkage_RAPHAEL2008_PlaneStrain UO2_Shrinkage_RAPHAEL2008_GeneralisedPlaneStrain UO2_Shrinkage_RAPHAEL2008_Tridimensional
$ 
~~~~

This can be tested in `MTest`. As an exemple, the following test
computes the swelling of uranium dioxide for a constant temperature for
a burn-up increasing from \(0\) to \(5\, at.%\) (time has no physical
meaning here):

~~~~{.cxx}
@ModellingHypothesis 'Tridimensional';
@Behaviour<generic> 'src/libUO2-generic.so' 'UO2_Shrinkage_RAPHAEL2008';

@ExternalStateVariable 'BurnUp_AtPercent' {0 : 0, 1 : 5};
@ExternalStateVariable 'Temperature' 800;

@Times{0, 1 in 100};
~~~~

## Improvements to the `MaterialProperty` DSL

### Dedicated tutorial

A new tutorial dedicated to the `MaterialProperty` DSL is available on
[this page](material-properties.html).

### Physical constants {#sec:mfront:material_property:issue_243}

The physical constants defined in the [`TFEL/PhysicalConstants`
library](physical-constants.html) are available through the
`PhysicalConstants` type alias. This alias is defined by taking the
numeric type and the usage of quantity.

#### Example of usage

~~~~{.cxx}
constexpr auto R = PhysicalConstants::R;
~~~~

## Improvements to the `Model` DSL

### New keywords `@StateVariable` and `@ExternalStateVariable`

The keywords `@StateVariable` and `@ExternalStateVariable` are
synomymous of the `@Output` and `@Input` keywords respectively. Those
aliases were introduced for consistency with behaviours.

### Defining types for parameters, inputs and outputs {#sec:mfront:model:issue_241}

The keywords `@StateVariable`, `@ExternalStateVariable`, `@Output`,
`@Input` and `@Parameters` now allow to specify the type of the
variables they define. See [this page](mfront-types.html) for details on
types supported by `MFront`.

Note that only scalar types are supported by models.

### Support for quantities {#sec:mfront:model:issue_242}

Quantities are now fully supported in the `Model` DSL.

### Physical constants {#sec:mfront:model:issue_244}

The physical constants defined in the [`TFEL/PhysicalConstants`
library](physical-constants.html) are available through the
`PhysicalConstants` type alias. This alias is defined by taking the
numeric type and the usage of quantity.

#### Example of usage

~~~~{.cxx}
constexpr auto R = PhysicalConstants::R;
~~~~

## Miscellaneous improvements

### Extension of the `derivative_type` metafunction to higher order derivatives 

In variable declaration, the `derivative_type` metafunction now accepts
multiple arguments defining the variable types with respect to which the
function type is derived.

#### Example

The following declarations

~~~~{.cxx}
// declaration valid in Version 4.0 and 4.1
@LocalVariable derivative_type<derivative_type<stress, temperature>, temperature> d2E_dT2;
~~~~

and

~~~~{.cxx}
// declaration valid in Version 4.1
@LocalVariable derivative_type<stress, temperature, temperature> d2E_dT2;
~~~~

 are now equivalent.

### Pedantic check

The `--pedantic` option of `MFront` now allows to check several criteria to ensure a good quality assurance of the `.mfront` file.

### Example of usage

~~~~{.bash}
$ mfront --pedantic MFrontFileName.mfront
~~~~

### Support for the `DTAU_DDF` tangent operator for the `generic` interface {#sec:mfront:generic_interface:issue_351}

The `generic` interface for finite strain behaviour can now return the
derivative of the Kirchhoff stress with respect to the spatial increment
of the deformation gradient
\(\ets{\tns{F}}\,\cdot\,\bts{\tns{F}}^{-1}\).

## Generation of climb tensors {#sec:mfront:model:issue_317}

For a given slip system of normal \(\vec{n}\), the climb tensor is
defined as \(\vec{n}\,\otimes\vec{n}\).

If sliding systems are declared, `MFront` generates a class containing
the definition of all normals, slip directions, orientation tensors (see
[this page](singlecrystal.html) for details).

This class now contains the climb tensors:

- The climb tensors of all slip systems are stored in a variable named
  `climb_tensors`
- The climb tensors all the `i`th family of slip systems' is stored in a
  variable named `climb_tensors`i.

> **Note**
> 
> In previous versions of `MFront`, climb tensors could have been
> computed using the normals using `buildFromVectorDiadicProduct`
> method from the `stensor` class.

# `MTest` improvements

## Support for `madnex` file {#sec:tfel:4.1:mtest:madnex_support}

### Exporting an `MTest` test to a `madnex` file

The `TFELMTest` library exposes a data structure named `TestDescription` which
describes an `MTest` file and two functions called respectively
`loadMTestFileContent` and `write`.

This data structure and functions are exported in `python`, as described
in Section @sec:tfel:4.1:python:mtest:export_mtest_file. Section
@sec:tfel:4.1:python:mtest:export_mtest_file:usage provides an example
of use.

#### The `TestDescription` data structure {#sec:tfel:4.1:mtest:TestDescription}

The `TestDescription` data structure exposes the following data members:

- `author`, which describes the author of the test.
- `date`, which describes the date at which the test has been created.
- `scheme`, which describes the type of test. Valid values are `mtest`
  and `ptest`.
- `description`, which describes a description of the test.
- `behaviour`, name of the behaviour to which the test is associated.
  This data member is **required** to export the file in the `madnex`
  file format.
- `material`, name of the material to which the test is associated. This
  data member can be empty.
- `content`, content of the `MTest` file. This content can be filled
  from an existing `MTest` file using the `loadMTestFileContent`
  function.

#### The `loadMTestFileContent` function {#sec:tfel:4.1:mtest:loadMTestFileContent}

The `loadMTestFileContent` function loads the content of an `MTest` file
and stores it in the `content` data member of a `TestDescription` data
structure.

#### The `write` function {#sec:tfel:4.1:mtest:write}

The `write` function exports an `MTest` test, described by a
`TestDescription` data structure, to a file.

The file format is deduced from the extension of the file.

Currently, only extensions associated with the [`madnex` file
format](https://github.com/thelfer/madnex) are supported if `TFEL` is
compiled with support of this file format. Those extensions are: `mdnx`,
`madnex` (deprecated) or `edf` (experimental data file, deprecated).
Note that the behaviour member of the metadata must be specified for
export in the `madnex` file format.

#### Best practices

We highly recommend to use the following substitution variables when
defining the test:

- `@interface@`, which is meant to be replaced by the interface to be
  used. This is very handy if the test can be run for different
  interfaces
- `@library@`, which is meant to be replaced by the path to the shared
  library containing the tested behaviour.
- `@behaviour@`, which contains the name of the function implementing
  the behaviour for the considered interface.

### Executing a test stored in a `madnex` file

To execute a test stored in a `madnex` file, the user must specify:

- the name of the `madnex` file
- the name of the test using the `--test` (or `-t`) command line
  argument.
- the name of the behaviour to which the test is associated using the
  `--behaviour` (or `-b`) command line argument.
- the name of the material is the behaviour to which the test is
  associated is associated with a material using the `--material` (or
  `-m`) command line argument. If the material is not specified, or if
  the special material name `<none>` is used, the behaviour associated
  with the test is assumed not associated with any material.

Note that the `--test` (or `-t`) command line argument can accept
regular expressions to select as set of tests.

#### Example of usage {.unnumbered}

The following example executes the `UniaxialTensileTest` test 
associated with the `Plasticity` behaviour (and not attached to any
material) using the behaviour `cyranoplasticity` compiled with the
`cyrano` interface in a shared library `libCyranoBehaviours.so` located
in the `src` subdirectory and stored in the `Plasticity.mdnx` file:

~~~~{.bash}
$ mtest  --behaviour=Plasticity --test=UniaxialTensileTest        \
         --@interface@=cyrano --@behaviour@="'cyranoplasticity'"  \
         --@library@="'src/libCyranoBehaviours.so'"               \
         Plasticity.mdnx
~~~~

### Execution all tests associated with a behaviour stored in a `madnex` file

The user can execute all tests associated with a behaviour using the
`--all-tests` command line arguments. The user must specify the name of
the behaviour (using the `--behaviour` (or `-b`) command line argument)
and optionally the name of the material (using the `--material` (or
`-m`) command line argument).

~~~~{.bash}
$ mtest  --behaviour=Plasticity --@interface@=cyrano \
         --@behaviour@="'cyranoplasticity'"          \
         --@library@="'src/libCyranoBehaviours.so'"  \ 
         --all-tests Plasticity.mdnx                 \
~~~~

The `--all-tests` command line argument is equivalent to `--test=".+"`.

### Alternative way to select a single test in a `madnex` file

`mtest` allows to select a test inside a `madnex` file using the
following path syntax:

~~~~
madnex:<file>:<material>:<behaviour>:<test>
~~~~

where:

- `<file>` is the path to the `madnex` file.
- `<material>` is the name of the material considered. This name can be
  empty or `<none>` if the considered test is not
  associated to a material.
- `<behaviour>` is the name of the behaviour
- `<test>` is the name of the test

## Support for a boundary condition modelling the effect of a mandrel in pipe modelling {#sec:tfel_4.1:mtest:mandrel}

The effect of a non-deformable mandrel located inside the pipe can be
modelled by defining the evolution of its radius \(R_{m}\) using the
`@MandrelRadiusEvolution` keyword. The inner radius \(R_{i}\) will then
satisfy the following unilateral boundary condition:

\[
R_{i} - R_{m} \geq 0
\]

This boundary condition is imposed by using a Lagrange multiplier. Its
value is given by the contact pressure in the output file. The total
pressure applied to the inner surface of the pipe is given by the sum of
the imposed pressure (if any) and the contact pressure. Only the imposed
inner pressure is used to compute the end cap effect.

This boundary condition is not compatible with:

- boundary conditions imposing the evolution of the inner radius of the
  pipe or the outer radius of the pipe.
- the modelling of a tight pipe. 

### Axial binding

If the evolution of the axial growth of the mandrel is defined using the
`@MandrelAxialGrowthEvolution` keyword, an axial binding between the
mandrel and the pipe is assumed, i.e. the difference between the axial
growth of the pipe \(\varepsilon^{p}_{zz}\) and the axial growth of the
mandrel \(\varepsilon^{p}_{zz}\) is assumed to be constant to its value
when the contact between the mandrel and the pipe is detected:

\[
\varepsilon^{p}_{zz}-\varepsilon^{m}_{zz}=\textrm{Cste}
\]

This axial boundary condition is not compatible with the boundary
condition imposing the evolution of the axial growth of the pipe.

## Wrappers around `3D` behaviours {#sec:tfel_4.1:mtest:small_strain_tridimensional_behaviour_wrapper}

The `SmallStrainTridimensionalBehaviourWrapper` class allows to wrap a
tridimensional behaviour into a behaviour usable in one of the following
modelling hypotheses:

- axisymmetrical generalised plane strain,
- generalised plane strain,
- plane strain.

Shear components of the strain tensor which are not meaningful for the
targeted modelling hypothesis are set to zero.

After the behaviour integration, only the meaningful components of the
stress tensor and stiffness matrix are retained.

### Treatment of the internal state variables

The internal state variables are all declared as scalars.

For instance, let us assume that the `3D` behaviour declares an internal
state variable called `s`. In this case, the wrapped behaviour will then
declared `6` internal state variables associated with `s` named
respectively `sXX`, `sYY`, `sZZ`, `sXY`, `sXZ`, `sYZ`.

### Usage in `MTest`

The following code shows how to wrap a tridimensional behaviour to be
used under the axisymmetrical plane strain modelling hypothesis:

~~~~{.cxx}
// Choice of the modelling hypothesis
@ModellingHypothesis "AxisymmetricalGeneralisedPlaneStrain";
// Declaration of the behaviour
@Behaviour<
    generic,                                   // interface name
    SmallStrainTridimensionalBehaviourWrapper> // wrapper
    "src/libBehaviour.so" "ImplicitNorton";
~~~~

### Usage in the `mtest` `python` module

The following code shows how to load a tridimensional behaviour and how
to wrap this behaviour to be used in plane strain:

~~~~{.python}
import mtest
import tfel.material
h = tfel.material.ModellingHypothesis.TRIDIMENSIONAL
b1 = mtest.Behaviour('generic', 'src/libBehaviour.so',
                     'ImplicitNorton', h)
# ['ElasticStrain', 'p']
print(b1.getInternalStateVariablesNames())
b2 = mtest.Behaviour(wrapper = 'SmallStrainTridimensionalBehaviourWrapper',
                     library = 'src/libBehaviour.so',
                     function = 'ImplicitNorton',
                     hypothesis = 'PlaneStrain')
# ['ElasticStrainXX', 'ElasticStrainYY', 'ElasticStrainZZ',
#  'ElasticStrainXY', 'ElasticStrainXZ', 'ElasticStrainYZ', 'p']
print(b2.getInternalStateVariablesNames())
~~~~

## A more flexible declaration of the rotation matrix {#sec:tfel_4.1:mtest:rotation_matrix}

The `@RotationMatrix` keyword now has a `Direction` option which now let
the user specify:

- a \(2D\) vector giving the first direction of orthotropy in modelling
  hypotheses of space dimension \(2\).
- two \(3D\) vectors giving respectively the first and second directions
  of orthotropy under the tridimensional modelling hypothesis.

The given vectors are not required to be normalised. In the \(3D\) case,
the second vector is not required to be orthogonal to the first one. If
not, the second direction of orthotropy is deduced from the second
vector by removing its projection along the first one.

### Usage

~~~~{.cpp}
// using direction in 2D
@RotationMatrix<Direction> {0,1};
~~~~

~~~~{.cpp}
// using directions in 3D
@RotationMatrix<Direction> {{0,1,0},{1,0,0}};
~~~~

## Wrapper around behaviours written in the logarithmic strain framework generated with the `aster` interface {#sec:tfel_4.1:mtest:aster_logarithmic_strain_wrapper}

Behaviours written in the logarithmic strain framework and generated
with the `aster` interface were not handled by `MTest` because the
logarithmic strain framework is treated by `code_aster` and not by the
`aster` interface.

However, this is an issue for users using `MTest` for the identification
of the behaviour, as described in [Issue
#55](https://github.com/thelfer/tfel/issues/55).

`MTest` now automatically wraps the behaviour to handle those
behaviours. The wrapper handles the pre and post-processing steps around
the behaviour integration.

## Support for material properties built using the `generic` interface {#sec:tfel:4.1:mtest:generic_material_property_support}

`MTest` support material properties built using the `generic` interface:

- The `mtest::MaterialProperty` can manipulate can load such material
  properties.
- The `@MaterialProperty` keyword now has a `generic` option.

### Example of usage

~~~~{.cxx}
@MaterialProperty<generic> 'YoungModulus' 'src/libGenericInconel600.so' 'Inconel600_YoungModulus';
~~~~

## Adding `computeIntegralValue` and `computeMeanValue` post-processings for tests on pipes {#sec:tfel:4.1:ptest:integral_value}

Added two `PipeTest` functions to calculate the integral and the average
of a scalar value in the thickness of the tube for a `ptest` problem.
Each function allows to calculate the corresponding quantities in the
current or initial configurations

### Example of usage

~~~~{.cxx}
@AdditionalOutputs {'mean_value_initial_configuration':'SRR',
                    'mean_value_current_configuration':'SRR'}; // compute mean values of SRR
@AdditionalOutputs {'integral_value_initial_configuration':'SRR',
                    'integral_value_current_configuration':'SRR'}; // compute integral values of SRR
~~~~

## Support of material properties generated with the `generic` interface

The `@MaterialProperty` keyword now have a `generic` option allowing to
load material properties generated with the `generic` interface.

### Example of usage

~~~~{.python}
@MaterialProperty<generic> 'YoungModulus' 'src/libGenericInconel600.so' 'Inconel600_YoungModulus';
~~~~

## Support of point-wise models generated with the `Cast3M` interface {#sec:tfel:4.1:mtest:castem_model_interface}

Point-wise models generated with the `Cast3M` interface are now
supported (see Sections @sec:tfel:4.1:mfront:castem_model_interface and
@sec:tfel:4.1:mfront:castem_model_interface2).

### Example of usage

~~~~{.python}
@Model 'src/libM5-umat.so' 'umatm5_deziroxoxidationmodel_srma2020';
~~~~

## Support for failure criteria for pipes {#sec:tfel:4.1:ptest:failure_criterion}

Failure criteria can be added to pipe modelling using the
`@FailureCriterion` keyword. Note that no failure criterion is currently
shipped with `MTest`. The failure criteria must be provided by external
librairies.

A failure criterion is called at the end of each time step to detect
failure of the pipe.

Each failure criterion adds a column to the output file giving the
status of the criterion:

- `0` means that the criterion is not met, i.e. no failure is detected
  and the pipe is sound.
- `1` means that the criterion is met, i.e. failure is detected and the
  pipe is broken.

In case of failure, three policies can be selected using the
`@FailurePolicy` keyword:

- `ReportOnly`: failure does not affect computation. The evaluation of
  the failure criteria only affects the output file.
- `StopComputation`: failure leads to reject the current time step. If
  substepping is enabled, the time step is divided by two. With this
  policy, one can thus only approach the failure time, but never go
  beyond.
- `FreezeState` (or `FreezeStateUntilEndOfComputation`): if a failure is
  detected, the state of the structure is freezed and do not evolve. No
  equilibrium is performed, the behaviour is no more called and `PipeTest`
  will output the same results again and again until the end of
  computation. This option may be useful when optimizing material
  parameters.

### Example of usage

~~~~{.python}
@FailurePolicy 'FreezeState';
@FailureCriterion 'ElongationAtBreak' {maximum_value : 1e-4};
~~~~

**Note** This example assumes that a failure criterion named
  `ElongationAtBreak` has been loaded from an external library.

## Support for oxidation models for pipes {#sec:tfel:4.1:ptest:oxidation_model}

The `@OxidationModel` keyword introduces a model which computes an
oxidation length at either the inner boundary or the outer boundary of
the pipe. This keyword must be followed by a data map with the following
entries:

- `model`: the name of model.
- `library`: the name of the library in which the model is available.
- `boundary`: the name of boundary on which the model is defined. This
  variable must be equal to `inner_boundary` or `outer_boundary`.

An oxidation model must define an internal state variable named
`OxidationLength`. The values of the material properties and external
state variables passed to an oxidation model are computed on the
boundary on which the model is defined.

The definition of a least one oxidation model automatically defines an
evolution named `OxidationStatus` which states if an integration point
is inside an oxidation layer.

### Example of usage

~~~~{.python}
@OxidationModel{
  model : 'umatm5deziroxoxidationmodel_srma2020b',
  library : 'src/libUmatM5.so',
  boundary : 'outer_boundary'
};
~~~~

## Support for the `DTAU_DDF` tangent operator in the `GenericBehaviour` class {#sec:mtest:issue_351}

The finite strain behaviours, generated by the `generic` interface and
supported by the `GenericBehaviour` class, can now use the derivative of
the Kirchoff stress with respect to the spatial increment of the
deformation gradient as tangent operator (See Section
@sec:mfront:generic_interface:issue_351 for details).

# `mfm-test-generator` improvements

## Support for `madnex` file {#sec:tfel:4.1:mfmtg:madnex_support}

### Exporting an `mfm-test-generator` test to a `madnex` file

The `MFMTestGenerator` library exposes a data structure named
`TestDescription` which describes an `mfm-test-generator` file and two
functions called respectively `loadMFMTestGeneratorFileContent` and
`write`.

This data structure and functions are exported in `python`, as described
in Section @sec:tfel:4.1:python:mfmtg:export_mfmtg_file. Section
@sec:tfel:4.1:python:mfmtg:export_mfmtg_file:usage provides an example
of use.

#### The `TestDescription` data structure {#sec:tfel:4.1:mfmtg:TestDescription}

The `TestDescription` data structure exposes the following data members:

- `author`, which describes the author of the test.
- `date`, which describes the date at which the test has been created.
- `scheme`, which describes the type of test. Valid values are `mtest`
  and `ptest`.
- `description`, which describes a description of the test.
- `behaviour`, name of the behaviour to which the test is associated.
  This data member is **required** to export the file in the `madnex`
  file format.
- `material`, name of the material to which the test is associated. This
  data member can be empty.
- `content`, content of the `mfm-test-generator` file. This content can
  be filled from an existing `mfm-test-generator` file using the
  `loadMFMTestGeneratorFileContent` function.

#### The `loadMFMTestGeneratorFileContent` function {#sec:tfel:4.1:mfmtg:loadMFMTestGeneratorFileContent}

The `loadMFMTestGeneratorFileContent` function loads the content of an
`mfm_test_generator` file and stores it in the `content` data member of
a `TestDescription` data structure.

#### The `write` function {#sec:tfel:4.1:mfmtg:write}

The `write` function exports an `mfm_test_generator` test, described by
a `TestDescription` data structure, to a file.

The file format is deduced from the extension of the file.

Currently, only extensions associated with the [`madnex` file
format](https://github.com/thelfer/madnex) are supported if `TFEL` is
compiled with support of this file format. Those extensions are: `mdnx`,
`madnex` (deprecated) or `edf` (experimental data file, deprecated).
Note that the behaviour member of the metadata must be specified for
export in the `madnex` file format.

#### Best practices

We highly recommend to use the following substitution variables when
defining the test:

- `@interface@`, which is meant to be replaced by the interface to be
  used. This is very handy if the test can be run for different
  interfaces
- `@library@`, which is meant to be replaced by the path to the shared
  library containing the tested behaviour.
- `@behaviour@`, which contains the name of the function implementing
  the behaviour for the considered interface.

# Improvements to the `python` bindings 

## Improvements to the `TFEL/System` `python` module

### Retrieving the unit system of an entry point {#sec:tfel_4.1:python:system:elm:unit_system}

The `ExternalLibraryManager` class now exposes the `getUnitSystem` to
retrieve the unit system associated with an entry point (material
property, behaviour, model).

### Retrieving information about behaviours' post-processings generated by the `generic` interface

The `generic` interface generates a function dedicated for each
initialize function and each modelling hypothesis supported (see Section
@sec:tfel_4.1:mfront:initialize_functions for details about initialize
functions).

The `ExternalLibraryManager` class now exposes the following methods:

- `getGenericBehaviourInitializeFunctionFunctions`, which returns the
  initialize functions associated with a behaviour.
- `getGenericBehaviourInitializeFunctionFunctionOutputs`, which returns the
  outputs of a initialize function function.
- `getGenericBehaviourInitializeFunctionFunctionOutputs`, which returns the
  types of the outputs of a initialize function function.

### Retrieving information about behaviours' post-processings generated by the `generic` interface

The `generic` interface generates a function dedicated for each
post-processing and each modelling hypothesis supported (see Section
@sec:tfel_4.1:mfront:postprocessings for details about
post-processings).

The `ExternalLibraryManager` class now exposes the following methods:

- `getGenericBehaviourPostProcessingFunctions`, which returns the
  post-processings associated with a behaviour.
- `getGenericBehaviourPostProcessingFunctionOutputs`, which returns the
  outputs of a post-processing function.
- `getGenericBehaviourPostProcessingFunctionOutputs`, which returns the
  types of the outputs of a post-processing function.

## Improvements to the `mtest` `python` module
The following `PipeTest` methods are now available:

- `computeMeanValue`, which computes the mean value of a scalar variable.
- `computeIntegralValue`, which computes the integral value of a scalar variable.

### Example of usage

~~~~{.python}
import std
from mtest import PipeTest, StudyCurrentState, SolverWorkSpace, \
     PipeTestConfiguration as ptc

t = PipeTest()
# geometry and meshing
t.setInnerRadius(4.2e-3)
t.setOuterRadius(4.7e-3)
t.setNumberOfElements(10)
t.setElementType('Linear')

# modelling hypothesis
t.setAxialLoading('None')

t.setTimes([0,1])
t.setInnerPressureEvolution(1.5e6)
t.setOuterPressureEvolution({0:1.5e6,1:10e6})

t.setBehaviour('LogarithmicStrain1D','castem','../behaviours/castem/libMFrontCastemBehaviours.so','umatelasticity')
t.setMaterialProperty('YoungModulus',150e9)
t.setMaterialProperty('PoissonRatio',0.3)
t.setExternalStateVariable('Temperature',{0:293.15,1:693.15})

t.setOutputFileName("pipe.res")

s  = StudyCurrentState()
wk = SolverWorkSpace()

t.completeInitialisation()
t.initializeCurrentState(s)
t.initializeWorkSpace(wk)

mean_ic     = t.computeMeanValue(s,'SRR',ptc.INTIAL_CONFIGURATION)
mean_cc     = t.computeMeanValue(s,'SRR',ptc.CURRENT_CONFIGURATION)
integral_ic = t.computeIntegralValue(s,'SRR',ptc.INTIAL_CONFIGURATION)
integral_cc = t.computeIntegralValue(s,'SRR',ptc.CURRENT_CONFIGURATION)
~~~~

## Support of named arguments in the constructor of the `Behaviour` class {#sec:tfel_4.1:pymtest:behaviour_constructor}

Named arguments are now supported in the `Behaviour` constructor. The
following arguments can be specified:

- `interface`: name of the interface to be used (optional).
- `library`: name of the shared library to be loaded.
- `function`: name of the function implementing the behaviour.
- `hypothesis`: hypothesis to be used.
- `wrapper`: name of the behaviour wrapper to be used (optional).

### Example of usage

The following piece of code loads a behaviour implemented in the
`src/libBehaviour.so` shared library by the `ImplicitNorton` function
for the `PlaneStrain` modelling hypothesis:

~~~~{.python}
b = mtest.Behaviour(library = 'src/libBehaviour.so',
                    function = 'ImplicitNorton',
                    hypothesis = 'PlaneStrain')
~~~~

## Exporting an `MTest`test to a `madnex` file {#sec:tfel:4.1:python:mtest:export_mtest_file}

The `TestDescription` data structure (see Section
@sec:tfel:4.1:mtest:TestDescription) and the `loadMTestFileContent` and
`write` functions (see Sections @sec:tfel:4.1:mtest:loadMTestFileContent
and @sec:tfel:4.1:mtest:write) are exposed in the `mtest` `python`
module.

### Example of usage {#sec:tfel:4.1:python:mtest:export_mtest_file:usage}

~~~~{.python}
import mtest

d = mtest.TestDescription()
d.author = 'John Doe'
d.date = '01/03/2022'
d.name = 'UniaxialTensileTest'
d.scheme = 'mtest'
d.behaviour = 'Plasticity'
mtest.loadMTestFileContent(d, 'Plasticity.mtest')

mtest.write(d,'Plasticity.mdnx')
~~~~

## The `mtest` python module

### Support for material properties generated with the `generic` interface

Material properties generated with the `generic` interface are supported
by the `mtest.MaterialProperty` class.

#### Example of usage

~~~~{.python}
import mtest
mp = mtest.MaterialProperty('src/libGenericInconel600.so', 'Inconel600_YoungModulus')
~~~~

### Definition of oxidation models in the `PipeTest` class {#sec:tfel:4.1:python:mtest:oxidation_model}

The `addOxidationModel` allows to define a new oxidation model. It
accepts three named arguments:

- `library`: the name of the library in which the model is available.
- `model`: the name of model.
- `boundary`: the name of boundary on which the model is defined. This
  variable must be equal to `inner_boundary` or `outer_boundary`.

#### Example of usage

~~~~{.python}
p.addOxidationModel(library = 'src/libUmatM5.so',
                    model = 'umatm5deziroxoxidationmodel_srma2020b',
                    boundary = 'outer_boundary')
~~~~

## The `mfm_test_generator` python module

### Exporting an `mfm-test-generator` test to a `madnex` file {#sec:tfel:4.1:python:mfmtg:export_mfmtg_file}

The `TestDescription` data structure (see Section
@sec:tfel:4.1:mfmtg:TestDescription) and the
`loadMFMTestGeneratorFileContent` and `write` functions (see Sections
@sec:tfel:4.1:mfmtg:loadMFMTestGeneratorFileContent and
@sec:tfel:4.1:mfmtg:write) are exposed in the `mfm_test_generator`
`python` module.

#### Example of usage {#sec:tfel:4.1:python:mfmtg:export_mfmtg_file:usage}

~~~~{.python}
import mfm_test_generator

d = mfm_test_generator.TestDescription()
d.author = 'John Doe'
d.date = '01/03/2022'
d.name = 'UniaxialTensileTest'
d.behaviour = 'Plasticity'
mfm_test_generator.loadMFMTestGeneratorFileContent(d, 'Plasticity.mfmtg')

mfm_test_generator.write(d,'Plasticity.mdnx')
~~~~

## Support for failure criteria {#sec:tfel:4.1:python:ptest:failure_criterion}

The `PipeTest` class have two methods related to failure criteria:

- `addFailureCriterion`, which corresponds to the `@FailureCriterion`
  keyword (see Section @sec:tfel:4.1:ptest:failure_criterion).
- `setFailurePolicy`, which corresponds to the `@FailurePolicy` keyword
  (see Section @sec:tfel:4.1:ptest:failure_criterion).

# `mfront-query` improvements

## General queries

### List of dependencies of an `MFront` file {#sec:tfel_4.1:mfront_query:list_dependencies}

The `--list-dependencies` query lists all the dependencies of an
`MFront` file.

If a dependency is encoded in a `madnex` file, an internal representation of
the path to this dependency is returned (see example below).

#### Example of usage {.unnumbered}

~~~~{.bash}
$ mfront-query --list-dependencies --search-path=generate   \
               --madnex-search-path=MaterialProperties.mdnx \
               Test.mfront 
madnex:generate/MaterialProperties.mdnx:MaterialProperty::YoungModulusTest
~~~~

## New queries on material properties

### Output of a material property{#sec:tfel_4.1:mfront_query:mp:output}

The `--output` query displays information about the output of a material
property.

#### Example of usage {.unnumbered}

~~~~{.cxx}
$ mfront-query --output Inconel600_YoungModulus.mfront 
- YoungModulus (E): The Young modulus of an isotropic material
~~~~

### Inputs of a material property{#sec:tfel_4.1:mfront_query:mp:inputs}

The `--inputs` query displays information about the inputs of a material
property. The `--state-variables` query is equivalent to `--inputs`.

#### Example of usage {.unnumbered}

~~~~{.cxx}
$ mfront-query --inputs Inconel600_YoungModulus.mfront 
- Temperature (TK): The temperature
~~~~

### Bounds and physical bounds {#sec:mfront_query:issue_254}

The following queries have been implemented:

- `--has-bounds`, which returns `true` if a variable has bounds, `false`
  otherwise.
- `--bounds-type`, which returns the bounds type associated to a
  variable.
- `--bounds-value`, which shows the bounds value associated as a range.
- `--has-physical-bounds`, which returns `true` if a variable has
  physical bounds, `false` otherwise.
- `--physical-bounds-type`, which returns the physical bounds type
  associated to a variable.
- `--physical-bounds-value`, which shows the physical bounds value
  associated as a range.

Those queries expects the external name of a variable as argument. The
variable may be the output of the material property, any of the input or
a parameter.

### Example of usage

~~~~{.cxx}
$ mfront-query --has-physical-bounds=Temperature Inconel600_YoungModulus.mfront 
- true
$ $ mfront-query --physical-bounds-type=Temperature Inconel600_YoungModulus.mfront
Lower
$ mfront-query --physical-bounds-value=Temperature Inconel600_YoungModulus.mfront
[0:*[
~~~~

### Unit system

The `--unit-system` query allows to retrieve the unit system. If no unit
system is declared, an empty string is returned.

## New queries on behaviours

### List of initialize functions

The list of initialize functions defined by a behaviour can be retrieved
using the `--initialize functions` query, as follows:

~~~~{.bash}
 mfront-query --initialize-functions Plasticity.mfront 
- ElasticStrainFromInitialStress: no description available.
~~~~

### List of initialize function variables

The list of initialize function' variables defined by a behaviour can be
retrieved using the `--initialize function-variables` query.

### List of post-processings

The list of post-processings defined by a behaviour can be retrieved
using the `--post-processings` query, as follows:

~~~~{.bash}
$ mfront-query --post-processings Elasticity.mfront 
- PrincipalStrain: compute the principal strain. Modified post-processing variables are:
  - PrincipalStrain (εᵖ)
~~~~

### List of post-processing variables

The list of post-processing' variables defined by a behaviour can be
retrieved using the `--post-processing-variables` query, as follows:

~~~~{.bash}
$ mfront-query --post-processing-variables Elasticity.mfront 
- PrincipalStrain (εᵖ)
~~~~

### List of `MTest` tests associated with a behaviour in a `madnex` file

The `--list-behaviour-mtest-tests` command line argument can be used to
display the list of tests associated with a behaviour in a `madnex`
file.

Optionnally, this command line argument accept the options
`sorted-by-behaviours` or `unsorted` (see the examples below).

#### Examples of usage

~~~~{.cxx}
$ mfront-query --list-behaviour-mtest-tests --test=".+Tensile.+" Plasticity.mdnx
- tests associated with behaviour Plasticity
    - UniaxialTensileTest
~~~~

~~~~{.cxx}
$ mfront-query --list-behaviour-mtest-tests=unsorted --test=".+Tensile.+" Plasticity.mdnx
UniaxialTensileTest
~~~~

### List of `mfm-test-generator` tests associated with a behaviour in a `madnex` file

The `--list-behaviour-mfm-test-generator-tests` command line argument
can be used to display the list of tests associated with a behaviour in
a `madnex` file.

Optionnally, this command line argument accept the options
`sorted-by-behaviours` or `unsorted` (see the examples below).

#### Examples of usage

~~~~{.cxx}
$ mfront-query --list-behaviour-mfm-test-generator-tests --test=".+Tensile.+" Plasticity.mdnx
~~~~

~~~~{.cxx}
$ mfront-query --list-behaviour-mfm-test-generator-tests=unsorted --test=".+Tensile.+" Plasticity.mdnx
~~~~

### Getting information about the climb tensors {#sec:mfront_query:model:issue_317}

The following queries are available to retrieve information about climb tensors:

- `--climb-tensors`: list all the climb tensors, sorted by
  family".
- `--climb-tensors-by-index`: list all the climb tensors.
- `--climb-tensors-by-slip-system`: list all the climb
  tensors sorted by slip systems.

### Display information about a code block{#sec:mfront_query:issue_323}

The query `--code-block` returns the information about code block.

#### Example of usage

~~~~{.bash}
$ mfront-query --code-block=Integrator UserDefinedViscoplasticityTest.mfront
- Integrator: 
  - code:
  feel -= this->deto;
  
  if(!perturbatedSystemEvaluation){
  }
  const auto& s = this->sig;
  const auto seq = sigmaeq(s);
  const auto iseq = 1/max(seq,(this->relative_value_for_the_equivalent_stress_lower_bound) * this->young);
  const auto dseq_ds = 3*deviator(s)*(iseq/2);
  const auto d2seq_dsds = (Stensor4::M()-(dseq_ds^dseq_ds))*iseq;
  const auto& n = dseq_ds;
  const auto& dn_ds = d2seq_dsds;
  feel += this->dp* n;
  dfeel_ddp = n;
  dfeel_ddeel += (2 * this->mu)*(this->theta) * ((this->dp) * dn_ds);
  const auto mfront_udvf_f = seq;
  const auto vp = [this, mfront_udvf_f] {
    if(mfront_udvf_f >= stress{0}){
     return strainrate{(this->A)*(std::pow(mfront_udvf_f,this->E))};
    }
    return strainrate{0};
  }();
  const auto dvp_dseqe = [this, mfront_udvf_f] {
    if(mfront_udvf_f >= stress{0}){
  return derivative_type<strainrate, stress>{(this->A)*((this->E)*(std::pow(mfront_udvf_f,(this->E)-(1))))};
    }
    return derivative_type<strainrate, stress>{0};
  }();
  fp -= (this->dt) * vp;
  dfp_ddeel += (2 * this->mu)*(this->theta) * (-(this->dt) * dvp_dseqe * dseq_ds);
~~~~

# `tfel-check` improvements

## Automatic declaration of substitutions for `TFEL` executables and `python` interpreter {#sec:tfel_4.1:tfel_check:default_substitutions}

The following substitutions are automatically declared: `@mfront@`,
`@mfront-query@`, `@mtest@`, `@mfront-doc@`, `@mfm-test-generator@`.

In `python` bindings are enabled, the `@python` substitution is also
automatically declared.

Those substitutions are declared after reading the configuration files
and after parsing the command line arguments, so those default
substitutions can be overriden by the user.

## Test for failure {#sec:tfel_4.1:tfel_check:test_failure}

The `shall_fail` option allows to specify if a given command is expected
to fail (or succeed).

## Testing the output of commands {#sec:tfel_4.1:tfel_check:command_outputs}

### The `expected_output` option

The `expected_output` option to `@Command` allows to specifiy the output
of a command. This option may be a string (single line output) or an
array of strings (multiple lines output).

Note that the output lines of commands are always trimmed on the end.

### The `expected_numerical_output` option

The `expected_numerical_output` option to `@Command` allows to test the
numerical output of a command. This option must be defined as a map with
two entries:

- `value`: the expected value
- `criterion_value`: the criterion value used to test the output

### The `output_validation_regex` option

The `output_validation_regex` allows to specifiy a regular expression
which shall validate the output of the command. The output of the
command is concatenated in a single string for the test.

# Issues fixed

## Issue 351: Add support of `DTAU_DDF` in `generic` interface

This feature is described in Sections
@sec:mfront:generic_interface:issue_351 and @sec:mtest:issue_351.

For more details, see <https://github.com/thelfer/tfel/issues/351>

## Issue 323: [mfront-query] display the code in a code block 

This feature is described in Section @sec:mfront_query:issue_323.

For more details, see <https://github.com/thelfer/tfel/issues/323>

## Issue 317: [mfront] Automatic declaration of  climb tensors

This feature is described in Sections @sec:mfront:model:issue_317 and
@sec:mfront_query:model:issue_317.

For more details, see <https://github.com/thelfer/tfel/issues/317>

## Issue 314: [tfel-material] Implement the inverse Langevin function

This feature is described in Section
sec:tfel_4.1:tfel_material:inverse_langevin_function.

For more details, see <https://github.com/thelfer/tfel/issues/314>

## Issue 302: [mfront] `mfront` interface for material properties shall generate overloaded functions for every numerical type supported

For more details, see <https://github.com/thelfer/tfel/issues/302>

￼
￼
## Issue 301: [mfront] `mfront` interface for material properties shall generate a version of supporting quantites

For more details, see <https://github.com/thelfer/tfel/issues/301>

## Issue 300: [mfront] Add a new `castem` interface for point-wise models handled by the `Model` DSL 

This feature is described in Section
@sec:tfel:4.1:mfront:castem_model_interface. See also Section
@sec:tfel:4.1:mtest:castem_model_interface.

For more details, see <https://github.com/thelfer/tfel/issues/300>

## Issue 299: [mfront] Add support for models in the `castem` interface for behaviours

This feature is described in Section
@sec:tfel:4.1:mfront:castem_model_interface2. See also Section
@sec:tfel:4.1:mtest:castem_model_interface.

For more details, see <https://github.com/thelfer/tfel/issues/299>

## Issue 298: Add the `@MaterialProperty` keyword for models described by the `Model` DSL

The `@MaterialProperty` and `@Coef` keywords have been added for
consistency with DSLs related to behaviours and in particular the
`@DefaultModel`, `@RungeKuttaModel` and `@ImplicitModel` DSLs. They are
equivalent to the `@ConstantMaterialProperty` keyword.

For more details, see <https://github.com/thelfer/tfel/issues/298>

## Issue 297: [mfront] Automatically include the `<iostream>` header in models

For more details, see <https://github.com/thelfer/tfel/issues/297>

## Issue 296: [mfront] Allow bidirectional convertion between the IntegrationResult enumeration and boolean values

Some code blocks return boolean values and others return instances of
the IntegrationResult enumeration. This is inconsistent from the user
point of view. Allowing bidirectional convertion between the
IntegrationResult enumeration and boolean values mitigates this issue.

For more details, see <https://github.com/thelfer/tfel/issues/296>

## Issue 295: [mfront] Allow failure of the behaviour initialisation

This feature is described in Section @sec:mfront:issue_295.

For more details, see <https://github.com/thelfer/tfel/issues/295>

## Issue 282: [mfront] Allow models based generic behaviours to define tangent operator blocks associated with the derivatives of state variables with respect to external state variables 

For more details, see <https://github.com/thelfer/tfel/issues/282>

## Issue 276: [mfront] Add unicode support the `Model` DSL

For more details, see <https://github.com/thelfer/tfel/issues/276>

## Issue 266: [tfel-math] Add constexpr evaluation of spline and derivative 

This feature is described in Section @sec:tfel_4.1:tfel_math:issue_266.

For more details, see <https://github.com/thelfer/tfel/issues/266>

## Issue 265: [mfront] Add MTest file generation capability for `generic` behaviours

For more details, see <https://github.com/thelfer/tfel/issues/265>

## Issue 264: Add support to compute linear intepolation

This feature is described in Section @sec:tfel_4.1:tfel_math:issue_264.

For more details, see <https://github.com/thelfer/tfel/issues/264>

## Issue 263: [tfel-math] Add an empty method to `TFEL`' array

For more details, see <https://github.com/thelfer/tfel/issues/263>

## Issue 262: Installation issue of the `master` branch (future version 4.1) with `gcc-8.1.0`

The issue is related to `MFront` tests using quantities such tests fails
to build with old compilers, i.e. `gcc` `8.1.0`.

The `enable-mfront-quantity-tests` option can be optionnaly specified to
enable or disable tests of behaviours using quantities. This option is
ON by default.

For more details, see <https://github.com/thelfer/tfel/issues/262>

## Issue 256: [tfel-math] Add support to clamp `TFEL/Math` arrays

This feature is described in Section @sec:tfel_4.1:tfel_math:issue_256.

For more details, see <https://github.com/thelfer/tfel/issues/256>

## Issue 255: [tfel-gossary] add physical bounds to entries of the glossary when appropriate

For more details, see <https://github.com/thelfer/tfel/issues/255>

## Issue 254: [mfront-query] add queries related to bounds for material properties

This feature is described in Section @sec:mfront_query:issue_254.

For more details, see <https://github.com/thelfer/tfel/issues/254>

## Issue 249: [mfront] The C++ interface for material properties is broken when declaring parameters with complex types

For more details, see <https://github.com/thelfer/tfel/issues/249>

## Issue 248: [mtest] Document the `@AdditionalOutputs` keyword

For more details, see <https://github.com/thelfer/tfel/issues/248>

## Issue 245: [mtest] add a `getOutputName` for `MaterialProperty` class

For more details, see <https://github.com/thelfer/tfel/issues/245>
￼

## Issue 244: [mfront] Support for `PhysicalConstants` in the `Model` DSL

This feature is described in Section @sec:mfront:model:issue_244.

For more details, see <https://github.com/thelfer/tfel/issues/244>

## Issue 243: [mfront] Support for `PhysicalConstants` in the `MaterialProperty` DSL

This feature is described in Section
@sec:mfront:material_property:issue_243

For more details, see <https://github.com/thelfer/tfel/issues/243>

## Issue 242: [mfront] Quantity support in the `Model` DSL

This feature is described in Section @sec:mfront:model:issue_242.

For more details, see <https://github.com/thelfer/tfel/issues/242>

## Issue 241: [mfront] Allow to specify types in the `Model` DSL

This feature is described in Section @sec:mfront:model:issue_241.

For more details, see <https://github.com/thelfer/tfel/issues/241>

## Issue 239: [mfront] Add support for defining an unit system enhancement mfront

This feature is described in Section @sec:mfront:issue_239.

For more details, see <https://github.com/thelfer/tfel/issues/239>

## Issue 238: [tfel-glossary] Translate all the description in english

For more details, see <https://github.com/thelfer/tfel/issues/238>

## Issue 237: [mfront] Add physical bounds declaration in the glossary. Automatically use those bounds in `MFront`

This feature is described in Section @sec:mfront:issue_239.

For more details, see <https://github.com/thelfer/tfel/issues/237>

## Issue 236: [mtest] Allow to use oxidation models in `ptest`

This feature is described in Sections @sec:tfel:4.1:ptest:oxidation_model
and @sec:tfel:4.1:python:mtest:oxidation_model.

For more details, see <https://github.com/thelfer/tfel/issues/236>.

## Issue 235: [mfront] Document the variable affecting the compilation of shared libraries

The compilation process used by `MFront` is now documented on this page:
<https://thelfer.github.io/tfel/web/compiling-mfront-shared-libraries.html>

For more details, see <https://github.com/thelfer/tfel/issues/235>.

## Issue 233: Add the ability to define failure criteria in `ptest`

This feature is described in Sections
@sec:tfel:4.1:ptest:failure_criterion and
@sec:tfel:4.1:python:ptest:failure_criterion.

For more details, see <https://github.com/thelfer/tfel/issues/233>.

## Issue 231:[tfel-check] Allow to compare the result of a command to a regular expression 

This feature is described in Section
@sec:tfel_4.1:tfel_check:command_outputs.

￼
For more details, see <https://github.com/thelfer/tfel/issues/231>.

## Issue 230: [tfel-check] All to compare commands output to numerical values

This feature is described in Section
@sec:tfel_4.1:tfel_check:command_outputs.

￼
For more details, see <https://github.com/thelfer/tfel/issues/230>.

￼
## Issue 229:[tfel-check] Allow to specify that a command shall fail 

This feature is described in Section
@sec:tfel_4.1:tfel_check:test_failure.

For more details, see <https://github.com/thelfer/tfel/issues/229>.

## Issue 225: [mfront] Separate file generation from DSLs

For more details, see <https://github.com/thelfer/tfel/issues/225>.

## Issue 224: [mfront] add DSL options to override parameters in material properties and point-wise models

For more details, see <https://github.com/thelfer/tfel/issues/224>.

## Issue 223: [mfront] Allow to define dsl options from a file

This feature is described in depth in Section
@sec:tfel_4.1:mfront:global_dsl_options:external_file.

For more details, see <https://github.com/thelfer/tfel/issues/223>.

## Issue 222: [mfront] Add a behaviour DSL option to define the modelling hypotheses to be treated

This feature is described in depth in Section
@sec:tfel:4.1:mfront:global_options:modelling_hypothesis.

For more details, see <https://github.com/thelfer/tfel/issues/222>.

## Issue 219: [mfront] Allow to override "overriden" parameters 

For more details, see <https://github.com/thelfer/tfel/issues/219>.

## Issue 216: [mfront] Add behaviour `DSL` options to override material properties, external state variables and parameters

This feature is described in Section
@sec:tfel:4.1:mfront:global_options:overriding_parameters.

For more details, see <https://github.com/thelfer/tfel/issues/216>.

## Issue 215: [mfront] Allow to override external state variables as parameters

Scalar external state variables can be overriden by parameters. Note
that if an external state variable is overriden, the increment of this
variable must also be overriden. If no overriding parameter is specified
for this increment, `MFront` will automatically define such a parameter
with a null default value.

For more details, see <https://github.com/thelfer/tfel/issues/215>.

## Issue 210: [tfel-check] Allow to split command in multiple strings

For more details, see <https://github.com/thelfer/tfel/issues/210>.

## Issue 209: [tfel-utilities] Convertion from `Data` to `std::vector` and `std::map` shall work if the object is empty

￼
For more details, see <https://github.com/thelfer/tfel/issues/209>.

￼
## Issue 208: [tfel-check] Automatic declaration of substitutions for `TFEL` executables and `python` interpreter 

This feature is described in Section
@sec:tfel_4.1:tfel_check:default_substitutions.

For more details, see <https://github.com/thelfer/tfel/issues/208>.

## Issue 207: [tfel-check] allow comments in config files

For more details, see <https://github.com/thelfer/tfel/issues/207>.

## Issue 200: [mfront] Generate a `setParameter` function for material properties generated with the `python` interface
￼

For more details, see <https://github.com/thelfer/tfel/issues/200>.

## Issue 199: [mfront-query] Add the `--parameters-file` query for material properties

For more details, see <https://github.com/thelfer/tfel/issues/199>.

## Issue 197: [mfront] Extension of the `derivative_type` metafunction to higher order derivatives

For more details, see <https://github.com/thelfer/tfel/issues/197>.

## Issue 192: [TFEL/System] Access to default values of parameters of a material property through the `ExternalMaterialPropertyDescription` class

This feature is described in Section
@sec:tfel_4.1:system:empd:getParameterDefaultValue.

For more details, see <https://github.com/thelfer/tfel/issues/192>.

## Issue 191: [TFEL/System] Ability to request bounds of variables (inputs, outputs) of a material property in the `ExternalMaterialPropertyDescription` class

Those features are described in Section @sec:tfel_4.1:system:empd:bounds.

For more details, see <https://github.com/thelfer/tfel/issues/191>.

## Issue 187: [mfront] Get `@Output` from compiled file through `python` bindings

This feature is described in Section
@sec:tfel_4.1:system:elm:getMaterialPropertyOutput.

For more details, see <https://github.com/thelfer/tfel/issues/187>.

## Issue 186: [mfront] Get `@Material`, `@Law`, `@Author`, `@Date`, `@Description` from compiled file through Python binding

Those features are described in Sections
@sec:tfel_4.1:system:elm:getLaw, @sec:tfel_4.1:system:elm:getAuthor,
@sec:tfel_4.1:system:elm:getDate and
@sec:tfel_4.1:system:elm:getDescription.

For more details, see <https://github.com/thelfer/tfel/issues/186>.

## Issue 183: [mfront] The `rkCastem` algorithm assumes that a variable `young` is defined

This feature is described in Section
@sec:tfel_4.1:mfront:rkCastem.

For more details, see <https://github.com/thelfer/tfel/issues/183>.

## Issue 181: [mfront-query] add query about the output of a material property

This feature is described in Section
@sec:tfel_4.1:mfront_query:mp:output.

For more details, see <https://github.com/thelfer/tfel/issues/181>

## Issue 180: [mfront-query] add query about the inputs of a material property

This feature is described in Section
@sec:tfel_4.1:mfront_query:mp:inputs.

For more details, see <https://github.com/thelfer/tfel/issues/180>

## Issue 177: [mtest] Support of material properties generated with the `generic` interface

The feature is described in Section
@sec:tfel:4.1:mtest:generic_material_property_support.

For more details, see : <https://github.com/thelfer/tfel/issues/177>.

## Issue 175: Add support for post-processing to mean-value and the integral value of a state variable

The feature is described in Section sec:tfel:4.1:ptest:integral_value.

For more details, see : <https://github.com/thelfer/tfel/issues/175>.

## Issue 172: [tfel-math-parser] Add the ability to derive the power function enhancement {#sec:tfel:4.1:issue:172}

Previous versions of `TFEL/Math` did not implement the evaluation the
derivative of the formula `power<N>(x)`. This is now implemented
properly.

For more details, see : <https://github.com/thelfer/tfel/issues/172>.

## Issue 171: ￼[tfel-math-parser] Correct derivative of exponents bug

The computation of the derivative of power functions lead to the
following issues in previous versions of `TFEL`. The derivative of
`a(x)**b(x)` with respect to `x` was computed as
`(b(x)*a(x)**b(x)/a(x))*da(x)+log(a(x))*a(x)**b(x)*db(x)` where `da(x)`
and `db(x)` denote the deriative of `a` and with respect to `x`.

1. The first term `b(x)*a(x)**b(x)/a(x)*da(x)` is singular if `a(x)` is
  null. This term is now computed as `b(x)*a(x)**(b(x)-1)*da(x)`.
2. The second term `log(a(x))*a(x)**b(x)*db(x)` is singular if `a(x)` is
  null although `log(a(x))*a(x)**b(x)` has a well defined limit as a
  tends to zero. The new implementation takes this case into account.

For more details, see : <https://github.com/thelfer/tfel/issues/171>.

## Issue 170: [tfel-math-parser] Add support for integer exponent

In previous versions, the formula `x**N` was evaluated using
`std::pow(x,N)` even if `N` was an integer.

This formula is now evaluated as:

- `tfel::math::power<N>(x)` if `N` is in the ranges `[-16:-1]` or
  `[2:16]`.
- `x**0` is evaluated as `1`.
- `x**1` is evaluated as `x`.
- `std::pow(x,N)` otherwise.

See also Issue 172.

For more details, see : <https://github.com/thelfer/tfel/issues/170>.

## Issue 169: [tfel-math-parser] Add feature to simplify formula

### More efficient evaluation of derivatives

In previous versions, the evaluation of derivatives of a formula could
lead to inefficient evaluation tree. For example, the derivative of the
formula `x+exp(y)` with respect to `x` was computed as `1+exp(y)*0`,
which required a useless evaluation of `exp(y)`. Version 4.1 evaluates
this derivative as `1`.

For more details, see : <https://github.com/thelfer/tfel/issues/169>.

## Issue 157: Disable `reportContractViolation` on GPUs

See Section @sec:tfel_4.1:tfel_exception:reportContractViolation_gpu.

For more details, see : <https://github.com/thelfer/tfel/issues/157>.

## Issue 150: [TFEL/Material] Add device specification

See Sections @sec:tfel_4.1:tfel_config:device_specification and @sec:tfel_4.1:tfel_material:gpu.

For more details, see : <https://github.com/thelfer/tfel/issues/150>.

## Issue 148: [TFEL/Material] `constexpr` all the things

See Section @sec:tfel_4.1:tfel_material:constexpr.

For more details, see : <https://github.com/thelfer/tfel/issues/148>.

## Issue 146: [mfront] Export dependencies of MFront files in madnex files

For more details, see : <https://github.com/thelfer/tfel/issues/146>.

## Issue 145: [mfront] Automatically add a madnex input file to the search paths 

This feature is described in Section
@sec:tfel_4.1:mfront:automatic_madnex_file_search_path.

For more details, see : <https://github.com/thelfer/tfel/issues/145>.

## Issue 144: [mfront-query] List dependencies of a `mfront` file

This feature is described in Section
@sec:tfel_4.1:mfront_query:list_dependencies.

For more details, see : <https://github.com/thelfer/tfel/issues/144>.

## Issue 143: [mfront] Add the ability to append a madnex paths to the search paths 

This feature is described in Section @sec:tfel_4.1:mfront:madnex_search_paths.

For more details, see : <https://github.com/thelfer/tfel/issues/143>.

## Issue 125: [mfront] Remove behaviour sources if not needed

For more details, see : <https://github.com/thelfer/tfel/issues/125>.

## Issue 124: [mfront] Add DSL option to disable runtime change of the out of bound policy

This feature is described in Section
@sec:tfel:4.1:mfront:global_options:out_of_bounds_policy_runtime_modification.

For more details, see : <https://github.com/thelfer/tfel/issues/124>.

## Issue 123: [mfront] Add DSL option to change the default out of bounds policy

This feature is described in Section
@sec:tfel:4.1:mfront:global_options:default_out_of_bounds_policy.

For more details, see : <https://github.com/thelfer/tfel/issues/123>.

## Issue 122: [mfront] remove usage of exceptions in the computeTangentOperator method generated by the `Implicit` DSL

For more details, see : <https://github.com/thelfer/tfel/issues/122>.

## Issue 111: [mtest] Support for `madnex` file

This feature is described in depth in Section @sec:tfel:4.1:mtest:madnex_support.

For more details, see : <https://github.com/thelfer/tfel/issues/111>.

## Issue 108: [mtest] Support for extended types in MTest

For more details, see : <https://github.com/thelfer/tfel/issues/108>.

## Issue 94: [mfront] Add option to disable initialization of parameters from text file

This feature is described in Section
@sec:tfel:4.1:mfront:global_options:parameters_initialization_from_file.

For more details, see : <https://github.com/thelfer/tfel/issues/94>.

## Issue 92: [mfront] create a generic interface for point wise models

This feature is described in Section
@sec:tfel:4.1:mfront:generic_model_interface.

For more details, see : <https://github.com/thelfer/tfel/issues/92>.

## Issue 91: [mfront] define build identifier using options

This feature is described in Section
@sec:tfel:4.1:mfront:global_options:build_identifier.

For more details, see : <https://github.com/thelfer/tfel/issues/90>.

## Issue 90: [mfront] create a generic interface for material properties

This feature is described in Section
@sec:tfel:4.1:mfront:generic_material_property_interface.

For more details, see : <https://github.com/thelfer/tfel/issues/90>.

## Issue 83: [mfront] Add a command line argument to retrieve the list of options associated with a domain specific language

This feature is described in Section
@sec:tfel_4.1:mfront:list_dsl_options.

For more details, see : <https://github.com/thelfer/tfel/issues/83>.

## Issue 82: [mfront] Ability to define DSL options on the command line

This feature is described in Section
@sec:tfel_4.1:mfront:global_dsl_options.

For more details, see : <https://github.com/thelfer/tfel/issues/82>.

## Issue 55: [mtest] Creation of a wrapper around behaviours written in the logarithmic strain framework generated with the `aster` interface

The wrapper is described in Section
@sec:tfel_4.1:mtest:aster_logarithmic_strain_wrapper.

For more details, see : <https://github.com/thelfer/tfel/issues/55>.

## Issue 57: [gallery] Mistakes in the Drucker Prager Cap model of the gallery

For more details, see : <https://github.com/thelfer/tfel/issues/57>.

## Issue 50: [mfront] Options to avoid the automatic declaration of the temperature as an external state variable {#sec:tfel:4.1:issue:50}

This option is described in depth in Section
@sec:tfel:4.1:mfront:global_options:temperature_as_first_external_state_variable.

For more details, see : <https://github.com/thelfer/tfel/issues/50>.

## Issue 44: [mtest] Add support for a boundary condition modelling the effet of a mandrel in pipe modelling

The feature is described in Section @sec:tfel_4.1:mtest:mandrel.

For more details, see : <https://github.com/thelfer/tfel/issues/38>.

## Issue 39: [tfel-config] Add query associated with python bindings support

`tfel-config` now supports an command line option named
`--python-bindings-support` which displays true if `python` bindings are
available, `false` otherwise.

For more details, see : <https://github.com/thelfer/tfel/issues/39>.

## Issue 38: [tfel-config] Add query associated with `madnex` support

`tfel-config` now supports an command line option named
`--madnex-support` which displays true if `madnex` files are supported,
`false` otherwise.

For more details, see : <https://github.com/thelfer/tfel/issues/38>.

## Issue 36: [mfront] Improved support for `MTest` file generation (all kind of state variables)

In previous versions, only scalar and symmetric tensors state variables
were supported when generation `MTest` file on integration failure.

All kind of state variables are now supported.

For more details, see : <https://github.com/thelfer/tfel/issues/36>.

## Issue 28: [mtest] Support of tensorial external state variables

The `Behaviour` class has new methods:

- `expandExternalStateVariablesNames` (wrapped in python)
- `getExternalStateVariableType`  (wrapped in python)
- `getExternalStateVariablePosition` (wrapped in python)

The `@ExternalStateVariable` now allows to define tensorial external
state variables components by components or an array as follows:

~~~~{.cpp}
// Definition of an symmetric tensor external state variable
@ExternalStateVariable<function> "g" {"0", "1", "2", "3 * t", "4", "5 * t"};
// Definition of an symmetric tensor external state variable components by components
@ExternalStateVariable<function> "g2XX"  "0";
@ExternalStateVariable<function> "g2YY"  "2";
@ExternalStateVariable<function> "g2ZZ"  "t";
@ExternalStateVariable<function> "g2XY"  "3";
@ExternalStateVariable<function> "g2XZ"  "4";
@ExternalStateVariable<function> "g2YZ"  "5*t";
~~~~

For more details, see : <https://github.com/thelfer/tfel/issues/28>.

## Issue 28: [mtest] Support of tensorial external state variables

For more details, see : <https://github.com/thelfer/tfel/issues/28>.

## Issue 27: [mfront] Better support of tensorial external state variables

The type of the external state variables are now exported.

- `ExternalLibraryManager` has a new method called
  `getUMATExternalStateVariablesTypes` (wrapped in `python`)
- `ExternalBehaviourDescription` has a new method called `evtypes`
  (wrapped in `python`)

Generation of `MTest` file also supports tensor external state variables.

For more details, see : <https://github.com/thelfer/tfel/issues/27>.

## Issue 26: Don't use explicit instanciation for static variables

For more details, see : <https://github.com/thelfer/tfel/issues/26>.

## Issue 25: [mfront] Add ability to define initialisation functions

This feature is described in Section
@sec:tfel_4.1:mfront:initialize_functions.

For more details, see : <https://github.com/thelfer/tfel/issues/25>.

## Issue 24: [mfront] Add ability to define post-processings

This feature is described in Section
@sec:tfel_4.1:mfront:postprocessings.

For more details, see : <https://github.com/thelfer/tfel/issues/24>.

## Issue 23: [mfront] Remove usage of C++ reserved variables names

Some exported variables used to contain a double underscore in their
names. Such names are reserved by the C++ standard for internal use by
the compilers.

For more details, see : <https://github.com/thelfer/tfel/issues/23>.

## Issue 22 : [mfront] Option to treat parameters as static variables

This option is described in depth in Section
@sec:tfel:4.1:mfront:global_options:parameters_as_static_variables.

For more details, see : <https://github.com/thelfer/tfel/issues/22>.

## Issue 20: [mtest] Declaration of a behaviour wrapper in python module

As described in Section @sec:tfel_4.1:pymtest:behaviour_constructor,
named parameters are now supported in the constructor of the `Behaviour`
class. The (optional) `wrapper` argument can now be used to specify a
behaviour wrapper.

Currently, two behaviours wrappers are available:

- `LogarithmicStrain1D`: which turns a small strain behaviour in
  axisymmetrical generalised plane strain into a finite strain behaviour
  in the logarithmic strain framework following the ETO-PK1 kinematic
  assumption used by the `MTest` for the simulation of pipe in finite
  strain.
- `SmallStrainTridimensionalBehaviourWrapper` which allows to use a
  small strain tridimensional behaviour under the following modelling
  hypotheses: axisymmetrical generalised plane strain, plane strain,
  axisymmetry.

For more details, see : <https://github.com/thelfer/tfel/issues/20>.

## Issue 18: [mtest] Usage of a `3D` behaviour in `ptest`

Thanks the `SmallStrainTridimensionalBehaviourWrapper` class detailled
in Section
@sec:tfel_4.1:mtest:small_strain_tridimensional_behaviour_wrapper,
tridimensional behaviours can be used under the following modelling
hypotheses: generalised plane strain, plane strain, axisymmetrical
generalised plane strain.

For more details, see : <https://github.com/thelfer/tfel/issues/18>.

## Issue 15: [mtest] New option for the declaration of the rotation matrix in `MTest`

This option is described in depth in Section
@sec:tfel_4.1:mtest:rotation_matrix.

For more details, see : <https://github.com/thelfer/tfel/issues/15>.

# References
