/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2025 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <TGUI/extlibs/IncludeSDL.hpp>
#include <TGUI/Backend/Renderer/SDL_GPU/BackendRenderTargetSDLGPU.hpp>

#if !TGUI_BUILD_AS_CXX_MODULE
    #include <TGUI/Backend/Renderer/BackendText.hpp>
    #include <TGUI/Backend/Window/Backend.hpp>
    #include <TGUI/Container.hpp>
#endif

#if !TGUI_EXPERIMENTAL_USE_STD_MODULE
    #include <numeric>
    #include <cmath>
#endif

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    // To edit the shaders, change the HSLS source files in src/Backend/Renderer/SDL_GPU/shaders and run compile_shaders.sh in
    // that folder to compile the shaders into different formats and generate the c++ code found below.

    static const std::uint8_t vertexShaderMSL[] = R"(
        #include <metal_stdlib>
        #include <simd/simd.h>

        using namespace metal;

        struct main0_out
        {
            float4 out_var_TEXCOORD0 [[user(locn0)]];
            float2 out_var_TEXCOORD1 [[user(locn1)]];
            float4 gl_Position [[position]];
        };

        struct main0_in
        {
            float2 in_var_TEXCOORD0 [[attribute(0)]];
            float4 in_var_TEXCOORD1 [[attribute(1)]];
            float2 in_var_TEXCOORD2 [[attribute(2)]];
        };

        vertex main0_out main0(main0_in in [[stage_in]])
        {
            main0_out out = {};
            out.out_var_TEXCOORD0 = in.in_var_TEXCOORD1;
            out.out_var_TEXCOORD1 = in.in_var_TEXCOORD2;
            out.gl_Position = float4(in.in_var_TEXCOORD0, 0.0, 1.0);
            return out;
        }
    )";

    static const std::uint8_t fragmentShaderMSL[] = R"(
        #include <metal_stdlib>
        #include <simd/simd.h>

        using namespace metal;

        struct main0_out
        {
            float4 out_var_SV_Target0 [[color(0)]];
        };

        struct main0_in
        {
            float4 in_var_TEXCOORD0 [[user(locn0)]];
            float2 in_var_TEXCOORD1 [[user(locn1)]];
        };

        fragment main0_out main0(main0_in in [[stage_in]], texture2d<float> Texture [[texture(0)]], sampler Sampler [[sampler(0)]])
        {
            main0_out out = {};
            out.out_var_SV_Target0 = Texture.sample(Sampler, in.in_var_TEXCOORD1) * in.in_var_TEXCOORD0;
            return out;
        }
    )";

    static const std::uint8_t vertexShaderSPV[] =
        "\x03\x02\x23\x07\x00\x00\x01\x00\x00\x00\x0e\x00\x1a\x00\x00\x00\x00\x00\x00\x00\x11\x00\x02\x00\x01\x00\x00\x00\x0e\x00\x03\x00\x00\x00\x00\x00\x01\x00\x00\x00"
        "\x0f\x00\x0b\x00\x00\x00\x00\x00\x01\x00\x00\x00\x6d\x61\x69\x6e\x00\x00\x00\x00\x02\x00\x00\x00\x03\x00\x00\x00\x04\x00\x00\x00\x05\x00\x00\x00\x06\x00\x00\x00"
        "\x07\x00\x00\x00\x03\x00\x03\x00\x05\x00\x00\x00\x58\x02\x00\x00\x05\x00\x07\x00\x02\x00\x00\x00\x69\x6e\x2e\x76\x61\x72\x2e\x54\x45\x58\x43\x4f\x4f\x52\x44\x30"
        "\x00\x00\x00\x00\x05\x00\x07\x00\x03\x00\x00\x00\x69\x6e\x2e\x76\x61\x72\x2e\x54\x45\x58\x43\x4f\x4f\x52\x44\x31\x00\x00\x00\x00\x05\x00\x07\x00\x04\x00\x00\x00"
        "\x69\x6e\x2e\x76\x61\x72\x2e\x54\x45\x58\x43\x4f\x4f\x52\x44\x32\x00\x00\x00\x00\x05\x00\x07\x00\x05\x00\x00\x00\x6f\x75\x74\x2e\x76\x61\x72\x2e\x54\x45\x58\x43"
        "\x4f\x4f\x52\x44\x30\x00\x00\x00\x05\x00\x07\x00\x06\x00\x00\x00\x6f\x75\x74\x2e\x76\x61\x72\x2e\x54\x45\x58\x43\x4f\x4f\x52\x44\x31\x00\x00\x00\x05\x00\x04\x00"
        "\x01\x00\x00\x00\x6d\x61\x69\x6e\x00\x00\x00\x00\x47\x00\x04\x00\x07\x00\x00\x00\x0b\x00\x00\x00\x00\x00\x00\x00\x47\x00\x04\x00\x02\x00\x00\x00\x1e\x00\x00\x00"
        "\x00\x00\x00\x00\x47\x00\x04\x00\x03\x00\x00\x00\x1e\x00\x00\x00\x01\x00\x00\x00\x47\x00\x04\x00\x04\x00\x00\x00\x1e\x00\x00\x00\x02\x00\x00\x00\x47\x00\x04\x00"
        "\x05\x00\x00\x00\x1e\x00\x00\x00\x00\x00\x00\x00\x47\x00\x04\x00\x06\x00\x00\x00\x1e\x00\x00\x00\x01\x00\x00\x00\x16\x00\x03\x00\x08\x00\x00\x00\x20\x00\x00\x00"
        "\x2b\x00\x04\x00\x08\x00\x00\x00\x09\x00\x00\x00\x00\x00\x00\x00\x2b\x00\x04\x00\x08\x00\x00\x00\x0a\x00\x00\x00\x00\x00\x80\x3f\x17\x00\x04\x00\x0b\x00\x00\x00"
        "\x08\x00\x00\x00\x02\x00\x00\x00\x20\x00\x04\x00\x0c\x00\x00\x00\x01\x00\x00\x00\x0b\x00\x00\x00\x17\x00\x04\x00\x0d\x00\x00\x00\x08\x00\x00\x00\x04\x00\x00\x00"
        "\x20\x00\x04\x00\x0e\x00\x00\x00\x01\x00\x00\x00\x0d\x00\x00\x00\x20\x00\x04\x00\x0f\x00\x00\x00\x03\x00\x00\x00\x0d\x00\x00\x00\x20\x00\x04\x00\x10\x00\x00\x00"
        "\x03\x00\x00\x00\x0b\x00\x00\x00\x13\x00\x02\x00\x11\x00\x00\x00\x21\x00\x03\x00\x12\x00\x00\x00\x11\x00\x00\x00\x3b\x00\x04\x00\x0c\x00\x00\x00\x02\x00\x00\x00"
        "\x01\x00\x00\x00\x3b\x00\x04\x00\x0e\x00\x00\x00\x03\x00\x00\x00\x01\x00\x00\x00\x3b\x00\x04\x00\x0c\x00\x00\x00\x04\x00\x00\x00\x01\x00\x00\x00\x3b\x00\x04\x00"
        "\x0f\x00\x00\x00\x05\x00\x00\x00\x03\x00\x00\x00\x3b\x00\x04\x00\x10\x00\x00\x00\x06\x00\x00\x00\x03\x00\x00\x00\x3b\x00\x04\x00\x0f\x00\x00\x00\x07\x00\x00\x00"
        "\x03\x00\x00\x00\x36\x00\x05\x00\x11\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x12\x00\x00\x00\xf8\x00\x02\x00\x13\x00\x00\x00\x3d\x00\x04\x00\x0b\x00\x00\x00"
        "\x14\x00\x00\x00\x02\x00\x00\x00\x3d\x00\x04\x00\x0d\x00\x00\x00\x15\x00\x00\x00\x03\x00\x00\x00\x3d\x00\x04\x00\x0b\x00\x00\x00\x16\x00\x00\x00\x04\x00\x00\x00"
        "\x51\x00\x05\x00\x08\x00\x00\x00\x17\x00\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\x51\x00\x05\x00\x08\x00\x00\x00\x18\x00\x00\x00\x14\x00\x00\x00\x01\x00\x00\x00"
        "\x50\x00\x07\x00\x0d\x00\x00\x00\x19\x00\x00\x00\x17\x00\x00\x00\x18\x00\x00\x00\x09\x00\x00\x00\x0a\x00\x00\x00\x3e\x00\x03\x00\x05\x00\x00\x00\x15\x00\x00\x00"
        "\x3e\x00\x03\x00\x06\x00\x00\x00\x16\x00\x00\x00\x3e\x00\x03\x00\x07\x00\x00\x00\x19\x00\x00\x00\xfd\x00\x01\x00\x38\x00\x01\x00";

    static const std::uint8_t fragmentShaderSPV[] =
        "\x03\x02\x23\x07\x00\x00\x01\x00\x00\x00\x0e\x00\x1c\x00\x00\x00\x00\x00\x00\x00\x11\x00\x02\x00\x01\x00\x00\x00\x0e\x00\x03\x00\x00\x00\x00\x00\x01\x00\x00\x00"
        "\x0f\x00\x08\x00\x04\x00\x00\x00\x01\x00\x00\x00\x6d\x61\x69\x6e\x00\x00\x00\x00\x02\x00\x00\x00\x03\x00\x00\x00\x04\x00\x00\x00\x10\x00\x03\x00\x01\x00\x00\x00"
        "\x07\x00\x00\x00\x03\x00\x03\x00\x05\x00\x00\x00\x58\x02\x00\x00\x05\x00\x06\x00\x05\x00\x00\x00\x74\x79\x70\x65\x2e\x32\x64\x2e\x69\x6d\x61\x67\x65\x00\x00\x00"
        "\x05\x00\x04\x00\x06\x00\x00\x00\x54\x65\x78\x74\x75\x72\x65\x00\x05\x00\x06\x00\x07\x00\x00\x00\x74\x79\x70\x65\x2e\x73\x61\x6d\x70\x6c\x65\x72\x00\x00\x00\x00"
        "\x05\x00\x04\x00\x08\x00\x00\x00\x53\x61\x6d\x70\x6c\x65\x72\x00\x05\x00\x07\x00\x02\x00\x00\x00\x69\x6e\x2e\x76\x61\x72\x2e\x54\x45\x58\x43\x4f\x4f\x52\x44\x30"
        "\x00\x00\x00\x00\x05\x00\x07\x00\x03\x00\x00\x00\x69\x6e\x2e\x76\x61\x72\x2e\x54\x45\x58\x43\x4f\x4f\x52\x44\x31\x00\x00\x00\x00\x05\x00\x07\x00\x04\x00\x00\x00"
        "\x6f\x75\x74\x2e\x76\x61\x72\x2e\x53\x56\x5f\x54\x61\x72\x67\x65\x74\x30\x00\x00\x05\x00\x04\x00\x01\x00\x00\x00\x6d\x61\x69\x6e\x00\x00\x00\x00\x05\x00\x07\x00"
        "\x09\x00\x00\x00\x74\x79\x70\x65\x2e\x73\x61\x6d\x70\x6c\x65\x64\x2e\x69\x6d\x61\x67\x65\x00\x00\x47\x00\x04\x00\x02\x00\x00\x00\x1e\x00\x00\x00\x00\x00\x00\x00"
        "\x47\x00\x04\x00\x03\x00\x00\x00\x1e\x00\x00\x00\x01\x00\x00\x00\x47\x00\x04\x00\x04\x00\x00\x00\x1e\x00\x00\x00\x00\x00\x00\x00\x47\x00\x04\x00\x06\x00\x00\x00"
        "\x22\x00\x00\x00\x02\x00\x00\x00\x47\x00\x04\x00\x06\x00\x00\x00\x21\x00\x00\x00\x00\x00\x00\x00\x47\x00\x04\x00\x08\x00\x00\x00\x22\x00\x00\x00\x02\x00\x00\x00"
        "\x47\x00\x04\x00\x08\x00\x00\x00\x21\x00\x00\x00\x00\x00\x00\x00\x16\x00\x03\x00\x0a\x00\x00\x00\x20\x00\x00\x00\x19\x00\x09\x00\x05\x00\x00\x00\x0a\x00\x00\x00"
        "\x01\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x20\x00\x04\x00\x0b\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00"
        "\x1a\x00\x02\x00\x07\x00\x00\x00\x20\x00\x04\x00\x0c\x00\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x17\x00\x04\x00\x0d\x00\x00\x00\x0a\x00\x00\x00\x04\x00\x00\x00"
        "\x20\x00\x04\x00\x0e\x00\x00\x00\x01\x00\x00\x00\x0d\x00\x00\x00\x17\x00\x04\x00\x0f\x00\x00\x00\x0a\x00\x00\x00\x02\x00\x00\x00\x20\x00\x04\x00\x10\x00\x00\x00"
        "\x01\x00\x00\x00\x0f\x00\x00\x00\x20\x00\x04\x00\x11\x00\x00\x00\x03\x00\x00\x00\x0d\x00\x00\x00\x13\x00\x02\x00\x12\x00\x00\x00\x21\x00\x03\x00\x13\x00\x00\x00"
        "\x12\x00\x00\x00\x1b\x00\x03\x00\x09\x00\x00\x00\x05\x00\x00\x00\x3b\x00\x04\x00\x0b\x00\x00\x00\x06\x00\x00\x00\x00\x00\x00\x00\x3b\x00\x04\x00\x0c\x00\x00\x00"
        "\x08\x00\x00\x00\x00\x00\x00\x00\x3b\x00\x04\x00\x0e\x00\x00\x00\x02\x00\x00\x00\x01\x00\x00\x00\x3b\x00\x04\x00\x10\x00\x00\x00\x03\x00\x00\x00\x01\x00\x00\x00"
        "\x3b\x00\x04\x00\x11\x00\x00\x00\x04\x00\x00\x00\x03\x00\x00\x00\x36\x00\x05\x00\x12\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x13\x00\x00\x00\xf8\x00\x02\x00"
        "\x14\x00\x00\x00\x3d\x00\x04\x00\x0d\x00\x00\x00\x15\x00\x00\x00\x02\x00\x00\x00\x3d\x00\x04\x00\x0f\x00\x00\x00\x16\x00\x00\x00\x03\x00\x00\x00\x3d\x00\x04\x00"
        "\x05\x00\x00\x00\x17\x00\x00\x00\x06\x00\x00\x00\x3d\x00\x04\x00\x07\x00\x00\x00\x18\x00\x00\x00\x08\x00\x00\x00\x56\x00\x05\x00\x09\x00\x00\x00\x19\x00\x00\x00"
        "\x17\x00\x00\x00\x18\x00\x00\x00\x57\x00\x06\x00\x0d\x00\x00\x00\x1a\x00\x00\x00\x19\x00\x00\x00\x16\x00\x00\x00\x00\x00\x00\x00\x85\x00\x05\x00\x0d\x00\x00\x00"
        "\x1b\x00\x00\x00\x1a\x00\x00\x00\x15\x00\x00\x00\x3e\x00\x03\x00\x04\x00\x00\x00\x1b\x00\x00\x00\xfd\x00\x01\x00\x38\x00\x01\x00";

    static const std::uint8_t vertexShaderDXIL[] =
        "\x44\x58\x42\x43\x65\xf4\x12\xe2\xf7\x6e\xc9\xbb\x42\xce\x13\x16\x43\xf8\xc3\xe6\x01\x00\x00\x00\x48\x0d\x00\x00\x07\x00\x00\x00\x3c\x00\x00\x00\x4c\x00\x00\x00"
        "\xc8\x00\x00\x00\x50\x01\x00\x00\x70\x02\x00\x00\x7c\x07\x00\x00\x98\x07\x00\x00\x53\x46\x49\x30\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x49\x53\x47\x31"
        "\x74\x00\x00\x00\x03\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x68\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x03\x03\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x68\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x01\x00\x00\x00\x0f\x0f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x68\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x02\x00\x00\x00\x03\x03\x00\x00\x00\x00\x00\x00\x54\x45\x58\x43\x4f\x4f\x52\x44\x00\x00\x00\x00"
        "\x4f\x53\x47\x31\x80\x00\x00\x00\x03\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x68\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00"
        "\x0f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x68\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x01\x00\x00\x00\x03\x0c\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x71\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x03\x00\x00\x00\x02\x00\x00\x00\x0f\x00\x00\x00\x00\x00\x00\x00\x54\x45\x58\x43\x4f\x4f\x52\x44"
        "\x00\x53\x56\x5f\x50\x6f\x73\x69\x74\x69\x6f\x6e\x00\x00\x00\x00\x50\x53\x56\x30\x18\x01\x00\x00\x34\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\x01\x00\x00\x00\x03\x03\x00\x03\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2e\x00\x00\x00"
        "\x00\x00\x00\x00\x34\x00\x00\x00\x00\x54\x45\x58\x43\x4f\x4f\x52\x44\x00\x54\x45\x58\x43\x4f\x4f\x52\x44\x00\x54\x45\x58\x43\x4f\x4f\x52\x44\x00\x54\x45\x58\x43"
        "\x4f\x4f\x52\x44\x00\x54\x45\x58\x43\x4f\x4f\x52\x44\x00\x6d\x61\x69\x6e\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00\x10\x00\x00\x00"
        "\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x42\x00\x03\x00\x00\x00\x0a\x00\x00\x00\x01\x00\x00\x00\x01\x01\x44\x00\x03\x00\x00\x00\x13\x00\x00\x00\x02\x00\x00\x00"
        "\x01\x02\x42\x00\x03\x00\x00\x00\x1c\x00\x00\x00\x00\x00\x00\x00\x01\x00\x44\x00\x03\x02\x00\x00\x25\x00\x00\x00\x01\x00\x00\x00\x01\x01\x42\x00\x03\x02\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x01\x02\x44\x03\x03\x04\x00\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00"
        "\x04\x00\x00\x00\x08\x00\x00\x00\x10\x00\x00\x00\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x53\x54\x41\x54\x04\x05\x00\x00\x60\x00\x01\x00\x41\x01\x00\x00"
        "\x44\x58\x49\x4c\x00\x01\x00\x00\x10\x00\x00\x00\xec\x04\x00\x00\x42\x43\xc0\xde\x21\x0c\x00\x00\x38\x01\x00\x00\x0b\x82\x20\x00\x02\x00\x00\x00\x13\x00\x00\x00"
        "\x07\x81\x23\x91\x41\xc8\x04\x49\x06\x10\x32\x39\x92\x01\x84\x0c\x25\x05\x08\x19\x1e\x04\x8b\x62\x80\x10\x45\x02\x42\x92\x0b\x42\x84\x10\x32\x14\x38\x08\x18\x4b"
        "\x0a\x32\x42\x88\x48\x90\x14\x20\x43\x46\x88\xa5\x00\x19\x32\x42\xe4\x48\x0e\x90\x11\x22\xc4\x50\x41\x51\x81\x8c\xe1\x83\xe5\x8a\x04\x21\x46\x06\x51\x18\x00\x00"
        "\x06\x00\x00\x00\x1b\x8c\xe0\xff\xff\xff\xff\x07\x40\x02\xa8\x0d\x84\xf0\xff\xff\xff\xff\x03\x20\x01\x00\x00\x00\x49\x18\x00\x00\x02\x00\x00\x00\x13\x82\x60\x42"
        "\x20\x00\x00\x00\x89\x20\x00\x00\x0f\x00\x00\x00\x32\x22\x08\x09\x20\x64\x85\x04\x13\x22\xa4\x84\x04\x13\x22\xe3\x84\xa1\x90\x14\x12\x4c\x88\x8c\x0b\x84\x84\x4c"
        "\x10\x30\x23\x00\x25\x00\x8a\x19\x80\x39\x02\x30\x98\x23\x40\x8a\x31\x44\x54\x44\x56\x0c\x20\xa2\x1a\xc2\x81\x80\x64\x20\x00\x00\x13\x14\x72\xc0\x87\x74\x60\x87"
        "\x36\x68\x87\x79\x68\x03\x72\xc0\x87\x0d\xaf\x50\x0e\x6d\xd0\x0e\x7a\x50\x0e\x6d\x00\x0f\x7a\x30\x07\x72\xa0\x07\x73\x20\x07\x6d\x90\x0e\x71\xa0\x07\x73\x20\x07"
        "\x6d\x90\x0e\x78\xa0\x07\x73\x20\x07\x6d\x90\x0e\x71\x60\x07\x7a\x30\x07\x72\xd0\x06\xe9\x30\x07\x72\xa0\x07\x73\x20\x07\x6d\x90\x0e\x76\x40\x07\x7a\x60\x07\x74"
        "\xd0\x06\xe6\x10\x07\x76\xa0\x07\x73\x20\x07\x6d\x60\x0e\x73\x20\x07\x7a\x30\x07\x72\xd0\x06\xe6\x60\x07\x74\xa0\x07\x76\x40\x07\x6d\xe0\x0e\x78\xa0\x07\x71\x60"
        "\x07\x7a\x30\x07\x72\xa0\x07\x76\x40\x07\x43\x9e\x00\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x3c\x06\x10\x00\x01\x00\x00\x00\x00\x00\x00\x00\x0c\x79\x10\x20"
        "\x00\x04\x00\x00\x00\x00\x00\x00\x00\xc8\x02\x01\x0e\x00\x00\x00\x32\x1e\x98\x14\x19\x11\x4c\x90\x8c\x09\x26\x47\xc6\x04\x43\xa2\x12\x18\x01\x28\x86\x32\x28\x8f"
        "\x22\x28\x93\x82\x28\x0a\xaa\x92\x18\x01\x28\x82\x42\x28\x03\xda\xb1\x18\x85\x01\x11\x10\x02\x80\x40\x20\x10\x08\x0c\x00\x00\x00\x79\x18\x00\x00\x6b\x00\x00\x00"
        "\x1a\x03\x4c\x90\x46\x02\x13\x44\x35\x20\xc3\x1b\x43\x81\x93\x4b\xb3\x0b\xa3\x2b\x4b\x01\x89\x71\xc1\x71\x81\x71\x89\xa1\xc1\xc1\xa1\x01\x41\x69\x0b\x4b\x73\x63"
        "\x01\x81\x31\x33\x23\xb3\x81\x09\x1b\x1b\x4b\xd9\x10\x04\x13\x04\x62\x98\x20\x10\xc4\x06\x61\x20\x36\x08\x04\x41\xc1\x6e\x6e\x82\x40\x14\x1b\x86\x03\x21\x26\x08"
        "\x02\xb0\x01\xd8\x30\x10\xcb\xb2\x21\x60\x36\x0c\x83\xd2\x4c\x10\x96\x68\x43\xf0\x90\x68\x0b\x4b\x73\x23\x42\x55\x84\x35\xf4\xf4\x24\x45\x34\x41\x28\x98\x09\x42"
        "\xd1\x6c\x08\x88\x09\x42\xe1\x4c\x10\x08\x63\x83\x70\x5d\x1b\x16\x42\x9a\xa8\x8a\x1a\x2c\x82\xc2\x36\x04\xc3\x04\xa1\x78\x26\x08\xc4\xb1\x41\xb8\xb8\x0d\xcb\x20"
        "\x4d\x94\x46\x0d\xdb\x40\x75\x13\x04\x02\xd9\x10\x7c\x1b\x96\x4f\x9a\x28\x30\xa0\x06\xeb\xa3\xb0\x0d\x43\xe6\x85\xc1\x86\x85\x90\x26\xaa\xb2\x86\x8d\xa0\xba\x0d"
        "\xcb\x20\x4d\x94\x66\x0d\xd6\x40\x61\x5c\xa6\xac\xbe\xa0\xde\xe6\xd2\xe8\xd2\xde\xdc\x26\x08\x05\xb4\x61\xf9\xca\x60\x32\x83\x6a\x1b\xb6\x8f\xea\x36\x0c\x63\x40"
        "\x06\x67\xb0\x61\x10\x03\x34\x00\x36\x14\x4a\x94\x06\x00\xc0\x22\xcd\x6d\x8e\x6e\x6e\x82\x40\x24\x34\xe6\xd2\xce\xbe\xd8\xc8\x26\x08\x84\x42\x63\x2e\xed\xec\x6b"
        "\x8e\x6e\x82\x40\x2c\x1b\x8c\x35\x60\x83\x36\x70\x83\x37\x80\x83\x2a\x6c\x6c\x76\x6d\x2e\x69\x64\x65\x6e\x74\x53\x82\xa0\x0a\x19\x9e\x8b\x5d\x99\xdc\x5c\xda\x9b"
        "\xdb\x94\x80\x68\x42\x86\xe7\x62\x17\xc6\x66\x57\x26\x37\x25\x28\xea\x90\xe1\xb9\xcc\xa1\x85\x91\x95\xc9\x35\xbd\x91\x95\xb1\x4d\x09\x90\x4a\x64\x78\x2e\x74\x79"
        "\x70\x65\x41\x6e\x6e\x6f\x74\x61\x74\x69\x6f\x6e\x73\x53\x82\xa6\x0e\x19\x9e\x8b\x5d\x5a\xd9\x5d\x12\xd9\x14\x5d\x18\x5d\xd9\x94\xe0\xa9\x43\x86\xe7\x52\xe6\x46"
        "\x27\x97\x07\xf5\x96\xe6\x46\x37\x37\x25\x48\x83\x2e\x64\x78\x2e\x63\x6f\x75\x6e\x74\x65\x72\x73\x53\x02\x38\x00\x79\x18\x00\x00\x4c\x00\x00\x00\x33\x08\x80\x1c"
        "\xc4\xe1\x1c\x66\x14\x01\x3d\x88\x43\x38\x84\xc3\x8c\x42\x80\x07\x79\x78\x07\x73\x98\x71\x0c\xe6\x00\x0f\xed\x10\x0e\xf4\x80\x0e\x33\x0c\x42\x1e\xc2\xc1\x1d\xce"
        "\xa1\x1c\x66\x30\x05\x3d\x88\x43\x38\x84\x83\x1b\xcc\x03\x3d\xc8\x43\x3d\x8c\x03\x3d\xcc\x78\x8c\x74\x70\x07\x7b\x08\x07\x79\x48\x87\x70\x70\x07\x7a\x70\x03\x76"
        "\x78\x87\x70\x20\x87\x19\xcc\x11\x0e\xec\x90\x0e\xe1\x30\x0f\x6e\x30\x0f\xe3\xf0\x0e\xf0\x50\x0e\x33\x10\xc4\x1d\xde\x21\x1c\xd8\x21\x1d\xc2\x61\x1e\x66\x30\x89"
        "\x3b\xbc\x83\x3b\xd0\x43\x39\xb4\x03\x3c\xbc\x83\x3c\x84\x03\x3b\xcc\xf0\x14\x76\x60\x07\x7b\x68\x07\x37\x68\x87\x72\x68\x07\x37\x80\x87\x70\x90\x87\x70\x60\x07"
        "\x76\x28\x07\x76\xf8\x05\x76\x78\x87\x77\x80\x87\x5f\x08\x87\x71\x18\x87\x72\x98\x87\x79\x98\x81\x2c\xee\xf0\x0e\xee\xe0\x0e\xf5\xc0\x0e\xec\x30\x03\x62\xc8\xa1"
        "\x1c\xe4\xa1\x1c\xcc\xa1\x1c\xe4\xa1\x1c\xdc\x61\x1c\xca\x21\x1c\xc4\x81\x1d\xca\x61\x06\xd6\x90\x43\x39\xc8\x43\x39\x98\x43\x39\xc8\x43\x39\xb8\xc3\x38\x94\x43"
        "\x38\x88\x03\x3b\x94\xc3\x2f\xbc\x83\x3c\xfc\x82\x3b\xd4\x03\x3b\xb0\xc3\x8c\xc8\x21\x07\x7c\x70\x03\x72\x10\x87\x73\x70\x03\x7b\x08\x07\x79\x60\x87\x70\xc8\x87"
        "\x77\xa8\x07\x7a\x98\x81\x3c\xe4\x80\x0f\x6e\x40\x0f\xe5\xd0\x0e\xf0\x00\x00\x00\x71\x20\x00\x00\x0b\x00\x00\x00\x16\x30\x0d\x97\xef\x3c\xfe\xe2\x00\x83\xd8\x3c"
        "\xd4\xe4\x17\xb7\x6d\x02\xd5\x70\xf9\xce\xe3\x4b\x93\x13\x11\x28\x35\x3d\xd4\xe4\x17\xb7\x6d\x00\x04\x03\x20\x0d\x00\x00\x00\x00\x00\x00\x00\x00\x48\x41\x53\x48"
        "\x14\x00\x00\x00\x00\x00\x00\x00\x42\x78\xeb\x6b\xf5\xec\xe3\x33\x3d\x18\xf4\x75\xc5\xe1\xac\x85\x44\x58\x49\x4c\xa8\x05\x00\x00\x60\x00\x01\x00\x6a\x01\x00\x00"
        "\x44\x58\x49\x4c\x00\x01\x00\x00\x10\x00\x00\x00\x90\x05\x00\x00\x42\x43\xc0\xde\x21\x0c\x00\x00\x61\x01\x00\x00\x0b\x82\x20\x00\x02\x00\x00\x00\x13\x00\x00\x00"
        "\x07\x81\x23\x91\x41\xc8\x04\x49\x06\x10\x32\x39\x92\x01\x84\x0c\x25\x05\x08\x19\x1e\x04\x8b\x62\x80\x10\x45\x02\x42\x92\x0b\x42\x84\x10\x32\x14\x38\x08\x18\x4b"
        "\x0a\x32\x42\x88\x48\x90\x14\x20\x43\x46\x88\xa5\x00\x19\x32\x42\xe4\x48\x0e\x90\x11\x22\xc4\x50\x41\x51\x81\x8c\xe1\x83\xe5\x8a\x04\x21\x46\x06\x51\x18\x00\x00"
        "\x06\x00\x00\x00\x1b\x8c\xe0\xff\xff\xff\xff\x07\x40\x02\xa8\x0d\x84\xf0\xff\xff\xff\xff\x03\x20\x01\x00\x00\x00\x49\x18\x00\x00\x02\x00\x00\x00\x13\x82\x60\x42"
        "\x20\x00\x00\x00\x89\x20\x00\x00\x0f\x00\x00\x00\x32\x22\x08\x09\x20\x64\x85\x04\x13\x22\xa4\x84\x04\x13\x22\xe3\x84\xa1\x90\x14\x12\x4c\x88\x8c\x0b\x84\x84\x4c"
        "\x10\x30\x23\x00\x25\x00\x8a\x19\x80\x39\x02\x30\x98\x23\x40\x8a\x31\x44\x54\x44\x56\x0c\x20\xa2\x1a\xc2\x81\x80\x64\x20\x00\x00\x13\x14\x72\xc0\x87\x74\x60\x87"
        "\x36\x68\x87\x79\x68\x03\x72\xc0\x87\x0d\xaf\x50\x0e\x6d\xd0\x0e\x7a\x50\x0e\x6d\x00\x0f\x7a\x30\x07\x72\xa0\x07\x73\x20\x07\x6d\x90\x0e\x71\xa0\x07\x73\x20\x07"
        "\x6d\x90\x0e\x78\xa0\x07\x73\x20\x07\x6d\x90\x0e\x71\x60\x07\x7a\x30\x07\x72\xd0\x06\xe9\x30\x07\x72\xa0\x07\x73\x20\x07\x6d\x90\x0e\x76\x40\x07\x7a\x60\x07\x74"
        "\xd0\x06\xe6\x10\x07\x76\xa0\x07\x73\x20\x07\x6d\x60\x0e\x73\x20\x07\x7a\x30\x07\x72\xd0\x06\xe6\x60\x07\x74\xa0\x07\x76\x40\x07\x6d\xe0\x0e\x78\xa0\x07\x71\x60"
        "\x07\x7a\x30\x07\x72\xa0\x07\x76\x40\x07\x43\x9e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x3c\x06\x10\x00\x01\x00\x00\x00\x00\x00\x00\x00\x0c\x79\x10\x20"
        "\x00\x04\x00\x00\x00\x00\x00\x00\x00\xc8\x02\x01\x0d\x00\x00\x00\x32\x1e\x98\x14\x19\x11\x4c\x90\x8c\x09\x26\x47\xc6\x04\x43\xa2\x12\x18\x01\x28\x89\x62\x28\x83"
        "\xf2\x28\x02\xaa\x92\x18\x01\x28\x82\x42\x28\x03\xda\xb1\x18\x85\x01\x11\x10\x02\x80\x40\x20\x10\x08\x0c\x00\x00\x79\x18\x00\x00\x55\x00\x00\x00\x1a\x03\x4c\x90"
        "\x46\x02\x13\x44\x35\x20\xc3\x1b\x43\x81\x93\x4b\xb3\x0b\xa3\x2b\x4b\x01\x89\x71\xc1\x71\x81\x71\x89\xa1\xc1\xc1\xa1\x01\x41\x69\x0b\x4b\x73\x63\x01\x81\x31\x33"
        "\x23\xb3\x81\x09\x1b\x1b\x4b\xd9\x10\x04\x13\x04\x62\x98\x20\x10\xc4\x06\x61\x20\x26\x08\x44\xb1\x41\x18\x0c\x0a\x76\x73\x13\x04\xc2\xd8\x30\x20\x09\x31\x41\x58"
        "\x9e\x0d\xc1\x32\x41\x10\x00\x12\x6d\x61\x69\x6e\x44\xa8\x8a\xb0\x86\x9e\x9e\xa4\x88\x26\x08\x85\x32\x41\x28\x96\x0d\x01\x31\x41\x28\x98\x09\x02\x71\x6c\x10\x28"
        "\x6a\xc3\x42\x3c\x50\x24\x45\xc3\x44\x44\xd5\x86\x60\x98\x20\x14\xcd\x04\x81\x40\x36\x08\x54\xb6\x61\x19\x1e\x28\xba\xa2\x01\x1b\x22\x6d\x82\x40\x24\x1b\x02\x6e"
        "\xc3\xc2\x3d\x50\xd4\x45\xc3\xc4\x45\xd5\x86\xc1\xda\xbc\x0d\x0b\xf1\x40\x91\x34\x0d\x18\x11\x69\x1b\x96\xe1\x81\xa2\x6b\x1a\xa6\x21\xaa\xb8\x4c\x59\x7d\x41\xbd"
        "\xcd\xa5\xd1\xa5\xbd\xb9\x4d\x10\x0a\x67\xc3\xc2\x89\x01\x34\x06\x12\x36\x60\x5c\xa4\x6d\x18\xc0\x20\x0c\xc8\x60\xc3\xf0\x95\x01\xb0\xa1\x68\x1c\x33\x00\x80\x2a"
        "\x6c\x6c\x76\x6d\x2e\x69\x64\x65\x6e\x74\x53\x82\xa0\x0a\x19\x9e\x8b\x5d\x99\xdc\x5c\xda\x9b\xdb\x94\x80\x68\x42\x86\xe7\x62\x17\xc6\x66\x57\x26\x37\x25\x30\xea"
        "\x90\xe1\xb9\xcc\xa1\x85\x91\x95\xc9\x35\xbd\x91\x95\xb1\x4d\x09\x92\x3a\x64\x78\x2e\x76\x69\x65\x77\x49\x64\x53\x74\x61\x74\x65\x53\x82\xa5\x0e\x19\x9e\x4b\x99"
        "\x1b\x9d\x5c\x1e\xd4\x5b\x9a\x1b\xdd\xdc\x94\xc0\x0c\x00\x00\x00\x79\x18\x00\x00\x4c\x00\x00\x00\x33\x08\x80\x1c\xc4\xe1\x1c\x66\x14\x01\x3d\x88\x43\x38\x84\xc3"
        "\x8c\x42\x80\x07\x79\x78\x07\x73\x98\x71\x0c\xe6\x00\x0f\xed\x10\x0e\xf4\x80\x0e\x33\x0c\x42\x1e\xc2\xc1\x1d\xce\xa1\x1c\x66\x30\x05\x3d\x88\x43\x38\x84\x83\x1b"
        "\xcc\x03\x3d\xc8\x43\x3d\x8c\x03\x3d\xcc\x78\x8c\x74\x70\x07\x7b\x08\x07\x79\x48\x87\x70\x70\x07\x7a\x70\x03\x76\x78\x87\x70\x20\x87\x19\xcc\x11\x0e\xec\x90\x0e"
        "\xe1\x30\x0f\x6e\x30\x0f\xe3\xf0\x0e\xf0\x50\x0e\x33\x10\xc4\x1d\xde\x21\x1c\xd8\x21\x1d\xc2\x61\x1e\x66\x30\x89\x3b\xbc\x83\x3b\xd0\x43\x39\xb4\x03\x3c\xbc\x83"
        "\x3c\x84\x03\x3b\xcc\xf0\x14\x76\x60\x07\x7b\x68\x07\x37\x68\x87\x72\x68\x07\x37\x80\x87\x70\x90\x87\x70\x60\x07\x76\x28\x07\x76\xf8\x05\x76\x78\x87\x77\x80\x87"
        "\x5f\x08\x87\x71\x18\x87\x72\x98\x87\x79\x98\x81\x2c\xee\xf0\x0e\xee\xe0\x0e\xf5\xc0\x0e\xec\x30\x03\x62\xc8\xa1\x1c\xe4\xa1\x1c\xcc\xa1\x1c\xe4\xa1\x1c\xdc\x61"
        "\x1c\xca\x21\x1c\xc4\x81\x1d\xca\x61\x06\xd6\x90\x43\x39\xc8\x43\x39\x98\x43\x39\xc8\x43\x39\xb8\xc3\x38\x94\x43\x38\x88\x03\x3b\x94\xc3\x2f\xbc\x83\x3c\xfc\x82"
        "\x3b\xd4\x03\x3b\xb0\xc3\x8c\xc8\x21\x07\x7c\x70\x03\x72\x10\x87\x73\x70\x03\x7b\x08\x07\x79\x60\x87\x70\xc8\x87\x77\xa8\x07\x7a\x98\x81\x3c\xe4\x80\x0f\x6e\x40"
        "\x0f\xe5\xd0\x0e\xf0\x00\x00\x00\x71\x20\x00\x00\x0b\x00\x00\x00\x16\x30\x0d\x97\xef\x3c\xfe\xe2\x00\x83\xd8\x3c\xd4\xe4\x17\xb7\x6d\x02\xd5\x70\xf9\xce\xe3\x4b"
        "\x93\x13\x11\x28\x35\x3d\xd4\xe4\x17\xb7\x6d\x00\x04\x03\x20\x0d\x00\x00\x00\x00\x61\x20\x00\x00\x3e\x00\x00\x00\x13\x04\x41\x2c\x10\x00\x00\x00\x05\x00\x00\x00"
        "\x54\x25\x40\x34\x03\x50\x0a\x85\x40\x33\x46\x00\x82\x20\x88\x7f\x23\x00\x00\x00\x23\x06\x09\x00\x82\x60\x60\x54\x43\x23\x2d\xc5\x88\x41\x02\x80\x20\x18\x18\x16"
        "\xe1\x4c\x87\x31\x62\x90\x00\x20\x08\x06\xc6\x55\x54\x54\x73\x8c\x18\x24\x00\x08\x82\x81\x81\x19\x56\x95\x20\x23\x06\x09\x00\x82\x60\x60\x64\xc7\x65\x39\xc9\x88"
        "\x41\x02\x80\x20\x18\x18\x1a\x82\x5d\x8d\x32\x62\x90\x00\x20\x08\x06\xc6\x96\x60\x58\xb4\x8c\x18\x24\x00\x08\x82\x81\xc1\x29\x59\xd6\x30\x23\x06\x09\x00\x82\x60"
        "\x80\x70\x8c\xa6\x4d\xc6\x88\x41\x02\x80\x20\x18\x20\x1c\xa3\x69\x4e\x31\x62\x90\x00\x20\x08\x06\x08\xc7\x68\x9a\x44\x8c\x18\x24\x00\x08\x82\x01\xc2\x31\x9a\x06"
        "\x0d\x23\x06\x09\x00\x82\x60\x80\x70\xcc\xa6\x4d\xc8\x88\x41\x02\x80\x20\x18\x20\x1c\xb3\x69\xce\x31\x62\x90\x00\x20\x08\x06\x08\xc7\x54\xda\x24\x8c\x18\x24\x00"
        "\x08\x82\x01\xc2\x31\x95\xe6\x04\x23\x06\x09\x00\x82\x60\x80\x70\x4c\xa5\x49\xc9\x88\x41\x02\x80\x20\x18\x20\x1c\x53\x69\x90\x82\x00\x00\x00\x00\x00\x00\x00\x00";

    static const std::uint8_t fragmentShaderDXIL[] =
        "\x44\x58\x42\x43\x6a\x4a\x04\xcd\xc1\x3b\x3f\x12\x1c\xd7\x40\xf8\x48\x17\x2c\xe8\x01\x00\x00\x00\x20\x0f\x00\x00\x07\x00\x00\x00\x3c\x00\x00\x00\x4c\x00\x00\x00"
        "\xa8\x00\x00\x00\xe4\x00\x00\x00\xd8\x01\x00\x00\x3c\x08\x00\x00\x58\x08\x00\x00\x53\x46\x49\x30\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x49\x53\x47\x31"
        "\x54\x00\x00\x00\x02\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x48\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x0f\x0f\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x48\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x01\x00\x00\x00\x03\x03\x00\x00\x00\x00\x00\x00\x54\x45\x58\x43"
        "\x4f\x4f\x52\x44\x00\x00\x00\x00\x4f\x53\x47\x31\x34\x00\x00\x00\x01\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x28\x00\x00\x00\x00\x00\x00\x00\x40\x00\x00\x00"
        "\x03\x00\x00\x00\x00\x00\x00\x00\x0f\x00\x00\x00\x00\x00\x00\x00\x53\x56\x5f\x54\x61\x72\x67\x65\x74\x00\x00\x00\x50\x53\x56\x30\xec\x00\x00\x00\x34\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\x02\x01\x00\x02\x01\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x13\x00\x00\x00\x02\x00\x00\x00\x18\x00\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0e\x00\x00\x00"
        "\x00\x00\x00\x00\x03\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x18\x00\x00\x00\x00\x54\x45\x58\x43\x4f\x4f\x52"
        "\x44\x00\x54\x45\x58\x43\x4f\x4f\x52\x44\x00\x6d\x61\x69\x6e\x00\x02\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x10\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00"
        "\x01\x00\x44\x00\x03\x02\x00\x00\x0a\x00\x00\x00\x01\x00\x00\x00\x01\x01\x42\x00\x03\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x44\x10\x03\x00\x00\x00"
        "\x01\x00\x00\x00\x02\x00\x00\x00\x04\x00\x00\x00\x08\x00\x00\x00\x0f\x00\x00\x00\x0f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x53\x54\x41\x54\x5c\x06\x00\x00"
        "\x60\x00\x00\x00\x97\x01\x00\x00\x44\x58\x49\x4c\x00\x01\x00\x00\x10\x00\x00\x00\x44\x06\x00\x00\x42\x43\xc0\xde\x21\x0c\x00\x00\x8e\x01\x00\x00\x0b\x82\x20\x00"
        "\x02\x00\x00\x00\x13\x00\x00\x00\x07\x81\x23\x91\x41\xc8\x04\x49\x06\x10\x32\x39\x92\x01\x84\x0c\x25\x05\x08\x19\x1e\x04\x8b\x62\x80\x14\x45\x02\x42\x92\x0b\x42"
        "\xa4\x10\x32\x14\x38\x08\x18\x4b\x0a\x32\x52\x88\x48\x90\x14\x20\x43\x46\x88\xa5\x00\x19\x32\x42\xe4\x48\x0e\x90\x91\x22\xc4\x50\x41\x51\x81\x8c\xe1\x83\xe5\x8a"
        "\x04\x29\x46\x06\x51\x18\x00\x00\x08\x00\x00\x00\x1b\x8c\xe0\xff\xff\xff\xff\x07\x40\x02\xa8\x0d\x84\xf0\xff\xff\xff\xff\x03\x20\x6d\x30\x86\xff\xff\xff\xff\x1f"
        "\x00\x09\xa8\x00\x49\x18\x00\x00\x03\x00\x00\x00\x13\x82\x60\x42\x20\x4c\x08\x06\x00\x00\x00\x00\x89\x20\x00\x00\x43\x00\x00\x00\x32\x22\x48\x09\x20\x64\x85\x04"
        "\x93\x22\xa4\x84\x04\x93\x22\xe3\x84\xa1\x90\x14\x12\x4c\x8a\x8c\x0b\x84\xa4\x4c\x10\x68\x23\x00\x25\x00\x14\x66\x00\xe6\x08\xc0\x60\x8e\x00\x29\xc6\x20\x84\x14"
        "\x42\xa6\x18\x80\x10\x52\x06\xa1\x9b\x86\xcb\x9f\xb0\x87\x90\xfc\x95\x90\x56\x62\xf2\x8b\xdb\x46\xc5\x18\x63\x10\x2a\xf7\x0c\x97\x3f\x61\x0f\x21\xf9\x21\xd0\x0c"
        "\x0b\x81\x82\x55\x18\x45\x18\x1b\x63\x0c\x42\xc8\xa0\x36\x47\x10\x14\x83\x91\x42\xc8\x23\x38\x10\x30\x8c\x40\x0c\x33\xb5\xc1\x38\xb0\x43\x38\xcc\xc3\x3c\xb8\x01"
        "\x2d\x94\x03\x3e\xd0\x43\x3d\xc8\x43\x39\xc8\x01\x29\xf0\x81\x3d\x94\xc3\x38\xd0\xc3\x3b\xc8\x03\x1f\x98\x03\x3b\xbc\x43\x38\xd0\x03\x1b\x80\x01\x1d\xf8\x01\x18"
        "\xf8\x81\x1e\xe8\x41\x3b\xa4\x03\x3c\xcc\xc3\x2f\xd0\x43\x3e\xc0\x43\x39\xa0\x80\xcc\x24\x06\xe3\xc0\x0e\xe1\x30\x0f\xf3\xe0\x06\xb4\x50\x0e\xf8\x40\x0f\xf5\x20"
        "\x0f\xe5\x20\x07\xa4\xc0\x07\xf6\x50\x0e\xe3\x40\x0f\xef\x20\x0f\x7c\x60\x0e\xec\xf0\x0e\xe1\x40\x0f\x6c\x00\x06\x74\xe0\x07\x60\xe0\x07\x48\x98\x94\xea\x4d\xd2"
        "\x14\x51\xc2\xe4\xb3\x00\xf3\x2c\x44\xc4\x4e\xc0\x44\xa0\x80\xd0\x4d\x04\x02\x00\x13\x14\x72\xc0\x87\x74\x60\x87\x36\x68\x87\x79\x68\x03\x72\xc0\x87\x0d\xaf\x50"
        "\x0e\x6d\xd0\x0e\x7a\x50\x0e\x6d\x00\x0f\x7a\x30\x07\x72\xa0\x07\x73\x20\x07\x6d\x90\x0e\x71\xa0\x07\x73\x20\x07\x6d\x90\x0e\x78\xa0\x07\x73\x20\x07\x6d\x90\x0e"
        "\x71\x60\x07\x7a\x30\x07\x72\xd0\x06\xe9\x30\x07\x72\xa0\x07\x73\x20\x07\x6d\x90\x0e\x76\x40\x07\x7a\x60\x07\x74\xd0\x06\xe6\x10\x07\x76\xa0\x07\x73\x20\x07\x6d"
        "\x60\x0e\x73\x20\x07\x7a\x30\x07\x72\xd0\x06\xe6\x60\x07\x74\xa0\x07\x76\x40\x07\x6d\xe0\x0e\x78\xa0\x07\x71\x60\x07\x7a\x30\x07\x72\xa0\x07\x76\x40\x07\x43\x9e"
        "\x00\x08\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x3c\x06\x10\x00\x01\x00\x00\x00\x00\x00\x00\x00\x0c\x79\x10\x20\x00\x04\x00\x00\x00\x00\x00\x00\x00\x18\xf2\x34"
        "\x40\x00\x0c\x00\x00\x00\x00\x00\x00\x00\x30\xe4\x81\x80\x00\x18\x00\x00\x00\x00\x00\x00\x00\x20\x0b\x04\x00\x00\x0e\x00\x00\x00\x32\x1e\x98\x14\x19\x11\x4c\x90"
        "\x8c\x09\x26\x47\xc6\x04\x43\x22\x25\x30\x02\x50\x0c\x45\x50\x12\x65\x50\x1e\x85\x50\x2c\x54\x4a\x62\x04\xa0\x08\x0a\xa1\x40\xc8\xce\x00\x10\x9e\x01\xa0\x3c\x16"
        "\x62\x10\x81\x40\x20\xcf\x03\x00\x79\x18\x00\x00\x7c\x00\x00\x00\x1a\x03\x4c\x90\x46\x02\x13\x44\x35\x20\xc3\x1b\x43\x81\x93\x4b\xb3\x0b\xa3\x2b\x4b\x01\x89\x71"
        "\xc1\x71\x81\x71\x89\xa1\xc1\xc1\xa1\x01\x41\x69\x0b\x4b\x73\x63\x01\x81\x31\x33\x23\xb3\x81\x09\x1b\x1b\x4b\xd9\x10\x04\x13\x04\xa2\x98\x20\x10\xc6\x06\x61\x20"
        "\x36\x08\x04\x41\x01\x6e\x6e\x82\x40\x1c\x1b\x86\x03\x21\x26\x08\xd6\xc4\x83\xaa\x0c\x8f\xae\x4e\xae\x6c\x82\x40\x20\x13\x04\x22\xd9\x20\x10\xcd\x86\x84\x50\x16"
        "\x86\x18\x18\xc2\xd9\x10\x3c\x13\x04\x8c\xe2\x31\x15\xd6\x06\xc7\x56\x26\xb7\x01\x21\x22\x89\x21\x06\x02\xd8\x10\x4c\x1b\x08\x08\x00\xa8\x09\x82\x00\x6c\x00\x36"
        "\x0c\xc4\x75\x6d\x08\xb0\x0d\xc3\x60\x65\x13\x84\xac\xda\x10\x6c\x24\xda\xc2\xd2\xdc\x88\x50\x15\x61\x0d\x3d\x3d\x49\x11\x4d\x10\x0a\x67\x82\x50\x3c\x1b\x02\x62"
        "\x82\x50\x40\x13\x84\x22\x9a\x20\x10\xca\x04\x81\x58\x36\x08\x64\x50\x06\x1b\x16\xc2\xfb\xc0\x20\x0c\xc4\x60\x18\x03\x02\x0c\xcc\x60\x43\x30\x6c\x10\xc8\x80\x0c"
        "\x36\x2c\x83\xf7\x81\x01\x1a\x88\xc1\x20\x06\x03\x18\xa4\xc1\x06\xe1\x0c\xd4\x80\xc9\x94\xd5\x17\x55\x98\xdc\x59\x19\xdd\x04\xa1\x90\x36\x2c\x04\x1b\x7c\x6d\x10"
        "\x06\x60\x30\x8c\x01\x01\x06\x66\xb0\x21\x70\x83\x0d\xc3\x1a\xbc\x01\xb0\xa1\xb0\x3a\x38\xa8\x00\x1a\x66\x6c\x6f\x61\x74\x73\x13\x04\x82\x61\x91\xe6\x36\x47\x37"
        "\x37\x41\x20\x1a\x1a\x73\x69\x67\x5f\x6c\x64\x34\xe6\xd2\xce\xbe\xe6\xe8\x88\xd0\x95\xe1\x7d\xb9\xbd\xc9\xb5\x6d\x50\xe4\x60\x0e\xe8\xa0\x0e\xec\x00\xb9\x83\x39"
        "\xc0\x83\xa1\x0a\x1b\x9b\x5d\x9b\x4b\x1a\x59\x99\x1b\xdd\x94\x20\xa8\x42\x86\xe7\x62\x57\x26\x37\x97\xf6\xe6\x36\x25\x20\x9a\x90\xe1\xb9\xd8\x85\xb1\xd9\x95\xc9"
        "\x4d\x09\x8a\x3a\x64\x78\x2e\x73\x68\x61\x64\x65\x72\x4d\x6f\x64\x65\x6c\x53\x02\xa4\x0c\x19\x9e\x8b\x5c\xd9\xdc\x5b\x9d\xdc\x58\xd9\xdc\x94\x80\xaa\x44\x86\xe7"
        "\x42\x97\x07\x57\x16\xe4\xe6\xf6\x46\x17\x46\x97\xf6\xe6\x36\x37\x25\xc8\xea\x90\xe1\xb9\xd8\xa5\x95\xdd\x25\x91\x4d\xd1\x85\xd1\x95\x4d\x09\xb6\x3a\x64\x78\x2e"
        "\x65\x6e\x74\x72\x79\x50\x6f\x69\x6e\x74\x73\x53\x02\x38\xe8\x42\x86\xe7\x32\xf6\x56\xe7\x46\x57\x26\x37\x37\x25\xc0\x03\x00\x00\x79\x18\x00\x00\x4c\x00\x00\x00"
        "\x33\x08\x80\x1c\xc4\xe1\x1c\x66\x14\x01\x3d\x88\x43\x38\x84\xc3\x8c\x42\x80\x07\x79\x78\x07\x73\x98\x71\x0c\xe6\x00\x0f\xed\x10\x0e\xf4\x80\x0e\x33\x0c\x42\x1e"
        "\xc2\xc1\x1d\xce\xa1\x1c\x66\x30\x05\x3d\x88\x43\x38\x84\x83\x1b\xcc\x03\x3d\xc8\x43\x3d\x8c\x03\x3d\xcc\x78\x8c\x74\x70\x07\x7b\x08\x07\x79\x48\x87\x70\x70\x07"
        "\x7a\x70\x03\x76\x78\x87\x70\x20\x87\x19\xcc\x11\x0e\xec\x90\x0e\xe1\x30\x0f\x6e\x30\x0f\xe3\xf0\x0e\xf0\x50\x0e\x33\x10\xc4\x1d\xde\x21\x1c\xd8\x21\x1d\xc2\x61"
        "\x1e\x66\x30\x89\x3b\xbc\x83\x3b\xd0\x43\x39\xb4\x03\x3c\xbc\x83\x3c\x84\x03\x3b\xcc\xf0\x14\x76\x60\x07\x7b\x68\x07\x37\x68\x87\x72\x68\x07\x37\x80\x87\x70\x90"
        "\x87\x70\x60\x07\x76\x28\x07\x76\xf8\x05\x76\x78\x87\x77\x80\x87\x5f\x08\x87\x71\x18\x87\x72\x98\x87\x79\x98\x81\x2c\xee\xf0\x0e\xee\xe0\x0e\xf5\xc0\x0e\xec\x30"
        "\x03\x62\xc8\xa1\x1c\xe4\xa1\x1c\xcc\xa1\x1c\xe4\xa1\x1c\xdc\x61\x1c\xca\x21\x1c\xc4\x81\x1d\xca\x61\x06\xd6\x90\x43\x39\xc8\x43\x39\x98\x43\x39\xc8\x43\x39\xb8"
        "\xc3\x38\x94\x43\x38\x88\x03\x3b\x94\xc3\x2f\xbc\x83\x3c\xfc\x82\x3b\xd4\x03\x3b\xb0\xc3\x8c\xc8\x21\x07\x7c\x70\x03\x72\x10\x87\x73\x70\x03\x7b\x08\x07\x79\x60"
        "\x87\x70\xc8\x87\x77\xa8\x07\x7a\x98\x81\x3c\xe4\x80\x0f\x6e\x40\x0f\xe5\xd0\x0e\xf0\x00\x00\x00\x71\x20\x00\x00\x12\x00\x00\x00\x46\x20\x0d\x97\xef\x3c\xbe\x10"
        "\x11\xc0\x44\x84\x40\x33\x2c\x84\x05\x4c\xc3\xe5\x3b\x8f\xbf\x38\xc0\x20\x36\x0f\x35\xf9\xc5\x6d\xdb\x00\x34\x5c\xbe\xf3\xf8\x12\xc0\x3c\x0b\xe1\x17\xb7\x6d\x02"
        "\xd5\x70\xf9\xce\xe3\x4b\x93\x13\x11\x28\x35\x3d\xd4\xe4\x17\xb7\x6d\x00\x04\x03\x20\x0d\x00\x00\x00\x00\x00\x00\x48\x41\x53\x48\x14\x00\x00\x00\x00\x00\x00\x00"
        "\x04\x6a\x7d\xaa\x8e\x43\xa0\xfc\x9e\x75\x54\x30\x4d\x5d\x1b\x34\x44\x58\x49\x4c\xc0\x06\x00\x00\x60\x00\x00\x00\xb0\x01\x00\x00\x44\x58\x49\x4c\x00\x01\x00\x00"
        "\x10\x00\x00\x00\xa8\x06\x00\x00\x42\x43\xc0\xde\x21\x0c\x00\x00\xa7\x01\x00\x00\x0b\x82\x20\x00\x02\x00\x00\x00\x13\x00\x00\x00\x07\x81\x23\x91\x41\xc8\x04\x49"
        "\x06\x10\x32\x39\x92\x01\x84\x0c\x25\x05\x08\x19\x1e\x04\x8b\x62\x80\x14\x45\x02\x42\x92\x0b\x42\xa4\x10\x32\x14\x38\x08\x18\x4b\x0a\x32\x52\x88\x48\x90\x14\x20"
        "\x43\x46\x88\xa5\x00\x19\x32\x42\xe4\x48\x0e\x90\x91\x22\xc4\x50\x41\x51\x81\x8c\xe1\x83\xe5\x8a\x04\x29\x46\x06\x51\x18\x00\x00\x08\x00\x00\x00\x1b\x8c\xe0\xff"
        "\xff\xff\xff\x07\x40\x02\xa8\x0d\x84\xf0\xff\xff\xff\xff\x03\x20\x6d\x30\x86\xff\xff\xff\xff\x1f\x00\x09\xa8\x00\x49\x18\x00\x00\x03\x00\x00\x00\x13\x82\x60\x42"
        "\x20\x4c\x08\x06\x00\x00\x00\x00\x89\x20\x00\x00\x43\x00\x00\x00\x32\x22\x48\x09\x20\x64\x85\x04\x93\x22\xa4\x84\x04\x93\x22\xe3\x84\xa1\x90\x14\x12\x4c\x8a\x8c"
        "\x0b\x84\xa4\x4c\x10\x68\x23\x00\x25\x00\x14\x66\x00\xe6\x08\xc0\x60\x8e\x00\x29\xc6\x20\x84\x14\x42\xa6\x18\x80\x10\x52\x06\xa1\x9b\x86\xcb\x9f\xb0\x87\x90\xfc"
        "\x95\x90\x56\x62\xf2\x8b\xdb\x46\xc5\x18\x63\x10\x2a\xf7\x0c\x97\x3f\x61\x0f\x21\xf9\x21\xd0\x0c\x0b\x81\x82\x55\x18\x45\x18\x1b\x63\x0c\x42\xc8\xa0\x36\x47\x10"
        "\x14\x83\x91\x42\xc8\x23\x38\x10\x30\x8c\x40\x0c\x33\xb5\xc1\x38\xb0\x43\x38\xcc\xc3\x3c\xb8\x01\x2d\x94\x03\x3e\xd0\x43\x3d\xc8\x43\x39\xc8\x01\x29\xf0\x81\x3d"
        "\x94\xc3\x38\xd0\xc3\x3b\xc8\x03\x1f\x98\x03\x3b\xbc\x43\x38\xd0\x03\x1b\x80\x01\x1d\xf8\x01\x18\xf8\x81\x1e\xe8\x41\x3b\xa4\x03\x3c\xcc\xc3\x2f\xd0\x43\x3e\xc0"
        "\x43\x39\xa0\x80\xcc\x24\x06\xe3\xc0\x0e\xe1\x30\x0f\xf3\xe0\x06\xb4\x50\x0e\xf8\x40\x0f\xf5\x20\x0f\xe5\x20\x07\xa4\xc0\x07\xf6\x50\x0e\xe3\x40\x0f\xef\x20\x0f"
        "\x7c\x60\x0e\xec\xf0\x0e\xe1\x40\x0f\x6c\x00\x06\x74\xe0\x07\x60\xe0\x07\x48\x98\x94\xea\x4d\xd2\x14\x51\xc2\xe4\xb3\x00\xf3\x2c\x44\xc4\x4e\xc0\x44\xa0\x80\xd0"
        "\x4d\x04\x02\x00\x13\x14\x72\xc0\x87\x74\x60\x87\x36\x68\x87\x79\x68\x03\x72\xc0\x87\x0d\xaf\x50\x0e\x6d\xd0\x0e\x7a\x50\x0e\x6d\x00\x0f\x7a\x30\x07\x72\xa0\x07"
        "\x73\x20\x07\x6d\x90\x0e\x71\xa0\x07\x73\x20\x07\x6d\x90\x0e\x78\xa0\x07\x73\x20\x07\x6d\x90\x0e\x71\x60\x07\x7a\x30\x07\x72\xd0\x06\xe9\x30\x07\x72\xa0\x07\x73"
        "\x20\x07\x6d\x90\x0e\x76\x40\x07\x7a\x60\x07\x74\xd0\x06\xe6\x10\x07\x76\xa0\x07\x73\x20\x07\x6d\x60\x0e\x73\x20\x07\x7a\x30\x07\x72\xd0\x06\xe6\x60\x07\x74\xa0"
        "\x07\x76\x40\x07\x6d\xe0\x0e\x78\xa0\x07\x71\x60\x07\x7a\x30\x07\x72\xa0\x07\x76\x40\x07\x43\x9e\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x86\x3c\x06\x10\x00"
        "\x01\x00\x00\x00\x00\x00\x00\x00\x0c\x79\x10\x20\x00\x04\x00\x00\x00\x00\x00\x00\x00\x18\xf2\x34\x40\x00\x0c\x00\x00\x00\x00\x00\x00\x00\x30\xe4\x81\x80\x00\x18"
        "\x00\x00\x00\x00\x00\x00\x00\x20\x0b\x04\x00\x00\x0d\x00\x00\x00\x32\x1e\x98\x14\x19\x11\x4c\x90\x8c\x09\x26\x47\xc6\x04\x43\x22\x25\x30\x02\x50\x12\xc5\x50\x04"
        "\x65\x50\x1e\x54\x4a\x62\x04\xa0\x08\x0a\xa1\x40\xc8\xce\x00\x10\x9e\x01\xa0\x3c\x16\x62\x10\x81\x40\x20\xcf\x03\x79\x18\x00\x00\x5a\x00\x00\x00\x1a\x03\x4c\x90"
        "\x46\x02\x13\x44\x35\x20\xc3\x1b\x43\x81\x93\x4b\xb3\x0b\xa3\x2b\x4b\x01\x89\x71\xc1\x71\x81\x71\x89\xa1\xc1\xc1\xa1\x01\x41\x69\x0b\x4b\x73\x63\x01\x81\x31\x33"
        "\x23\xb3\x81\x09\x1b\x1b\x4b\xd9\x10\x04\x13\x04\xa2\x98\x20\x10\xc6\x06\x61\x20\x26\x08\xc4\xb1\x41\x18\x0c\x0a\x70\x73\x13\x04\x02\xd9\x30\x20\x09\x31\x41\xb0"
        "\x22\x02\x13\x04\x22\xd9\x20\x10\xc6\x86\x84\x58\x98\x86\x18\x1a\xc2\xd9\x10\x3c\x13\x04\x4c\xda\x80\x10\x11\xd3\x10\x03\x01\x6c\x08\xa4\x0d\x04\x04\x00\xd3\x04"
        "\x21\x9b\x36\x04\xd5\x04\x41\x00\x48\xb4\x85\xa5\xb9\x11\xa1\x2a\xc2\x1a\x7a\x7a\x92\x22\x9a\x20\x14\xcc\x04\xa1\x68\x36\x04\xc4\x04\xa1\x70\x26\x08\xc5\x33\x41"
        "\x20\x94\x09\x02\xb1\x6c\x10\x3e\x30\xd8\xb0\x10\x99\xb6\x71\xdd\xe0\x11\x5b\x18\x6c\x08\x86\x0d\xc2\xf7\x6d\x58\x86\x4c\xdb\xc6\xa0\x1b\xba\x61\x23\x83\x0d\x82"
        "\x18\x94\x01\x93\x29\xab\x2f\xaa\x30\xb9\xb3\x32\xba\x09\x42\x01\x6d\x58\x88\x33\xd0\xd0\x80\xdb\x06\x8f\xd8\xc2\x60\x43\x90\x06\x1b\x06\x33\x50\x03\x60\x43\x71"
        "\x61\x6b\x40\x01\x55\xd8\xd8\xec\xda\x5c\xd2\xc8\xca\xdc\xe8\xa6\x04\x41\x15\x32\x3c\x17\xbb\x32\xb9\xb9\xb4\x37\xb7\x29\x01\xd1\x84\x0c\xcf\xc5\x2e\x8c\xcd\xae"
        "\x4c\x6e\x4a\x60\xd4\x21\xc3\x73\x99\x43\x0b\x23\x2b\x93\x6b\x7a\x23\x2b\x63\x9b\x12\x24\x65\xc8\xf0\x5c\xe4\xca\xe6\xde\xea\xe4\xc6\xca\xe6\xa6\x04\x53\x1d\x32"
        "\x3c\x17\xbb\xb4\xb2\xbb\x24\xb2\x29\xba\x30\xba\xb2\x29\x41\x55\x87\x0c\xcf\xa5\xcc\x8d\x4e\x2e\x0f\xea\x2d\xcd\x8d\x6e\x6e\x4a\xb0\x06\x00\x00\x79\x18\x00\x00"
        "\x4c\x00\x00\x00\x33\x08\x80\x1c\xc4\xe1\x1c\x66\x14\x01\x3d\x88\x43\x38\x84\xc3\x8c\x42\x80\x07\x79\x78\x07\x73\x98\x71\x0c\xe6\x00\x0f\xed\x10\x0e\xf4\x80\x0e"
        "\x33\x0c\x42\x1e\xc2\xc1\x1d\xce\xa1\x1c\x66\x30\x05\x3d\x88\x43\x38\x84\x83\x1b\xcc\x03\x3d\xc8\x43\x3d\x8c\x03\x3d\xcc\x78\x8c\x74\x70\x07\x7b\x08\x07\x79\x48"
        "\x87\x70\x70\x07\x7a\x70\x03\x76\x78\x87\x70\x20\x87\x19\xcc\x11\x0e\xec\x90\x0e\xe1\x30\x0f\x6e\x30\x0f\xe3\xf0\x0e\xf0\x50\x0e\x33\x10\xc4\x1d\xde\x21\x1c\xd8"
        "\x21\x1d\xc2\x61\x1e\x66\x30\x89\x3b\xbc\x83\x3b\xd0\x43\x39\xb4\x03\x3c\xbc\x83\x3c\x84\x03\x3b\xcc\xf0\x14\x76\x60\x07\x7b\x68\x07\x37\x68\x87\x72\x68\x07\x37"
        "\x80\x87\x70\x90\x87\x70\x60\x07\x76\x28\x07\x76\xf8\x05\x76\x78\x87\x77\x80\x87\x5f\x08\x87\x71\x18\x87\x72\x98\x87\x79\x98\x81\x2c\xee\xf0\x0e\xee\xe0\x0e\xf5"
        "\xc0\x0e\xec\x30\x03\x62\xc8\xa1\x1c\xe4\xa1\x1c\xcc\xa1\x1c\xe4\xa1\x1c\xdc\x61\x1c\xca\x21\x1c\xc4\x81\x1d\xca\x61\x06\xd6\x90\x43\x39\xc8\x43\x39\x98\x43\x39"
        "\xc8\x43\x39\xb8\xc3\x38\x94\x43\x38\x88\x03\x3b\x94\xc3\x2f\xbc\x83\x3c\xfc\x82\x3b\xd4\x03\x3b\xb0\xc3\x8c\xc8\x21\x07\x7c\x70\x03\x72\x10\x87\x73\x70\x03\x7b"
        "\x08\x07\x79\x60\x87\x70\xc8\x87\x77\xa8\x07\x7a\x98\x81\x3c\xe4\x80\x0f\x6e\x40\x0f\xe5\xd0\x0e\xf0\x00\x00\x00\x71\x20\x00\x00\x12\x00\x00\x00\x46\x20\x0d\x97"
        "\xef\x3c\xbe\x10\x11\xc0\x44\x84\x40\x33\x2c\x84\x05\x4c\xc3\xe5\x3b\x8f\xbf\x38\xc0\x20\x36\x0f\x35\xf9\xc5\x6d\xdb\x00\x34\x5c\xbe\xf3\xf8\x12\xc0\x3c\x0b\xe1"
        "\x17\xb7\x6d\x02\xd5\x70\xf9\xce\xe3\x4b\x93\x13\x11\x28\x35\x3d\xd4\xe4\x17\xb7\x6d\x00\x04\x03\x20\x0d\x00\x00\x61\x20\x00\x00\x3a\x00\x00\x00\x13\x04\x41\x2c"
        "\x10\x00\x00\x00\x05\x00\x00\x00\xf4\x46\x00\x88\xcc\x00\x14\x42\x29\x94\x5c\xe1\x51\x29\x83\x12\xa0\x31\x03\x00\x23\x06\x09\x00\x82\x60\x00\x65\x05\x74\x5d\xc9"
        "\x88\x41\x02\x80\x20\x18\x40\x9a\x41\x60\x98\x32\x62\x90\x00\x20\x08\x06\x86\x97\x68\x99\xa4\x8c\x18\x24\x00\x08\x82\x81\xf1\x29\x9b\x56\x2c\x23\x06\x09\x00\x82"
        "\x60\x60\x80\xc1\xb2\x6d\x14\x33\x62\x90\x00\x20\x08\x06\x46\x18\x30\x1c\x77\x34\x23\x06\x09\x00\x82\x60\x60\x88\x41\xd3\x75\x95\x33\x62\x90\x00\x20\x08\x06\xc6"
        "\x18\x38\x9e\xa7\x3c\x23\x06\x0f\x00\x82\x60\xd0\x88\x01\x83\x1c\x46\x91\x24\xdf\x07\x25\xa3\x09\x01\x30\x9a\x20\x04\xa3\x09\x83\x30\x9a\x40\x0c\x46\x24\xf2\x31"
        "\x22\x91\x8f\x11\x89\x7c\x8c\x48\xe4\x33\x62\x90\x00\x20\x08\x06\x08\x1b\x5c\x68\x80\x06\x61\x40\x8c\x18\x24\x00\x08\x82\x01\xc2\x06\x17\x1a\xa0\xc1\x34\x8c\x18"
        "\x24\x00\x08\x82\x01\xc2\x06\x17\x1a\xa0\x01\x18\x08\x23\x06\x09\x00\x82\x60\x80\xb0\xc1\x85\x06\x68\x40\x05\x08\x00\x00\x00\x00";

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    static bool compareClipRects(const SDL_Rect& rect1, const SDL_Rect& rect2)
    {
        return (rect1.x == rect2.x) && (rect1.y == rect2.y)  && (rect1.w == rect2.w)  && (rect1.h == rect2.h);
    };

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    BackendRenderTargetSDLGPU::BackendRenderTargetSDLGPU(SDL_GPUDevice* device, SDL_GPUTextureFormat swapchainTextureFormat) :
        m_device{device}
    {
        SDL_GPUShaderFormat backendFormats = SDL_GetGPUShaderFormats(device);

        SDL_GPUShaderCreateInfo shaderInfoVertex = {};
        shaderInfoVertex.stage = SDL_GPU_SHADERSTAGE_VERTEX;
        shaderInfoVertex.num_samplers = 0;
        shaderInfoVertex.num_uniform_buffers = 0;
        shaderInfoVertex.num_storage_buffers = 0;
        shaderInfoVertex.num_storage_textures = 0;

        SDL_GPUShaderCreateInfo shaderInfoFragment = {};
        shaderInfoFragment.stage = SDL_GPU_SHADERSTAGE_FRAGMENT;
        shaderInfoFragment.num_samplers = 1;
        shaderInfoFragment.num_uniform_buffers = 0;
        shaderInfoFragment.num_storage_buffers = 0;
        shaderInfoFragment.num_storage_textures = 0;

        if (backendFormats & SDL_GPU_SHADERFORMAT_SPIRV)
        {
            shaderInfoVertex.code = vertexShaderSPV;
            shaderInfoVertex.code_size = sizeof(vertexShaderSPV)-1;
            shaderInfoVertex.entrypoint = "main";
            shaderInfoVertex.format = SDL_GPU_SHADERFORMAT_SPIRV;

            shaderInfoFragment.code = fragmentShaderSPV;
            shaderInfoFragment.code_size = sizeof(fragmentShaderSPV)-1;
            shaderInfoFragment.entrypoint = "main";
            shaderInfoFragment.format = SDL_GPU_SHADERFORMAT_SPIRV;
        }
        else if (backendFormats & SDL_GPU_SHADERFORMAT_MSL)
        {
            shaderInfoVertex.code = vertexShaderMSL;
            shaderInfoVertex.code_size = sizeof(vertexShaderMSL)-1;
            shaderInfoVertex.entrypoint = "main0";
            shaderInfoVertex.format = SDL_GPU_SHADERFORMAT_MSL;

            shaderInfoFragment.code = fragmentShaderMSL;
            shaderInfoFragment.code_size = sizeof(fragmentShaderMSL)-1;
            shaderInfoFragment.entrypoint = "main0";
            shaderInfoFragment.format = SDL_GPU_SHADERFORMAT_MSL;
        }
        else if (backendFormats & SDL_GPU_SHADERFORMAT_DXIL)
        {
            shaderInfoVertex.code = vertexShaderDXIL;
            shaderInfoVertex.code_size = sizeof(vertexShaderDXIL)-1;
            shaderInfoVertex.entrypoint = "main";
            shaderInfoVertex.format = SDL_GPU_SHADERFORMAT_DXIL;

            shaderInfoFragment.code = fragmentShaderDXIL;
            shaderInfoFragment.code_size = sizeof(fragmentShaderDXIL)-1;
            shaderInfoFragment.entrypoint = "main";
            shaderInfoFragment.format = SDL_GPU_SHADERFORMAT_DXIL;
        }
        else
            throw Exception("Creating BackendRenderTargetSDLGPU failed: no supported shader format found");

        SDL_GPUShader* vertexShader = SDL_CreateGPUShader(device, &shaderInfoVertex);
        if (!vertexShader)
            throw Exception("Creating BackendRenderTargetSDLGPU failed: failed to create vertex shader");

        SDL_GPUShader* fragmentShader = SDL_CreateGPUShader(device, &shaderInfoFragment);
        if (!fragmentShader)
            throw Exception("Creating BackendRenderTargetSDLGPU failed: failed to create fragment shader");

        // Position is stored as x,y in the first 2 floats
        // Color is stored as r,g,b,a in the next 4 bytes
        // Texture coordinate is stored as u,v in the last 2 floats
        static_assert(sizeof(Vertex) == 8 + 4 + 8, "Size of tgui::Vertex has to match the SDL_GPU backend shader input");
        static_assert(sizeof(unsigned int) == 4, "Indices are stored as 32bit values");

        SDL_GPUColorTargetBlendState blendState = {};
        blendState.enable_blend = true;
        blendState.src_color_blendfactor = SDL_GPU_BLENDFACTOR_SRC_ALPHA;
        blendState.dst_color_blendfactor = SDL_GPU_BLENDFACTOR_ONE_MINUS_SRC_ALPHA;
        blendState.color_blend_op = SDL_GPU_BLENDOP_ADD;
        blendState.src_alpha_blendfactor = SDL_GPU_BLENDFACTOR_ONE;
        blendState.dst_alpha_blendfactor = SDL_GPU_BLENDFACTOR_ONE_MINUS_SRC_ALPHA;
        blendState.alpha_blend_op = SDL_GPU_BLENDOP_ADD;

        std::array<SDL_GPUColorTargetDescription, 1> colorTargetDescriptions = {{ {} }};
        colorTargetDescriptions[0].format = swapchainTextureFormat;
        colorTargetDescriptions[0].blend_state = blendState;

        std::array<SDL_GPUVertexBufferDescription, 1> vertexBuffers = {{ {} }};
        vertexBuffers[0].slot = 0;
        vertexBuffers[0].input_rate = SDL_GPU_VERTEXINPUTRATE_VERTEX;
        vertexBuffers[0].instance_step_rate = 0;
        vertexBuffers[0].pitch = sizeof(Vertex);

        std::array<SDL_GPUVertexAttribute, 3> vertexAttributes = {{ {}, {}, {} }};
        vertexAttributes[0].buffer_slot = 0;
        vertexAttributes[0].format = SDL_GPU_VERTEXELEMENTFORMAT_FLOAT2;
        vertexAttributes[0].location = 0;
        vertexAttributes[0].offset = 0;
        vertexAttributes[1].buffer_slot = 0;
        vertexAttributes[1].format = SDL_GPU_VERTEXELEMENTFORMAT_UBYTE4_NORM;
        vertexAttributes[1].location = 1;
        vertexAttributes[1].offset = sizeof(float) * 2;
        vertexAttributes[2].buffer_slot = 0;
        vertexAttributes[2].format = SDL_GPU_VERTEXELEMENTFORMAT_FLOAT2;
        vertexAttributes[2].location = 2;
        vertexAttributes[2].offset = sizeof(float) * 2 + sizeof(std::uint8_t) * 4;

        SDL_GPUGraphicsPipelineCreateInfo pipelineCreateInfo = {};
        pipelineCreateInfo.vertex_shader = vertexShader;
        pipelineCreateInfo.fragment_shader = fragmentShader;
        pipelineCreateInfo.vertex_input_state.vertex_buffer_descriptions = vertexBuffers.data();
        pipelineCreateInfo.vertex_input_state.num_vertex_buffers = static_cast<std::uint32_t>(vertexBuffers.size());
        pipelineCreateInfo.vertex_input_state.vertex_attributes = vertexAttributes.data();
        pipelineCreateInfo.vertex_input_state.num_vertex_attributes = static_cast<std::uint32_t>(vertexAttributes.size());
        pipelineCreateInfo.primitive_type = SDL_GPU_PRIMITIVETYPE_TRIANGLELIST;
        pipelineCreateInfo.target_info.has_depth_stencil_target = false;
        pipelineCreateInfo.target_info.color_target_descriptions = colorTargetDescriptions.data();
        pipelineCreateInfo.target_info.num_color_targets = static_cast<std::uint32_t>(colorTargetDescriptions.size());

        m_pipeline = SDL_CreateGPUGraphicsPipeline(device, &pipelineCreateInfo);
        if (!m_pipeline)
            throw Exception("Creating BackendRenderTargetSDLGPU failed: failed to create the pipeline");

        SDL_ReleaseGPUShader(device, vertexShader);
        SDL_ReleaseGPUShader(device, fragmentShader);

        SDL_GPUSamplerCreateInfo samplerInfoNearest = {};
        samplerInfoNearest.min_filter = SDL_GPU_FILTER_NEAREST;
        samplerInfoNearest.mag_filter = SDL_GPU_FILTER_NEAREST;
        samplerInfoNearest.mipmap_mode = SDL_GPU_SAMPLERMIPMAPMODE_NEAREST;
        samplerInfoNearest.address_mode_u = SDL_GPU_SAMPLERADDRESSMODE_REPEAT;
        samplerInfoNearest.address_mode_v = SDL_GPU_SAMPLERADDRESSMODE_REPEAT;
        samplerInfoNearest.address_mode_w = SDL_GPU_SAMPLERADDRESSMODE_REPEAT;
        m_samplerNearest = SDL_CreateGPUSampler(m_device, &samplerInfoNearest);
        if (!m_samplerNearest)
            throw Exception("Creating BackendRenderTargetSDLGPU failed: failed to create sampler");

        SDL_GPUSamplerCreateInfo samplerInfoLinear = samplerInfoNearest;
        samplerInfoLinear.min_filter = SDL_GPU_FILTER_LINEAR;
        samplerInfoLinear.mag_filter = SDL_GPU_FILTER_LINEAR;
        samplerInfoLinear.mipmap_mode = SDL_GPU_SAMPLERMIPMAPMODE_LINEAR;
        m_samplerLinear = SDL_CreateGPUSampler(m_device, &samplerInfoLinear);
        if (!m_samplerLinear)
            throw Exception("Creating BackendRenderTargetSDLGPU failed: failed to create linear sampler");

        // Create a solid white 1x1 texture to pass to the shader when we aren't drawing a texture
        m_emptyTexture = std::make_unique<BackendTextureSDLGPU>(m_device);
        auto pixels = MakeUniqueForOverwrite<std::uint8_t[]>(4); // 4 bytes to store RGBA values
        for (unsigned int i = 0; i < 4; ++i)
            pixels[i] = 255;
        m_emptyTexture->load({1,1}, std::move(pixels), false);
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    BackendRenderTargetSDLGPU::~BackendRenderTargetSDLGPU()
    {
        SDL_ReleaseGPUSampler(m_device, m_samplerLinear);
        SDL_ReleaseGPUSampler(m_device, m_samplerNearest);
        SDL_ReleaseGPUGraphicsPipeline(m_device, m_pipeline);
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BackendRenderTargetSDLGPU::setClearColor(const Color&)
    {
        // This function isn't used by this backend
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BackendRenderTargetSDLGPU::clearScreen()
    {
        // This function isn't used by this backend
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BackendRenderTargetSDLGPU::setView(FloatRect view, FloatRect viewport, Vector2f targetSize)
    {
        BackendRenderTarget::setView(view, viewport, targetSize);

        m_projectionTransform = Transform();
        m_projectionTransform.translate({-1 - (2.f * (view.left / view.width)), 1 + (2.f * (view.top / view.height))});
        m_projectionTransform.scale({2.f / view.width, -2.f / view.height});
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    bool BackendRenderTargetSDLGPU::prepareDrawGui(const std::shared_ptr<RootContainer>& root, SDL_GPUCommandBuffer* cmdBuffer, SDL_GPUCopyPass* copyPass)
    {
        // Buffers should already be empty, but this could have been skipped if prepareDrawGui or drawGui previously aborted with an error
        m_drawCommands.clear();
        m_vertices.clear();
        m_indices.clear();
        if (m_vertexBuffer)
        {
            SDL_ReleaseGPUBuffer(m_device, m_vertexBuffer);
            m_vertexBuffer = nullptr;
        }
        if (m_indexBuffer)
        {
            SDL_ReleaseGPUBuffer(m_device, m_indexBuffer);
            m_indexBuffer = nullptr;
        }

        if (!m_device || (m_targetSize.x == 0) || (m_targetSize.y == 0) || (m_viewRect.width <= 0) || (m_viewRect.height <= 0))
            return false;

        m_pixelsPerPoint = {m_viewport.width / m_viewRect.width, m_viewport.height / m_viewRect.height};
        m_clipRect = {0, 0, static_cast<int>(std::round(m_targetSize.x)), static_cast<int>(std::round(m_targetSize.y))};

        root->draw(*this, {});

        if (m_drawCommands.empty() || m_vertices.empty() || m_indices.empty())
            return true; // Nothing to do

        SDL_GPUBufferCreateInfo vertexBufferCreateInfo = {};
        vertexBufferCreateInfo.usage = SDL_GPU_BUFFERUSAGE_VERTEX;
        vertexBufferCreateInfo.size = static_cast<std::uint32_t>(m_vertices.size() * sizeof(Vertex));
        m_vertexBuffer = SDL_CreateGPUBuffer(m_device, &vertexBufferCreateInfo);
        if (!m_vertexBuffer)
            return false;

        SDL_GPUBufferCreateInfo indexBufferCreateInfo = {};
        indexBufferCreateInfo.usage = SDL_GPU_BUFFERUSAGE_INDEX;
        indexBufferCreateInfo.size = static_cast<std::uint32_t>(m_indices.size() * sizeof(unsigned int));
        m_indexBuffer = SDL_CreateGPUBuffer(m_device, &indexBufferCreateInfo);
        if (!m_indexBuffer)
            return false;

        SDL_GPUTransferBufferCreateInfo transferBufferCreateInfo = {};
        transferBufferCreateInfo.usage = SDL_GPU_TRANSFERBUFFERUSAGE_UPLOAD;
        transferBufferCreateInfo.size = static_cast<std::uint32_t>((m_vertices.size() * sizeof(Vertex)) + (m_indices.size() * sizeof(unsigned int)));
        SDL_GPUTransferBuffer* transferBuffer = SDL_CreateGPUTransferBuffer(m_device, &transferBufferCreateInfo);
        if (!transferBuffer)
            return false;

        void* data = SDL_MapGPUTransferBuffer(m_device, transferBuffer, false);
        if (!data)
        {
            SDL_ReleaseGPUTransferBuffer(m_device, transferBuffer);
            return false;
        }

        auto* vertexData = reinterpret_cast<Vertex*>(data);
        auto* indexData = reinterpret_cast<unsigned int*>(vertexData + m_vertices.size());
        std::memcpy(vertexData, m_vertices.data(), m_vertices.size() * sizeof(Vertex));
        std::memcpy(indexData, m_indices.data(), m_indices.size() * sizeof(unsigned int));
        SDL_UnmapGPUTransferBuffer(m_device, transferBuffer);

        const bool usingExistingCopyPass = (copyPass != nullptr);
        if (!copyPass)
        {
            copyPass = SDL_BeginGPUCopyPass(cmdBuffer);
            if (!copyPass)
            {
                SDL_ReleaseGPUTransferBuffer(m_device, transferBuffer);
                return false;
            }
        }

        SDL_GPUTransferBufferLocation transferBufferVertexLocation = {};
        transferBufferVertexLocation.transfer_buffer = transferBuffer;
        transferBufferVertexLocation.offset = 0;

        SDL_GPUBufferRegion vertexBufferRegion = {};
        vertexBufferRegion.buffer = m_vertexBuffer;
        vertexBufferRegion.offset = 0;
        vertexBufferRegion.size = static_cast<std::uint32_t>(m_vertices.size() * sizeof(Vertex));
        SDL_UploadToGPUBuffer(copyPass, &transferBufferVertexLocation, &vertexBufferRegion, false);

        SDL_GPUTransferBufferLocation transferBufferIndexLocation = {};
        transferBufferIndexLocation.transfer_buffer = transferBuffer;
        transferBufferIndexLocation.offset = static_cast<std::uint32_t>(m_vertices.size() * sizeof(Vertex));

        SDL_GPUBufferRegion indexBufferRegion = {};
        indexBufferRegion.buffer = m_indexBuffer;
        indexBufferRegion.offset = 0;
        indexBufferRegion.size = static_cast<std::uint32_t>(m_indices.size() * sizeof(unsigned int));
        SDL_UploadToGPUBuffer(copyPass, &transferBufferIndexLocation, &indexBufferRegion, false);

        SDL_ReleaseGPUTransferBuffer(m_device, transferBuffer);

        if (!usingExistingCopyPass)
            SDL_EndGPUCopyPass(copyPass);

        m_vertices.clear();
        m_indices.clear();
        return true;
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BackendRenderTargetSDLGPU::drawGui(SDL_GPURenderPass* renderPass)
    {
        if (!m_device || (m_targetSize.x == 0) || (m_targetSize.y == 0) || (m_viewRect.width <= 0) || (m_viewRect.height <= 0))
            return;

        if (m_drawCommands.empty() || !m_vertexBuffer || !m_indexBuffer)
            return;

        const SDL_Rect defaultClipRect = {0, 0, static_cast<int>(std::round(m_targetSize.x)), static_cast<int>(std::round(m_targetSize.y))};

        SDL_GPUBufferBinding vertexBufferBinding = {};
        vertexBufferBinding.buffer = m_vertexBuffer;
        vertexBufferBinding.offset = 0;

        SDL_GPUBufferBinding indexBufferBinding = {};
        indexBufferBinding.buffer = m_indexBuffer;
        indexBufferBinding.offset = 0;

        SDL_BindGPUGraphicsPipeline(renderPass, m_pipeline);
        SDL_BindGPUVertexBuffers(renderPass, 0, &vertexBufferBinding, 1);
        SDL_BindGPUIndexBuffer(renderPass, &indexBufferBinding, SDL_GPU_INDEXELEMENTSIZE_32BIT);
        SDL_SetGPUScissor(renderPass, &defaultClipRect);

        SDL_GPUViewport viewport = {m_viewport.left, m_viewport.top, m_viewport.width, m_viewport.height, 0, 0};
        SDL_SetGPUViewport(renderPass, &viewport);

        SDL_GPUTextureSamplerBinding samplerBinding = {};
        samplerBinding.texture = m_emptyTexture->getInternalTexture();
        samplerBinding.sampler = m_samplerNearest;
        SDL_BindGPUFragmentSamplers(renderPass, 0, &samplerBinding, 1);

        std::size_t indexOffset = 0;
        std::shared_ptr<BackendTexture> lastTexture;
        SDL_Rect lastClipRect = defaultClipRect;
        for (const DrawCommand& drawCommand : m_drawCommands)
        {
            if (lastTexture != drawCommand.texture)
            {
                if (drawCommand.texture)
                {
                    TGUI_ASSERT(std::dynamic_pointer_cast<BackendTextureSDLGPU>(drawCommand.texture), "BackendRenderTargetSDLGPU requires textures of type BackendTextureSDLGPU");
                    const auto texture = std::static_pointer_cast<BackendTextureSDLGPU>(drawCommand.texture);

                    samplerBinding.texture = texture->getInternalTexture();
                    samplerBinding.sampler = texture->isSmooth() ? m_samplerLinear : m_samplerNearest;
                }
                else
                {
                    samplerBinding.texture = m_emptyTexture->getInternalTexture();
                    samplerBinding.sampler = m_samplerNearest;
                }

                SDL_BindGPUFragmentSamplers(renderPass, 0, &samplerBinding, 1);
                lastTexture = drawCommand.texture;
            }

            if (!compareClipRects(lastClipRect, drawCommand.clipRect))
            {
                SDL_SetGPUScissor(renderPass, &drawCommand.clipRect);
                lastClipRect = drawCommand.clipRect;
            }

            SDL_DrawGPUIndexedPrimitives(renderPass, static_cast<std::uint32_t>(drawCommand.indexCount), 1, static_cast<std::uint32_t>(indexOffset), 0, 0);

            indexOffset += drawCommand.indexCount;
        }

        if (!compareClipRects(lastClipRect, defaultClipRect))
            SDL_SetGPUScissor(renderPass, &defaultClipRect);

        SDL_ReleaseGPUBuffer(m_device, m_vertexBuffer);
        SDL_ReleaseGPUBuffer(m_device, m_indexBuffer);

        m_vertexBuffer = nullptr;
        m_indexBuffer = nullptr;

        m_drawCommands.clear();
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BackendRenderTargetSDLGPU::drawVertexArray(const RenderStates& states, const Vertex* vertices,
        std::size_t vertexCount, const unsigned int* indices, std::size_t indexCount, const std::shared_ptr<BackendTexture>& texture)
    {
        if (vertexCount == 0)
            return;

        const std::size_t exisingVerticesCount = m_vertices.size();
        const std::size_t exisingIndicesCount = m_indices.size();
        const Transform finalTransform = m_projectionTransform * states.transform;

        m_vertices.insert(m_vertices.end(), vertices, vertices + vertexCount);
        std::for_each(m_vertices.begin() + static_cast<std::ptrdiff_t>(exisingVerticesCount), m_vertices.end(),
                      [&finalTransform](Vertex& vertex) { vertex.position = finalTransform.transformPoint(vertex.position); });

        if (indices)
        {
            m_indices.insert(m_indices.end(), indices, indices + indexCount);
            std::for_each(m_indices.begin() + static_cast<std::ptrdiff_t>(exisingIndicesCount), m_indices.end(),
                          [exisingVerticesCount](unsigned int& index) { index += static_cast<unsigned int>(exisingVerticesCount); });
        }
        else
        {
            m_indices.resize(m_indices.size() + vertexCount);
            std::iota(m_indices.begin() + static_cast<std::ptrdiff_t>(exisingIndicesCount), m_indices.end(), static_cast<unsigned int>(exisingVerticesCount));
            indexCount = vertexCount;
        }

        // Combine draw commands when the texture didn't change
        if (!m_drawCommands.empty() && (m_drawCommands.back().texture == texture) && compareClipRects(m_drawCommands.back().clipRect, m_clipRect))
            m_drawCommands.back().indexCount += indexCount;
        else
            m_drawCommands.push_back({indexCount, texture, m_clipRect});
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BackendRenderTargetSDLGPU::updateClipping(FloatRect clipRect, FloatRect clipViewport)
    {
        if ((clipViewport.width > 0) && (clipViewport.height > 0) && (clipRect.width > 0) && (clipRect.height > 0))
        {
            m_pixelsPerPoint = {clipViewport.width / clipRect.width, clipViewport.height / clipRect.height};
            m_clipRect = {
                static_cast<int>(std::floor(clipViewport.left)),
                static_cast<int>(std::floor(clipViewport.top)),
                static_cast<int>(std::ceil(clipViewport.width)),
                static_cast<int>(std::ceil(clipViewport.height))
            };
        }
        else // Clip the entire window
        {
            m_pixelsPerPoint = {1, 1};
            m_clipRect = {0, 0, 0, 0};
        }
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
