use std::{
    fmt::{self, Debug},
    hash,
    marker::PhantomData,
};

use base_db::{FileId, SourceDatabase};
use salsa::InternKey;
use syntax::Parse;
use triomphe::Arc;
use vfs::VfsPath;

use crate::{
    HirFileId, InFile,
    ast_id::AstIdMap,
    attributes::{AttributeDefId, AttributesWithOwner},
    body::{Body, BodySourceMap, scope::ExprScopes},
    data::{
        FunctionData, GlobalConstantData, GlobalVariableData, OverrideData, StructData,
        TypeAliasData,
    },
    expression_store::{ExpressionSourceMap, ExpressionStore},
    hir_file_id::HirFileIdRepr,
    module_data::{
        Function, GlobalConstant, GlobalVariable, ModuleInfo, ModuleItemId, Override, Struct,
        TypeAlias,
    },
    resolver::Resolver,
};

#[salsa::query_group(DefDatabaseStorage)]
pub trait DefDatabase: InternDatabase + SourceDatabase {
    fn parse_or_resolve(
        &self,
        key: HirFileId,
    ) -> Parse;

    fn get_path(
        &self,
        key: HirFileId,
    ) -> VfsPath;

    fn get_file_id(
        &self,
        key: VfsPath,
    ) -> Result<FileId, ()>;

    fn ast_id_map(
        &self,
        key: HirFileId,
    ) -> Arc<AstIdMap>;

    #[salsa::invoke(ModuleInfo::module_info_query)]
    fn module_info(
        &self,
        key: HirFileId,
    ) -> Arc<ModuleInfo>;

    #[salsa::invoke(Body::body_with_source_map_query)]
    fn body_with_source_map(
        &self,
        key: DefinitionWithBodyId,
    ) -> (Arc<Body>, Arc<BodySourceMap>);

    #[salsa::invoke(Body::body_query)]
    fn body(
        &self,
        key: DefinitionWithBodyId,
    ) -> Arc<Body>;

    #[salsa::invoke(ExprScopes::expression_scopes_query)]
    fn expression_scopes(
        &self,
        key: DefinitionWithBodyId,
    ) -> Arc<ExprScopes>;

    #[salsa::invoke(signature_with_source_map)]
    fn signature_with_source_map(
        &self,
        key: DefinitionWithBodyId,
    ) -> (Arc<ExpressionStore>, Arc<ExpressionSourceMap>);

    #[salsa::invoke(FunctionData::query)]
    fn function_data(
        &self,
        key: FunctionId,
    ) -> (Arc<FunctionData>, Arc<ExpressionSourceMap>);

    #[salsa::invoke(StructData::query)]
    fn struct_data(
        &self,
        key: StructId,
    ) -> (Arc<StructData>, Arc<ExpressionSourceMap>);

    #[salsa::invoke(TypeAliasData::type_alias_data_query)]
    fn type_alias_data(
        &self,
        key: TypeAliasId,
    ) -> (Arc<TypeAliasData>, Arc<ExpressionSourceMap>);

    #[salsa::invoke(GlobalVariableData::global_var_data_query)]
    fn global_var_data(
        &self,
        key: GlobalVariableId,
    ) -> (Arc<GlobalVariableData>, Arc<ExpressionSourceMap>);

    #[salsa::invoke(GlobalConstantData::global_constant_data_query)]
    fn global_constant_data(
        &self,
        key: GlobalConstantId,
    ) -> (Arc<GlobalConstantData>, Arc<ExpressionSourceMap>);

    #[salsa::invoke(OverrideData::override_data_query)]
    fn override_data(
        &self,
        key: OverrideId,
    ) -> (Arc<OverrideData>, Arc<ExpressionSourceMap>);

    #[salsa::invoke(AttributesWithOwner::attrs_query)]
    fn attrs(
        &self,
        key: AttributeDefId,
    ) -> (Arc<AttributesWithOwner>, Arc<ExpressionSourceMap>);
}

fn signature_with_source_map(
    database: &dyn DefDatabase,
    key: DefinitionWithBodyId,
) -> (Arc<ExpressionStore>, Arc<ExpressionSourceMap>) {
    match key {
        DefinitionWithBodyId::Function(id) => {
            let (data, source_map) = database.function_data(id);
            (data.store.clone(), source_map)
        },
        DefinitionWithBodyId::GlobalVariable(id) => {
            let (data, source_map) = database.global_var_data(id);
            (data.store.clone(), source_map)
        },
        DefinitionWithBodyId::GlobalConstant(id) => {
            let (data, source_map) = database.global_constant_data(id);
            (data.store.clone(), source_map)
        },
        DefinitionWithBodyId::Override(id) => {
            let (data, source_map) = database.override_data(id);
            (data.store.clone(), source_map)
        },
    }
}

fn get_path(
    database: &dyn DefDatabase,
    file_id: HirFileId,
) -> VfsPath {
    match file_id.0 {
        HirFileIdRepr::FileId(file_id) => database.file_path(file_id),
    }
}

#[expect(clippy::unnecessary_wraps, reason = "Needed for salsa")]
fn get_file_id(
    database: &dyn DefDatabase,
    path: VfsPath,
) -> Result<FileId, ()> {
    Ok(database.file_id(path))
}

fn parse_or_resolve(
    database: &dyn DefDatabase,
    file_id: HirFileId,
) -> Parse {
    match file_id.0 {
        HirFileIdRepr::FileId(file_id) => database.parse(file_id),
    }
}

fn ast_id_map(
    database: &dyn DefDatabase,
    file_id: HirFileId,
) -> Arc<AstIdMap> {
    let parsed = database.parse_or_resolve(file_id);
    let map = AstIdMap::from_source(&parsed.tree());
    Arc::new(map)
}

#[salsa::query_group(InternDatabaseStorage)]
pub trait InternDatabase: SourceDatabase {
    #[salsa::interned]
    fn intern_function(
        &self,
        location: Location<Function>,
    ) -> FunctionId;
    #[salsa::interned]
    fn intern_global_variable(
        &self,
        location: Location<GlobalVariable>,
    ) -> GlobalVariableId;
    #[salsa::interned]
    fn intern_global_constant(
        &self,
        location: Location<GlobalConstant>,
    ) -> GlobalConstantId;
    #[salsa::interned]
    fn intern_override(
        &self,
        location: Location<Override>,
    ) -> OverrideId;
    #[salsa::interned]
    fn intern_struct(
        &self,
        location: Location<Struct>,
    ) -> StructId;
    #[salsa::interned]
    fn intern_type_alias(
        &self,
        location: Location<TypeAlias>,
    ) -> TypeAliasId;
}

pub type Location<T> = InFile<ModuleItemId<T>>;

pub struct Interned<T>(salsa::InternId, PhantomData<T>);

impl<T> hash::Hash for Interned<T> {
    fn hash<H: hash::Hasher>(
        &self,
        state: &mut H,
    ) {
        self.0.hash(state);
    }
}

impl<T> PartialEq for Interned<T> {
    fn eq(
        &self,
        other: &Self,
    ) -> bool {
        self.0 == other.0
    }
}

impl<T> Eq for Interned<T> {}

impl<T> Clone for Interned<T> {
    fn clone(&self) -> Self {
        *self
    }
}

impl<T> Copy for Interned<T> {}

impl<T> fmt::Debug for Interned<T> {
    fn fmt(
        &self,
        formatter: &mut fmt::Formatter<'_>,
    ) -> fmt::Result {
        formatter.debug_tuple("Interned").field(&self.0).finish()
    }
}

impl<T> InternKey for Interned<T> {
    fn from_intern_id(v: salsa::InternId) -> Self {
        Self(v, PhantomData)
    }

    fn as_intern_id(&self) -> salsa::InternId {
        self.0
    }
}

macro_rules! intern_id {
    ($id:ident, $loc:ty, $lookup:ident) => {
        #[derive(Debug, Clone, Copy, Eq, PartialEq, Hash)]
        pub struct $id(salsa::InternId);
        impl InternKey for $id {
            fn from_intern_id(v: salsa::InternId) -> Self {
                $id(v)
            }

            fn as_intern_id(&self) -> salsa::InternId {
                self.0
            }
        }

        impl Lookup for $id {
            type Data = $loc;

            fn lookup(
                &self,
                database: &dyn DefDatabase,
            ) -> $loc {
                database.$lookup(*self)
            }
        }
    };
}

pub trait Lookup: Sized {
    type Data;
    fn lookup(
        &self,
        database: &dyn DefDatabase,
    ) -> Self::Data;
}

intern_id!(FunctionId, Location<Function>, lookup_intern_function);
intern_id!(
    GlobalVariableId,
    Location<GlobalVariable>,
    lookup_intern_global_variable
);
intern_id!(
    GlobalConstantId,
    Location<GlobalConstant>,
    lookup_intern_global_constant
);
intern_id!(OverrideId, Location<Override>, lookup_intern_override);
intern_id!(StructId, Location<Struct>, lookup_intern_struct);
intern_id!(TypeAliasId, Location<TypeAlias>, lookup_intern_type_alias);

/// Module items with a body.
#[derive(PartialEq, Eq, Hash, Debug, Clone, Copy)]
pub enum DefinitionWithBodyId {
    Function(FunctionId),
    GlobalVariable(GlobalVariableId),
    GlobalConstant(GlobalConstantId),
    Override(OverrideId),
}

impl DefinitionWithBodyId {
    pub fn file_id(
        self,
        database: &dyn DefDatabase,
    ) -> HirFileId {
        match self {
            Self::Function(id) => id.lookup(database).file_id,
            Self::GlobalVariable(id) => id.lookup(database).file_id,
            Self::GlobalConstant(id) => id.lookup(database).file_id,
            Self::Override(id) => id.lookup(database).file_id,
        }
    }

    pub fn resolver(
        self,
        database: &dyn DefDatabase,
    ) -> Resolver {
        let file_id = self.file_id(database);
        let module_info = database.module_info(file_id);
        Resolver::default().push_module_scope(file_id, module_info)
    }
}

/// All module items.
#[derive(PartialEq, Eq, Hash, Debug, Clone, Copy)]
pub enum ModuleDefinitionId {
    Function(FunctionId),
    GlobalVariable(GlobalVariableId),
    GlobalConstant(GlobalConstantId),
    Override(OverrideId),
    Struct(StructId),
    TypeAlias(TypeAliasId),
}

impl ModuleDefinitionId {
    pub fn file_id(
        self,
        database: &dyn DefDatabase,
    ) -> HirFileId {
        match self {
            Self::Function(id) => id.lookup(database).file_id,
            Self::GlobalVariable(id) => id.lookup(database).file_id,
            Self::GlobalConstant(id) => id.lookup(database).file_id,
            Self::Override(id) => id.lookup(database).file_id,
            Self::Struct(id) => id.lookup(database).file_id,
            Self::TypeAlias(id) => id.lookup(database).file_id,
        }
    }

    pub fn resolver(
        self,
        database: &dyn DefDatabase,
    ) -> Resolver {
        let file_id = self.file_id(database);
        let module_info = database.module_info(file_id);
        Resolver::default().push_module_scope(file_id, module_info)
    }
}

impl From<DefinitionWithBodyId> for ModuleDefinitionId {
    fn from(value: DefinitionWithBodyId) -> Self {
        match value {
            DefinitionWithBodyId::Function(id) => Self::Function(id),
            DefinitionWithBodyId::GlobalVariable(id) => Self::GlobalVariable(id),
            DefinitionWithBodyId::GlobalConstant(id) => Self::GlobalConstant(id),
            DefinitionWithBodyId::Override(id) => Self::Override(id),
        }
    }
}
