#![expect(clippy::missing_const_for_fn, reason = "Implementation is TODO")]

use std::{
    fmt, fs, mem,
    path::{Path, PathBuf},
};

use xshell::Shell;

use crate::{
    flags::{self, CodegenType},
    project_root,
};

pub(crate) mod assists_doc_tests;
pub(crate) mod diagnostics_docs;
pub(crate) mod feature_docs;
mod lints;
mod parser_inline_tests;

impl flags::Codegen {
    #[expect(clippy::unnecessary_wraps, reason = "Intended interface")]
    pub(crate) fn run(
        self,
        _sh: &Shell,
    ) -> anyhow::Result<()> {
        match self.rtype.unwrap_or_default() {
            flags::CodegenType::All => {
                assists_doc_tests::generate(self.check);
                parser_inline_tests::generate(self.check);
                feature_docs::generate(self.check);
                diagnostics_docs::generate(self.check);
                // lints::generate(self.check); // disabled upstream, so double check this!
            },
            flags::CodegenType::AssistsDocTests => assists_doc_tests::generate(self.check),
            flags::CodegenType::DiagnosticsDocs => diagnostics_docs::generate(self.check),
            flags::CodegenType::LintDefinitions => lints::generate(self.check),
            flags::CodegenType::ParserTests => parser_inline_tests::generate(self.check),
            flags::CodegenType::FeatureDocs => feature_docs::generate(self.check),
        }
        Ok(())
    }
}

#[derive(Clone)]
pub(crate) struct CommentBlock {
    pub(crate) id: String,
    pub(crate) line: usize,
    pub(crate) contents: Vec<String>,
    is_doc: bool,
}

impl CommentBlock {
    fn extract(
        tag: &str,
        text: &str,
    ) -> Vec<Self> {
        assert!(tag.starts_with(char::is_uppercase));

        let tag = format!("{tag}:");
        let mut blocks = Self::extract_untagged(text);
        blocks.retain_mut(|block| {
            let first = block.contents.remove(0);
            let Some(id) = first.strip_prefix(&tag) else {
                return false;
            };

            assert!(
                !block.is_doc,
                "Use plain (non-doc) comments with tags like {tag}:\n    {first}"
            );

            id.trim().clone_into(&mut block.id);
            true
        });
        blocks
    }

    fn extract_untagged(text: &str) -> Vec<Self> {
        let mut result = Vec::new();

        let lines = text.lines().map(str::trim_start);

        let dummy_block = Self {
            id: String::new(),
            line: 0,
            contents: Vec::new(),
            is_doc: false,
        };
        let mut block = dummy_block.clone();
        for (line_num, line) in lines.enumerate() {
            match line.strip_prefix("//") {
                Some(mut contents) if !contents.starts_with('/') => {
                    if let Some('/' | '!') = contents.chars().next() {
                        contents = &contents[1..];
                        block.is_doc = true;
                    }
                    if contents.starts_with(' ') {
                        contents = &contents[1..];
                    }
                    block.contents.push(contents.to_owned());
                },
                _ => {
                    if !block.contents.is_empty() {
                        let block = mem::replace(&mut block, dummy_block.clone());
                        result.push(block);
                    }
                    block.line = line_num + 2;
                },
            }
        }
        if !block.contents.is_empty() {
            result.push(block);
        }
        result
    }
}

#[derive(Debug)]
pub(crate) struct Location {
    pub(crate) file: PathBuf,
    pub(crate) line: usize,
}

impl fmt::Display for Location {
    fn fmt(
        &self,
        formatter: &mut fmt::Formatter<'_>,
    ) -> fmt::Result {
        let path = self
            .file
            .strip_prefix(project_root())
            .unwrap()
            .display()
            .to_string();
        let path = path.replace('\\', "/");
        let name = self.file.file_name().unwrap();
        write!(
            formatter,
            " [{}](https://github.com/wgsl-analyzer/wgsl-analyzer/blob/master/{path}#L{}) ",
            name.to_str().unwrap(),
            self.line
        )
    }
}

fn add_preamble(
    cg: CodegenType,
    mut text: String,
) -> String {
    let preamble = format!("//! Generated by `cargo xtask codegen {cg}`, do not edit by hand.\n\n");
    text.insert_str(0, &preamble);
    text
}

/// Checks that the `file` has the specified `contents`. If that is not the
/// case, updates the file and then fails the test.
#[expect(clippy::print_stderr, reason = "TODO")]
fn ensure_file_contents(
    cg: CodegenType,
    file: &Path,
    contents: &str,
    check: bool,
) -> bool {
    let contents = normalize_newlines(contents);
    if let Ok(old_contents) = fs::read_to_string(file)
        && normalize_newlines(&old_contents) == contents
    {
        // File is already up to date.
        return false;
    }

    let display_path = file.strip_prefix(project_root()).unwrap_or(file);
    if check {
        panic!(
            "{} was not up-to-date{}",
            file.display(),
            if std::env::var("CI").is_ok() {
                format!(
                    "\n    NOTE: run `cargo xtask codegen {cg}` locally and commit the updated files\n"
                )
            } else {
                String::new()
            }
        );
    } else {
        eprintln!(
            "\n\x1b[31;1merror\x1b[0m: {} was not up-to-date, updating\n",
            display_path.display()
        );

        if let Some(parent) = file.parent() {
            fs::create_dir_all(parent).unwrap();
        }
        fs::write(file, contents).unwrap();
        true
    }
}

fn normalize_newlines(string: &str) -> String {
    string.replace("\r\n", "\n")
}
