/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#pragma once

#include <arrow-glib/version.h>

/**
 * SECTION: version
 * @section_id: version-macros
 * @title: Version related macros
 * @include: arrow-dataset-glib/arrow-dataset-glib.h
 *
 * Apache Arrow Dataset GLib provides macros that can be used by C pre-processor.
 * They are useful to check version related things at compile time.
 */

/**
 * GADATASET_VERSION_MAJOR:
 *
 * The major version.
 *
 * Since: 17.0.0
 */
#define GADATASET_VERSION_MAJOR (23)

/**
 * GADATASET_VERSION_MINOR:
 *
 * The minor version.
 *
 * Since: 17.0.0
 */
#define GADATASET_VERSION_MINOR (0)

/**
 * GADATASET_VERSION_MICRO:
 *
 * The micro version.
 *
 * Since: 17.0.0
 */
#define GADATASET_VERSION_MICRO (0)

/**
 * GADATASET_VERSION_TAG:
 *
 * The version tag. Normally, it's an empty string. It's "SNAPSHOT"
 * for snapshot version.
 *
 * Since: 17.0.0
 */
#define GADATASET_VERSION_TAG   ""

/**
 * GADATASET_VERSION_CHECK:
 * @major: A major version to check for.
 * @minor: A minor version to check for.
 * @micro: A micro version to check for.
 *
 * You can use this macro in C pre-processor.
 *
 * Returns: %TRUE if the compile time Apache Arrow GLib version is the
 *   same as or newer than the passed version, %FALSE otherwise.
 *
 * Since: 17.0.0
 */
#define GADATASET_VERSION_CHECK(major, minor, micro)       \
  (GADATASET_VERSION_MAJOR > (major) ||                    \
   (GADATASET_VERSION_MAJOR == (major) &&                  \
    GADATASET_VERSION_MINOR > (minor)) ||                  \
   (GADATASET_VERSION_MAJOR == (major) &&                  \
    GADATASET_VERSION_MINOR == (minor) &&                  \
    GADATASET_VERSION_MICRO >= (micro)))

/**
 * GADATASET_DISABLE_DEPRECATION_WARNINGS:
 *
 * If this macro is defined, no deprecated warnings are produced.
 *
 * You must define this macro before including the
 * arrow-glib/arrow-glib.h header.
 *
 * Since: 17.0.0
 */

#ifdef GADATASET_DISABLE_DEPRECATION_WARNINGS
#  define GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_FOR(function)
#  define GADATASET_UNAVAILABLE(major, minor)
#else
#  define GADATASET_DEPRECATED G_DEPRECATED
#  define GADATASET_DEPRECATED_FOR(function) G_DEPRECATED_FOR(function)
#  define GADATASET_UNAVAILABLE(major, minor) G_UNAVAILABLE(major, minor)
#endif

/**
 * GADATASET_VERSION_23_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 23.0.0
 */
#define GADATASET_VERSION_23_0 G_ENCODE_VERSION(23, 0)

/**
 * GADATASET_VERSION_22_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 22.0.0
 */
#define GADATASET_VERSION_22_0 G_ENCODE_VERSION(22, 0)

/**
 * GADATASET_VERSION_21_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 21.0.0
 */
#define GADATASET_VERSION_21_0 G_ENCODE_VERSION(21, 0)

/**
 * GADATASET_VERSION_20_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 20.0.0
 */
#define GADATASET_VERSION_20_0 G_ENCODE_VERSION(20, 0)

/**
 * GADATASET_VERSION_19_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 19.0.0
 */
#define GADATASET_VERSION_19_0 G_ENCODE_VERSION(19, 0)

/**
 * GADATASET_VERSION_18_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 18.0.0
 */
#define GADATASET_VERSION_18_0 G_ENCODE_VERSION(18, 0)

/**
 * GADATASET_VERSION_17_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 17.0.0
 */
#define GADATASET_VERSION_17_0 G_ENCODE_VERSION(17, 0)

/**
 * GADATASET_VERSION_16_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 16.0.0
 */
#define GADATASET_VERSION_16_0 G_ENCODE_VERSION(16, 0)

/**
 * GADATASET_VERSION_15_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 15.0.0
 */
#define GADATASET_VERSION_15_0 G_ENCODE_VERSION(15, 0)

/**
 * GADATASET_VERSION_14_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 14.0.0
 */
#define GADATASET_VERSION_14_0 G_ENCODE_VERSION(14, 0)

/**
 * GADATASET_VERSION_13_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 13.0.0
 */
#define GADATASET_VERSION_13_0 G_ENCODE_VERSION(13, 0)

/**
 * GADATASET_VERSION_12_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 12.0.0
 */
#define GADATASET_VERSION_12_0 G_ENCODE_VERSION(12, 0)

/**
 * GADATASET_VERSION_11_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 11.0.0
 */
#define GADATASET_VERSION_11_0 G_ENCODE_VERSION(11, 0)

/**
 * GADATASET_VERSION_10_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 10.0.0
 */
#define GADATASET_VERSION_10_0 G_ENCODE_VERSION(10, 0)

/**
 * GADATASET_VERSION_9_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 9.0.0
 */
#define GADATASET_VERSION_9_0 G_ENCODE_VERSION(9, 0)

/**
 * GADATASET_VERSION_8_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 8.0.0
 */
#define GADATASET_VERSION_8_0 G_ENCODE_VERSION(8, 0)

/**
 * GADATASET_VERSION_7_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 7.0.0
 */
#define GADATASET_VERSION_7_0 G_ENCODE_VERSION(7, 0)

/**
 * GADATASET_VERSION_6_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 6.0.0
 */
#define GADATASET_VERSION_6_0 G_ENCODE_VERSION(6, 0)

/**
 * GADATASET_VERSION_5_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 5.0.0
 */
#define GADATASET_VERSION_5_0 G_ENCODE_VERSION(5, 0)

/**
 * GADATASET_VERSION_4_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 4.0.0
 */
#define GADATASET_VERSION_4_0 G_ENCODE_VERSION(4, 0)

/**
 * GADATASET_VERSION_3_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 3.0.0
 */
#define GADATASET_VERSION_3_0 G_ENCODE_VERSION(3, 0)

/**
 * GADATASET_VERSION_2_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 2.0.0
 */
#define GADATASET_VERSION_2_0 G_ENCODE_VERSION(2, 0)

/**
 * GADATASET_VERSION_1_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.0.0
 */
#define GADATASET_VERSION_1_0 G_ENCODE_VERSION(1, 0)

/**
 * GADATASET_VERSION_0_17:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.17.0
 */
#define GADATASET_VERSION_0_17 G_ENCODE_VERSION(0, 17)

/**
 * GADATASET_VERSION_0_16:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.16.0
 */
#define GADATASET_VERSION_0_16 G_ENCODE_VERSION(0, 16)

/**
 * GADATASET_VERSION_0_15:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.15.0
 */
#define GADATASET_VERSION_0_15 G_ENCODE_VERSION(0, 15)

/**
 * GADATASET_VERSION_0_14:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.14.0
 */
#define GADATASET_VERSION_0_14 G_ENCODE_VERSION(0, 14)

/**
 * GADATASET_VERSION_0_13:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.13.0
 */
#define GADATASET_VERSION_0_13 G_ENCODE_VERSION(0, 13)

/**
 * GADATASET_VERSION_0_12:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.12.0
 */
#define GADATASET_VERSION_0_12 G_ENCODE_VERSION(0, 12)

/**
 * GADATASET_VERSION_0_11:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.11.0
 */
#define GADATASET_VERSION_0_11 G_ENCODE_VERSION(0, 11)

/**
 * GADATASET_VERSION_0_10:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.10.0
 */
#define GADATASET_VERSION_0_10 G_ENCODE_VERSION(0, 10)

/**
 * GADATASET_VERSION_MIN_REQUIRED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GADATASET_VERSION_0_10.
 *
 * If you use any functions that is defined by newer version than
 * %GADATASET_VERSION_MIN_REQUIRED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * arrow-dataset-glib/arrow-dataset-glib.h header.
 *
 * Since: 17.0.0
 */
#ifndef GADATASET_VERSION_MIN_REQUIRED
#  define GADATASET_VERSION_MIN_REQUIRED GARROW_VERSION_MIN_REQUIRED
#endif

/**
 * GADATASET_VERSION_MAX_ALLOWED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GADATASET_VERSION_0_10.
 *
 * If you use any functions that is defined by newer version than
 * %GADATASET_VERSION_MAX_ALLOWED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * arrow-dataset-glib/arrow-dataset-glib.h header.
 *
 * Since: 17.0.0
 */
#ifndef GADATASET_VERSION_MAX_ALLOWED
#  define GADATASET_VERSION_MAX_ALLOWED GARROW_VERSION_MAX_ALLOWED
#endif

#if (defined(_WIN32) || defined(__CYGWIN__)) && defined(_MSC_VER) &&   !defined(GADATASET_STATIC_COMPILATION)
#  define GADATASET_EXPORT __declspec(dllexport)
#  define GADATASET_IMPORT __declspec(dllimport)
#else
#  define GADATASET_EXPORT
#  define GADATASET_IMPORT
#endif

#ifdef GADATASET_COMPILATION
#  define GADATASET_API GADATASET_EXPORT
#else
#  define GADATASET_API GADATASET_IMPORT
#endif

#define GADATASET_EXTERN GADATASET_API extern

#define GADATASET_AVAILABLE_IN_ALL GADATASET_EXTERN

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_23_0
#  define GADATASET_DEPRECATED_IN_23_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_23_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_23_0
#  define GADATASET_DEPRECATED_IN_23_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_23_0
#  define GADATASET_AVAILABLE_IN_23_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(23, 0)
#else
#  define GADATASET_AVAILABLE_IN_23_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_22_0
#  define GADATASET_DEPRECATED_IN_22_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_22_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_22_0
#  define GADATASET_DEPRECATED_IN_22_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_22_0
#  define GADATASET_AVAILABLE_IN_22_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(22, 0)
#else
#  define GADATASET_AVAILABLE_IN_22_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_21_0
#  define GADATASET_DEPRECATED_IN_21_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_21_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_21_0
#  define GADATASET_DEPRECATED_IN_21_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_21_0
#  define GADATASET_AVAILABLE_IN_21_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(21, 0)
#else
#  define GADATASET_AVAILABLE_IN_21_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_20_0
#  define GADATASET_DEPRECATED_IN_20_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_20_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_20_0
#  define GADATASET_DEPRECATED_IN_20_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_20_0
#  define GADATASET_AVAILABLE_IN_20_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(20, 0)
#else
#  define GADATASET_AVAILABLE_IN_20_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_19_0
#  define GADATASET_DEPRECATED_IN_19_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_19_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_19_0
#  define GADATASET_DEPRECATED_IN_19_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_19_0
#  define GADATASET_AVAILABLE_IN_19_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(19, 0)
#else
#  define GADATASET_AVAILABLE_IN_19_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_18_0
#  define GADATASET_DEPRECATED_IN_18_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_18_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_18_0
#  define GADATASET_DEPRECATED_IN_18_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_18_0
#  define GADATASET_AVAILABLE_IN_18_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(18, 0)
#else
#  define GADATASET_AVAILABLE_IN_18_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_17_0
#  define GADATASET_DEPRECATED_IN_17_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_17_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_17_0
#  define GADATASET_DEPRECATED_IN_17_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_17_0
#  define GADATASET_AVAILABLE_IN_17_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(17, 0)
#else
#  define GADATASET_AVAILABLE_IN_17_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_16_0
#  define GADATASET_DEPRECATED_IN_16_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_16_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_16_0
#  define GADATASET_DEPRECATED_IN_16_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_16_0
#  define GADATASET_AVAILABLE_IN_16_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(16, 0)
#else
#  define GADATASET_AVAILABLE_IN_16_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_15_0
#  define GADATASET_DEPRECATED_IN_15_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_15_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_15_0
#  define GADATASET_DEPRECATED_IN_15_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_15_0
#  define GADATASET_AVAILABLE_IN_15_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(15, 0)
#else
#  define GADATASET_AVAILABLE_IN_15_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_14_0
#  define GADATASET_DEPRECATED_IN_14_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_14_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_14_0
#  define GADATASET_DEPRECATED_IN_14_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_14_0
#  define GADATASET_AVAILABLE_IN_14_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(14, 0)
#else
#  define GADATASET_AVAILABLE_IN_14_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_13_0
#  define GADATASET_DEPRECATED_IN_13_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_13_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_13_0
#  define GADATASET_DEPRECATED_IN_13_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_13_0
#  define GADATASET_AVAILABLE_IN_13_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(13, 0)
#else
#  define GADATASET_AVAILABLE_IN_13_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_12_0
#  define GADATASET_DEPRECATED_IN_12_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_12_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_12_0
#  define GADATASET_DEPRECATED_IN_12_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_12_0
#  define GADATASET_AVAILABLE_IN_12_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(12, 0)
#else
#  define GADATASET_AVAILABLE_IN_12_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_11_0
#  define GADATASET_DEPRECATED_IN_11_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_11_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_11_0
#  define GADATASET_DEPRECATED_IN_11_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_11_0
#  define GADATASET_AVAILABLE_IN_11_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(11, 0)
#else
#  define GADATASET_AVAILABLE_IN_11_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_10_0
#  define GADATASET_DEPRECATED_IN_10_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_10_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_10_0
#  define GADATASET_DEPRECATED_IN_10_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_10_0
#  define GADATASET_AVAILABLE_IN_10_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(10, 0)
#else
#  define GADATASET_AVAILABLE_IN_10_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_9_0
#  define GADATASET_DEPRECATED_IN_9_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_9_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_9_0
#  define GADATASET_DEPRECATED_IN_9_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_9_0
#  define GADATASET_AVAILABLE_IN_9_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(9, 0)
#else
#  define GADATASET_AVAILABLE_IN_9_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_8_0
#  define GADATASET_DEPRECATED_IN_8_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_8_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_8_0
#  define GADATASET_DEPRECATED_IN_8_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_8_0
#  define GADATASET_AVAILABLE_IN_8_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(8, 0)
#else
#  define GADATASET_AVAILABLE_IN_8_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_7_0
#  define GADATASET_DEPRECATED_IN_7_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_7_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_7_0
#  define GADATASET_DEPRECATED_IN_7_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_7_0
#  define GADATASET_AVAILABLE_IN_7_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(7, 0)
#else
#  define GADATASET_AVAILABLE_IN_7_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_6_0
#  define GADATASET_DEPRECATED_IN_6_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_6_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_6_0
#  define GADATASET_DEPRECATED_IN_6_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_6_0
#  define GADATASET_AVAILABLE_IN_6_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(6, 0)
#else
#  define GADATASET_AVAILABLE_IN_6_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_5_0
#  define GADATASET_DEPRECATED_IN_5_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_5_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_5_0
#  define GADATASET_DEPRECATED_IN_5_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_5_0
#  define GADATASET_AVAILABLE_IN_5_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(5, 0)
#else
#  define GADATASET_AVAILABLE_IN_5_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_4_0
#  define GADATASET_DEPRECATED_IN_4_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_4_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_4_0
#  define GADATASET_DEPRECATED_IN_4_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_4_0
#  define GADATASET_AVAILABLE_IN_4_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(4, 0)
#else
#  define GADATASET_AVAILABLE_IN_4_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_3_0
#  define GADATASET_DEPRECATED_IN_3_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_3_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_3_0
#  define GADATASET_DEPRECATED_IN_3_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_3_0
#  define GADATASET_AVAILABLE_IN_3_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(3, 0)
#else
#  define GADATASET_AVAILABLE_IN_3_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_2_0
#  define GADATASET_DEPRECATED_IN_2_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_2_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_2_0
#  define GADATASET_DEPRECATED_IN_2_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_2_0
#  define GADATASET_AVAILABLE_IN_2_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(2, 0)
#else
#  define GADATASET_AVAILABLE_IN_2_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_1_0
#  define GADATASET_DEPRECATED_IN_1_0               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_1_0_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_1_0
#  define GADATASET_DEPRECATED_IN_1_0_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_1_0
#  define GADATASET_AVAILABLE_IN_1_0 GADATASET_EXTERN GADATASET_UNAVAILABLE(1, 0)
#else
#  define GADATASET_AVAILABLE_IN_1_0 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_0_17
#  define GADATASET_DEPRECATED_IN_0_17               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_0_17_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_0_17
#  define GADATASET_DEPRECATED_IN_0_17_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_0_17
#  define GADATASET_AVAILABLE_IN_0_17 GADATASET_EXTERN GADATASET_UNAVAILABLE(0, 17)
#else
#  define GADATASET_AVAILABLE_IN_0_17 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_0_16
#  define GADATASET_DEPRECATED_IN_0_16               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_0_16_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_0_16
#  define GADATASET_DEPRECATED_IN_0_16_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_0_16
#  define GADATASET_AVAILABLE_IN_0_16 GADATASET_EXTERN GADATASET_UNAVAILABLE(0, 16)
#else
#  define GADATASET_AVAILABLE_IN_0_16 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_0_15
#  define GADATASET_DEPRECATED_IN_0_15               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_0_15_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_0_15
#  define GADATASET_DEPRECATED_IN_0_15_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_0_15
#  define GADATASET_AVAILABLE_IN_0_15 GADATASET_EXTERN GADATASET_UNAVAILABLE(0, 15)
#else
#  define GADATASET_AVAILABLE_IN_0_15 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_0_14
#  define GADATASET_DEPRECATED_IN_0_14               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_0_14_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_0_14
#  define GADATASET_DEPRECATED_IN_0_14_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_0_14
#  define GADATASET_AVAILABLE_IN_0_14 GADATASET_EXTERN GADATASET_UNAVAILABLE(0, 14)
#else
#  define GADATASET_AVAILABLE_IN_0_14 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_0_13
#  define GADATASET_DEPRECATED_IN_0_13               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_0_13_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_0_13
#  define GADATASET_DEPRECATED_IN_0_13_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_0_13
#  define GADATASET_AVAILABLE_IN_0_13 GADATASET_EXTERN GADATASET_UNAVAILABLE(0, 13)
#else
#  define GADATASET_AVAILABLE_IN_0_13 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_0_12
#  define GADATASET_DEPRECATED_IN_0_12               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_0_12_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_0_12
#  define GADATASET_DEPRECATED_IN_0_12_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_0_12
#  define GADATASET_AVAILABLE_IN_0_12 GADATASET_EXTERN GADATASET_UNAVAILABLE(0, 12)
#else
#  define GADATASET_AVAILABLE_IN_0_12 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_0_11
#  define GADATASET_DEPRECATED_IN_0_11               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_0_11_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_0_11
#  define GADATASET_DEPRECATED_IN_0_11_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_0_11
#  define GADATASET_AVAILABLE_IN_0_11 GADATASET_EXTERN GADATASET_UNAVAILABLE(0, 11)
#else
#  define GADATASET_AVAILABLE_IN_0_11 GADATASET_EXTERN
#endif

#if GADATASET_VERSION_MIN_REQUIRED >= GADATASET_VERSION_0_10
#  define GADATASET_DEPRECATED_IN_0_10               GADATASET_DEPRECATED
#  define GADATASET_DEPRECATED_IN_0_10_FOR(function) GADATASET_DEPRECATED_FOR(function)
#else
#  define GADATASET_DEPRECATED_IN_0_10
#  define GADATASET_DEPRECATED_IN_0_10_FOR(function)
#endif

#if GADATASET_VERSION_MAX_ALLOWED < GADATASET_VERSION_0_10
#  define GADATASET_AVAILABLE_IN_0_10 GADATASET_EXTERN GADATASET_UNAVAILABLE(0, 10)
#else
#  define GADATASET_AVAILABLE_IN_0_10 GADATASET_EXTERN
#endif
