










package com.adacore.liblktlang;
import com.adacore.langkit_support.LangkitSupport;

import java.util.Arrays;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

import java.lang.StringBuilder;
import java.lang.Iterable;
import java.lang.reflect.Method;

import java.math.BigInteger;

import java.io.File;
import java.nio.ByteOrder;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import com.oracle.truffle.api.CompilerDirectives;

import org.graalvm.nativeimage.CurrentIsolate;
import org.graalvm.nativeimage.ImageInfo;
import org.graalvm.nativeimage.IsolateThread;
import org.graalvm.nativeimage.StackValue;
import org.graalvm.nativeimage.UnmanagedMemory;
import org.graalvm.nativeimage.c.CContext;
import org.graalvm.nativeimage.c.function.CEntryPoint;
import org.graalvm.nativeimage.c.function.CEntryPointLiteral;
import org.graalvm.nativeimage.c.function.CFunction;
import org.graalvm.nativeimage.c.function.CFunctionPointer;
import org.graalvm.nativeimage.c.function.InvokeCFunctionPointer;
import org.graalvm.nativeimage.c.struct.*;
import org.graalvm.nativeimage.c.type.*;
import org.graalvm.word.PointerBase;
import org.graalvm.word.Pointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordFactory;


/*
====================

This is the Java bindings of the liblktlang API.
You can use all functionalities of the library in a Java environment.
Those bindings call the native library using JNI and Native-Image C API.

====================
*/
public final class Liblktlang {

    // ==========
    // Native entry points
    // ==========

    /**
     * This method is the only valid callback to pass to a native event
     * handler for unit requests events.
     * This method will dispatch the execution according to the passed
     * analysis context.
     */
    @CEntryPoint
    static void unitRequested(
        final IsolateThread thread,
        final AnalysisContextNative contextNative,
        final TextNative nameNative,
        final AnalysisUnitNative fromNative,
        final byte foundNative,
        final byte isNotFoundErrorNative
    ) {
        try(
            final AnalysisContext context = AnalysisContext.wrap(
                contextNative
            );
            final Text text = Text.wrap(nameNative)
        ) {
            // Get the callback from the context event handler
            final EventHandler.UnitRequestedCallback callback = context
                .getEventHandler()
                .getUnitRequestCallback();

            // Call the callback
            if(callback != null) {
                callback.invoke(
                    context,
                    text.getContent(),
                    AnalysisUnit.wrap(fromNative),
                    foundNative != 0,
                    isNotFoundErrorNative != 0
                );
            }
        }
    }

    /**
     * This method is the only valid callback to pass to a native event
     * handler for unit parsing events.
     * This method will dispatch the execution according to the passed
     * analysis context.
     */
    @CEntryPoint
    static void unitParsed(
        final IsolateThread thread,
        final AnalysisContextNative contextNative,
        final AnalysisUnitNative unitNative,
        final byte reparsedNative
    ) {
        try(
            final AnalysisContext context = AnalysisContext.wrap(
                contextNative
            )
        ) {
            // Get the callback from the context event handler
            final EventHandler.UnitParsedCallback callback = context
                .getEventHandler()
                .getUnitParsedCallback();

            // Call the callback
            if(callback != null) {
                callback.invoke(
                    context,
                    AnalysisUnit.wrap(unitNative),
                    reparsedNative != 0
                );
            }
        }
    }

    // ==========
    // Static values
    // ==========

    /** The default grammar rule to parse the inputs. */
    private static final GrammarRule DEFAULT_GRAMMAR_RULE =
        GrammarRule.MAIN_RULE_RULE;

    /** The os name in lower case. */
    private static final String OS =
            System.getProperty("os.name").toLowerCase();

    /** The byte order of the system. */
    private static final ByteOrder BYTE_ORDER = ByteOrder.nativeOrder();

    /** Charset value that will be used to decode Langkit's buffers. */
    private static final Charset ORDERED_UTF_32 =
        BYTE_ORDER == ByteOrder.BIG_ENDIAN ? StandardCharsets.UTF_32BE
                                           : StandardCharsets.UTF_32LE;

    /** A map to store node descriptions associated to their camel name. */
    public static final Map<String, Reflection.Node>
        NODE_DESCRIPTION_MAP = new HashMap<>();

    static {
        // Fill the node description map and set the node kind descriptions
        NODE_DESCRIPTION_MAP.put(
            "LktNode",
            LktNode.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Argument",
            Argument.description
        );
        NodeKind.ARGUMENT.setDescription(
            Argument.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseLexerCaseRuleAlt",
            BaseLexerCaseRuleAlt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerCaseRuleCondAlt",
            LexerCaseRuleCondAlt.description
        );
        NodeKind.LEXER_CASE_RULE_COND_ALT.setDescription(
            LexerCaseRuleCondAlt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerCaseRuleDefaultAlt",
            LexerCaseRuleDefaultAlt.description
        );
        NodeKind.LEXER_CASE_RULE_DEFAULT_ALT.setDescription(
            LexerCaseRuleDefaultAlt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseMatchBranch",
            BaseMatchBranch.description
        );
        NODE_DESCRIPTION_MAP.put(
            "MatchBranch",
            MatchBranch.description
        );
        NodeKind.MATCH_BRANCH.setDescription(
            MatchBranch.description
        );
        NODE_DESCRIPTION_MAP.put(
            "PatternMatchBranch",
            PatternMatchBranch.description
        );
        NodeKind.PATTERN_MATCH_BRANCH.setDescription(
            PatternMatchBranch.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockExprClause",
            BlockExprClause.description
        );
        NodeKind.BLOCK_EXPR_CLAUSE.setDescription(
            BlockExprClause.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockStringLine",
            BlockStringLine.description
        );
        NodeKind.BLOCK_STRING_LINE.setDescription(
            BlockStringLine.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ClassQualifier",
            ClassQualifier.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ClassQualifierAbsent",
            ClassQualifierAbsent.description
        );
        NodeKind.CLASS_QUALIFIER_ABSENT.setDescription(
            ClassQualifierAbsent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ClassQualifierPresent",
            ClassQualifierPresent.description
        );
        NodeKind.CLASS_QUALIFIER_PRESENT.setDescription(
            ClassQualifierPresent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Decl",
            Decl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseGrammarRuleDecl",
            BaseGrammarRuleDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarRuleDecl",
            GrammarRuleDecl.description
        );
        NodeKind.GRAMMAR_RULE_DECL.setDescription(
            GrammarRuleDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SyntheticLexerDecl",
            SyntheticLexerDecl.description
        );
        NodeKind.SYNTHETIC_LEXER_DECL.setDescription(
            SyntheticLexerDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseValDecl",
            BaseValDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NodeDecl",
            NodeDecl.description
        );
        NodeKind.NODE_DECL.setDescription(
            NodeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SelfDecl",
            SelfDecl.description
        );
        NodeKind.SELF_DECL.setDescription(
            SelfDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "UserValDecl",
            UserValDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BindingValDecl",
            BindingValDecl.description
        );
        NodeKind.BINDING_VAL_DECL.setDescription(
            BindingValDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumLitDecl",
            EnumLitDecl.description
        );
        NodeKind.ENUM_LIT_DECL.setDescription(
            EnumLitDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExplicitlyTypedDecl",
            ExplicitlyTypedDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ComponentDecl",
            ComponentDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FieldDecl",
            FieldDecl.description
        );
        NodeKind.FIELD_DECL.setDescription(
            FieldDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunParamDecl",
            FunParamDecl.description
        );
        NodeKind.FUN_PARAM_DECL.setDescription(
            FunParamDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LambdaParamDecl",
            LambdaParamDecl.description
        );
        NodeKind.LAMBDA_PARAM_DECL.setDescription(
            LambdaParamDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DynVarDecl",
            DynVarDecl.description
        );
        NodeKind.DYN_VAR_DECL.setDescription(
            DynVarDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "MatchValDecl",
            MatchValDecl.description
        );
        NodeKind.MATCH_VAL_DECL.setDescription(
            MatchValDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ValDecl",
            ValDecl.description
        );
        NodeKind.VAL_DECL.setDescription(
            ValDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunDecl",
            FunDecl.description
        );
        NodeKind.FUN_DECL.setDescription(
            FunDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnvSpecDecl",
            EnvSpecDecl.description
        );
        NodeKind.ENV_SPEC_DECL.setDescription(
            EnvSpecDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ErrorDecl",
            ErrorDecl.description
        );
        NodeKind.ERROR_DECL.setDescription(
            ErrorDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericDecl",
            GenericDecl.description
        );
        NodeKind.GENERIC_DECL.setDescription(
            GenericDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarDecl",
            GrammarDecl.description
        );
        NodeKind.GRAMMAR_DECL.setDescription(
            GrammarDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerDecl",
            LexerDecl.description
        );
        NodeKind.LEXER_DECL.setDescription(
            LexerDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerFamilyDecl",
            LexerFamilyDecl.description
        );
        NodeKind.LEXER_FAMILY_DECL.setDescription(
            LexerFamilyDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SynthFunDecl",
            SynthFunDecl.description
        );
        NodeKind.SYNTH_FUN_DECL.setDescription(
            SynthFunDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SynthParamDecl",
            SynthParamDecl.description
        );
        NodeKind.SYNTH_PARAM_DECL.setDescription(
            SynthParamDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TypeDecl",
            TypeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "AnyTypeDecl",
            AnyTypeDecl.description
        );
        NodeKind.ANY_TYPE_DECL.setDescription(
            AnyTypeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassAltDecl",
            EnumClassAltDecl.description
        );
        NodeKind.ENUM_CLASS_ALT_DECL.setDescription(
            EnumClassAltDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunctionType",
            FunctionType.description
        );
        NodeKind.FUNCTION_TYPE.setDescription(
            FunctionType.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericParamTypeDecl",
            GenericParamTypeDecl.description
        );
        NodeKind.GENERIC_PARAM_TYPE_DECL.setDescription(
            GenericParamTypeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NamedTypeDecl",
            NamedTypeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BasicClassDecl",
            BasicClassDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ClassDecl",
            ClassDecl.description
        );
        NodeKind.CLASS_DECL.setDescription(
            ClassDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassDecl",
            EnumClassDecl.description
        );
        NodeKind.ENUM_CLASS_DECL.setDescription(
            EnumClassDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumTypeDecl",
            EnumTypeDecl.description
        );
        NodeKind.ENUM_TYPE_DECL.setDescription(
            EnumTypeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "StructDecl",
            StructDecl.description
        );
        NodeKind.STRUCT_DECL.setDescription(
            StructDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TraitDecl",
            TraitDecl.description
        );
        NodeKind.TRAIT_DECL.setDescription(
            TraitDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DeclAnnotation",
            DeclAnnotation.description
        );
        NodeKind.DECL_ANNOTATION.setDescription(
            DeclAnnotation.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DeclAnnotationArgs",
            DeclAnnotationArgs.description
        );
        NodeKind.DECL_ANNOTATION_ARGS.setDescription(
            DeclAnnotationArgs.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DynEnvWrapper",
            DynEnvWrapper.description
        );
        NodeKind.DYN_ENV_WRAPPER.setDescription(
            DynEnvWrapper.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ElsifBranch",
            ElsifBranch.description
        );
        NodeKind.ELSIF_BRANCH.setDescription(
            ElsifBranch.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassCase",
            EnumClassCase.description
        );
        NodeKind.ENUM_CLASS_CASE.setDescription(
            EnumClassCase.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExcludesNull",
            ExcludesNull.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExcludesNullAbsent",
            ExcludesNullAbsent.description
        );
        NodeKind.EXCLUDES_NULL_ABSENT.setDescription(
            ExcludesNullAbsent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExcludesNullPresent",
            ExcludesNullPresent.description
        );
        NodeKind.EXCLUDES_NULL_PRESENT.setDescription(
            ExcludesNullPresent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Expr",
            Expr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "AnyOf",
            AnyOf.description
        );
        NodeKind.ANY_OF.setDescription(
            AnyOf.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ArrayLiteral",
            ArrayLiteral.description
        );
        NodeKind.ARRAY_LITERAL.setDescription(
            ArrayLiteral.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseCallExpr",
            BaseCallExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "CallExpr",
            CallExpr.description
        );
        NodeKind.CALL_EXPR.setDescription(
            CallExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LogicCallExpr",
            LogicCallExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LogicPredicate",
            LogicPredicate.description
        );
        NodeKind.LOGIC_PREDICATE.setDescription(
            LogicPredicate.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LogicPropagateCall",
            LogicPropagateCall.description
        );
        NodeKind.LOGIC_PROPAGATE_CALL.setDescription(
            LogicPropagateCall.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BinOp",
            BinOp.description
        );
        NodeKind.BIN_OP.setDescription(
            BinOp.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockExpr",
            BlockExpr.description
        );
        NodeKind.BLOCK_EXPR.setDescription(
            BlockExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "CastExpr",
            CastExpr.description
        );
        NodeKind.CAST_EXPR.setDescription(
            CastExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DotExpr",
            DotExpr.description
        );
        NodeKind.DOT_EXPR.setDescription(
            DotExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ErrorOnNull",
            ErrorOnNull.description
        );
        NodeKind.ERROR_ON_NULL.setDescription(
            ErrorOnNull.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericInstantiation",
            GenericInstantiation.description
        );
        NodeKind.GENERIC_INSTANTIATION.setDescription(
            GenericInstantiation.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarExpr",
            GrammarExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarCut",
            GrammarCut.description
        );
        NodeKind.GRAMMAR_CUT.setDescription(
            GrammarCut.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarDiscard",
            GrammarDiscard.description
        );
        NodeKind.GRAMMAR_DISCARD.setDescription(
            GrammarDiscard.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarDontSkip",
            GrammarDontSkip.description
        );
        NodeKind.GRAMMAR_DONT_SKIP.setDescription(
            GrammarDontSkip.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarList",
            GrammarList.description
        );
        NodeKind.GRAMMAR_LIST.setDescription(
            GrammarList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarNull",
            GrammarNull.description
        );
        NodeKind.GRAMMAR_NULL.setDescription(
            GrammarNull.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOpt",
            GrammarOpt.description
        );
        NodeKind.GRAMMAR_OPT.setDescription(
            GrammarOpt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOptError",
            GrammarOptError.description
        );
        NodeKind.GRAMMAR_OPT_ERROR.setDescription(
            GrammarOptError.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOptErrorGroup",
            GrammarOptErrorGroup.description
        );
        NodeKind.GRAMMAR_OPT_ERROR_GROUP.setDescription(
            GrammarOptErrorGroup.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOptGroup",
            GrammarOptGroup.description
        );
        NodeKind.GRAMMAR_OPT_GROUP.setDescription(
            GrammarOptGroup.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOrExpr",
            GrammarOrExpr.description
        );
        NodeKind.GRAMMAR_OR_EXPR.setDescription(
            GrammarOrExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarPick",
            GrammarPick.description
        );
        NodeKind.GRAMMAR_PICK.setDescription(
            GrammarPick.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarImplicitPick",
            GrammarImplicitPick.description
        );
        NodeKind.GRAMMAR_IMPLICIT_PICK.setDescription(
            GrammarImplicitPick.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarPredicate",
            GrammarPredicate.description
        );
        NodeKind.GRAMMAR_PREDICATE.setDescription(
            GrammarPredicate.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarRuleRef",
            GrammarRuleRef.description
        );
        NodeKind.GRAMMAR_RULE_REF.setDescription(
            GrammarRuleRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarSkip",
            GrammarSkip.description
        );
        NodeKind.GRAMMAR_SKIP.setDescription(
            GrammarSkip.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarStopCut",
            GrammarStopCut.description
        );
        NodeKind.GRAMMAR_STOP_CUT.setDescription(
            GrammarStopCut.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ParseNodeExpr",
            ParseNodeExpr.description
        );
        NodeKind.PARSE_NODE_EXPR.setDescription(
            ParseNodeExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TokenLit",
            TokenLit.description
        );
        NodeKind.TOKEN_LIT.setDescription(
            TokenLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TokenNoCaseLit",
            TokenNoCaseLit.description
        );
        NodeKind.TOKEN_NO_CASE_LIT.setDescription(
            TokenNoCaseLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TokenPatternConcat",
            TokenPatternConcat.description
        );
        NodeKind.TOKEN_PATTERN_CONCAT.setDescription(
            TokenPatternConcat.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TokenPatternLit",
            TokenPatternLit.description
        );
        NodeKind.TOKEN_PATTERN_LIT.setDescription(
            TokenPatternLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TokenRef",
            TokenRef.description
        );
        NodeKind.TOKEN_REF.setDescription(
            TokenRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Id",
            Id.description
        );
        NodeKind.ID.setDescription(
            Id.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DefId",
            DefId.description
        );
        NodeKind.DEF_ID.setDescription(
            DefId.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ModuleRefId",
            ModuleRefId.description
        );
        NodeKind.MODULE_REF_ID.setDescription(
            ModuleRefId.description
        );
        NODE_DESCRIPTION_MAP.put(
            "RefId",
            RefId.description
        );
        NodeKind.REF_ID.setDescription(
            RefId.description
        );
        NODE_DESCRIPTION_MAP.put(
            "IfExpr",
            IfExpr.description
        );
        NodeKind.IF_EXPR.setDescription(
            IfExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Isa",
            Isa.description
        );
        NodeKind.ISA.setDescription(
            Isa.description
        );
        NODE_DESCRIPTION_MAP.put(
            "KeepExpr",
            KeepExpr.description
        );
        NodeKind.KEEP_EXPR.setDescription(
            KeepExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LambdaExpr",
            LambdaExpr.description
        );
        NodeKind.LAMBDA_EXPR.setDescription(
            LambdaExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Lit",
            Lit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BigNumLit",
            BigNumLit.description
        );
        NodeKind.BIG_NUM_LIT.setDescription(
            BigNumLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "CharLit",
            CharLit.description
        );
        NodeKind.CHAR_LIT.setDescription(
            CharLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NullLit",
            NullLit.description
        );
        NodeKind.NULL_LIT.setDescription(
            NullLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NumLit",
            NumLit.description
        );
        NodeKind.NUM_LIT.setDescription(
            NumLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "StringLit",
            StringLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockStringLit",
            BlockStringLit.description
        );
        NodeKind.BLOCK_STRING_LIT.setDescription(
            BlockStringLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SingleLineStringLit",
            SingleLineStringLit.description
        );
        NodeKind.SINGLE_LINE_STRING_LIT.setDescription(
            SingleLineStringLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "PatternSingleLineStringLit",
            PatternSingleLineStringLit.description
        );
        NodeKind.PATTERN_SINGLE_LINE_STRING_LIT.setDescription(
            PatternSingleLineStringLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LogicAssign",
            LogicAssign.description
        );
        NodeKind.LOGIC_ASSIGN.setDescription(
            LogicAssign.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LogicExpr",
            LogicExpr.description
        );
        NodeKind.LOGIC_EXPR.setDescription(
            LogicExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LogicPropagate",
            LogicPropagate.description
        );
        NodeKind.LOGIC_PROPAGATE.setDescription(
            LogicPropagate.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LogicUnify",
            LogicUnify.description
        );
        NodeKind.LOGIC_UNIFY.setDescription(
            LogicUnify.description
        );
        NODE_DESCRIPTION_MAP.put(
            "MatchExpr",
            MatchExpr.description
        );
        NodeKind.MATCH_EXPR.setDescription(
            MatchExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NotExpr",
            NotExpr.description
        );
        NodeKind.NOT_EXPR.setDescription(
            NotExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ParenExpr",
            ParenExpr.description
        );
        NodeKind.PAREN_EXPR.setDescription(
            ParenExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "RaiseExpr",
            RaiseExpr.description
        );
        NodeKind.RAISE_EXPR.setDescription(
            RaiseExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SubscriptExpr",
            SubscriptExpr.description
        );
        NodeKind.SUBSCRIPT_EXPR.setDescription(
            SubscriptExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TryExpr",
            TryExpr.description
        );
        NodeKind.TRY_EXPR.setDescription(
            TryExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "UnOp",
            UnOp.description
        );
        NodeKind.UN_OP.setDescription(
            UnOp.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FullDecl",
            FullDecl.description
        );
        NodeKind.FULL_DECL.setDescription(
            FullDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarListSep",
            GrammarListSep.description
        );
        NodeKind.GRAMMAR_LIST_SEP.setDescription(
            GrammarListSep.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Import",
            Import.description
        );
        NodeKind.IMPORT.setDescription(
            Import.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LangkitRoot",
            LangkitRoot.description
        );
        NodeKind.LANGKIT_ROOT.setDescription(
            LangkitRoot.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerCaseRule",
            LexerCaseRule.description
        );
        NodeKind.LEXER_CASE_RULE.setDescription(
            LexerCaseRule.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerCaseRuleSend",
            LexerCaseRuleSend.description
        );
        NodeKind.LEXER_CASE_RULE_SEND.setDescription(
            LexerCaseRuleSend.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ListKind",
            ListKind.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ListKindOne",
            ListKindOne.description
        );
        NodeKind.LIST_KIND_ONE.setDescription(
            ListKindOne.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ListKindZero",
            ListKindZero.description
        );
        NodeKind.LIST_KIND_ZERO.setDescription(
            ListKindZero.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LktNodeBaseList",
            LktNodeBaseList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ArgumentList",
            ArgumentList.description
        );
        NodeKind.ARGUMENT_LIST.setDescription(
            ArgumentList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseLexerCaseRuleAltList",
            BaseLexerCaseRuleAltList.description
        );
        NodeKind.BASE_LEXER_CASE_RULE_ALT_LIST.setDescription(
            BaseLexerCaseRuleAltList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseMatchBranchList",
            BaseMatchBranchList.description
        );
        NodeKind.BASE_MATCH_BRANCH_LIST.setDescription(
            BaseMatchBranchList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockStringLineList",
            BlockStringLineList.description
        );
        NodeKind.BLOCK_STRING_LINE_LIST.setDescription(
            BlockStringLineList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "CallExprList",
            CallExprList.description
        );
        NodeKind.CALL_EXPR_LIST.setDescription(
            CallExprList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DeclAnnotationList",
            DeclAnnotationList.description
        );
        NodeKind.DECL_ANNOTATION_LIST.setDescription(
            DeclAnnotationList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ElsifBranchList",
            ElsifBranchList.description
        );
        NodeKind.ELSIF_BRANCH_LIST.setDescription(
            ElsifBranchList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassAltDeclList",
            EnumClassAltDeclList.description
        );
        NodeKind.ENUM_CLASS_ALT_DECL_LIST.setDescription(
            EnumClassAltDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassCaseList",
            EnumClassCaseList.description
        );
        NodeKind.ENUM_CLASS_CASE_LIST.setDescription(
            EnumClassCaseList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumLitDeclList",
            EnumLitDeclList.description
        );
        NodeKind.ENUM_LIT_DECL_LIST.setDescription(
            EnumLitDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExprList",
            ExprList.description
        );
        NodeKind.EXPR_LIST.setDescription(
            ExprList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "AnyOfList",
            AnyOfList.description
        );
        NodeKind.ANY_OF_LIST.setDescription(
            AnyOfList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FullDeclList",
            FullDeclList.description
        );
        NodeKind.FULL_DECL_LIST.setDescription(
            FullDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DeclBlock",
            DeclBlock.description
        );
        NodeKind.DECL_BLOCK.setDescription(
            DeclBlock.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericParamDeclList",
            GenericParamDeclList.description
        );
        NodeKind.GENERIC_PARAM_DECL_LIST.setDescription(
            GenericParamDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunParamDeclList",
            FunParamDeclList.description
        );
        NodeKind.FUN_PARAM_DECL_LIST.setDescription(
            FunParamDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarExprList",
            GrammarExprList.description
        );
        NodeKind.GRAMMAR_EXPR_LIST.setDescription(
            GrammarExprList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarExprListList",
            GrammarExprListList.description
        );
        NodeKind.GRAMMAR_EXPR_LIST_LIST.setDescription(
            GrammarExprListList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ImportList",
            ImportList.description
        );
        NodeKind.IMPORT_LIST.setDescription(
            ImportList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LambdaParamDeclList",
            LambdaParamDeclList.description
        );
        NodeKind.LAMBDA_PARAM_DECL_LIST.setDescription(
            LambdaParamDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LktNodeList",
            LktNodeList.description
        );
        NodeKind.LKT_NODE_LIST.setDescription(
            LktNodeList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "PatternDetailList",
            PatternDetailList.description
        );
        NodeKind.PATTERN_DETAIL_LIST.setDescription(
            PatternDetailList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "PatternList",
            PatternList.description
        );
        NodeKind.PATTERN_LIST.setDescription(
            PatternList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "RefIdList",
            RefIdList.description
        );
        NodeKind.REF_ID_LIST.setDescription(
            RefIdList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TypeRefList",
            TypeRefList.description
        );
        NodeKind.TYPE_REF_LIST.setDescription(
            TypeRefList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SyntheticTypeRefList",
            SyntheticTypeRefList.description
        );
        NodeKind.SYNTHETIC_TYPE_REF_LIST.setDescription(
            SyntheticTypeRefList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NullCondQualifier",
            NullCondQualifier.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NullCondQualifierAbsent",
            NullCondQualifierAbsent.description
        );
        NodeKind.NULL_COND_QUALIFIER_ABSENT.setDescription(
            NullCondQualifierAbsent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NullCondQualifierPresent",
            NullCondQualifierPresent.description
        );
        NodeKind.NULL_COND_QUALIFIER_PRESENT.setDescription(
            NullCondQualifierPresent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Op",
            Op.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpAmp",
            OpAmp.description
        );
        NodeKind.OP_AMP.setDescription(
            OpAmp.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpAnd",
            OpAnd.description
        );
        NodeKind.OP_AND.setDescription(
            OpAnd.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpDiv",
            OpDiv.description
        );
        NodeKind.OP_DIV.setDescription(
            OpDiv.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpEq",
            OpEq.description
        );
        NodeKind.OP_EQ.setDescription(
            OpEq.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpGt",
            OpGt.description
        );
        NodeKind.OP_GT.setDescription(
            OpGt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpGte",
            OpGte.description
        );
        NodeKind.OP_GTE.setDescription(
            OpGte.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpLogicAnd",
            OpLogicAnd.description
        );
        NodeKind.OP_LOGIC_AND.setDescription(
            OpLogicAnd.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpLogicOr",
            OpLogicOr.description
        );
        NodeKind.OP_LOGIC_OR.setDescription(
            OpLogicOr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpLt",
            OpLt.description
        );
        NodeKind.OP_LT.setDescription(
            OpLt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpLte",
            OpLte.description
        );
        NodeKind.OP_LTE.setDescription(
            OpLte.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpMinus",
            OpMinus.description
        );
        NodeKind.OP_MINUS.setDescription(
            OpMinus.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpMult",
            OpMult.description
        );
        NodeKind.OP_MULT.setDescription(
            OpMult.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpNe",
            OpNe.description
        );
        NodeKind.OP_NE.setDescription(
            OpNe.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpOr",
            OpOr.description
        );
        NodeKind.OP_OR.setDescription(
            OpOr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpOrInt",
            OpOrInt.description
        );
        NodeKind.OP_OR_INT.setDescription(
            OpOrInt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpPlus",
            OpPlus.description
        );
        NodeKind.OP_PLUS.setDescription(
            OpPlus.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Pattern",
            Pattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "AnyTypePattern",
            AnyTypePattern.description
        );
        NodeKind.ANY_TYPE_PATTERN.setDescription(
            AnyTypePattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BindingPattern",
            BindingPattern.description
        );
        NodeKind.BINDING_PATTERN.setDescription(
            BindingPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BoolPattern",
            BoolPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BoolPatternFalse",
            BoolPatternFalse.description
        );
        NodeKind.BOOL_PATTERN_FALSE.setDescription(
            BoolPatternFalse.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BoolPatternTrue",
            BoolPatternTrue.description
        );
        NodeKind.BOOL_PATTERN_TRUE.setDescription(
            BoolPatternTrue.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EllipsisPattern",
            EllipsisPattern.description
        );
        NodeKind.ELLIPSIS_PATTERN.setDescription(
            EllipsisPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExtendedPattern",
            ExtendedPattern.description
        );
        NodeKind.EXTENDED_PATTERN.setDescription(
            ExtendedPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FilteredPattern",
            FilteredPattern.description
        );
        NodeKind.FILTERED_PATTERN.setDescription(
            FilteredPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "IntegerPattern",
            IntegerPattern.description
        );
        NodeKind.INTEGER_PATTERN.setDescription(
            IntegerPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ListPattern",
            ListPattern.description
        );
        NodeKind.LIST_PATTERN.setDescription(
            ListPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NotPattern",
            NotPattern.description
        );
        NodeKind.NOT_PATTERN.setDescription(
            NotPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NullPattern",
            NullPattern.description
        );
        NodeKind.NULL_PATTERN.setDescription(
            NullPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OrPattern",
            OrPattern.description
        );
        NodeKind.OR_PATTERN.setDescription(
            OrPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ParenPattern",
            ParenPattern.description
        );
        NodeKind.PAREN_PATTERN.setDescription(
            ParenPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "RegexPattern",
            RegexPattern.description
        );
        NodeKind.REGEX_PATTERN.setDescription(
            RegexPattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TuplePattern",
            TuplePattern.description
        );
        NodeKind.TUPLE_PATTERN.setDescription(
            TuplePattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TypePattern",
            TypePattern.description
        );
        NodeKind.TYPE_PATTERN.setDescription(
            TypePattern.description
        );
        NODE_DESCRIPTION_MAP.put(
            "PatternDetail",
            PatternDetail.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FieldPatternDetail",
            FieldPatternDetail.description
        );
        NodeKind.FIELD_PATTERN_DETAIL.setDescription(
            FieldPatternDetail.description
        );
        NODE_DESCRIPTION_MAP.put(
            "PropertyPatternDetail",
            PropertyPatternDetail.description
        );
        NodeKind.PROPERTY_PATTERN_DETAIL.setDescription(
            PropertyPatternDetail.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SelectorPatternDetail",
            SelectorPatternDetail.description
        );
        NodeKind.SELECTOR_PATTERN_DETAIL.setDescription(
            SelectorPatternDetail.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SelectorCall",
            SelectorCall.description
        );
        NodeKind.SELECTOR_CALL.setDescription(
            SelectorCall.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TypeRef",
            TypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DefaultListTypeRef",
            DefaultListTypeRef.description
        );
        NodeKind.DEFAULT_LIST_TYPE_REF.setDescription(
            DefaultListTypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunctionTypeRef",
            FunctionTypeRef.description
        );
        NodeKind.FUNCTION_TYPE_REF.setDescription(
            FunctionTypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericTypeRef",
            GenericTypeRef.description
        );
        NodeKind.GENERIC_TYPE_REF.setDescription(
            GenericTypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SimpleTypeRef",
            SimpleTypeRef.description
        );
        NodeKind.SIMPLE_TYPE_REF.setDescription(
            SimpleTypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "VarBind",
            VarBind.description
        );
        NodeKind.VAR_BIND.setDescription(
            VarBind.description
        );
    }

    // ==========
    // Util functions
    // ==========

    /**
     * Get the string representing the memory.
     *
     * @param pointer The pointer to start displaying the memory from.
     * @param count The number of bytes to display from the pointer.
     * @return The string representing the memory as hex bytes.
     */
    private static String dumpMemory(
        final Pointer pointer,
        final long count
    ) {
        final StringBuilder res = new StringBuilder();
        for(int i = 0 ; i < count ; i++) {
            final byte toDump = pointer.readByte(i);
            res.append(String.format("%02x ", toDump));
        }
        return res.toString();
    }

    /**
     * Convert a Java string to a C string by allocating memory.
     *
     * @param jString The Java string to convert.
     * @return The native C char pointer. This pointer MUST be freed.
     */
    @CompilerDirectives.TruffleBoundary
    private static CCharPointer toCString(
        final String jString
    ) {
        final UnsignedWord size = WordFactory.unsigned(jString.length() + 1);
        final CCharPointer res = UnmanagedMemory.calloc(size);
        if(jString.length() > 0) {
            CTypeConversion.toCString(
                jString,
                StandardCharsets.UTF_8,
                res,
                size
            );
        }

        return res;
    }

    /**
     * Convert a Java byte array in a native byte array.
     *
     * @param bytes The Java bytes.
     * @return The native C char pointer. This pointer MUST be freed.
     */
    private static CCharPointer toCBytes(
        final byte[] bytes
    ) {
        final UnsignedWord size = WordFactory.unsigned(bytes.length);
        final CCharPointer res = UnmanagedMemory.malloc(size);
        for(int i = 0 ; i < bytes.length ; i++) {
            res.write(i, bytes[i]);
        }
        return res;
    }

    /**
     * Convert a native-image C string to a Java string.
     *
     * @param pointer The char pointer to convert to a Java string.
     * @return The Java string.
     */
    @CompilerDirectives.TruffleBoundary
    private static String toJString(
        final CCharPointer pointer
    ) {
        if (pointer.isNull()) {
            return null;
        }
        return CTypeConversion.toJavaString(pointer);
    }

    /**
     * This function decode a utf 32 int array in a
     * string without calling Java charset.
     *
     * @param chars The int array to decode.
     * @return The resulting string.
     */
    @CompilerDirectives.TruffleBoundary
    private static String decodeUTF32(
        final byte[] toDecode
    ) {
        return new String(toDecode, ORDERED_UTF_32);
    }

    /**
     * This function encode a given string to a int array
     * according to the utf 32 standard.
     *
     * @param toEncode The string to encode.
     * @return The encoded string in an int array.
     */
    @CompilerDirectives.TruffleBoundary
    private static byte[] encodeUTF32(
        final String toEncode
    ) {
        return toEncode.getBytes(ORDERED_UTF_32);
    }

    /**
     * Get the string representation of the given string.
     * This function escaped needed chars and format the string.
     *
     * @param source The source string to get the representation for.
     * @return The representation of the string.
     */
    private static String stringRepresentation(
        final String source
    ) {
        return source
            .replace("\"", "\\\"")
            .replace("\n", "\\x0a");
    }

    /**
     * Convert a C Langkit exception to the LangkitException class.
     */
    private static LangkitException wrapException(
        final LangkitExceptionNative exc
    ) {
        final PointerWrapper stackTrace =
            PointerWrapper.wrap(exc.get_stack_trace());
        return new LangkitException(
            exc.get_kind(),
            toJString(exc.get_information()),
            new LangkitStackTrace(stackTrace)
        );
    }

    /**
      * Return the exception raised by the last C API call, or null if the last
      * call was successful.
      */
    private static LangkitException getLastException() {
        LangkitException result = null;

        if(ImageInfo.inImageCode()) {
            final LangkitExceptionNative exceptionNative =
                NI_LIB.lkt_get_last_exception();
            if(exceptionNative.isNonNull()) {
                result = wrapException(exceptionNative);
            }
        } else {
            result = JNI_LIB.lkt_get_last_exception();
        }
        return result;
    }

    /**
     * Check the last exception raised by langkit and throw it.
     *
     * @throws The last langkit exception if there is one.
     */
    @CompilerDirectives.TruffleBoundary
    private static void checkException() throws LangkitException {
        LangkitException exc = getLastException();
        if(exc != null)
            throw exc;
    }

    /**
     * Create a native array from the given rewriting nodes array.
     * The returned pointer must be freed using UnmanagedMemory#free methods.
     */
    private static WordPointer rewritingNodesToNative(
        final RewritingNode[] rewritingNodes
    ) {
        final WordPointer res = UnmanagedMemory.malloc(
            rewritingNodes.length * SizeOf.get(WordPointer.class)
        );
        for(int i = 0; i < rewritingNodes.length; i++) {
            res.write(i, rewritingNodes[i].unwrap());
        }
        return res;
    }

    // ==========
    // Util interfaces
    // ==========

    /**
     * Interface to visit the parse tree.
     */
    public static interface BasicVisitor<T> {
        T visit(LktNode node);
        T visit(Argument node);
        T visit(LexerCaseRuleCondAlt node);
        T visit(LexerCaseRuleDefaultAlt node);
        T visit(MatchBranch node);
        T visit(PatternMatchBranch node);
        T visit(BlockExprClause node);
        T visit(BlockStringLine node);
        T visit(ClassQualifierAbsent node);
        T visit(ClassQualifierPresent node);
        T visit(GrammarRuleDecl node);
        T visit(SyntheticLexerDecl node);
        T visit(NodeDecl node);
        T visit(SelfDecl node);
        T visit(BindingValDecl node);
        T visit(EnumLitDecl node);
        T visit(FieldDecl node);
        T visit(FunParamDecl node);
        T visit(LambdaParamDecl node);
        T visit(DynVarDecl node);
        T visit(MatchValDecl node);
        T visit(ValDecl node);
        T visit(FunDecl node);
        T visit(EnvSpecDecl node);
        T visit(ErrorDecl node);
        T visit(GenericDecl node);
        T visit(GrammarDecl node);
        T visit(LexerDecl node);
        T visit(LexerFamilyDecl node);
        T visit(SynthFunDecl node);
        T visit(SynthParamDecl node);
        T visit(AnyTypeDecl node);
        T visit(EnumClassAltDecl node);
        T visit(FunctionType node);
        T visit(GenericParamTypeDecl node);
        T visit(ClassDecl node);
        T visit(EnumClassDecl node);
        T visit(EnumTypeDecl node);
        T visit(StructDecl node);
        T visit(TraitDecl node);
        T visit(DeclAnnotation node);
        T visit(DeclAnnotationArgs node);
        T visit(DynEnvWrapper node);
        T visit(ElsifBranch node);
        T visit(EnumClassCase node);
        T visit(ExcludesNullAbsent node);
        T visit(ExcludesNullPresent node);
        T visit(AnyOf node);
        T visit(ArrayLiteral node);
        T visit(CallExpr node);
        T visit(LogicPredicate node);
        T visit(LogicPropagateCall node);
        T visit(BinOp node);
        T visit(BlockExpr node);
        T visit(CastExpr node);
        T visit(DotExpr node);
        T visit(ErrorOnNull node);
        T visit(GenericInstantiation node);
        T visit(GrammarCut node);
        T visit(GrammarDiscard node);
        T visit(GrammarDontSkip node);
        T visit(GrammarList node);
        T visit(GrammarNull node);
        T visit(GrammarOpt node);
        T visit(GrammarOptError node);
        T visit(GrammarOptErrorGroup node);
        T visit(GrammarOptGroup node);
        T visit(GrammarOrExpr node);
        T visit(GrammarPick node);
        T visit(GrammarImplicitPick node);
        T visit(GrammarPredicate node);
        T visit(GrammarRuleRef node);
        T visit(GrammarSkip node);
        T visit(GrammarStopCut node);
        T visit(ParseNodeExpr node);
        T visit(TokenLit node);
        T visit(TokenNoCaseLit node);
        T visit(TokenPatternConcat node);
        T visit(TokenPatternLit node);
        T visit(TokenRef node);
        T visit(Id node);
        T visit(DefId node);
        T visit(ModuleRefId node);
        T visit(RefId node);
        T visit(IfExpr node);
        T visit(Isa node);
        T visit(KeepExpr node);
        T visit(LambdaExpr node);
        T visit(BigNumLit node);
        T visit(CharLit node);
        T visit(NullLit node);
        T visit(NumLit node);
        T visit(BlockStringLit node);
        T visit(SingleLineStringLit node);
        T visit(PatternSingleLineStringLit node);
        T visit(LogicAssign node);
        T visit(LogicExpr node);
        T visit(LogicPropagate node);
        T visit(LogicUnify node);
        T visit(MatchExpr node);
        T visit(NotExpr node);
        T visit(ParenExpr node);
        T visit(RaiseExpr node);
        T visit(SubscriptExpr node);
        T visit(TryExpr node);
        T visit(UnOp node);
        T visit(FullDecl node);
        T visit(GrammarListSep node);
        T visit(Import node);
        T visit(LangkitRoot node);
        T visit(LexerCaseRule node);
        T visit(LexerCaseRuleSend node);
        T visit(ListKindOne node);
        T visit(ListKindZero node);
        T visit(ArgumentList node);
        T visit(BaseLexerCaseRuleAltList node);
        T visit(BaseMatchBranchList node);
        T visit(BlockStringLineList node);
        T visit(CallExprList node);
        T visit(DeclAnnotationList node);
        T visit(ElsifBranchList node);
        T visit(EnumClassAltDeclList node);
        T visit(EnumClassCaseList node);
        T visit(EnumLitDeclList node);
        T visit(ExprList node);
        T visit(AnyOfList node);
        T visit(FullDeclList node);
        T visit(DeclBlock node);
        T visit(GenericParamDeclList node);
        T visit(FunParamDeclList node);
        T visit(GrammarExprList node);
        T visit(GrammarExprListList node);
        T visit(ImportList node);
        T visit(LambdaParamDeclList node);
        T visit(LktNodeList node);
        T visit(PatternDetailList node);
        T visit(PatternList node);
        T visit(RefIdList node);
        T visit(TypeRefList node);
        T visit(SyntheticTypeRefList node);
        T visit(NullCondQualifierAbsent node);
        T visit(NullCondQualifierPresent node);
        T visit(OpAmp node);
        T visit(OpAnd node);
        T visit(OpDiv node);
        T visit(OpEq node);
        T visit(OpGt node);
        T visit(OpGte node);
        T visit(OpLogicAnd node);
        T visit(OpLogicOr node);
        T visit(OpLt node);
        T visit(OpLte node);
        T visit(OpMinus node);
        T visit(OpMult node);
        T visit(OpNe node);
        T visit(OpOr node);
        T visit(OpOrInt node);
        T visit(OpPlus node);
        T visit(AnyTypePattern node);
        T visit(BindingPattern node);
        T visit(BoolPatternFalse node);
        T visit(BoolPatternTrue node);
        T visit(EllipsisPattern node);
        T visit(ExtendedPattern node);
        T visit(FilteredPattern node);
        T visit(IntegerPattern node);
        T visit(ListPattern node);
        T visit(NotPattern node);
        T visit(NullPattern node);
        T visit(OrPattern node);
        T visit(ParenPattern node);
        T visit(RegexPattern node);
        T visit(TuplePattern node);
        T visit(TypePattern node);
        T visit(FieldPatternDetail node);
        T visit(PropertyPatternDetail node);
        T visit(SelectorPatternDetail node);
        T visit(SelectorCall node);
        T visit(DefaultListTypeRef node);
        T visit(FunctionTypeRef node);
        T visit(GenericTypeRef node);
        T visit(SimpleTypeRef node);
        T visit(VarBind node);
    }

    /**
     * Abstract class to visit the parse tree with default behavior.
     */
    public static abstract class DefaultVisitor<T> implements BasicVisitor<T> {

        /**
         * The default behavior to use when a ``visit`` method was not
         * overriden by the parent class.
         */
        protected final
        Function<LktNode, T> defaultBehavior;

        /**
         * Create the default behavior function and return it.
         *
         * This method is called by the ``DefaultVisitor`` constructor. You do
         * not need to call it yourself. This method can be overriden to change
         * the default behavior. The first argument is the node that is
         * currently being visited.
         */
        protected Function<LktNode, T> createDefaultBehavior() {
            return (node) -> null;
        }

        /**
         * Construct a DefaultVisitor with a default behavior that does
         * nothing.
         */
        public DefaultVisitor () {
            this.defaultBehavior = createDefaultBehavior();
        }

        @Override
        public T visit(LktNode node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(Argument node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LexerCaseRuleCondAlt node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LexerCaseRuleDefaultAlt node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(MatchBranch node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(PatternMatchBranch node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BlockExprClause node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BlockStringLine node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ClassQualifierAbsent node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ClassQualifierPresent node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarRuleDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SyntheticLexerDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(NodeDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SelfDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BindingValDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EnumLitDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FieldDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FunParamDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LambdaParamDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(DynVarDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(MatchValDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ValDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FunDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EnvSpecDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ErrorDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GenericDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LexerDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LexerFamilyDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SynthFunDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SynthParamDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(AnyTypeDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EnumClassAltDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FunctionType node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GenericParamTypeDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ClassDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EnumClassDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EnumTypeDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(StructDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TraitDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(DeclAnnotation node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(DeclAnnotationArgs node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(DynEnvWrapper node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ElsifBranch node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EnumClassCase node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ExcludesNullAbsent node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ExcludesNullPresent node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(AnyOf node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ArrayLiteral node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(CallExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LogicPredicate node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LogicPropagateCall node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BinOp node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BlockExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(CastExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(DotExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ErrorOnNull node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GenericInstantiation node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarCut node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarDiscard node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarDontSkip node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarNull node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarOpt node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarOptError node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarOptErrorGroup node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarOptGroup node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarOrExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarPick node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarImplicitPick node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarPredicate node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarRuleRef node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarSkip node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarStopCut node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ParseNodeExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TokenLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TokenNoCaseLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TokenPatternConcat node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TokenPatternLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TokenRef node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(Id node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(DefId node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ModuleRefId node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(RefId node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(IfExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(Isa node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(KeepExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LambdaExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BigNumLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(CharLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(NullLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(NumLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BlockStringLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SingleLineStringLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(PatternSingleLineStringLit node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LogicAssign node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LogicExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LogicPropagate node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LogicUnify node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(MatchExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(NotExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ParenExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(RaiseExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SubscriptExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TryExpr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(UnOp node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FullDecl node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarListSep node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(Import node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LangkitRoot node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LexerCaseRule node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LexerCaseRuleSend node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ListKindOne node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ListKindZero node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ArgumentList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BaseLexerCaseRuleAltList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BaseMatchBranchList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BlockStringLineList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(CallExprList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(DeclAnnotationList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ElsifBranchList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EnumClassAltDeclList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EnumClassCaseList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EnumLitDeclList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ExprList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(AnyOfList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FullDeclList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(DeclBlock node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GenericParamDeclList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FunParamDeclList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarExprList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GrammarExprListList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ImportList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LambdaParamDeclList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(LktNodeList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(PatternDetailList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(PatternList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(RefIdList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TypeRefList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SyntheticTypeRefList node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(NullCondQualifierAbsent node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(NullCondQualifierPresent node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpAmp node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpAnd node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpDiv node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpEq node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpGt node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpGte node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpLogicAnd node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpLogicOr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpLt node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpLte node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpMinus node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpMult node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpNe node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpOr node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpOrInt node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OpPlus node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(AnyTypePattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BindingPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BoolPatternFalse node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(BoolPatternTrue node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(EllipsisPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ExtendedPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FilteredPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(IntegerPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ListPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(NotPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(NullPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(OrPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(ParenPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(RegexPattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TuplePattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(TypePattern node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FieldPatternDetail node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(PropertyPatternDetail node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SelectorPatternDetail node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SelectorCall node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(DefaultListTypeRef node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(FunctionTypeRef node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(GenericTypeRef node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(SimpleTypeRef node) {
            return defaultBehavior.apply(node);
        }
        @Override
        public T visit(VarBind node) {
            return defaultBehavior.apply(node);
        }
    }

    /**
     * Interface to visit the parse tree with a parameter.
     */
    public static interface ParamVisitor<T, P> {
        T visit(LktNode node, P param);
        T visit(Argument node, P param);
        T visit(LexerCaseRuleCondAlt node, P param);
        T visit(LexerCaseRuleDefaultAlt node, P param);
        T visit(MatchBranch node, P param);
        T visit(PatternMatchBranch node, P param);
        T visit(BlockExprClause node, P param);
        T visit(BlockStringLine node, P param);
        T visit(ClassQualifierAbsent node, P param);
        T visit(ClassQualifierPresent node, P param);
        T visit(GrammarRuleDecl node, P param);
        T visit(SyntheticLexerDecl node, P param);
        T visit(NodeDecl node, P param);
        T visit(SelfDecl node, P param);
        T visit(BindingValDecl node, P param);
        T visit(EnumLitDecl node, P param);
        T visit(FieldDecl node, P param);
        T visit(FunParamDecl node, P param);
        T visit(LambdaParamDecl node, P param);
        T visit(DynVarDecl node, P param);
        T visit(MatchValDecl node, P param);
        T visit(ValDecl node, P param);
        T visit(FunDecl node, P param);
        T visit(EnvSpecDecl node, P param);
        T visit(ErrorDecl node, P param);
        T visit(GenericDecl node, P param);
        T visit(GrammarDecl node, P param);
        T visit(LexerDecl node, P param);
        T visit(LexerFamilyDecl node, P param);
        T visit(SynthFunDecl node, P param);
        T visit(SynthParamDecl node, P param);
        T visit(AnyTypeDecl node, P param);
        T visit(EnumClassAltDecl node, P param);
        T visit(FunctionType node, P param);
        T visit(GenericParamTypeDecl node, P param);
        T visit(ClassDecl node, P param);
        T visit(EnumClassDecl node, P param);
        T visit(EnumTypeDecl node, P param);
        T visit(StructDecl node, P param);
        T visit(TraitDecl node, P param);
        T visit(DeclAnnotation node, P param);
        T visit(DeclAnnotationArgs node, P param);
        T visit(DynEnvWrapper node, P param);
        T visit(ElsifBranch node, P param);
        T visit(EnumClassCase node, P param);
        T visit(ExcludesNullAbsent node, P param);
        T visit(ExcludesNullPresent node, P param);
        T visit(AnyOf node, P param);
        T visit(ArrayLiteral node, P param);
        T visit(CallExpr node, P param);
        T visit(LogicPredicate node, P param);
        T visit(LogicPropagateCall node, P param);
        T visit(BinOp node, P param);
        T visit(BlockExpr node, P param);
        T visit(CastExpr node, P param);
        T visit(DotExpr node, P param);
        T visit(ErrorOnNull node, P param);
        T visit(GenericInstantiation node, P param);
        T visit(GrammarCut node, P param);
        T visit(GrammarDiscard node, P param);
        T visit(GrammarDontSkip node, P param);
        T visit(GrammarList node, P param);
        T visit(GrammarNull node, P param);
        T visit(GrammarOpt node, P param);
        T visit(GrammarOptError node, P param);
        T visit(GrammarOptErrorGroup node, P param);
        T visit(GrammarOptGroup node, P param);
        T visit(GrammarOrExpr node, P param);
        T visit(GrammarPick node, P param);
        T visit(GrammarImplicitPick node, P param);
        T visit(GrammarPredicate node, P param);
        T visit(GrammarRuleRef node, P param);
        T visit(GrammarSkip node, P param);
        T visit(GrammarStopCut node, P param);
        T visit(ParseNodeExpr node, P param);
        T visit(TokenLit node, P param);
        T visit(TokenNoCaseLit node, P param);
        T visit(TokenPatternConcat node, P param);
        T visit(TokenPatternLit node, P param);
        T visit(TokenRef node, P param);
        T visit(Id node, P param);
        T visit(DefId node, P param);
        T visit(ModuleRefId node, P param);
        T visit(RefId node, P param);
        T visit(IfExpr node, P param);
        T visit(Isa node, P param);
        T visit(KeepExpr node, P param);
        T visit(LambdaExpr node, P param);
        T visit(BigNumLit node, P param);
        T visit(CharLit node, P param);
        T visit(NullLit node, P param);
        T visit(NumLit node, P param);
        T visit(BlockStringLit node, P param);
        T visit(SingleLineStringLit node, P param);
        T visit(PatternSingleLineStringLit node, P param);
        T visit(LogicAssign node, P param);
        T visit(LogicExpr node, P param);
        T visit(LogicPropagate node, P param);
        T visit(LogicUnify node, P param);
        T visit(MatchExpr node, P param);
        T visit(NotExpr node, P param);
        T visit(ParenExpr node, P param);
        T visit(RaiseExpr node, P param);
        T visit(SubscriptExpr node, P param);
        T visit(TryExpr node, P param);
        T visit(UnOp node, P param);
        T visit(FullDecl node, P param);
        T visit(GrammarListSep node, P param);
        T visit(Import node, P param);
        T visit(LangkitRoot node, P param);
        T visit(LexerCaseRule node, P param);
        T visit(LexerCaseRuleSend node, P param);
        T visit(ListKindOne node, P param);
        T visit(ListKindZero node, P param);
        T visit(ArgumentList node, P param);
        T visit(BaseLexerCaseRuleAltList node, P param);
        T visit(BaseMatchBranchList node, P param);
        T visit(BlockStringLineList node, P param);
        T visit(CallExprList node, P param);
        T visit(DeclAnnotationList node, P param);
        T visit(ElsifBranchList node, P param);
        T visit(EnumClassAltDeclList node, P param);
        T visit(EnumClassCaseList node, P param);
        T visit(EnumLitDeclList node, P param);
        T visit(ExprList node, P param);
        T visit(AnyOfList node, P param);
        T visit(FullDeclList node, P param);
        T visit(DeclBlock node, P param);
        T visit(GenericParamDeclList node, P param);
        T visit(FunParamDeclList node, P param);
        T visit(GrammarExprList node, P param);
        T visit(GrammarExprListList node, P param);
        T visit(ImportList node, P param);
        T visit(LambdaParamDeclList node, P param);
        T visit(LktNodeList node, P param);
        T visit(PatternDetailList node, P param);
        T visit(PatternList node, P param);
        T visit(RefIdList node, P param);
        T visit(TypeRefList node, P param);
        T visit(SyntheticTypeRefList node, P param);
        T visit(NullCondQualifierAbsent node, P param);
        T visit(NullCondQualifierPresent node, P param);
        T visit(OpAmp node, P param);
        T visit(OpAnd node, P param);
        T visit(OpDiv node, P param);
        T visit(OpEq node, P param);
        T visit(OpGt node, P param);
        T visit(OpGte node, P param);
        T visit(OpLogicAnd node, P param);
        T visit(OpLogicOr node, P param);
        T visit(OpLt node, P param);
        T visit(OpLte node, P param);
        T visit(OpMinus node, P param);
        T visit(OpMult node, P param);
        T visit(OpNe node, P param);
        T visit(OpOr node, P param);
        T visit(OpOrInt node, P param);
        T visit(OpPlus node, P param);
        T visit(AnyTypePattern node, P param);
        T visit(BindingPattern node, P param);
        T visit(BoolPatternFalse node, P param);
        T visit(BoolPatternTrue node, P param);
        T visit(EllipsisPattern node, P param);
        T visit(ExtendedPattern node, P param);
        T visit(FilteredPattern node, P param);
        T visit(IntegerPattern node, P param);
        T visit(ListPattern node, P param);
        T visit(NotPattern node, P param);
        T visit(NullPattern node, P param);
        T visit(OrPattern node, P param);
        T visit(ParenPattern node, P param);
        T visit(RegexPattern node, P param);
        T visit(TuplePattern node, P param);
        T visit(TypePattern node, P param);
        T visit(FieldPatternDetail node, P param);
        T visit(PropertyPatternDetail node, P param);
        T visit(SelectorPatternDetail node, P param);
        T visit(SelectorCall node, P param);
        T visit(DefaultListTypeRef node, P param);
        T visit(FunctionTypeRef node, P param);
        T visit(GenericTypeRef node, P param);
        T visit(SimpleTypeRef node, P param);
        T visit(VarBind node, P param);
    }

    // ==========
    // Util classes
    // ==========

    /**
     * This class represents a pointer and can hold NI and JNI addresses.
     */
    public static final class PointerWrapper {

        // ----- Instance attributes -----

        /** The pointer NI value. */
        private PointerBase ni;

        /** The pointer JNI value. */
        private final long jni;

        // ----- Constructors -----

        /**
         * Create a new custom pointer from a NI pointer based value.
         *
         * @param niPointer The pointer based value.
         */
        PointerWrapper(
            final PointerBase niPointer
        ) {
            this.ni = niPointer;
            this.jni = -1;
        }

        /**
         * Create a new custom pointer from a long value.
         *
         * @param jniPointer The pointer in a long value.
         */
        PointerWrapper(
            final long jniPointer
        ) {
            this.jni = jniPointer;
        }

        /**
         * Wrap the given NI pointer in the Java class.
         *
         * @param niPointer The NI pointer to wrap.
         * @return The wrapped pointer.
         */
        static PointerWrapper wrap(
            final PointerBase niPointer
        ) {
            return new PointerWrapper(niPointer);
        }

        /**
         * Get the null pointer according to the execution mode.
         *
         * @return The null custom pointer.
         */
        public static PointerWrapper nullPointer() {

            if(ImageInfo.inImageCode()) {
                return new PointerWrapper(WordFactory.nullPointer());
            } else {
                return new PointerWrapper(0L);
            }

        }

        // ----- Instance methods -----

        /**
         * Get the pointer as an NI pointer based value.
         *
         * @return The pointer based value for NI.
         */
        public <T extends PointerBase> T ni() {
            return (T) this.ni;
        }

        /**
         * Get the pointer as a long Java value.
         *
         * @return The pointer as a long value for JNI.
         */
        public long jni() {
            return this.jni;
        }

        /**
         * Get if the pointer is null.
         *
         * @return True if the pointer is null, false else.
         */
        public boolean isNull() {

            if(ImageInfo.inImageCode()) {
                return this.ni.isNull();
            } else {
                return this.jni == 0;
            }

        }

        // ----- Override methods -----

        @Override
        public String toString() {

            if(ImageInfo.inImageCode()) {
                return "PointerWrapper{"
                    + this.ni.rawValue()
                    + "}";
            } else {
                return "PointerWrapper{"
                    + this.jni
                    + "}";
            }

        }

        @Override
        public boolean equals(Object o) {
            if(o == this) return true;
            if(!(o instanceof PointerWrapper)) return false;
            final PointerWrapper other = (PointerWrapper) o;
            if(ImageInfo.inImageCode()) {
                return this.ni.equal(other.ni);
            } else {
                return this.jni == other.jni;
            }
        }

        @Override
        public int hashCode() {

            if(ImageInfo.inImageCode()) {
                return (int) this.ni.rawValue();
            } else {
                return (int) this.jni;
            }

        }

    }

    // ==========
    // Reflection utils
    // ==========

    public static final class Reflection extends LangkitSupport.Reflection {
        /**
        * This class represents the description of a node.
        */
        public static final class Node extends LangkitSupport.Reflection.Node {

            // ----- Instance attributes -----

            /** Kind of the node. This kind is null if the node is abstract */
            public final NodeKind kind;

            /** Whether the node is a token node */
            public final boolean isTokenNode;

            /** Whether the node is a list node */
            public final boolean isListNode;

            /** Java class of the node */
            public final Class<? extends LktNode> clazz;

            /** Simple name of the Java class of the node */
            public final String className;

            /** Fields of the node, sorted by parsing order */
            public final String[] fields;

            /** Map containing description for all fields of the node */
            public final Map<String, Field> fieldDescriptions;

            // ----- Constructors -----

            /** Create a new node description with its kind and class */
            public Node (
                NodeKind kind,
                final boolean isTokenNode,
                final boolean isListNode,
                final Class<? extends LktNode> clazz,
                final String className,
                final String[] fields,
                final Map<String, Field> fieldDescriptions
            ) {
                this.kind = kind;
                this.isTokenNode = isTokenNode;
                this.isListNode = isListNode;
                this.clazz = clazz;
                this.className = className;
                this.fieldDescriptions = fieldDescriptions;
                this.fields = fields;
            }

        }

        /**
        * This class represents the description of a node field.
        */
        public static final class Field extends
            LangkitSupport.Reflection.Field {

            // ----- Instance attributes -----

            /** The Java method for the field */
            public final Method javaMethod;

            /** The parameters of the method */
            public final List<Param> params;

            /** The generic member reference of this field */
            public final MemberReference memberRef;

            // ----- Constructors -----

            /**
            * Create a new field description.
            *
            * @param method The Java method to access the field.
            * @param params The parameters of the field call.
            */
            public Field(
                final Method javaMethod,
                final List<Param> params,
                final MemberReference memberRef
            ) {
                this.javaMethod = javaMethod;
                this.params = params;
                this.memberRef = memberRef;
            }

            // ----- Getters -----

            /** The parameters of the method */
            public List<Param> getParams() {
                return this.params;
            }

            /** The Java method for the field */
            public Method getJavaMethod() {
                return this.javaMethod;
            }

        }

        /**
        * This class represents a parameter description.
        */
        public static class Param extends LangkitSupport.Reflection.Param {

            // ----- Instance attributes -----

            /** The type of the argument */
            public final Class<?> type;

            /** The name of the parameter */
            public final String name;

            /** The optional default value of the parameter */
            public final Optional<Object> defaultValue;

            // ----- Constructors -----

            /**
            * Create a new langkit parameter.
            *
            * @param type The type of the parameter.
            * @param name The name of the parameter.
            */
            public Param(
                final Class<?> type,
                final String name
            ) {
                this.type = type;
                this.name = name;
                this.defaultValue = Optional.empty();
            }

            /** Create a new parameter description with a default value. */
            public Param(
                final Class<?> type,
                final String name,
                final Object defaultValue
            ) {
                this.type = type;
                this.name = name;
                this.defaultValue = Optional.ofNullable(defaultValue);
            }

            // ----- Getters -----

            /** The type of the parameter */
            public Class<?> getType() {
                return this.type;
            }

            /** The optional default value of the parameter */
            public Optional<Object> getDefaultValue() {
                return this.defaultValue;
            }

        }
    }

    // ==========
    // Language specific extensions
    // ==========

    


    // ==========
    // Defining the JNI bindings library
    // ==========

    
    
    
    
    
    
    

    /** This class contains all native function definitions for JNI */
    public static final class JNI_LIB {

        // ----- Static initializer -----

        static {
            if(!ImageInfo.inImageCode()) {
                // Load the needed libraries
                if(OS.indexOf("win") < 0) {
                    System.loadLibrary("langkit_sigsegv_handler");
                }
                System.loadLibrary(
                    "lktlang_jni"
                );

                // Initialize the JNI library
                lkt_initialize();

                // Register the library finalizer
                Runtime.getRuntime().addShutdownHook(
                    new Thread(JNI_LIB::lkt_finalize)
                );
            }
        }

        // ----- Language specific functions -----

        


        // ----- Lifecycle functions ------

        /** Function to initialize the JNI library */
        public static native void lkt_initialize();

        /** Function to finalize the JNI library */
        public static native void lkt_finalize();

        // ----- Stack trace functions ------

        @CompilerDirectives.TruffleBoundary
        public static native int
        lkt_stack_trace_size(long trace);

        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper
        lkt_stack_trace_element(long trace, int index);

        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper
        lkt_create_stack_trace(PointerWrapper[] elements);

        @CompilerDirectives.TruffleBoundary
        public static native void
        lkt_destroy_stack_trace(long trace);

        @CompilerDirectives.TruffleBoundary
        public static native String
        lkt_symbolize_stack_trace(long trace);

        // ----- Exception functions ------

        /** Get the last langkit exception */
        @CompilerDirectives.TruffleBoundary
        public static native LangkitException lkt_get_last_exception();

        // ----- Text functions -----

        /** Create a new text from its content */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_create_text(
            byte[] utf32Content
        );

        /** Destroy the given text */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_destroy_text(
            Text text
        );

        // ----- Rewriting apply result functions -----

        /** Get the diagnostics from the rewriting apply result */
        @CompilerDirectives.TruffleBoundary
        public static native Diagnostic[]
        lkt_rewriting_get_result_diagnostics(
            int diagnosticsCount,
            long diagnosticsReference
        );

        /** Free the rewriting apply result structure */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_free_apply_result(
            RewritingApplyResult applyResult
        );

        // ----- File reader functions -----

        /** Decrease the reference counter of the given file reader */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_dec_ref_file_reader(
            FileReader fileReader
        );

        // ----- Unit provider functions -----

        /** Decrease the ref counter of the unit provider */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_dec_ref_unit_provider(
            UnitProvider unitProvider
        );

        // ----- Event handler functions -----

        /** Create a new event handler */
        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper lkt_create_event_handler(
            EventHandler.UnitRequestedCallback unitRequestedCallback,
            EventHandler.UnitParsedCallback unitParsedCallback
        );

        /** Decrease the ref counter of the event handler */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_dec_ref_event_handler(
            EventHandler eventHandler
        );

        // ----- Token functions -----

        /** Get the next token */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_token_next(
            Token token
        );

        /** Get the previous token */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_token_previous(
            Token token
        );

        /** Get if the tokens are equivalent */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_token_is_equivalent(
            Token left,
            Token right
        );

        /** Get text between the two tokens */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_token_range_text(
            Token start,
            Token end
        );

        // ----- Analysis context functions -----

        /** Create a new analysis context */
        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper lkt_create_analysis_context(
            String charset,
            FileReader fileReader,
            UnitProvider unitProvider,
            EventHandler eventHandler,
            boolean withTrivia,
            int tabstop
        );

        /** Increase the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_context_incref(
            long context
        );

        /** Decrease the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_context_decref(
            long context
        );

        // ----- Analysis unit functions -----

        /** Get the analysis unit from a file */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        lkt_get_analysis_unit_from_file(
            AnalysisContext context,
            String fileName,
            String charset,
            boolean reparse,
            int grammarRule
        );

        /** Get the analysis unit from a buffer */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        lkt_get_analysis_unit_from_buffer(
            AnalysisContext context,
            String fileName,
            String charset,
            String buffer,
            long bufferSize,
            int grammarRule
        );

        /** Get the analysis unit from the unit provider. */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        lkt_get_analysis_unit_from_provider(
            AnalysisContext context,
            Text name,
            int kind,
            String charset,
            boolean reparse
        );

        /** Get the root of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_unit_root(
            AnalysisUnit unit
        );

        /** Get the file name of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_unit_filename(
            AnalysisUnit unit
        );

        /** Get the token count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_unit_token_count(
            AnalysisUnit unit
        );

        /** Get the trivia count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_unit_trivia_count(
            AnalysisUnit unit
        );

        /** Get the first token of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_unit_first_token(
            AnalysisUnit unit
        );

        /** Get the last token of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_unit_last_token(
            AnalysisUnit unit
        );

        /** Get the context of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisContext lkt_unit_context(
            AnalysisUnit unit
        );

        /** Get the number of diagnostic in the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_unit_diagnostic_count(
            AnalysisUnit unit
        );

        /** Get the nth diagnostic of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Diagnostic lkt_unit_diagnostic(
            AnalysisUnit unit,
            int n
        );

        // ----- Rewriting context functions -----

        /** Start a rewriting session and return the new context */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingContext
        lkt_rewriting_start_rewriting(
            AnalysisContext analysisContext
        );

        /** Get the analysis context from the given rewriting context */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisContext
        lkt_rewriting_handle_to_context(
            RewritingContext rewritingContext
        );

        /** Get a pointer to the rewriting units owned by the context */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingUnit[] lkt_rewriting_unit_handles(
            RewritingContext rewritingContext
        );

        /** Create a node in the rewriting context and return it */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_create_node(
            RewritingContext rewritingContext,
            int nodeKind
        );

        /** Create a node in the rewriting context with the given children */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode
        lkt_rewriting_create_regular_node(
            RewritingContext rewritingContext,
            int nodeKind,
            RewritingNode[] children
        );

        /** Create a token node in the rewriting context and return it */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode
        lkt_rewriting_create_token_node(
            RewritingContext rewritingContext,
            int nodeKind,
            Text nodeText
        );

        /** Create a new node tree from the given template */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode
        lkt_rewriting_create_from_template(
            RewritingContext rewriting_context,
            Text template_text,
            RewritingNode[] arguments,
            int rule
        );

        /** Apply the rewriting session and close it fi success */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingApplyResult lkt_rewriting_apply(
            RewritingContext rewritingContext
        );

        /** Abort the rewriting session */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_abort_rewriting(
            RewritingContext rewritingContext
        );

        // ----- Rewriting unit functions -----

        /** Get the rewriting unit associated */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingUnit lkt_rewriting_unit_to_handle(
            AnalysisUnit analysisUnit
        );

        /** Get the analysis unit corresponding to the given rewriting unit */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit lkt_rewriting_handle_to_unit(
            RewritingUnit rewritingUnit
        );

        /** Get the root of the given rewriting unit */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_unit_root(
            RewritingUnit rewritingUnit
        );

        /** Set the root of the rewriting unit to the rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_unit_set_root(
            RewritingUnit rewritingUnit,
            RewritingNode rewritingNode
        );

        /** Unparse the given rewriting unit and return its textual value */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_rewriting_unit_unparse(
            RewritingUnit rewritingUnit
        );

        // ----- Rewriting node functions -----

        /** Get the rewriting node from the given parsed node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_node_to_handle(
            Entity entity
        );

        /** Get the parsed node from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_rewriting_handle_to_node(
            RewritingNode rewritingNode
        );

        /** Get the rewriting context from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingContext
        lkt_rewriting_node_to_context(
            RewritingNode rewriting_node
        );

        /** Clone the given rewriting node and return the copy */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_clone(
            RewritingNode toClone
        );

        /** Unparse the given rewriting node in the given text */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_rewriting_node_unparse(
            RewritingNode rewritingNode
        );

        /** Get the kind of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_rewriting_kind(
            RewritingNode rewritingNode
        );

        /** Get the rewriting node image */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_rewriting_node_image(
            RewritingNode rewritingNode
        );

        /** Return whether the node is tied to a rewriting unit */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_rewriting_tied(
            RewritingNode rewritingNode
        );

        /** Return the parent of the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_parent(
            RewritingNode rewritingNode
        );

        /** Get the rewriting node children */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode[] lkt_rewriting_children(
            RewritingNode rewritingNode
        );

        /** Get the child at the given member reference */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_child(
            RewritingNode parent,
            int childMemberReference
        );

        /** Set the given child at the given member reference */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_set_child(
            RewritingNode parent,
            int childMemberReference,
            RewritingNode newChild
        );

        /** Replace the rewriting node by the new one */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_replace(
            RewritingNode rewritingNode,
            RewritingNode newNode
        );

        /** Get the first child of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_first_child(
            RewritingNode parent
        );

        /** Get the last child of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_last_child(
            RewritingNode parent
        );

        /** Get the next child from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_next_child(
            RewritingNode rewritingNode
        );

        /** Get the previous child from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_previous_child(
            RewritingNode rewritingNode
        );

        /** Insert the provided rewriting node before the other node */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_insert_before(
            RewritingNode rewritingNode,
            RewritingNode toInsert
        );

        /** Insert the provided rewriting node after the other node */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_insert_after(
            RewritingNode rewritingNode,
            RewritingNode toInsert
        );

        /**
         * Insert the provided rewriting node at the beginning of the
         * children
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_first(
            RewritingNode rewritingNode,
            RewritingNode toInsert
        );

        /** Insert the provided rewriting node at the end of the children */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_last(
            RewritingNode rewritingNode,
            RewritingNode toInsert
        );

        /** Remove the given rewriting node from its list parent */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_remove_child(
            RewritingNode toRemove
        );

        /** Get the text of the rewriting token node */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_rewriting_text(
            RewritingNode rewritingNode
        );

        /** Set the text of the rewriting token node */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_set_text(
            RewritingNode rewritingNode,
            Text text
        );

        // ----- Iterator functions -----


        // ----- Node functions -----

        /** Return whether the two given entities are equal */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_node_is_equivalent(
            Entity entity_left,
            Entity entity_right
        );

        /** Get the hash of a node */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_node_hash(
            Entity entity
        );

        /** Get the node kind */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_node_kind(
            Entity entity
        );

        /** Get the node text */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_node_text(
            Entity entity
        );

        /** Get the node source location range */
        @CompilerDirectives.TruffleBoundary
        public static native SourceLocationRange lkt_node_sloc_range(
            Entity entity
        );

        /** Get the node children count */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_node_children_count(
            Entity entity
        );

        /** Get the node nth child */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_node_child(
            Entity entity,
            int n
        );

        /** Get if the node is a token node */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_node_is_token_node(
            Entity entity
        );

        /** Get the unit of the node */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit lkt_node_unit(
            Entity entity
        );

        /** Get the entity image of the node */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_node_image(
            Entity entity
        );

        // ----- Node fields accessors and properties -----

        
    

            

        /** Isomethod of lkt_lkt_node_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_parent(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_parents langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native LktNode[] lkt_lkt_node_parents(
            boolean
            with_self,
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_children langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native LktNode[] lkt_lkt_node_children(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_token_start langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_lkt_node_token_start(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_token_end langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_lkt_node_token_end(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_child_index langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_lkt_node_child_index(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_previous_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_previous_sibling(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_next_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_next_sibling(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit lkt_lkt_node_unit(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_is_ghost langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_lkt_node_is_ghost(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_full_sloc_image langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_lkt_node_full_sloc_image(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_completion_item_kind_to_int langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_lkt_node_completion_item_kind_to_int(
            CompletionItemKind
            kind,
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_set_solver_debug_mode langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_lkt_node_p_set_solver_debug_mode(
            boolean
            enable,
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_basic_trait_gen langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_basic_trait_gen(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_basic_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_basic_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_node_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_node_gen_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_node_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_indexable_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_indexable_gen_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_indexable_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_indexable_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_token_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_token_node_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_error_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_error_node_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_char_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_int_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_bool_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_bool_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_bigint_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_bigint_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_string_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_string_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_symbol_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_symbol_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_property_error_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_property_error_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_regexp_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_regexp_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_entity_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_entity_gen_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_entity_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_entity_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_logicvar_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_logicvar_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_equation_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_equation_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_array_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_array_gen_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_array_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_array_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_astlist_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_astlist_gen_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_astlist_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_astlist_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_node_builder_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_node_builder_gen_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_node_builder_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_node_builder_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_iterator_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_iterator_gen_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_iterator_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_iterator_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_analysis_unit_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_analysis_unit_gen_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_analysis_unit_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_analysis_unit_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_topmost_invalid_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_topmost_invalid_decl(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_nameres_diagnostics langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native SolverDiagnostic[] lkt_lkt_node_p_nameres_diagnostics(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_solve_enclosing_context langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native SolverResult lkt_lkt_node_p_solve_enclosing_context(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_xref_entry_point langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_lkt_node_p_xref_entry_point(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_complete langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native CompleteItem[] lkt_lkt_node_p_complete(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_argument_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_argument_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_argument_f_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_argument_f_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_base_lexer_case_rule_alt_f_send langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_lexer_case_rule_alt_f_send(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_cond_alt_f_cond_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_cond_alt_f_cond_exprs(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_base_match_branch_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_match_branch_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_base_match_branch_p_match_part langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_match_branch_p_match_part(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_match_branch_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_match_branch_f_decl(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_pattern_match_branch_f_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_pattern_match_branch_f_pattern(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_block_expr_clause_f_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_block_expr_clause_f_clause(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_class_qualifier_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_class_qualifier_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_decl_f_syn_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_f_syn_name(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_custom_image langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_decl_p_custom_image(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_decl_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_decl_p_decl_type_name(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_def_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DefId[] lkt_decl_p_def_ids(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_as_bare_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_p_as_bare_decl(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_get_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_p_get_type(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_get_cast_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_p_get_cast_type(
            Entity
            cast_to,
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_get_keep_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_p_get_keep_type(
            Entity
            keep_type,
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_get_suffix_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_p_get_suffix_type(
            Entity
            prefix_type,
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_is_generic langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_decl_p_is_generic(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_return_type_is_instantiated langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_decl_p_return_type_is_instantiated(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_is_instantiated langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_decl_p_is_instantiated(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Symbol lkt_decl_p_name(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_full_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_decl_p_full_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_base_grammar_rule_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_grammar_rule_decl_f_expr(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_explicitly_typed_decl_f_decl_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_explicitly_typed_decl_f_decl_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_component_decl_f_default_val langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_component_decl_f_default_val(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_field_decl_f_trait_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_field_decl_f_trait_ref(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_fun_param_decl_f_decl_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_fun_param_decl_f_decl_annotations(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_val_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_val_decl_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_fun_decl_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_fun_decl_f_params(
            Entity node
        );
            

        /** Isomethod of lkt_fun_decl_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_fun_decl_f_return_type(
            Entity node
        );
            

        /** Isomethod of lkt_fun_decl_f_trait_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_fun_decl_f_trait_ref(
            Entity node
        );
            

        /** Isomethod of lkt_fun_decl_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_fun_decl_f_body(
            Entity node
        );
            

        /** Isomethod of lkt_fun_decl_p_is_dynamic_combiner langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_fun_decl_p_is_dynamic_combiner(
            Entity node
        );
            

        /** Isomethod of lkt_fun_decl_p_find_all_overrides langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native FunDecl[] lkt_fun_decl_p_find_all_overrides(
            AnalysisUnit[]
            units,
            Entity node
        );

        
    

            

        /** Isomethod of lkt_env_spec_decl_f_actions langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_env_spec_decl_f_actions(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_generic_decl_f_generic_param_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_decl_f_generic_param_decls(
            Entity node
        );
            

        /** Isomethod of lkt_generic_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_decl_f_decl(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_decl_f_rules(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_decl_f_rules(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_family_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_family_decl_f_rules(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_type_decl_f_traits langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_decl_f_traits(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_f_syn_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_decl_f_syn_base_type(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_def_id langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_decl_p_def_id(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_decl_p_base_type(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_base_type_if_entity langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_decl_p_base_type_if_entity(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_generic_param_type_decl_f_has_class langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_param_type_decl_f_has_class(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_named_type_decl_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_named_type_decl_f_decls(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_enum_class_decl_f_branches langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_enum_class_decl_f_branches(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_enum_type_decl_f_literals langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_enum_type_decl_f_literals(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_decl_annotation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_annotation_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_decl_annotation_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_annotation_f_args(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_decl_annotation_args_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_annotation_args_f_args(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_elsif_branch_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_elsif_branch_f_cond_expr(
            Entity node
        );
            

        /** Isomethod of lkt_elsif_branch_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_elsif_branch_f_then_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_enum_class_case_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_enum_class_case_f_decls(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_excludes_null_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_excludes_null_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_expr_p_get_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_expr_p_get_type(
            Entity node
        );
            

        /** Isomethod of lkt_expr_p_get_generic_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_expr_p_get_generic_type(
            Entity node
        );
            

        /** Isomethod of lkt_expr_p_get_expected_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_expr_p_get_expected_type(
            Entity node
        );
            

        /** Isomethod of lkt_expr_p_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_expr_p_referenced_decl(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_any_of_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_any_of_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_any_of_f_values langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_any_of_f_values(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_array_literal_f_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_array_literal_f_exprs(
            Entity node
        );
            

        /** Isomethod of lkt_array_literal_f_element_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_array_literal_f_element_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_base_call_expr_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_call_expr_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_base_call_expr_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_call_expr_f_args(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_bin_op_f_left langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_bin_op_f_left(
            Entity node
        );
            

        /** Isomethod of lkt_bin_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_bin_op_f_op(
            Entity node
        );
            

        /** Isomethod of lkt_bin_op_f_right langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_bin_op_f_right(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_block_expr_f_clauses langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_block_expr_f_clauses(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_cast_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_cast_expr_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_cast_expr_f_null_cond langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_cast_expr_f_null_cond(
            Entity node
        );
            

        /** Isomethod of lkt_cast_expr_f_excludes_null langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_cast_expr_f_excludes_null(
            Entity node
        );
            

        /** Isomethod of lkt_cast_expr_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_cast_expr_f_dest_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_dot_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_dot_expr_f_prefix(
            Entity node
        );
            

        /** Isomethod of lkt_dot_expr_f_null_cond langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_dot_expr_f_null_cond(
            Entity node
        );
            

        /** Isomethod of lkt_dot_expr_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_dot_expr_f_suffix(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_error_on_null_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_error_on_null_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_generic_instantiation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_instantiation_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_generic_instantiation_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_instantiation_f_args(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_grammar_discard_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_discard_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_dont_skip_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_dont_skip_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_dont_skip_f_dont_skip langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_dont_skip_f_dont_skip(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_list_f_list_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_f_list_type(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_list_f_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_f_kind(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_list_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_list_f_sep langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_f_sep(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_null_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_null_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_opt_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_error_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_opt_error_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_error_group_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_opt_error_group_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_group_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_opt_group_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_or_expr_f_sub_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_or_expr_f_sub_exprs(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_pick_f_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_pick_f_exprs(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_grammar_predicate_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_predicate_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_predicate_f_prop_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_predicate_f_prop_ref(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_rule_ref_f_node_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_rule_ref_f_node_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_skip_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_skip_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_stop_cut_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_stop_cut_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_parse_node_expr_f_node_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_parse_node_expr_f_node_name(
            Entity node
        );
            

        /** Isomethod of lkt_parse_node_expr_f_sub_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_parse_node_expr_f_sub_exprs(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_token_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DecodedStringValue lkt_token_lit_p_denoted_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_token_no_case_lit_f_lit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_token_no_case_lit_f_lit(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_token_pattern_concat_f_left langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_token_pattern_concat_f_left(
            Entity node
        );
            

        /** Isomethod of lkt_token_pattern_concat_f_right langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_token_pattern_concat_f_right(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_token_pattern_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DecodedStringValue lkt_token_pattern_lit_p_denoted_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_token_ref_f_token_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_token_ref_f_token_name(
            Entity node
        );
            

        /** Isomethod of lkt_token_ref_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_token_ref_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_id_p_custom_image langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_id_p_custom_image(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_def_id_p_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_def_id_p_name(
            Entity node
        );
            

        /** Isomethod of lkt_def_id_p_get_implementatinons langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DefId[] lkt_def_id_p_get_implementatinons(
            AnalysisUnit[]
            units,
            Entity node
        );
            

        /** Isomethod of lkt_def_id_p_decl_detail langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_def_id_p_decl_detail(
            Entity node
        );
            

        /** Isomethod of lkt_def_id_p_completion_item_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_def_id_p_completion_item_kind(
            Entity node
        );
            

        /** Isomethod of lkt_def_id_p_doc langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_def_id_p_doc(
            Entity node
        );
            

        /** Isomethod of lkt_def_id_p_find_all_references langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native RefResult[] lkt_def_id_p_find_all_references(
            AnalysisUnit[]
            units,
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_ref_id_p_referenced_defining_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_ref_id_p_referenced_defining_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_if_expr_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_if_expr_f_cond_expr(
            Entity node
        );
            

        /** Isomethod of lkt_if_expr_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_if_expr_f_then_expr(
            Entity node
        );
            

        /** Isomethod of lkt_if_expr_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_if_expr_f_alternatives(
            Entity node
        );
            

        /** Isomethod of lkt_if_expr_f_else_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_if_expr_f_else_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_isa_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_isa_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_isa_f_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_isa_f_pattern(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_keep_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_keep_expr_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_keep_expr_f_null_cond langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_keep_expr_f_null_cond(
            Entity node
        );
            

        /** Isomethod of lkt_keep_expr_f_keep_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_keep_expr_f_keep_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lambda_expr_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lambda_expr_f_params(
            Entity node
        );
            

        /** Isomethod of lkt_lambda_expr_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lambda_expr_f_return_type(
            Entity node
        );
            

        /** Isomethod of lkt_lambda_expr_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lambda_expr_f_body(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_char_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DecodedCharValue lkt_char_lit_p_denoted_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_null_lit_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_null_lit_f_dest_type(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_string_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DecodedStringValue lkt_string_lit_p_denoted_value(
            Entity node
        );
            

        /** Isomethod of lkt_string_lit_p_is_prefixed_string langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_string_lit_p_is_prefixed_string(
            Entity node
        );
            

        /** Isomethod of lkt_string_lit_p_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Char lkt_string_lit_p_prefix(
            Entity node
        );
            

        /** Isomethod of lkt_string_lit_p_is_regexp_literal langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_string_lit_p_is_regexp_literal(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_block_string_lit_f_lines langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_block_string_lit_f_lines(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_logic_assign_f_dest_var langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_logic_assign_f_dest_var(
            Entity node
        );
            

        /** Isomethod of lkt_logic_assign_f_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_logic_assign_f_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_logic_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_logic_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_logic_propagate_f_dest_var langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_logic_propagate_f_dest_var(
            Entity node
        );
            

        /** Isomethod of lkt_logic_propagate_f_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_logic_propagate_f_call(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_logic_unify_f_lhs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_logic_unify_f_lhs(
            Entity node
        );
            

        /** Isomethod of lkt_logic_unify_f_rhs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_logic_unify_f_rhs(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_match_expr_f_match_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_match_expr_f_match_expr(
            Entity node
        );
            

        /** Isomethod of lkt_match_expr_f_branches langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_match_expr_f_branches(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_not_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_not_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_paren_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_paren_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_raise_expr_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_raise_expr_f_dest_type(
            Entity node
        );
            

        /** Isomethod of lkt_raise_expr_f_except_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_raise_expr_f_except_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_subscript_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_subscript_expr_f_prefix(
            Entity node
        );
            

        /** Isomethod of lkt_subscript_expr_f_null_cond langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_subscript_expr_f_null_cond(
            Entity node
        );
            

        /** Isomethod of lkt_subscript_expr_f_index langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_subscript_expr_f_index(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_try_expr_f_try_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_try_expr_f_try_expr(
            Entity node
        );
            

        /** Isomethod of lkt_try_expr_f_or_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_try_expr_f_or_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_un_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_un_op_f_op(
            Entity node
        );
            

        /** Isomethod of lkt_un_op_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_un_op_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_full_decl_f_doc langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_full_decl_f_doc(
            Entity node
        );
            

        /** Isomethod of lkt_full_decl_f_decl_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_full_decl_f_decl_annotations(
            Entity node
        );
            

        /** Isomethod of lkt_full_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_full_decl_f_decl(
            Entity node
        );
            

        /** Isomethod of lkt_full_decl_p_has_annotation langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_full_decl_p_has_annotation(
            Symbol
            name,
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_list_sep_f_token langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_sep_f_token(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_list_sep_f_extra langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_sep_f_extra(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_import_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_import_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_import_p_referenced_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit lkt_import_p_referenced_unit(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_langkit_root_f_imports langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_langkit_root_f_imports(
            Entity node
        );
            

        /** Isomethod of lkt_langkit_root_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_langkit_root_f_decls(
            Entity node
        );
            

        /** Isomethod of lkt_langkit_root_p_fetch_prelude langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit lkt_langkit_root_p_fetch_prelude(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_lexer_case_rule_f_alts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_f_alts(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_send_f_sent langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_send_f_sent(
            Entity node
        );
            

        /** Isomethod of lkt_lexer_case_rule_send_f_match_size langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_send_f_match_size(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_null_cond_qualifier_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_null_cond_qualifier_p_as_bool(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_binding_pattern_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_binding_pattern_f_decl(
            Entity node
        );
            

        /** Isomethod of lkt_binding_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_binding_pattern_f_sub_pattern(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_ellipsis_pattern_f_binding langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_ellipsis_pattern_f_binding(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_extended_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_extended_pattern_f_sub_pattern(
            Entity node
        );
            

        /** Isomethod of lkt_extended_pattern_f_details langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_extended_pattern_f_details(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_filtered_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_filtered_pattern_f_sub_pattern(
            Entity node
        );
            

        /** Isomethod of lkt_filtered_pattern_f_predicate langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_filtered_pattern_f_predicate(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_list_pattern_f_sub_patterns langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_list_pattern_f_sub_patterns(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_not_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_not_pattern_f_sub_pattern(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_or_pattern_f_left_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_or_pattern_f_left_sub_pattern(
            Entity node
        );
            

        /** Isomethod of lkt_or_pattern_f_right_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_or_pattern_f_right_sub_pattern(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_paren_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_paren_pattern_f_sub_pattern(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_tuple_pattern_f_sub_patterns langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_tuple_pattern_f_sub_patterns(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_type_pattern_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_pattern_f_type_name(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_field_pattern_detail_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_field_pattern_detail_f_id(
            Entity node
        );
            

        /** Isomethod of lkt_field_pattern_detail_f_expected_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_field_pattern_detail_f_expected_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_property_pattern_detail_f_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_property_pattern_detail_f_call(
            Entity node
        );
            

        /** Isomethod of lkt_property_pattern_detail_f_expected_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_property_pattern_detail_f_expected_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_selector_pattern_detail_f_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_selector_pattern_detail_f_call(
            Entity node
        );
            

        /** Isomethod of lkt_selector_pattern_detail_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_selector_pattern_detail_f_sub_pattern(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_selector_call_f_quantifier langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_selector_call_f_quantifier(
            Entity node
        );
            

        /** Isomethod of lkt_selector_call_f_binding langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_selector_call_f_binding(
            Entity node
        );
            

        /** Isomethod of lkt_selector_call_f_selector_call langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_selector_call_f_selector_call(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_type_ref_p_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_ref_p_referenced_decl(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_function_type_ref_f_param_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_function_type_ref_f_param_types(
            Entity node
        );
            

        /** Isomethod of lkt_function_type_ref_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_function_type_ref_f_return_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_generic_type_ref_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_type_ref_f_type_name(
            Entity node
        );
            

        /** Isomethod of lkt_generic_type_ref_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_type_ref_f_args(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_simple_type_ref_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_simple_type_ref_f_type_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_var_bind_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_var_bind_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_var_bind_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_var_bind_f_expr(
            Entity node
        );


    }


    // ==========
    // Defining the Native-Image bindings library
    // ==========

    
    
    
    
    
    
    

    /** This class contains the directives for the shared lib loading */
    public static final class LibDirectives implements CContext.Directives {
        @Override
        public List<String> getHeaderFiles() {
            List<String> res = new ArrayList<>();
            res.add("<liblktlang.h>");
            res.add("<stdlib.h>");
            return res;
        }

        @Override
        public List<String> getLibraries() {
            List<String> res = new ArrayList<>();
            res.add("lktlang");
            return res;
        }
    }

    // ===== Language specific structures =====

    


    // ===== Constant structures =====

    /** The structure for the langkit exceptions */
    @CContext(LibDirectives.class)
    @CStruct("lkt_exception")
    public interface LangkitExceptionNative extends PointerBase {
        @CField("kind") public int get_kind();
        @CField("kind") public void set_kind(
            int kind
        );

        @CField("information") public CCharPointer get_information();
        @CField("information") public void set_information(
            CCharPointer information
        );

        @CField("stack_trace") public PointerBase get_stack_trace();
        @CField("stack_trace") public void set_stack_trace(
            PointerBase stack_trace
        );
    }

    /** The big integers are just pointers */
    public interface BigIntegerNative extends Pointer {}

    /** The structure for the symbols */
    @CContext(LibDirectives.class)
    @CStruct("lkt_symbol_type")
    public interface SymbolNative extends PointerBase {
        @CField("thin_sym") public int get_thin_sym();
        @CField("thin_sym") public void set_thin_sym(
            int data
        );

        @CField("table") public VoidPointer get_table();
        @CField("table") public void set_table(
            VoidPointer bounds
        );
    }

    /** The string wrappers are just pointers */
    public interface StringNative extends Pointer {}

    /** The structure for the text */
    @CContext(LibDirectives.class)
    @CStruct("lkt_text")
    public interface TextNative extends PointerBase {
        @CField("chars") public CIntPointer get_chars();
        @CField("chars") public void set_chars(
            CIntPointer chars
        );

        @CField("length") public long get_length();
        @CField("length") public void set_length(
            long length
        );

        @CField("is_allocated") public int get_is_allocated();
        @CField("is_allocated") public void set_is_allocated(
            int is_allocated
        );
    }

    /** The structure for the source locations */
    @CContext(LibDirectives.class)
    @CStruct("lkt_source_location")
    public interface SourceLocationNative extends PointerBase {
        @CField("line") public int get_line();
        @CField("line") public void set_line(
            int line
        );

        @CField("column") public short get_column();
        @CField("column") public void set_column(
            short column
        );
    }

    /** The structure for the source location ranges */
    @CContext(LibDirectives.class)
    @CStruct("lkt_source_location_range")
    public interface SourceLocationRangeNative extends PointerBase {
        @CField("start.line") public int get_start_line();
        @CField("start.line") public void set_start_line(
            int start_line
        );

        @CField("start.column") public short get_start_column();
        @CField("start.column") public void set_start_column(
            short start_column
        );

        @CField("end.line") public int get_end_line();
        @CField("end.line") public void set_end_line(
            int end_line
        );

        @CField("end.column") public short get_end_column();
        @CField("end.column") public void set_end_column(
            short end_column
        );
    }

    /** The structure for the diagnostic */
    @CContext(LibDirectives.class)
    @CStruct("lkt_diagnostic")
    public interface DiagnosticNative extends PointerBase {
        @CField("sloc_range.start.line") public int get_start_line();
        @CField("sloc_range.start.line") public void set_start_line(
            int start_line
        );

        @CField("sloc_range.start.column") public short get_start_column();
        @CField("sloc_range.start.column") public void set_start_column(
            short start_column
        );

        @CField("sloc_range.end.line") public int get_end_line();
        @CField("sloc_range.end.line") public void set_end_line(
            int end_line
        );

        @CField("sloc_range.end.column") public short get_end_column();
        @CField("sloc_range.end.column") public void set_end_column(
            short end_column
        );

        @CField("message.chars") public CIntPointer get_message_chars();
        @CField("message.chars") public void set_message_chars(
            CIntPointer chars
        );

        @CField("message.length") public long get_message_length();
        @CField("message.length") public void set_message_length(
            long length
        );

        @CField("message.is_allocated") public int get_message_is_allocated();
        @CField("message.is_allocated") public void set_message_is_allocated(
            int is_allocated
        );
    }

    /** The file reader is just a pointer */
    public interface FileReaderNative extends Pointer {}

    /** The unit provider is just a pointer */
    public interface UnitProviderNative extends Pointer {}

    /** The event handler is just a pointer */
    public interface EventHandlerNative extends Pointer {}

    /** The event handler unit requested callback type */
    public interface UnitRequestedFunctionPointer extends CFunctionPointer {
        @InvokeCFunctionPointer
        void invoke(
            VoidPointer data,
            AnalysisContextNative context,
            TextNative name,
            AnalysisUnitNative from,
            boolean found,
            boolean is_not_found_error
        );
    }

    /** The event handler unit parsed callback type */
    public interface UnitParsedFunctionPointer extends CFunctionPointer {
        @InvokeCFunctionPointer
        void invoke(
            VoidPointer data,
            AnalysisContextNative context,
            AnalysisUnitNative unit,
            boolean reparsed
        );
    }

    /** Anonymous structure for the token data handler */
    @RawStructure
    public interface TokenDataHandlerNative extends PointerBase {
        @RawField public long get_version();
        @RawField public void set_version(long version);
    }

    /** The structure representing a token */
    @CContext(LibDirectives.class)
    @CStruct("lkt_token")
    public interface TokenNative extends PointerBase {
        @CField("context") public AnalysisContextNative get_context();
        @CField("context") public void set_context(
            AnalysisContextNative context
        );

        @CField("token_data") public TokenDataHandlerNative get_data();
        @CField("token_data") public void set_data(
            TokenDataHandlerNative data
        );

        @CField("token_index") public int get_token_index();
        @CField("token_index") public void set_token_index(
            int token_index
        );

        @CField("trivia_index") public int get_trivia_index();
        @CField("trivia_index") public void set_trivia_index(
            int trivia_index
        );
    }

    /** Anonymous strucutre for analysis context */
    @RawStructure
    public interface AnalysisContextNative extends PointerBase {
        @RawField public long get_serial_number();
        @RawField public void set_serial_number(long serial_number);
    }

    /** Anonymous strucutre for analysis unit */
    @RawStructure
    public interface AnalysisUnitNative extends PointerBase {
        @RawField public long get_version_number();
        @RawField public void set_version_number(long version_number);
    }

    /** The structure for reswriting apply results */
    @CContext(LibDirectives.class)
    @CStruct("lkt_rewriting_apply_result")
    public interface RewritingApplyResultNative extends PointerBase {
        @CField("success") public int get_success();
        @CField("success") public void set_success(int success);

        @CField("unit") public AnalysisUnitNative get_unit();
        @CField("unit") public void set_unit(AnalysisUnitNative unit);

        @CField("diagnostics_count") public int get_diagnostics_count();
        @CField("diagnostics_count") public void set_diagnostics_count(
            int diagnostics_count
        );

        @CField("diagnostics") public DiagnosticNative get_diagnostics();
        @CField("diagnostics") public void set_diagnostics(
            DiagnosticNative diagnostics
        );
    }

    /** The rewriting context type is just a pointer */
    public interface RewritingContextNative extends Pointer {}

    /** The rewriting unit native type is just a pointer */
    public interface RewritingUnitNative extends Pointer {}

    /** The rewriting node native type is just a pointer */
    public interface RewritingNodeNative extends Pointer {}

    // ===== Generated structures =====

        
    
    

    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_metadata")
    public interface MetadataNative extends PointerBase {
        @CField("dummy") public byte get_dummy();
        @CField("dummy") public void set_dummy(byte dummy);
    }

        
    
    

    /** The structure for the langkit lkt_internal_entity_info */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_entity_info")
    public interface EntityInfoNative extends PointerBase {
        @CField("md.dummy")
        public byte
        get_md_dummy();

        @CField("md.dummy")
        public void
        set_md_dummy(
            byte val
        );

        @CFieldAddress("md.dummy")
        public <T extends PointerBase> T address_md_dummy();

        @CField("rebindings")
        public Pointer
        get_rebindings();

        @CField("rebindings")
        public void
        set_rebindings(
            Pointer val
        );

        @CFieldAddress("rebindings")
        public <T extends PointerBase> T address_rebindings();

        @CField("from_rebound")
        public byte
        get_from_rebound();

        @CField("from_rebound")
        public void
        set_from_rebound(
            byte val
        );

        @CFieldAddress("from_rebound")
        public <T extends PointerBase> T address_from_rebound();


            
        @CFieldAddress("md")
        public MetadataNative
        address_md();

    }

    
    

    /** The structure for the langkit lkt_node */
    @CContext(LibDirectives.class)
    @CStruct("lkt_node")
    public interface EntityNative extends PointerBase {
        @CField("node")
        public Pointer
        get_node();

        @CField("node")
        public void
        set_node(
            Pointer val
        );

        @CFieldAddress("node")
        public <T extends PointerBase> T address_node();

        @CField("info.md.dummy")
        public byte
        get_info_md_dummy();

        @CField("info.md.dummy")
        public void
        set_info_md_dummy(
            byte val
        );

        @CFieldAddress("info.md.dummy")
        public <T extends PointerBase> T address_info_md_dummy();

        @CField("info.rebindings")
        public Pointer
        get_info_rebindings();

        @CField("info.rebindings")
        public void
        set_info_rebindings(
            Pointer val
        );

        @CFieldAddress("info.rebindings")
        public <T extends PointerBase> T address_info_rebindings();

        @CField("info.from_rebound")
        public byte
        get_info_from_rebound();

        @CField("info.from_rebound")
        public void
        set_info_from_rebound(
            byte val
        );

        @CFieldAddress("info.from_rebound")
        public <T extends PointerBase> T address_info_from_rebound();


            
        @CFieldAddress("info.md")
        public EntityInfoNative
        address_info();

    }

        
    
    

    /** The structure for the langkit lkt_internal_complete_item */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_complete_item")
    public interface CompleteItemNative extends PointerBase {
        @CField("declaration.node")
        public Pointer
        get_declaration_node();

        @CField("declaration.node")
        public void
        set_declaration_node(
            Pointer val
        );

        @CFieldAddress("declaration.node")
        public <T extends PointerBase> T address_declaration_node();

        @CField("declaration.info.md.dummy")
        public byte
        get_declaration_info_md_dummy();

        @CField("declaration.info.md.dummy")
        public void
        set_declaration_info_md_dummy(
            byte val
        );

        @CFieldAddress("declaration.info.md.dummy")
        public <T extends PointerBase> T address_declaration_info_md_dummy();

        @CField("declaration.info.rebindings")
        public Pointer
        get_declaration_info_rebindings();

        @CField("declaration.info.rebindings")
        public void
        set_declaration_info_rebindings(
            Pointer val
        );

        @CFieldAddress("declaration.info.rebindings")
        public <T extends PointerBase> T address_declaration_info_rebindings();

        @CField("declaration.info.from_rebound")
        public byte
        get_declaration_info_from_rebound();

        @CField("declaration.info.from_rebound")
        public void
        set_declaration_info_from_rebound(
            byte val
        );

        @CFieldAddress("declaration.info.from_rebound")
        public <T extends PointerBase> T address_declaration_info_from_rebound();


            
        @CFieldAddress("declaration.node")
        public EntityNative
        address_declaration();

    }

        
    
    

    /** The structure for the langkit lkt_internal_decoded_char_value */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_decoded_char_value")
    public interface DecodedCharValueNative extends PointerBase {
        @CField("value")
        public int
        get_value();

        @CField("value")
        public void
        set_value(
            int val
        );

        @CFieldAddress("value")
        public <T extends PointerBase> T address_value();

        @CField("has_error")
        public byte
        get_has_error();

        @CField("has_error")
        public void
        set_has_error(
            byte val
        );

        @CFieldAddress("has_error")
        public <T extends PointerBase> T address_has_error();

        @CField("error_sloc")
        public SourceLocationNative
        get_error_sloc();

        @CField("error_sloc")
        public void
        set_error_sloc(
            SourceLocationNative val
        );

        @CFieldAddress("error_sloc")
        public <T extends PointerBase> T address_error_sloc();

        @CField("error_message")
        public StringNative
        get_error_message();

        @CField("error_message")
        public void
        set_error_message(
            StringNative val
        );

        @CFieldAddress("error_message")
        public <T extends PointerBase> T address_error_message();


    }

        
    
    

    /** The structure for the langkit lkt_internal_decoded_string_value */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_decoded_string_value")
    public interface DecodedStringValueNative extends PointerBase {
        @CField("value")
        public StringNative
        get_value();

        @CField("value")
        public void
        set_value(
            StringNative val
        );

        @CFieldAddress("value")
        public <T extends PointerBase> T address_value();

        @CField("has_error")
        public byte
        get_has_error();

        @CField("has_error")
        public void
        set_has_error(
            byte val
        );

        @CFieldAddress("has_error")
        public <T extends PointerBase> T address_has_error();

        @CField("error_sloc")
        public SourceLocationNative
        get_error_sloc();

        @CField("error_sloc")
        public void
        set_error_sloc(
            SourceLocationNative val
        );

        @CFieldAddress("error_sloc")
        public <T extends PointerBase> T address_error_sloc();

        @CField("error_message")
        public StringNative
        get_error_message();

        @CField("error_message")
        public void
        set_error_message(
            StringNative val
        );

        @CFieldAddress("error_message")
        public <T extends PointerBase> T address_error_message();


    }

        
        
        
        
    
    

    /** The structure for the langkit lkt_internal_logic_context */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_logic_context")
    public interface LogicContextNative extends PointerBase {
        @CField("ref_node.node")
        public Pointer
        get_ref_node_node();

        @CField("ref_node.node")
        public void
        set_ref_node_node(
            Pointer val
        );

        @CFieldAddress("ref_node.node")
        public <T extends PointerBase> T address_ref_node_node();

        @CField("ref_node.info.md.dummy")
        public byte
        get_ref_node_info_md_dummy();

        @CField("ref_node.info.md.dummy")
        public void
        set_ref_node_info_md_dummy(
            byte val
        );

        @CFieldAddress("ref_node.info.md.dummy")
        public <T extends PointerBase> T address_ref_node_info_md_dummy();

        @CField("ref_node.info.rebindings")
        public Pointer
        get_ref_node_info_rebindings();

        @CField("ref_node.info.rebindings")
        public void
        set_ref_node_info_rebindings(
            Pointer val
        );

        @CFieldAddress("ref_node.info.rebindings")
        public <T extends PointerBase> T address_ref_node_info_rebindings();

        @CField("ref_node.info.from_rebound")
        public byte
        get_ref_node_info_from_rebound();

        @CField("ref_node.info.from_rebound")
        public void
        set_ref_node_info_from_rebound(
            byte val
        );

        @CFieldAddress("ref_node.info.from_rebound")
        public <T extends PointerBase> T address_ref_node_info_from_rebound();

        @CField("decl_node.node")
        public Pointer
        get_decl_node_node();

        @CField("decl_node.node")
        public void
        set_decl_node_node(
            Pointer val
        );

        @CFieldAddress("decl_node.node")
        public <T extends PointerBase> T address_decl_node_node();

        @CField("decl_node.info.md.dummy")
        public byte
        get_decl_node_info_md_dummy();

        @CField("decl_node.info.md.dummy")
        public void
        set_decl_node_info_md_dummy(
            byte val
        );

        @CFieldAddress("decl_node.info.md.dummy")
        public <T extends PointerBase> T address_decl_node_info_md_dummy();

        @CField("decl_node.info.rebindings")
        public Pointer
        get_decl_node_info_rebindings();

        @CField("decl_node.info.rebindings")
        public void
        set_decl_node_info_rebindings(
            Pointer val
        );

        @CFieldAddress("decl_node.info.rebindings")
        public <T extends PointerBase> T address_decl_node_info_rebindings();

        @CField("decl_node.info.from_rebound")
        public byte
        get_decl_node_info_from_rebound();

        @CField("decl_node.info.from_rebound")
        public void
        set_decl_node_info_from_rebound(
            byte val
        );

        @CFieldAddress("decl_node.info.from_rebound")
        public <T extends PointerBase> T address_decl_node_info_from_rebound();


            
        @CFieldAddress("ref_node.node")
        public EntityNative
        address_ref_node();

            
        @CFieldAddress("decl_node.node")
        public EntityNative
        address_decl_node();

    }

        
        
        
    
    

    /** The structure for the langkit lkt_internal_ref_result */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_ref_result")
    public interface RefResultNative extends PointerBase {
        @CField("ref.node")
        public Pointer
        get_ref_node();

        @CField("ref.node")
        public void
        set_ref_node(
            Pointer val
        );

        @CFieldAddress("ref.node")
        public <T extends PointerBase> T address_ref_node();

        @CField("ref.info.md.dummy")
        public byte
        get_ref_info_md_dummy();

        @CField("ref.info.md.dummy")
        public void
        set_ref_info_md_dummy(
            byte val
        );

        @CFieldAddress("ref.info.md.dummy")
        public <T extends PointerBase> T address_ref_info_md_dummy();

        @CField("ref.info.rebindings")
        public Pointer
        get_ref_info_rebindings();

        @CField("ref.info.rebindings")
        public void
        set_ref_info_rebindings(
            Pointer val
        );

        @CFieldAddress("ref.info.rebindings")
        public <T extends PointerBase> T address_ref_info_rebindings();

        @CField("ref.info.from_rebound")
        public byte
        get_ref_info_from_rebound();

        @CField("ref.info.from_rebound")
        public void
        set_ref_info_from_rebound(
            byte val
        );

        @CFieldAddress("ref.info.from_rebound")
        public <T extends PointerBase> T address_ref_info_from_rebound();


            
        @CFieldAddress("ref.node")
        public EntityNative
        address_ref();

    }

        
    
    

    /** The structure for the langkit lkt_internal_solver_diagnostic */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_solver_diagnostic")
    public interface SolverDiagnosticNative extends PointerBase {
        @CField("message_template")
        public StringNative
        get_message_template();

        @CField("message_template")
        public void
        set_message_template(
            StringNative val
        );

        @CFieldAddress("message_template")
        public <T extends PointerBase> T address_message_template();

        @CField("args")
        public LktNodeArrayNative
        get_args();

        @CField("args")
        public void
        set_args(
            LktNodeArrayNative val
        );

        @CFieldAddress("args")
        public <T extends PointerBase> T address_args();

        @CField("location")
        public Pointer
        get_location();

        @CField("location")
        public void
        set_location(
            Pointer val
        );

        @CFieldAddress("location")
        public <T extends PointerBase> T address_location();

        @CField("contexts")
        public LogicContextArrayNative
        get_contexts();

        @CField("contexts")
        public void
        set_contexts(
            LogicContextArrayNative val
        );

        @CFieldAddress("contexts")
        public <T extends PointerBase> T address_contexts();

        @CField("round")
        public int
        get_round();

        @CField("round")
        public void
        set_round(
            int val
        );

        @CFieldAddress("round")
        public <T extends PointerBase> T address_round();


    }

        
    
    

    /** The structure for the langkit lkt_internal_solver_result */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_solver_result")
    public interface SolverResultNative extends PointerBase {
        @CField("success")
        public byte
        get_success();

        @CField("success")
        public void
        set_success(
            byte val
        );

        @CFieldAddress("success")
        public <T extends PointerBase> T address_success();

        @CField("diagnostics")
        public SolverDiagnosticArrayNative
        get_diagnostics();

        @CField("diagnostics")
        public void
        set_diagnostics(
            SolverDiagnosticArrayNative val
        );

        @CFieldAddress("diagnostics")
        public <T extends PointerBase> T address_diagnostics();


    }


    // ===== Generated arrays =====

    
    

    /**
     * The native structure of the lkt_internal_complete_item_array Langkit array.
     * This structure is described as a `RawStruct` because native-image
     * doesn't handle flexible array members. Regarding this statement, array
     * elements are accessible through the `addressOfItems` method which
     * returns a pointer to the first value.
     */
    @RawStructure
    public interface CompleteItemArrayNative extends PointerBase {
        @RawField
        public int getASize();
        @RawField
        public void setASize(int n);

        @RawField
        public int getBRefCount();
        @RawField
        public void setBRefCount(int refCount);

        @RawField
        public byte getItems();
        @RawField
        public void setItems(byte items);
        @RawFieldAddress
        public <T extends PointerBase> T addressOfItems();
    }

    
    

    /**
     * The native structure of the lkt_node_array Langkit array.
     * This structure is described as a `RawStruct` because native-image
     * doesn't handle flexible array members. Regarding this statement, array
     * elements are accessible through the `addressOfItems` method which
     * returns a pointer to the first value.
     */
    @RawStructure
    public interface LktNodeArrayNative extends PointerBase {
        @RawField
        public int getASize();
        @RawField
        public void setASize(int n);

        @RawField
        public int getBRefCount();
        @RawField
        public void setBRefCount(int refCount);

        @RawField
        public byte getItems();
        @RawField
        public void setItems(byte items);
        @RawFieldAddress
        public <T extends PointerBase> T addressOfItems();
    }

    
    

    /**
     * The native structure of the lkt_node_array Langkit array.
     * This structure is described as a `RawStruct` because native-image
     * doesn't handle flexible array members. Regarding this statement, array
     * elements are accessible through the `addressOfItems` method which
     * returns a pointer to the first value.
     */
    @RawStructure
    public interface DefIdArrayNative extends PointerBase {
        @RawField
        public int getASize();
        @RawField
        public void setASize(int n);

        @RawField
        public int getBRefCount();
        @RawField
        public void setBRefCount(int refCount);

        @RawField
        public byte getItems();
        @RawField
        public void setItems(byte items);
        @RawFieldAddress
        public <T extends PointerBase> T addressOfItems();
    }

    
    

    /**
     * The native structure of the lkt_node_array Langkit array.
     * This structure is described as a `RawStruct` because native-image
     * doesn't handle flexible array members. Regarding this statement, array
     * elements are accessible through the `addressOfItems` method which
     * returns a pointer to the first value.
     */
    @RawStructure
    public interface FunDeclArrayNative extends PointerBase {
        @RawField
        public int getASize();
        @RawField
        public void setASize(int n);

        @RawField
        public int getBRefCount();
        @RawField
        public void setBRefCount(int refCount);

        @RawField
        public byte getItems();
        @RawField
        public void setItems(byte items);
        @RawFieldAddress
        public <T extends PointerBase> T addressOfItems();
    }

    
    

    /**
     * The native structure of the lkt_internal_logic_context_array Langkit array.
     * This structure is described as a `RawStruct` because native-image
     * doesn't handle flexible array members. Regarding this statement, array
     * elements are accessible through the `addressOfItems` method which
     * returns a pointer to the first value.
     */
    @RawStructure
    public interface LogicContextArrayNative extends PointerBase {
        @RawField
        public int getASize();
        @RawField
        public void setASize(int n);

        @RawField
        public int getBRefCount();
        @RawField
        public void setBRefCount(int refCount);

        @RawField
        public byte getItems();
        @RawField
        public void setItems(byte items);
        @RawFieldAddress
        public <T extends PointerBase> T addressOfItems();
    }

    
    

    /**
     * The native structure of the lkt_internal_ref_result_array Langkit array.
     * This structure is described as a `RawStruct` because native-image
     * doesn't handle flexible array members. Regarding this statement, array
     * elements are accessible through the `addressOfItems` method which
     * returns a pointer to the first value.
     */
    @RawStructure
    public interface RefResultArrayNative extends PointerBase {
        @RawField
        public int getASize();
        @RawField
        public void setASize(int n);

        @RawField
        public int getBRefCount();
        @RawField
        public void setBRefCount(int refCount);

        @RawField
        public byte getItems();
        @RawField
        public void setItems(byte items);
        @RawFieldAddress
        public <T extends PointerBase> T addressOfItems();
    }

    
    

    /**
     * The native structure of the lkt_internal_solver_diagnostic_array Langkit array.
     * This structure is described as a `RawStruct` because native-image
     * doesn't handle flexible array members. Regarding this statement, array
     * elements are accessible through the `addressOfItems` method which
     * returns a pointer to the first value.
     */
    @RawStructure
    public interface SolverDiagnosticArrayNative extends PointerBase {
        @RawField
        public int getASize();
        @RawField
        public void setASize(int n);

        @RawField
        public int getBRefCount();
        @RawField
        public void setBRefCount(int refCount);

        @RawField
        public byte getItems();
        @RawField
        public void setItems(byte items);
        @RawFieldAddress
        public <T extends PointerBase> T addressOfItems();
    }

    
    

    /**
     * The native structure of the lkt_analysis_unit_array Langkit array.
     * This structure is described as a `RawStruct` because native-image
     * doesn't handle flexible array members. Regarding this statement, array
     * elements are accessible through the `addressOfItems` method which
     * returns a pointer to the first value.
     */
    @RawStructure
    public interface AnalysisUnitArrayNative extends PointerBase {
        @RawField
        public int getASize();
        @RawField
        public void setASize(int n);

        @RawField
        public int getBRefCount();
        @RawField
        public void setBRefCount(int refCount);

        @RawField
        public byte getItems();
        @RawField
        public void setItems(byte items);
        @RawFieldAddress
        public <T extends PointerBase> T addressOfItems();
    }


    // ===== Generated iterators =====


    // ===== Native function definitions =====

    /** This class contains all native function definitions for NI */
    @CContext(LibDirectives.class)
    public static final class NI_LIB {

        // ----- Language specific functions -----

        


        // ----- Entry point literals -----

        /**
         * This entry point literal provide a pointer to the unit requested
         * callback.
         */
        public static final CEntryPointLiteral<UnitRequestedFunctionPointer>
            unitRequestedFunction = CEntryPointLiteral.create(
                Liblktlang.class,
                "unitRequested",
                IsolateThread.class,
                AnalysisContextNative.class,
                TextNative.class,
                AnalysisUnitNative.class,
                byte.class,
                byte.class
            );

        /**
         * This entry point literal provide a pointer to the unit parsed
         * callback.
         */
        public static final CEntryPointLiteral<UnitParsedFunctionPointer>
            unitParsedFunction = CEntryPointLiteral.create(
                Liblktlang.class,
                "unitParsed",
                IsolateThread.class,
                AnalysisContextNative.class,
                AnalysisUnitNative.class,
                byte.class
            );

        // ----- Util functions -----

        /** Util function to free langkit side allocated memory */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_free(
            PointerBase pointer
        );

        // ----- Stack trace functions -----

        /**
         * Return the number of entries in the given stack trace.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int
        lkt_stack_trace_size(VoidPointer trace);

        /**
         * Return the stack trace item at the given index. The given index must
         * be non-negative and lower than the stack trace size.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native VoidPointer
        lkt_stack_trace_element(VoidPointer trace, int index);

        /**
         * Allocate and return a stack trace for the given entries.
         *
         * The result must be deallocated with the ``destroy_stack_trace``
         * function when done with it.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native VoidPointer
        lkt_create_stack_trace(int size, WordPointer elements);

        /**
         * Deallocate a stack trace that was created with
         * ``create_stack_trace``.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void
        lkt_destroy_stack_trace(VoidPointer trace);

        /**
         * Convert a stack trace to a multi-line human readable trace.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native CCharPointer
        lkt_symbolize_stack_trace(VoidPointer trace);

        // ----- Exception functions -----

        /** Get the last exception raised by langkit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native LangkitExceptionNative
        lkt_get_last_exception();

        // ----- Big integer functions -----

        /** Create a big integer from a text */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native BigIntegerNative lkt_create_big_integer(
            TextNative text
        );

        /** Get the text representation of a big integer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_big_integer_text(
            BigIntegerNative big_integer,
            TextNative text
        );

        /** Decrease the reference counter of the big integer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_big_integer_decref(
            BigIntegerNative big_integer
        );

        // ----- Symbol functions -----

        /** Create a new symbol in the given context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_context_symbol(
            AnalysisContextNative context,
            TextNative text,
            SymbolNative res
        );

        /** Get the text of a given symbol */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_symbol_text(
            SymbolNative symbol,
            TextNative text
        );

        // ----- String functions -----

        /** Create a new string wrapper in langkit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native StringNative lkt_create_string(
            CIntPointer content,
            int length
        );

        /** Decrease the reference counter of a string */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_string_dec_ref(
            StringNative string
        );

        // ----- Text functions -----

        /** Destroy a text in the memory */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_destroy_text(
            TextNative text
        );

        // ----- Rewriting result functions -----

        /** Free a rewriting apply result */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_free_apply_result(
            RewritingApplyResultNative apply_result
        );

        // ----- File reader functions -----

        /** Decrease the reference counter of the given file reader */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_dec_ref_file_reader(
            FileReaderNative fileReader
        );

        // ----- Unit provider functions -----

        /** Decrease the ref counter of the unit provider */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_dec_ref_unit_provider(
            UnitProviderNative unitProvider
        );

        // ----- Event handler functions -----

        /** Create a new event handler */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native EventHandlerNative lkt_create_event_handler(
            VoidPointer data,
            VoidPointer destroy_callback,
            UnitRequestedFunctionPointer unit_requested_func,
            UnitParsedFunctionPointer unit_parsed_func
        );

        /** Decrease the ref counter of the event handler */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_dec_ref_event_handler(
            EventHandlerNative eventHandler
        );

        // ----- Token functions -----

        /**
         * Kind for this token.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_get_kind(
            TokenNative token
        );

        /**
         * Return the source location range of the given token.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_token_sloc_range(
            TokenNative token,
            SourceLocationRangeNative result
        );

        /** Get the next token */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_token_next(
            TokenNative token,
            TokenNative res
        );

        /** Get the previous token */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_token_previous(
            TokenNative token,
            TokenNative res
        );

        /** Get if two tokens are equivalent */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native byte lkt_token_is_equivalent(
            TokenNative left,
            TokenNative right
        );

        /** Get the text in a token range */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_token_range_text(
            TokenNative start,
            TokenNative end,
            TextNative res
        );

        // ----- Analysis context functions -----

        /** Allocate a new analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative
        lkt_allocate_analysis_context();

        /** Create a new analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void
        lkt_initialize_analysis_context(
            AnalysisContextNative context,
            CCharPointer charset,
            FileReaderNative file_reader,
            UnitProviderNative unit_provider,
            EventHandlerNative event_handler,
            int with_trivia,
            int tab_stop
        );

        /** Increase the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_context_incref(
            AnalysisContextNative context
        );

        /** Decrease the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_context_decref(
            AnalysisContextNative context
        );

        // ----- Analysis unit functions -----

        /** Get a unit from a file */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        lkt_get_analysis_unit_from_file(
            AnalysisContextNative context,
            CCharPointer file_name,
            CCharPointer charset,
            int reparse,
            int rule
        );

        /** Get a unit from a buffer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        lkt_get_analysis_unit_from_buffer(
            AnalysisContextNative context,
            CCharPointer file_name,
            CCharPointer charset,
            CCharPointer buffer,
            long buffer_size,
            int rule
        );

        /** Get a unit from the unit provider. */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        lkt_get_analysis_unit_from_provider(
            AnalysisContextNative context,
            TextNative name,
            int kind,
            CCharPointer charset,
            int reparse
        );

        /** Get the root of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_unit_root(
            AnalysisUnitNative unit,
            EntityNative res
        );

        /** Get the file name for a given unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native CCharPointer lkt_unit_filename(
            AnalysisUnitNative unit
        );

        /** Get the token count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_unit_token_count(
            AnalysisUnitNative unit
        );

        /** Get the trivia count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_unit_trivia_count(
            AnalysisUnitNative unit
        );

        /** Get the first token of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_unit_first_token(
            AnalysisUnitNative unit,
            TokenNative res
        );

        /** Get the last token of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_unit_last_token (
            AnalysisUnitNative unit,
            TokenNative res
        );

        /** Get the context for a given unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative lkt_unit_context(
            AnalysisUnitNative unit
        );

        /** Get the diagnostic count of the unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_unit_diagnostic_count(
            AnalysisUnitNative unit
        );

        /** Get the nth diagnostic for the unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_unit_diagnostic(
            AnalysisUnitNative unit,
            int n,
            DiagnosticNative diagnostic
        );

        // ----- Rewriting context functions -----

        /** Start a new rewriting session on the given analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingContextNative
        lkt_rewriting_start_rewriting(
            AnalysisContextNative analysis_context
        );

        /** Get the analysis context from the given rewriting context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative
        lkt_rewriting_handle_to_context(
            RewritingContextNative rewriting_context
        );

        /** Get a pointer to the rewriting units owned by the context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native WordPointer lkt_rewriting_unit_handles(
            RewritingContextNative rewriting_context
        );

        /** Create a node in the rewriting context and return it */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_create_node(
            RewritingContextNative rewriting_context,
            int node_kind
        );

        /** Create a node in the rewriting context with the given children */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_create_regular_node(
            RewritingContextNative rewriting_context,
            int node_kind,
            WordPointer children,
            int count
        );

        /** Create a token node in the rewriting context and return it */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_create_token_node(
            RewritingContextNative rewriting_context,
            int node_kind,
            TextNative node_text
        );

        /** Create a new node tree from the given template */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_create_from_template(
            RewritingContextNative rewriting_context,
            TextNative template_text,
            WordPointer arguments,
            int count,
            int rule
        );

        /** Apply the rewriting session and close it if success */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_apply(
            RewritingContextNative rewriting_context,
            RewritingApplyResultNative apply_result
        );

        /** Abort the rewriting session */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_abort_rewriting(
            RewritingContextNative rewriting_context
        );

        // ----- Rewriting unit functions -----

        /** Get the rewriting unit corresponding to the given analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingUnitNative
        lkt_rewriting_unit_to_handle(
            AnalysisUnitNative unit
        );

        /** Get the analysis unit corresponding to the given rewriting unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        lkt_rewriting_handle_to_unit(
            RewritingUnitNative rewriting_unit
        );

        /** Get the root of the given rewriting unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative lkt_rewriting_unit_root(
            RewritingUnitNative rewriting_unit
        );

        /** Set the root of the rewriting unit to the rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_unit_set_root(
            RewritingUnitNative rewriting_unit,
            RewritingNodeNative rewriting_node
        );

        /** Unparse the rewriting unit to get its textual content */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_unit_unparse(
            RewritingUnitNative rewriting_unit,
            TextNative result
        );

        // ----- Rewriting node functions -----

        /** Get the rewriting node from the given parsed node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_node_to_handle(
            Pointer node
        );

        /** Get the parsed node from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native Pointer lkt_rewriting_handle_to_node(
            RewritingNodeNative rewriting_node
        );

        /** Get the rewriting context from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingContextNative
        lkt_rewriting_node_to_context(
            RewritingNodeNative rewriting_node
        );

        /** Clone the given rewriting node and return the copy */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative lkt_rewriting_clone(
            RewritingNodeNative to_clone
        );

        /** Unparse the given rewriting node in the given text */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_node_unparse(
            RewritingNodeNative rewriting_node,
            TextNative result
        );

        /** Get the kind of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_rewriting_kind(
            RewritingNodeNative rewriting_node
        );

        /** Get the rewriting node image */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_node_image(
            RewritingNodeNative rewriting_node,
            TextNative result
        );

        /** Return whether the node is tied to a rewriting unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_rewriting_tied(
            RewritingNodeNative rewriting_node
        );

        /** Return the parent of the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative lkt_rewriting_parent(
            RewritingNodeNative rewriting_node
        );

        /** Get the rewriting node children */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_children(
            RewritingNodeNative rewriting_node,
            WordPointer result_reference,
            CIntPointer result_count
        );

        /** Get the child at the given member reference */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative lkt_rewriting_child(
            RewritingNodeNative parent,
            int child_member_reference
        );

        /** Set the given child at the given member reference */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_set_child(
            RewritingNodeNative parent,
            int child_member_reference,
            RewritingNodeNative new_child
        );

        /** Replace the rewriting node by the new one */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_replace(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative new_node
        );

        /** Get the first child of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_first_child(
            RewritingNodeNative parent
        );

        /** Get the last child of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_last_child(
            RewritingNodeNative parent
        );

        /** Get the next child from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_next_child(
            RewritingNodeNative rewriting_node
        );

        /** Get the previous child from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_previous_child(
            RewritingNodeNative rewriting_node
        );

        /** Insert the provided rewriting node before the other node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_before(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative to_insert
        );

        /** Insert the provided rewriting node after the other node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_after(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative to_insert
        );

        /**
         * Insert the provided rewriting node at the beginning of the
         * children
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_first(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative to_insert
        );

        /** Insert the provided rewriting node at the end of the children */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_last(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative to_insert
        );

        /** Remove the given rewriting node from its list parent */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_remove_child(
            RewritingNodeNative to_remove
        );

        /** Get the text of the rewriting token node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_text(
            RewritingNodeNative rewriting_node,
            TextNative result
        );

        /** Set the text of the rewriting token node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_set_text(
            RewritingNodeNative rewriting_node,
            TextNative text
        );

        // ----- Array functions -----

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native CompleteItemArrayNative lkt_internal_complete_item_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_complete_item_array_dec_ref(CompleteItemArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native LktNodeArrayNative lkt_node_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_array_dec_ref(LktNodeArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native LogicContextArrayNative lkt_internal_logic_context_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_logic_context_array_dec_ref(LogicContextArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RefResultArrayNative lkt_internal_ref_result_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_ref_result_array_dec_ref(RefResultArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native SolverDiagnosticArrayNative lkt_internal_solver_diagnostic_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_solver_diagnostic_array_dec_ref(SolverDiagnosticArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitArrayNative lkt_analysis_unit_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_analysis_unit_array_dec_ref(AnalysisUnitArrayNative array);


        // ----- Iterator functions -----


        // ----- Structure functions -----

            
        
    


            
        
    


        
    


            
        
    


            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_decoded_char_value_dec_ref(
            DecodedCharValueNative structNative
        );

            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_decoded_string_value_dec_ref(
            DecodedStringValueNative structNative
        );

            
            
            
            
        
    


            
            
            
        
    


            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_solver_diagnostic_dec_ref(
            SolverDiagnosticNative structNative
        );

            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_solver_result_dec_ref(
            SolverResultNative structNative
        );


        // ----- Node functions -----

        /** Create a bare entity from a node pointer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_create_bare_entity(
            Pointer node,
            EntityNative result
        );

        /** Return whether the two given entities are equal */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_is_equivalent(
            EntityNative entity_left,
            EntityNative entity_right
        );

        /** Get the hash of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_hash(
            EntityNative entity
        );

        /** Get the type of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_kind(
            EntityNative entity
        );

        /** Get the text from a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_text(
            EntityNative entity,
            TextNative text
        );

        /** Get the source location range for a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_sloc_range(
            EntityNative entity,
            SourceLocationRangeNative slocr
        );

        /** Get the number of children for a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_children_count(
            EntityNative entity
        );

        /** Get the nth child for the node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_child(
            EntityNative entity,
            int n,
            EntityNative res
        );

        /** Get if the node is a token node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_is_token_node(
            EntityNative entity
        );

        /** Get the unit of the node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative lkt_node_unit(
            EntityNative entity
        );

        /** Get the image of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_image(
            EntityNative entity,
            TextNative text
        );

        // ----- Node fields accessors and properties -----

        
    

            

        /** Isomethod of lkt_lkt_node_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_parent(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_parents langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_parents(
            EntityNative node,
            byte with_self,
            WordPointer result
        );
            

        /** Isomethod of lkt_lkt_node_children langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_children(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_lkt_node_token_start langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_token_start(
            EntityNative node,
            TokenNative result
        );
            

        /** Isomethod of lkt_lkt_node_token_end langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_token_end(
            EntityNative node,
            TokenNative result
        );
            

        /** Isomethod of lkt_lkt_node_child_index langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_child_index(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of lkt_lkt_node_previous_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_previous_sibling(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_next_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_next_sibling(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_unit(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_lkt_node_is_ghost langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_is_ghost(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_lkt_node_full_sloc_image langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_full_sloc_image(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_lkt_node_completion_item_kind_to_int langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_completion_item_kind_to_int(
            EntityNative node,
            int kind,
            CIntPointer result
        );
            

        /** Isomethod of lkt_lkt_node_p_set_solver_debug_mode langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_set_solver_debug_mode(
            EntityNative node,
            byte enable,
            CCharPointer result
        );
            

        /** Isomethod of lkt_lkt_node_p_basic_trait_gen langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_basic_trait_gen(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_basic_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_basic_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_node_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_node_gen_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_node_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_indexable_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_indexable_gen_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_indexable_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_indexable_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_token_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_token_node_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_error_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_error_node_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_char_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_int_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_bool_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_bool_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_bigint_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_bigint_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_string_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_string_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_symbol_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_symbol_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_property_error_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_property_error_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_regexp_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_regexp_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_entity_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_entity_gen_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_entity_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_entity_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_logicvar_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_logicvar_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_equation_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_equation_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_array_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_array_gen_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_array_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_array_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_astlist_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_astlist_gen_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_astlist_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_astlist_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_node_builder_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_node_builder_gen_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_node_builder_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_node_builder_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_iterator_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_iterator_gen_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_iterator_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_iterator_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_analysis_unit_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_analysis_unit_gen_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_analysis_unit_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_analysis_unit_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_topmost_invalid_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_topmost_invalid_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_nameres_diagnostics langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_nameres_diagnostics(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_lkt_node_p_solve_enclosing_context langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_solve_enclosing_context(
            EntityNative node,
            SolverResultNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_xref_entry_point langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_xref_entry_point(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_lkt_node_p_complete langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_complete(
            EntityNative node,
            WordPointer result
        );

        
    

            

        /** Isomethod of lkt_argument_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_argument_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_argument_f_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_argument_f_value(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_base_lexer_case_rule_alt_f_send langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_lexer_case_rule_alt_f_send(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_cond_alt_f_cond_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_cond_alt_f_cond_exprs(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_base_match_branch_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_match_branch_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_base_match_branch_p_match_part langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_match_branch_p_match_part(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_match_branch_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_match_branch_f_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_pattern_match_branch_f_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_pattern_match_branch_f_pattern(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_block_expr_clause_f_clause langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_block_expr_clause_f_clause(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_class_qualifier_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_class_qualifier_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_decl_f_syn_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_f_syn_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_decl_p_custom_image langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_custom_image(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_decl_p_decl_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_decl_type_name(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_decl_p_def_ids langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_def_ids(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_decl_p_as_bare_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_as_bare_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_decl_p_get_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_get_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_decl_p_get_cast_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_get_cast_type(
            EntityNative node,
            EntityNative cast_to,
            EntityNative result
        );
            

        /** Isomethod of lkt_decl_p_get_keep_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_get_keep_type(
            EntityNative node,
            EntityNative keep_type,
            EntityNative result
        );
            

        /** Isomethod of lkt_decl_p_get_suffix_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_get_suffix_type(
            EntityNative node,
            EntityNative prefix_type,
            EntityNative result
        );
            

        /** Isomethod of lkt_decl_p_is_generic langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_is_generic(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_decl_p_return_type_is_instantiated langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_return_type_is_instantiated(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_decl_p_is_instantiated langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_is_instantiated(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_decl_p_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_name(
            EntityNative node,
            SymbolNative result
        );
            

        /** Isomethod of lkt_decl_p_full_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_full_name(
            EntityNative node,
            WordPointer result
        );

        
    

            

        /** Isomethod of lkt_base_grammar_rule_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_grammar_rule_decl_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_explicitly_typed_decl_f_decl_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_explicitly_typed_decl_f_decl_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_component_decl_f_default_val langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_component_decl_f_default_val(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_field_decl_f_trait_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_field_decl_f_trait_ref(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_fun_param_decl_f_decl_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_param_decl_f_decl_annotations(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_val_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_val_decl_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_fun_decl_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_decl_f_params(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_fun_decl_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_decl_f_return_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_fun_decl_f_trait_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_decl_f_trait_ref(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_fun_decl_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_decl_f_body(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_fun_decl_p_is_dynamic_combiner langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_decl_p_is_dynamic_combiner(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_fun_decl_p_find_all_overrides langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_decl_p_find_all_overrides(
            EntityNative node,
            AnalysisUnitArrayNative units,
            WordPointer result
        );

        
    

            

        /** Isomethod of lkt_env_spec_decl_f_actions langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_env_spec_decl_f_actions(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_generic_decl_f_generic_param_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_decl_f_generic_param_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_generic_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_decl_f_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_decl_f_rules(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lexer_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_decl_f_rules(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lexer_family_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_family_decl_f_rules(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_type_decl_f_traits langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_f_traits(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_type_decl_f_syn_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_f_syn_base_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_type_decl_p_def_id langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_def_id(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_type_decl_p_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_base_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_type_decl_p_base_type_if_entity langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_base_type_if_entity(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_generic_param_type_decl_f_has_class langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_param_type_decl_f_has_class(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_named_type_decl_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_named_type_decl_f_decls(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_enum_class_decl_f_branches langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_enum_class_decl_f_branches(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_enum_type_decl_f_literals langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_enum_type_decl_f_literals(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_decl_annotation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_annotation_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_decl_annotation_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_annotation_f_args(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_decl_annotation_args_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_annotation_args_f_args(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_elsif_branch_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_elsif_branch_f_cond_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_elsif_branch_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_elsif_branch_f_then_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_enum_class_case_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_enum_class_case_f_decls(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_excludes_null_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_excludes_null_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_expr_p_get_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_get_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_expr_p_get_generic_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_get_generic_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_expr_p_get_expected_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_get_expected_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_expr_p_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_referenced_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_any_of_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_any_of_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_any_of_f_values langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_any_of_f_values(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_array_literal_f_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_array_literal_f_exprs(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_array_literal_f_element_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_array_literal_f_element_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_base_call_expr_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_call_expr_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_base_call_expr_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_call_expr_f_args(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_bin_op_f_left langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_bin_op_f_left(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_bin_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_bin_op_f_op(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_bin_op_f_right langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_bin_op_f_right(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_block_expr_f_clauses langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_block_expr_f_clauses(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_cast_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_cast_expr_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_cast_expr_f_null_cond langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_cast_expr_f_null_cond(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_cast_expr_f_excludes_null langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_cast_expr_f_excludes_null(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_cast_expr_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_cast_expr_f_dest_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_dot_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_dot_expr_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_dot_expr_f_null_cond langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_dot_expr_f_null_cond(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_dot_expr_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_dot_expr_f_suffix(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_error_on_null_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_error_on_null_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_generic_instantiation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_instantiation_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_generic_instantiation_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_instantiation_f_args(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_grammar_discard_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_discard_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_dont_skip_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_dont_skip_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_dont_skip_f_dont_skip langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_dont_skip_f_dont_skip(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_list_f_list_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_f_list_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_list_f_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_f_kind(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_list_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_list_f_sep langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_f_sep(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_null_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_null_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_opt_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_error_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_opt_error_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_error_group_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_opt_error_group_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_group_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_opt_group_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_or_expr_f_sub_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_or_expr_f_sub_exprs(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_pick_f_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_pick_f_exprs(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_grammar_predicate_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_predicate_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_predicate_f_prop_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_predicate_f_prop_ref(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_rule_ref_f_node_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_rule_ref_f_node_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_skip_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_skip_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_stop_cut_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_stop_cut_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_parse_node_expr_f_node_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_parse_node_expr_f_node_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_parse_node_expr_f_sub_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_parse_node_expr_f_sub_exprs(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_token_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_lit_p_denoted_value(
            EntityNative node,
            DecodedStringValueNative result
        );

        
    

            

        /** Isomethod of lkt_token_no_case_lit_f_lit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_no_case_lit_f_lit(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_token_pattern_concat_f_left langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_pattern_concat_f_left(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_token_pattern_concat_f_right langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_pattern_concat_f_right(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_token_pattern_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_pattern_lit_p_denoted_value(
            EntityNative node,
            DecodedStringValueNative result
        );

        
    

            

        /** Isomethod of lkt_token_ref_f_token_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_ref_f_token_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_token_ref_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_ref_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_id_p_custom_image langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_id_p_custom_image(
            EntityNative node,
            WordPointer result
        );

        
    

            

        /** Isomethod of lkt_def_id_p_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_def_id_p_name(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_def_id_p_get_implementatinons langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_def_id_p_get_implementatinons(
            EntityNative node,
            AnalysisUnitArrayNative units,
            WordPointer result
        );
            

        /** Isomethod of lkt_def_id_p_decl_detail langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_def_id_p_decl_detail(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_def_id_p_completion_item_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_def_id_p_completion_item_kind(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of lkt_def_id_p_doc langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_def_id_p_doc(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_def_id_p_find_all_references langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_def_id_p_find_all_references(
            EntityNative node,
            AnalysisUnitArrayNative units,
            WordPointer result
        );

        
    


        
    

            

        /** Isomethod of lkt_ref_id_p_referenced_defining_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_ref_id_p_referenced_defining_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_if_expr_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_if_expr_f_cond_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_if_expr_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_if_expr_f_then_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_if_expr_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_if_expr_f_alternatives(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_if_expr_f_else_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_if_expr_f_else_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_isa_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_isa_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_isa_f_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_isa_f_pattern(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_keep_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_keep_expr_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_keep_expr_f_null_cond langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_keep_expr_f_null_cond(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_keep_expr_f_keep_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_keep_expr_f_keep_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lambda_expr_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lambda_expr_f_params(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lambda_expr_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lambda_expr_f_return_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lambda_expr_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lambda_expr_f_body(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_char_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_char_lit_p_denoted_value(
            EntityNative node,
            DecodedCharValueNative result
        );

        
    

            

        /** Isomethod of lkt_null_lit_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_null_lit_f_dest_type(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_string_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_string_lit_p_denoted_value(
            EntityNative node,
            DecodedStringValueNative result
        );
            

        /** Isomethod of lkt_string_lit_p_is_prefixed_string langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_string_lit_p_is_prefixed_string(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_string_lit_p_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_string_lit_p_prefix(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of lkt_string_lit_p_is_regexp_literal langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_string_lit_p_is_regexp_literal(
            EntityNative node,
            CCharPointer result
        );

        
    

            

        /** Isomethod of lkt_block_string_lit_f_lines langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_block_string_lit_f_lines(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_logic_assign_f_dest_var langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_logic_assign_f_dest_var(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_logic_assign_f_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_logic_assign_f_value(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_logic_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_logic_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_logic_propagate_f_dest_var langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_logic_propagate_f_dest_var(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_logic_propagate_f_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_logic_propagate_f_call(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_logic_unify_f_lhs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_logic_unify_f_lhs(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_logic_unify_f_rhs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_logic_unify_f_rhs(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_match_expr_f_match_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_match_expr_f_match_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_match_expr_f_branches langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_match_expr_f_branches(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_not_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_not_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_paren_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_paren_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_raise_expr_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_raise_expr_f_dest_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_raise_expr_f_except_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_raise_expr_f_except_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_subscript_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_subscript_expr_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_subscript_expr_f_null_cond langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_subscript_expr_f_null_cond(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_subscript_expr_f_index langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_subscript_expr_f_index(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_try_expr_f_try_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_try_expr_f_try_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_try_expr_f_or_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_try_expr_f_or_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_un_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_un_op_f_op(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_un_op_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_un_op_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_full_decl_f_doc langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_full_decl_f_doc(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_full_decl_f_decl_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_full_decl_f_decl_annotations(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_full_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_full_decl_f_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_full_decl_p_has_annotation langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_full_decl_p_has_annotation(
            EntityNative node,
            SymbolNative name,
            CCharPointer result
        );

        
    

            

        /** Isomethod of lkt_grammar_list_sep_f_token langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_sep_f_token(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_list_sep_f_extra langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_sep_f_extra(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_import_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_import_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_import_p_referenced_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_import_p_referenced_unit(
            EntityNative node,
            WordPointer result
        );

        
    

            

        /** Isomethod of lkt_langkit_root_f_imports langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_langkit_root_f_imports(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_langkit_root_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_langkit_root_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_langkit_root_p_fetch_prelude langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_langkit_root_p_fetch_prelude(
            EntityNative node,
            WordPointer result
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lexer_case_rule_f_alts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_f_alts(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_send_f_sent langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_send_f_sent(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lexer_case_rule_send_f_match_size langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_send_f_match_size(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_null_cond_qualifier_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_null_cond_qualifier_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_binding_pattern_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_binding_pattern_f_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_binding_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_binding_pattern_f_sub_pattern(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_ellipsis_pattern_f_binding langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_ellipsis_pattern_f_binding(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_extended_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_extended_pattern_f_sub_pattern(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_extended_pattern_f_details langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_extended_pattern_f_details(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_filtered_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_filtered_pattern_f_sub_pattern(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_filtered_pattern_f_predicate langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_filtered_pattern_f_predicate(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_list_pattern_f_sub_patterns langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_list_pattern_f_sub_patterns(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_not_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_not_pattern_f_sub_pattern(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_or_pattern_f_left_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_or_pattern_f_left_sub_pattern(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_or_pattern_f_right_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_or_pattern_f_right_sub_pattern(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_paren_pattern_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_paren_pattern_f_sub_pattern(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_tuple_pattern_f_sub_patterns langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_tuple_pattern_f_sub_patterns(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_type_pattern_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_pattern_f_type_name(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_field_pattern_detail_f_id langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_field_pattern_detail_f_id(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_field_pattern_detail_f_expected_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_field_pattern_detail_f_expected_value(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_property_pattern_detail_f_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_property_pattern_detail_f_call(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_property_pattern_detail_f_expected_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_property_pattern_detail_f_expected_value(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_selector_pattern_detail_f_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_selector_pattern_detail_f_call(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_selector_pattern_detail_f_sub_pattern langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_selector_pattern_detail_f_sub_pattern(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_selector_call_f_quantifier langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_selector_call_f_quantifier(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_selector_call_f_binding langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_selector_call_f_binding(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_selector_call_f_selector_call langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_selector_call_f_selector_call(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_type_ref_p_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_ref_p_referenced_decl(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_function_type_ref_f_param_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_function_type_ref_f_param_types(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_function_type_ref_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_function_type_ref_f_return_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_generic_type_ref_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_type_ref_f_type_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_generic_type_ref_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_type_ref_f_args(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_simple_type_ref_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_simple_type_ref_f_type_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_var_bind_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_var_bind_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_var_bind_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_var_bind_f_expr(
            EntityNative node,
            EntityNative result
        );


    }


    // ==========
    // Exceptions
    // ==========

    /**
     * This class represents exception during symbol manipulation.
     */
    public static final class SymbolException extends RuntimeException {
        public SymbolException(
            final String symbol
        ) {
            super("Invalid symbol : '" + symbol + "'");
        }
    }

    /**
     * This class reprsents exception during enum manipulation.
     */
    public static final class EnumException extends RuntimeException {
        public EnumException(
            final String msg
        ) {
            super(msg);
        }
    }

    /**
     * This class represents an exception in the references manipulation.
     */
    public static final class ReferenceException extends RuntimeException {
        public ReferenceException(
            final String msg
        ) {
            super(msg);
        }
    }

    public static class LangkitStackTrace {
        // ----- Instance attributes -----

        /**
         * Array of elements for the stack trace
         */
        private final PointerWrapper[] elements;

        /**
         * Cached symbolized stack trace: symbolization is done on demand, but
         * at most once.
         */
        private String symbolized;

        /**
         * Create a copy of a stack trace from an existing C API value.
         */
        public LangkitStackTrace(final PointerWrapper ptr) {
            final int size =
                ImageInfo.inImageCode()
                ? NI_LIB.lkt_stack_trace_size(ptr.ni())
                : JNI_LIB.lkt_stack_trace_size(ptr.jni());
            this.elements = new PointerWrapper[size];
            for (int i = 0; i < size; ++i) {
                this.elements[i] =
                    ImageInfo.inImageCode()
                    ? PointerWrapper.wrap(
                        NI_LIB.lkt_stack_trace_element(ptr.ni(), i))
                    : JNI_LIB.lkt_stack_trace_element(ptr.jni(), i);
            }
            this.symbolized = null;
        }

        /**
         * Convert a stack trace to a multi-line human readable trace.
         */
        public final String symbolize() {
            // If this stack trace has not been symbolized yet, do it now.
            if (this.symbolized == null) {
                // First, re-create a stack trace C API value from this Java
                // object.
                if(ImageInfo.inImageCode()) {
                    // Create the C API level "elements" array.
                    final WordPointer elements = UnmanagedMemory.malloc(
                        this.elements.length * SizeOf.get(WordPointer.class));
                    for(int i = 0; i < this.elements.length; ++i) {
                        elements.write(i, this.elements[i].ni());
                    }

                    // Create the C API level stack trace.
                    final PointerWrapper stackTrace =
                        PointerWrapper.wrap(
                            NI_LIB.lkt_create_stack_trace(
                                this.elements.length,
                                elements));
                    UnmanagedMemory.free(elements);

                    // Finally run symbolization.
                    final CCharPointer cString =
                        NI_LIB.lkt_symbolize_stack_trace(
                            (VoidPointer) stackTrace.ni());
                    this.symbolized = toJString(cString);
                    NI_LIB.lkt_free(cString);

                    // And destroy the stack trace.
                    NI_LIB.lkt_destroy_stack_trace(stackTrace.ni());
                } else {
                    final PointerWrapper stackTrace =
                        JNI_LIB.lkt_create_stack_trace(this.elements);
                    this.symbolized =
                        JNI_LIB.lkt_symbolize_stack_trace(
                            stackTrace.jni());
                    JNI_LIB.lkt_destroy_stack_trace(stackTrace.jni());
                }
            }
            return this.symbolized;
        }
    }

    /**
     * This class wraps the exceptions from the langkit native library.
     */
    public static class LangkitException extends RuntimeException {

        // ----- Instance attributes -----

        /** The kind of the langkit exception. */
        public final ExceptionKind kind;

        /** Native stack trace.  */
        private final LangkitStackTrace internalStackTrace;

        // ----- Constructors -----

        /**
         * Create a new langkit exception.
         *
         * @param kind The kind of the exception represented by an integer
         *   which will be mapped to an enum value.
         * @param message The message of the exception.
         * @param stackTrace The native stack trace associated with the
         * exception.
         */
        public LangkitException(
            final int kind,
            final String message,
            final LangkitStackTrace nativeStackTrace
        ) {
            super(message);
            this.kind = ExceptionKind.fromC(kind);
            this.internalStackTrace = nativeStackTrace;
        }

        // ----- Instance methods -----

        /**
         * Return the native stack trace associated with this exception as a
         * multi-line human readable string.
         */
        public final String nativeStackTrace() {
            return internalStackTrace.symbolize();
        }

    }

    // ==========
    // Enum definitions
    // ==========

    // ===== Constants enumeration =====

    /**
     * Kind for this token.
     */
    public enum TokenKind implements LangkitSupport.TokenKindInterface {

        // ----- Enum values -----

        NO_TOKEN(-1, "No_Token"),
        LKT_AMP(0, "Amp"),
        LKT_AND_KW(1, "And_Kw"),
        LKT_AT(2, "At"),
        LKT_BIG_NUMBER(3, "Big_Number"),
        LKT_BIND_KW(4, "Bind_Kw"),
        LKT_BLOCK_STRING_LINE(5, "Block_String_Line"),
        LKT_CASE_KW(6, "Case_Kw"),
        LKT_CHAR(7, "Char"),
        LKT_CLASS_KW(8, "Class_Kw"),
        LKT_COLON(9, "Colon"),
        LKT_COMB(10, "Comb"),
        LKT_COMMA(11, "Comma"),
        LKT_COMMENT(12, "Comment"),
        LKT_DISCARD_KW(13, "Discard_Kw"),
        LKT_DIV(14, "Div"),
        LKT_DOC_COMMENT(15, "Doc_Comment"),
        LKT_DOT(16, "Dot"),
        LKT_DYN_VAR_KW(17, "Dyn_Var_Kw"),
        LKT_E_Q(18, "E_Q"),
        LKT_ELIF_KW(19, "Elif_Kw"),
        LKT_ELLIPSIS(20, "Ellipsis"),
        LKT_ELSE_KW(21, "Else_Kw"),
        LKT_ENUM_KW(22, "Enum_Kw"),
        LKT_EQUAL(23, "Equal"),
        LKT_EXCL_MARK(24, "Excl_Mark"),
        LKT_FAT_RIGHT_ARROW(25, "Fat_Right_Arrow"),
        LKT_FUN_KW(26, "Fun_Kw"),
        LKT_G_T(27, "G_T"),
        LKT_G_T_E(28, "G_T_E"),
        LKT_GENERIC_KW(29, "Generic_Kw"),
        LKT_GRAMMAR_KW(30, "Grammar_Kw"),
        LKT_IDENTIFIER(31, "Identifier"),
        LKT_IF_KW(32, "If_Kw"),
        LKT_IMPLEMENTS_KW(33, "Implements_Kw"),
        LKT_IMPORT_KW(34, "Import_Kw"),
        LKT_IN_KW(35, "In_Kw"),
        LKT_INT_MARK(36, "Int_Mark"),
        LKT_IS_KW(37, "Is_Kw"),
        LKT_L_BRACE(38, "L_Brace"),
        LKT_L_BRACK(39, "L_Brack"),
        LKT_L_PAR(40, "L_Par"),
        LKT_L_T(41, "L_T"),
        LKT_L_T_E(42, "L_T_E"),
        LKT_LEFT_ARROW(43, "Left_Arrow"),
        LKT_LEXER_KW(44, "Lexer_Kw"),
        LKT_LEXING_FAILURE(45, "Lexing_Failure"),
        LKT_MATCH_KW(46, "Match_Kw"),
        LKT_MINUS(47, "Minus"),
        LKT_N_E(48, "N_E"),
        LKT_NOT_KW(49, "Not_Kw"),
        LKT_NULL_KW(50, "Null_Kw"),
        LKT_NUMBER(51, "Number"),
        LKT_OR_KW(52, "Or_Kw"),
        LKT_P_STRING(53, "P_String"),
        LKT_PERCENT(54, "Percent"),
        LKT_PIPE(55, "Pipe"),
        LKT_PLUS(56, "Plus"),
        LKT_PRIVATE_KW(57, "Private_Kw"),
        LKT_PUBLIC_KW(58, "Public_Kw"),
        LKT_R_BRACE(59, "R_Brace"),
        LKT_R_BRACK(60, "R_Brack"),
        LKT_R_PAR(61, "R_Par"),
        LKT_RAISE_KW(62, "Raise_Kw"),
        LKT_RIGHT_ARROW(63, "Right_Arrow"),
        LKT_SEMICOLON(64, "Semicolon"),
        LKT_STRING(65, "String"),
        LKT_STRUCT_KW(66, "Struct_Kw"),
        LKT_TERMINATION(67, "Termination"),
        LKT_THEN_KW(68, "Then_Kw"),
        LKT_TIMES(69, "Times"),
        LKT_TRAIT_KW(70, "Trait_Kw"),
        LKT_TRY_KW(71, "Try_Kw"),
        LKT_TWO_SIDED_ARROW(72, "Two_Sided_Arrow"),
        LKT_VAL_KW(73, "Val_Kw"),
        LKT_WHEN_KW(74, "When_Kw"),
        LKT_WHITESPACE(75, "Whitespace"),
        ;

        // ----- Class attributes -----

        /** Singleton that represents the none token kind. */
        public static final TokenKind NONE = NO_TOKEN;

        /** The map from int to enum values. */
        private static final Map<Integer, TokenKind> map = new HashMap<>();

        // ----- Instance attributes -----

        /** The value of the enum instance. */
        private final int value;

        /** The name of the enum instance in the Langkit DSL. */
        public final String name;

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(TokenKind elem : TokenKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private TokenKind(
            final int value,
            final String name
        ) {
            this.value = value;
            this.name = name;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to a token kind
         * value.
         */
        static TokenKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static TokenKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get TokenKind from " + cValue
                );
            return (TokenKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    /**
     * Enumerated type describing all possible exceptions that need to be
     * handled in the C bindings.
     */
    public enum ExceptionKind {

        // ----- Enum values -----

        FILE_READ_ERROR(0),
        BAD_TYPE_ERROR(1),
        OUT_OF_BOUNDS_ERROR(2),
        INVALID_INPUT(3),
        INVALID_SYMBOL_ERROR(4),
        INVALID_UNIT_NAME_ERROR(5),
        NATIVE_EXCEPTION(6),
        PRECONDITION_FAILURE(7),
        PROPERTY_ERROR(8),
        TEMPLATE_ARGS_ERROR(9),
        TEMPLATE_FORMAT_ERROR(10),
        TEMPLATE_INSTANTIATION_ERROR(11),
        STALE_REFERENCE_ERROR(12),
        SYNTAX_ERROR(13),
        UNKNOWN_CHARSET(14),
        MALFORMED_TREE_ERROR(15),
        ;

        // ----- Class attributes -----

        /** Singleton that represents the none expcetion kind. */
        public static final ExceptionKind NONE =
            FILE_READ_ERROR;

        /** The map from int to enum values. */
        private static final Map<Integer, ExceptionKind> map =
            new HashMap<>();

        // ----- Instance ttributes -----

        /** The value of the enum instance. */
        private final int value;

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(ExceptionKind elem : ExceptionKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private ExceptionKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to a exception
         * kind value.
         */
        static ExceptionKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static ExceptionKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get ExceptionKind from " + cValue
                );
            return (ExceptionKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    /**
     * Kind of AST nodes in parse trees.
     */
    public enum NodeKind implements LangkitSupport.NodeKindInterface {

        // ----- Enum values -----

            
        // LKT_NODE is abstract
            
        /**
         * Argument for function calls or for annotations.
         *
         * This node type has no derivation.
         */
        ARGUMENT(1),
            
        // BASE_LEXER_CASE_RULE_ALT is abstract
            
        /**
         * Alternative of a case rule which sends the token only if the kind of
         * the previous token is among a given set.
         *
         * This node type has no derivation.
         */
        LEXER_CASE_RULE_COND_ALT(2),
            
        /**
         * Default alternative of a case rule which sends the token if all the
         * previous alternatives failed.
         *
         * This node type has no derivation.
         */
        LEXER_CASE_RULE_DEFAULT_ALT(3),
            
        // BASE_MATCH_BRANCH is abstract
            
        /**
         * Branch inside a match expression. Classic limited Lkt syntax based
         * on ``case <name> : <type> => <expr>``, for the moment, the only
         * supported syntax in Lkt.
         *
         * This node type has no derivation.
         */
        MATCH_BRANCH(4),
            
        /**
         * Branch inside a match expression. LKQL pattern based syntax ``case
         * <pattern> => <expr>``.
         *
         * This node type has no derivation.
         */
        PATTERN_MATCH_BRANCH(5),
            
        /**
         * Clause (value declaration or dynamic variable binding) in a block
         * expression.
         *
         * This node type has no derivation.
         */
        BLOCK_EXPR_CLAUSE(6),
            
        /**
         * A single line in a block string literal.
         *
         * This node type has no derivation.
         */
        BLOCK_STRING_LINE(7),
            
        // CLASS_QUALIFIER is abstract
            
        /**
         * This node type has no derivation.
         */
        CLASS_QUALIFIER_ABSENT(8),
            
        /**
         * This node type has no derivation.
         */
        CLASS_QUALIFIER_PRESENT(9),
            
        // DECL is abstract
            
        // BASE_GRAMMAR_RULE_DECL is abstract
            
        /**
         * Declaration of a grammar rule inside of a grammar.
         *
         * This node type has no derivation.
         */
        GRAMMAR_RULE_DECL(10),
            
        /**
         * This node type has no derivation.
         */
        SYNTHETIC_LEXER_DECL(11),
            
        // BASE_VAL_DECL is abstract
            
        /**
         * Synthetic declaration for the implicit "node" variable available in
         * properties.
         *
         * This node type has no derivation.
         */
        NODE_DECL(12),
            
        /**
         * Synthetic declaration for the implicit "self" variable available in
         * properties.
         *
         * This node type has no derivation.
         */
        SELF_DECL(13),
            
        // USER_VAL_DECL is abstract
            
        /**
         * Variable declaration in pattern
         *
         * This node type has no derivation.
         */
        BINDING_VAL_DECL(14),
            
        /**
         * Enum literal declaration.
         *
         * This node type has no derivation.
         */
        ENUM_LIT_DECL(15),
            
        // EXPLICITLY_TYPED_DECL is abstract
            
        // COMPONENT_DECL is abstract
            
        /**
         * Field declaration.
         *
         * This node type has no derivation.
         */
        FIELD_DECL(16),
            
        /**
         * Function parameter declaration.
         *
         * This node type has no derivation.
         */
        FUN_PARAM_DECL(17),
            
        /**
         * Function parameter declaration.
         *
         * This node type has no derivation.
         */
        LAMBDA_PARAM_DECL(18),
            
        /**
         * Dynamic variable declaration.
         *
         * This node type has no derivation.
         */
        DYN_VAR_DECL(19),
            
        /**
         * Value declaration in a match branch.
         *
         * This node type has no derivation.
         */
        MATCH_VAL_DECL(20),
            
        /**
         * Value declaration.
         *
         * This node type has no derivation.
         */
        VAL_DECL(21),
            
        /**
         * Function declaration.
         *
         * This node type has no derivation.
         */
        FUN_DECL(22),
            
        /**
         * Env spec declaration.
         *
         * Each node type can have one or no env spec. Env specs contains only
         * a list of env actions.
         *
         * This node type has no derivation.
         */
        ENV_SPEC_DECL(23),
            
        /**
         * Placeholder node for syntax errors in lists of declarations.
         *
         * This node type has no derivation.
         */
        ERROR_DECL(24),
            
        /**
         * Generic entity declaration.
         *
         * This node type has no derivation.
         */
        GENERIC_DECL(25),
            
        /**
         * Declaration of a language's grammar.
         *
         * This node type has no derivation.
         */
        GRAMMAR_DECL(26),
            
        /**
         * Declaration of a language's lexer.
         *
         * This node type has no derivation.
         */
        LEXER_DECL(27),
            
        /**
         * Declaration of a token family.
         *
         * This node type has no derivation.
         */
        LEXER_FAMILY_DECL(28),
            
        /**
         * Logic function declaration.
         *
         * This node type has no derivation.
         */
        SYNTH_FUN_DECL(29),
            
        /**
         * Logic function parameter declaration.
         *
         * This node type has no derivation.
         */
        SYNTH_PARAM_DECL(30),
            
        // TYPE_DECL is abstract
            
        /**
         * Internal type to represent a type that can be matched with anything.
         *
         * This node type has no derivation.
         */
        ANY_TYPE_DECL(31),
            
        /**
         * Alternative for an enum class decl.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_ALT_DECL(32),
            
        /**
         * Function type.
         *
         * This node type has no derivation.
         */
        FUNCTION_TYPE(33),
            
        /**
         * Declaration of a parameter type in a generic declaration.
         *
         * This node type has no derivation.
         */
        GENERIC_PARAM_TYPE_DECL(34),
            
        // NAMED_TYPE_DECL is abstract
            
        // BASIC_CLASS_DECL is abstract
            
        /**
         * Declaration for a LK class. This only cover node classes for the
         * moment, but might be extended to support regular classes in the
         * future.
         *
         * This node type has no derivation.
         */
        CLASS_DECL(35),
            
        /**
         * Declaration for a LK class. This only cover node classes for the
         * moment, but might be extended to support regular classes in the
         * future.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_DECL(36),
            
        /**
         * Enum type declaration.
         *
         * This node type has no derivation.
         */
        ENUM_TYPE_DECL(37),
            
        /**
         * Declaration for a LK struct.
         *
         * This node type has no derivation.
         */
        STRUCT_DECL(38),
            
        /**
         * Trait declaration. For the moment, a Trait can just be used to group
         * behavior for built-in types. It's not usable as a type-bound since
         * we don't have generics, and you cannot implement one either.
         *
         * The reason they're added is to lay down the basics of what we want
         * the Lkt type system to be.
         *
         * TODO: Traits are *not* types. They're treated as such in the grammar
         * for convenience for now, but it's probably not a good idea. Migrate
         * away from this.
         *
         * This node type has no derivation.
         */
        TRAIT_DECL(39),
            
        /**
         * Compile time annotation attached to a declaration.
         *
         * This node type has no derivation.
         */
        DECL_ANNOTATION(40),
            
        /**
         * List of arguments for an annotation with a call syntax. This
         * intermediate node is necessary in order to determine after parsing
         * whether there is no argument list, or if the list is empty.
         *
         * This node type has no derivation.
         */
        DECL_ANNOTATION_ARGS(41),
            
        /**
         * Synthetic node to instantiate a DynamicEnvironment for generics.
         *
         * This node type has no derivation.
         */
        DYN_ENV_WRAPPER(42),
            
        /**
         * Elsif branch of an if expression.
         *
         * This node type has no derivation.
         */
        ELSIF_BRANCH(43),
            
        /**
         * Case branch for an enum class declaration.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_CASE(44),
            
        // EXCLUDES_NULL is abstract
            
        /**
         * This node type has no derivation.
         */
        EXCLUDES_NULL_ABSENT(45),
            
        /**
         * This node type has no derivation.
         */
        EXCLUDES_NULL_PRESENT(46),
            
        // EXPR is abstract
            
        /**
         * "Any of" expression.
         *
         * This node type has no derivation.
         */
        ANY_OF(47),
            
        /**
         * Literal for an array value.
         *
         * This node type has no derivation.
         */
        ARRAY_LITERAL(48),
            
        // BASE_CALL_EXPR is abstract
            
        /**
         * Call expression.
         *
         * This node type has no derivation.
         */
        CALL_EXPR(49),
            
        // LOGIC_CALL_EXPR is abstract
            
        /**
         * Class for "predicate" equations.
         *
         * This node type has no derivation.
         */
        LOGIC_PREDICATE(50),
            
        /**
         * Class for the call inside "propagate" equations.
         *
         * This node type has no derivation.
         */
        LOGIC_PROPAGATE_CALL(51),
            
        /**
         * Binary operator expression.
         *
         * This node type has no derivation.
         */
        BIN_OP(52),
            
        /**
         * Block expression.
         *
         * This node type has no derivation.
         */
        BLOCK_EXPR(53),
            
        /**
         * Cast expression.
         *
         * This node type has no derivation.
         */
        CAST_EXPR(54),
            
        /**
         * Dotted expression.
         *
         * This node type has no derivation.
         */
        DOT_EXPR(55),
            
        /**
         * Expression that throws an error if LHS is null.
         *
         * This node type has no derivation.
         */
        ERROR_ON_NULL(56),
            
        /**
         * Generic instantiation.
         *
         * This node type has no derivation.
         */
        GENERIC_INSTANTIATION(57),
            
        // GRAMMAR_EXPR is abstract
            
        /**
         * Grammar expression for a cut.
         *
         * This node type has no derivation.
         */
        GRAMMAR_CUT(58),
            
        /**
         * Grammar expression to discard the match.
         *
         * This node type has no derivation.
         */
        GRAMMAR_DISCARD(59),
            
        /**
         * Grammar expression (error recovery) to ensure that any nested skip
         * parser calls won't skip certain parse results.
         *
         * This node type has no derivation.
         */
        GRAMMAR_DONT_SKIP(60),
            
        /**
         * Grammar expression to parse lists of results. Results can be
         * separated by a separator. List can be empty ('*') or not ('+').
         *
         * This node type has no derivation.
         */
        GRAMMAR_LIST(61),
            
        /**
         * Grammar expression to parse a null node.
         *
         * This node type has no derivation.
         */
        GRAMMAR_NULL(62),
            
        /**
         * Grammar expression for an optional parsing result.
         *
         * This node type has no derivation.
         */
        GRAMMAR_OPT(63),
            
        /**
         * Grammar expression for an optional parsing result. Missing result
         * creates an error, but parsing continues.
         *
         * This node type has no derivation.
         */
        GRAMMAR_OPT_ERROR(64),
            
        /**
         * Grammar expression for a group of optional parsing results. Failure
         * to parse an optional result creates an error, but parsing continues.
         *
         * This node type has no derivation.
         */
        GRAMMAR_OPT_ERROR_GROUP(65),
            
        /**
         * Grammar expression for a group of optional parsing results.
         *
         * This node type has no derivation.
         */
        GRAMMAR_OPT_GROUP(66),
            
        /**
         * Grammar ``Or`` expression (disjunctive choice between several
         * grammar options).
         *
         * This node type has no derivation.
         */
        GRAMMAR_OR_EXPR(67),
            
        /**
         * Grammar expression to pick the significant parse out of a list of
         * parses (will automatically discard token results).
         *
         * Derived nodes: ``GrammarImplicitPick``
         */
        GRAMMAR_PICK(68),
            
        /**
         * Implicit pick operation.
         *
         * This node type has no derivation.
         */
        GRAMMAR_IMPLICIT_PICK(69),
            
        /**
         * Grammar expression for a predicate: Only parse something if the
         * predicate (that is a reference to a node property) returns True.
         *
         * This node type has no derivation.
         */
        GRAMMAR_PREDICATE(70),
            
        /**
         * Grammar expression for a reference to another grammar rule.
         *
         * This node type has no derivation.
         */
        GRAMMAR_RULE_REF(71),
            
        /**
         * Grammar expression (error recovery) to skip a parsing result.
         *
         * This node type has no derivation.
         */
        GRAMMAR_SKIP(72),
            
        /**
         * Grammar expression for a StopCut.
         *
         * This node type has no derivation.
         */
        GRAMMAR_STOP_CUT(73),
            
        /**
         * Expression for the parsing of a Node.
         *
         * This node type has no derivation.
         */
        PARSE_NODE_EXPR(74),
            
        /**
         * Grammar expression for a token literal.
         *
         * This node type has no derivation.
         */
        TOKEN_LIT(75),
            
        /**
         * Grammar expression for a case insensitive token literal.
         *
         * This node type has no derivation.
         */
        TOKEN_NO_CASE_LIT(76),
            
        /**
         * Grammar expression for the concatenation of two patterns.
         *
         * This node type has no derivation.
         */
        TOKEN_PATTERN_CONCAT(77),
            
        /**
         * Grammar expression for a pattern literal.
         *
         * This node type has no derivation.
         */
        TOKEN_PATTERN_LIT(78),
            
        /**
         * Grammar expression for a token reference.
         *
         * This node type has no derivation.
         */
        TOKEN_REF(79),
            
        /**
         * Identifier.
         *
         * Derived nodes: ``DefId``, ``ModuleRefId``, ``RefId``
         */
        ID(80),
            
        /**
         * Defining identifier.
         *
         * This node type has no derivation.
         */
        DEF_ID(81),
            
        /**
         * Id referencing a langkit module.
         *
         * This node type has no derivation.
         */
        MODULE_REF_ID(82),
            
        /**
         * Reference identifier.
         *
         * This node type has no derivation.
         */
        REF_ID(83),
            
        /**
         * If expression.
         *
         * This node type has no derivation.
         */
        IF_EXPR(84),
            
        /**
         * Isa expression.
         *
         * This node type has no derivation.
         */
        ISA(85),
            
        /**
         * Keep expression.
         *
         * This node type has no derivation.
         */
        KEEP_EXPR(86),
            
        /**
         * Lambda expression.
         *
         * This node type has no derivation.
         */
        LAMBDA_EXPR(87),
            
        // LIT is abstract
            
        /**
         * Big number literal expression.
         *
         * This node type has no derivation.
         */
        BIG_NUM_LIT(88),
            
        /**
         * Character literal expression.
         *
         * This node type has no derivation.
         */
        CHAR_LIT(89),
            
        /**
         * Null literal expression.
         *
         * This node type has no derivation.
         */
        NULL_LIT(90),
            
        /**
         * Number literal expression.
         *
         * This node type has no derivation.
         */
        NUM_LIT(91),
            
        // STRING_LIT is abstract
            
        /**
         * String literal expression, made of multiple line strings.
         *
         * The denoted string value is the concatenation of all line string
         * items. Each line string item must be either:
         *
         * * The empty string designator (``|"``), to denote an empty line
         *   (``\n``).
         *
         * * ``|" <content>``, to designate a non-empty line. The space before
         *   ``<content>`` is mandatory, and is not included in the denoted
         *   string value. ``<content>`` can be anything that appear in a
         *   regular string literal: escape sequences are interpreted the same
         *   way.
         *
         * This node type has no derivation.
         */
        BLOCK_STRING_LIT(92),
            
        /**
         * Single line string literal expression.
         *
         * Note that in order to reduce the size of the node type hierarchy, we
         * define only one node (StringLit) for all our string literals (only
         * regular strings and pattern string literals at the moment). This
         * will also make it easy to add new string prefixes in the future.
         *
         * Derived nodes: ``PatternSingleLineStringLit``
         */
        SINGLE_LINE_STRING_LIT(93),
            
        /**
         * Pattern single line string literal expression.
         *
         * This node type has no derivation.
         */
        PATTERN_SINGLE_LINE_STRING_LIT(94),
            
        /**
         * Class for "assign to logic var" equations.
         *
         * This node type has no derivation.
         */
        LOGIC_ASSIGN(95),
            
        /**
         * Class for logic expressions (any ``basic_expr`` starting with %).
         *
         * This node type has no derivation.
         */
        LOGIC_EXPR(96),
            
        /**
         * Class for "propagate" equations.
         *
         * This node type has no derivation.
         */
        LOGIC_PROPAGATE(97),
            
        /**
         * Class for "unify" equations.
         *
         * This node type has no derivation.
         */
        LOGIC_UNIFY(98),
            
        /**
         * Binary operator expression.
         *
         * This node type has no derivation.
         */
        MATCH_EXPR(99),
            
        /**
         * Boolean negation expression.
         *
         * This node type has no derivation.
         */
        NOT_EXPR(100),
            
        /**
         * Parenthesized expression.
         *
         * This node type has no derivation.
         */
        PAREN_EXPR(101),
            
        /**
         * Raise expression.
         *
         * This node type has no derivation.
         */
        RAISE_EXPR(102),
            
        /**
         * Array subscript expression.
         *
         * This node type has no derivation.
         */
        SUBSCRIPT_EXPR(103),
            
        /**
         * Try expression.
         *
         * This node type has no derivation.
         */
        TRY_EXPR(104),
            
        /**
         * Unary operator expression.
         *
         * This node type has no derivation.
         */
        UN_OP(105),
            
        /**
         * Container for an lkt declaration. Contains the decl node plus the
         * documentation and annotations.
         *
         * This node type has no derivation.
         */
        FULL_DECL(106),
            
        /**
         * Specification for the separator of a list parser.
         *
         * This node type has no derivation.
         */
        GRAMMAR_LIST_SEP(107),
            
        /**
         * Statement to import another source file.
         *
         * This node type has no derivation.
         */
        IMPORT(108),
            
        /**
         * For the moment, root node of a lkt compilation unit.
         *
         * This node type has no derivation.
         */
        LANGKIT_ROOT(109),
            
        /**
         * Lexer construct to introduce a conditional lexing action.
         *
         * This node type has no derivation.
         */
        LEXER_CASE_RULE(110),
            
        /**
         * Lexer construction used by case alternatives to represent the token
         * to send if that alternative was chosen.
         *
         * This node type has no derivation.
         */
        LEXER_CASE_RULE_SEND(111),
            
        // LIST_KIND is abstract
            
        /**
         * This node type has no derivation.
         */
        LIST_KIND_ONE(112),
            
        /**
         * This node type has no derivation.
         */
        LIST_KIND_ZERO(113),
            
        // LKT_NODE_BASE_LIST is abstract
            
        /**
         * List of Argument.
         *
         * This node type has no derivation.
         */
        ARGUMENT_LIST(114),
            
        /**
         * List of BaseLexerCaseRuleAlt.
         *
         * This node type has no derivation.
         */
        BASE_LEXER_CASE_RULE_ALT_LIST(115),
            
        /**
         * List of BaseMatchBranch.
         *
         * This node type has no derivation.
         */
        BASE_MATCH_BRANCH_LIST(116),
            
        /**
         * List of BlockStringLine.
         *
         * This node type has no derivation.
         */
        BLOCK_STRING_LINE_LIST(117),
            
        /**
         * List of CallExpr.
         *
         * This node type has no derivation.
         */
        CALL_EXPR_LIST(118),
            
        /**
         * List of DeclAnnotation.
         *
         * This node type has no derivation.
         */
        DECL_ANNOTATION_LIST(119),
            
        /**
         * List of ElsifBranch.
         *
         * This node type has no derivation.
         */
        ELSIF_BRANCH_LIST(120),
            
        /**
         * List of EnumClassAltDecl.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_ALT_DECL_LIST(121),
            
        /**
         * List of EnumClassCase.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_CASE_LIST(122),
            
        /**
         * List of EnumLitDecl.
         *
         * This node type has no derivation.
         */
        ENUM_LIT_DECL_LIST(123),
            
        /**
         * List of Expr.
         *
         * This list node can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BinOp``, ``BlockExpr``, ``CallExpr``,
         * ``CastExpr``, ``DotExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicAssign``, ``LogicExpr``,
         * ``LogicPredicate``, ``LogicPropagate``, ``LogicUnify``,
         * ``MatchExpr``, ``NotExpr``, ``ParenExpr``, ``RaiseExpr``, ``RefId``,
         * ``SubscriptExpr``, ``TryExpr``, ``UnOp``
         *
         * Derived nodes: ``AnyOfList``
         */
        EXPR_LIST(124),
            
        /**
         * Pipe-separated list of expressions.
         *
         * This is used to represent the "values" operand of an ``AnyOf``
         * expression.
         *
         * This list node can contain one of the following nodes:
         * ``ArrayLiteral``, ``BlockExpr``, ``CallExpr``, ``CastExpr``,
         * ``DotExpr``, ``ErrorOnNull``, ``GenericInstantiation``, ``IfExpr``,
         * ``KeepExpr``, ``LambdaExpr``, ``Lit``, ``LogicExpr``,
         * ``LogicPredicate``, ``MatchExpr``, ``ParenExpr``, ``RaiseExpr``,
         * ``RefId``, ``SubscriptExpr``, ``TryExpr``
         *
         * This node type has no derivation.
         */
        ANY_OF_LIST(125),
            
        /**
         * List of FullDecl.
         *
         * Derived nodes: ``DeclBlock``, ``GenericParamDeclList``
         */
        FULL_DECL_LIST(126),
            
        /**
         * List of declarations that also introduces a containing lexical
         * scope.
         *
         * This node type has no derivation.
         */
        DECL_BLOCK(127),
            
        /**
         * Comma-separated list of generic parameter types.
         *
         * This node type has no derivation.
         */
        GENERIC_PARAM_DECL_LIST(128),
            
        /**
         * List of FunParamDecl.
         *
         * This node type has no derivation.
         */
        FUN_PARAM_DECL_LIST(129),
            
        /**
         * List of GrammarExpr.
         *
         * This node type has no derivation.
         */
        GRAMMAR_EXPR_LIST(130),
            
        /**
         * List of ASTList[GrammarExpr].
         *
         * This node type has no derivation.
         */
        GRAMMAR_EXPR_LIST_LIST(131),
            
        /**
         * List of Import.
         *
         * This node type has no derivation.
         */
        IMPORT_LIST(132),
            
        /**
         * List of LambdaParamDecl.
         *
         * This node type has no derivation.
         */
        LAMBDA_PARAM_DECL_LIST(133),
            
        /**
         * List of LktNode.
         *
         * This list node can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BinOp``, ``BlockExprClause``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``DotExpr``, ``ErrorDecl``,
         * ``ErrorOnNull``, ``FullDecl``, ``GenericInstantiation``, ``IfExpr``,
         * ``Isa``, ``KeepExpr``, ``LambdaExpr``, ``LexerCaseRule``, ``Lit``,
         * ``LogicAssign``, ``LogicExpr``, ``LogicPredicate``,
         * ``LogicPropagate``, ``LogicUnify``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * This node type has no derivation.
         */
        LKT_NODE_LIST(134),
            
        /**
         * List of PatternDetail.
         *
         * This node type has no derivation.
         */
        PATTERN_DETAIL_LIST(135),
            
        /**
         * List of Pattern.
         *
         * This list node can contain one of the following nodes:
         * ``BindingPattern``, ``BoolPattern``, ``EllipsisPattern``,
         * ``ExtendedPattern``, ``IntegerPattern``, ``ListPattern``,
         * ``NotPattern``, ``NullPattern``, ``ParenPattern``, ``RegexPattern``,
         * ``TuplePattern``, ``TypePattern``
         *
         * This node type has no derivation.
         */
        PATTERN_LIST(136),
            
        /**
         * List of RefId.
         *
         * This node type has no derivation.
         */
        REF_ID_LIST(137),
            
        /**
         * List of TypeRef.
         *
         * This list node can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * Derived nodes: ``SyntheticTypeRefList``
         */
        TYPE_REF_LIST(138),
            
        /**
         * Synthetic list of type references, used to create synthetic type
         * declarations.
         *
         * This node type has no derivation.
         */
        SYNTHETIC_TYPE_REF_LIST(139),
            
        // NULL_COND_QUALIFIER is abstract
            
        /**
         * This node type has no derivation.
         */
        NULL_COND_QUALIFIER_ABSENT(140),
            
        /**
         * This node type has no derivation.
         */
        NULL_COND_QUALIFIER_PRESENT(141),
            
        // OP is abstract
            
        /**
         * This node type has no derivation.
         */
        OP_AMP(142),
            
        /**
         * This node type has no derivation.
         */
        OP_AND(143),
            
        /**
         * This node type has no derivation.
         */
        OP_DIV(144),
            
        /**
         * This node type has no derivation.
         */
        OP_EQ(145),
            
        /**
         * This node type has no derivation.
         */
        OP_GT(146),
            
        /**
         * This node type has no derivation.
         */
        OP_GTE(147),
            
        /**
         * This node type has no derivation.
         */
        OP_LOGIC_AND(148),
            
        /**
         * This node type has no derivation.
         */
        OP_LOGIC_OR(149),
            
        /**
         * This node type has no derivation.
         */
        OP_LT(150),
            
        /**
         * This node type has no derivation.
         */
        OP_LTE(151),
            
        /**
         * This node type has no derivation.
         */
        OP_MINUS(152),
            
        /**
         * This node type has no derivation.
         */
        OP_MULT(153),
            
        /**
         * This node type has no derivation.
         */
        OP_NE(154),
            
        /**
         * This node type has no derivation.
         */
        OP_OR(155),
            
        /**
         * This node type has no derivation.
         */
        OP_OR_INT(156),
            
        /**
         * This node type has no derivation.
         */
        OP_PLUS(157),
            
        // PATTERN is abstract
            
        /**
         * Pattern that allows to match any type. Only usable as an
         * ExtendedPattern's left side pattern:
         *
         * .. code::
         *
         *    *(f_field: BasicDecl(...))
         *
         * For the general case of matching any value, the idiom is to use a
         * binding pattern with no right hand side:
         *
         * .. code::
         *
         *    _ => true
         *
         * This node type has no derivation.
         */
        ANY_TYPE_PATTERN(158),
            
        /**
         * Pattern comprising a binding name and a value pattern.
         *
         * For instance:
         *
         * .. code::
         *
         *    o@ObjectDecl
         *
         * This node type has no derivation.
         */
        BINDING_PATTERN(159),
            
        // BOOL_PATTERN is abstract
            
        /**
         * This node type has no derivation.
         */
        BOOL_PATTERN_FALSE(160),
            
        /**
         * This node type has no derivation.
         */
        BOOL_PATTERN_TRUE(161),
            
        /**
         * Pattern to match any remaining number of elements in a list pattern.
         *
         * This node type has no derivation.
         */
        ELLIPSIS_PATTERN(162),
            
        /**
         * Pattern that takes a base pattern, and adds details to match on the
         * shape of what is being matched. The syntactic form is:
         *
         * .. code::
         *
         *    <sub_pattern>(<detail>, <detail>, ...)
         *
         * For instance:
         *
         * .. code::
         *
         *    ObjectDecl(any children: AspectAssoc)
         *
         * This node type has no derivation.
         */
        EXTENDED_PATTERN(163),
            
        /**
         * Pattern with a filtering predicate, of the form: ``<pattern> when
         * <predicate>``
         *
         * For instance:
         *
         * .. code::
         *
         *    o@ObjectDecl when o.children.length == 3
         *
         * This node type has no derivation.
         */
        FILTERED_PATTERN(164),
            
        /**
         * Pattern to match on integers.
         *
         * This node type has no derivation.
         */
        INTEGER_PATTERN(165),
            
        /**
         * Pattern to match on lists.
         *
         * This node type has no derivation.
         */
        LIST_PATTERN(166),
            
        /**
         * Pattern that matches if its inner pattern doesn't match.
         *
         * For instance:
         *
         * .. code::
         *
         *    val non_objects = select not ObjectDecl
         *
         * This node type has no derivation.
         */
        NOT_PATTERN(167),
            
        /**
         * Null pattern. Will only match the null value.
         *
         * This node type has no derivation.
         */
        NULL_PATTERN(168),
            
        /**
         * Pattern that matches if any of its subpatterns matches.
         *
         * For instance:
         *
         * .. code::
         *
         *    val value_decls = select ObjectDecl | ParamSpec
         *
         * This node type has no derivation.
         */
        OR_PATTERN(169),
            
        /**
         * A syntactically parenthesized pattern. Has no effect, only used to
         * disambiguate syntax.
         *
         * This node type has no derivation.
         */
        PAREN_PATTERN(170),
            
        /**
         * Pattern that considers the value as text and matches it against the
         * given regular expression.
         *
         * This node type has no derivation.
         */
        REGEX_PATTERN(171),
            
        /**
         * Pattern to match on tuples.
         *
         * This node type has no derivation.
         */
        TUPLE_PATTERN(172),
            
        /**
         * Pattern matching on a specific type.
         *
         * This node type has no derivation.
         */
        TYPE_PATTERN(173),
            
        // PATTERN_DETAIL is abstract
            
        /**
         * Pattern detail denoting an access to a field.
         *
         * This node type has no derivation.
         */
        FIELD_PATTERN_DETAIL(174),
            
        /**
         * Pattern detail denoting an access to a property in a node pattern.
         *
         * This node type has no derivation.
         */
        PROPERTY_PATTERN_DETAIL(175),
            
        /**
         * Pattern detail denoting the use of a selector in a node pattern
         *
         * This node type has no derivation.
         */
        SELECTOR_PATTERN_DETAIL(176),
            
        /**
         * Root node for selector patterns
         *
         * This node type has no derivation.
         */
        SELECTOR_CALL(177),
            
        // TYPE_REF is abstract
            
        /**
         * "list" type reference in parsers.
         *
         * This node type has no derivation.
         */
        DEFAULT_LIST_TYPE_REF(178),
            
        /**
         * Reference to a function type.
         *
         * This node type has no derivation.
         */
        FUNCTION_TYPE_REF(179),
            
        /**
         * Reference to a generic type.
         *
         * This node type has no derivation.
         */
        GENERIC_TYPE_REF(180),
            
        /**
         * Simple reference to a type.
         *
         * This node type has no derivation.
         */
        SIMPLE_TYPE_REF(181),
            
        /**
         * Dynamic var bind expression.
         *
         * This node type has no derivation.
         */
        VAR_BIND(182),
        ;

        // ----- Class attributes -----

        /** Map containing relation from node kind value and enum instance. */
        private static final Map<Integer, NodeKind> map = new HashMap<>();

        // ----- Instance attributes -----

        /** Integer value of the node kind. */
        public final int value;

        /** Description associated to the node kind. */
        private Reflection.Node description;

        // ----- Constructors -----

        static {
            // Initialize the lookup map
            for(NodeKind elem : NodeKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private NodeKind(
            final int value
        ) {
            this.value = value;

            // The "description" field is intiialized with null to avoid
            // static code execution order issues.
            this.description = null;
        }

        // ----- Class methods -----

        /**
         * Get the enum instance from the given C integer value.
         *
         * @throws EnumException If the given C value is not a valid enum
         *                       value.
         */
        public static NodeKind fromC(
            final int cValue
        ) {
            if(!map.containsKey(cValue))
                throw new EnumException("Cannot get NodeKind from " + cValue);
            return (NodeKind) map.get(cValue);
        }

        // ----- Instance methods -----

        /** Get the C integer value of the enum instance. */
        public int toC() {
            return this.value;
        }

        public Reflection.Node getDescription() {
            return this.description;
        }

        void setDescription(Reflection.Node description) {
            this.description = description;
        }

    }

    /** This enum contains all language nodes' member references */
    public enum MemberReference implements
        LangkitSupport.MemberReferenceInterface {

        // ----- Enum values -----

            
        LKT_COMPLETE_ITEM_DECLARATION(1),
            
        LKT_DECODED_CHAR_VALUE_VALUE(2),
            
        LKT_DECODED_CHAR_VALUE_HAS_ERROR(3),
            
        LKT_DECODED_CHAR_VALUE_ERROR_SLOC(4),
            
        LKT_DECODED_CHAR_VALUE_ERROR_MESSAGE(5),
            
        LKT_DECODED_STRING_VALUE_VALUE(6),
            
        LKT_DECODED_STRING_VALUE_HAS_ERROR(7),
            
        LKT_DECODED_STRING_VALUE_ERROR_SLOC(8),
            
        LKT_DECODED_STRING_VALUE_ERROR_MESSAGE(9),
            
        LKT_LOGIC_CONTEXT_REF_NODE(10),
            
        LKT_LOGIC_CONTEXT_DECL_NODE(11),
            
        LKT_REF_RESULT_REF(12),
            
        LKT_SOLVER_DIAGNOSTIC_MESSAGE_TEMPLATE(13),
            
        LKT_SOLVER_DIAGNOSTIC_ARGS(14),
            
        LKT_SOLVER_DIAGNOSTIC_LOCATION(15),
            
        LKT_SOLVER_DIAGNOSTIC_CONTEXTS(16),
            
        LKT_SOLVER_DIAGNOSTIC_ROUND(17),
            
        LKT_SOLVER_RESULT_SUCCESS(18),
            
        LKT_SOLVER_RESULT_DIAGNOSTICS(19),
            
        LKT_ARGUMENT_F_NAME(20),
            
        LKT_ARGUMENT_F_VALUE(21),
            
        LKT_BASE_LEXER_CASE_RULE_ALT_F_SEND(22),
            
        LKT_LEXER_CASE_RULE_COND_ALT_F_COND_EXPRS(23),
            
        LKT_BASE_MATCH_BRANCH_F_EXPR(24),
            
        LKT_MATCH_BRANCH_F_DECL(25),
            
        LKT_PATTERN_MATCH_BRANCH_F_PATTERN(26),
            
        LKT_BLOCK_EXPR_CLAUSE_F_CLAUSE(27),
            
        LKT_DECL_F_SYN_NAME(28),
            
        LKT_BASE_GRAMMAR_RULE_DECL_F_EXPR(29),
            
        LKT_EXPLICITLY_TYPED_DECL_F_DECL_TYPE(30),
            
        LKT_COMPONENT_DECL_F_DEFAULT_VAL(31),
            
        LKT_FIELD_DECL_F_TRAIT_REF(32),
            
        LKT_FUN_PARAM_DECL_F_DECL_ANNOTATIONS(33),
            
        LKT_VAL_DECL_F_EXPR(34),
            
        LKT_FUN_DECL_F_PARAMS(35),
            
        LKT_FUN_DECL_F_RETURN_TYPE(36),
            
        LKT_FUN_DECL_F_TRAIT_REF(37),
            
        LKT_FUN_DECL_F_BODY(38),
            
        LKT_ENV_SPEC_DECL_F_ACTIONS(39),
            
        LKT_GENERIC_DECL_F_GENERIC_PARAM_DECLS(40),
            
        LKT_GENERIC_DECL_F_DECL(41),
            
        LKT_GRAMMAR_DECL_F_RULES(42),
            
        LKT_LEXER_DECL_F_RULES(43),
            
        LKT_LEXER_FAMILY_DECL_F_RULES(44),
            
        LKT_TYPE_DECL_F_TRAITS(45),
            
        LKT_TYPE_DECL_F_SYN_BASE_TYPE(46),
            
        LKT_GENERIC_PARAM_TYPE_DECL_F_HAS_CLASS(47),
            
        LKT_NAMED_TYPE_DECL_F_DECLS(48),
            
        LKT_ENUM_CLASS_DECL_F_BRANCHES(49),
            
        LKT_ENUM_TYPE_DECL_F_LITERALS(50),
            
        LKT_DECL_ANNOTATION_F_NAME(51),
            
        LKT_DECL_ANNOTATION_F_ARGS(52),
            
        LKT_DECL_ANNOTATION_ARGS_F_ARGS(53),
            
        LKT_ELSIF_BRANCH_F_COND_EXPR(54),
            
        LKT_ELSIF_BRANCH_F_THEN_EXPR(55),
            
        LKT_ENUM_CLASS_CASE_F_DECLS(56),
            
        LKT_ANY_OF_F_EXPR(57),
            
        LKT_ANY_OF_F_VALUES(58),
            
        LKT_ARRAY_LITERAL_F_EXPRS(59),
            
        LKT_ARRAY_LITERAL_F_ELEMENT_TYPE(60),
            
        LKT_BASE_CALL_EXPR_F_NAME(61),
            
        LKT_BASE_CALL_EXPR_F_ARGS(62),
            
        LKT_BIN_OP_F_LEFT(63),
            
        LKT_BIN_OP_F_OP(64),
            
        LKT_BIN_OP_F_RIGHT(65),
            
        LKT_BLOCK_EXPR_F_CLAUSES(66),
            
        LKT_CAST_EXPR_F_EXPR(67),
            
        LKT_CAST_EXPR_F_NULL_COND(68),
            
        LKT_CAST_EXPR_F_EXCLUDES_NULL(69),
            
        LKT_CAST_EXPR_F_DEST_TYPE(70),
            
        LKT_DOT_EXPR_F_PREFIX(71),
            
        LKT_DOT_EXPR_F_NULL_COND(72),
            
        LKT_DOT_EXPR_F_SUFFIX(73),
            
        LKT_ERROR_ON_NULL_F_EXPR(74),
            
        LKT_GENERIC_INSTANTIATION_F_NAME(75),
            
        LKT_GENERIC_INSTANTIATION_F_ARGS(76),
            
        LKT_GRAMMAR_DISCARD_F_EXPR(77),
            
        LKT_GRAMMAR_DONT_SKIP_F_EXPR(78),
            
        LKT_GRAMMAR_DONT_SKIP_F_DONT_SKIP(79),
            
        LKT_GRAMMAR_LIST_F_LIST_TYPE(80),
            
        LKT_GRAMMAR_LIST_F_KIND(81),
            
        LKT_GRAMMAR_LIST_F_EXPR(82),
            
        LKT_GRAMMAR_LIST_F_SEP(83),
            
        LKT_GRAMMAR_NULL_F_NAME(84),
            
        LKT_GRAMMAR_OPT_F_EXPR(85),
            
        LKT_GRAMMAR_OPT_ERROR_F_EXPR(86),
            
        LKT_GRAMMAR_OPT_ERROR_GROUP_F_EXPR(87),
            
        LKT_GRAMMAR_OPT_GROUP_F_EXPR(88),
            
        LKT_GRAMMAR_OR_EXPR_F_SUB_EXPRS(89),
            
        LKT_GRAMMAR_PICK_F_EXPRS(90),
            
        LKT_GRAMMAR_PREDICATE_F_EXPR(91),
            
        LKT_GRAMMAR_PREDICATE_F_PROP_REF(92),
            
        LKT_GRAMMAR_RULE_REF_F_NODE_NAME(93),
            
        LKT_GRAMMAR_SKIP_F_NAME(94),
            
        LKT_GRAMMAR_STOP_CUT_F_EXPR(95),
            
        LKT_PARSE_NODE_EXPR_F_NODE_NAME(96),
            
        LKT_PARSE_NODE_EXPR_F_SUB_EXPRS(97),
            
        LKT_TOKEN_NO_CASE_LIT_F_LIT(98),
            
        LKT_TOKEN_PATTERN_CONCAT_F_LEFT(99),
            
        LKT_TOKEN_PATTERN_CONCAT_F_RIGHT(100),
            
        LKT_TOKEN_REF_F_TOKEN_NAME(101),
            
        LKT_TOKEN_REF_F_EXPR(102),
            
        LKT_IF_EXPR_F_COND_EXPR(103),
            
        LKT_IF_EXPR_F_THEN_EXPR(104),
            
        LKT_IF_EXPR_F_ALTERNATIVES(105),
            
        LKT_IF_EXPR_F_ELSE_EXPR(106),
            
        LKT_ISA_F_EXPR(107),
            
        LKT_ISA_F_PATTERN(108),
            
        LKT_KEEP_EXPR_F_EXPR(109),
            
        LKT_KEEP_EXPR_F_NULL_COND(110),
            
        LKT_KEEP_EXPR_F_KEEP_TYPE(111),
            
        LKT_LAMBDA_EXPR_F_PARAMS(112),
            
        LKT_LAMBDA_EXPR_F_RETURN_TYPE(113),
            
        LKT_LAMBDA_EXPR_F_BODY(114),
            
        LKT_NULL_LIT_F_DEST_TYPE(115),
            
        LKT_BLOCK_STRING_LIT_F_LINES(116),
            
        LKT_LOGIC_ASSIGN_F_DEST_VAR(117),
            
        LKT_LOGIC_ASSIGN_F_VALUE(118),
            
        LKT_LOGIC_EXPR_F_EXPR(119),
            
        LKT_LOGIC_PROPAGATE_F_DEST_VAR(120),
            
        LKT_LOGIC_PROPAGATE_F_CALL(121),
            
        LKT_LOGIC_UNIFY_F_LHS(122),
            
        LKT_LOGIC_UNIFY_F_RHS(123),
            
        LKT_MATCH_EXPR_F_MATCH_EXPR(124),
            
        LKT_MATCH_EXPR_F_BRANCHES(125),
            
        LKT_NOT_EXPR_F_EXPR(126),
            
        LKT_PAREN_EXPR_F_EXPR(127),
            
        LKT_RAISE_EXPR_F_DEST_TYPE(128),
            
        LKT_RAISE_EXPR_F_EXCEPT_EXPR(129),
            
        LKT_SUBSCRIPT_EXPR_F_PREFIX(130),
            
        LKT_SUBSCRIPT_EXPR_F_NULL_COND(131),
            
        LKT_SUBSCRIPT_EXPR_F_INDEX(132),
            
        LKT_TRY_EXPR_F_TRY_EXPR(133),
            
        LKT_TRY_EXPR_F_OR_EXPR(134),
            
        LKT_UN_OP_F_OP(135),
            
        LKT_UN_OP_F_EXPR(136),
            
        LKT_FULL_DECL_F_DOC(137),
            
        LKT_FULL_DECL_F_DECL_ANNOTATIONS(138),
            
        LKT_FULL_DECL_F_DECL(139),
            
        LKT_GRAMMAR_LIST_SEP_F_TOKEN(140),
            
        LKT_GRAMMAR_LIST_SEP_F_EXTRA(141),
            
        LKT_IMPORT_F_NAME(142),
            
        LKT_LANGKIT_ROOT_F_IMPORTS(143),
            
        LKT_LANGKIT_ROOT_F_DECLS(144),
            
        LKT_LEXER_CASE_RULE_F_EXPR(145),
            
        LKT_LEXER_CASE_RULE_F_ALTS(146),
            
        LKT_LEXER_CASE_RULE_SEND_F_SENT(147),
            
        LKT_LEXER_CASE_RULE_SEND_F_MATCH_SIZE(148),
            
        LKT_BINDING_PATTERN_F_DECL(149),
            
        LKT_BINDING_PATTERN_F_SUB_PATTERN(150),
            
        LKT_ELLIPSIS_PATTERN_F_BINDING(151),
            
        LKT_EXTENDED_PATTERN_F_SUB_PATTERN(152),
            
        LKT_EXTENDED_PATTERN_F_DETAILS(153),
            
        LKT_FILTERED_PATTERN_F_SUB_PATTERN(154),
            
        LKT_FILTERED_PATTERN_F_PREDICATE(155),
            
        LKT_LIST_PATTERN_F_SUB_PATTERNS(156),
            
        LKT_NOT_PATTERN_F_SUB_PATTERN(157),
            
        LKT_OR_PATTERN_F_LEFT_SUB_PATTERN(158),
            
        LKT_OR_PATTERN_F_RIGHT_SUB_PATTERN(159),
            
        LKT_PAREN_PATTERN_F_SUB_PATTERN(160),
            
        LKT_TUPLE_PATTERN_F_SUB_PATTERNS(161),
            
        LKT_TYPE_PATTERN_F_TYPE_NAME(162),
            
        LKT_FIELD_PATTERN_DETAIL_F_ID(163),
            
        LKT_FIELD_PATTERN_DETAIL_F_EXPECTED_VALUE(164),
            
        LKT_PROPERTY_PATTERN_DETAIL_F_CALL(165),
            
        LKT_PROPERTY_PATTERN_DETAIL_F_EXPECTED_VALUE(166),
            
        LKT_SELECTOR_PATTERN_DETAIL_F_CALL(167),
            
        LKT_SELECTOR_PATTERN_DETAIL_F_SUB_PATTERN(168),
            
        LKT_SELECTOR_CALL_F_QUANTIFIER(169),
            
        LKT_SELECTOR_CALL_F_BINDING(170),
            
        LKT_SELECTOR_CALL_F_SELECTOR_CALL(171),
            
        LKT_FUNCTION_TYPE_REF_F_PARAM_TYPES(172),
            
        LKT_FUNCTION_TYPE_REF_F_RETURN_TYPE(173),
            
        LKT_GENERIC_TYPE_REF_F_TYPE_NAME(174),
            
        LKT_GENERIC_TYPE_REF_F_ARGS(175),
            
        LKT_SIMPLE_TYPE_REF_F_TYPE_NAME(176),
            
        LKT_VAR_BIND_F_NAME(177),
            
        LKT_VAR_BIND_F_EXPR(178),
            
        LKT_PARENT(179),
            
        LKT_PARENTS(180),
            
        LKT_CHILDREN(181),
            
        LKT_TOKEN_START(182),
            
        LKT_TOKEN_END(183),
            
        LKT_CHILD_INDEX(184),
            
        LKT_PREVIOUS_SIBLING(185),
            
        LKT_NEXT_SIBLING(186),
            
        LKT_UNIT(187),
            
        LKT_IS_GHOST(188),
            
        LKT_FULL_SLOC_IMAGE(189),
            
        LKT_COMPLETION_ITEM_KIND_TO_INT(190),
            
        LKT_LKT_NODE_P_SET_SOLVER_DEBUG_MODE(191),
            
        LKT_LKT_NODE_P_BASIC_TRAIT_GEN(192),
            
        LKT_LKT_NODE_P_BASIC_TRAIT(193),
            
        LKT_LKT_NODE_P_NODE_GEN_TRAIT(194),
            
        LKT_LKT_NODE_P_NODE_TRAIT(195),
            
        LKT_LKT_NODE_P_INDEXABLE_GEN_TRAIT(196),
            
        LKT_LKT_NODE_P_INDEXABLE_TRAIT(197),
            
        LKT_LKT_NODE_P_TOKEN_NODE_TRAIT(198),
            
        LKT_LKT_NODE_P_ERROR_NODE_TRAIT(199),
            
        LKT_LKT_NODE_P_CHAR_TYPE(200),
            
        LKT_LKT_NODE_P_INT_TYPE(201),
            
        LKT_LKT_NODE_P_BOOL_TYPE(202),
            
        LKT_LKT_NODE_P_BIGINT_TYPE(203),
            
        LKT_LKT_NODE_P_STRING_TYPE(204),
            
        LKT_LKT_NODE_P_SYMBOL_TYPE(205),
            
        LKT_LKT_NODE_P_PROPERTY_ERROR_TYPE(206),
            
        LKT_LKT_NODE_P_REGEXP_TYPE(207),
            
        LKT_LKT_NODE_P_ENTITY_GEN_TYPE(208),
            
        LKT_LKT_NODE_P_ENTITY_TYPE(209),
            
        LKT_LKT_NODE_P_LOGICVAR_TYPE(210),
            
        LKT_LKT_NODE_P_EQUATION_TYPE(211),
            
        LKT_LKT_NODE_P_ARRAY_GEN_TYPE(212),
            
        LKT_LKT_NODE_P_ARRAY_TYPE(213),
            
        LKT_LKT_NODE_P_ASTLIST_GEN_TYPE(214),
            
        LKT_LKT_NODE_P_ASTLIST_TYPE(215),
            
        LKT_LKT_NODE_P_NODE_BUILDER_GEN_TYPE(216),
            
        LKT_LKT_NODE_P_NODE_BUILDER_TYPE(217),
            
        LKT_LKT_NODE_P_ITERATOR_GEN_TRAIT(218),
            
        LKT_LKT_NODE_P_ITERATOR_TRAIT(219),
            
        LKT_LKT_NODE_P_ANALYSIS_UNIT_GEN_TRAIT(220),
            
        LKT_LKT_NODE_P_ANALYSIS_UNIT_TRAIT(221),
            
        LKT_LKT_NODE_P_TOPMOST_INVALID_DECL(222),
            
        LKT_LKT_NODE_P_NAMERES_DIAGNOSTICS(223),
            
        LKT_LKT_NODE_P_SOLVE_ENCLOSING_CONTEXT(224),
            
        LKT_LKT_NODE_P_XREF_ENTRY_POINT(225),
            
        LKT_LKT_NODE_P_COMPLETE(226),
            
        LKT_BASE_MATCH_BRANCH_P_MATCH_PART(227),
            
        LKT_CLASS_QUALIFIER_P_AS_BOOL(228),
            
        LKT_DECL_P_CUSTOM_IMAGE(229),
            
        LKT_DECL_P_DECL_TYPE_NAME(230),
            
        LKT_DECL_P_DEF_IDS(231),
            
        LKT_DECL_P_AS_BARE_DECL(232),
            
        LKT_DECL_P_GET_TYPE(233),
            
        LKT_DECL_P_GET_CAST_TYPE(234),
            
        LKT_DECL_P_GET_KEEP_TYPE(235),
            
        LKT_DECL_P_GET_SUFFIX_TYPE(236),
            
        LKT_DECL_P_IS_GENERIC(237),
            
        LKT_DECL_P_RETURN_TYPE_IS_INSTANTIATED(238),
            
        LKT_DECL_P_IS_INSTANTIATED(239),
            
        LKT_DECL_P_NAME(240),
            
        LKT_DECL_P_FULL_NAME(241),
            
        LKT_FUN_DECL_P_IS_DYNAMIC_COMBINER(242),
            
        LKT_FUN_DECL_P_FIND_ALL_OVERRIDES(243),
            
        LKT_TYPE_DECL_P_DEF_ID(244),
            
        LKT_TYPE_DECL_P_BASE_TYPE(245),
            
        LKT_TYPE_DECL_P_BASE_TYPE_IF_ENTITY(246),
            
        LKT_EXCLUDES_NULL_P_AS_BOOL(247),
            
        LKT_EXPR_P_GET_TYPE(248),
            
        LKT_EXPR_P_GET_GENERIC_TYPE(249),
            
        LKT_EXPR_P_GET_EXPECTED_TYPE(250),
            
        LKT_EXPR_P_REFERENCED_DECL(251),
            
        LKT_TOKEN_LIT_P_DENOTED_VALUE(252),
            
        LKT_TOKEN_PATTERN_LIT_P_DENOTED_VALUE(253),
            
        LKT_ID_P_CUSTOM_IMAGE(254),
            
        LKT_DEF_ID_P_NAME(255),
            
        LKT_DEF_ID_P_GET_IMPLEMENTATINONS(256),
            
        LKT_DEF_ID_P_DECL_DETAIL(257),
            
        LKT_DEF_ID_P_COMPLETION_ITEM_KIND(258),
            
        LKT_DEF_ID_P_DOC(259),
            
        LKT_DEF_ID_P_FIND_ALL_REFERENCES(260),
            
        LKT_REF_ID_P_REFERENCED_DEFINING_NAME(261),
            
        LKT_CHAR_LIT_P_DENOTED_VALUE(262),
            
        LKT_STRING_LIT_P_DENOTED_VALUE(263),
            
        LKT_STRING_LIT_P_IS_PREFIXED_STRING(264),
            
        LKT_STRING_LIT_P_PREFIX(265),
            
        LKT_STRING_LIT_P_IS_REGEXP_LITERAL(266),
            
        LKT_FULL_DECL_P_HAS_ANNOTATION(267),
            
        LKT_IMPORT_P_REFERENCED_UNIT(268),
            
        LKT_LANGKIT_ROOT_P_FETCH_PRELUDE(269),
            
        LKT_NULL_COND_QUALIFIER_P_AS_BOOL(270),
            
        LKT_TYPE_REF_P_REFERENCED_DECL(271),
        ;

        // ----- Class attributes -----

        /** Map containing relation from native value to Jave instance. */
        private static Map<Integer, MemberReference> map = new HashMap<>();

        // ----- Instance attribtues -----

        /** Native value if the member reference. */
        private final int value;

        // ----- Constructors -----

        /** Private constructor. */
        private MemberReference (
            final int value
        ) {
            this.value = value;
        }

        static {
            // Initialize the lookup map
            for(MemberReference elem : MemberReference.values()) {
                map.put(elem.value, elem);
            }
        }

        // ----- Class methods -----

        /**
         * Get the enum instance from the given C integer value.
         *
         * @throws EnumException If the given C value is not a valid enum
         *                       value.
         */
        public static MemberReference fromC(
            final int cValue
        ) {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get MemberReference from " + cValue
                );
            return (MemberReference) map.get(cValue);
        }

        // ----- Instance methods -----

        /** Get the native value of the enum instance. */
        public int toC() {
            return this.value;
        }

    }

    // ===== Generated enums =====

    
    

    /**
     * Specify a kind of analysis unit. Specification units provide an
     * interface to the outer world while body units provide an implementation
     * for the corresponding interface.
     */
    public enum AnalysisUnitKind {

        // ----- Enum values -----

        UNIT_SPECIFICATION(0),
        UNIT_BODY(1),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final AnalysisUnitKind NONE =
            UNIT_SPECIFICATION;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, AnalysisUnitKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(AnalysisUnitKind elem : AnalysisUnitKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private AnalysisUnitKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to an enum value.
         */
        static AnalysisUnitKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static AnalysisUnitKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get AnalysisUnitKind from " + cValue
                );
            return (AnalysisUnitKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Type of completion item. Refer to the official LSP specification.
     */
    public enum CompletionItemKind {

        // ----- Enum values -----

        TEXT_KIND(0),
        METHOD_KIND(1),
        FUNCTION_KIND(2),
        CONSTRUCTOR_KIND(3),
        FIELD_KIND(4),
        VARIABLE_KIND(5),
        CLASS_KIND(6),
        INTERFACE_KIND(7),
        MODULE_KIND(8),
        PROPERTY_KIND(9),
        UNIT_KIND(10),
        VALUE_KIND(11),
        ENUM_KIND(12),
        KEYWORD_KIND(13),
        SNIPPET_KIND(14),
        COLOR_KIND(15),
        FILE_KIND(16),
        REFERENCE_KIND(17),
        FOLDER_KIND(18),
        ENUM_MEMBER_KIND(19),
        CONSTANT_KIND(20),
        STRUCT_KIND(21),
        EVENT_KIND(22),
        OPERATOR_KIND(23),
        TYPE_PARAMETER_KIND(24),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final CompletionItemKind NONE =
            TEXT_KIND;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, CompletionItemKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(CompletionItemKind elem : CompletionItemKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private CompletionItemKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to an enum value.
         */
        static CompletionItemKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static CompletionItemKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get CompletionItemKind from " + cValue
                );
            return (CompletionItemKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Discriminant for DesignatedEnv structures.
     */
    public enum DesignatedEnvKind {

        // ----- Enum values -----

        NONE_ENUM(0),
        CURRENT_ENV(1),
        NAMED_ENV(2),
        DIRECT_ENV(3),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final DesignatedEnvKind NONE =
            NONE_ENUM;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, DesignatedEnvKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(DesignatedEnvKind elem : DesignatedEnvKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private DesignatedEnvKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to an enum value.
         */
        static DesignatedEnvKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static DesignatedEnvKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get DesignatedEnvKind from " + cValue
                );
            return (DesignatedEnvKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Gramar rule to use for parsing.
     */
    public enum GrammarRule {

        // ----- Enum values -----

        MAIN_RULE_RULE(0),
        ID_RULE(1),
        REF_ID_RULE(2),
        TYPE_REF_ID_RULE(3),
        DEF_ID_RULE(4),
        DOC_RULE(5),
        IMPORT_STMT_RULE(6),
        IMPORTS_RULE(7),
        LEXER_DECL_RULE(8),
        GRAMMAR_DECL_RULE(9),
        GRAMMAR_RULE_RULE(10),
        LEXER_RULE_RULE(11),
        LEXER_FAMILY_DECL_RULE(12),
        LEXER_CASE_RULE_RULE(13),
        LEXER_CASE_ALT_RULE(14),
        LEXER_CASE_SEND_RULE(15),
        GRAMMAR_PRIMARY_RULE(16),
        GRAMMAR_EXPR_RULE(17),
        GRAMMAR_PICK_RULE(18),
        GRAMMAR_IMPLICIT_PICK_RULE(19),
        GRAMMAR_OPT_RULE(20),
        GRAMMAR_OPT_ERROR_RULE(21),
        GRAMMAR_CUT_RULE(22),
        GRAMMAR_STOPCUT_RULE(23),
        GRAMMAR_OR_EXPR_RULE(24),
        GRAMMAR_DISCARD_EXPR_RULE(25),
        TOKEN_LITERAL_RULE(26),
        TOKEN_NO_CASE_LITERAL_RULE(27),
        TOKEN_PATTERN_RULE(28),
        TOKEN_PATTERN_LITERAL_RULE(29),
        PARSE_NODE_EXPR_RULE(30),
        GRAMMAR_RULE_REF_RULE(31),
        GRAMMAR_LIST_EXPR_RULE(32),
        GRAMMAR_LIST_SEP_RULE(33),
        GRAMMAR_SKIP_RULE(34),
        GRAMMAR_NULL_RULE(35),
        GRAMMAR_TOKEN_RULE(36),
        TYPE_DECL_RULE(37),
        GENERIC_DECL_RULE(38),
        GENERIC_PARAM_TYPE_RULE(39),
        ENUM_LIT_DECL_RULE(40),
        FUN_DECL_RULE(41),
        LAMBDA_PARAM_DECL_RULE(42),
        FUN_PARAM_DECL_RULE(43),
        FUN_PARAM_LIST_RULE(44),
        LAMBDA_PARAM_LIST_RULE(45),
        FIELD_DECL_RULE(46),
        BARE_DECL_RULE(47),
        DECL_RULE(48),
        TYPE_MEMBER_REF_RULE(49),
        TYPE_EXPR_RULE(50),
        TYPE_REF_RULE(51),
        TYPE_LIST_RULE(52),
        DECLS_RULE(53),
        DECL_BLOCK_RULE(54),
        VAL_DECL_RULE(55),
        DYNVAR_DECL_RULE(56),
        VAR_BIND_RULE(57),
        ENV_SPEC_ACTION_RULE(58),
        ENV_SPEC_DECL_RULE(59),
        BLOCK_RULE(60),
        PATTERN_RULE(61),
        FIL_PATTERN_RULE(62),
        VALUE_PATTERN_RULE(63),
        REGEX_PATTERN_RULE(64),
        BOOL_PATTERN_RULE(65),
        ELLIPSIS_PATTERN_RULE(66),
        INTEGER_PATTERN_RULE(67),
        LIST_PATTERN_RULE(68),
        TUPLE_PATTERN_RULE(69),
        PATTERN_ARG_RULE(70),
        SELECTOR_CALL_RULE(71),
        EXPR_RULE(72),
        REL_RULE(73),
        EQ_RULE(74),
        ARITH_1_RULE(75),
        ARITH_2_RULE(76),
        ARITH_3_RULE(77),
        ISA_OR_PRIMARY_RULE(78),
        LOGIC_PROPAGATE_CALL_RULE(79),
        PRIMARY_RULE(80),
        MATCH_EXPR_RULE(81),
        NUM_LIT_RULE(82),
        BIG_NUM_LIT_RULE(83),
        STRING_LIT_RULE(84),
        BLOCK_STRING_LIT_RULE(85),
        CHAR_LIT_RULE(86),
        IF_EXPR_RULE(87),
        RAISE_EXPR_RULE(88),
        TRY_EXPR_RULE(89),
        ARRAY_LITERAL_RULE(90),
        CALLABLE_REF_RULE(91),
        NULL_COND_QUAL_RULE(92),
        BASIC_EXPR_RULE(93),
        TERM_RULE(94),
        BASIC_NAME_RULE(95),
        LAMBDA_EXPR_RULE(96),
        NULL_LIT_RULE(97),
        ARGUMENT_RULE(98),
        ARGS_RULE(99),
        DECL_ANNOTATION_ARGS_RULE(100),
        DECL_ANNOTATION_RULE(101),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final GrammarRule NONE =
            MAIN_RULE_RULE;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, GrammarRule> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(GrammarRule elem : GrammarRule.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private GrammarRule(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to an enum value.
         */
        static GrammarRule fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static GrammarRule fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get GrammarRule from " + cValue
                );
            return (GrammarRule) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**

     */
    public enum LookupKind {

        // ----- Enum values -----

        RECURSIVE(0),
        FLAT(1),
        MINIMAL(2),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final LookupKind NONE =
            RECURSIVE;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, LookupKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(LookupKind elem : LookupKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private LookupKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to an enum value.
         */
        static LookupKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static LookupKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get LookupKind from " + cValue
                );
            return (LookupKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }


    // ==========
    // Java wrapping classes
    // ==========

    // ===== Constant structure wrapping classes =====

    /**
     * This class provides static methods to help wrapping and unwrapping
     * native boolean values.
     */
    public static final class BooleanWrapper {

        // ----- Graal C API methods -----

        static boolean wrap(
            final CCharPointer pointer
        ) {
            return pointer.read() != 0;
        }

        static boolean wrap(
            final byte nativeValue
        ) {
            return nativeValue != 0;
        }

    }

    /**
     * This class provides static methods to help wrapping and unwrapping
     * native integer values.
     */
    public static final class IntegerWrapper {

        // ----- Graal C API methods -----

        static int wrap(
            final CIntPointer pointer
        ) {
            return pointer.read();
        }

        static int wrap(
            final int nativeValue
        ) {
            return nativeValue;
        }

    }

    /**
     * This class wraps the langkit characters which are 32 bit wide.
     */
    public static final class Char implements LangkitSupport.CharInterface {

        // ----- Class attributes -----

        /** Singleton that represents the none char. */
        public static final Char NONE = new Char(0);

        // ----- Instance attributes -----

        /** The value of the character. */
        public final int value;

        // ----- Constructors -----

        /**
         * Create a new character from its value. In langkit characters are
         * 32 bit wide so represented by Java integer.
         *
         * @param value The value of the character.
         */
        Char(
            final int value
        ) {
            this.value = value;
        }

        /**
         * Create a character from its integer value.
         *
         * @param value The character value.
         * @return The newly created character.
         */
        public static Char create(
            final int value
        ) {
            return new Char(value);
        }

        /**
         * Create a character from a Java character.
         *
         * @param value The source value of the character.
         * @return The newly created character.
         */
        public static Char create(
            final char value
        ) {
            return new Char((int) value);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given NI pointer in a Java class.
         *
         * @param pointer The NI pointer to wrap.
         * @return The wrapped character.
         */
        static Char wrap(
            final CIntPointer pointer
        ) {
            return wrap(pointer.read());
        }

        /**
         * Wrap an integer to a character.
         *
         * @param value The value of the character in an integer.
         * @return The newly created character.
         */
        static Char wrap(
            final int value
        ) {
            return new Char(value);
        }

        /**
         * Unwrap the character in the given int pointer.
         *
         * @param pointer The pointer to unwrap the character in.
         */
        void unwrap(
            final CIntPointer pointer
        ) {
            pointer.write(this.value);
        }

        /**
         * Unwrap the character in a Java integer.
         *
         * @return The character value in a Java integer.
         */
        int unwrap() {
            return this.value;
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            final ByteBuffer buffer = ByteBuffer.allocate(4);
            buffer.order(BYTE_ORDER);
            buffer.putInt(this.value);
            return decodeUTF32(buffer.array());
        }

        @Override
        public boolean equals(Object o) {
            if(o == this) return true;
            if(!(o instanceof Char)) return false;
            Char other = (Char) o;
            return this.value == other.value;
        }

    }

    /**
     * Arbitrarily large integer.
     */
    static final class BigIntegerWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none big integer. */
        public static final BigInteger NONE = BigInteger.ZERO;

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer which points to a native big integer.
         *
         * @param pointer The pointer to the native big integer.
         * @return The Java big integer.
         */
        static BigInteger wrap(
            final WordPointer pointer
        ) {
            return wrap((BigIntegerNative) pointer.read());
        }

        /**
         * Wrap the given native big integer in a Java big integer.
         *
         * @param bigIntegerNative The big integer native value.
         * @return The Java big integer.
         */
        static BigInteger wrap(
            final BigIntegerNative bigIntegerNative
        ) {
            final String representation = getRepresentation(bigIntegerNative);
            return new BigInteger(representation);
        }

        /**
         * Unwrap the given Java big integer in the given word pointer.
         *
         * @param bigInteger The big integer to unwrap.
         * @param pointer The word pointer to place the big integer in.
         */
        static void unwrap(
            final BigInteger bigInteger,
            final WordPointer pointer
        ) {
            pointer.write(unwrap(bigInteger));
        }

        /**
         * Unwrap the given big integer.
         *
         * @param bigInteger The big integer to unwrap.
         * @return The native big integer newly allocated.
         */
        static BigIntegerNative unwrap(
            final BigInteger bigInteger
        ) {
            // Create the representation of the big integer
            final String representation = bigInteger.toString();
            try(final Text bigIntegerText = Text.create(representation)) {
                TextNative bigIntegerTextNative = StackValue.get(
                    TextNative.class
                );
                bigIntegerText.unwrap(bigIntegerTextNative);

                // Create the big intger by calling the native function
                return NI_LIB.lkt_create_big_integer(
                    bigIntegerTextNative
                );
            }
        }

        /**
         * Release the big integer pointed by the given pointer.
         *
         * @param pointer The pointer to the big integer to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((BigIntegerNative) pointer.read());
        }

        /**
         * Release the given native big integer.
         *
         * @param bigIntegerNative The native big integer to release.
         */
        static void release(
            final BigIntegerNative bigIntegerNative
        ) {
            NI_LIB.lkt_big_integer_decref(bigIntegerNative);
        }

        // ----- Class methods -----

        /**
         * Get the string representation of the given native big integer.
         *
         * @param bigIntegerNative The native big integer to get the
         * representation from.
         */
        private static String getRepresentation(
            final BigIntegerNative bigIntegerNative
        ) {
            // Allocate the stack value for the text
            final TextNative bigIntegerTextNative = StackValue.get(
                TextNative.class
            );
            Text.NONE.unwrap(bigIntegerTextNative);

            // Call the native function
            NI_LIB.lkt_big_integer_text(
                bigIntegerNative,
                bigIntegerTextNative
            );

            // Wrap the text and return the result
            try(final Text bigIntegerText = Text.wrap(bigIntegerTextNative)) {
                return bigIntegerText.getContent();
            }
        }

    }

    /**
     * Reference to a symbol. Symbols are owned by analysis contexts, so they
     * must not outlive them. This type exists only in the C API, and roughly
     * wraps the corresponding Ada type (an array fat pointer).
     */
    public static final class Symbol implements LangkitSupport.SymbolInterface {

        // ----- Class attributes -----

        /** Singleton that represents the none symbol. */
        public static final Symbol NONE = new Symbol("");

        // ----- Instance attributes

        /** The text of the symbol. */
        public final String text;

        // ----- Constructors -----

        /**
         * Create a new symbol from its text.
         *
         * @param text The symbol text.
         */
        Symbol(
            final String text
        ) {
            this.text = text;
        }

        /**
         * Public access to the symbol creation.
         *
         * @param text The text of the symbol.
         */
        public static Symbol create(
            final String text
        ) {
            return new Symbol(text);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to a native symbol.
         *
         * @param pointer The pointer to the native symbol.
         * @return The wrapped symbol.
         */
        static Symbol wrap(
            final WordPointer pointer
        ) {
            return wrap((SymbolNative) pointer.read());
        }

        /**
         * Wrap the given symbol native value in a Java value.
         *
         * @param symbolNative The symbol native value.
         * @return The wrapped symbol.
         */
        static Symbol wrap(
            final SymbolNative symbolNative
        ) {
            // Get the symbol text
            final TextNative textNative = StackValue.get(TextNative.class);
            Text.NONE.unwrap(textNative);
            NI_LIB.lkt_symbol_text(
                symbolNative,
                textNative
            );

            // Return the new symbol
            try(final Text text = Text.wrap(textNative)) {
                return new Symbol(text.getContent());
            }
        }

        /**
         * Unwrap the symbol in the given native structure.
         *
         * @param symbolNative The native structure to unwrap in.
         */
        void unwrap(
            final SymbolNative symbolNative,
            final AnalysisContext context
        ) {
            // Unwrap the symbol text
            try(Text text = Text.create(this.text)) {
                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                text.unwrap(textNative);

                // Call the symbol creation
                final int resCode = NI_LIB.lkt_context_symbol(
                    context.reference.ni(),
                    textNative,
                    symbolNative
                );

                // Check the symbol creation success
                if(resCode == 0) {
                    throw new SymbolException(this.text);
                }
            }
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.text;
        }

    }

    /**
     * Type to contain Unicode text data.
     */
    static final class StringWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none string. */
        public static final String NONE = "";

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer that points to a native string.
         *
         * @param pointer The pointer to the native string.
         * @return The Java string wrapped from the native string.
         */
        static String wrap(
            final WordPointer pointer
        ) {
            return wrap((StringNative) pointer.read());
        }

        /**
         * Wrap a native string wrapper in a Java string.
         *
         * @param stringNative The native string to wrap.
         * @return The Java string created from the native one.
         */
        static String wrap(
            final StringNative stringNative
        ) {
            return getNativeStringContent(stringNative);
        }

        /**
         * Unwrap the string in the given word pointer.
         *
         * @param string The string to unwrap.
         * @param pointer The word pointer to place the native string in.
         */
        static void unwrap(
            final String string,
            final WordPointer pointer
        ) {
            pointer.write(unwrap(string));
        }

        /**
         * Unwrap the given string in a native one.
         *
         * @param string The string to unwrap.
         * @return The native string unwrapped.
         */
        static StringNative unwrap(
            final String string
        ) {
            return nativeStringFromContent(string);
        }

        /**
         * Release the native string at the given pointer.
         *
         * @param pointer The pointer to the native string.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((StringNative) pointer.read());
        }

        /**
         * Release the given native string.
         *
         * @param stringNative The native string to release.
         */
        static void release(
            final StringNative stringNative
        ) {
            NI_LIB.lkt_string_dec_ref(stringNative);
        }

        // ----- Class methods -----

        /**
         * Get the content of the given native string in a Java one.
         *
         * @param stringNative The native string to get the content from.
         * @return The Java string.
         */
        private static String getNativeStringContent(
            final StringNative stringNative
        ) {
            // Prepare the working variable
            final Pointer pointer = (Pointer) stringNative;
            final int length = pointer.readInt(0);
            final byte[] contentArray = new byte[length * 4];

            // Get the content from the native string
            for(int i = 0 ; i < contentArray.length ; i++) {
                contentArray[i] = pointer.readByte(i + 8);
            }

            // Return the decoded string
            return decodeUTF32(contentArray);
        }

        /**
         * Create a native string from a Java string.
         *
         * @param string The Java string to create the native string from.
         * @return The native string.
         */
        private static StringNative nativeStringFromContent(
            final String string
        ) {
            // Encode the string in UTF-32
            final byte[] contentArray = encodeUTF32(string);
            final int length = string.length();

            // Create the native array
            final Pointer contentArrayNative = UnmanagedMemory.malloc(
                contentArray.length
            );
            for(int i = 0 ; i < contentArray.length ; i++) {
                contentArrayNative.writeByte(i, contentArray[i]);
            }

            // Create the native string
            final StringNative res = NI_LIB.lkt_create_string(
                (CIntPointer) contentArrayNative,
                length
            );

            // Free the memory
            UnmanagedMemory.free(contentArrayNative);

            // Return the result
            return res;
        }

    }

    /**
     * String encoded in UTF-32 (native endianness).
     */
    public static final class Text extends LangkitSupport.Text {

        // ----- Class attributes -----

        /** Singleton that represents the none text. */
        public static final Text NONE = new Text(
            PointerWrapper.nullPointer(),
            0,
            false
        );

        // ----- Instance attributes -----

        /** The pointer to the characters. */
        private final PointerWrapper charPointer;

        /** The size of the text. */
        private final long length;

        /** If the text is allocated. */
        private final boolean isAllocated;

        /** If the text object is the owner of its buffer. */
        private final boolean isOwner;

        /** The content of the text in a Java string. */
        private String content;

        // ----- Constructors -----

        /**
         * Create a new text from its content.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         */
        Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated
        ) {
            this(
                charPointer,
                length,
                isAllocated,
                false,
                null
            );
        }

        /**
         * Create a new text from its content and buffer.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         * @param contentArray The characters of the text (for JNI)
         */
        Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated,
            final byte[] contentArray
        ) {
            this(
                charPointer,
                length,
                isAllocated,
                false,
                contentArray
            );
        }

        /**
         * Create a new text from its content and buffer.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         * @param isOwner If the Java object owns the text buffer.
         * @param contentArray The characters of the text
         * (as strings, this is only used in JNI mode).
         */
        private Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated,
            final boolean isOwner,
            final byte[] contentArray
        ) {
            this.charPointer = charPointer;
            this.length = length;
            this.isAllocated = isAllocated;
            this.isOwner = isOwner;

            if(contentArray != null) {
                this.content = decodeUTF32(contentArray);
            } else {
                this.content = null;
            }
        }

        /**
         * Create a new langkit text from its string content.
         *
         * @param content The content of the text in a Java string.
         * @return The newly created text.
         */
        public static Text create(
            final String content
        ) {
            final byte[] contentArray = encodeUTF32(content);

            if(ImageInfo.inImageCode()) {
                final PointerWrapper charPointer = new PointerWrapper(
                    toCBytes(contentArray)
                );
                return new Text(
                    charPointer,
                    (long) content.length(),
                    false,
                    true,
                    contentArray
                );
            } else {
                return JNI_LIB.lkt_create_text(contentArray);
            }
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the native text pointed by the given pointer.
         *
         * @param pointer The pointer to the native text to wrap.
         * @return The wrapped text.
         */
        static Text wrap(
            final WordPointer pointer
        ) {
            return wrap((TextNative) pointer.read());
        }

        /**
         * Wrap a text native value in the Java class.
         *
         * @param textNative The text native NI value.
         * @return The newly wrapped text.
         */
        static Text wrap(
            final TextNative textNative
        ) {
            return new Text(
                new PointerWrapper(textNative.get_chars()),
                textNative.get_length(),
                textNative.get_is_allocated() != 0
            );
        }

        /**
         * Unwrap the text in the given NI pointer.
         *
         * @param textNative The NI pointer to the native text structure.
         */
        void unwrap(
            final TextNative textNative
        ) {
            textNative.set_chars(this.charPointer.ni());
            textNative.set_length(this.length);
            textNative.set_is_allocated(this.isAllocated ? 1 : 0);
        }

        // ----- Instance methods -----

        /** The pointer to the characters. */
        public final PointerWrapper getCharPointer() {
            return charPointer;
        }

        /** The size of the text. */
        public final long getLength() {
            return length;
        }

        /** If the text is allocated. */
        public final boolean getIsAllocated() {
            return isAllocated;
        }

        /** If the text object is the owner of its buffer. */
        public final boolean getIsOwner() {
            return isOwner;
        }

        /**
         * Get the content of the text in a Java string.
         *
         * @return the content of the text.
         */
        public String getContent() {
            // Content is null only when using the Graal C API.
            if(this.content == null) {
                final byte[] contentArray = new byte[(int) this.length * 4];
                for(int i = 0 ; i < contentArray.length ; i++) {
                    contentArray[i] = (
                        (CCharPointer) this.charPointer.ni()
                    ).read(i);
                }
                this.content = decodeUTF32(contentArray);
            }

            return this.content;
        }

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                if(this.isOwner) {
                    UnmanagedMemory.free(this.charPointer.ni());
                } else {
                    final TextNative textNative = StackValue.get(
                        TextNative.class
                    );
                    this.unwrap(textNative);
                    NI_LIB.lkt_destroy_text(textNative);
                }
            } else {
                JNI_LIB.lkt_destroy_text(this);
            }
            checkException();

        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.getContent();
        }

    }

    /**
     * Location in a source file. Line and column numbers are one-based.
     */
    public static final
    class SourceLocation extends LangkitSupport.SourceLocation {

        // ----- Class attributes -----

        /** Singleton that represents the none source location. */
        public static final SourceLocation NONE = new SourceLocation(
            0,
            (short) 0
        );

        // ----- Constructors -----

        /**
         * Create a new source location from a line and a column.
         *
         * @param line The line of the source location.
         * @param column The column of the source location.
         */
        SourceLocation(
            final int line,
            final short column
        ) {
            super(line, column);
        }

        /**
         * Create a source location from its line and column.
         *
         * @param line The line.
         * @param column The column.
         * @return The newly create source location.
         */
        public static SourceLocation create(
            final int line,
            final short column
        ) {
            return new SourceLocation(
                line,
                column
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given source location in the Java class.
         *
         * @param pointer Pointer to the native source location.
         * @return The newly wrapper source location.
         */
        static SourceLocation wrap(
            final WordPointer pointer
        ) {
            return wrap((SourceLocationNative) pointer.read());
        }

        /**
         * Wrap a source location native value in the Java class.
         *
         * @param sourceLocationNative The source location NI native value.
         * @return The newly wrapped source location.
         */
        static SourceLocation wrap(
            final SourceLocationNative sourceLocationNative
        ) {
            return new SourceLocation(
                sourceLocationNative.get_line(),
                sourceLocationNative.get_column()
            );
        }

        /**
         * Uwrap the source location in the given NI pointer.
         *
         * @param sourceLocationNative The NI pointer to the native structure
         *  to fill.
         */
        public void unwrap(
            final SourceLocationNative sourceLocationNative
        ) {
            sourceLocationNative.set_line(this.line);
            sourceLocationNative.set_column(this.column);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.line + ":" + this.column;
        }

    }

    /**
     * Location of a span of text in a source file.
     */
    public static final
    class SourceLocationRange extends LangkitSupport.SourceLocationRange {

        // ----- Class attributes -----

        /** Singleton that represents the none source location range. */
        public static final SourceLocationRange NONE =
            new SourceLocationRange(
                SourceLocation.NONE,
                SourceLocation.NONE
            );

        // ----- Constructors -----

        /**
         * Create a source location range from its bounds.
         *
         * @param start The start of the range.
         * @param end The end of the range.
         */
        SourceLocationRange(
            final SourceLocation start,
            final SourceLocation end
        ) {
            super(start, end);
        }

        /**
         * Create a new source location range from its bounds.
         *
         * @param start The starting bound.
         * @param end The ending bound.
         * @return The newly created source location range.
         */
        public static SourceLocationRange create(
            final SourceLocation start,
            final SourceLocation end
        ) {
            return new SourceLocationRange(
                start,
                end
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given source location range in the Java class.*
         *
         * @param pointer The pointer to the native source location range.
         * @return The newly wrapped source location range.
         */
        static SourceLocationRange wrap(
            final WordPointer pointer
        ) {
            return wrap((SourceLocationRangeNative) pointer.read());
        }

        /**
         * Wrap a source location range native value in the Java class.
         *
         * @param sourceLocationRangeNative The source location range NI
         * native value.
         * @return The newly wrapped source location range.
         */
        static SourceLocationRange wrap(
            final SourceLocationRangeNative sourceLocationRangeNative
        ) {
            return new SourceLocationRange(
                new SourceLocation(
                    sourceLocationRangeNative.get_start_line(),
                    sourceLocationRangeNative.get_start_column()
                ),
                new SourceLocation(
                    sourceLocationRangeNative.get_end_line(),
                    sourceLocationRangeNative.get_end_column()
                )
            );
        }

        /**
         * Uwrap the source location range in the given NI pointer.
         *
         * @param sourceLocationRangeNative The NI pointer to the native
         * structure to fill.
         */
        void unwrap(
            final SourceLocationRangeNative sourceLocationRangeNative
        ) {
            sourceLocationRangeNative.set_start_line(this.start.line);
            sourceLocationRangeNative.set_start_column(this.start.column);
            sourceLocationRangeNative.set_end_line(this.end.line);
            sourceLocationRangeNative.set_end_column(this.end.column);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.start.toString() + "-" + this.end.toString();
        }

    }

    /**
     * Diagnostic for an analysis unit: cannot open the source file, parsing
     * error, ...
     */
    public static final class Diagnostic extends LangkitSupport.Diagnostic {

        // ----- Class attributes -----

        /** Singleton that represents the none diagnostic. */
        public static final Diagnostic NONE = new Diagnostic(
            SourceLocationRange.NONE,
            Text.NONE
        );

        // ----- Instance attributes -----

        /** The source location range of the diagnostic. */
        public final SourceLocationRange sourceLocationRange;

        /** The message of the diagnostic. */
        public final Text message;

        // ----- Getters -----

        public final SourceLocationRange getSourceLocationRange() {
            return sourceLocationRange;
        }

        public final Text getMessage() {
            return message;
        }

        // ----- Constructors -----

        /**
         * Create a diagnostic from its content.
         *
         * @param sourceLocationRange The range of the diagnostic.
         * @param message The message of the diagnostic.
         */
        Diagnostic(
            final SourceLocationRange sourceLocationRange,
            final Text message
        ) {
            this.sourceLocationRange = sourceLocationRange;
            this.message = message;
        }

        /**
         * Create a new diagnostic from its content.
         *
         * @param sourceLocationRange The source location range concerned by
         * this diagnostic.
         * @param message The message of the diagnostic.
         * @return The newly created diagnostic
         */
        public static Diagnostic create(
            final SourceLocationRange sourceLocationRange,
            final Text message
        ) {
            return new Diagnostic(
                sourceLocationRange,
                message
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to a native diagnostic.
         *
         * @param pointer The pointer to the native diagnostic.
         * @return The wrapped diagnostic.
         */
        static Diagnostic wrap(
            final WordPointer pointer
        ) {
            return wrap((DiagnosticNative) pointer.read());
        }

        /**
         * Wrap a diagnostic native value in the Java class.
         *
         * @param diagnosticNative The diagnostic NI native value.
         * @return The newly wrapped diagnostic.
         */
        static Diagnostic wrap(
            final DiagnosticNative diagnosticNative
        ) {
            return new Diagnostic(
                new SourceLocationRange(
                    new SourceLocation(
                        diagnosticNative.get_start_line(),
                        diagnosticNative.get_start_column()
                    ),
                    new SourceLocation(
                        diagnosticNative.get_end_line(),
                        diagnosticNative.get_end_column()
                    )
                ),
                new Text(
                    new PointerWrapper(diagnosticNative.get_message_chars()),
                    diagnosticNative.get_message_length(),
                    diagnosticNative.get_message_is_allocated() != 0
                )
            );
        }

        /**
         * Unwrap the diagnostic in the given NI pointer.
         *
         * @param diagnosticNative The pointer to the native structure to
         * fill.
         */
        public void unwrap(
            final DiagnosticNative diagnosticNative
        ) {
            diagnosticNative.set_start_line(
                this.sourceLocationRange.start.line
            );
            diagnosticNative.set_start_column(
                this.sourceLocationRange.start.column
            );
            diagnosticNative.set_end_line(
                this.sourceLocationRange.end.line
            );
            diagnosticNative.set_end_column(
                this.sourceLocationRange.end.column
            );
            diagnosticNative.set_message_chars(
                this.message.charPointer.ni()
            );
            diagnosticNative.set_message_length(
                this.message.length
            );
            diagnosticNative.set_message_is_allocated(
                this.message.isAllocated ? 1 : 0
            );
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.message.toString()
                + " at <"
                + this.sourceLocationRange.toString()
                + ">";
        }

    }

    /**
     * Result of applying a rewriting session.
     *
     * On success, ``Success`` is true.
     *
     * On failure, ``Success`` is false, ``Unit`` is set to the unit on which
     * rewriting failed, and ``Diagnostics`` is set to related rewriting
     * errors.
     */
    public static final class RewritingApplyResult extends
        LangkitSupport.RewritingApplyResult {

        // ----- Instance attributes -----

        /**
         * If the rewriting failed, this is the unit on which the rewriting
         * failed. Otherwise it is the None unit.
         */
        public final AnalysisUnit unit;

        /**
         * Number of diagnostics in the result. 0 if the apply has been a
         * success.
         */
        private final int diagnosticsCount;

        /** Pointer to the diagnostic array, null if successful. */
        private final PointerWrapper diagnosticsReference;

        /** Cache for the unwrapped diagnostics. */
        private Diagnostic[] diagnostics;

        // ----- Constructors ------

        /** Create a new rewriting result with its content. */
        RewritingApplyResult(
            final boolean success,
            final AnalysisUnit unit,
            final int diagnosticsCount,
            final PointerWrapper diagnosticsReference
        ) {
            super(success);
            this.unit = unit;
            this.diagnosticsCount = diagnosticsCount;
            this.diagnosticsReference = diagnosticsReference;
            this.diagnostics = null;
        }

        /** Shortcut creation method for success. */
        static RewritingApplyResult success() {
            return new RewritingApplyResult(
                true,
                AnalysisUnit.NONE,
                0,
                PointerWrapper.nullPointer()
            );
        }

        // ----- Graal C API methods -----

        /** Wrap a pointer to a native rewriting apply result. */
        static RewritingApplyResult wrap(
            final Pointer pointer
        ) {
            return wrap((RewritingApplyResultNative) pointer.readWord(0));
        }

        /** Wrap a native rewriting apply result to the Java class. */
        static RewritingApplyResult wrap(
            final RewritingApplyResultNative rewritingApplyResultNative
        ) {
            if(rewritingApplyResultNative.get_success() > 0) {
                return RewritingApplyResult.success();
            }
            return new RewritingApplyResult(
                false,
                AnalysisUnit.wrap(rewritingApplyResultNative.get_unit()),
                rewritingApplyResultNative.get_diagnostics_count(),
                new PointerWrapper(
                    rewritingApplyResultNative.get_diagnostics()
                )
            );
        }

        /** Unwrap the rewriting apply result to the provided native value. */
        void unwrap(
            final RewritingApplyResultNative rewritingApplyResultNative
        ) {
            rewritingApplyResultNative.set_success(this.success ? 1 : 0);
            rewritingApplyResultNative.set_unit(this.unit.unwrap());
            rewritingApplyResultNative.set_diagnostics_count(
                this.diagnosticsCount
            );
            rewritingApplyResultNative.set_diagnostics(
                (DiagnosticNative) this.diagnosticsReference.ni()
            );
        }

        // ----- Instance methods -----

        /** Unwrap all diagnostics, cache the result and return it */
        public Diagnostic[] getDiagnostics() {
            if(this.diagnostics == null) {
                if(this.success) {
                    this.diagnostics = new Diagnostic[0];
                } else {

                    if(ImageInfo.inImageCode()) {
                        this.diagnostics =
                            new Diagnostic[this.diagnosticsCount];
                        final Pointer diagnostics =
                            (Pointer) this.diagnosticsReference.ni();
                        DiagnosticNative diagnosticNative;
                        final int diagnosticNativeSize = SizeOf.get(
                            DiagnosticNative.class
                        );
                        for(int i = 0; i < this.diagnosticsCount; i++) {
                            diagnosticNative = WordFactory.unsigned(
                                diagnostics.add(i * diagnosticNativeSize)
                                        .rawValue()
                            );
                            this.diagnostics[i] = Diagnostic.wrap(
                                diagnosticNative
                            );
                        }
                    } else {
                        this.diagnostics =
                            JNI_LIB
                                .lkt_rewriting_get_result_diagnostics(
                                    this.diagnosticsCount,
                                    this.diagnosticsReference.jni()
                                );
                    }

                }
            }
            return this.diagnostics;
        }

        /**
         * Free the result of the ``Apply`` operation.
         */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                final RewritingApplyResultNative resultNative =
                    StackValue.get(RewritingApplyResultNative.class);
                unwrap(resultNative);
                NI_LIB.lkt_rewriting_free_apply_result(resultNative);
            } else {
                JNI_LIB.lkt_rewriting_free_apply_result(this);
            }

        }

    }

    /**
     * Interface to override how source files are fetched and decoded.
     */
    public static final class FileReader implements
        LangkitSupport.FileReaderInterface {

        // ----- Class attributes -----

        /** Singleton that represents the none file reader. */
        public static final FileReader NONE = new FileReader(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the file reader */
        private final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new file reader from its native reference.
         *
         * @param reference The reference to the native file reader.
         */
        FileReader(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to a native file reader.
         *
         * @param pointer The pointer to the native file reader.
         * @return The newly wrapped file reader.
         */
        static FileReader wrap(
            final WordPointer pointer
        ) {
            return wrap((FileReaderNative) pointer.read());
        }

        /**
         * Wrap the given file reader in the Java class.
         *
         * @param fileReaderNative The native file reader to wrap.
         * @return The wrapped file reader.
         */
        static FileReader wrap(
            final FileReaderNative fileReaderNative
        ) {
            return new FileReader(
                new PointerWrapper(fileReaderNative)
            );
        }

        /**
         * Unwrap the file reader in the given pointer.
         *
         * @param pointer The pointer to unwrap the file reader in.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(0, this.unwrap());
        }

        /**
         * Get the unwrapped file reader.
         *
         * @return The unwrapped native file reader.
         */
        FileReaderNative unwrap() {
            return (FileReaderNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_dec_ref_file_reader(
                    this.reference.ni()
                );
            } else {
                JNI_LIB.lkt_dec_ref_file_reader(this);
            }
            checkException();

        }

    }

    /**
     * Interface to fetch analysis units from a name and a unit kind.
     *
     * The unit provider mechanism provides an abstraction which assumes that
     * to any couple (unit name, unit kind) we can associate at most one source
     * file. This means that several couples can be associated to the same
     * source file, but on the other hand, only one one source file can be
     * associated to a couple.
     *
     * This is used to make the semantic analysis able to switch from one
     * analysis units to another.
     *
     * See the documentation of each unit provider for the exact semantics of
     * the unit name/kind information.
     */
    public static final class UnitProvider implements
        LangkitSupport.UnitProviderInterface {

        // ----- Class attributes -----

        /** Singleton that represents the none unit provider. */
        public static final UnitProvider NONE = new UnitProvider(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the unit provider */
        private final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new unit provider with the reference to the native one.
         *
         * @param reference The reference to the native unit provider.
         */
        UnitProvider(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to a native unit provider.
         *
         * @param pointer The pointer to the native unit provider.
         * @return The wrapped unit provider.
         */
        static UnitProvider wrap(
            final WordPointer pointer
        ) {
            return wrap((UnitProviderNative) pointer.read());
        }

        /**
         * Wrap the given native unit provider.
         *
         * @param unitProviderNative The native unit provider.
         * @return The wrapped unit provider.
         */
        static UnitProvider wrap(
            final UnitProviderNative unitProviderNative
        ) {
            return new UnitProvider(
                new PointerWrapper(unitProviderNative)
            );
        }

        /**
         * Unwrap the unit provider in the given native pointer.
         *
         * @param pointer The pointer to place the native unit provider in.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(this.unwrap());
        }

        /**
         * Get the native unit provider.
         *
         * @return The native unit provider.
         */
        UnitProviderNative unwrap() {
            return (UnitProviderNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_dec_ref_unit_provider(this.reference.ni());
            } else {
                JNI_LIB.lkt_dec_ref_unit_provider(this);
            }
            checkException();

        }

    }

    /**
     * Interface to handle events sent by the analysis context.
     */
    public static final class EventHandler implements
        LangkitSupport.EventHandlerInterface {

        // ----- Class attributes -----

        /** Singleton that represents the none event handler. */
        public static final EventHandler NONE = new EventHandler(
            PointerWrapper.nullPointer(),
            null,
            null
        );

        /** This map contains all created event handlers. */
        private static final Map<PointerWrapper, EventHandler>
            eventHandlerCache = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native event handler. */
        private final PointerWrapper reference;

        /**
         * The Java callback when an analysis unit is requested in the
         * associated context.
         */
        private final UnitRequestedCallback unitRequestedCallback;

        /**
         * The Java callback when an analysis unit is parsed in the
         * associated context.
         */
        private final UnitParsedCallback unitParsedCallback;

        // ----- Constructors -----

        /**
         * Create a new event handler from its native reference.
         *
         * @param reference The reference to the native event handler.
         * @param unitRequestedCallback The callback for unit requests.
         * @param unitParsedCallback The callback for unit parsing.
         */
        EventHandler(
            final PointerWrapper reference,
            final UnitRequestedCallback unitRequestedCallback,
            final UnitParsedCallback unitParsedCallback
        ) {
            this.reference = reference;
            this.unitRequestedCallback = unitRequestedCallback;
            this.unitParsedCallback = unitParsedCallback;
        }

        /**
         * Create a new even handler with its callbacks. Callbacks can be null.
         *
         * @param unitRequestedCallback The callback for analysis unit requests.
         * @param unitParsedCallback The callback for analysis unit parsing.
         */
        public static EventHandler create(
            final UnitRequestedCallback unitRequestedCallback,
            final UnitParsedCallback unitParsedCallback
        ) {
            // Prepare the reference to the native event handler
            final PointerWrapper reference;

            if(ImageInfo.inImageCode()) {
                // Get the current thread
                final IsolateThread thread = CurrentIsolate.getCurrentThread();

                // Create the native event handler
                final EventHandlerNative resNative =
                    NI_LIB.lkt_create_event_handler(
                        (VoidPointer) thread,
                        WordFactory.nullPointer(),
                        NI_LIB.unitRequestedFunction.getFunctionPointer(),
                        NI_LIB.unitParsedFunction.getFunctionPointer()
                    );

                // Set the result to the created event handler
                reference = new PointerWrapper(resNative);
            } else {
                reference = JNI_LIB.lkt_create_event_handler(
                    unitRequestedCallback,
                    unitParsedCallback
                );
            }

            // Return the new event handler wrapped object
            return new EventHandler(
                reference,
                unitRequestedCallback,
                unitParsedCallback
            );
        }

        /**
         * Get event handler Java object from its native pointer.
         *
         * @param reference The pointer to the native event handler.
         * @return The associated Java event handler.
         */
        static EventHandler fromReference(
            final PointerWrapper reference
        ) {
            if(eventHandlerCache.containsKey(reference)) {
                return eventHandlerCache.get(reference);
            } else {
                throw new ReferenceException(
                    "Cannot get event handler from this reference: " +
                    reference.toString()
                );
            }
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to an event handler.
         *
         * @param pointer The pointer to the native event handler.
         * @return The wrapped event handler.
         */
        EventHandler wrap(
            final WordPointer pointer
        ) {
            return wrap((EventHandlerNative) pointer.read());
        }

        /**
         * Wrap the given native event handler.
         *
         * @param eventHandlerNative The native value of the event handler.
         * @return The wrapped event handler.
         */
        EventHandler wrap(
            final EventHandlerNative eventHandlerNative
        ) {
            return fromReference(new PointerWrapper(eventHandlerNative));
        }

        /**
         * Unwrap the event handler in the given native pointer.
         *
         * @param pointer The pointer to place the native event handler.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(this.unwrap());
        }

        /**
         * Unwrap the event handler to a native value.
         *
         * @return The native value of the event handler.
         */
        EventHandlerNative unwrap() {
            return (EventHandlerNative) this.reference.ni();
        }

        // ----- Getters -----

        public UnitRequestedCallback getUnitRequestCallback() {
            return this.unitRequestedCallback;
        }

        public UnitParsedCallback getUnitParsedCallback() {
            return this.unitParsedCallback;
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_dec_ref_event_handler(this.reference.ni());
            } else {
                JNI_LIB.lkt_dec_ref_event_handler(this);
            }
            checkException();

        }

        // ----- Inner classes -----

        /**
         * Callback that will be called when a unit is requested from the
         * context ``Context``.
         *
         * ``Name`` is the name of the requested unit.
         *
         * ``From`` is the unit from which the unit was requested.
         *
         * ``Found`` indicates whether the requested unit was found or not.
         *
         * ``Is_Not_Found_Error`` indicates whether the fact that the unit was
         * not found is an error or not.
         *
         * .. warning:: The interface of this callback is probably subject to
         *    change, so should be treated as experimental.
         */
        @FunctionalInterface
        public interface UnitRequestedCallback {
            void invoke(
                AnalysisContext context,
                String name,
                AnalysisUnit from,
                boolean found,
                boolean isNotFoundError
            );
        }

        /**
         * Callback that will be called when any unit is parsed from the
         * context ``Context``.
         *
         * ``Unit`` is the resulting unit.
         *
         * ``Reparsed`` indicates whether the unit was reparsed, or whether it
         * was the first parse.
         */
        @FunctionalInterface
        public interface UnitParsedCallback {
            void invoke(
                AnalysisContext context,
                AnalysisUnit unit,
                boolean reparsed
            );
        }

    }

    /**
     * Reference to a token in an analysis unit.
     */
    public static class Token implements LangkitSupport.TokenInterface {

        // ----- Instance attributes -----

        /**
         * We only store the reference to the context to avoid ref-count
         * problems. To access the token context go throught the
         * analysis unit.
         */
        protected final PointerWrapper contextRef;

        /** The unit of the token. */
        public final AnalysisUnit unit;

        /** The pointer to the token data handler. */
        protected final PointerWrapper tokenDataHandler;

        /** The index of the token. */
        public final int tokenIndex;

        /** The trivia index. */
        public final int triviaIndex;

        /** The kind of the token. */
        public final TokenKind kind;

        /** The text of the token. */
        protected final String text;

        /** The source location range of the token. */
        public final SourceLocationRange sourceLocationRange;

        // ----- Constructors -----

        /**
         * Create a new token from its content.
         *
         * @param contextRef The context of the token.
         * @param unit The unit which owns the token.
         * @param tokenDataHandler The pointer to the token data.
         * @param tokenIndex The index of the token.
         * @param triviaIndex The trivia index of the token.
         * @param kind The kind of the token in an integer.
         * @param text The text of the token.
         * @param sourceLocationRange The location of the token.
         */
        Token(
            final PointerWrapper contextRef,
            final AnalysisUnit unit,
            final PointerWrapper tokenDataHandler,
            final int tokenIndex,
            final int triviaIndex,
            final TokenKind kind,
            final String text,
            final SourceLocationRange sourceLocationRange
        ) {
            this.contextRef = contextRef;
            this.unit = unit;
            this.tokenDataHandler = tokenDataHandler;
            this.tokenIndex = tokenIndex;
            this.triviaIndex = triviaIndex;
            this.kind = kind;
            this.text = text;
            this.sourceLocationRange = sourceLocationRange;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the pointer to a native token.
         *
         * @param pointer The pointer to the native token.
         * @param unit The analysis unit which owns the token.
         * @return The wrapped token or a none value if the token is a none
         * one.
         */
        static Token wrap(
            final WordPointer pointer,
            final AnalysisUnit unit
        ) {
            return wrap(
                (TokenNative) pointer.read(),
                unit
            );
        }

        /**
         * Wrap a native token value in the Java class.
         *
         * @param tokenNative The native NI token value.
         * @param unit The analysis unit that owns the token.
         * @return The wrapped token or a none value if the token data
         * handler is null.
         */
        static Token wrap(
            final TokenNative tokenNative,
            final AnalysisUnit unit
        ) {
            if(tokenNative.get_data().isNull())
                return NONE(unit);

            // Fetch the token kind, source location range and text from the
            // tokenNative reference.
            final TokenKind kind = TokenKind.fromC(
                NI_LIB.lkt_token_get_kind(tokenNative)
            );

            final SourceLocationRangeNative slocRangeNative =
                StackValue.get(SourceLocationRangeNative.class);
            NI_LIB.lkt_token_sloc_range(
                tokenNative,
                slocRangeNative
            );
            final SourceLocationRange slocRange =
                SourceLocationRange.wrap(slocRangeNative);

            final TextNative textNative = StackValue.get(TextNative.class);
            NI_LIB.lkt_token_range_text(
                tokenNative,
                tokenNative,
                textNative
            );
            final String text = Text.wrap(textNative).getContent();
            NI_LIB.lkt_destroy_text(textNative);

            // Wrap them in a high-level Token instance
            return new Token(
                new PointerWrapper(tokenNative.get_context()),
                unit,
                new PointerWrapper(tokenNative.get_data()),
                tokenNative.get_token_index(),
                tokenNative.get_trivia_index(),
                kind,
                text,
                slocRange
            );
        }

        /**
         * Unwrap the token in the given NI pointer.
         *
         * @param tokenNative The NI pointer to the native structure to
         * fill.
         */
        public void unwrap(
            final TokenNative tokenNative
        ) {
            tokenNative.set_context(this.contextRef.ni());
            tokenNative.set_data(this.tokenDataHandler.ni());
            tokenNative.set_token_index(this.tokenIndex);
            tokenNative.set_trivia_index(this.triviaIndex);
        }

        // ----- Getters -----

        public SourceLocationRange getSourceLocationRange() {
            return this.sourceLocationRange;
        }

        public AnalysisUnit getUnit() {
            return this.unit;
        }

        public String getText() {
            return this.text;
        }

        public TokenKind getKind() {
            return this.kind;
        }

        public boolean isTrivia() {
            return this.triviaIndex != 0;
        }

        public boolean isNone() {
            return false;
        }

        // ----- Class methods -----

        /**
         * Get a none token for the given unit.
         *
         * @param unit The unit to get a none token for.
         * @return The none token for the given unit.
         */
        public static Token NONE(
            final AnalysisUnit unit
        ) {
            return NoToken.getInstance(unit);
        }

        /**
         * Get the text from the start token to the end token.
         *
         * @param start The start token.
         * @param end The end token.
         * @return The text between the two tokens.
         */
        @CompilerDirectives.TruffleBoundary
        public static String textRange(
            final Token start,
            final Token end
        ) {

            if(ImageInfo.inImageCode()) {
                final TokenNative startNative = StackValue.get(
                    TokenNative.class
                );
                start.unwrap(startNative);

                final TokenNative endNative = StackValue.get(
                    TokenNative.class
                );
                end.unwrap(endNative);

                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                Text.NONE.unwrap(textNative);
                NI_LIB.lkt_token_range_text(
                    startNative,
                    endNative,
                    textNative
                );
                try(final Text resText = Text.wrap(textNative)) {
                    return resText.getContent();
                }
            } else {
                try(
                    final Text resText = JNI_LIB.lkt_token_range_text(
                        start,
                        end
                    )
                ) {
                    return resText.getContent();
                }
            }

        }

        // ----- Instance methods -----

        /**
         * Get the next token.
         *
         * @return The next token in the source.
         */
        public Token next() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(tokenNative);

                final TokenNative nextNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.lkt_token_next(
                    tokenNative,
                    nextNative
                );
                return wrap(nextNative, this.unit);
            } else {
                return JNI_LIB.lkt_token_next(this);
            }

        }

        /**
         * Get the previous token.
         *
         * @return The previous token in the source.
         */
        public Token previous() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(tokenNative);

                final TokenNative previousNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.lkt_token_previous(
                    tokenNative,
                    previousNative
                );
                return wrap(previousNative, this.unit);
            } else {
                return JNI_LIB.lkt_token_previous(this);
            }

        }

        /**
         * Check of the token is equivalent to the other one.
         *
         * @param other The other token to compare with.
         */
        public boolean isEquivalent(
            final LangkitSupport.TokenInterface other
        ) {

            final Token castOther = (Token) other;

            if(ImageInfo.inImageCode()) {
                final TokenNative leftNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(leftNative);

                final TokenNative rightNative = StackValue.get(
                    TokenNative.class
                );
                castOther.unwrap(rightNative);

                return NI_LIB.lkt_token_is_equivalent(
                    leftNative,
                    rightNative
                ) != 0;
            } else {
                return JNI_LIB.lkt_token_is_equivalent(this, castOther);
            }

        }

        // ----- Override methods -----

        @Override
        @CompilerDirectives.TruffleBoundary
        public String toString() {
            return "<Token Kind="
                + this.kind.name
                + " Text=\""
                + stringRepresentation(this.getText())
                + "\">";
        }

        @Override
        public boolean equals(
            Object o
        ) {
            if(o == this) return true;
            if(!(o instanceof Token)) return false;
            final Token other = (Token) o;
            return this.tokenDataHandler.equals(other.tokenDataHandler) &&
                    this.tokenIndex == other.tokenIndex &&
                    this.triviaIndex == other.triviaIndex;
        }

        // ----- Inner classes -----

        /**
        * This class represents the absence of token.
        */
        private static final class NoToken extends Token {

            // ----- Class attributes -----

            /** The map of the no token instances */
            private static final Map<AnalysisUnit, NoToken> instances
                = new HashMap<>();

            // ----- Constructors -----

            /**
            * Create a new no token for the given analysis unit.
            * The constructor is private to avoid too many instances.
            *
            * @param unit The analysis unit to create the no token for.
            */
            private NoToken(
                final PointerWrapper contextRef,
                final AnalysisUnit unit
            ) {
                super(
                    contextRef,
                    unit,
                    PointerWrapper.nullPointer(),
                    0,
                    0,
                    TokenKind.fromC(-1),
                    null,
                    SourceLocationRange.NONE
                );
            }

            /**
            * Get the no token instance for the given analysis unit.
            *
            * @param unit The unit to get the no token instance for.
            * @return The no token instance.
            */
            static NoToken getInstance(
                final AnalysisUnit unit
            ) {
                if(!instances.containsKey(unit)) {
                    try(AnalysisContext context = unit.getContext()) {
                        instances.put(
                            unit,
                            new NoToken(context.reference, unit)
                        );
                    }
                }
                return instances.get(unit);
            }

            // ----- Getters -----

            @Override
            public String getText() {
                return "";
            }

            // ----- Instance methods -----

            @Override
            public Token next() {
                return this;
            }

            @Override
            public Token previous() {
                return this;
            }

            @Override
            public boolean isEquivalent(
                final LangkitSupport.TokenInterface other
            ) {
                return other instanceof NoToken;
            }

            @Override
            public boolean isNone() {
                return true;
            }

            @Override
            public void unwrap(
                final TokenNative tokenNative
            ) {
                tokenNative.set_context(this.contextRef.ni());
                tokenNative.set_data(this.tokenDataHandler.ni());
                tokenNative.set_token_index(this.tokenIndex);
                tokenNative.set_trivia_index(this.triviaIndex);
            }

            // ----- Override methods -----

            @Override
            @CompilerDirectives.TruffleBoundary
            public String toString() {
                return "<Token Kind="
                    + this.kind.name
                    + " Text=\"\">";
            }

            @Override
            public boolean equals(
                Object o
            ) {
                return o == this;
            }

        }

    }

    /**
     * This type represents a context for all source analysis. This is the
     * first type you need to create to use Liblktlang. It will contain the
     * results of all analysis, and is the main holder for all the data.
     *
     * You can create several analysis contexts if you need to, which enables
     * you, for example to:
     *
     * * analyze several different projects at the same time;
     *
     * * analyze different parts of the same projects in parallel.
     *
     * In the current design, contexts always keep all of their analysis units
     * allocated. If you need to get this memory released, the only option at
     * your disposal is to destroy your analysis context instance.
     */
    public static final
    class AnalysisContext implements LangkitSupport.AnalysisContextInterface {

        // ----- Class attributes -----

        /** Singleton that represents the none analysis context. */
        public static final AnalysisContext NONE = new AnalysisContext(
            PointerWrapper.nullPointer(),
            null
        );

        /** This map contains all created analysis contexts. */
        private static final Map<PointerWrapper, AnalysisContext> contextCache
            = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native analysis context. */
        private final PointerWrapper reference;

        /** The event handler associated with the context. */
        private EventHandler eventHandler;

        /**
         * The rewriting context associated with this analysis context.
         * It can be the none value.
         */
        RewritingContext rewritingContext = RewritingContext.NONE;

        // ----- Constructors -----

        /**
         * Create a new analysis unit from its reference.
         *
         * @param reference The native analysis context.
         * @param eventHandler The associated event handler.
         */
        private AnalysisContext(
            final PointerWrapper reference,
            final EventHandler eventHandler
        ) {
            this.reference = reference;
            this.eventHandler = eventHandler;

            increaseRefCounter(this);
        }

        /**
         * Create a new analysis context from scratch with its configuration.
         *
         * @param charset The charset for the analysis context, it can be null.
         * @param fileReader The file reader for the analysis context, it
         * can be null.
         * @param unitProvider The unit provider for the analysis context,
         * it can be null.
         * @param eventHandler The event handler for the analysis context,
         * it can be null.
         * @param withTrivia If the analysis context should include trivias.
         * @param tabStop The effect of the tabulations on the column number.
         */
        private AnalysisContext(
            final String charset,
            final FileReader fileReader,
            final UnitProvider unitProvider,
            final EventHandler eventHandler,
            final boolean withTrivia,
            final int tabStop
        ) {
            // Call the function to allocate the native analysis context
            final PointerWrapper reference;
            if(ImageInfo.inImageCode()) {
                reference = new PointerWrapper(
                    NI_LIB.lkt_allocate_analysis_context()
                );
            } else {
                reference = JNI_LIB.lkt_create_analysis_context(
                    charset,
                    (fileReader == null ?
                        FileReader.NONE :
                        fileReader),
                    (unitProvider == null ?
                        UnitProvider.NONE :
                        unitProvider),
                    (eventHandler == null ?
                        EventHandler.NONE :
                        eventHandler),
                    withTrivia,
                    tabStop
                );
            }

            // Place the context in the cache for potention callbacks during
            // context initialization.
            this.reference = reference;
            this.eventHandler = eventHandler;
            contextCache.put(this.reference, this);

            // Perform the context initialization
            if(ImageInfo.inImageCode()) {
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                NI_LIB.lkt_initialize_analysis_context(
                    (AnalysisContextNative) reference.ni(),
                    charsetNative,
                    (fileReader == null ?
                        WordFactory.nullPointer() :
                        fileReader.reference.ni()),
                    (unitProvider == null ?
                        WordFactory.nullPointer() :
                        unitProvider.reference.ni()),
                    (eventHandler == null ?
                        WordFactory.nullPointer() :
                        eventHandler.reference.ni()),
                    (withTrivia ? 1 : 0),
                    tabStop
                );
                if(charset != null) UnmanagedMemory.free(charsetNative);
            }
        }

        /**
         * Create a new analysis context with the default parameters.
         *
         * @return The newly create analysis unit.
         */
        public static AnalysisContext create() {
            return new AnalysisContext(
                null,
                null,
                null,
                null,
                true,
                8
            );
        }

        /**
         * Create an analysis context with its parameters.
         *
         * @param charset The charset for the analysis context, it can be null.
         * @param fileReader The file reader for the analysis context, it
         * can be null.
         * @param unitProvider The unit provider for the analysis context,
         * it can be null.
         * @param eventHandler The event handler for the analysis context,
         * it can be null.
         * @param withTrivia If the analysis context should include trivias.
         * @param tabStop The effect of the tabulations on the column number.
         * @return The newly create analysis unit.
         */
        public static AnalysisContext create(
            final String charset,
            final FileReader fileReader,
            final UnitProvider unitProvider,
            final EventHandler eventHandler,
            final boolean withTrivia,
            final int tabStop
        ) {
            return new AnalysisContext(
                charset,
                fileReader,
                unitProvider,
                eventHandler,
                withTrivia,
                tabStop
            );
        }

        /**
         * Get the analysis context object from its reference.
         *
         * @param reference The native reference to the analysis context.
         * @param eventHandler The corresponding event handler.
         * @param setEventHandler Whether to set the result's eventHandler
         * field to eventHandler when there is already a cached
         * AnalysisContext.
         * @return The Java analysis unit associated with the reference.
         */
        static AnalysisContext fromReference(
            final PointerWrapper reference,
            final EventHandler eventHandler,
            final boolean setEventHandler
        ) {
            if(contextCache.containsKey(reference)) {
                final AnalysisContext res = contextCache.get(reference);
                increaseRefCounter(res);
                if(setEventHandler)
                    res.eventHandler = eventHandler;
                return res;
            } else {
                final AnalysisContext result =
                    new AnalysisContext(reference, eventHandler);
                contextCache.put(reference, result);
                return result;
            }
        }

        /**
         * Get the analysis context object from its reference.
         *
         * @param reference The native reference to the analysis context.
         * @return The Java analysis unit associated with the reference.
         */
        static AnalysisContext fromReference(
            final PointerWrapper reference
        ) {
            return fromReference(reference, null, false);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a native pointer to the native analysis context in the
         * Java class.
         *
         * @param pointer The pointer to the NI analysis context
         * native value.
         * @return The newly wrapped analysis context.
         */
        static AnalysisContext wrap(
            final WordPointer pointer
        ) {
            return wrap((AnalysisContextNative) pointer.read());
        }

        /**
         * Wrap an analysis context native value in the Java class.
         *
         * @param analysisContextNative The NI analysis context native value.
         * @return The newly wrapped analysis context.
         */
        static AnalysisContext wrap(
            final AnalysisContextNative analysisContextNative
        ) {
            return fromReference(new PointerWrapper(analysisContextNative));
        }

        /**
         * Unwrap the analysis context in the given native pointer.
         *
         * @param pointer The pointer to place the native analysis unit.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(this.unwrap());
        }

        /**
         * Get the native value of the analysis context.
         *
         * @return The native analysis context.
         */
        AnalysisContextNative unwrap() {
            return (AnalysisContextNative) this.reference.ni();
        }

        // ----- Getters -----

        public EventHandler getEventHandler() {
            return this.eventHandler;
        }

        /**
         * Get the currently open rewriting context associated to this
         * analysis context. The None rewriting context is returned if the
         * current context hasn't started a rewriting session.
         * @see AnalysisContext#startRewriting()
         */
        public RewritingContext getRewritingContext() {
            return this.rewritingContext;
        }

        // ----- Class methods -----

        /**
         * Increase the reference counter of the given context.
         *
         * @param context The context to increase the reference counter of.
         */
        private static void increaseRefCounter(
            final AnalysisContext context
        ) {
            // Increase the context reference counter of the context if not null
            if(!context.reference.isNull()) {
                if(ImageInfo.inImageCode()) {
                    NI_LIB.lkt_context_incref(context.reference.ni());
                } else {
                    JNI_LIB.lkt_context_incref(context.reference.jni());
                }
            }
        }

        // ----- Instance methods -----

        /**
         * Get an analysis unit from the given file in the current context.
         *
         * @param fileName The file to get the analysis unit from.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName
        ) {
            return this.getUnitFromFile(
                fileName,
                null,
                false,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given file in the current context
         * with additional parameters.
         *
         * @param fileName The file to get the analysis unit from.
         * @param charset The charset of the given file.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName,
            final String charset
        ) {
            return this.getUnitFromFile(
                fileName,
                charset,
                false,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given file in the current context
         * with additional parameters.
         *
         * @param fileName The file to get the analysis unit from.
         * @param charset The charset of the given file.
         * @param reparse If the file should be reparsed.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName,
            final String charset,
            final boolean reparse
        ) {
            return this.getUnitFromFile(
                fileName,
                charset,
                reparse,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given file in the current context
         * with additional parameters.
         *
         * @param fileName The file to get the analysis unit from.
         * @param charset The charset of the given file.
         * @param reparse If the file should be reparsed.
         * @param rule The grammar rule to parse the source with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName,
            final String charset,
            final boolean reparse,
            final GrammarRule rule
        ) {

            if(ImageInfo.inImageCode()) {
                final CCharPointer fileNameNative = toCString(fileName);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                final AnalysisUnitNative resNative =
                    NI_LIB.lkt_get_analysis_unit_from_file(
                    this.reference.ni(),
                    fileNameNative,
                    charsetNative,
                    (reparse ? 1 : 0),
                    rule.toC()
                );
                UnmanagedMemory.free(fileNameNative);
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.lkt_get_analysis_unit_from_file(
                    this,
                    fileName,
                    charset,
                    reparse,
                    rule.toC()
                );
            }

        }

        /**
         * Get an analysis unit from the given buffer in the current context.
         *
         * @param buffer The buffer to parse.
         * @param name The name of the buffer.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromBuffer(
            final String buffer,
            final String name
        ) {
            return this.getUnitFromBuffer(
                buffer,
                name,
                null,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given buffer in the current context
         * with additional parameters.
         *
         * @param buffer The buffer to parse.
         * @param name The name of the buffer.
         * @param charset The charset of the buffer.
         * @param rule The rule to parse the buffer with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromBuffer(
            final String buffer,
            final String name,
            final String charset,
            final GrammarRule rule
        ) {

            if(ImageInfo.inImageCode()) {
                final CCharPointer bufferNative = toCString(buffer);
                final CCharPointer nameNative = toCString(name);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                final AnalysisUnitNative resNative =
                    NI_LIB.lkt_get_analysis_unit_from_buffer(
                    this.reference.ni(),
                    nameNative,
                    charsetNative,
                    bufferNative,
                    buffer.length(),
                    rule.toC()
                );
                UnmanagedMemory.free(bufferNative);
                UnmanagedMemory.free(nameNative);
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.lkt_get_analysis_unit_from_buffer(
                    this,
                    name,
                    charset,
                    buffer,
                    buffer.length(),
                    rule.toC()
                );
            }

        }

        /**
         * Get an analysis unit from the given unit name and unit kind in the
         * current context with additional parameters.
         *
         * @param name Name of the unit.
         * @param kind Kind of the unit.
         * @param charset The charset of the buffer.
         * @param rule The rule to parse the buffer with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromProvider(
            final Text name,
            final AnalysisUnitKind kind,
            final String charset,
            final boolean reparse
        ) {
            if(ImageInfo.inImageCode()) {
                TextNative nameNative = StackValue.get(TextNative.class);
                name.unwrap(nameNative);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);
                final AnalysisUnitNative resNative =
                    NI_LIB.lkt_get_analysis_unit_from_provider(
                    this.reference.ni(),
                    nameNative,
                    kind.toC(),
                    charsetNative,
                    (reparse ? 1 : 0)
                );
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.lkt_get_analysis_unit_from_provider(
                    this,
                    name,
                    kind.toC(),
                    charset,
                    reparse
                );
            }
        }

        public AnalysisUnit getUnitFromProvider(
            final Text name,
            final AnalysisUnitKind kind
        ) {
            return this.getUnitFromProvider(name, kind, "", false);
        }

        /**
         * Start a rewriting session for Context.
         *
         * This handle will keep track of all changes to do on Context's
         * analysis units. Once the set of changes is complete, call the Apply
         * procedure to actually update Context. This makes it possible to
         * inspect the "old" Context state while creating the list of changes.
         *
         * There can be only one rewriting session per analysis context, so
         * this will raise an Existing_Rewriting_Handle_Error exception if
         * Context already has a living rewriting session.
         */
        public RewritingContext startRewriting() {
            final RewritingContext res;

            if(ImageInfo.inImageCode()) {
                final RewritingContextNative resNative =
                    NI_LIB.lkt_rewriting_start_rewriting(
                        this.reference.ni()
                    );
                res = RewritingContext.wrap(resNative);
            } else {
                res = JNI_LIB.lkt_rewriting_start_rewriting(this);
            }

            checkException();
            this.rewritingContext = res;
            return this.rewritingContext;
        }

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_context_decref(this.reference.ni());
            } else {
                JNI_LIB.lkt_context_decref(this.reference.jni());
            }
            checkException();

        }

        


    }

    /**
     * This type represents the analysis of a single file.
     *
     * This type has strong-reference semantics and is ref-counted.
     * Furthermore, a reference to a unit contains an implicit reference to the
     * context that owns it. This means that keeping a reference to a unit will
     * keep the context and all the unit it contains allocated.
     */
    public static final class AnalysisUnit extends LangkitSupport.AnalysisUnit {

        // ----- Class attributes -----

        /** Singleton that represents the none analysis unit. */
        public static final AnalysisUnit NONE = new AnalysisUnit(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the native analysis unit. */
        private final PointerWrapper reference;

        /** The cache for the unit root. */
        private LktNode root;

        // ----- Constructors -----

        /**
         * Create a new analysis unit from its value.
         *
         * @param reference The native analysis unit native value in
         * a pointer wrapper.
         */
        AnalysisUnit(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native analysis unit in the Java class.
         *
         * @param pointer The pointer the native analysis unit value.
         * @return The newly wrapped analysis unit.
         */
        static AnalysisUnit wrap(
            final WordPointer pointer
        ) {
            return wrap((AnalysisUnitNative) pointer.read());
        }

        /**
         * Wrap a NI analysis unit native value in the Java class.
         *
         * @param analysisUnitNative The NI analysis unit native value.
         * @return The newly wrapped analysis unit.
         */
        static AnalysisUnit wrap(
            final AnalysisUnitNative analysisUnitNative
        ) {
            return new AnalysisUnit(new PointerWrapper(analysisUnitNative));
        }

        /**
         * Unwrap the analysis unit in the given pointer.
         *
         * @param pointer The pointer to place the native analysis unit in.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(this.unwrap());
        }

        /**
         * Unwrap the analysis unit as a native value.
         *
         * @return The native analysis unit.
         */
        AnalysisUnitNative unwrap() {
            return (AnalysisUnitNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /**
         * Get the root node of the analysis unit.
         *
         * @return The root node.
         */
        public LktNode getRoot() {
            if(this.root == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    NI_LIB.lkt_unit_root(
                        this.reference.ni(),
                        entityNative
                    );
                    this.root = LktNode.fromEntity(
                        Entity.wrap(entityNative)
                    );
                } else {
                    this.root = LktNode.fromEntity(
                        JNI_LIB.lkt_unit_root(this)
                    );
                }

            }
            return this.root;
        }

        /**
         * Get the analysis unit file name with its full path.
         *
         * @return The unit file name.
         */
        public String getFileName() {
            return this.getFileName(true);
        }

        /**
         * Get the unit's file name.
         *
         * @param fullPath If the method should return the
         * file full absolute path.
         * @return The file name.
         */
        @CompilerDirectives.TruffleBoundary
        public String getFileName(
            final boolean fullPath
        ) {
            final String absoluteFile;

            if(ImageInfo.inImageCode()) {
                final CCharPointer resNative = NI_LIB.lkt_unit_filename(
                    this.reference.ni()
                );
                absoluteFile = toJString(resNative);
                NI_LIB.lkt_free(resNative);
            } else {
                absoluteFile = JNI_LIB.lkt_unit_filename(this);
            }

            if(fullPath) {
                return absoluteFile;
            } else {
                return new File(absoluteFile).getName();
            }
        }

        /**
         * Get the number of tokens in the analysis unit.
         *
         * @return The number of token.
         */
        public int getTokenCount() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.lkt_unit_token_count(this.reference.ni());
            } else {
                return JNI_LIB.lkt_unit_token_count(this);
            }

        }

        /**
         * Get the number of trivia in the analysis unit.
         *
         * @return The number of trivia.
         */
        public int getTriviaCount() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.lkt_unit_trivia_count(
                    this.reference.ni()
                );
            } else {
                return JNI_LIB.lkt_unit_trivia_count(this);
            }

        }

        /**
         * Return the first token of the analysis unit.
         *
         * @return The first token.
         */
        public Token getFirstToken() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.lkt_unit_first_token(
                    this.reference.ni(),
                    tokenNative
                );
                return Token.wrap(tokenNative, this);
            } else {
                return JNI_LIB.lkt_unit_first_token(this);
            }

        }

        /**
         * Return the last token of the analysis unit.
         *
         * @return The last token.
         */
        public Token getLastToken() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.lkt_unit_last_token(
                    this.reference.ni(),
                    tokenNative
                );
                return Token.wrap(tokenNative, this);
            } else {
                return JNI_LIB.lkt_unit_last_token(this);
            }

        }

        /**
         * Get the text of the analysis unit in a string.
         *
         * @return The text of the analysis unit source.
         */
        public String getText() {
            return Token.textRange(
                this.getFirstToken(),
                this.getLastToken()
            );
        }

        /**
         * Get the analysis context that owns the unit.
         *
         * @return The owning analysis context.
         */
        public AnalysisContext getContext() {

            if(ImageInfo.inImageCode()) {
                final AnalysisContextNative contextNative =
                    NI_LIB.lkt_unit_context(this.reference.ni());
                return AnalysisContext.wrap(contextNative);
            } else {
                return JNI_LIB.lkt_unit_context(this);
            }

        }

        /**
         * Get the list of associated diagnostics. Those are parsing errors.
         *
         * @return The diagnostics of the unit.
         */
        public Diagnostic[] getDiagnostics() {
            final int diagnosticCount;

            if(ImageInfo.inImageCode()) {
                diagnosticCount = NI_LIB.lkt_unit_diagnostic_count(
                    this.reference.ni()
                );
            } else {
                diagnosticCount = JNI_LIB.lkt_unit_diagnostic_count(
                    this
                );
            }

            Diagnostic[] res = new Diagnostic[diagnosticCount];

            if(ImageInfo.inImageCode()) {
                final DiagnosticNative diagnosticNative = StackValue.get(
                    DiagnosticNative.class
                );
                for(int i = 0 ; i < diagnosticCount ; i++) {
                    NI_LIB.lkt_unit_diagnostic(
                        this.reference.ni(),
                        i,
                        diagnosticNative
                    );
                    res[i] = Diagnostic.wrap(diagnosticNative);
                }
            } else {
                for(int i = 0 ; i < diagnosticCount ; i++) {
                    res[i] = JNI_LIB.lkt_unit_diagnostic(this, i);
                }
            }

            return res;
        }

        /**
         * Return the rewriting handle corresponding to Unit
         */
        public RewritingUnit getRewritingUnit() {
            final RewritingUnit res;

            if(ImageInfo.inImageCode()) {
                RewritingUnitNative rewritingUnitNative =
                    NI_LIB.lkt_rewriting_unit_to_handle(
                        this.unwrap()
                    );
                res = RewritingUnit.wrap(rewritingUnitNative);
            } else {
                res = JNI_LIB.lkt_rewriting_unit_to_handle(this);
            }

            checkException();
            return res;
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return "<AnalysisUnit \"" + this.getFileName(false) + "\">";
        }

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof AnalysisUnit)) return false;
            final AnalysisUnit other = (AnalysisUnit) o;
            return this.reference.equals(other.reference);
        }

    }

    /**
     * Handle for an analysis context rewriting session
     */
    public static final class RewritingContext implements
        LangkitSupport.RewritingContextInterface {

        // ----- Class attributes -----

        /** Singleton representing the none rewriting context */
        public static final RewritingContext NONE = new RewritingContext(
            PointerWrapper.nullPointer()
        );

        /**
         * This map contains all wrapped rewriting context associated to their
         * address.
         */
        private static final Map<PointerWrapper, RewritingContext> contextCache
            = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native rewriting context handle. */
        private final PointerWrapper reference;

        /**
         * Cache for the analysis context associated with this rewriting
         * context.
         */
        private AnalysisContext analysisContext;

        /** Whether the rewriting context has already been closed. */
        private boolean closed;

        // ----- Constructors -----

        /** Create a new rewriting context with its native reference. */
        private RewritingContext(
            final PointerWrapper reference
        ) {
            this.reference = reference;
            this.closed = false;
        }

        /**
         * From the given native reference, get the associate rewriting
         * context Java object. A native reference can only have one
         * associated Java instance.
         */
        @CompilerDirectives.TruffleBoundary
        static RewritingContext fromReference(
            final PointerWrapper reference
        ) {
            if(!contextCache.containsKey(reference)) {
                contextCache.put(
                    reference,
                    new RewritingContext(reference)
                );
            }
            final RewritingContext res = contextCache.get(reference);
            res.closed = false;
            return res;
        }

        // ----- Graal C API methods -----

        /** Wrap a native pointer to a native rewriting context. */
        static RewritingContext wrap(
            final Pointer pointer
        ) {
            return wrap((RewritingContextNative) pointer.readWord(0));
        }

        /** Wrap a native rewriting context. */
        static RewritingContext wrap(
            final RewritingContextNative rewritingContextNative
        ) {
            return fromReference(new PointerWrapper(rewritingContextNative));
        }

        /** Unwrap the analysis context in the given native pointer. */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /** Unwrap the rewriting context as a native value. */
        RewritingContextNative unwrap() {
            return (RewritingContextNative) this.reference.ni();
        }

        // ----- Getters -----

        public boolean isClosed() {
            return this.closed;
        }


        /**
         * Return the analysis context associated to Handle
         */
        public AnalysisContext getAnalysisContext() {
            if(this.analysisContext == null) {

                if(ImageInfo.inImageCode()) {
                    this.analysisContext = AnalysisContext.wrap(
                        NI_LIB.lkt_rewriting_handle_to_context(
                            this.unwrap()
                        )
                    );
                } else {
                    this.analysisContext =
                        JNI_LIB.lkt_rewriting_handle_to_context(this);
                }

            }
            return this.analysisContext;
        }

        // ----- Instance methods -----

        /**
         * Return the list of unit rewriting handles in the given context
         * handle for units that the Apply primitive will modify.
         */
        public RewritingUnit[] rewritingUnits() {

            if(ImageInfo.inImageCode()) {
                // Get the native array
                final WordPointer unitArrayNative =
                    NI_LIB.lkt_rewriting_unit_handles(this.unwrap());

                // Fill the Java result list
                final List<RewritingUnit> resList = new ArrayList<>();
                int cursor = 0;
                while(((Pointer) unitArrayNative.read(cursor)).isNonNull()) {
                    resList.add(RewritingUnit.wrap(
                        (RewritingUnitNative) unitArrayNative.read(cursor)
                    ));
                    cursor++;
                }

                // Free the native array
                NI_LIB.lkt_free(unitArrayNative);

                // Return the Java list as an array
                return resList.toArray(new RewritingUnit[0]);
            } else {
                return JNI_LIB.lkt_rewriting_unit_handles(this);
            }

        }

        /**
         * Create a new node of the given Kind, with empty text (for token
         * nodes) or children (for regular nodes).
         */
        public RewritingNode createNode(
            final NodeKind kind
        ) {
            final RewritingNode res;

            if(ImageInfo.inImageCode()) {
                res = RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_create_node(
                        this.unwrap(),
                        kind.toC()
                    )
                );
            } else {
                res = JNI_LIB.lkt_rewriting_create_node(
                    this,
                    kind.toC()
                );
            }

            checkException();
            return res;
        }

        /**
         * Create a new regular node of the given Kind and assign it the given
         * Children.
         *
         * Except for lists, which can have any number of children, the size of
         * Children must match the number of children associated to the given
         * Kind. Besides, all given children must not be tied.
         */
        public RewritingNode createNode(
            final LangkitSupport.NodeKindInterface kind,
            final LangkitSupport.RewritingNodeInterface... children
        ) {
            final NodeKind castKind = (NodeKind) kind;
            final RewritingNode[] castChildren = Arrays.copyOf(
                children, children.length, RewritingNode[].class);
            final RewritingNode res;

            if(ImageInfo.inImageCode()) {
                final WordPointer childrenNative =
                    rewritingNodesToNative(castChildren);
                res = RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_create_regular_node(
                        this.unwrap(),
                        castKind.toC(),
                        childrenNative,
                        castChildren.length
                    )
                );
                UnmanagedMemory.free(childrenNative);
            } else {
                res = JNI_LIB.lkt_rewriting_create_regular_node(
                    this,
                    castKind.toC(),
                    castChildren
                );
            }

            checkException();
            return res;
        }

        /**
         * Create a new token node with the given Kind and Text
         */
        public RewritingNode createTokenNode(
            final LangkitSupport.NodeKindInterface kind,
            final String text
        ) {
            final NodeKind castKind = (NodeKind) kind;
            try (
                final Text nodeText = Text.create(text);
            ) {
                final RewritingNode res;

                if(ImageInfo.inImageCode()) {
                    final TextNative nodeTextNative = StackValue.get(
                        TextNative.class
                    );
                    nodeText.unwrap(nodeTextNative);
                    res = RewritingNode.wrap(
                        NI_LIB.lkt_rewriting_create_token_node(
                            this.unwrap(),
                            castKind.toC(),
                            nodeTextNative
                        )
                    );
                } else {
                    res = JNI_LIB.lkt_rewriting_create_token_node(
                        this,
                        castKind.toC(),
                        nodeText
                    );
                }

                checkException();
                return res;
            }
        }

        @CompilerDirectives.TruffleBoundary
        public RewritingNode createFromTemplate(
            final String template,
            final String rule,
            final LangkitSupport.RewritingNodeInterface... arguments
        ) {
            final RewritingNode[] castArguments = Arrays.copyOf(
                arguments, arguments.length, RewritingNode[].class);

            return this.createFromTemplate(
                template, GrammarRule.valueOf(rule.toUpperCase()),
                                              castArguments);
        }

        /**
         * Create a tree of new nodes from the given Template string, replacing
         * placeholders with nodes in Arguments and parsed according to the
         * given grammar Rule.
         */
        public RewritingNode createFromTemplate(
            final String template,
            final GrammarRule rule,
            final RewritingNode... arguments
        ) {
            try (
                final Text templateText = Text.create(template);
            ) {
                final RewritingNode res;

                if(ImageInfo.inImageCode()) {
                    final WordPointer argumentsNative =
                        rewritingNodesToNative(arguments);
                    TextNative templateTextNative =
                        StackValue.get(TextNative.class);
                    templateText.unwrap(templateTextNative);
                    res = RewritingNode.wrap(
                        NI_LIB.lkt_rewriting_create_from_template(
                            this.unwrap(),
                            templateTextNative,
                            argumentsNative,
                            arguments.length,
                            rule.toC()
                        )
                    );
                    UnmanagedMemory.free(argumentsNative);
                } else {
                    res = JNI_LIB.lkt_rewriting_create_from_template(
                        this,
                        templateText,
                        arguments,
                        rule.toC()
                    );
                }

                checkException();
                return res;
            }
        }

        /**
         * Apply all modifications to Handle's analysis context. If that
         * worked, close Handle and return (Success => True). Otherwise,
         * reparsing did not work, so keep Handle and its Context unchanged and
         * return details about the error that happened.
         *
         * Note that on success, this invalidates all related unit/node
         * handles.
         */
        public RewritingApplyResult apply() {
            final RewritingApplyResult res;

            if(ImageInfo.inImageCode()) {
                RewritingApplyResultNative resNative = StackValue.get(
                    RewritingApplyResultNative.class
                );
                NI_LIB.lkt_rewriting_apply(
                    this.unwrap(),
                    resNative
                );
                res = RewritingApplyResult.wrap(resNative);
            } else {
                res = JNI_LIB.lkt_rewriting_apply(this);
            }

            this.closed = res.success;
            return res;
        }

        /**
         * Discard all modifications registered in Handle and close Handle.
         * This invalidates all related unit/node handles.
         */
        public void close() {
            if(!this.closed) {
                if(this.analysisContext != null) {
                    this.analysisContext.close();
                    this.analysisContext = null;
                }

                if(ImageInfo.inImageCode()) {
                    NI_LIB.lkt_rewriting_abort_rewriting(this.unwrap());
                } else {
                    JNI_LIB.lkt_rewriting_abort_rewriting(this);
                }

                checkException();
                this.closed = true;
            }
        }

    }

    /**
     * Handle for the process of rewriting an analysis unit. Such handles are
     * owned by a Rewriting_Handle instance.
     */
    public static final class RewritingUnit {

        // ----- Class methods -----

        /** Singleton representing the none value for the rewriting unit. */
        public static final RewritingUnit NONE = new RewritingUnit(
            PointerWrapper.nullPointer()
        );

        // ----- Instance methods -----

        /** Reference to the native rewriting unit. */
        private final PointerWrapper reference;

        /** Cache for the analysis unit associated to the rewriting unit. */
        private AnalysisUnit analysisUnit;

        /** Cache for the unit root rewriting node. */
        private RewritingNode root;

        // ----- Constructors -----

        /** Create a new rewriting unit with its native reference. */
        RewritingUnit(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /** Wrap the given pointer to the native rewriting unit. */
        static RewritingUnit wrap(
            final Pointer pointer
        ) {
            return wrap((RewritingUnitNative) pointer.readWord(0));
        }

        /** Wrap the native rewriting unit. */
        static RewritingUnit wrap(
            final RewritingUnitNative rewritingUnitNative
        ) {
            return new RewritingUnit(new PointerWrapper(rewritingUnitNative));
        }

        /** Unwrap the rewriting unit in the given pointer. */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /** Unwrap the rewriting unit and return its native value. */
        RewritingUnitNative unwrap() {
            return (RewritingUnitNative) this.reference.ni();
        }

        // ----- Getters -----

        /**
         * Return the unit corresponding to Handle
         */
        public AnalysisUnit getAnalysisUnit() {
            if(this.analysisUnit == null) {

                if(ImageInfo.inImageCode()) {
                    this.analysisUnit = AnalysisUnit.wrap(
                        NI_LIB.lkt_rewriting_handle_to_unit(
                            this.unwrap()
                        )
                    );
                } else {
                    this.analysisUnit =
                        JNI_LIB.lkt_rewriting_handle_to_unit(this);
                }

            }
            return this.analysisUnit;
        }

        /**
         * Return the node handle corresponding to the root of the unit which
         * Handle designates.
         */
        public RewritingNode getRoot() {
            if(this.root == null) {

                if(ImageInfo.inImageCode()) {
                    this.root = RewritingNode.wrap(
                        NI_LIB.lkt_rewriting_unit_root(this.unwrap())
                    );
                } else {
                    this.root = JNI_LIB.lkt_rewriting_unit_root(this);
                }

            }
            return this.root;
        }

        // ----- Setters -----

        /**
         * Set the root node for the unit Handle to Root. This unties the
         * previous root handle. If Root is not No_Node_Rewriting_Handle, this
         * also ties Root to Handle.
         *
         * Root must not already be tied to another analysis unit handle.
         */
        public void setRoot(
            final RewritingNode root
        ) {
            this.root = root;

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_unit_set_root(
                    this.unwrap(),
                    root.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_unit_set_root(this, root);
            }
        }

        // ----- Instance methods -----

        /**
         * Return the text associated to the given unit.
         */
        public String unparse() {
            final Text unparseText;

            if(ImageInfo.inImageCode()) {
                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                NI_LIB.lkt_rewriting_unit_unparse(
                    this.unwrap(),
                    textNative
                );
                unparseText = Text.wrap(textNative);
            } else {
                unparseText = JNI_LIB.lkt_rewriting_unit_unparse(this);
            }

            String res = unparseText.getContent();
            unparseText.close();
            return res;
        }

        // ----- Override methods -----

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof RewritingUnit)) return false;
            final RewritingUnit other = (RewritingUnit) o;
            return this.reference.equals(other.reference);
        }

    }

    /**
     * Handle for the process of rewriting an AST node. Such handles are owned
     * by a Rewriting_Handle instance.
     */
    public static final class RewritingNode implements
        LangkitSupport.RewritingNodeInterface {

        // ----- Class attributes -----

        /** Singleton representing the none value for rewriting node. */
        public static final RewritingNode NONE = new RewritingNode(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the native rewriting node. */
        private final PointerWrapper reference;

        /** Kind of the rewriting node. */
        private NodeKind kind;

        /** Cache for the associated parsed node. */
        private LktNode parsedNode;

        /** Cache for the rewriting context containing this node. */
        private RewritingContext rewritingContext;

        // ----- Constructors -----

        /** Create a new rewriting node with its native reference. */
        RewritingNode(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API -----

        /** Wrap a pointer to a native rewriting node. */
        static RewritingNode wrap(
            final Pointer pointer
        ) {
            return wrap((RewritingNodeNative) pointer.readWord(0));
        }

        /** Wrap the native rewriting node. */
        static RewritingNode wrap(
            final RewritingNodeNative rewritingNodeNative
        ) {
            return new RewritingNode(new PointerWrapper(rewritingNodeNative));
        }

        /** Unwrap the rewriting node in the given pointer. */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /** Unwrap the rewriting node and return its native value. */
        RewritingNodeNative unwrap() {
            return (RewritingNodeNative) this.reference.ni();
        }

        // ----- Getters -----

        public RewritingNode getNONE() {
            return this.NONE;
        }

        /**
         * Return the kind corresponding to Handle's node
         */
        public NodeKind getKind() {
            if(this.kind == null) {
                final int kindNative;

                if(ImageInfo.inImageCode()) {
                    kindNative = NI_LIB.lkt_rewriting_kind(
                        this.unwrap()
                    );
                } else {
                    kindNative = JNI_LIB.lkt_rewriting_kind(this);
                }

                this.kind = NodeKind.fromC(kindNative);
            }
            return this.kind;
        }

        /**
         * Return the node which the given rewriting Handle relates to. This
         * can be the null entity if this handle designates a new node.
         */
        public LktNode getParsedNode() {
            if(this.parsedNode == null) {
                final Entity nodeEntity;

                if(ImageInfo.inImageCode()) {
                    final Pointer nodeNative =
                        NI_LIB.lkt_rewriting_handle_to_node(
                            this.unwrap()
                        );
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    NI_LIB.lkt_create_bare_entity(
                        nodeNative,
                        entityNative
                    );
                    nodeEntity = Entity.wrap(entityNative);
                } else {
                    nodeEntity = JNI_LIB.lkt_rewriting_handle_to_node(
                        this
                    );
                }

                this.parsedNode = LktNode.fromEntity(nodeEntity);
            }
            return this.parsedNode;
        }

        /**
         * Return a handle for the rewriting context to which Handle belongs
         */
        public RewritingContext getRewritingContext() {
            if(this.rewritingContext == null) {

                if(ImageInfo.inImageCode()) {
                    this.rewritingContext = RewritingContext.wrap(
                        NI_LIB.lkt_rewriting_node_to_context(
                            this.unwrap()
                        )
                    );
                } else {
                    this.rewritingContext =
                        JNI_LIB.lkt_rewriting_node_to_context(this);
                }

            }
            return this.rewritingContext;
        }

        /** Get whether the rewriting node is a none node. */
        public boolean isNone() {
            return this.reference.isNull();
        }

        // ----- Instance methods -----

        /**
         * Create a clone of the Handle node tree. The result is not tied to
         * any analysis unit tree.
         */
        public RewritingNode clone() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_clone(this.unwrap())
                );
            } else {
                return JNI_LIB.lkt_rewriting_clone(this);
            }

        }

        /**
         * Turn the given rewritten node Handle designates into text. This is
         * the text that is used in Apply in order to re-create an analysis
         * unit.
         */
        public String unparse() {
            final Text text;

            if(ImageInfo.inImageCode()) {
                TextNative textNative = StackValue.get(TextNative.class);
                NI_LIB.lkt_rewriting_node_unparse(
                    this.unwrap(),
                    textNative
                );
                text = Text.wrap(textNative);
            } else {
                text = JNI_LIB.lkt_rewriting_node_unparse(this);
            }

            final String res = text.getContent();
            text.close();
            return res;
        }

        /**
         * Return a representation of ``Handle`` as a string.
         */
        public String image() {
            final Text text;

            if(ImageInfo.inImageCode()) {
                TextNative textNative = StackValue.get(TextNative.class);
                NI_LIB.lkt_rewriting_node_image(
                    this.unwrap(),
                    textNative
                );
                text = Text.wrap(textNative);
            } else {
                text = JNI_LIB.lkt_rewriting_node_image(this);
            }

            final String res = text.getContent();
            text.close();
            return res;
        }

        /**
         * Return whether this node handle is tied to an analysis unit. If it
         * is not, it can be passed as the Child parameter to Set_Child.
         */
        public boolean isTied() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.lkt_rewriting_tied(
                    this.unwrap()
                ) > 0;
            } else {
                return JNI_LIB.lkt_rewriting_tied(this);
            }

        }

        /**
         * Return a handle for the node that is the parent of Handle's node.
         * This is ``No_Rewriting_Handle`` for a node that is not tied to any
         * tree yet.
         */
        public RewritingNode parent() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_parent(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_parent(this);
            }

        }

        /**
         * Return the list of children for ``Handle``.
         */
        public RewritingNode[] children() {

            if(ImageInfo.inImageCode()) {
                // Call the native function
                final WordPointer childrenPointer =
                    StackValue.get(WordPointer.class);
                final CIntPointer countPointer =
                    StackValue.get(CIntPointer.class);
                NI_LIB.lkt_rewriting_children(
                    this.unwrap(),
                    childrenPointer,
                    countPointer
                );
                final WordPointer children = childrenPointer.read();
                final int count = countPointer.read();

                // Create the Java array and fill it
                final RewritingNode[] res = new RewritingNode[count];
                for(int i = 0; i < res.length; i++) {
                    res[i] = RewritingNode.wrap(
                        (RewritingNodeNative) children.read(i)
                    );
                }

                // Free the native children
                NI_LIB.lkt_free(children);

                // Return the result
                return res;
            } else {
                return JNI_LIB.lkt_rewriting_children(this);
            }

        }

        /**
         * Return the node that is in the syntax ``Field`` for ``Handle``
         */
        public RewritingNode getChild(
            final MemberReference childMember
        ) {
            final RewritingNode res;

            if(ImageInfo.inImageCode()) {
                res = RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_child(
                        this.unwrap(),
                        childMember.toC()
                    )
                );
            } else {
                res = JNI_LIB.lkt_rewriting_child(
                    this,
                    childMember.toC()
                );
            }

            checkException();
            return res;
        }

        /**
         * If ``Child`` is ``No_Rewriting_Node``, untie the syntax field in
         * ``Handle`` corresponding to ``Field``, so it can be attached to
         * another one. Otherwise, ``Child`` must have no parent as it will be
         * tied to ``Handle``'s tree.
         */
        public void setChild(
            LangkitSupport.MemberReferenceInterface childMember,
            LangkitSupport.RewritingNodeInterface child
        ) {

            final MemberReference castChildMember =
                (MemberReference) childMember;
            final RewritingNode castChild = (RewritingNode) child;

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_set_child(
                    this.unwrap(),
                    castChildMember.toC(),
                    castChild.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_set_child(
                    this,
                    castChildMember.toC(),
                    castChild
                );
            }
            checkException();

        }

        /**
         * If Handle is the root of an analysis unit, untie it and set New_Node
         * as its new root. Otherwise, replace Handle with New_Node in Handle's
         * parent node.
         *
         * Note that: * Handle must be tied to an existing analysis unit
         * handle. * New_Node must not already be tied to another analysis unit
         * handle.
         */
        public void replace(
            final LangkitSupport.RewritingNodeInterface newNode
        ) {

            final RewritingNode nn = (RewritingNode) newNode;

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_replace(
                    this.unwrap(),
                    nn.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_replace(this, nn);
            }
            checkException();

        }

        /**
         * Assuming ``Handle`` refers to a list node, return a handle to its
         * first child, or ``No_Node_Rewriting_Handle``` if it has no child
         * node.
         */
        public RewritingNode firstChild() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_first_child(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_first_child(this);
            }

        }

        /**
         * Assuming ``Handle`` refers to a list node, return a handle to its
         * last child, or ``No_Node_Rewriting_Handle``` if it has no child
         * node.
         */
        public RewritingNode lastChild() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_last_child(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_last_child(this);
            }

        }

        /**
         * Assuming ``Handle`` refers to the child of a list node, return a
         * handle to its next sibling, or ``No_Node_Rewriting_Handle``` if it
         * is the last sibling.
         */
        public RewritingNode nextChild() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_next_child(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_next_child(this);
            }

        }

        /**
         * Assuming ``Handle`` refers to the child of a list node, return a
         * handle to its previous sibling, or ``No_Node_Rewriting_Handle``` if
         * it is the first sibling.
         */
        public RewritingNode previousChild() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_previous_child(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_previous_child(this);
            }

        }

        /**
         * Assuming ``Handle`` refers to the child of a list node, insert
         * ``New_Sibling`` as a new child in this list, right before
         * ``Handle``.
         */
        public void insertBefore(
            final LangkitSupport.RewritingNodeInterface toInsert
        ) {

            final RewritingNode castToInsert = (RewritingNode) toInsert;

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_insert_before(
                    this.unwrap(),
                    castToInsert.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_insert_before(this, castToInsert);
            }
            checkException();

        }

        /**
         * Assuming ``Handle`` refers to the child of a list node, insert
         * ``New_Sibling`` as a new child in this list, right before
         * ``Handle``.
         */
        public void insertAfter(
            final LangkitSupport.RewritingNodeInterface toInsert
        ) {

            final RewritingNode castToInsert = (RewritingNode) toInsert;

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_insert_after(
                    this.unwrap(),
                    castToInsert.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_insert_after(this, castToInsert);
            }
            checkException();

        }

        /**
         * Assuming ``Handle`` refers to a list node, insert ``New_Child`` to
         * be the first child in this list.
         */
        public void insertFirst(
            final LangkitSupport.RewritingNodeInterface toInsert
        ) {

            final RewritingNode castToInsert = (RewritingNode) toInsert;

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_insert_first(
                    this.unwrap(),
                    castToInsert.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_insert_first(this, castToInsert);
            }
            checkException();

        }

        /**
         * Assuming ``Handle`` refers to a list node, insert ``New_Child`` to
         * be the last child in this list.
         */
        public void insertLast(
            final LangkitSupport.RewritingNodeInterface toInsert
        ) {

            final RewritingNode castToInsert = (RewritingNode) toInsert;

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_insert_last(
                    this.unwrap(),
                    castToInsert.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_insert_last(this, castToInsert);
            }
            checkException();

        }

        /**
         * Assuming Handle refers to the child of a list node, remove it from
         * that list.
         */
        public void removeFromParent() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_remove_child(this.unwrap());
            } else {
                JNI_LIB.lkt_rewriting_remove_child(this);
            }
            checkException();

        }

        /** Get the text of the token node. */
        public String getText() {
            final Text resultText;

            if(ImageInfo.inImageCode()) {
                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                NI_LIB.lkt_rewriting_text(
                    this.unwrap(),
                    textNative
                );
                resultText = Text.wrap(textNative);
            } else {
                resultText = JNI_LIB.lkt_rewriting_text(this);
            }

            final String res = resultText.getContent();
            resultText.close();
            return res;
        }

        /**
         * Override text associated to the given token node.
         */
        public void setText(
            final String text
        ) {
            try (
                final Text nodeText = Text.create(text);
            ) {

                if(ImageInfo.inImageCode()) {
                    final TextNative textNative = StackValue.get(
                        TextNative.class
                    );
                    nodeText.unwrap(textNative);
                    NI_LIB.lkt_rewriting_set_text(
                        this.unwrap(),
                        textNative
                    );
                } else {
                    JNI_LIB.lkt_rewriting_set_text(this, nodeText);
                }
                checkException();

            }
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.image();
        }

        @Override
        public boolean equals(Object o) {
            if(o == this) return true;
            if(!(o instanceof RewritingNode)) return false;
            final RewritingNode other = (RewritingNode) o;
            return this.reference.equals(other.reference);
        }

    }

    // ===== Generated structure wrapping classes =====

        
    
    

    /**

     */
    public static final class Metadata
    {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Metadata NONE = new Metadata();

        // ----- Instance attributes ------

        /** The dummy field is always false (0) */
        final boolean dummy = false;

        // ----- Constructors -----

        /**
         * An empty constructor because the structure is empty
         */
        Metadata() {}

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The None instance because the structure is empty.
         */
        static Metadata wrap(
            final WordPointer pointer
        ) {
            return wrap((MetadataNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The None instance because the structure is empty.
         */
        static Metadata wrap(
            final MetadataNative structNative
        ) {
            return Metadata.NONE;
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final MetadataNative structNative
        ) {
            // Set the dummy field to 0 to make sure the memory is correctly
            // initialized.
            structNative.set_dummy((byte) 0);
        }


    }

        
    
    

    /**

     */
    public static final class EntityInfo
    {


        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final EntityInfo NONE = new EntityInfo(
            Metadata.NONE,PointerWrapper.nullPointer(),false
        );

        // ----- Instance attributes -----

        public final
        Metadata
        md;
        public final
        PointerWrapper
        rebindings;
        public final
        boolean
        fromRebound;


        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        EntityInfo(
            final Metadata md,final PointerWrapper rebindings,final boolean fromRebound
        ) {
            this.md = md;
            this.rebindings = rebindings;
            this.fromRebound = fromRebound;
        }

        /**
         * Create a new structure with the field values.
         */
        public static EntityInfo create(
            final Metadata md,final PointerWrapper rebindings,final boolean fromRebound
        ) {
            return new EntityInfo(
                md,rebindings,fromRebound
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static EntityInfo wrap(
            final WordPointer pointer
        ) {
            return wrap((EntityInfoNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static EntityInfo wrap(
            final EntityInfoNative structNative
        ) {
            return new EntityInfo(
                Metadata.NONE,PointerWrapper.wrap(structNative.get_rebindings()),BooleanWrapper.wrap(structNative.get_from_rebound())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final EntityInfoNative structNative
        ) {
            MetadataNative mdNative = structNative.address_md();this.md.unwrap(mdNative);
            Pointer rebindingsNative = structNative.address_rebindings();rebindingsNative.writeWord(0, this.rebindings.ni());
            CCharPointer fromReboundNative = structNative.address_from_rebound();fromReboundNative.write(this.fromRebound ? (byte) 1 : (byte) 0);
        }


    }

    
    

    /**

     */
    public static final class Entity
    {


        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Entity NONE = new Entity(
            PointerWrapper.nullPointer(),EntityInfo.NONE
        );

        // ----- Instance attributes -----

        public final
        PointerWrapper
        node;
        public final
        EntityInfo
        info;


        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        Entity(
            final PointerWrapper node,final EntityInfo info
        ) {
            this.node = node;
            this.info = info;
        }

        /**
         * Create a new structure with the field values.
         */
        public static Entity create(
            final PointerWrapper node,final EntityInfo info
        ) {
            return new Entity(
                node,info
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static Entity wrap(
            final WordPointer pointer
        ) {
            return wrap((EntityNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static Entity wrap(
            final EntityNative structNative
        ) {
            return new Entity(
                PointerWrapper.wrap(structNative.get_node()),EntityInfo.wrap(structNative.address_info())
            );
        }

        /**
         * Special wrapping method to construct a new entity structure
         * from a native bare node pointer.
         */
        static Entity wrapBareNode(
            final Pointer bareNode
        ) {
            return new Entity(
                PointerWrapper.wrap(bareNode),EntityInfo.NONE
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final EntityNative structNative
        ) {
            Pointer nodeNative = structNative.address_node();nodeNative.writeWord(0, this.node.ni());
            EntityInfoNative infoNative = structNative.address_info();this.info.unwrap(infoNative);
        }


    }

        
    
    

    /**
     * Completion item for language servers
     */
    public static final class CompleteItem
    implements LangkitSupport.CompletionItemInterface
    {


        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final CompleteItem NONE = new CompleteItem(
            Decl.NONE
        );

        // ----- Instance attributes -----

        public final
        Decl
        declaration;


        // ----- Getters -----


        public
        Decl
        gDecl() {
            return this.declaration;
        }


        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        CompleteItem(
            final Decl declaration
        ) {
            this.declaration = declaration;
        }

        /**
         * Create a new structure with the field values.
         */
        public static CompleteItem create(
            final Decl declaration
        ) {
            return new CompleteItem(
                declaration
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static CompleteItem wrap(
            final WordPointer pointer
        ) {
            return wrap((CompleteItemNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static CompleteItem wrap(
            final CompleteItemNative structNative
        ) {
            return new CompleteItem(
                Decl.fromEntity(Entity.wrap(structNative.address_declaration()))
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final CompleteItemNative structNative
        ) {
            EntityNative declarationNative = structNative.address_declaration();this.declaration.unwrap(declarationNative);
        }


    }

        
    
    

    /**
     * Result for ``CharLit.p_denoted_value``.
     *
     * If that property is successful, set ``has_error`` to false and ``value``
     * to the decoded character value. Otherwise, set ``has_error`` to true and
     * ``error_sloc`` and ``error_message`` to give information about the
     * decoding failure.
     */
    public static final class DecodedCharValue
    {


        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final DecodedCharValue NONE = new DecodedCharValue(
            Char.NONE,false,SourceLocation.NONE,StringWrapper.NONE
        );

        // ----- Instance attributes -----

        public final
        Char
        value;
        public final
        boolean
        hasError;
        public final
        SourceLocation
        errorSloc;
        public final
        String
        errorMessage;


        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        DecodedCharValue(
            final Char value,final boolean hasError,final SourceLocation errorSloc,final String errorMessage
        ) {
            this.value = value;
            this.hasError = hasError;
            this.errorSloc = errorSloc;
            this.errorMessage = errorMessage;
        }

        /**
         * Create a new structure with the field values.
         */
        public static DecodedCharValue create(
            final Char value,final boolean hasError,final SourceLocation errorSloc,final String errorMessage
        ) {
            return new DecodedCharValue(
                value,hasError,errorSloc,errorMessage
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static DecodedCharValue wrap(
            final WordPointer pointer
        ) {
            return wrap((DecodedCharValueNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static DecodedCharValue wrap(
            final DecodedCharValueNative structNative
        ) {
            return new DecodedCharValue(
                Char.wrap(structNative.get_value()),BooleanWrapper.wrap(structNative.get_has_error()),SourceLocation.wrap(structNative.get_error_sloc()),StringWrapper.wrap(structNative.get_error_message())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final DecodedCharValueNative structNative
        ) {
            CIntPointer valueNative = structNative.address_value();valueNative.write(this.value.value);
            CCharPointer hasErrorNative = structNative.address_has_error();hasErrorNative.write(this.hasError ? (byte) 1 : (byte) 0);
            SourceLocationNative errorSlocNative = structNative.address_error_sloc();this.errorSloc.unwrap(errorSlocNative);
            WordPointer errorMessageNative = structNative.address_error_message();StringWrapper.unwrap(this.errorMessage, errorMessageNative);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(DecodedCharValueNative structNative) {
            NI_LIB.lkt_internal_decoded_char_value_dec_ref(structNative);
        }

    }

        
    
    

    /**
     * Result for ``StringLit.p_denoted_value``.
     *
     * If that property is successful, set ``has_error`` to false and ``value``
     * to the decoded string value. Otherwise, set ``has_error`` to true and
     * ``error_sloc`` and ``error_message`` to give information about the
     * decoding failure.
     */
    public static final class DecodedStringValue
    {


        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final DecodedStringValue NONE = new DecodedStringValue(
            StringWrapper.NONE,false,SourceLocation.NONE,StringWrapper.NONE
        );

        // ----- Instance attributes -----

        public final
        String
        value;
        public final
        boolean
        hasError;
        public final
        SourceLocation
        errorSloc;
        public final
        String
        errorMessage;


        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        DecodedStringValue(
            final String value,final boolean hasError,final SourceLocation errorSloc,final String errorMessage
        ) {
            this.value = value;
            this.hasError = hasError;
            this.errorSloc = errorSloc;
            this.errorMessage = errorMessage;
        }

        /**
         * Create a new structure with the field values.
         */
        public static DecodedStringValue create(
            final String value,final boolean hasError,final SourceLocation errorSloc,final String errorMessage
        ) {
            return new DecodedStringValue(
                value,hasError,errorSloc,errorMessage
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static DecodedStringValue wrap(
            final WordPointer pointer
        ) {
            return wrap((DecodedStringValueNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static DecodedStringValue wrap(
            final DecodedStringValueNative structNative
        ) {
            return new DecodedStringValue(
                StringWrapper.wrap(structNative.get_value()),BooleanWrapper.wrap(structNative.get_has_error()),SourceLocation.wrap(structNative.get_error_sloc()),StringWrapper.wrap(structNative.get_error_message())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final DecodedStringValueNative structNative
        ) {
            WordPointer valueNative = structNative.address_value();StringWrapper.unwrap(this.value, valueNative);
            CCharPointer hasErrorNative = structNative.address_has_error();hasErrorNative.write(this.hasError ? (byte) 1 : (byte) 0);
            SourceLocationNative errorSlocNative = structNative.address_error_sloc();this.errorSloc.unwrap(errorSlocNative);
            WordPointer errorMessageNative = structNative.address_error_message();StringWrapper.unwrap(this.errorMessage, errorMessageNative);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(DecodedStringValueNative structNative) {
            NI_LIB.lkt_internal_decoded_string_value_dec_ref(structNative);
        }

    }

        
        
        
        
    
    

    /**
     * Describes an interpretation of a reference. Can be attached to logic
     * atoms (e.g. Binds) to indicate under which interpretation this
     * particular atom was produced, which can in turn be used to produce
     * informative diagnostics for resolution failures.
     */
    public static final class LogicContext
    implements LangkitSupport.LogicContextInterface
    {


        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final LogicContext NONE = new LogicContext(
            LktNode.NONE,LktNode.NONE
        );

        // ----- Instance attributes -----

        public final
        LktNode
        refNode;
        public final
        LktNode
        declNode;


        // ----- Getters -----


        public
        LktNode
        gRefNode() {
            return this.refNode;
        }

        public
        LktNode
        gDeclNode() {
            return this.declNode;
        }


        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        LogicContext(
            final LktNode refNode,final LktNode declNode
        ) {
            this.refNode = refNode;
            this.declNode = declNode;
        }

        /**
         * Create a new structure with the field values.
         */
        public static LogicContext create(
            final LktNode refNode,final LktNode declNode
        ) {
            return new LogicContext(
                refNode,declNode
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static LogicContext wrap(
            final WordPointer pointer
        ) {
            return wrap((LogicContextNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static LogicContext wrap(
            final LogicContextNative structNative
        ) {
            return new LogicContext(
                LktNode.fromEntity(Entity.wrap(structNative.address_ref_node())),LktNode.fromEntity(Entity.wrap(structNative.address_decl_node()))
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final LogicContextNative structNative
        ) {
            EntityNative refNodeNative = structNative.address_ref_node();this.refNode.unwrap(refNodeNative);
            EntityNative declNodeNative = structNative.address_decl_node();this.declNode.unwrap(declNodeNative);
        }


    }

        
        
        
    
    

    /**
     * Reference result struct
     */
    public static final class RefResult
    implements LangkitSupport.RefResultInterface
    {


        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final RefResult NONE = new RefResult(
            RefId.NONE
        );

        // ----- Instance attributes -----

        public final
        RefId
        ref;


        // ----- Getters -----


        public
        RefId
        gRef() {
            return this.ref;
        }


        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        RefResult(
            final RefId ref
        ) {
            this.ref = ref;
        }

        /**
         * Create a new structure with the field values.
         */
        public static RefResult create(
            final RefId ref
        ) {
            return new RefResult(
                ref
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static RefResult wrap(
            final WordPointer pointer
        ) {
            return wrap((RefResultNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static RefResult wrap(
            final RefResultNative structNative
        ) {
            return new RefResult(
                RefId.fromEntity(Entity.wrap(structNative.address_ref()))
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final RefResultNative structNative
        ) {
            EntityNative refNative = structNative.address_ref();this.ref.unwrap(refNative);
        }


    }

        
    
    

    /**
     * A raw diagnostic produced by a solver resolution failure. This contains
     * as much information as possible to allow formatters down the chain to
     * filter/choose which diagnostics to show among a set of diagnostics
     * produced for a single equation.
     *
     * * ``Message_Template`` is a string explaining the error, which may
     *   contain holes represented by the ``{}`` characters. Literal opening
     *   braces are encoded as ``{{``.
     *
     * * ``Args`` is an array of nodes, which are to be plugged in the holes of
     *   the template in the same order (i.e. the first argument goes into the
     *   first hole of the template, etc.).
     *
     * * ``Location`` is a node which indicates the location of the error.
     *
     * * ``Contexts`` is the array of contexts that were deemed relevant for
     *   this error.
     *
     * * ``Round`` is the solver round during which this diagnostic was
     *   emitted.
     */
    public static final class SolverDiagnostic
    implements LangkitSupport.SolverDiagnosticInterface
    {


        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final SolverDiagnostic NONE = new SolverDiagnostic(
            StringWrapper.NONE,LktNodeArrayWrapper.NONE,LktNode.NONE,LogicContextArrayWrapper.NONE,0
        );

        // ----- Instance attributes -----

        public final
        String
        messageTemplate;
        public final
        LktNode[]
        args;
        public final
        LktNode
        location;
        public final
        LogicContext[]
        contexts;
        public final
        int
        round;


        // ----- Getters -----


        public
        String
        gMessageTemplate() {
            return this.messageTemplate;
        }

        public
        LktNode[]
        gArgs() {
            return this.args;
        }

        public
        LktNode
        gLocation() {
            return this.location;
        }

        public
        LogicContext[]
        gContexts() {
            return this.contexts;
        }


        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        SolverDiagnostic(
            final String messageTemplate,final LktNode[] args,final LktNode location,final LogicContext[] contexts,final int round
        ) {
            this.messageTemplate = messageTemplate;
            this.args = args;
            this.location = location;
            this.contexts = contexts;
            this.round = round;
        }

        /**
         * Create a new structure with the field values.
         */
        public static SolverDiagnostic create(
            final String messageTemplate,final LktNode[] args,final LktNode location,final LogicContext[] contexts,final int round
        ) {
            return new SolverDiagnostic(
                messageTemplate,args,location,contexts,round
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static SolverDiagnostic wrap(
            final WordPointer pointer
        ) {
            return wrap((SolverDiagnosticNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static SolverDiagnostic wrap(
            final SolverDiagnosticNative structNative
        ) {
            return new SolverDiagnostic(
                StringWrapper.wrap(structNative.get_message_template()),LktNodeArrayWrapper.wrap(structNative.get_args()),LktNode.fromEntity(Entity.wrapBareNode(structNative.get_location())),LogicContextArrayWrapper.wrap(structNative.get_contexts()),IntegerWrapper.wrap(structNative.get_round())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final SolverDiagnosticNative structNative
        ) {
            WordPointer messageTemplateNative = structNative.address_message_template();StringWrapper.unwrap(this.messageTemplate, messageTemplateNative);
            WordPointer argsNative = structNative.address_args();LktNodeArrayWrapper.unwrap(this.args, argsNative);
            Pointer locationNative = structNative.address_location();locationNative.writeWord(0, this.location.entity.node.ni());
            WordPointer contextsNative = structNative.address_contexts();LogicContextArrayWrapper.unwrap(this.contexts, contextsNative);
            CIntPointer roundNative = structNative.address_round();roundNative.write(this.round);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(SolverDiagnosticNative structNative) {
            NI_LIB.lkt_internal_solver_diagnostic_dec_ref(structNative);
        }

    }

        
    
    

    /**
     * A pair returned by the ``Solve_With_Diagnostic`` primitive, consisting
     * of:
     *
     * * A ``Success`` field indicating whether resolution was successful or
     *   not.
     *
     * * A ``Diagnostics`` field containing an array of diagnostics which may
     *   be non-empty if ``Success`` is ``False``.
     */
    public static final class SolverResult
    {


        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final SolverResult NONE = new SolverResult(
            false,SolverDiagnosticArrayWrapper.NONE
        );

        // ----- Instance attributes -----

        public final
        boolean
        success;
        public final
        SolverDiagnostic[]
        diagnostics;


        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        SolverResult(
            final boolean success,final SolverDiagnostic[] diagnostics
        ) {
            this.success = success;
            this.diagnostics = diagnostics;
        }

        /**
         * Create a new structure with the field values.
         */
        public static SolverResult create(
            final boolean success,final SolverDiagnostic[] diagnostics
        ) {
            return new SolverResult(
                success,diagnostics
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static SolverResult wrap(
            final WordPointer pointer
        ) {
            return wrap((SolverResultNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static SolverResult wrap(
            final SolverResultNative structNative
        ) {
            return new SolverResult(
                BooleanWrapper.wrap(structNative.get_success()),SolverDiagnosticArrayWrapper.wrap(structNative.get_diagnostics())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final SolverResultNative structNative
        ) {
            CCharPointer successNative = structNative.address_success();successNative.write(this.success ? (byte) 1 : (byte) 0);
            WordPointer diagnosticsNative = structNative.address_diagnostics();SolverDiagnosticArrayWrapper.unwrap(this.diagnostics, diagnosticsNative);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(SolverResultNative structNative) {
            NI_LIB.lkt_internal_solver_result_dec_ref(structNative);
        }

    }


    // ===== Generated array utility classes =====

    
    

    /**
     * This class represents the lkt_internal_complete_item_array Java wrapping class
     */
    public static final class CompleteItemArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final CompleteItem[] NONE = new CompleteItem[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static CompleteItem[] wrap(
            final WordPointer pointer
        ) {
            return wrap((CompleteItemArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static CompleteItem[] wrap(
            final CompleteItemArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.getASize();
            final CompleteItem[] content = new CompleteItem[size];
            final Pointer nativeItems = nativeArray.addressOfItems();
            Pointer nativeItem;
            CompleteItemNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(CompleteItemNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = CompleteItem.wrap(toRead);
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            CompleteItem[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final CompleteItemArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static CompleteItemArrayNative unwrap(
            CompleteItem[] source
            
        ) {
            // Create a new native array with the size
            final CompleteItemArrayNative res = NI_LIB.lkt_internal_complete_item_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.addressOfItems();
            Pointer nativeItem;
            CompleteItemNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(CompleteItemNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((CompleteItemArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final CompleteItemArrayNative arrayNative
        ) {
            NI_LIB.lkt_internal_complete_item_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static CompleteItem[] jniWrap(
            final CompleteItem[] jniContent
        ) {
            final CompleteItem[] content =
                new CompleteItem[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static CompleteItem[] jniUnwrap(CompleteItem[] content) {
            final CompleteItem[] res =
                new CompleteItem[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the lkt_node_array Java wrapping class
     */
    public static final class LktNodeArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final LktNode[] NONE = new LktNode[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static LktNode[] wrap(
            final WordPointer pointer
        ) {
            return wrap((LktNodeArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static LktNode[] wrap(
            final LktNodeArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.getASize();
            final LktNode[] content = new LktNode[size];
            final Pointer nativeItems = nativeArray.addressOfItems();
            Pointer nativeItem;
            EntityNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = LktNode.fromEntity(Entity.wrap(toRead));
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            LktNode[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static LktNodeArrayNative unwrap(
            LktNode[] source
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative res = NI_LIB.lkt_node_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.addressOfItems();
            Pointer nativeItem;
            EntityNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((LktNodeArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final LktNodeArrayNative arrayNative
        ) {
            NI_LIB.lkt_node_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static LktNode[] jniWrap(
            final Entity[] jniContent
        ) {
            final LktNode[] content =
                new LktNode[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    LktNode.fromEntity(jniContent[i]);
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static Entity[] jniUnwrap(LktNode[] content) {
            final Entity[] res =
                new Entity[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = (content[i] != null ? content[i].entity : null);
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the lkt_node_array Java wrapping class
     */
    public static final class DefIdArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final DefId[] NONE = new DefId[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static DefId[] wrap(
            final WordPointer pointer
        ) {
            return wrap((LktNodeArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static DefId[] wrap(
            final LktNodeArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.getASize();
            final DefId[] content = new DefId[size];
            final Pointer nativeItems = nativeArray.addressOfItems();
            Pointer nativeItem;
            EntityNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = DefId.fromEntity(Entity.wrap(toRead));
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            DefId[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static LktNodeArrayNative unwrap(
            DefId[] source
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative res = NI_LIB.lkt_node_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.addressOfItems();
            Pointer nativeItem;
            EntityNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((LktNodeArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final LktNodeArrayNative arrayNative
        ) {
            NI_LIB.lkt_node_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static DefId[] jniWrap(
            final Entity[] jniContent
        ) {
            final DefId[] content =
                new DefId[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    DefId.fromEntity(jniContent[i]);
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static Entity[] jniUnwrap(DefId[] content) {
            final Entity[] res =
                new Entity[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = (content[i] != null ? content[i].entity : null);
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the lkt_node_array Java wrapping class
     */
    public static final class FunDeclArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final FunDecl[] NONE = new FunDecl[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static FunDecl[] wrap(
            final WordPointer pointer
        ) {
            return wrap((LktNodeArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static FunDecl[] wrap(
            final LktNodeArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.getASize();
            final FunDecl[] content = new FunDecl[size];
            final Pointer nativeItems = nativeArray.addressOfItems();
            Pointer nativeItem;
            EntityNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = FunDecl.fromEntity(Entity.wrap(toRead));
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            FunDecl[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static LktNodeArrayNative unwrap(
            FunDecl[] source
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative res = NI_LIB.lkt_node_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.addressOfItems();
            Pointer nativeItem;
            EntityNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((LktNodeArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final LktNodeArrayNative arrayNative
        ) {
            NI_LIB.lkt_node_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static FunDecl[] jniWrap(
            final Entity[] jniContent
        ) {
            final FunDecl[] content =
                new FunDecl[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    FunDecl.fromEntity(jniContent[i]);
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static Entity[] jniUnwrap(FunDecl[] content) {
            final Entity[] res =
                new Entity[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = (content[i] != null ? content[i].entity : null);
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the lkt_internal_logic_context_array Java wrapping class
     */
    public static final class LogicContextArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final LogicContext[] NONE = new LogicContext[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static LogicContext[] wrap(
            final WordPointer pointer
        ) {
            return wrap((LogicContextArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static LogicContext[] wrap(
            final LogicContextArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.getASize();
            final LogicContext[] content = new LogicContext[size];
            final Pointer nativeItems = nativeArray.addressOfItems();
            Pointer nativeItem;
            LogicContextNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(LogicContextNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = LogicContext.wrap(toRead);
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            LogicContext[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final LogicContextArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static LogicContextArrayNative unwrap(
            LogicContext[] source
            
        ) {
            // Create a new native array with the size
            final LogicContextArrayNative res = NI_LIB.lkt_internal_logic_context_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.addressOfItems();
            Pointer nativeItem;
            LogicContextNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(LogicContextNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((LogicContextArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final LogicContextArrayNative arrayNative
        ) {
            NI_LIB.lkt_internal_logic_context_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static LogicContext[] jniWrap(
            final LogicContext[] jniContent
        ) {
            final LogicContext[] content =
                new LogicContext[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static LogicContext[] jniUnwrap(LogicContext[] content) {
            final LogicContext[] res =
                new LogicContext[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the lkt_internal_ref_result_array Java wrapping class
     */
    public static final class RefResultArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final RefResult[] NONE = new RefResult[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static RefResult[] wrap(
            final WordPointer pointer
        ) {
            return wrap((RefResultArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static RefResult[] wrap(
            final RefResultArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.getASize();
            final RefResult[] content = new RefResult[size];
            final Pointer nativeItems = nativeArray.addressOfItems();
            Pointer nativeItem;
            RefResultNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(RefResultNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = RefResult.wrap(toRead);
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            RefResult[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final RefResultArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static RefResultArrayNative unwrap(
            RefResult[] source
            
        ) {
            // Create a new native array with the size
            final RefResultArrayNative res = NI_LIB.lkt_internal_ref_result_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.addressOfItems();
            Pointer nativeItem;
            RefResultNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(RefResultNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((RefResultArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final RefResultArrayNative arrayNative
        ) {
            NI_LIB.lkt_internal_ref_result_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static RefResult[] jniWrap(
            final RefResult[] jniContent
        ) {
            final RefResult[] content =
                new RefResult[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static RefResult[] jniUnwrap(RefResult[] content) {
            final RefResult[] res =
                new RefResult[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the lkt_internal_solver_diagnostic_array Java wrapping class
     */
    public static final class SolverDiagnosticArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final SolverDiagnostic[] NONE = new SolverDiagnostic[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static SolverDiagnostic[] wrap(
            final WordPointer pointer
        ) {
            return wrap((SolverDiagnosticArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static SolverDiagnostic[] wrap(
            final SolverDiagnosticArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.getASize();
            final SolverDiagnostic[] content = new SolverDiagnostic[size];
            final Pointer nativeItems = nativeArray.addressOfItems();
            Pointer nativeItem;
            SolverDiagnosticNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(SolverDiagnosticNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = SolverDiagnostic.wrap(toRead);
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            SolverDiagnostic[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final SolverDiagnosticArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static SolverDiagnosticArrayNative unwrap(
            SolverDiagnostic[] source
            
        ) {
            // Create a new native array with the size
            final SolverDiagnosticArrayNative res = NI_LIB.lkt_internal_solver_diagnostic_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.addressOfItems();
            Pointer nativeItem;
            SolverDiagnosticNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(SolverDiagnosticNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((SolverDiagnosticArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final SolverDiagnosticArrayNative arrayNative
        ) {
            NI_LIB.lkt_internal_solver_diagnostic_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static SolverDiagnostic[] jniWrap(
            final SolverDiagnostic[] jniContent
        ) {
            final SolverDiagnostic[] content =
                new SolverDiagnostic[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static SolverDiagnostic[] jniUnwrap(SolverDiagnostic[] content) {
            final SolverDiagnostic[] res =
                new SolverDiagnostic[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = content[i];
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the lkt_analysis_unit_array Java wrapping class
     */
    public static final class AnalysisUnitArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final AnalysisUnit[] NONE = new AnalysisUnit[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static AnalysisUnit[] wrap(
            final WordPointer pointer
        ) {
            return wrap((AnalysisUnitArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static AnalysisUnit[] wrap(
            final AnalysisUnitArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.getASize();
            final AnalysisUnit[] content = new AnalysisUnit[size];
            final Pointer nativeItems = nativeArray.addressOfItems();
            Pointer nativeItem;
            WordPointer toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(WordPointer.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = AnalysisUnit.wrap(toRead);
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            AnalysisUnit[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final AnalysisUnitArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static AnalysisUnitArrayNative unwrap(
            AnalysisUnit[] source
            
        ) {
            // Create a new native array with the size
            final AnalysisUnitArrayNative res = NI_LIB.lkt_analysis_unit_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.addressOfItems();
            Pointer nativeItem;
            WordPointer toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(WordPointer.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                toWrite.write(source[i].unwrap());
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((AnalysisUnitArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final AnalysisUnitArrayNative arrayNative
        ) {
            NI_LIB.lkt_analysis_unit_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static AnalysisUnit[] jniWrap(
            final AnalysisUnit[] jniContent
        ) {
            final AnalysisUnit[] content =
                new AnalysisUnit[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static AnalysisUnit[] jniUnwrap(AnalysisUnit[] content) {
            final AnalysisUnit[] res =
                new AnalysisUnit[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = content[i];
            }
            return res;
        }

    }


    // ===== Generated iterator wrapping classes =====


    // ===== Node classes =====

    

    /**
     * Data type for all nodes. Nodes are assembled to make up a tree.  See the
     * node primitives below to inspect such trees.
     *
     * Unlike for contexts and units, this type has weak-reference semantics:
     * keeping a reference to a node has no effect on the decision to keep the
     * unit that it owns allocated. This means that once all references to the
     * context and units related to a node are dropped, the context and its
     * units are deallocated and the node becomes a stale reference: most
     * operations on it will raise a ``Stale_Reference_Error``.
     *
     * Note that since reparsing an analysis unit deallocates all the nodes it
     * contains, this operation makes all reference to these nodes stale as
     * well.
     */
    public static abstract
    class LktNode
    implements LangkitSupport.LspNodeInterface, LangkitSupport.NodeInterface
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                LktNode.class,
                "LktNode",
                new String[] {
                    
                },
                new HashMap<>(
                    
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "parent",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "parent",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_PARENT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "parents",
                        new Class[]{boolean.class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        boolean.class,
                        "withSelf",
                        true
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "parents",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_PARENTS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "children",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "children",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CHILDREN
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "tokenStart",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "token_start",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TOKEN_START
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "tokenEnd",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "token_end",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TOKEN_END
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "childIndex",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "child_index",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CHILD_INDEX
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "previousSibling",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "previous_sibling",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_PREVIOUS_SIBLING
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "nextSibling",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "next_sibling",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_NEXT_SIBLING
                        )
                    );
                }
                
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "isGhost",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "is_ghost",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_IS_GHOST
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "fullSlocImage",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "full_sloc_image",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FULL_SLOC_IMAGE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "completionItemKindToInt",
                        new Class[]{CompletionItemKind.class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        CompletionItemKind.class,
                        "kind"
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "completion_item_kind_to_int",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_COMPLETION_ITEM_KIND_TO_INT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pSetSolverDebugMode",
                        new Class[]{boolean.class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        boolean.class,
                        "enable"
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_set_solver_debug_mode",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_SET_SOLVER_DEBUG_MODE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pBasicTraitGen",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_basic_trait_gen",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_BASIC_TRAIT_GEN
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pBasicTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_basic_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_BASIC_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pNodeGenTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_node_gen_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_NODE_GEN_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pNodeTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_node_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_NODE_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pIndexableGenTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_indexable_gen_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_INDEXABLE_GEN_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pIndexableTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_indexable_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_INDEXABLE_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pTokenNodeTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_token_node_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_TOKEN_NODE_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pErrorNodeTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_error_node_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ERROR_NODE_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pCharType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_char_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_CHAR_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pIntType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_int_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_INT_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pBoolType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_bool_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_BOOL_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pBigintType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_bigint_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_BIGINT_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pStringType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_string_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_STRING_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pSymbolType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_symbol_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_SYMBOL_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pPropertyErrorType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_property_error_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_PROPERTY_ERROR_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pRegexpType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_regexp_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_REGEXP_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pEntityGenType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_entity_gen_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ENTITY_GEN_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pEntityType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_entity_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ENTITY_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pLogicvarType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_logicvar_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_LOGICVAR_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pEquationType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_equation_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_EQUATION_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pArrayGenType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_array_gen_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ARRAY_GEN_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pArrayType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_array_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ARRAY_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pAstlistGenType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_astlist_gen_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ASTLIST_GEN_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pAstlistType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_astlist_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ASTLIST_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pNodeBuilderGenType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_node_builder_gen_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_NODE_BUILDER_GEN_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pNodeBuilderType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_node_builder_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_NODE_BUILDER_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pIteratorGenTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_iterator_gen_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ITERATOR_GEN_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pIteratorTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_iterator_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ITERATOR_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pAnalysisUnitGenTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_analysis_unit_gen_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ANALYSIS_UNIT_GEN_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pAnalysisUnitTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_analysis_unit_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ANALYSIS_UNIT_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pTopmostInvalidDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_topmost_invalid_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_TOPMOST_INVALID_DECL
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pNameresDiagnostics",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_nameres_diagnostics",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_NAMERES_DIAGNOSTICS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pSolveEnclosingContext",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_solve_enclosing_context",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_SOLVE_ENCLOSING_CONTEXT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pXrefEntryPoint",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_xref_entry_point",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_XREF_ENTRY_POINT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pComplete",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_complete",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_COMPLETE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LktNode NONE = new NoneNode();

        /** The entity of the node. */
        public final Entity entity;

        /** Cache for the associated rewriting node. */
        protected RewritingNode rewritingNode;

        /** The analysis unit that owns the node. */
        protected AnalysisUnit unit;

        /** The cache for the image of the node. */
        protected String image;

        // ----- Constructors -----

        /**
         * Create a new node with its entity.
         *
         * @param entity The node's entity.
         */
        protected LktNode(
            final Entity entity
        ) {
            this.entity = entity;
            this.unit = null;
            this.image = null;
        }

        /**
         * Get a node from the given entity.
         *
         * @param entity The entity to get the node from.
         * @return The newly created node.
         */
        public static LktNode fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LktNode.NONE :
                dispatchNodeCreation(entity);
        }

        /**
         * Dispatch the node creation to return the valid Java object
         * according to the node kind.
         *
         * @param entity The entity to create the node from.
         * @return The wrapped node in the correct class.
         */
        protected static LktNode dispatchNodeCreation(
            final Entity entity
        ) {
            int nodeKind = -1;

            if(ImageInfo.inImageCode()) {
                EntityNative entityNative = StackValue.get(
                    EntityNative.class
                );
                entity.unwrap(entityNative);
                nodeKind = NI_LIB.lkt_node_kind(entityNative);
            } else {
                nodeKind = JNI_LIB.lkt_node_kind(entity);
            }

            switch(nodeKind) {
                case 1:
                    return entity.node.isNull() ?
                        Argument.NONE :
                        new Argument(entity);
                case 2:
                    return entity.node.isNull() ?
                        LexerCaseRuleCondAlt.NONE :
                        new LexerCaseRuleCondAlt(entity);
                case 3:
                    return entity.node.isNull() ?
                        LexerCaseRuleDefaultAlt.NONE :
                        new LexerCaseRuleDefaultAlt(entity);
                case 4:
                    return entity.node.isNull() ?
                        MatchBranch.NONE :
                        new MatchBranch(entity);
                case 5:
                    return entity.node.isNull() ?
                        PatternMatchBranch.NONE :
                        new PatternMatchBranch(entity);
                case 6:
                    return entity.node.isNull() ?
                        BlockExprClause.NONE :
                        new BlockExprClause(entity);
                case 7:
                    return entity.node.isNull() ?
                        BlockStringLine.NONE :
                        new BlockStringLine(entity);
                case 8:
                    return entity.node.isNull() ?
                        ClassQualifierAbsent.NONE :
                        new ClassQualifierAbsent(entity);
                case 9:
                    return entity.node.isNull() ?
                        ClassQualifierPresent.NONE :
                        new ClassQualifierPresent(entity);
                case 10:
                    return entity.node.isNull() ?
                        GrammarRuleDecl.NONE :
                        new GrammarRuleDecl(entity);
                case 11:
                    return entity.node.isNull() ?
                        SyntheticLexerDecl.NONE :
                        new SyntheticLexerDecl(entity);
                case 12:
                    return entity.node.isNull() ?
                        NodeDecl.NONE :
                        new NodeDecl(entity);
                case 13:
                    return entity.node.isNull() ?
                        SelfDecl.NONE :
                        new SelfDecl(entity);
                case 14:
                    return entity.node.isNull() ?
                        BindingValDecl.NONE :
                        new BindingValDecl(entity);
                case 15:
                    return entity.node.isNull() ?
                        EnumLitDecl.NONE :
                        new EnumLitDecl(entity);
                case 16:
                    return entity.node.isNull() ?
                        FieldDecl.NONE :
                        new FieldDecl(entity);
                case 17:
                    return entity.node.isNull() ?
                        FunParamDecl.NONE :
                        new FunParamDecl(entity);
                case 18:
                    return entity.node.isNull() ?
                        LambdaParamDecl.NONE :
                        new LambdaParamDecl(entity);
                case 19:
                    return entity.node.isNull() ?
                        DynVarDecl.NONE :
                        new DynVarDecl(entity);
                case 20:
                    return entity.node.isNull() ?
                        MatchValDecl.NONE :
                        new MatchValDecl(entity);
                case 21:
                    return entity.node.isNull() ?
                        ValDecl.NONE :
                        new ValDecl(entity);
                case 22:
                    return entity.node.isNull() ?
                        FunDecl.NONE :
                        new FunDecl(entity);
                case 23:
                    return entity.node.isNull() ?
                        EnvSpecDecl.NONE :
                        new EnvSpecDecl(entity);
                case 24:
                    return entity.node.isNull() ?
                        ErrorDecl.NONE :
                        new ErrorDecl(entity);
                case 25:
                    return entity.node.isNull() ?
                        GenericDecl.NONE :
                        new GenericDecl(entity);
                case 26:
                    return entity.node.isNull() ?
                        GrammarDecl.NONE :
                        new GrammarDecl(entity);
                case 27:
                    return entity.node.isNull() ?
                        LexerDecl.NONE :
                        new LexerDecl(entity);
                case 28:
                    return entity.node.isNull() ?
                        LexerFamilyDecl.NONE :
                        new LexerFamilyDecl(entity);
                case 29:
                    return entity.node.isNull() ?
                        SynthFunDecl.NONE :
                        new SynthFunDecl(entity);
                case 30:
                    return entity.node.isNull() ?
                        SynthParamDecl.NONE :
                        new SynthParamDecl(entity);
                case 31:
                    return entity.node.isNull() ?
                        AnyTypeDecl.NONE :
                        new AnyTypeDecl(entity);
                case 32:
                    return entity.node.isNull() ?
                        EnumClassAltDecl.NONE :
                        new EnumClassAltDecl(entity);
                case 33:
                    return entity.node.isNull() ?
                        FunctionType.NONE :
                        new FunctionType(entity);
                case 34:
                    return entity.node.isNull() ?
                        GenericParamTypeDecl.NONE :
                        new GenericParamTypeDecl(entity);
                case 35:
                    return entity.node.isNull() ?
                        ClassDecl.NONE :
                        new ClassDecl(entity);
                case 36:
                    return entity.node.isNull() ?
                        EnumClassDecl.NONE :
                        new EnumClassDecl(entity);
                case 37:
                    return entity.node.isNull() ?
                        EnumTypeDecl.NONE :
                        new EnumTypeDecl(entity);
                case 38:
                    return entity.node.isNull() ?
                        StructDecl.NONE :
                        new StructDecl(entity);
                case 39:
                    return entity.node.isNull() ?
                        TraitDecl.NONE :
                        new TraitDecl(entity);
                case 40:
                    return entity.node.isNull() ?
                        DeclAnnotation.NONE :
                        new DeclAnnotation(entity);
                case 41:
                    return entity.node.isNull() ?
                        DeclAnnotationArgs.NONE :
                        new DeclAnnotationArgs(entity);
                case 42:
                    return entity.node.isNull() ?
                        DynEnvWrapper.NONE :
                        new DynEnvWrapper(entity);
                case 43:
                    return entity.node.isNull() ?
                        ElsifBranch.NONE :
                        new ElsifBranch(entity);
                case 44:
                    return entity.node.isNull() ?
                        EnumClassCase.NONE :
                        new EnumClassCase(entity);
                case 45:
                    return entity.node.isNull() ?
                        ExcludesNullAbsent.NONE :
                        new ExcludesNullAbsent(entity);
                case 46:
                    return entity.node.isNull() ?
                        ExcludesNullPresent.NONE :
                        new ExcludesNullPresent(entity);
                case 47:
                    return entity.node.isNull() ?
                        AnyOf.NONE :
                        new AnyOf(entity);
                case 48:
                    return entity.node.isNull() ?
                        ArrayLiteral.NONE :
                        new ArrayLiteral(entity);
                case 49:
                    return entity.node.isNull() ?
                        CallExpr.NONE :
                        new CallExpr(entity);
                case 50:
                    return entity.node.isNull() ?
                        LogicPredicate.NONE :
                        new LogicPredicate(entity);
                case 51:
                    return entity.node.isNull() ?
                        LogicPropagateCall.NONE :
                        new LogicPropagateCall(entity);
                case 52:
                    return entity.node.isNull() ?
                        BinOp.NONE :
                        new BinOp(entity);
                case 53:
                    return entity.node.isNull() ?
                        BlockExpr.NONE :
                        new BlockExpr(entity);
                case 54:
                    return entity.node.isNull() ?
                        CastExpr.NONE :
                        new CastExpr(entity);
                case 55:
                    return entity.node.isNull() ?
                        DotExpr.NONE :
                        new DotExpr(entity);
                case 56:
                    return entity.node.isNull() ?
                        ErrorOnNull.NONE :
                        new ErrorOnNull(entity);
                case 57:
                    return entity.node.isNull() ?
                        GenericInstantiation.NONE :
                        new GenericInstantiation(entity);
                case 58:
                    return entity.node.isNull() ?
                        GrammarCut.NONE :
                        new GrammarCut(entity);
                case 59:
                    return entity.node.isNull() ?
                        GrammarDiscard.NONE :
                        new GrammarDiscard(entity);
                case 60:
                    return entity.node.isNull() ?
                        GrammarDontSkip.NONE :
                        new GrammarDontSkip(entity);
                case 61:
                    return entity.node.isNull() ?
                        GrammarList.NONE :
                        new GrammarList(entity);
                case 62:
                    return entity.node.isNull() ?
                        GrammarNull.NONE :
                        new GrammarNull(entity);
                case 63:
                    return entity.node.isNull() ?
                        GrammarOpt.NONE :
                        new GrammarOpt(entity);
                case 64:
                    return entity.node.isNull() ?
                        GrammarOptError.NONE :
                        new GrammarOptError(entity);
                case 65:
                    return entity.node.isNull() ?
                        GrammarOptErrorGroup.NONE :
                        new GrammarOptErrorGroup(entity);
                case 66:
                    return entity.node.isNull() ?
                        GrammarOptGroup.NONE :
                        new GrammarOptGroup(entity);
                case 67:
                    return entity.node.isNull() ?
                        GrammarOrExpr.NONE :
                        new GrammarOrExpr(entity);
                case 68:
                    return entity.node.isNull() ?
                        GrammarPick.NONE :
                        new GrammarPick(entity);
                case 69:
                    return entity.node.isNull() ?
                        GrammarImplicitPick.NONE :
                        new GrammarImplicitPick(entity);
                case 70:
                    return entity.node.isNull() ?
                        GrammarPredicate.NONE :
                        new GrammarPredicate(entity);
                case 71:
                    return entity.node.isNull() ?
                        GrammarRuleRef.NONE :
                        new GrammarRuleRef(entity);
                case 72:
                    return entity.node.isNull() ?
                        GrammarSkip.NONE :
                        new GrammarSkip(entity);
                case 73:
                    return entity.node.isNull() ?
                        GrammarStopCut.NONE :
                        new GrammarStopCut(entity);
                case 74:
                    return entity.node.isNull() ?
                        ParseNodeExpr.NONE :
                        new ParseNodeExpr(entity);
                case 75:
                    return entity.node.isNull() ?
                        TokenLit.NONE :
                        new TokenLit(entity);
                case 76:
                    return entity.node.isNull() ?
                        TokenNoCaseLit.NONE :
                        new TokenNoCaseLit(entity);
                case 77:
                    return entity.node.isNull() ?
                        TokenPatternConcat.NONE :
                        new TokenPatternConcat(entity);
                case 78:
                    return entity.node.isNull() ?
                        TokenPatternLit.NONE :
                        new TokenPatternLit(entity);
                case 79:
                    return entity.node.isNull() ?
                        TokenRef.NONE :
                        new TokenRef(entity);
                case 80:
                    return entity.node.isNull() ?
                        Id.NONE :
                        new Id(entity);
                case 81:
                    return entity.node.isNull() ?
                        DefId.NONE :
                        new DefId(entity);
                case 82:
                    return entity.node.isNull() ?
                        ModuleRefId.NONE :
                        new ModuleRefId(entity);
                case 83:
                    return entity.node.isNull() ?
                        RefId.NONE :
                        new RefId(entity);
                case 84:
                    return entity.node.isNull() ?
                        IfExpr.NONE :
                        new IfExpr(entity);
                case 85:
                    return entity.node.isNull() ?
                        Isa.NONE :
                        new Isa(entity);
                case 86:
                    return entity.node.isNull() ?
                        KeepExpr.NONE :
                        new KeepExpr(entity);
                case 87:
                    return entity.node.isNull() ?
                        LambdaExpr.NONE :
                        new LambdaExpr(entity);
                case 88:
                    return entity.node.isNull() ?
                        BigNumLit.NONE :
                        new BigNumLit(entity);
                case 89:
                    return entity.node.isNull() ?
                        CharLit.NONE :
                        new CharLit(entity);
                case 90:
                    return entity.node.isNull() ?
                        NullLit.NONE :
                        new NullLit(entity);
                case 91:
                    return entity.node.isNull() ?
                        NumLit.NONE :
                        new NumLit(entity);
                case 92:
                    return entity.node.isNull() ?
                        BlockStringLit.NONE :
                        new BlockStringLit(entity);
                case 93:
                    return entity.node.isNull() ?
                        SingleLineStringLit.NONE :
                        new SingleLineStringLit(entity);
                case 94:
                    return entity.node.isNull() ?
                        PatternSingleLineStringLit.NONE :
                        new PatternSingleLineStringLit(entity);
                case 95:
                    return entity.node.isNull() ?
                        LogicAssign.NONE :
                        new LogicAssign(entity);
                case 96:
                    return entity.node.isNull() ?
                        LogicExpr.NONE :
                        new LogicExpr(entity);
                case 97:
                    return entity.node.isNull() ?
                        LogicPropagate.NONE :
                        new LogicPropagate(entity);
                case 98:
                    return entity.node.isNull() ?
                        LogicUnify.NONE :
                        new LogicUnify(entity);
                case 99:
                    return entity.node.isNull() ?
                        MatchExpr.NONE :
                        new MatchExpr(entity);
                case 100:
                    return entity.node.isNull() ?
                        NotExpr.NONE :
                        new NotExpr(entity);
                case 101:
                    return entity.node.isNull() ?
                        ParenExpr.NONE :
                        new ParenExpr(entity);
                case 102:
                    return entity.node.isNull() ?
                        RaiseExpr.NONE :
                        new RaiseExpr(entity);
                case 103:
                    return entity.node.isNull() ?
                        SubscriptExpr.NONE :
                        new SubscriptExpr(entity);
                case 104:
                    return entity.node.isNull() ?
                        TryExpr.NONE :
                        new TryExpr(entity);
                case 105:
                    return entity.node.isNull() ?
                        UnOp.NONE :
                        new UnOp(entity);
                case 106:
                    return entity.node.isNull() ?
                        FullDecl.NONE :
                        new FullDecl(entity);
                case 107:
                    return entity.node.isNull() ?
                        GrammarListSep.NONE :
                        new GrammarListSep(entity);
                case 108:
                    return entity.node.isNull() ?
                        Import.NONE :
                        new Import(entity);
                case 109:
                    return entity.node.isNull() ?
                        LangkitRoot.NONE :
                        new LangkitRoot(entity);
                case 110:
                    return entity.node.isNull() ?
                        LexerCaseRule.NONE :
                        new LexerCaseRule(entity);
                case 111:
                    return entity.node.isNull() ?
                        LexerCaseRuleSend.NONE :
                        new LexerCaseRuleSend(entity);
                case 112:
                    return entity.node.isNull() ?
                        ListKindOne.NONE :
                        new ListKindOne(entity);
                case 113:
                    return entity.node.isNull() ?
                        ListKindZero.NONE :
                        new ListKindZero(entity);
                case 114:
                    return entity.node.isNull() ?
                        ArgumentList.NONE :
                        new ArgumentList(entity);
                case 115:
                    return entity.node.isNull() ?
                        BaseLexerCaseRuleAltList.NONE :
                        new BaseLexerCaseRuleAltList(entity);
                case 116:
                    return entity.node.isNull() ?
                        BaseMatchBranchList.NONE :
                        new BaseMatchBranchList(entity);
                case 117:
                    return entity.node.isNull() ?
                        BlockStringLineList.NONE :
                        new BlockStringLineList(entity);
                case 118:
                    return entity.node.isNull() ?
                        CallExprList.NONE :
                        new CallExprList(entity);
                case 119:
                    return entity.node.isNull() ?
                        DeclAnnotationList.NONE :
                        new DeclAnnotationList(entity);
                case 120:
                    return entity.node.isNull() ?
                        ElsifBranchList.NONE :
                        new ElsifBranchList(entity);
                case 121:
                    return entity.node.isNull() ?
                        EnumClassAltDeclList.NONE :
                        new EnumClassAltDeclList(entity);
                case 122:
                    return entity.node.isNull() ?
                        EnumClassCaseList.NONE :
                        new EnumClassCaseList(entity);
                case 123:
                    return entity.node.isNull() ?
                        EnumLitDeclList.NONE :
                        new EnumLitDeclList(entity);
                case 124:
                    return entity.node.isNull() ?
                        ExprList.NONE :
                        new ExprList(entity);
                case 125:
                    return entity.node.isNull() ?
                        AnyOfList.NONE :
                        new AnyOfList(entity);
                case 126:
                    return entity.node.isNull() ?
                        FullDeclList.NONE :
                        new FullDeclList(entity);
                case 127:
                    return entity.node.isNull() ?
                        DeclBlock.NONE :
                        new DeclBlock(entity);
                case 128:
                    return entity.node.isNull() ?
                        GenericParamDeclList.NONE :
                        new GenericParamDeclList(entity);
                case 129:
                    return entity.node.isNull() ?
                        FunParamDeclList.NONE :
                        new FunParamDeclList(entity);
                case 130:
                    return entity.node.isNull() ?
                        GrammarExprList.NONE :
                        new GrammarExprList(entity);
                case 131:
                    return entity.node.isNull() ?
                        GrammarExprListList.NONE :
                        new GrammarExprListList(entity);
                case 132:
                    return entity.node.isNull() ?
                        ImportList.NONE :
                        new ImportList(entity);
                case 133:
                    return entity.node.isNull() ?
                        LambdaParamDeclList.NONE :
                        new LambdaParamDeclList(entity);
                case 134:
                    return entity.node.isNull() ?
                        LktNodeList.NONE :
                        new LktNodeList(entity);
                case 135:
                    return entity.node.isNull() ?
                        PatternDetailList.NONE :
                        new PatternDetailList(entity);
                case 136:
                    return entity.node.isNull() ?
                        PatternList.NONE :
                        new PatternList(entity);
                case 137:
                    return entity.node.isNull() ?
                        RefIdList.NONE :
                        new RefIdList(entity);
                case 138:
                    return entity.node.isNull() ?
                        TypeRefList.NONE :
                        new TypeRefList(entity);
                case 139:
                    return entity.node.isNull() ?
                        SyntheticTypeRefList.NONE :
                        new SyntheticTypeRefList(entity);
                case 140:
                    return entity.node.isNull() ?
                        NullCondQualifierAbsent.NONE :
                        new NullCondQualifierAbsent(entity);
                case 141:
                    return entity.node.isNull() ?
                        NullCondQualifierPresent.NONE :
                        new NullCondQualifierPresent(entity);
                case 142:
                    return entity.node.isNull() ?
                        OpAmp.NONE :
                        new OpAmp(entity);
                case 143:
                    return entity.node.isNull() ?
                        OpAnd.NONE :
                        new OpAnd(entity);
                case 144:
                    return entity.node.isNull() ?
                        OpDiv.NONE :
                        new OpDiv(entity);
                case 145:
                    return entity.node.isNull() ?
                        OpEq.NONE :
                        new OpEq(entity);
                case 146:
                    return entity.node.isNull() ?
                        OpGt.NONE :
                        new OpGt(entity);
                case 147:
                    return entity.node.isNull() ?
                        OpGte.NONE :
                        new OpGte(entity);
                case 148:
                    return entity.node.isNull() ?
                        OpLogicAnd.NONE :
                        new OpLogicAnd(entity);
                case 149:
                    return entity.node.isNull() ?
                        OpLogicOr.NONE :
                        new OpLogicOr(entity);
                case 150:
                    return entity.node.isNull() ?
                        OpLt.NONE :
                        new OpLt(entity);
                case 151:
                    return entity.node.isNull() ?
                        OpLte.NONE :
                        new OpLte(entity);
                case 152:
                    return entity.node.isNull() ?
                        OpMinus.NONE :
                        new OpMinus(entity);
                case 153:
                    return entity.node.isNull() ?
                        OpMult.NONE :
                        new OpMult(entity);
                case 154:
                    return entity.node.isNull() ?
                        OpNe.NONE :
                        new OpNe(entity);
                case 155:
                    return entity.node.isNull() ?
                        OpOr.NONE :
                        new OpOr(entity);
                case 156:
                    return entity.node.isNull() ?
                        OpOrInt.NONE :
                        new OpOrInt(entity);
                case 157:
                    return entity.node.isNull() ?
                        OpPlus.NONE :
                        new OpPlus(entity);
                case 158:
                    return entity.node.isNull() ?
                        AnyTypePattern.NONE :
                        new AnyTypePattern(entity);
                case 159:
                    return entity.node.isNull() ?
                        BindingPattern.NONE :
                        new BindingPattern(entity);
                case 160:
                    return entity.node.isNull() ?
                        BoolPatternFalse.NONE :
                        new BoolPatternFalse(entity);
                case 161:
                    return entity.node.isNull() ?
                        BoolPatternTrue.NONE :
                        new BoolPatternTrue(entity);
                case 162:
                    return entity.node.isNull() ?
                        EllipsisPattern.NONE :
                        new EllipsisPattern(entity);
                case 163:
                    return entity.node.isNull() ?
                        ExtendedPattern.NONE :
                        new ExtendedPattern(entity);
                case 164:
                    return entity.node.isNull() ?
                        FilteredPattern.NONE :
                        new FilteredPattern(entity);
                case 165:
                    return entity.node.isNull() ?
                        IntegerPattern.NONE :
                        new IntegerPattern(entity);
                case 166:
                    return entity.node.isNull() ?
                        ListPattern.NONE :
                        new ListPattern(entity);
                case 167:
                    return entity.node.isNull() ?
                        NotPattern.NONE :
                        new NotPattern(entity);
                case 168:
                    return entity.node.isNull() ?
                        NullPattern.NONE :
                        new NullPattern(entity);
                case 169:
                    return entity.node.isNull() ?
                        OrPattern.NONE :
                        new OrPattern(entity);
                case 170:
                    return entity.node.isNull() ?
                        ParenPattern.NONE :
                        new ParenPattern(entity);
                case 171:
                    return entity.node.isNull() ?
                        RegexPattern.NONE :
                        new RegexPattern(entity);
                case 172:
                    return entity.node.isNull() ?
                        TuplePattern.NONE :
                        new TuplePattern(entity);
                case 173:
                    return entity.node.isNull() ?
                        TypePattern.NONE :
                        new TypePattern(entity);
                case 174:
                    return entity.node.isNull() ?
                        FieldPatternDetail.NONE :
                        new FieldPatternDetail(entity);
                case 175:
                    return entity.node.isNull() ?
                        PropertyPatternDetail.NONE :
                        new PropertyPatternDetail(entity);
                case 176:
                    return entity.node.isNull() ?
                        SelectorPatternDetail.NONE :
                        new SelectorPatternDetail(entity);
                case 177:
                    return entity.node.isNull() ?
                        SelectorCall.NONE :
                        new SelectorCall(entity);
                case 178:
                    return entity.node.isNull() ?
                        DefaultListTypeRef.NONE :
                        new DefaultListTypeRef(entity);
                case 179:
                    return entity.node.isNull() ?
                        FunctionTypeRef.NONE :
                        new FunctionTypeRef(entity);
                case 180:
                    return entity.node.isNull() ?
                        GenericTypeRef.NONE :
                        new GenericTypeRef(entity);
                case 181:
                    return entity.node.isNull() ?
                        SimpleTypeRef.NONE :
                        new SimpleTypeRef(entity);
                case 182:
                    return entity.node.isNull() ?
                        VarBind.NONE :
                        new VarBind(entity);
                default:
                    throw new EnumException(
                        "Cannot find the node type from " + nodeKind
                    );
            }
        }

        // ----- Graal C API methods -----

        /**
         * Util internal method to unwrap a node to a native entity struct
         */
        void unwrap(
            final EntityNative entityNative
        ) {
            this.entity.unwrap(entityNative);
        }

        // ----- Getters -----

        public Reflection.Node getDescription() {
            return LktNode.description;
        }

        public NodeKind getKind() {
            return this.getDescription().kind;
        }

        public String getClassName() {
            return this.getDescription().className;
        }

        public boolean isTokenNode() {
            return this.getDescription().isTokenNode;
        }

        public boolean isListNode() {
            return this.getDescription().isListNode;
        }

        public String[] getFieldNames() {
            return this.getDescription().fields;
        }

        public Map<String, Reflection.Field> getFieldDescriptions()
        {
            return this.getDescription().fieldDescriptions;
        }

        @CompilerDirectives.TruffleBoundary
        public Reflection.Field getFieldDescription(
            final String name
        ) {
            return this.getDescription()
                .fieldDescriptions
                .get(name);
        }

        public boolean isNone() {
            return this.entity.node.isNull();
        }

        // ----- Instance methods -----

        /**
         * Get the analysis unit of the node.
         *
         * @return The unit which owns the node.
         */
        public AnalysisUnit getUnit() {
            if(this.unit == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    this.entity.unwrap(entityNative);

                    final AnalysisUnitNative unitNative =
                        NI_LIB.lkt_node_unit(entityNative);
                    this.unit = AnalysisUnit.wrap(unitNative);
                } else {
                    this.unit = JNI_LIB.lkt_node_unit(this.entity);
                }

            }
            return this.unit;
        }

        /**
         * Get the children count of the node.
         *
         * @return The children count.
         */
        public int getChildrenCount() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                return NI_LIB.lkt_node_children_count(thisNative);
            } else {
                return JNI_LIB.lkt_node_children_count(this.entity);
            }

        }

        /**
         * Get the child at the given position.
         *
         * @param n The index of the child to get.
         * @return The child at the given index.
         */
        public LktNode getChild(
            final int n
        ) {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final EntityNative resNative = StackValue.get(
                    EntityNative.class
                );
                NI_LIB.lkt_node_child(
                    thisNative,
                    n,
                    resNative
                );

                return fromEntity(Entity.wrap(resNative));
            } else {
                return fromEntity(JNI_LIB.lkt_node_child(
                    this.entity,
                    n
                ));
            }

        }

        /**
         * Get the text of the node.
         *
         * @return The text of the node.
         */
        public String getText() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final TextNative resNative = StackValue.get(TextNative.class);
                Text.NONE.unwrap(resNative);
                NI_LIB.lkt_node_text(
                    thisNative,
                    resNative
                );

                try(final Text resText = Text.wrap(resNative)) {
                    return resText.getContent();
                }
            } else {
                try(
                    final Text resText = JNI_LIB.lkt_node_text(
                        this.entity
                    );
                ) {
                    return resText.getContent();
                }
            }

        }

        /**
         * Get the image of the node.
         *
         * @return The node's image.
         */
        public String getImage() {
            if(this.image == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative thisNative = StackValue.get(
                        EntityNative.class
                    );
                    this.entity.unwrap(thisNative);

                    final TextNative resNative = StackValue.get(
                        TextNative.class
                    );
                    Text.NONE.unwrap(resNative);
                    NI_LIB.lkt_node_image(thisNative, resNative);

                    try(final Text resText = Text.wrap(resNative)) {
                        this.image = resText.getContent();
                    }
                } else {
                    try(
                        final Text resText = JNI_LIB.lkt_node_image(
                            this.entity
                        )
                    ) {
                        this.image = resText.getContent();
                    }
                }

            }

            return this.image;
        }

        /**
         * Get the source location range of the node.
         *
         * @return The source location range of the node.
         */
        public SourceLocationRange getSourceLocationRange() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final SourceLocationRangeNative resNative = StackValue.get(
                    SourceLocationRangeNative.class
                );
                NI_LIB.lkt_node_sloc_range(
                    thisNative,
                    resNative
                );

                return SourceLocationRange.wrap(resNative);
            } else {
                return JNI_LIB.lkt_node_sloc_range(this.entity);
            }

        }

        /**
         * Return the rewriting handle corresponding to Node.
         *
         * The owning unit of Node must be free of diagnostics.
         */
        public RewritingNode getRewritingNode() {
            if(this.rewritingNode == null) {
                final RewritingNode tmp;

                if(ImageInfo.inImageCode()) {
                    tmp = RewritingNode.wrap(
                        NI_LIB.lkt_rewriting_node_to_handle(
                            (Pointer) this.entity.node.ni()
                        )
                    );
                } else {
                    tmp = JNI_LIB.lkt_rewriting_node_to_handle(
                        this.entity
                    );
                }
                checkException();
                this.rewritingNode = tmp;

            }
            return this.rewritingNode;
        }

        // ----- Dumping methods -----

        /**
         * Return the parsing tree in a string.
         *
         * @return The string containing the representation of the parsing tree
         * from the node.
         */
        @CompilerDirectives.TruffleBoundary
        public String dumpTree() {
            final StringBuilder builder = new StringBuilder();
            this.dumpTree(builder);
            return builder.toString();
        }

        /**
         * Dump the parse tree in the given string builder.
         *
         * @param builder The builder to dump the parse tree in.
         */
        @CompilerDirectives.TruffleBoundary
        public void dumpTree(
            final StringBuilder builder
        ) {
            this.dumpTree(builder, "");
        }

        /**
         * Dump a node field in the given string builder.
         *
         * @param builder The string builder to put the file in.
         * @param indent The current indentation string.
         * @param name The name of the field.
         * @param value The value of the field.
         */
        protected static void dumpField(
            final StringBuilder builder,
            final String indent,
            final String name,
            final LktNode value
        ) {
            builder.append(indent)
                .append(name)
                .append(":\n");
            value.dumpTree(builder, indent + "  ");
        }

        /**
         * Dump the parse tree in the given string builder with the indent
         * level.
         *
         * @param builder The builder to dump the tree in.
         * @param indent The starting indent level.
         */
        @CompilerDirectives.TruffleBoundary
        protected void dumpTree(
            final StringBuilder builder,
            String indent
        ) {
            // Prepare the working variables
            String image = this.getImage();
            image = image.substring(1, image.length());
            final int childrenCount = this.getChildrenCount();

            // Print the node
            builder.append(indent)
                .append(image);
            if(this.isTokenNode()) {
                builder.append(": ")
                    .append(this.getText());
            }
            builder.append('\n');

            // Print the field of the node
            indent = indent + "|";
            if(this.isListNode()) {
                for(int i = 0 ; i < childrenCount ; i++) {
                    final LktNode child = this.getChild(i);
                    dumpField(builder, indent, "item_" + i, child);
                }
            } else {
                for(int i = 0 ; i < childrenCount ; i++) {
                    final LktNode child = this.getChild(i);
                    final String name = this.getFieldNames()[i];
                    dumpField(builder, indent, name, child);
                }
            }
        }

        // ----- Visitor methods -----

        /**
         * Accept the given visitor.
         *
         * @param visitor The visitor to accept.
         * @return The result of the visit.
         */
        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        /**
         * Accept the given visitor.
         *
         * @param visitor The visitor to accept.
         * @param param The parameter of the visit.
         * @return The result of the visit.
         */
        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * Return the syntactic parent for this node. Return null for the root
         * node.
         */
        public LktNode parent(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_parent(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_parent(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }

        
    


        /**
         * Return an array that contains the lexical parents, this node
         * included iff ``with_self`` is True. Nearer parents are first in the
         * list.
         */
        public LktNode[] parents(
            final boolean withSelf
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                byte withSelfNative = (withSelf ? (byte) 1 : (byte) 0);

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_parents(
                    thisNative,
                    withSelfNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode[] res = LktNodeArrayWrapper.NONE;
                if(propException == null) {
                    res = LktNodeArrayWrapper.wrap(resNative);

                    LktNodeArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final LktNode[] res = JNI_LIB.lkt_lkt_node_parents(
                    withSelf,
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return an array that contains the direct lexical children.
         *
         * .. warning:: This constructs a whole array every-time you call it,
         *    and as such is less efficient than calling the ``Child`` built-
         *    in.
         */
        public LktNode[] children(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_children(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode[] res = LktNodeArrayWrapper.NONE;
                if(propException == null) {
                    res = LktNodeArrayWrapper.wrap(resNative);

                    LktNodeArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final LktNode[] res = JNI_LIB.lkt_lkt_node_children(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return the first token used to parse this node.
         */
        public Token tokenStart(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                // Get the node unit
                final AnalysisUnit currentUnit = this.getUnit();


                // Unwrap the arguments

                // Create the result native
                final TokenNative resNative =
                    StackValue.get(TokenNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_token_start(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Token res = Token.NONE(currentUnit);
                if(propException == null) {
                    res = Token.wrap(resNative, currentUnit);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Token res = JNI_LIB.lkt_lkt_node_token_start(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return the last token used to parse this node.
         */
        public Token tokenEnd(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                // Get the node unit
                final AnalysisUnit currentUnit = this.getUnit();


                // Unwrap the arguments

                // Create the result native
                final TokenNative resNative =
                    StackValue.get(TokenNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_token_end(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Token res = Token.NONE(currentUnit);
                if(propException == null) {
                    res = Token.wrap(resNative, currentUnit);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Token res = JNI_LIB.lkt_lkt_node_token_end(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return the 0-based index for Node in its parent's children.
         */
        public int childIndex(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CIntPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_child_index(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                int res = 0;
                if(propException == null) {
                    res = IntegerWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final int res = JNI_LIB.lkt_lkt_node_child_index(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return the node's previous sibling, or null if there is no such
         * sibling.
         */
        public LktNode previousSibling(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_previous_sibling(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_previous_sibling(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }

        
    


        /**
         * Return the node's next sibling, or null if there is no such sibling.
         */
        public LktNode nextSibling(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_next_sibling(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_next_sibling(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }

        
    

        
    


        /**
         * Return whether the node is a ghost.
         *
         * Unlike regular nodes, ghost nodes cover no token in the input
         * source: they are logically located instead between two tokens. Both
         * the ``token_start`` and the ``token_end`` of all ghost nodes is the
         * token right after this logical position.
         */
        public boolean isGhost(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_is_ghost(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_lkt_node_is_ghost(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return a string containing the filename + the sloc in GNU conformant
         * format. Useful to create diagnostics from a node.
         */
        public String fullSlocImage(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_full_sloc_image(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                String res = StringWrapper.NONE;
                if(propException == null) {
                    res = StringWrapper.wrap(resNative);

                    StringWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final String res = JNI_LIB.lkt_lkt_node_full_sloc_image(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Convert a CompletionItemKind enum to its corresponding integer
         * value.
         */
        public int completionItemKindToInt(
            final CompletionItemKind kind
        ) {

            // Verify that arguments are not null
            if(kind == null) throw new IllegalArgumentException(
                "Argument 'kind' of type " +
                "CompletionItemKind cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                int kindNative = kind.toC();

                // Create the result native
                final CIntPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_completion_item_kind_to_int(
                    thisNative,
                    kindNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                int res = 0;
                if(propException == null) {
                    res = IntegerWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final int res = JNI_LIB.lkt_lkt_node_completion_item_kind_to_int(
                    kind,
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Enable or disable the solver traces for debugging purposes.
         */
        public boolean pSetSolverDebugMode(
            final boolean enable
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                byte enableNative = (enable ? (byte) 1 : (byte) 0);

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_p_set_solver_debug_mode(
                    thisNative,
                    enableNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_lkt_node_p_set_solver_debug_mode(
                    enable,
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Unit method. Return the ``BasicTrait`` builtin generic trait.
         */
        public GenericDecl pBasicTraitGen(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_basic_trait_gen(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_basic_trait_gen(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ``BasicTrait`` builtin trait.
         */
        public TraitDecl pBasicTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_basic_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TraitDecl res = TraitDecl.NONE;
                if(propException == null) {
                    res = TraitDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_basic_trait(
                    this.entity
                );

                // Wrap and return the result
                return TraitDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ``Node`` builtin generic trait.
         */
        public GenericDecl pNodeGenTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_node_gen_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_node_gen_trait(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ``Node`` builtin trait.
         */
        public TraitDecl pNodeTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_node_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TraitDecl res = TraitDecl.NONE;
                if(propException == null) {
                    res = TraitDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_node_trait(
                    this.entity
                );

                // Wrap and return the result
                return TraitDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ``Node`` builtin generic trait.
         */
        public GenericDecl pIndexableGenTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_indexable_gen_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_indexable_gen_trait(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ``Node`` builtin trait.
         */
        public TraitDecl pIndexableTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_indexable_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TraitDecl res = TraitDecl.NONE;
                if(propException == null) {
                    res = TraitDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_indexable_trait(
                    this.entity
                );

                // Wrap and return the result
                return TraitDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ``TokenNode`` builtin trait.
         */
        public NamedTypeDecl pTokenNodeTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_token_node_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_token_node_trait(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ``ErrorNode`` builtin trait.
         */
        public NamedTypeDecl pErrorNodeTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_error_node_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_error_node_trait(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the character builtin type.
         */
        public NamedTypeDecl pCharType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_char_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_char_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the integer builtin type.
         */
        public NamedTypeDecl pIntType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_int_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_int_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the boolean builtin type.
         */
        public NamedTypeDecl pBoolType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_bool_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_bool_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the big integer builtin type.
         */
        public NamedTypeDecl pBigintType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_bigint_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_bigint_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the string builtin type.
         */
        public NamedTypeDecl pStringType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_string_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_string_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the string builtin type.
         */
        public NamedTypeDecl pSymbolType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_symbol_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_symbol_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the property error builtin type.
         */
        public NamedTypeDecl pPropertyErrorType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_property_error_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_property_error_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the regexp builtin type.
         */
        public NamedTypeDecl pRegexpType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_regexp_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_regexp_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the logicvar builtin type.
         */
        public GenericDecl pEntityGenType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_entity_gen_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_entity_gen_type(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the logicvar builtin type.
         */
        public NamedTypeDecl pEntityType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_entity_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_entity_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the logicvar builtin type.
         */
        public NamedTypeDecl pLogicvarType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_logicvar_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_logicvar_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the logicvar builtin type.
         */
        public NamedTypeDecl pEquationType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_equation_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_equation_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the array builtin generic type.
         */
        public GenericDecl pArrayGenType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_array_gen_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_array_gen_type(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the array builtin type.
         */
        public NamedTypeDecl pArrayType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_array_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_array_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ASTList builtin generic type.
         */
        public GenericDecl pAstlistGenType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_astlist_gen_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_astlist_gen_type(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ASTList builtin type.
         */
        public NamedTypeDecl pAstlistType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_astlist_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_astlist_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the NodeBuilder builtin generic type.
         */
        public GenericDecl pNodeBuilderGenType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_node_builder_gen_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_node_builder_gen_type(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the NodeBuilder builtin type.
         */
        public NamedTypeDecl pNodeBuilderType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_node_builder_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_node_builder_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the Iterator builtin generic trait.
         */
        public GenericDecl pIteratorGenTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_iterator_gen_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_iterator_gen_trait(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the Iterator builtin trait.
         */
        public TraitDecl pIteratorTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_iterator_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TraitDecl res = TraitDecl.NONE;
                if(propException == null) {
                    res = TraitDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_iterator_trait(
                    this.entity
                );

                // Wrap and return the result
                return TraitDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ``AnalysisUnit`` builtin generic trait.
         */
        public GenericDecl pAnalysisUnitGenTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_analysis_unit_gen_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_analysis_unit_gen_trait(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Unit method. Return the ``AnalysisUnit`` builtin trait.
         */
        public TraitDecl pAnalysisUnitTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_analysis_unit_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TraitDecl res = TraitDecl.NONE;
                if(propException == null) {
                    res = TraitDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_analysis_unit_trait(
                    this.entity
                );

                // Wrap and return the result
                return TraitDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Return the topmost (from ``Self`` to the root node) FullDecl
         * annotated with ``@invalid``, null otherwise.
         */
        public LktNode pTopmostInvalidDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_topmost_invalid_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_topmost_invalid_decl(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }

        
    

        public LangkitSupport.SolverDiagnosticInterface[]
        gNameresDiagnostics (
            
        ) {
            return pNameresDiagnostics(
                
            );
        }

        /**
         * If name resolution on this lkt compilation unit fails, this returns
         * all the diagnostics that were produced while resolving it.
         */
        public SolverDiagnostic[] pNameresDiagnostics(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_p_nameres_diagnostics(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                SolverDiagnostic[] res = SolverDiagnosticArrayWrapper.NONE;
                if(propException == null) {
                    res = SolverDiagnosticArrayWrapper.wrap(resNative);

                    SolverDiagnosticArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final SolverDiagnostic[] res = JNI_LIB.lkt_lkt_node_p_nameres_diagnostics(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Finds the nearest parent that is an xref_entry_point and solve its
         * equation.
         */
        public SolverResult pSolveEnclosingContext(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final SolverResultNative resNative =
                    StackValue.get(SolverResultNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_solve_enclosing_context(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                SolverResult res = SolverResult.NONE;
                if(propException == null) {
                    res = SolverResult.wrap(resNative);

                    SolverResult.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final SolverResult res = JNI_LIB.lkt_lkt_node_p_solve_enclosing_context(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        public boolean
        gXrefEntryPoint (
            
        ) {
            return pXrefEntryPoint(
                
            );
        }

        /**
         * Designates entities that are entry point for the xref solving
         * infrastructure. If this returns true, then nameres_diagnostics can
         * be called on it.
         */
        public boolean pXrefEntryPoint(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_p_xref_entry_point(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_lkt_node_p_xref_entry_point(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        public LangkitSupport.CompletionItemInterface[]
        gCompleteItems (
            
        ) {
            return pComplete(
                
            );
        }

        /**
         * Return an array of completion item for language server clients
         */
        public CompleteItem[] pComplete(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_p_complete(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                CompleteItem[] res = CompleteItemArrayWrapper.NONE;
                if(propException == null) {
                    res = CompleteItemArrayWrapper.wrap(resNative);

                    CompleteItemArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final CompleteItem[] res = JNI_LIB.lkt_lkt_node_p_complete(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Override methods -----

        @Override
        public String toString() {
            return this.getImage();
        }

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof LktNode)) return false;
            final LktNode other = (LktNode) o;

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final EntityNative otherNative = StackValue.get(
                    EntityNative.class
                );
                other.entity.unwrap(otherNative);

                return NI_LIB.lkt_node_is_equivalent(
                    thisNative,
                    otherNative
                ) != 0;
            } else {
                return JNI_LIB.lkt_node_is_equivalent(
                    this.entity, other.entity
                ) != 0;
            }
        }

        @Override
        public int hashCode() {
            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                return NI_LIB.lkt_node_hash(thisNative);
            } else {
                return JNI_LIB.lkt_node_hash(this.entity);
            }

        }

        // ----- Inner classes -----

        /**
         * This class represents the none node without any concrete type.
         */
        private static final class NoneNode extends LktNode {
            NoneNode() {super(Entity.NONE);}
        }

    }

    // ===== Generated node wrapping classes =====

    
    

    /**
     * Argument for function calls or for annotations.
     *
     * This node type has no derivation.
     */
    public static 
    class Argument
    extends LktNode
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ARGUMENT,
                false,
                false,
                Argument.class,
                "Argument",
                new String[] {
                    "f_name","f_value"
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = Argument.class.getMethod(
                        "fName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ARGUMENT_F_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Argument.class.getMethod(
                        "fValue",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_value",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ARGUMENT_F_VALUE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Argument NONE =
            new Argument(
                Entity.NONE
            );

        // ----- Constructors -----

        protected Argument(
            final Entity entity
        ) {
            super(entity);
        }

        public static Argument fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Argument.NONE :

                new Argument(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return Argument.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * This field may be null even when there are no parsing errors.
         */
        public RefId fName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_argument_f_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                RefId res = RefId.NONE;
                if(propException == null) {
                    res = RefId.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_argument_f_name(
                    this.entity
                );

                // Wrap and return the result
                return RefId.fromEntity(res);
            }

        }

        
    


        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BinOp``, ``BlockExpr``, ``CallExpr``,
         * ``CastExpr``, ``DotExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicAssign``, ``LogicExpr``,
         * ``LogicPredicate``, ``LogicPropagate``, ``LogicUnify``,
         * ``MatchExpr``, ``NotExpr``, ``ParenExpr``, ``RaiseExpr``, ``RefId``,
         * ``SubscriptExpr``, ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fValue(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_argument_f_value(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_argument_f_value(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Base class for the different kind of alternatives allowed in a case
     * rule.
     *
     * Derived nodes: ``LexerCaseRuleCondAlt``, ``LexerCaseRuleDefaultAlt``
     */
    public static abstract
    class BaseLexerCaseRuleAlt
    extends LktNode
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                BaseLexerCaseRuleAlt.class,
                "BaseLexerCaseRuleAlt",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BaseLexerCaseRuleAlt.class.getMethod(
                        "fSend",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_send",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BASE_LEXER_CASE_RULE_ALT_F_SEND
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseLexerCaseRuleAlt NONE =
            new BaseLexerCaseRuleAltNone();

        // ----- Constructors -----

        protected BaseLexerCaseRuleAlt(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseLexerCaseRuleAlt fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseLexerCaseRuleAlt.NONE :

                (BaseLexerCaseRuleAlt) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BaseLexerCaseRuleAlt.description;
        }


        // ----- Field accessors -----

        
    


        /**
         * When there are no parsing errors, this field is never null.
         */
        public LexerCaseRuleSend fSend(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_base_lexer_case_rule_alt_f_send(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LexerCaseRuleSend res = LexerCaseRuleSend.NONE;
                if(propException == null) {
                    res = LexerCaseRuleSend.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_base_lexer_case_rule_alt_f_send(
                    this.entity
                );

                // Wrap and return the result
                return LexerCaseRuleSend.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BaseLexerCaseRuleAlt.
         */
        private static final class BaseLexerCaseRuleAltNone extends BaseLexerCaseRuleAlt {
            BaseLexerCaseRuleAltNone() {super(Entity.NONE);}
        }


    }


    
    

    /**
     * Alternative of a case rule which sends the token only if the kind of the
     * previous token is among a given set.
     *
     * This node type has no derivation.
     */
    public static 
    class LexerCaseRuleCondAlt
    extends BaseLexerCaseRuleAlt
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LEXER_CASE_RULE_COND_ALT,
                false,
                false,
                LexerCaseRuleCondAlt.class,
                "LexerCaseRuleCondAlt",
                new String[] {
                    "f_cond_exprs","f_send"
                },
                new HashMap<>(
                    BaseLexerCaseRuleAlt.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = LexerCaseRuleCondAlt.class.getMethod(
                        "fCondExprs",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_cond_exprs",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LEXER_CASE_RULE_COND_ALT_F_COND_EXPRS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LexerCaseRuleCondAlt NONE =
            new LexerCaseRuleCondAlt(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LexerCaseRuleCondAlt(
            final Entity entity
        ) {
            super(entity);
        }

        public static LexerCaseRuleCondAlt fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LexerCaseRuleCondAlt.NONE :

                new LexerCaseRuleCondAlt(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LexerCaseRuleCondAlt.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * When there are no parsing errors, this field is never null.
         */
        public RefIdList fCondExprs(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lexer_case_rule_cond_alt_f_cond_exprs(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                RefIdList res = RefIdList.NONE;
                if(propException == null) {
                    res = RefIdList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lexer_case_rule_cond_alt_f_cond_exprs(
                    this.entity
                );

                // Wrap and return the result
                return RefIdList.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Default alternative of a case rule which sends the token if all the
     * previous alternatives failed.
     *
     * This node type has no derivation.
     */
    public static 
    class LexerCaseRuleDefaultAlt
    extends BaseLexerCaseRuleAlt
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LEXER_CASE_RULE_DEFAULT_ALT,
                false,
                false,
                LexerCaseRuleDefaultAlt.class,
                "LexerCaseRuleDefaultAlt",
                new String[] {
                    "f_send"
                },
                new HashMap<>(
                    BaseLexerCaseRuleAlt.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LexerCaseRuleDefaultAlt NONE =
            new LexerCaseRuleDefaultAlt(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LexerCaseRuleDefaultAlt(
            final Entity entity
        ) {
            super(entity);
        }

        public static LexerCaseRuleDefaultAlt fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LexerCaseRuleDefaultAlt.NONE :

                new LexerCaseRuleDefaultAlt(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LexerCaseRuleDefaultAlt.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Abstract base class for match branches, exists to accommodate the
     * existence of two different syntaxes.
     *
     * Derived nodes: ``MatchBranch``, ``PatternMatchBranch``
     */
    public static abstract
    class BaseMatchBranch
    extends LktNode
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                BaseMatchBranch.class,
                "BaseMatchBranch",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BaseMatchBranch.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BASE_MATCH_BRANCH_F_EXPR
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = BaseMatchBranch.class.getMethod(
                        "pMatchPart",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_match_part",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BASE_MATCH_BRANCH_P_MATCH_PART
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseMatchBranch NONE =
            new BaseMatchBranchNone();

        // ----- Constructors -----

        protected BaseMatchBranch(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseMatchBranch fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseMatchBranch.NONE :

                (BaseMatchBranch) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BaseMatchBranch.description;
        }


        // ----- Field accessors -----

        
    


        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BinOp``, ``BlockExpr``, ``CallExpr``,
         * ``CastExpr``, ``DotExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicAssign``, ``LogicExpr``,
         * ``LogicPredicate``, ``LogicPropagate``, ``LogicUnify``,
         * ``MatchExpr``, ``NotExpr``, ``ParenExpr``, ``RaiseExpr``, ``RefId``,
         * ``SubscriptExpr``, ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_base_match_branch_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_base_match_branch_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    


        /**
         * Return the "match" part of the branch, either a pattern branch or a
         * legacy match branch with variable declaration.
         */
        public LktNode pMatchPart(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_base_match_branch_p_match_part(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_base_match_branch_p_match_part(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BaseMatchBranch.
         */
        private static final class BaseMatchBranchNone extends BaseMatchBranch {
            BaseMatchBranchNone() {super(Entity.NONE);}
        }


    }


    
    

    /**
     * Branch inside a match expression. Classic limited Lkt syntax based on
     * ``case <name> : <type> => <expr>``, for the moment, the only supported
     * syntax in Lkt.
     *
     * This node type has no derivation.
     */
    public static 
    class MatchBranch
    extends BaseMatchBranch
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.MATCH_BRANCH,
                false,
                false,
                MatchBranch.class,
                "MatchBranch",
                new String[] {
                    "f_decl","f_expr"
                },
                new HashMap<>(
                    BaseMatchBranch.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = MatchBranch.class.getMethod(
                        "fDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_MATCH_BRANCH_F_DECL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final MatchBranch NONE =
            new MatchBranch(
                Entity.NONE
            );

        // ----- Constructors -----

        protected MatchBranch(
            final Entity entity
        ) {
            super(entity);
        }

        public static MatchBranch fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                MatchBranch.NONE :

                new MatchBranch(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return MatchBranch.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * When there are no parsing errors, this field is never null.
         */
        public MatchValDecl fDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_match_branch_f_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                MatchValDecl res = MatchValDecl.NONE;
                if(propException == null) {
                    res = MatchValDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_match_branch_f_decl(
                    this.entity
                );

                // Wrap and return the result
                return MatchValDecl.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Branch inside a match expression. LKQL pattern based syntax ``case
     * <pattern> => <expr>``.
     *
     * This node type has no derivation.
     */
    public static 
    class PatternMatchBranch
    extends BaseMatchBranch
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.PATTERN_MATCH_BRANCH,
                false,
                false,
                PatternMatchBranch.class,
                "PatternMatchBranch",
                new String[] {
                    "f_pattern","f_expr"
                },
                new HashMap<>(
                    BaseMatchBranch.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = PatternMatchBranch.class.getMethod(
                        "fPattern",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_pattern",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_PATTERN_MATCH_BRANCH_F_PATTERN
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final PatternMatchBranch NONE =
            new PatternMatchBranch(
                Entity.NONE
            );

        // ----- Constructors -----

        protected PatternMatchBranch(
            final Entity entity
        ) {
            super(entity);
        }

        public static PatternMatchBranch fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                PatternMatchBranch.NONE :

                new PatternMatchBranch(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return PatternMatchBranch.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * This field can contain one of the following nodes:
         * ``BindingPattern``, ``BoolPattern``, ``ExtendedPattern``,
         * ``FilteredPattern``, ``IntegerPattern``, ``ListPattern``,
         * ``NotPattern``, ``NullPattern``, ``OrPattern``, ``ParenPattern``,
         * ``RegexPattern``, ``TuplePattern``, ``TypePattern``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Pattern fPattern(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_pattern_match_branch_f_pattern(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Pattern res = Pattern.NONE;
                if(propException == null) {
                    res = Pattern.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_pattern_match_branch_f_pattern(
                    this.entity
                );

                // Wrap and return the result
                return Pattern.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Clause (value declaration or dynamic variable binding) in a block
     * expression.
     *
     * This node type has no derivation.
     */
    public static 
    class BlockExprClause
    extends LktNode
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.BLOCK_EXPR_CLAUSE,
                false,
                false,
                BlockExprClause.class,
                "BlockExprClause",
                new String[] {
                    "f_clause"
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BlockExprClause.class.getMethod(
                        "fClause",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_clause",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BLOCK_EXPR_CLAUSE_F_CLAUSE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BlockExprClause NONE =
            new BlockExprClause(
                Entity.NONE
            );

        // ----- Constructors -----

        protected BlockExprClause(
            final Entity entity
        ) {
            super(entity);
        }

        public static BlockExprClause fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BlockExprClause.NONE :

                new BlockExprClause(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BlockExprClause.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * This field can contain one of the following nodes: ``ValDecl``,
         * ``VarBind``
         *
         * When there are no parsing errors, this field is never null.
         */
        public LktNode fClause(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_block_expr_clause_f_clause(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_block_expr_clause_f_clause(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * A single line in a block string literal.
     *
     * This node type has no derivation.
     */
    public static 
    class BlockStringLine
    extends LktNode
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.BLOCK_STRING_LINE,
                true,
                false,
                BlockStringLine.class,
                "BlockStringLine",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BlockStringLine NONE =
            new BlockStringLine(
                Entity.NONE
            );

        // ----- Constructors -----

        protected BlockStringLine(
            final Entity entity
        ) {
            super(entity);
        }

        public static BlockStringLine fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BlockStringLine.NONE :

                new BlockStringLine(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BlockStringLine.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Whether this generic parameter type must be a class.
     *
     * Derived nodes: ``ClassQualifierAbsent``, ``ClassQualifierPresent``
     */
    public static abstract
    class ClassQualifier
    extends LktNode
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                ClassQualifier.class,
                "ClassQualifier",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ClassQualifier.class.getMethod(
                        "pAsBool",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_as_bool",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CLASS_QUALIFIER_P_AS_BOOL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ClassQualifier NONE =
            new ClassQualifierNone();

        // ----- Constructors -----

        protected ClassQualifier(
            final Entity entity
        ) {
            super(entity);
        }

        public static ClassQualifier fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ClassQualifier.NONE :

                (ClassQualifier) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ClassQualifier.description;
        }


        // ----- Field accessors -----

        
    


        /**
         * Return whether this node is present
         */
        public boolean pAsBool(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_class_qualifier_p_as_bool(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_class_qualifier_p_as_bool(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * ClassQualifier.
         */
        private static final class ClassQualifierNone extends ClassQualifier {
            ClassQualifierNone() {super(Entity.NONE);}
        }


    }


    
    

    /**
     * This node type has no derivation.
     */
    public static 
    class ClassQualifierAbsent
    extends ClassQualifier
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.CLASS_QUALIFIER_ABSENT,
                false,
                false,
                ClassQualifierAbsent.class,
                "ClassQualifierAbsent",
                new String[] {
                    
                },
                new HashMap<>(
                    ClassQualifier.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ClassQualifierAbsent NONE =
            new ClassQualifierAbsent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ClassQualifierAbsent(
            final Entity entity
        ) {
            super(entity);
        }

        public static ClassQualifierAbsent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ClassQualifierAbsent.NONE :

                new ClassQualifierAbsent(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ClassQualifierAbsent.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * This node type has no derivation.
     */
    public static 
    class ClassQualifierPresent
    extends ClassQualifier
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.CLASS_QUALIFIER_PRESENT,
                false,
                false,
                ClassQualifierPresent.class,
                "ClassQualifierPresent",
                new String[] {
                    
                },
                new HashMap<>(
                    ClassQualifier.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ClassQualifierPresent NONE =
            new ClassQualifierPresent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ClassQualifierPresent(
            final Entity entity
        ) {
            super(entity);
        }

        public static ClassQualifierPresent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ClassQualifierPresent.NONE :

                new ClassQualifierPresent(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ClassQualifierPresent.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Base class for declarations. Encompasses regular declarations as well as
     * special declarations such as grammars, grammar rules, etc.
     *
     * Derived nodes: ``BaseGrammarRuleDecl``, ``BaseValDecl``,
     * ``EnvSpecDecl``, ``ErrorDecl``, ``GenericDecl``, ``GrammarDecl``,
     * ``LexerDecl``, ``LexerFamilyDecl``, ``SynthFunDecl``,
     * ``SynthParamDecl``, ``TypeDecl``
     */
    public static abstract
    class Decl
    extends LktNode
    implements LangkitSupport.DeclarationInterface
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                Decl.class,
                "Decl",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "fSynName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_syn_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_F_SYN_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pCustomImage",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_custom_image",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_CUSTOM_IMAGE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pDeclTypeName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_decl_type_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_DECL_TYPE_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pDefIds",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_def_ids",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_DEF_IDS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pAsBareDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_as_bare_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_AS_BARE_DECL
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pGetType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_get_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_GET_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pGetCastType",
                        new Class[]{TypeDecl.class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        TypeDecl.class,
                        "castTo"
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_get_cast_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_GET_CAST_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pGetKeepType",
                        new Class[]{TypeDecl.class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        TypeDecl.class,
                        "keepType"
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_get_keep_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_GET_KEEP_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pGetSuffixType",
                        new Class[]{TypeDecl.class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        TypeDecl.class,
                        "prefixType"
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_get_suffix_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_GET_SUFFIX_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pIsGeneric",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_is_generic",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_IS_GENERIC
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pReturnTypeIsInstantiated",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_return_type_is_instantiated",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_RETURN_TYPE_IS_INSTANTIATED
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pIsInstantiated",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_is_instantiated",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_IS_INSTANTIATED
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pFullName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_full_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_FULL_NAME
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Decl NONE =
            new DeclNone();

        // ----- Constructors -----

        protected Decl(
            final Entity entity
        ) {
            super(entity);
        }

        public static Decl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Decl.NONE :

                (Decl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return Decl.description;
        }


        // ----- Field accessors -----

        
    


        /**
         * This field may be null even when there are no parsing errors.
         */
        public DefId fSynName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_f_syn_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DefId res = DefId.NONE;
                if(propException == null) {
                    res = DefId.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_f_syn_name(
                    this.entity
                );

                // Wrap and return the result
                return DefId.fromEntity(res);
            }

        }

        
    


        /**
         * Return the image string using entity information.
         */
        public String pCustomImage(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_decl_p_custom_image(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                String res = StringWrapper.NONE;
                if(propException == null) {
                    res = StringWrapper.wrap(resNative);

                    StringWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final String res = JNI_LIB.lkt_decl_p_custom_image(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return the name of the declaration type, as it should be seen by
         * users/shown in diagnostics.
         */
        public String pDeclTypeName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_decl_p_decl_type_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                String res = StringWrapper.NONE;
                if(propException == null) {
                    res = StringWrapper.wrap(resNative);

                    StringWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final String res = JNI_LIB.lkt_decl_p_decl_type_name(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        public LangkitSupport.DefiningNameInterface[]
        gDefiningNames (
            
        ) {
            return pDefIds(
                
            );
        }

        /**
         * Return all the defining names that this declaration defines.
         */
        public DefId[] pDefIds(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_decl_p_def_ids(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DefId[] res = DefIdArrayWrapper.NONE;
                if(propException == null) {
                    res = DefIdArrayWrapper.wrap(resNative);

                    DefIdArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final DefId[] res = JNI_LIB.lkt_decl_p_def_ids(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Get this declaration without rebindings information.
         */
        public Decl pAsBareDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_p_as_bare_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Decl res = Decl.NONE;
                if(propException == null) {
                    res = Decl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_p_as_bare_decl(
                    this.entity
                );

                // Wrap and return the result
                return Decl.fromEntity(res);
            }

        }

        
    


        /**
         * Return the type of the Decl.
         */
        public TypeDecl pGetType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_p_get_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_p_get_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * If we are casting an entity (Self) to something that is not an
         * entity, make it an entity.
         */
        public TypeDecl pGetCastType(
            final TypeDecl castTo
        ) {

            // Verify that arguments are not null
            if(castTo == null) throw new IllegalArgumentException(
                "Argument 'castTo' of type " +
                "TypeDecl cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                EntityNative castToNative = StackValue.get(EntityNative.class);castTo.entity.unwrap(castToNative);

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_p_get_cast_type(
                    thisNative,
                    castToNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_p_get_cast_type(
                    (castTo != null ? castTo.entity : null),
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Return the type of Entity when we only keep elements of type
         * keep_type. If we are casting an entity (Self) to something that is
         * not an entity, make it an entity.
         */
        public TypeDecl pGetKeepType(
            final TypeDecl keepType
        ) {

            // Verify that arguments are not null
            if(keepType == null) throw new IllegalArgumentException(
                "Argument 'keepType' of type " +
                "TypeDecl cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                EntityNative keepTypeNative = StackValue.get(EntityNative.class);keepType.entity.unwrap(keepTypeNative);

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_p_get_keep_type(
                    thisNative,
                    keepTypeNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_p_get_keep_type(
                    (keepType != null ? keepType.entity : null),
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * If we are accessing a ParseField of an entity, then that field's
         * type also needs to be an entity.
         */
        public TypeDecl pGetSuffixType(
            final TypeDecl prefixType
        ) {

            // Verify that arguments are not null
            if(prefixType == null) throw new IllegalArgumentException(
                "Argument 'prefixType' of type " +
                "TypeDecl cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                EntityNative prefixTypeNative = StackValue.get(EntityNative.class);prefixType.entity.unwrap(prefixTypeNative);

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_p_get_suffix_type(
                    thisNative,
                    prefixTypeNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_p_get_suffix_type(
                    (prefixType != null ? prefixType.entity : null),
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }

        
    


        /**
         * Returns whether the Decl is generic.
         */
        public boolean pIsGeneric(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_decl_p_is_generic(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_decl_p_is_generic(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return True if the return type of this function is instantiated.
         */
        public boolean pReturnTypeIsInstantiated(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_decl_p_return_type_is_instantiated(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_decl_p_return_type_is_instantiated(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return True if Self is an instantiated declaration, meaning that it
         * does not use any of its declared generic types.
         */
        public boolean pIsInstantiated(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_decl_p_is_instantiated(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_decl_p_is_instantiated(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return the symbol corresponding to the name of this declaration.
         */
        public Symbol pName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final SymbolNative resNative =
                    StackValue.get(SymbolNative.class);

                // Call the native function
                NI_LIB.lkt_decl_p_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Symbol res = Symbol.NONE;
                if(propException == null) {
                    res = Symbol.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Symbol res = JNI_LIB.lkt_decl_p_name(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return the full name of this decl, as it should be seen by
         * users/shown in diagnostics.
         */
        public String pFullName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_decl_p_full_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                String res = StringWrapper.NONE;
                if(propException == null) {
                    res = StringWrapper.wrap(resNative);

                    StringWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final String res = JNI_LIB.lkt_decl_p_full_name(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * Decl.
         */
        private static final class DeclNone extends Decl {
            DeclNone() {super(Entity.NONE);}
        }


    }


    
    

    /**
     * Base class for grammar rules inside of grammars/lexers.
     *
     * Derived nodes: ``GrammarRuleDecl``, ``SyntheticLexerDecl``
     */
    public static abstract
    class BaseGrammarRuleDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                BaseGrammarRuleDecl.class,
                "BaseGrammarRuleDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BaseGrammarRuleDecl.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BASE_GRAMMAR_RULE_DECL_F_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseGrammarRuleDecl NONE =
            new BaseGrammarRuleDeclNone();

        // ----- Constructors -----

        protected BaseGrammarRuleDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseGrammarRuleDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseGrammarRuleDecl.NONE :

                (BaseGrammarRuleDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BaseGrammarRuleDecl.description;
        }


        // ----- Field accessors -----

        
    


        /**
         * This field may be null even when there are no parsing errors.
         */
        public GrammarExpr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_base_grammar_rule_decl_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarExpr res = GrammarExpr.NONE;
                if(propException == null) {
                    res = GrammarExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_base_grammar_rule_decl_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return GrammarExpr.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BaseGrammarRuleDecl.
         */
        private static final class BaseGrammarRuleDeclNone extends BaseGrammarRuleDecl {
            BaseGrammarRuleDeclNone() {super(Entity.NONE);}
        }


    }


    
    

    /**
     * Declaration of a grammar rule inside of a grammar.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarRuleDecl
    extends BaseGrammarRuleDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_RULE_DECL,
                false,
                false,
                GrammarRuleDecl.class,
                "GrammarRuleDecl",
                new String[] {
                    "f_syn_name","f_expr"
                },
                new HashMap<>(
                    BaseGrammarRuleDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarRuleDecl NONE =
            new GrammarRuleDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarRuleDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarRuleDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarRuleDecl.NONE :

                new GrammarRuleDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarRuleDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * This node type has no derivation.
     */
    public static 
    class SyntheticLexerDecl
    extends BaseGrammarRuleDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.SYNTHETIC_LEXER_DECL,
                false,
                false,
                SyntheticLexerDecl.class,
                "SyntheticLexerDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    BaseGrammarRuleDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final SyntheticLexerDecl NONE =
            new SyntheticLexerDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected SyntheticLexerDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static SyntheticLexerDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                SyntheticLexerDecl.NONE :

                new SyntheticLexerDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return SyntheticLexerDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Abstract class for named values declarations, such as parameters, local
     * value bindings, fields, etc.
     *
     * Derived nodes: ``NodeDecl``, ``SelfDecl``, ``UserValDecl``
     */
    public static abstract
    class BaseValDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                BaseValDecl.class,
                "BaseValDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseValDecl NONE =
            new BaseValDeclNone();

        // ----- Constructors -----

        protected BaseValDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseValDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseValDecl.NONE :

                (BaseValDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BaseValDecl.description;
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BaseValDecl.
         */
        private static final class BaseValDeclNone extends BaseValDecl {
            BaseValDeclNone() {super(Entity.NONE);}
        }


    }


    
    

    /**
     * Synthetic declaration for the implicit "node" variable available in
     * properties.
     *
     * This node type has no derivation.
     */
    public static 
    class NodeDecl
    extends BaseValDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.NODE_DECL,
                false,
                false,
                NodeDecl.class,
                "NodeDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    BaseValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final NodeDecl NONE =
            new NodeDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected NodeDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static NodeDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                NodeDecl.NONE :

                new NodeDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return NodeDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Synthetic declaration for the implicit "self" variable available in
     * properties.
     *
     * This node type has no derivation.
     */
    public static 
    class SelfDecl
    extends BaseValDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.SELF_DECL,
                false,
                false,
                SelfDecl.class,
                "SelfDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    BaseValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final SelfDecl NONE =
            new SelfDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected SelfDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static SelfDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                SelfDecl.NONE :

                new SelfDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return SelfDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Class for user declared val declarations (not synthetic).
     *
     * Derived nodes: ``BindingValDecl``, ``EnumLitDecl``,
     * ``ExplicitlyTypedDecl``, ``FunDecl``
     */
    public static abstract
    class UserValDecl
    extends BaseValDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                UserValDecl.class,
                "UserValDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    BaseValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final UserValDecl NONE =
            new UserValDeclNone();

        // ----- Constructors -----

        protected UserValDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static UserValDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                UserValDecl.NONE :

                (UserValDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return UserValDecl.description;
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * UserValDecl.
         */
        private static final class UserValDeclNone extends UserValDecl {
            UserValDeclNone() {super(Entity.NONE);}
        }


    }


    
    

    /**
     * Variable declaration in pattern
     *
     * This node type has no derivation.
     */
    public static 
    class BindingValDecl
    extends UserValDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.BINDING_VAL_DECL,
                false,
                false,
                BindingValDecl.class,
                "BindingValDecl",
                new String[] {
                    "f_syn_name"
                },
                new HashMap<>(
                    UserValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BindingValDecl NONE =
            new BindingValDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected BindingValDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static BindingValDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BindingValDecl.NONE :

                new BindingValDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BindingValDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Enum literal declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class EnumLitDecl
    extends UserValDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ENUM_LIT_DECL,
                false,
                false,
                EnumLitDecl.class,
                "EnumLitDecl",
                new String[] {
                    "f_syn_name"
                },
                new HashMap<>(
                    UserValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final EnumLitDecl NONE =
            new EnumLitDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected EnumLitDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static EnumLitDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                EnumLitDecl.NONE :

                new EnumLitDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return EnumLitDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Subset of user declared value declarations for values that have a type
     * that can be syntactically annotated by the user.
     *
     * Derived nodes: ``ComponentDecl``, ``DynVarDecl``, ``MatchValDecl``,
     * ``ValDecl``
     */
    public static abstract
    class ExplicitlyTypedDecl
    extends UserValDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                ExplicitlyTypedDecl.class,
                "ExplicitlyTypedDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    UserValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ExplicitlyTypedDecl.class.getMethod(
                        "fDeclType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_decl_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXPLICITLY_TYPED_DECL_F_DECL_TYPE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ExplicitlyTypedDecl NONE =
            new ExplicitlyTypedDeclNone();

        // ----- Constructors -----

        protected ExplicitlyTypedDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static ExplicitlyTypedDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ExplicitlyTypedDecl.NONE :

                (ExplicitlyTypedDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ExplicitlyTypedDecl.description;
        }


        // ----- Field accessors -----

        
    


        /**
         * This field can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * This field may be null even when there are no parsing errors.
         */
        public TypeRef fDeclType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_explicitly_typed_decl_f_decl_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_explicitly_typed_decl_f_decl_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * ExplicitlyTypedDecl.
         */
        private static final class ExplicitlyTypedDeclNone extends ExplicitlyTypedDecl {
            ExplicitlyTypedDeclNone() {super(Entity.NONE);}
        }


    }


    
    

    /**
     * Subset of explicitly typed declarations for value declarations that:
     *
     * 1. Have an optional default value.
     *
     * 2. Are part of a bigger declaration that can be referred to via a call
     *    expression (either a type or a function).
     *
     * Derived nodes: ``FieldDecl``, ``FunParamDecl``, ``LambdaParamDecl``
     */
    public static abstract
    class ComponentDecl
    extends ExplicitlyTypedDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                ComponentDecl.class,
                "ComponentDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    ExplicitlyTypedDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ComponentDecl.class.getMethod(
                        "fDefaultVal",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_default_val",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_COMPONENT_DECL_F_DEFAULT_VAL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ComponentDecl NONE =
            new ComponentDeclNone();

        // ----- Constructors -----

        protected ComponentDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static ComponentDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ComponentDecl.NONE :

                (ComponentDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ComponentDecl.description;
        }


        // ----- Field accessors -----

        
    


        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BinOp``, ``BlockExpr``, ``CallExpr``,
         * ``CastExpr``, ``DotExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicAssign``, ``LogicExpr``,
         * ``LogicPredicate``, ``LogicPropagate``, ``LogicUnify``,
         * ``MatchExpr``, ``NotExpr``, ``ParenExpr``, ``RaiseExpr``, ``RefId``,
         * ``SubscriptExpr``, ``TryExpr``, ``UnOp``
         *
         * This field may be null even when there are no parsing errors.
         */
        public Expr fDefaultVal(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_component_decl_f_default_val(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_component_decl_f_default_val(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * ComponentDecl.
         */
        private static final class ComponentDeclNone extends ComponentDecl {
            ComponentDeclNone() {super(Entity.NONE);}
        }


    }


    
    

    /**
     * Field declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class FieldDecl
    extends ComponentDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.FIELD_DECL,
                false,
                false,
                FieldDecl.class,
                "FieldDecl",
                new String[] {
                    "f_syn_name","f_decl_type","f_trait_ref","f_default_val"
                },
                new HashMap<>(
                    ComponentDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = FieldDecl.class.getMethod(
                        "fTraitRef",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_trait_ref",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FIELD_DECL_F_TRAIT_REF
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final FieldDecl NONE =
            new FieldDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected FieldDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static FieldDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                FieldDecl.NONE :

                new FieldDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return FieldDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * This field may be null even when there are no parsing errors.
         */
        public DotExpr fTraitRef(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_field_decl_f_trait_ref(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DotExpr res = DotExpr.NONE;
                if(propException == null) {
                    res = DotExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_field_decl_f_trait_ref(
                    this.entity
                );

                // Wrap and return the result
                return DotExpr.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Function parameter declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class FunParamDecl
    extends ComponentDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.FUN_PARAM_DECL,
                false,
                false,
                FunParamDecl.class,
                "FunParamDecl",
                new String[] {
                    "f_decl_annotations","f_syn_name","f_decl_type","f_default_val"
                },
                new HashMap<>(
                    ComponentDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = FunParamDecl.class.getMethod(
                        "fDeclAnnotations",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_decl_annotations",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_PARAM_DECL_F_DECL_ANNOTATIONS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final FunParamDecl NONE =
            new FunParamDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected FunParamDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static FunParamDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                FunParamDecl.NONE :

                new FunParamDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return FunParamDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * When there are no parsing errors, this field is never null.
         */
        public DeclAnnotationList fDeclAnnotations(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_fun_param_decl_f_decl_annotations(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DeclAnnotationList res = DeclAnnotationList.NONE;
                if(propException == null) {
                    res = DeclAnnotationList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_fun_param_decl_f_decl_annotations(
                    this.entity
                );

                // Wrap and return the result
                return DeclAnnotationList.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Function parameter declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class LambdaParamDecl
    extends ComponentDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LAMBDA_PARAM_DECL,
                false,
                false,
                LambdaParamDecl.class,
                "LambdaParamDecl",
                new String[] {
                    "f_syn_name","f_decl_type","f_default_val"
                },
                new HashMap<>(
                    ComponentDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LambdaParamDecl NONE =
            new LambdaParamDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LambdaParamDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static LambdaParamDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LambdaParamDecl.NONE :

                new LambdaParamDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LambdaParamDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Dynamic variable declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class DynVarDecl
    extends ExplicitlyTypedDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.DYN_VAR_DECL,
                false,
                false,
                DynVarDecl.class,
                "DynVarDecl",
                new String[] {
                    "f_syn_name","f_decl_type"
                },
                new HashMap<>(
                    ExplicitlyTypedDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final DynVarDecl NONE =
            new DynVarDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected DynVarDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static DynVarDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                DynVarDecl.NONE :

                new DynVarDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return DynVarDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Value declaration in a match branch.
     *
     * This node type has no derivation.
     */
    public static 
    class MatchValDecl
    extends ExplicitlyTypedDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.MATCH_VAL_DECL,
                false,
                false,
                MatchValDecl.class,
                "MatchValDecl",
                new String[] {
                    "f_syn_name","f_decl_type"
                },
                new HashMap<>(
                    ExplicitlyTypedDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final MatchValDecl NONE =
            new MatchValDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected MatchValDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static MatchValDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                MatchValDecl.NONE :

                new MatchValDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return MatchValDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Value declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class ValDecl
    extends ExplicitlyTypedDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.VAL_DECL,
                false,
                false,
                ValDecl.class,
                "ValDecl",
                new String[] {
                    "f_syn_name","f_decl_type","f_expr"
                },
                new HashMap<>(
                    ExplicitlyTypedDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ValDecl.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_VAL_DECL_F_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ValDecl NONE =
            new ValDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ValDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static ValDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ValDecl.NONE :

                new ValDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ValDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BinOp``, ``BlockExpr``, ``CallExpr``,
         * ``CastExpr``, ``DotExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicAssign``, ``LogicExpr``,
         * ``LogicPredicate``, ``LogicPropagate``, ``LogicUnify``,
         * ``MatchExpr``, ``NotExpr``, ``ParenExpr``, ``RaiseExpr``, ``RefId``,
         * ``SubscriptExpr``, ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_val_decl_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_val_decl_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Function declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class FunDecl
    extends UserValDecl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.FUN_DECL,
                false,
                false,
                FunDecl.class,
                "FunDecl",
                new String[] {
                    "f_syn_name","f_params","f_return_type","f_trait_ref","f_body"
                },
                new HashMap<>(
                    UserValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = FunDecl.class.getMethod(
                        "fParams",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_params",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_DECL_F_PARAMS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = FunDecl.class.getMethod(
                        "fReturnType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_return_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_DECL_F_RETURN_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = FunDecl.class.getMethod(
                        "fTraitRef",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_trait_ref",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_DECL_F_TRAIT_REF
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = FunDecl.class.getMethod(
                        "fBody",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_body",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_DECL_F_BODY
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = FunDecl.class.getMethod(
                        "pIsDynamicCombiner",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_is_dynamic_combiner",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_DECL_P_IS_DYNAMIC_COMBINER
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = FunDecl.class.getMethod(
                        "pFindAllOverrides",
                        new Class[]{AnalysisUnit[].class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        AnalysisUnit[].class,
                        "units"
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_find_all_overrides",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_DECL_P_FIND_ALL_OVERRIDES
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final FunDecl NONE =
            new FunDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected FunDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static FunDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                FunDecl.NONE :

                new FunDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return FunDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * When there are no parsing errors, this field is never null.
         */
        public FunParamDeclList fParams(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_fun_decl_f_params(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                FunParamDeclList res = FunParamDeclList.NONE;
                if(propException == null) {
                    res = FunParamDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_fun_decl_f_params(
                    this.entity
                );

                // Wrap and return the result
                return FunParamDeclList.fromEntity(res);
            }

        }

        
    


        /**
         * This field can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * When there are no parsing errors, this field is never null.
         */
        public TypeRef fReturnType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_fun_decl_f_return_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_fun_decl_f_return_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }

        
    


        /**
         * This field may be null even when there are no parsing errors.
         */
        public DotExpr fTraitRef(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_fun_decl_f_trait_ref(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DotExpr res = DotExpr.NONE;
                if(propException == null) {
                    res = DotExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_fun_decl_f_trait_ref(
                    this.entity
                );

                // Wrap and return the result
                return DotExpr.fromEntity(res);
            }

        }

        
    


        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BinOp``, ``BlockExpr``, ``CallExpr``,
         * ``CastExpr``, ``DotExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicAssign``, ``LogicExpr``,
         * ``LogicPredicate``, ``LogicPropagate``, ``LogicUnify``,
         * ``MatchExpr``, ``NotExpr``, ``ParenExpr``, ``RaiseExpr``, ``RefId``,
         * ``SubscriptExpr``, ``TryExpr``, ``UnOp``
         *
         * This field may be null even when there are no parsing errors.
         */
        public Expr fBody(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_fun_decl_f_body(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_fun_decl_f_body(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    


        /**
         * When this property is used as a a combinder inside an NPropagate
         * equation, return whether it expects a dynamic number of arguments.
         */
        public boolean pIsDynamicCombiner(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_fun_decl_p_is_dynamic_combiner(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_fun_decl_p_is_dynamic_combiner(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    


        /**
         * Return the list of all RefId that refer to this DefId.
         */
        public FunDecl[] pFindAllOverrides(
            final AnalysisUnit[] units
        ) {

            // Verify that arguments are not null
            if(units == null) throw new IllegalArgumentException(
                "Argument 'units' of type " +
                "AnalysisUnit[] cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                AnalysisUnitArrayNative unitsNative = AnalysisUnitArrayWrapper.unwrap(units);

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_fun_decl_p_find_all_overrides(
                    thisNative,
                    unitsNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                FunDecl[] res = FunDeclArrayWrapper.NONE;
                if(propException == null) {
                    res = FunDeclArrayWrapper.wrap(resNative);

                    FunDeclArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters
                AnalysisUnitArrayWrapper.release(
                    unitsNative
                );


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final FunDecl[] res = JNI_LIB.lkt_fun_decl_p_find_all_overrides(
                    units,
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Env spec declaration.
     *
     * Each node type can have one or no env spec. Env specs contains only a
     * list of env actions.
     *
     * This node type has no derivation.
     */
    public static 
    class EnvSpecDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ENV_SPEC_DECL,
                false,
                false,
                EnvSpecDecl.class,
                "EnvSpecDecl",
                new String[] {
                    "f_syn_name","f_actions"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = EnvSpecDecl.class.getMethod(
                        "fActions",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_actions",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ENV_SPEC_DECL_F_ACTIONS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final EnvSpecDecl NONE =
            new EnvSpecDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected EnvSpecDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static EnvSpecDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                EnvSpecDecl.NONE :

                new EnvSpecDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return EnvSpecDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * When there are no parsing errors, this field is never null.
         */
        public CallExprList fActions(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_env_spec_decl_f_actions(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                CallExprList res = CallExprList.NONE;
                if(propException == null) {
                    res = CallExprList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_env_spec_decl_f_actions(
                    this.entity
                );

                // Wrap and return the result
                return CallExprList.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Placeholder node for syntax errors in lists of declarations.
     *
     * This node type has no derivation.
     */
    public static 
    class ErrorDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ERROR_DECL,
                false,
                false,
                ErrorDecl.class,
                "ErrorDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ErrorDecl NONE =
            new ErrorDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ErrorDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static ErrorDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ErrorDecl.NONE :

                new ErrorDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ErrorDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Generic entity declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class GenericDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GENERIC_DECL,
                false,
                false,
                GenericDecl.class,
                "GenericDecl",
                new String[] {
                    "f_generic_param_decls","f_decl"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GenericDecl.class.getMethod(
                        "fGenericParamDecls",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_generic_param_decls",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GENERIC_DECL_F_GENERIC_PARAM_DECLS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GenericDecl.class.getMethod(
                        "fDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GENERIC_DECL_F_DECL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GenericDecl NONE =
            new GenericDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GenericDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static GenericDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GenericDecl.NONE :

                new GenericDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GenericDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * When there are no parsing errors, this field is never null.
         */
        public GenericParamDeclList fGenericParamDecls(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_generic_decl_f_generic_param_decls(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericParamDeclList res = GenericParamDeclList.NONE;
                if(propException == null) {
                    res = GenericParamDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_generic_decl_f_generic_param_decls(
                    this.entity
                );

                // Wrap and return the result
                return GenericParamDeclList.fromEntity(res);
            }

        }

        
    


        /**
         * This field can contain one of the following nodes: ``DynVarDecl``,
         * ``EnvSpecDecl``, ``FieldDecl``, ``FunDecl``, ``GenericDecl``,
         * ``GrammarDecl``, ``GrammarRuleDecl``, ``LexerDecl``,
         * ``NamedTypeDecl``, ``ValDecl``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Decl fDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_generic_decl_f_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Decl res = Decl.NONE;
                if(propException == null) {
                    res = Decl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_generic_decl_f_decl(
                    this.entity
                );

                // Wrap and return the result
                return Decl.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Declaration of a language's grammar.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_DECL,
                false,
                false,
                GrammarDecl.class,
                "GrammarDecl",
                new String[] {
                    "f_syn_name","f_rules"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GrammarDecl.class.getMethod(
                        "fRules",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_rules",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_DECL_F_RULES
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarDecl NONE =
            new GrammarDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarDecl.NONE :

                new GrammarDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * When there are no parsing errors, this field is never null.
         */
        public FullDeclList fRules(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_decl_f_rules(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                FullDeclList res = FullDeclList.NONE;
                if(propException == null) {
                    res = FullDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_decl_f_rules(
                    this.entity
                );

                // Wrap and return the result
                return FullDeclList.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Declaration of a language's lexer.
     *
     * This node type has no derivation.
     */
    public static 
    class LexerDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LEXER_DECL,
                false,
                false,
                LexerDecl.class,
                "LexerDecl",
                new String[] {
                    "f_syn_name","f_rules"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = LexerDecl.class.getMethod(
                        "fRules",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_rules",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LEXER_DECL_F_RULES
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LexerDecl NONE =
            new LexerDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LexerDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static LexerDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LexerDecl.NONE :

                new LexerDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LexerDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``FullDecl``, ``LexerCaseRule``
         *
         * When there are no parsing errors, this field is never null.
         */
        public LktNodeList fRules(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lexer_decl_f_rules(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNodeList res = LktNodeList.NONE;
                if(propException == null) {
                    res = LktNodeList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lexer_decl_f_rules(
                    this.entity
                );

                // Wrap and return the result
                return LktNodeList.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Declaration of a token family.
     *
     * This node type has no derivation.
     */
    public static 
    class LexerFamilyDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LEXER_FAMILY_DECL,
                false,
                false,
                LexerFamilyDecl.class,
                "LexerFamilyDecl",
                new String[] {
                    "f_syn_name","f_rules"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = LexerFamilyDecl.class.getMethod(
                        "fRules",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_rules",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LEXER_FAMILY_DECL_F_RULES
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LexerFamilyDecl NONE =
            new LexerFamilyDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LexerFamilyDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static LexerFamilyDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LexerFamilyDecl.NONE :

                new LexerFamilyDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LexerFamilyDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    


        /**
         * When there are no parsing errors, this field is never null.
         */
        public FullDeclList fRules(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lexer_family_decl_f_rules(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                FullDeclList res = FullDeclList.NONE;
                if(propException == null) {
                    res = FullDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lexer_family_decl_f_rules(
                    this.entity
                );

                // Wrap and return the result
                return FullDeclList.fromEntity(res);
            }

        }


        // ----- Inner classes -----



    }


    
    

    /**
     * Logic function declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class SynthFunDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.SYNTH_FUN_DECL,
                false,
                false,
                SynthFunDecl.class,
                "SynthFunDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final SynthFunDecl NONE =
            new SynthFunDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected SynthFunDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static SynthFunDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                SynthFunDecl.NONE :

                new SynthFunDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return SynthFunDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Logic function parameter declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class SynthParamDecl
    extends Decl
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.SYNTH_PARAM_DECL,
                false,
                false,
                SynthParamDecl.class,
                "SynthParamDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final SynthParamDecl NONE =
            new SynthParamDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected SynthParamDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static SynthParamDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                SynthParamDecl.NONE :

                new SynthParamDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return SynthParamDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----



    }


    
    

    /**
     * Abstract base class for type declarations.
     *
     * Derived nodes: ``AnyTypeDecl``, ``EnumClassAltDecl``, ``FunctionType``,
     * ``GenericParamTypeDecl``, ``NamedTypeDecl``
     */
    public static abstract
    class TypeDecl
    extends Decl
    implements LangkitSupport.TypeInterface
    {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                TypeDecl.class,
                "TypeDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "fTraits",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_traits",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_F_TRAITS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "fSynBaseType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_syn_base_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_F_SYN_BASE_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pDefId",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_def_id",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_DEF_ID
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pBaseType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_base_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_BASE_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pBaseTypeIfEntity",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_base_type_if_entity",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_BASE_TYPE_IF_ENTITY
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
 