%define OT_Distribution_doc
"Base class for probability distributions.

Notes
-----
In OpenTURNS a :class:`~openturns.Distribution` maps the concept of *probability distribution*."
%enddef
%feature("docstring") OT::DistributionImplementation
OT_Distribution_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeCDF_doc
"Compute the cumulative distribution function.

Parameters
----------
X : sequence of float, 2-d sequence of float
    CDF input(s).

Returns
-------
F : float, :class:`~openturns.Point`
    CDF value(s) at input(s) :math:`X`.

Notes
-----
The cumulative distribution function is defined as:

.. math::

    F_{\\vect{X}}(\\vect{x}) = \\Prob{\\bigcap_{i=1}^n X_i \\leq x_i},
                             \\quad \\vect{x} \\in \\supp{\\vect{X}}"
%enddef
%feature("docstring") OT::DistributionImplementation::computeCDF
OT_Distribution_computeCDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeCDFGradient_doc
"Compute the gradient of the cumulative distribution function.

Parameters
----------
X : sequence of float
    CDF input.

Returns
-------
dFdtheta : :class:`~openturns.Point`
    Partial derivatives of the CDF with respect to the distribution
    parameters at input :math:`X`."
%enddef
%feature("docstring") OT::DistributionImplementation::computeCDFGradient
OT_Distribution_computeCDFGradient_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeCharacteristicFunction_doc
"Compute the characteristic function.

Parameters
----------
t : float
    Characteristic function input.

Returns
-------
phi : complex
    Characteristic function value at input :math:`t`.

Notes
-----
The characteristic function is defined as:

.. math::
    \\phi_X(t) = \\mathbb{E}\\left[\\exp(- i t X)\\right],
                \\quad t \\in \\Rset

OpenTURNS features a generic implementation of the characteristic function for
all its univariate distributions (both continuous and discrete). This default
implementation might be time consuming, especially as the modulus of :math:`t` gets
high. Only some univariate distributions benefit from dedicated more efficient
implementations."
%enddef
%feature("docstring") OT::DistributionImplementation::computeCharacteristicFunction
OT_Distribution_computeCharacteristicFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeComplementaryCDF_doc
"Compute the complementary cumulative distribution function.

Parameters
----------
X : sequence of float, 2-d sequence of float
    Complementary CDF input(s).

Returns
-------
C : float, :class:`~openturns.Point`
    Complementary CDF value(s) at input(s) :math:`X`.

Notes
-----
The complementary cumulative distribution function.

.. math::

    1 - F_{\\vect{X}}(\\vect{x}) = 1 - \\Prob{\\bigcap_{i=1}^n X_i \\leq x_i}, \\quad \\vect{x} \\in \\supp{\\vect{X}}

.. warning::
    This is not the survival function (except for 1-dimensional
    distributions).

See Also
--------
computeSurvivalFunction"
%enddef
%feature("docstring") OT::DistributionImplementation::computeComplementaryCDF
OT_Distribution_computeComplementaryCDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeConditionalCDF_doc
"Compute the conditional cumulative distribution function.

Parameters
----------
Xn : float, sequence of float
    Conditional CDF input (last component).
Xcond : sequence of float, 2-d sequence of float with size :math:`n-1`
    Conditioning values for the other components.

Returns
-------
F : float, sequence of float
    Conditional CDF value(s) at input :math:`X_n`, :math:`X_{cond}`.

Notes
-----
The conditional cumulative distribution function of the last component with
respect to the other fixed components is defined as follows:

.. math::

    F_{X_n \\mid X_1, \\ldots, X_{n - 1}}(x_n) =
        \\Prob{X_n \\leq x_n \\mid X_1=x_1, \\ldots, X_{n-1}=x_{n-1}},
        \\quad x_n \\in \\supp{X_n}"
%enddef
%feature("docstring") OT::DistributionImplementation::computeConditionalCDF
OT_Distribution_computeConditionalCDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeConditionalDDF_doc
"Compute the conditional derivative density function of the last component.

With respect to the other fixed components.

Parameters
----------
Xn : float
    Conditional DDF input (last component).
Xcond : sequence of float with dimension :math:`n-1`
    Conditioning values for the other components.

Returns
-------
d : float
    Conditional DDF value at input :math:`X_n`, :math:`X_{cond}`.

See Also
--------
computeDDF, computeConditionalCDF"
%enddef
%feature("docstring") OT::DistributionImplementation::computeConditionalDDF
OT_Distribution_computeConditionalDDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeConditionalPDF_doc
"Compute the conditional probability density function.

Conditional PDF of the last component with respect to the other fixed components.

Parameters
----------
Xn : float, sequence of float
    Conditional PDF input (last component).
Xcond : sequence of float, 2-d sequence of float with size :math:`n-1`
    Conditioning values for the other components.

Returns
-------
F : float, sequence of float
    Conditional PDF value(s) at input :math:`X_n`, :math:`X_{cond}`.

See Also
--------
computePDF, computeConditionalCDF"
%enddef
%feature("docstring") OT::DistributionImplementation::computeConditionalPDF
OT_Distribution_computeConditionalPDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeConditionalQuantile_doc
"Compute the conditional quantile function of the last component.

Conditional quantile with respect to the other fixed components.

Parameters
----------
p : float, sequence of float, :math:`0 < p < 1`
    Conditional quantile function input.
Xcond : sequence of float, 2-d sequence of float with size :math:`n-1`
    Conditioning values for the other components.

Returns
-------
X1 : float
    Conditional quantile at input :math:`p`, :math:`X_{cond}`.

See Also
--------
computeQuantile, computeConditionalCDF"
%enddef
%feature("docstring") OT::DistributionImplementation::computeConditionalQuantile
OT_Distribution_computeConditionalQuantile_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeSequentialConditionalCDF_doc
"Compute the sequential conditional cumulative distribution functions.

Parameters
----------
X : sequence of float, with size :math:`d`
    Values to be taken sequentially as argument and conditioning part of the CDF.

Returns
-------
F : sequence of float
    Conditional CDF values at input.

Notes
-----
The sequential conditional cumulative distribution function is defined as follows:

.. math::

    F^{seq}_{X_1,\\ldots,X_d}(x_1,\\ldots,x_d) = \\left(\\Prob{X_n \\leq x_n \\mid X_1=x_1, \\ldots, X_{n-1}=x_{n-1}}\\right)_{i=1,\\ldots,d}

ie its :math:`n`-th component is the conditional CDF of :math:`X_n` at :math:`x_n` given that :math:`X_1=x_1,\\ldots,X_{n-1}=x_{n-1}`. For :math:`n=1` it reduces to :math:`\\Prob{X_1 \\leq x_1}`, ie the CDF of the first component at :math:`x_1`."

%enddef
%feature("docstring") OT::DistributionImplementation::computeSequentialConditionalCDF
OT_Distribution_computeSequentialConditionalCDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeSequentialConditionalDDF_doc
"Compute the sequential conditional derivative density function.

Parameters
----------
X : sequence of float, with size :math:`d`
    Values to be taken sequentially as argument and conditioning part of the DDF.

Returns
-------
ddf : sequence of float
    Conditional DDF values at input.

Notes
-----
The sequential conditional derivative density function is defined as follows:

.. math::

    ddf^{seq}_{X_1,\\ldots,X_d}(x_1,\\ldots,x_d) = \\left(\\dfrac{d^2}{d\\,x_n^2}\\Prob{X_n \\leq x_n \\mid X_1=x_1, \\ldots, 
    X_{n-1}=x_{n-1}}\\right)_{i=1,\\ldots,d}

ie its :math:`n`-th component is the conditional DDF of :math:`X_n` at :math:`x_n` given that :math:`X_1=x_1,
\\ldots,X_{n-1}=x_{n-1}`. For :math:`n=1` it reduces to :math:`\\dfrac{d^2}{d\\,x_1^2}\\Prob{X_1 \\leq x_1}`, ie the DDF of 
the first component at :math:`x_1`."

%enddef
%feature("docstring") OT::DistributionImplementation::computeSequentialConditionalDDF
OT_Distribution_computeSequentialConditionalDDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeSequentialConditionalPDF_doc
"Compute the sequential conditional probability density function.

Parameters
----------
X : sequence of float, with size :math:`d`
    Values to be taken sequentially as argument and conditioning part of the PDF.

Returns
-------
pdf : sequence of float
    Conditional PDF values at input.

Notes
-----
The sequential conditional density function is defined as follows:

.. math::

    pdf^{seq}_{X_1,\\ldots,X_d}(x_1,\\ldots,x_d) = \\left(\\dfrac{d}{d\\,x_n}\\Prob{X_n \\leq x_n \\mid X_1=x_1, \\ldots, X_{n-1}=x_{n-1}}\\right)_{i=1,\\ldots,d}

ie its :math:`n`-th component is the conditional PDF of :math:`X_n` at :math:`x_n` given that :math:`X_1=x_1,\\ldots,X_{n-1}=x_{n-1}`. For :math:`n=1` it reduces to :math:`\\dfrac{d}{d\\,x_1}\\Prob{X_1 \\leq x_1}`, ie the PDF of the first component at :math:`x_1`."

%enddef
%feature("docstring") OT::DistributionImplementation::computeSequentialConditionalPDF
OT_Distribution_computeSequentialConditionalPDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeSequentialConditionalQuantile_doc
"Compute the conditional quantile function of the last component.

Parameters
----------
q : sequence of float in :math:`[0,1]`, with size :math:`d`
    Values to be taken sequentially as the argument of the conditional quantile.

Returns
-------
Q : sequence of float
    Conditional quantiles values at input.

Notes
-----
The sequential conditional quantile function is defined as follows:

.. math::

    Q^{seq}_{X_1,\\ldots,X_d}(q_1,\\ldots,q_d) = \\left(F^{-1}(q_n|X_1=x_1,\\ldots,X_{n-1}=x_{n_1}\\right)_{i=1,\\ldots,d}

where :math:`x_1,\\ldots,x_{n-1}` are defined recursively as :math:`x_1=F_1^{-1}(q_1)` and given :math:`(x_i)_{i=1,
\\ldots,n-1}`, :math:`x_n=F^{-1}(q_n|X_1=x_1,\\ldots,X_{n-1}=x_{n_1})`: the conditioning part is the set of already 
computed conditional quantiles."

%enddef
%feature("docstring") OT::DistributionImplementation::computeSequentialConditionalQuantile
OT_Distribution_computeSequentialConditionalQuantile_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeDDF_doc
"Compute the derivative density function.

Parameters
----------
X : sequence of float, 2-d sequence of float
    PDF input(s).

Returns
-------
d : :class:`~openturns.Point`, :class:`~openturns.Sample`
    DDF value(s) at input(s) :math:`X`.

Notes
-----
The derivative density function is the gradient of the probability density
function with respect to :math:`\\vect{x}`:

.. math::

    \\vect{\\nabla}_{\\vect{x}} f_{\\vect{X}}(\\vect{x}) =
        \\Tr{\\left(\\frac{\\partial f_{\\vect{X}}(\\vect{x})}{\\partial x_i},
                  \\quad i = 1, \\ldots, n\\right)},
        \\quad \\vect{x} \\in \\supp{\\vect{X}}"
%enddef
%feature("docstring") OT::DistributionImplementation::computeDDF
OT_Distribution_computeDDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeEntropy_doc
"Compute the entropy of the distribution.

Returns
-------
e : float
    Entropy of the distribution.

Notes
-----
The entropy of a distribution is defined by:

.. math::

    \\cE_X = \\Expect{-\\log(p_X(\\vect{X}))}

Where the random vector :math:`\\vect{X}` follows the probability
distribution of interest, and :math:`p_X` is either the *probability
density function* of :math:`\\vect{X}` if it is continuous or the
*probability distribution function* if it is discrete.
"
%enddef
%feature("docstring") OT::DistributionImplementation::computeEntropy
OT_Distribution_computeEntropy_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeGeneratingFunction_doc
"Compute the probability-generating function.

Parameters
----------
z : float or complex
    Probability-generating function input.

Returns
-------
g : float
    Probability-generating function value at input :math:`X`.

Notes
-----
The probability-generating function is defined as follows:

.. math::

    G_X(z) = \\Expect{z^X}, \\quad z \\in \\Cset

This function only exists for discrete distributions. OpenTURNS implements
this method for univariate distributions only.

See Also
--------
isDiscrete"
%enddef
%feature("docstring") OT::DistributionImplementation::computeGeneratingFunction
OT_Distribution_computeGeneratingFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeLogCharacteristicFunction_doc
"Compute the logarithm of the characteristic function.

Parameters
----------
t : float
    Characteristic function input.

Returns
-------
phi : complex
    Logarithm of the characteristic function value at input :math:`t`.

Notes
-----
OpenTURNS features a generic implementation of the characteristic function for
all its univariate distributions (both continuous and discrete). This default
implementation might be time consuming, especially as the modulus of :math:`t` gets
high. Only some univariate distributions benefit from dedicated more efficient
implementations.

See Also
--------
computeCharacteristicFunction"
%enddef
%feature("docstring") OT::DistributionImplementation::computeLogCharacteristicFunction
OT_Distribution_computeLogCharacteristicFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeLogGeneratingFunction_doc
"Compute the logarithm of the probability-generating function.

Parameters
----------
z : float or complex
    Probability-generating function input.

Returns
-------
lg : float
    Logarithm of the probability-generating function value at input :math:`X`.

Notes
-----
This function only exists for discrete distributions. OpenTURNS implements
this method for univariate distributions only.

See Also
--------
isDiscrete, computeGeneratingFunction"
%enddef
%feature("docstring") OT::DistributionImplementation::computeLogGeneratingFunction
OT_Distribution_computeLogGeneratingFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeLogPDF_doc
"Compute the logarithm of the probability density function.

Parameters
----------
X : sequence of float, 2-d sequence of float
    PDF input(s).

Returns
-------
f : float, :class:`~openturns.Point`
    Logarithm of the PDF value(s) at input(s) :math:`X`."
%enddef
%feature("docstring") OT::DistributionImplementation::computeLogPDF
OT_Distribution_computeLogPDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeLogPDFGradient_doc
"Compute the gradient of the log probability density function.

Parameters
----------
X : sequence of float
    PDF input.

Returns
-------
dfdtheta : :class:`~openturns.Point`
    Partial derivatives of the logPDF with respect to the distribution
    parameters at input :math:`X`."
%enddef
%feature("docstring") OT::DistributionImplementation::computeLogPDFGradient
OT_Distribution_computeLogPDFGradient_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computePDF_doc
"Compute the probability density function.

Parameters
----------
X : sequence of float, 2-d sequence of float
    PDF input(s).

Returns
-------
f : float, :class:`~openturns.Point`
    PDF value(s) at input(s) :math:`X`.

Notes
-----
The probability density function is defined as follows:

.. math::

    f_{\\vect{X}}(\\vect{x}) = \\frac{\\partial^n F_{\\vect{X}}(\\vect{x})}
                                  {\\prod_{i=1}^n \\partial x_i},
                             \\quad \\vect{x} \\in \\supp{\\vect{X}}"
%enddef
%feature("docstring") OT::DistributionImplementation::computePDF
OT_Distribution_computePDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computePDFGradient_doc
"Compute the gradient of the probability density function.

Parameters
----------
X : sequence of float
    PDF input.

Returns
-------
dfdtheta : :class:`~openturns.Point`
    Partial derivatives of the PDF with respect to the distribution
    parameters at input :math:`X`."
%enddef
%feature("docstring") OT::DistributionImplementation::computePDFGradient
OT_Distribution_computePDFGradient_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeProbability_doc
  "Compute the interval probability.

Parameters
----------
interval : :class:`~openturns.Interval`
    An interval, possibly multivariate.

Returns
-------
P : float
    Interval probability.

Notes
-----
This computes the probability that the random vector :math:`\\vect{X}` lies in
the hyper-rectangular region formed by the vectors :math:`\\vect{a}` and
:math:`\\vect{b}`:

.. math::

    \\Prob{\\bigcap\\limits_{i=1}^n a_i < X_i \\leq b_i} =
        \\sum\\limits_{\\vect{c}} (-1)^{n(\\vect{c})}
            F_{\\vect{X}}\\left(\\vect{c}\\right)

where the sum runs over the :math:`2^n` vectors such that
:math:`\\vect{c} = \\Tr{(c_i, i = 1, \\ldots, n)}` with :math:`c_i \\in [a_i, b_i]`,
and :math:`n(\\vect{c})` is the number of components in
:math:`\\vect{c}` such that :math:`c_i = a_i`."
%enddef
%feature("docstring") OT::DistributionImplementation::computeProbability
OT_Distribution_computeProbability_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeQuantile_doc
"Compute the quantile function.

Parameters
----------
p : float (or sequence of float), :math:`0 < p < 1`
    Quantile function input (a probability).
tail : bool, optional (default=False)
    Whether p should be interpreted as the complementary probability.

Returns
-------
X : :class:`~openturns.Point` (or :class:`~openturns.Sample`)
    Quantile at probability level :math:`p`.

Notes
-----
The quantile function is also known as the inverse cumulative distribution
function:

.. math::

    Q_{\\vect{X}}(p) = F_{\\vect{X}}^{-1}(p),
                      \\quad p \\in [0; 1]"
%enddef
%feature("docstring") OT::DistributionImplementation::computeQuantile
OT_Distribution_computeQuantile_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeRadialDistributionCDF_doc
"Compute the cumulative distribution function of the squared radius.

For the underlying standard spherical distribution (for elliptical
distributions only).

Parameters
----------
r2 : float, :math:`0 \\leq r^2`
    Squared radius.

Returns
-------
F : float
    CDF value at input :math:`r^2`.

Notes
-----
This is the CDF of the sum of the squared independent, standard, identically
distributed components:

.. math::

    R^2 = \\sqrt{\\sum\\limits_{i=1}^n U_i^2}"
%enddef
%feature("docstring") OT::DistributionImplementation::computeRadialDistributionCDF
OT_Distribution_computeRadialDistributionCDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeScalarQuantile_doc
"Compute the quantile function for univariate distributions.

Parameters
----------
p : float, :math:`0 < p < 1`
    Quantile function input (a probability).

Returns
-------
X : float
    Quantile at probability level :math:`p`.

Notes
-----
The quantile function is also known as the inverse cumulative distribution
function:

.. math::

    Q_X(p) = F_X^{-1}(p), \\quad p \\in [0; 1]

See Also
--------
computeQuantile"
%enddef
%feature("docstring") OT::DistributionImplementation::computeScalarQuantile
OT_Distribution_computeScalarQuantile_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeSurvivalFunction_doc
"Compute the survival function.

Parameters
----------
x : sequence of float, 2-d sequence of float
    Survival function input(s).

Returns
-------
S : float, :class:`~openturns.Point`
    Survival function value(s) at input(s) `x`.

Notes
-----
The survival function of the random vector :math:`\\vect{X}` is defined as follows:

.. math::

 
    S_{\\vect{X}}(\\vect{x}) = \\Prob{\\bigcap_{i=1}^d X_i > x_i}
             \\quad \\forall \\vect{x} \\in \\Rset^d

.. warning::

    This is not the complementary cumulative distribution function (except for
    1-dimensional distributions).

See Also
--------
computeComplementaryCDF"
%enddef
%feature("docstring") OT::DistributionImplementation::computeSurvivalFunction
OT_Distribution_computeSurvivalFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeInverseSurvivalFunction_doc
"Compute the inverse survival function.

Parameters
----------
p : float, :math:`p \\in [0; 1]`
    Level of the survival function.

Returns
-------
x : :class:`~openturns.Point`
    Point :math:`\\vect{x}` such that :math:`S_{\\vect{X}}(\\vect{x}) = p` with iso-quantile components.

Notes
-----
The inverse survival function writes: :math:`S^{-1}(p)  =  \\vect{x}^p` where :math:`S( \\vect{x}^p) = \\Prob{\\bigcap_{i=1}^d X_i > x_i^p}`. OpenTURNS returns the point :math:`\\vect{x}^p` such that 
:math:`\\Prob{ X_1 > x_1^p}   =  \\dots = \\Prob{ X_d > x_d^p}`.

See Also
--------
computeQuantile, computeSurvivalFunction"
%enddef
%feature("docstring") OT::DistributionImplementation::computeInverseSurvivalFunction
OT_Distribution_computeInverseSurvivalFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawCDF_doc
"Draw the cumulative distribution function.

Available constructors:
    drawCDF(*x_min, x_max, pointNumber, logScale*)

    drawCDF(*lowerCorner, upperCorner, pointNbrInd, logScaleX, logScaleY*)

    drawCDF(*lowerCorner, upperCorner*)

Parameters
----------
x_min : float, optional
    The min-value of the mesh of the x-axis.
    Defaults uses the quantile associated to the probability level
    `Distribution-QMin` from the :class:`~openturns.ResourceMap`.
x_max : float, optional, :math:`x_{\\max} > x_{\\min}`
    The max-value of the mesh of the y-axis.
    Defaults uses the quantile associated to the probability level
    `Distribution-QMax` from the :class:`~openturns.ResourceMap`.
pointNumber : int
    The number of points that is used for meshing each axis.
    Defaults uses `DistributionImplementation-DefaultPointNumber` from the
    :class:`~openturns.ResourceMap`.
logScale : bool
    Flag to tell if the plot is done on a logarithmic scale. Default is *False*.
lowerCorner : sequence of float, of dimension 2, optional
    The lower corner :math:`[x_{min}, y_{min}]`.
upperCorner : sequence of float, of dimension 2, optional
    The upper corner :math:`[x_{max}, y_{max}]`.
pointNbrInd : :class:`~openturns.Indices`, of dimension 2
    Number of points that is used for meshing each axis.
logScaleX : bool
    Flag to tell if the plot is done on a logarithmic scale for X. Default is *False*.
logScaleY : bool
    Flag to tell if the plot is done on a logarithmic scale for Y. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the CDF.

Notes
-----
Only valid for univariate and bivariate distributions.

See Also
--------
computeCDF, viewer.View, ResourceMap

Examples
--------
View the CDF of a univariate distribution:

>>> import openturns as ot
>>> dist = ot.Normal()
>>> graph = dist.drawCDF()
>>> graph.setLegends(['normal cdf'])

View the iso-lines CDF of a bivariate distribution:

>>> import openturns as ot
>>> dist = ot.Normal(2)
>>> graph2 = dist.drawCDF()
>>> graph2.setLegends(['iso- normal cdf'])
>>> graph3 = dist.drawCDF([-10, -5],[5, 10], [511, 511])
"
%enddef
%feature("docstring") OT::DistributionImplementation::drawCDF
OT_Distribution_drawCDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawMarginal1DCDF_doc
"Draw the cumulative distribution function of a margin.

Parameters
----------
i : int, :math:`1 \\leq i \\leq n`
    The index of the margin of interest.
x_min : float
    The starting value that is used for meshing the x-axis.
x_max : float, :math:`x_{\\max} > x_{\\min}`
    The ending value that is used for meshing the x-axis.
n_points : int
    The number of points that is used for meshing the x-axis.
logScale : bool
    Flag to tell if the plot is done on a logarithmic scale. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the CDF of the requested margin.

See Also
--------
computeCDF, getMarginal, viewer.View, ResourceMap

Examples
--------

>>> import openturns as ot
>>> from openturns.viewer import View
>>> distribution = ot.Normal(10)
>>> graph = distribution.drawMarginal1DCDF(2, -6.0, 6.0, 100)
>>> view = View(graph)
>>> view.show()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawMarginal1DCDF
OT_Distribution_drawMarginal1DCDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeMinimumVolumeLevelSet_doc
"Compute the confidence domain with minimum volume.

Parameters
----------
alpha : float, :math:`\\alpha \\in [0,1]`
    The confidence level.

Returns
-------
levelSet : :class:`~openturns.LevelSet`
    The minimum volume domain of measure :math:`\\alpha`.

Notes
-----
We consider an absolutely continuous measure :math:`\\mu` with density function :math:`p`. 

The minimum volume confidence domain :math:`A^*_{\\alpha}` is the set of minimum volume and which measure is at least :math:`\\alpha`. It is defined by:

.. math::

    A^*_{\\alpha} = \\argmin_{A \\in \\Rset^d\\, | \\, \\mu(A) \\geq \\alpha} \\lambda(A)


where :math:`\\lambda` is the Lebesgue measure on :math:`\\Rset^d`. Under some general conditions on :math:`\\mu` (for example, no flat regions), the set  :math:`A^*_{\\alpha}` is unique and realises the minimum: :math:`\\mu(A^*_{\\alpha}) = \\alpha`. We show that :math:`A^*_{\\alpha}` writes:

.. math::

    A^*_{\\alpha} = \\{ \\vect{x} \\in \\Rset^d \\, | \\, p(\\vect{x}) \\geq p_{\\alpha} \\}

for a certain :math:`p_{\\alpha} >0`.

If we consider the random variable :math:`Y = p(\\vect{X})`, with cumulative distribution function :math:`F_Y`, then :math:`p_{\\alpha}` is defined by:

.. math::

    1-F_Y(p_{\\alpha}) = \\alpha


Thus the minimum volume domain of confidence :math:`\\alpha` is the interior of the domain which frontier is the :math:`1-\\alpha` quantile of :math:`Y`. It can be determined with simulations of :math:`Y`.

Examples
--------
Create a sample from a Normal distribution:

>>> import openturns as ot
>>> sample = ot.Normal().getSample(10)
>>> ot.ResourceMap.SetAsUnsignedInteger('DistributionFactory-DefaultBootstrapSize', 100)

Fit a Normal distribution and extract the asymptotic parameters distribution:

>>> fittedRes = ot.NormalFactory().buildEstimator(sample)
>>> paramDist = fittedRes.getParameterDistribution()

Determine the confidence region of minimum volume of the native parameters at level 0.9:

>>> levelSet = paramDist.computeMinimumVolumeLevelSet(0.9)
>>> ot.ResourceMap.Reload()
"
%enddef
%feature("docstring") OT::DistributionImplementation::computeMinimumVolumeLevelSet
OT_Distribution_computeMinimumVolumeLevelSet_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeMinimumVolumeLevelSetWithThreshold_doc
"Compute the confidence domain with minimum volume.

Refer to :func:`computeMinimumVolumeLevelSet()`

Parameters
----------
alpha : float, :math:`\\alpha \\in [0,1]`
    The confidence level.

Returns
-------
levelSet : :class:`~openturns.LevelSet`
    The minimum volume domain of measure :math:`\\alpha`.
level : float
    The value :math:`p_{\\alpha}` of the density function defining the frontier of the domain.

Examples
--------
Create a sample from a Normal distribution:

>>> import openturns as ot
>>> sample = ot.Normal().getSample(10)
>>> ot.ResourceMap.SetAsUnsignedInteger('DistributionFactory-DefaultBootstrapSize', 100)

Fit a Normal distribution and extract the asymptotic parameters distribution:

>>> fittedRes = ot.NormalFactory().buildEstimator(sample)
>>> paramDist = fittedRes.getParameterDistribution()

Determine the confidence region of minimum volume of the native parameters at level 0.9 with PDF threshold:

>>> levelSet, threshold = paramDist.computeMinimumVolumeLevelSetWithThreshold(0.9)
"
%enddef
%feature("docstring") OT::DistributionImplementation::computeMinimumVolumeLevelSetWithThreshold
OT_Distribution_computeMinimumVolumeLevelSetWithThreshold_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeMinimumVolumeInterval_doc
"Compute the confidence interval with minimum volume.

Parameters
----------
alpha : float, :math:`\\alpha \\in [0,1]`
    The confidence level.

Returns
-------
confInterval : :class:`~openturns.Interval`
    The confidence interval of level :math:`\\alpha`.

Notes
-----
We consider an absolutely continuous measure :math:`\\mu` with density function :math:`p`. 

The minimum volume confidence interval :math:`I^*_{\\alpha}` is the cartesian product :math:`I^*_{\\alpha} = [a_1, b_1] \\times \\dots \\times [a_d, b_d]` where :math:`[a_i, b_i]   = \\argmin_{I \\in \\Rset \\, | \\, \\mu_i(I) = \\beta} \\lambda_i(I)` and :math:`\\mu(I^*_{\\alpha})  =  \\alpha` with :math:`\\lambda` is the Lebesgue measure on :math:`\\Rset^d`. 

This problem resorts to solving :math:`d` univariate non linear equations: for a fixed value :math:`\\beta`, we find each intervals :math:`[a_i, b_i]` such that:

.. math::
    :nowrap:

    \\begin{eqnarray*}
    F_i(b_i) - F_i(a_i) & = & \\beta \\\\
    p_i(b_i) & = & p_i(a_i)
    \\end{eqnarray*}

which consists of finding the bound :math:`a_i` such that:

.. math::

    p_i(a_i) =  p_i(F_i^{-1}(\\beta + F_i(a_i)))

To find :math:`\\beta`, we use the Brent algorithm:  :math:`\\mu([\\vect{a}(\\beta); \\vect{b}(\\beta)] = g(\\beta) = \\alpha` with :math:`g` a non linear function.

Examples
--------
Create a sample from a Normal distribution:

>>> import openturns as ot
>>> sample = ot.Normal().getSample(10)
>>> ot.ResourceMap.SetAsUnsignedInteger('DistributionFactory-DefaultBootstrapSize', 100)

Fit a Normal distribution and extract the asymptotic parameters distribution:

>>> fittedRes = ot.NormalFactory().buildEstimator(sample)
>>> paramDist = fittedRes.getParameterDistribution()

Determine the confidence interval of the native parameters at level 0.9 with minimum volume:

>>> ot.ResourceMap.SetAsUnsignedInteger('Distribution-MinimumVolumeLevelSetSamplingSize', 1000)
>>> confInt = paramDist.computeMinimumVolumeInterval(0.9)
>>> ot.ResourceMap.Reload()
"
%enddef
%feature("docstring") OT::DistributionImplementation::computeMinimumVolumeInterval
OT_Distribution_computeMinimumVolumeInterval_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeMinimumVolumeIntervalWithMarginalProbability_doc
"Compute the confidence interval with minimum volume.

Refer to :func:`computeMinimumVolumeInterval()`

Parameters
----------
alpha : float, :math:`\\alpha \\in [0,1]`
    The confidence level.

Returns
-------
confInterval : :class:`~openturns.Interval`
    The confidence interval of level :math:`\\alpha`.
marginalProb : float
    The value :math:`\\beta` which is the common marginal probability of each marginal interval.

Examples
--------
Create a sample from a Normal distribution:

>>> import openturns as ot
>>> sample = ot.Normal().getSample(10)
>>> ot.ResourceMap.SetAsUnsignedInteger('DistributionFactory-DefaultBootstrapSize', 100)

Fit a Normal distribution and extract the asymptotic parameters distribution:

>>> fittedRes = ot.NormalFactory().buildEstimator(sample)
>>> paramDist = fittedRes.getParameterDistribution()

Determine the confidence interval of the native parameters at level 0.9 with minimum volume:

>>> ot.ResourceMap.SetAsUnsignedInteger('Distribution-MinimumVolumeLevelSetSamplingSize', 1000)
>>> confInt, marginalProb = paramDist.computeMinimumVolumeIntervalWithMarginalProbability(0.9)
>>> ot.ResourceMap.Reload()
"
%enddef
%feature("docstring") OT::DistributionImplementation::computeMinimumVolumeIntervalWithMarginalProbability
OT_Distribution_computeMinimumVolumeIntervalWithMarginalProbability_doc

// ---------------------------------------------------------------------


%define OT_Distribution_computeBilateralConfidenceInterval_doc
"Compute a bilateral confidence interval.

Parameters
----------
alpha : float, :math:`\\alpha \\in [0,1]`
    The confidence level.

Returns
-------
confInterval : :class:`~openturns.Interval`
    The confidence interval of level :math:`\\alpha`.

Notes
-----
We consider an absolutely continuous measure :math:`\\mu` with density function :math:`p`. 

The bilateral confidence interval :math:`I^*_{\\alpha}` is the cartesian product :math:`I^*_{\\alpha} = [a_1, b_1] \\times \\dots \\times [a_d, b_d]` where :math:`a_i = F_i^{-1}((1-\\beta)/2)` and :math:`b_i = F_i^{-1}((1+\\beta)/2)` for all :math:`i` and which verifies :math:`\\mu(I^*_{\\alpha}) = \\alpha`. 

Examples
--------
Create a sample from a Normal distribution:

>>> import openturns as ot
>>> sample = ot.Normal().getSample(10)
>>> ot.ResourceMap.SetAsUnsignedInteger('DistributionFactory-DefaultBootstrapSize', 100)

Fit a Normal distribution and extract the asymptotic parameters distribution:

>>> fittedRes = ot.NormalFactory().buildEstimator(sample)
>>> paramDist = fittedRes.getParameterDistribution()

Determine the bilateral confidence interval at level 0.9:

>>> confInt = paramDist.computeBilateralConfidenceInterval(0.9)"
%enddef
%feature("docstring") OT::DistributionImplementation::computeBilateralConfidenceInterval
OT_Distribution_computeBilateralConfidenceInterval_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeBilateralConfidenceIntervalWithMarginalProbability_doc
"Compute a bilateral confidence interval.

Refer to :func:`computeBilateralConfidenceInterval()`

Parameters
----------
alpha : float, :math:`\\alpha \\in [0,1]`
    The confidence level.

Returns
-------
confInterval : :class:`~openturns.Interval`
    The confidence interval of level :math:`\\alpha`.
marginalProb : float
    The value :math:`\\beta` which is the common marginal probability of each marginal interval.

Examples
--------
Create a sample from a Normal distribution:

>>> import openturns as ot
>>> sample = ot.Normal().getSample(10)
>>> ot.ResourceMap.SetAsUnsignedInteger('DistributionFactory-DefaultBootstrapSize', 100)

Fit a Normal distribution and extract the asymptotic parameters distribution:

>>> fittedRes = ot.NormalFactory().buildEstimator(sample)
>>> paramDist = fittedRes.getParameterDistribution()

Determine the bilateral confidence interval at level 0.9 with marginal probability:

>>> confInt, marginalProb = paramDist.computeBilateralConfidenceIntervalWithMarginalProbability(0.9)"
%enddef
%feature("docstring") OT::DistributionImplementation::computeBilateralConfidenceIntervalWithMarginalProbability
OT_Distribution_computeBilateralConfidenceIntervalWithMarginalProbability_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeUnilateralConfidenceInterval_doc
"Compute a unilateral confidence interval.

Parameters
----------
alpha : float, :math:`\\alpha \\in [0,1]`
    The confidence level.
tail : boolean
    `True` indicates the interval is bounded by an lower value.
    `False` indicates the interval is bounded by an upper value.
    Default value is `False`.

Returns
-------
confInterval : :class:`~openturns.Interval`
    The unilateral confidence interval of level :math:`\\alpha`.

Notes
-----
We consider an absolutely continuous measure :math:`\\mu`.

The left unilateral confidence interval :math:`I^*_{\\alpha}` is the cartesian product :math:`I^*_{\\alpha} = ]-\\infty, b_1] \\times \\dots \\times ]-\\infty, b_d]` where :math:`b_i = F_i^{-1}(\\beta)` for all :math:`i` and which verifies :math:`\\mu(I^*_{\\alpha}) = \\alpha`. 
It means that :math:`\\vect{b}` is the quantile of level :math:`\\alpha` of the measure :math:`\\mu`, with iso-quantile components.

The right unilateral confidence interval :math:`I^*_{\\alpha}` is the cartesian product :math:`I^*_{\\alpha} = ]a_1; +\\infty[ \\times \\dots \\times ]a_d; +\\infty[` where :math:`a_i = F_i^{-1}(1-\\beta)` for all :math:`i` and which verifies :math:`\\mu(I^*_{\\alpha}) = \\alpha`. 
It means that :math:`S_{\\mu}^{-1}(\\vect{a}) = \\alpha` with iso-quantile components, where :math:`S_{\\mu}` is the survival function of the measure :math:`\\mu`.

Examples
--------
Create a sample from a Normal distribution:

>>> import openturns as ot
>>> sample = ot.Normal().getSample(10)
>>> ot.ResourceMap.SetAsUnsignedInteger('DistributionFactory-DefaultBootstrapSize', 100)

Fit a Normal distribution and extract the asymptotic parameters distribution: 

>>> fittedRes = ot.NormalFactory().buildEstimator(sample)
>>> paramDist = fittedRes.getParameterDistribution()

Determine the right unilateral confidence interval at level 0.9:

>>> confInt = paramDist.computeUnilateralConfidenceInterval(0.9)

Determine the left unilateral confidence interval at level 0.9:

>>> confInt = paramDist.computeUnilateralConfidenceInterval(0.9, True)
"
%enddef
%feature("docstring") OT::DistributionImplementation::computeUnilateralConfidenceInterval
OT_Distribution_computeUnilateralConfidenceInterval_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeUnilateralConfidenceIntervalWithMarginalProbability_doc
"Compute a unilateral confidence interval.

Refer to :func:`computeUnilateralConfidenceInterval()`

Parameters
----------
alpha : float, :math:`\\alpha \\in [0,1]`
    The confidence level.
tail : boolean
    `True` indicates the interval is bounded by an lower value.
    `False` indicates the interval is bounded by an upper value.
    Default value is `False`.

Returns
-------
confInterval : :class:`~openturns.Interval`
    The unilateral confidence interval of level :math:`\\alpha`.
marginalProb : float
    The value :math:`\\beta` which is the common marginal probability of each marginal interval.

Examples
--------
Create a sample from a Normal distribution:

>>> import openturns as ot
>>> sample = ot.Normal().getSample(10)
>>> ot.ResourceMap.SetAsUnsignedInteger('DistributionFactory-DefaultBootstrapSize', 100)

Fit a Normal distribution and extract the asymptotic parameters distribution: 

>>> fittedRes = ot.NormalFactory().buildEstimator(sample)
>>> paramDist = fittedRes.getParameterDistribution()

Determine the right unilateral confidence interval at level 0.9:

>>> confInt, marginalProb = paramDist.computeUnilateralConfidenceIntervalWithMarginalProbability(0.9, False)

Determine the left unilateral confidence interval at level 0.9:

>>> confInt, marginalProb = paramDist.computeUnilateralConfidenceIntervalWithMarginalProbability(0.9, True)
>>> ot.ResourceMap.Reload()
"
%enddef
%feature("docstring") OT::DistributionImplementation::computeUnilateralConfidenceIntervalWithMarginalProbability
OT_Distribution_computeUnilateralConfidenceIntervalWithMarginalProbability_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawMarginal1DPDF_doc
"Draw the probability density function of a margin.

Parameters
----------
i : int, :math:`1 \\leq i \\leq n`
    The index of the margin of interest.
x_min : float
    The starting value that is used for meshing the x-axis.
x_max : float, :math:`x_{\\max} > x_{\\min}`
    The ending value that is used for meshing the x-axis.
n_points : int
    The number of points that is used for meshing the x-axis.
logScale : bool
    Flag to tell if the plot is done on a logarithmic scale. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the PDF of the requested margin.

See Also
--------
computePDF, getMarginal, viewer.View, ResourceMap

Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> distribution = ot.Normal(10)
>>> graph = distribution.drawMarginal1DPDF(2, -6.0, 6.0, 100)
>>> view = View(graph)
>>> view.show()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawMarginal1DPDF
OT_Distribution_drawMarginal1DPDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawMarginal1DLogPDF_doc
"Draw the log-probability density function of a margin.

Parameters
----------
i : int, :math:`1 \\leq i \\leq n`
    The index of the margin of interest.
x_min : float
    The starting value that is used for meshing the x-axis.
x_max : float, :math:`x_{\\max} > x_{\\min}`
    The ending value that is used for meshing the x-axis.
n_points : int
    The number of points that is used for meshing the x-axis.
logScale : bool
    Flag to tell if the plot is done on a logarithmic scale. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the log-PDF of the requested margin.

See Also
--------
computeLogPDF, getMarginal, viewer.View, ResourceMap

Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> distribution = ot.Normal(10)
>>> graph = distribution.drawMarginal1DLogPDF(2, -6.0, 6.0, 100)
>>> view = View(graph)
>>> view.show()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawMarginal1DLogPDF
OT_Distribution_drawMarginal1DLogPDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawMarginal2DCDF_doc
"Draw the cumulative distribution function of a couple of margins.

Parameters
----------
i : int, :math:`1 \\leq i \\leq n`
    The index of the first margin of interest.
j : int, :math:`1 \\leq i \\neq j \\leq n`
    The index of the second margin of interest.
x_min : list of 2 floats
    The starting values that are used for meshing the x- and y- axes.
x_max : list of 2 floats, :math:`x_{\\max} > x_{\\min}`
    The ending values that are used for meshing the x- and y- axes.
n_points : list of 2 ints
    The number of points that are used for meshing the x- and y- axes.
logScaleX : bool
    Flag to tell if the plot is done on a logarithmic scale for X. Default is *False*.
logScaleY : bool
    Flag to tell if the plot is done on a logarithmic scale for Y. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the marginal CDF of the requested couple of
    margins.

See Also
--------
computeCDF, getMarginal, viewer.View, ResourceMap

Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> distribution = ot.Normal(10)
>>> graph = distribution.drawMarginal2DCDF(2, 3, [-6.0] * 2, [6.0] * 2, [100] * 2)
>>> view = View(graph)
>>> view.show()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawMarginal2DCDF
OT_Distribution_drawMarginal2DCDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawMarginal2DPDF_doc
"Draw the probability density function of a couple of margins.

Parameters
----------
i : int, :math:`1 \\leq i \\leq n`
    The index of the first margin of interest.
j : int, :math:`1 \\leq i \\neq j \\leq n`
    The index of the second margin of interest.
x_min : list of 2 floats
    The starting values that are used for meshing the x- and y- axes.
x_max : list of 2 floats, :math:`x_{\\max} > x_{\\min}`
    The ending values that are used for meshing the x- and y- axes.
n_points : list of 2 ints
    The number of points that are used for meshing the x- and y- axes.
logScaleX : bool
    Flag to tell if the plot is done on a logarithmic scale for X. Default is *False*.
logScaleY : bool
    Flag to tell if the plot is done on a logarithmic scale for Y. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the marginal PDF of the requested couple of
    margins.

See Also
--------
computePDF, getMarginal, viewer.View, ResourceMap

Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> distribution = ot.Normal(10)
>>> graph = distribution.drawMarginal2DPDF(2, 3, [-6.0] * 2, [6.0] * 2, [100] * 2)
>>> view = View(graph)
>>> view.show()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawMarginal2DPDF
OT_Distribution_drawMarginal2DPDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawMarginal2DLogPDF_doc
"Draw the log-probability density function of a couple of margins.

Parameters
----------
i : int, :math:`1 \\leq i \\leq n`
    The index of the first margin of interest.
j : int, :math:`1 \\leq i \\neq j \\leq n`
    The index of the second margin of interest.
x_min : list of 2 floats
    The starting values that are used for meshing the x- and y- axes.
x_max : list of 2 floats, :math:`x_{\\max} > x_{\\min}`
    The ending values that are used for meshing the x- and y- axes.
n_points : list of 2 ints
    The number of points that are used for meshing the x- and y- axes.
logScaleX : bool
    Flag to tell if the plot is done on a logarithmic scale for X. Default is *False*.
logScaleY : bool
    Flag to tell if the plot is done on a logarithmic scale for Y. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the marginal log-PDF of the requested couple of
    margins.

See Also
--------
computeLogPDF, getMarginal, viewer.View, ResourceMap

Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> distribution = ot.Normal(10)
>>> graph = distribution.drawMarginal2DLogPDF(2, 3, [-6.0] * 2, [6.0] * 2, [100] * 2)
>>> view = View(graph)
>>> view.show()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawMarginal2DLogPDF
OT_Distribution_drawMarginal2DLogPDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawPDF_doc
"Draw the graph or of iso-lines of probability density function.

Available constructors:
    drawPDF(*x_min, x_max, pointNumber, logScale*)

    drawPDF(*lowerCorner, upperCorner, pointNbrInd, logScaleX, logScaleY*)

    drawPDF(*lowerCorner, upperCorner*)

Parameters
----------
x_min : float, optional
    The min-value of the mesh of the x-axis.
    Defaults uses the quantile associated to the probability level
    `Distribution-QMin` from the :class:`~openturns.ResourceMap`.
x_max : float, optional, :math:`x_{\\max} > x_{\\min}`
    The max-value of the mesh of the y-axis.
    Defaults uses the quantile associated to the probability level
    `Distribution-QMax` from the :class:`~openturns.ResourceMap`.
pointNumber : int
    The number of points that is used for meshing each axis.
    Defaults uses `DistributionImplementation-DefaultPointNumber` from the
    :class:`~openturns.ResourceMap`.
logScale : bool
    Flag to tell if the plot is done on a logarithmic scale. Default is *False*.
lowerCorner : sequence of float, of dimension 2, optional
    The lower corner :math:`[x_{min}, y_{min}]`.
upperCorner : sequence of float, of dimension 2, optional
    The upper corner :math:`[x_{max}, y_{max}]`.
pointNbrInd : :class:`~openturns.Indices`, of dimension 2
    Number of points that is used for meshing each axis.
logScaleX : bool
    Flag to tell if the plot is done on a logarithmic scale for X. Default is *False*.
logScaleY : bool
    Flag to tell if the plot is done on a logarithmic scale for Y. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the PDF or its iso_lines.

Notes
-----
Only valid for univariate and bivariate distributions.

See Also
--------
computePDF, viewer.View, ResourceMap

Examples
--------
View the PDF of a univariate distribution:

>>> import openturns as ot
>>> dist = ot.Normal()
>>> graph = dist.drawPDF()
>>> graph.setLegends(['normal pdf'])

View the iso-lines PDF of a bivariate distribution:

>>> import openturns as ot
>>> dist = ot.Normal(2)
>>> graph2 = dist.drawPDF()
>>> graph2.setLegends(['iso- normal pdf'])
>>> graph3 = dist.drawPDF([-10, -5],[5, 10], [511, 511])
"
%enddef
%feature("docstring") OT::DistributionImplementation::drawPDF
OT_Distribution_drawPDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawLogPDF_doc
"Draw the graph or of iso-lines of log-probability density function.

Available constructors:
    drawLogPDF(*x_min, x_max, pointNumber, logScale*)

    drawLogPDF(*lowerCorner, upperCorner, pointNbrInd, logScaleX, logScaleY*)

    drawLogPDF(*lowerCorner, upperCorner*)

Parameters
----------
x_min : float, optional
    The min-value of the mesh of the x-axis.
    Defaults uses the quantile associated to the probability level
    `Distribution-QMin` from the :class:`~openturns.ResourceMap`.
x_max : float, optional, :math:`x_{\\max} > x_{\\min}`
    The max-value of the mesh of the y-axis.
    Defaults uses the quantile associated to the probability level
    `Distribution-QMax` from the :class:`~openturns.ResourceMap`.
pointNumber : int
    The number of points that is used for meshing each axis.
    Defaults uses `DistributionImplementation-DefaultPointNumber` from the
    :class:`~openturns.ResourceMap`.
logScale : bool
    Flag to tell if the plot is done on a logarithmic scale. Default is *False*.
lowerCorner : sequence of float, of dimension 2, optional
    The lower corner :math:`[x_{min}, y_{min}]`.
upperCorner : sequence of float, of dimension 2, optional
    The upper corner :math:`[x_{max}, y_{max}]`.
pointNbrInd : :class:`~openturns.Indices`, of dimension 2
    Number of points that is used for meshing each axis.
logScaleX : bool
    Flag to tell if the plot is done on a logarithmic scale for X. Default is *False*.
logScaleY : bool
    Flag to tell if the plot is done on a logarithmic scale for Y. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the log-PDF or its iso_lines.

Notes
-----
Only valid for univariate and bivariate distributions.

See Also
--------
computeLogPDF, viewer.View, ResourceMap

Examples
--------
View the log-PDF of a univariate distribution:

>>> import openturns as ot
>>> dist = ot.Normal()
>>> graph = dist.drawLogPDF()
>>> graph.setLegends(['normal log-pdf'])

View the iso-lines log-PDF of a bivariate distribution:

>>> import openturns as ot
>>> dist = ot.Normal(2)
>>> graph2 = dist.drawLogPDF()
>>> graph2.setLegends(['iso- normal pdf'])
>>> graph3 = dist.drawLogPDF([-10, -5],[5, 10], [511, 511])
"
%enddef
%feature("docstring") OT::DistributionImplementation::drawLogPDF
OT_Distribution_drawLogPDF_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawQuantile_doc
"Draw the quantile function.

Parameters
----------
q_min : float, in :math:`[0,1]`
    The min value of the mesh of the x-axis.
q_max : float, in :math:`[0,1]`
    The max value of the mesh of the x-axis.
n_points : int, optional
    The number of points that is used for meshing the quantile curve.
    Defaults uses `DistributionImplementation-DefaultPointNumber` from the
    :class:`~openturns.ResourceMap`.
logScale : bool
    Flag to tell if the plot is done on a logarithmic scale. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the quantile function.

Notes
-----
This is implemented for univariate and bivariate distributions only.
In the case of bivariate distributions, defined by its CDF :math:`F` and its marginals :math:`(F_1, F_2)`, the quantile of order :math:`q` is the point :math:`(F_1(u),F_2(u))` defined by

.. math::

    F(F_1(u), F_2(u)) = q


See Also
--------
computeQuantile, viewer.View, ResourceMap

Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> distribution = ot.Normal()
>>> graph = distribution.drawQuantile()
>>> view = View(graph)
>>> view.show()
>>> distribution = ot.ComposedDistribution([ot.Normal(), ot.Exponential(1.0)], ot.ClaytonCopula(0.5))
>>> graph = distribution.drawQuantile()
>>> view = View(graph)
>>> view.show()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawQuantile
OT_Distribution_drawQuantile_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getCDFEpsilon_doc
"Accessor to the CDF computation precision.

Returns
-------
CDFEpsilon : float
    CDF computation precision."
%enddef
%feature("docstring") OT::DistributionImplementation::getCDFEpsilon
OT_Distribution_getCDFEpsilon_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getCentralMoment_doc
"Accessor to the componentwise central moments.

Parameters
----------
k : int
    The order of the central moment.

Returns
-------
m : :class:`~openturns.Point`
    Componentwise central moment of order :math:`k`.

Notes
-----
Central moments are centered with respect to the first-order moment:

.. math::

    \\vect{m}^{(k)}_0 = \\Tr{\\left(\\Expect{\\left(X_i - \\mu_i\\right)^k},
                                 \\quad i = 1, \\ldots, n\\right)}

See Also
--------
getMoment"
%enddef
%feature("docstring") OT::DistributionImplementation::getCentralMoment
OT_Distribution_getCentralMoment_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getShiftedMoment_doc
"Accessor to the componentwise shifted moments.

Parameters
----------
k : int
    The order of the shifted moment.
shift : sequence of float
    The shift of the moment.

Returns
-------
m : :class:`~openturns.Point`
    Componentwise central moment of order :math:`k`.

Notes
-----
The moments are centered with respect to the given shift :math:`\\vect{s}`:

.. math::

    \\vect{m}^{(k)}_0 = \\Tr{\\left(\\Expect{\\left(X_i - s_i\\right)^k},
                                 \\quad i = 1, \\ldots, n\\right)}

See Also
--------
getMoment, getCentralMoment"
%enddef
%feature("docstring") OT::DistributionImplementation::getShiftedMoment
OT_Distribution_getShiftedMoment_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getCholesky_doc
"Accessor to the Cholesky factor of the covariance matrix.

Returns
-------
L : :class:`~openturns.SquareMatrix`
    Cholesky factor of the covariance matrix.

See Also
--------
getCovariance"
%enddef
  %feature("docstring") OT::DistributionImplementation::getCholesky
OT_Distribution_getCholesky_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getCopula_doc
"Accessor to the copula of the distribution.

Returns
-------
C : :class:`~openturns.Distribution`
    Copula of the distribution.

See Also
--------
ComposedDistribution"
%enddef
%feature("docstring") OT::DistributionImplementation::getCopula
OT_Distribution_getCopula_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getCorrelation_doc
" Accessor to the correlation matrix.

Returns
-------
R : :class:`~openturns.CorrelationMatrix`
    The correlation matrix of the distribution.

Notes
-----

.. math::

   R_{ij} = \\dfrac{C_{ij}}{\\sigma_i\\sigma_j}

where the :math:`(\\sigma_i)` are the margin standard deviations
and :math:`(C_{ij})` the covariance coefficients.
"
%enddef
%feature("docstring") OT::DistributionImplementation::getCorrelation
OT_Distribution_getCorrelation_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getCovariance_doc
"Accessor to the covariance matrix.

Returns
-------
Sigma : :class:`~openturns.CovarianceMatrix`
    Covariance matrix.

Notes
-----
The covariance is the second-order central moment. It is defined as:

.. math::

    \\mat{\\Sigma} & = \\Cov{\\vect{X}} \\\\
                 & = \\Expect{\\left(\\vect{X} - \\vect{\\mu}\\right)
                             \\Tr{\\left(\\vect{X} - \\vect{\\mu}\\right)}}"
%enddef
%feature("docstring") OT::DistributionImplementation::getCovariance
OT_Distribution_getCovariance_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getDescription_doc
"Accessor to the componentwise description.

Returns
-------
description : :class:`~openturns.Description`
    Description of the components of the distribution.

See Also
--------
setDescription"
%enddef
%feature("docstring") OT::DistributionImplementation::getDescription
OT_Distribution_getDescription_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getDimension_doc
"Accessor to the dimension of the distribution.

Returns
-------
n : int
    The number of components in the distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::getDimension
OT_Distribution_getDimension_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getDispersionIndicator_doc
"Dispersion indicator accessor.

Defines a generic metric of the dispersion. When the standard deviation is not
defined it falls back to the interquartile.
Only available for 1-d distributions.

Returns
-------
dispersion : float
    Standard deviation or interquartile."
%enddef
%feature("docstring") OT::DistributionImplementation::getDispersionIndicator
OT_Distribution_getDispersionIndicator_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getInverseCholesky_doc
"Accessor to the inverse Cholesky factor of the covariance matrix.

Returns
-------
Linv : :class:`~openturns.SquareMatrix`
    Inverse Cholesky factor of the covariance matrix.

See also
--------
getCholesky"
%enddef
%feature("docstring") OT::DistributionImplementation::getInverseCholesky
OT_Distribution_getInverseCholesky_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getInverseIsoProbabilisticTransformation_doc
"Accessor to the inverse iso-probabilistic transformation.

Returns
-------
Tinv : :class:`~openturns.Function`
    Inverse iso-probabilistic transformation.

Notes
-----
The inverse iso-probabilistic transformation is defined as follows:

.. math::

    T^{-1}: \\left|\\begin{array}{rcl}
                \\Rset^n & \\rightarrow & \\supp{\\vect{X}} \\\\
                \\vect{u} & \\mapsto & \\vect{x}
            \\end{array}\\right.

See also
--------
getIsoProbabilisticTransformation"
%enddef
%feature("docstring") OT::DistributionImplementation::getInverseIsoProbabilisticTransformation
OT_Distribution_getInverseIsoProbabilisticTransformation_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getIsoProbabilisticTransformation_doc
"Accessor to the iso-probabilistic transformation.

Refer to :ref:`isoprobabilistic_transformation`.

Returns
-------
T : :class:`~openturns.Function`
    Iso-probabilistic transformation.

Notes
-----
The iso-probabilistic transformation is defined as follows:

.. math::

    T: \\left|\\begin{array}{rcl}
            \\supp{\\vect{X}} & \\rightarrow & \\Rset^n \\\\
            \\vect{x} & \\mapsto & \\vect{u}
       \\end{array}\\right.

An iso-probabilistic transformation is a *diffeomorphism* [#diff]_ from
:math:`\\supp{\\vect{X}}` to :math:`\\Rset^d` that maps realizations
:math:`\\vect{x}` of a random vector :math:`\\vect{X}` into realizations
:math:`\\vect{y}` of another random vector :math:`\\vect{Y}` while
preserving probabilities. It is hence defined so that it satisfies:

.. math::
    :nowrap:

    \\begin{eqnarray*}
        \\Prob{\\bigcap_{i=1}^d X_i \\leq x_i}
            & = & \\Prob{\\bigcap_{i=1}^d Y_i \\leq y_i} \\\\
        F_{\\vect{X}}(\\vect{x})
            & = & F_{\\vect{Y}}(\\vect{y})
    \\end{eqnarray*}

**The present** implementation of the iso-probabilistic transformation maps
realizations :math:`\\vect{x}` into realizations :math:`\\vect{u}` of a
random vector :math:`\\vect{U}` with *spherical distribution* [#spherical]_.
To be more specific:

    - if the distribution is elliptical, then the transformed distribution is
      simply made spherical using the **Nataf (linear) transformation**.
    - if the distribution has an elliptical Copula, then the transformed
      distribution is made spherical using the **generalized Nataf
      transformation**.
    - otherwise, the transformed distribution is the standard multivariate
      Normal distribution and is obtained by means of the **Rosenblatt
      transformation**.

.. [#diff] A differentiable map :math:`f` is called a *diffeomorphism* if it
    is a bijection and its inverse :math:`f^{-1}` is differentiable as well.
    Hence, the iso-probabilistic transformation implements a gradient (and
    even a Hessian).

.. [#spherical] A distribution is said to be *spherical* if it is invariant by
    rotation. Mathematically, :math:`\\vect{U}` has a spherical distribution
    if:

    .. math::

        \\mat{Q}\\,\\vect{U} \\sim \\vect{U},
        \\quad \\forall \\mat{Q} \\in \\cO_n(\\Rset)

See also
--------
getInverseIsoProbabilisticTransformation, isElliptical, hasEllipticalCopula"
%enddef
%feature("docstring") OT::DistributionImplementation::getIsoProbabilisticTransformation
OT_Distribution_getIsoProbabilisticTransformation_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getKendallTau_doc
"Accessor to the Kendall coefficients matrix.

Returns
-------
tau: :class:`~openturns.SquareMatrix`
    Kendall coefficients matrix.

Notes
-----
The Kendall coefficients matrix is defined as:

.. math::

    \\mat{\\tau} = \\Big[& \\Prob{X_i < x_i \\cap X_j < x_j
                              \\cup
                              X_i > x_i \\cap X_j > x_j} \\\\
                      & - \\Prob{X_i < x_i \\cap X_j > x_j
                                \\cup
                                X_i > x_i \\cap X_j < x_j},
                      \\quad i,j = 1, \\ldots, n\\Big]

See Also
--------
getSpearmanCorrelation"
%enddef
%feature("docstring") OT::DistributionImplementation::getKendallTau
OT_Distribution_getKendallTau_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getKurtosis_doc
"Accessor to the componentwise kurtosis.

Returns
-------
k : :class:`~openturns.Point`
    Componentwise kurtosis.

Notes
-----
The kurtosis is the fourth-order central moment standardized by the standard deviation:

.. math::

    \\vect{\\kappa} = \\Tr{\\left(\\Expect{\\left(\\frac{X_i - \\mu_i}
                                                 {\\sigma_i}\\right)^4},
                              \\quad i = 1, \\ldots, n\\right)}"
%enddef
%feature("docstring") OT::DistributionImplementation::getKurtosis
OT_Distribution_getKurtosis_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getMarginal_doc
"Accessor to marginal distributions.

Parameters
----------
i : int or list of ints, :math:`1 \\leq i \\leq n`
    Component(s) indice(s).

Returns
-------
distribution : :class:`~openturns.Distribution`
    The marginal distribution of the selected component(s)."
%enddef
%feature("docstring") OT::DistributionImplementation::getMarginal
OT_Distribution_getMarginal_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getMean_doc
"Accessor to the mean.

Returns
-------
k : :class:`~openturns.Point`
    Mean.

Notes
-----
The mean is the first-order moment:

.. math::

    \\vect{\\mu} = \\Tr{\\left(\\Expect{X_i}, \\quad i = 1, \\ldots, n\\right)}"
%enddef
%feature("docstring") OT::DistributionImplementation::getMean
OT_Distribution_getMean_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getMoment_doc
"Accessor to the componentwise moments.

Parameters
----------
k : int
    The order of the moment.

Returns
-------
m : :class:`~openturns.Point`
    Componentwise moment of order :math:`k`.

Notes
-----
The componentwise moment of order :math:`k` is defined as:

.. math::

    \\vect{m}^{(k)} = \\Tr{\\left(\\Expect{X_i^k}, \\quad i = 1, \\ldots, n\\right)}"
%enddef
%feature("docstring") OT::DistributionImplementation::getMoment
OT_Distribution_getMoment_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getPDFEpsilon_doc
"Accessor to the PDF computation precision.

Returns
-------
PDFEpsilon : float
    PDF computation precision."
%enddef
%feature("docstring") OT::DistributionImplementation::getPDFEpsilon
OT_Distribution_getPDFEpsilon_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getParametersCollection_doc
"Accessor to the parameter of the distribution.

Returns
-------
parameters : :class:`~openturns.PointWithDescription`
    Dictionary-like object with parameters names and values."
%enddef
%feature("docstring") OT::DistributionImplementation::getParametersCollection
OT_Distribution_getParametersCollection_doc

// ---------------------------------------------------------------------

%define OT_Distribution_setParameter_doc
"Accessor to the parameter of the distribution.

Parameters
----------
parameter : sequence of float
    Parameter values."
%enddef
%feature("docstring") OT::DistributionImplementation::setParameter
OT_Distribution_setParameter_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getParameter_doc
"Accessor to the parameter of the distribution.

Returns
-------
parameter : :class:`~openturns.Point`
    Parameter values."
%enddef
%feature("docstring") OT::DistributionImplementation::getParameter
OT_Distribution_getParameter_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getParameterDescription_doc
"Accessor to the parameter description of the distribution.

Returns
-------
description : :class:`~openturns.Description`
    Parameter names."
%enddef
%feature("docstring") OT::DistributionImplementation::getParameterDescription
OT_Distribution_getParameterDescription_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getParameterDimension_doc
"Accessor to the number of parameters in the distribution.

Returns
-------
n_parameters : int
    Number of parameters in the distribution.

See Also
--------
getParametersCollection"
%enddef
%feature("docstring") OT::DistributionImplementation::getParameterDimension
OT_Distribution_getParameterDimension_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getPearsonCorrelation_doc
"Accessor to the Pearson correlation matrix.

Returns
-------
R : :class:`~openturns.CorrelationMatrix`
    Pearson's correlation matrix.

See Also
--------
getCovariance

Notes
-----
Pearson's correlation is defined as the normalized covariance matrix:

.. math::

    \\mat{\\rho} & = \\left[\\frac{\\Cov{X_i, X_j}}{\\sqrt{\\Var{X_i}\\Var{X_j}}},
                         \\quad i,j = 1, \\ldots, n\\right] \\\\
               & = \\left[\\frac{\\Sigma_{i,j}}{\\sqrt{\\Sigma_{i,i}\\Sigma_{j,j}}},
                         \\quad i,j = 1, \\ldots, n\\right]"
%enddef
%feature("docstring") OT::DistributionImplementation::getPearsonCorrelation
OT_Distribution_getPearsonCorrelation_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getPositionIndicator_doc
"Position indicator accessor.

Defines a generic metric of the position. When the mean is not defined it falls
back to the median.
Available only for 1-d distributions.

Returns
-------
position : float
    Mean or median of the distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::getPositionIndicator
OT_Distribution_getPositionIndicator_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getRange_doc
"Accessor to the range of the distribution.

Returns
-------
range : :class:`~openturns.Interval`
    Range of the distribution.

Notes
-----
The *mathematical* range is the smallest closed interval outside of which the
PDF is zero. The *numerical* range is the interval outside of which the PDF is
rounded to zero in double precision.

See Also
--------
getSupport"
%enddef
%feature("docstring") OT::DistributionImplementation::getRange
OT_Distribution_getRange_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getRealization_doc
"Accessor to a pseudo-random realization from the distribution.

Refer to :ref:`distribution_realization`.

Returns
-------
point : :class:`~openturns.Point`
    A pseudo-random realization of the distribution.

See Also
--------
getSample, getSampleByInversion, getSampleByQMC, RandomGenerator"
%enddef
%feature("docstring") OT::DistributionImplementation::getRealization
OT_Distribution_getRealization_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getRoughness_doc
"Accessor to roughness of the distribution.

Returns
-------
r : float
    Roughness of the distribution.

Notes
-----
The roughness of the distribution is defined as the :math:`\\cL^2`-norm of its
PDF:

.. math::

    r = \\int_{\\supp{\\vect{X}}} f_{\\vect{X}}(\\vect{x})^2 \\di{\\vect{x}}

See Also
--------
computePDF"
%enddef
%feature("docstring") OT::DistributionImplementation::getRoughness
OT_Distribution_getRoughness_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getSample_doc
"Accessor to a pseudo-random sample from the distribution.

Parameters
----------
size : int
    Sample size.

Returns
-------
sample : :class:`~openturns.Sample`
    A pseudo-random sample of the distribution.

See Also
--------
getRealization, RandomGenerator, getSampleByInversion, getSampleByQMC"
%enddef
%feature("docstring") OT::DistributionImplementation::getSample
OT_Distribution_getSample_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getSampleByInversion_doc
"Accessor to a pseudo-random sample from the distribution.

Parameters
----------
size : int
    Sample size.

Returns
-------
sample : :class:`~openturns.Sample`
    A pseudo-random sample of the distribution based on conditional quantiles.

See Also
--------
getRealization, RandomGenerator, getSample, getSampleByQMC"
%enddef
%feature("docstring") OT::DistributionImplementation::getSampleByInversion
OT_Distribution_getSampleByInversion_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getSampleByQMC_doc
"Accessor to a low discrepancy sample from the distribution.

Parameters
----------
size : int
    Sample size.

Returns
-------
sample : :class:`~openturns.Sample`
    A low discrepancy sample of the distribution based on Sobol's sequences
    and conditional quantiles.

See Also
--------
getRealization, RandomGenerator, getSample, getSampleByInversion"
%enddef
%feature("docstring") OT::DistributionImplementation::getSampleByQMC
OT_Distribution_getSampleByQMC_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getShapeMatrix_doc
"Accessor to the shape matrix of the underlying copula if it is elliptical.

Returns
-------
shape : :class:`~openturns.CorrelationMatrix`
    Shape matrix of the elliptical copula of a distribution.

Notes
-----
This is not the Pearson correlation matrix.

See Also
--------
getPearsonCorrelation"
%enddef
%feature("docstring") OT::DistributionImplementation::getShapeMatrix
OT_Distribution_getShapeMatrix_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getSkewness_doc
"Accessor to the componentwise skewness.

Returns
-------
d : :class:`~openturns.Point`
    Componentwise skewness.

Notes
-----
The skewness is the third-order central moment standardized by the standard deviation:

.. math::

    \\vect{\\delta} = \\Tr{\\left(\\Expect{\\left(\\frac{X_i - \\mu_i}
                                                 {\\sigma_i}\\right)^3},
                              \\quad i = 1, \\ldots, n\\right)}"
%enddef
%feature("docstring") OT::DistributionImplementation::getSkewness
OT_Distribution_getSkewness_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getSpearmanCorrelation_doc
"Accessor to the Spearman correlation matrix.

Returns
-------
R : :class:`~openturns.CorrelationMatrix`
    Spearman's correlation matrix.

Notes
-----
Spearman's (rank) correlation is defined as the normalized covariance matrix
of the copula (ie that of the uniform margins):

.. math::

    \\mat{\\rho_S} = \\left[\\frac{\\Cov{F_{X_i}(X_i), F_{X_j}(X_j)}}
                              {\\sqrt{\\Var{F_{X_i}(X_i)} \\Var{F_{X_j}(X_j)}}},
                         \\quad i,j = 1, \\ldots, n\\right]

See Also
--------
getKendallTau"
%enddef
%feature("docstring") OT::DistributionImplementation::getSpearmanCorrelation
OT_Distribution_getSpearmanCorrelation_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getStandardDeviation_doc
"Accessor to the componentwise standard deviation.

The standard deviation is the square root of the variance.

Returns
-------
sigma : :class:`~openturns.Point`
    Componentwise standard deviation.

See Also
--------
getCovariance"
%enddef
%feature("docstring") OT::DistributionImplementation::getStandardDeviation
OT_Distribution_getStandardDeviation_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getStandardDistribution_doc
"Accessor to the standard distribution.

Returns
-------
standard_distribution : :class:`~openturns.Distribution`
    Standard distribution.

Notes
-----
The standard distribution is determined according to the distribution
properties. This is the target distribution achieved by the iso-probabilistic
transformation.

See Also
--------
getIsoProbabilisticTransformation"
%enddef
%feature("docstring") OT::DistributionImplementation::getStandardDistribution
OT_Distribution_getStandardDistribution_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getStandardRepresentative_doc
"Accessor to the standard representative distribution in the parametric family.

Returns
-------
std_repr_dist : :class:`~openturns.Distribution`
    Standard representative distribution.

Notes
-----
The standard representative distribution is defined on a distribution by distribution basis, most of the time by scaling the distribution with bounded support to :math:`[0,1]` or by standardizing (ie zero mean, unit variance) the distributions with unbounded support. It is the member of the family for which orthonormal polynomials will be built using generic algorithms of orthonormalization."
%enddef
%feature("docstring") OT::DistributionImplementation::getStandardRepresentative
OT_Distribution_getStandardRepresentative_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getSupport_doc
"Accessor to the support of the distribution.

Parameters
----------
interval : :class:`~openturns.Interval`
    An interval to intersect with the support of the discrete part of the distribution.

Returns
-------
support : :class:`~openturns.Interval`
    The intersection of the support of the discrete part of the distribution with the given `interval`.

Notes
-----
The mathematical support :math:`\\supp{\\vect{X}}` of the discrete part of a distribution is the collection of points with nonzero probability.

This is yet implemented for discrete distributions only.

See Also
--------
getRange"
%enddef
%feature("docstring") OT::DistributionImplementation::getSupport
OT_Distribution_getSupport_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getProbabilities_doc
"Accessor to the discrete probability levels.

Returns
-------
probabilities : :class:`~openturns.Point`
    The probability levels of a discrete distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::getProbabilities
OT_Distribution_getProbabilities_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getSingularities_doc
"Accessor to the singularities of the PDF function.

It is defined for univariate distributions only, and gives all the singularities (ie discontinuities of any order) strictly inside of the range of the distribution.

Returns
-------
singularities : :class:`~openturns.Point`
    The singularities of the PDF of an univariate distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::getSingularities
OT_Distribution_getSingularities_doc

// ---------------------------------------------------------------------

%define OT_Distribution_hasEllipticalCopula_doc
"Test whether the copula of the distribution is elliptical or not.

Returns
-------
test : bool
    Answer.

See Also
--------
isElliptical"
%enddef
%feature("docstring") OT::DistributionImplementation::hasEllipticalCopula
OT_Distribution_hasEllipticalCopula_doc

// ---------------------------------------------------------------------

%define OT_Distribution_hasIndependentCopula_doc
"Test whether the copula of the distribution is the independent one.

Returns
-------
test : bool
    Answer."
%enddef
%feature("docstring") OT::DistributionImplementation::hasIndependentCopula
OT_Distribution_hasIndependentCopula_doc

// ---------------------------------------------------------------------

%define OT_Distribution_isContinuous_doc
"Test whether the distribution is continuous or not.

Returns
-------
test : bool
    Answer."
%enddef
%feature("docstring") OT::DistributionImplementation::isContinuous
OT_Distribution_isContinuous_doc

// ---------------------------------------------------------------------

%define OT_Distribution_isCopula_doc
"Test whether the distribution is a copula or not.

Returns
-------
test : bool
    Answer.

Notes
-----
A copula is a distribution with uniform margins on [0; 1]."
%enddef
%feature("docstring") OT::DistributionImplementation::isCopula
OT_Distribution_isCopula_doc

// ---------------------------------------------------------------------

%define OT_Distribution_isDiscrete_doc
"Test whether the distribution is discrete or not.

Returns
-------
test : bool
    Answer."
%enddef
%feature("docstring") OT::DistributionImplementation::isDiscrete
OT_Distribution_isDiscrete_doc

// ---------------------------------------------------------------------

%define OT_Distribution_isElliptical_doc
"Test whether the distribution is elliptical or not.

Returns
-------
test : bool
    Answer.

Notes
-----
A multivariate distribution is said to be *elliptical* if its characteristic
function is of the form:

.. math::

    \\phi(\\vect{t}) = \\exp\\left(i \\Tr{\\vect{t}} \\vect{\\mu}\\right)
                     \\Psi\\left(\\Tr{\\vect{t}} \\mat{\\Sigma} \\vect{t}\\right),
                     \\quad \\vect{t} \\in \\Rset^n

for specified vector :math:`\\vect{\\mu}` and positive-definite matrix
:math:`\\mat{\\Sigma}`. The function :math:`\\Psi` is known as the
*characteristic generator* of the elliptical distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::isElliptical
OT_Distribution_isElliptical_doc

// ---------------------------------------------------------------------

%define OT_Distribution_isIntegral_doc
"Test whether the distribution is integer-valued or not.

Returns
-------
test : bool
    Answer."
%enddef
%feature("docstring") OT::DistributionImplementation::isIntegral
OT_Distribution_isIntegral_doc

// ---------------------------------------------------------------------

%define OT_Distribution_setDescription_doc
"Accessor to the componentwise description.

Parameters
----------
description : sequence of str
    Description of the components of the distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::setDescription
OT_Distribution_setDescription_doc

// ---------------------------------------------------------------------

%define OT_Distribution_setParametersCollection_doc
"Accessor to the parameter of the distribution.

Parameters
----------
parameters : :class:`~openturns.PointWithDescription`
    Dictionary-like object with parameters names and values."
%enddef
%feature("docstring") OT::DistributionImplementation::setParametersCollection
OT_Distribution_setParametersCollection_doc

// ---------------------------------------------------------------------

%define OT_Distribution_cos_doc
"Transform distribution by cosine function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::cos
OT_Distribution_cos_doc

// ---------------------------------------------------------------------

%define OT_Distribution_sin_doc
"Transform distribution by sine function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::sin
OT_Distribution_sin_doc

// ---------------------------------------------------------------------

%define OT_Distribution_tan_doc
"Transform distribution by tangent function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::tan
OT_Distribution_tan_doc

// ---------------------------------------------------------------------

%define OT_Distribution_acos_doc
"Transform distribution by arccosine function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::acos
OT_Distribution_acos_doc

// ---------------------------------------------------------------------

%define OT_Distribution_asin_doc
  "Transform distribution by arcsine function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::asin
OT_Distribution_asin_doc

// ---------------------------------------------------------------------

%define OT_Distribution_atan_doc
"Transform distribution by arctangent function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::atan
OT_Distribution_atan_doc

// ---------------------------------------------------------------------

%define OT_Distribution_cosh_doc
"Transform distribution by cosh function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::cosh
OT_Distribution_cosh_doc

// ---------------------------------------------------------------------

%define OT_Distribution_sinh_doc
"Transform distribution by sinh function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::sinh
OT_Distribution_sinh_doc

// ---------------------------------------------------------------------

%define OT_Distribution_tanh_doc
"Transform distribution by tanh function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::tanh
OT_Distribution_tanh_doc

// ---------------------------------------------------------------------

%define OT_Distribution_acosh_doc
"Transform distribution by acosh function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::acosh
OT_Distribution_acosh_doc

// ---------------------------------------------------------------------

%define OT_Distribution_asinh_doc
"Transform distribution by asinh function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::asinh
OT_Distribution_asinh_doc

// ---------------------------------------------------------------------

%define OT_Distribution_atanh_doc
"Transform distribution by atanh function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::atanh
OT_Distribution_atanh_doc

// ---------------------------------------------------------------------

%define OT_Distribution_exp_doc
"Transform distribution by exponential function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::exp
OT_Distribution_exp_doc

// ---------------------------------------------------------------------

%define OT_Distribution_log_doc
"Transform distribution by natural logarithm function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::log
OT_Distribution_log_doc

// ---------------------------------------------------------------------

%define OT_Distribution_ln_doc
"Transform distribution by natural logarithm function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::ln
OT_Distribution_ln_doc

// ---------------------------------------------------------------------

%define OT_Distribution_inverse_doc
"Transform distribution by inverse function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::inverse
OT_Distribution_inverse_doc

// ---------------------------------------------------------------------

%define OT_Distribution_sqr_doc
"Transform distribution by square function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::sqr
OT_Distribution_sqr_doc

// ---------------------------------------------------------------------

%define OT_Distribution_sqrt_doc
"Transform distribution by square root function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::sqrt
OT_Distribution_sqrt_doc

// ---------------------------------------------------------------------

%define OT_Distribution_cbrt_doc
"Transform distribution by cubic root function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::cbrt
OT_Distribution_cbrt_doc

// ---------------------------------------------------------------------

%define OT_Distribution_abs_doc
"Transform distribution by absolute value function.

Returns
-------
dist : :class:`~openturns.Distribution`
    The transformed distribution."
%enddef
%feature("docstring") OT::DistributionImplementation::abs
OT_Distribution_abs_doc

// ---------------------------------------------------------------------

%define OT_Distribution_setIntegrationNodesNumber_doc
"Accessor to the number of Gauss integration points.

Parameters
----------
N : int
    Number of integration points."
%enddef
%feature("docstring") OT::DistributionImplementation::setIntegrationNodesNumber
OT_Distribution_setIntegrationNodesNumber_doc

// ---------------------------------------------------------------------

%define OT_Distribution_getIntegrationNodesNumber_doc
"Accessor to the number of Gauss integration points.

Returns
-------
N : int
    Number of integration points."
%enddef
%feature("docstring") OT::DistributionImplementation::getIntegrationNodesNumber
OT_Distribution_getIntegrationNodesNumber_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawSurvivalFunction_doc
"Draw the cumulative distribution function.

Available constructors:
    drawSurvivalFunction(*x_min, x_max, pointNumber, logScale*)

    drawSurvivalFunction(*lowerCorner, upperCorner, pointNbrInd, logScaleX, logScaleY*)

    drawSurvivalFunction(*lowerCorner, upperCorner*)

Parameters
----------
x_min : float, optional
    The min-value of the mesh of the x-axis.
    Defaults uses the quantile associated to the probability level
    `Distribution-QMin` from the :class:`~openturns.ResourceMap`.
x_max : float, optional, :math:`x_{\\max} > x_{\\min}`
    The max-value of the mesh of the y-axis.
    Defaults uses the quantile associated to the probability level
    `Distribution-QMax` from the :class:`~openturns.ResourceMap`.
pointNumber : int
    The number of points that is used for meshing each axis.
    Defaults uses `DistributionImplementation-DefaultPointNumber` from the
    :class:`~openturns.ResourceMap`.
logScale : bool
    Flag to tell if the plot is done on a logarithmic scale. Default is *False*.
lowerCorner : sequence of float, of dimension 2, optional
    The lower corner :math:`[x_{min}, y_{min}]`.
upperCorner : sequence of float, of dimension 2, optional
    The upper corner :math:`[x_{max}, y_{max}]`.
pointNbrInd : :class:`~openturns.Indices`, of dimension 2
    Number of points that is used for meshing each axis.
logScaleX : bool
    Flag to tell if the plot is done on a logarithmic scale for X. Default is *False*.
logScaleY : bool
    Flag to tell if the plot is done on a logarithmic scale for Y. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the SurvivalFunction.

Notes
-----
Only valid for univariate and bivariate distributions.

See Also
--------
computeSurvivalFunction, viewer.View, ResourceMap

Examples
--------
View the SurvivalFunction of a univariate distribution:

>>> import openturns as ot
>>> dist = ot.Normal()
>>> graph = dist.drawSurvivalFunction()
>>> graph.setLegends(['normal cdf'])

View the iso-lines SurvivalFunction of a bivariate distribution:

>>> import openturns as ot
>>> dist = ot.Normal(2)
>>> graph2 = dist.drawSurvivalFunction()
>>> graph2.setLegends(['iso- normal cdf'])
>>> graph3 = dist.drawSurvivalFunction([-10, -5],[5, 10], [511, 511])
"
%enddef
%feature("docstring") OT::DistributionImplementation::drawSurvivalFunction
OT_Distribution_drawSurvivalFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawMarginal1DSurvivalFunction_doc
"Draw the cumulative distribution function of a margin.

Parameters
----------
i : int, :math:`1 \\leq i \\leq n`
    The index of the margin of interest.
x_min : float
    The starting value that is used for meshing the x-axis.
x_max : float, :math:`x_{\\max} > x_{\\min}`
    The ending value that is used for meshing the x-axis.
n_points : int
    The number of points that is used for meshing the x-axis.
logScale : bool
    Flag to tell if the plot is done on a logarithmic scale. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the SurvivalFunction of the requested margin.

See Also
--------
computeSurvivalFunction, getMarginal, viewer.View, ResourceMap

Examples
--------

>>> import openturns as ot
>>> from openturns.viewer import View
>>> distribution = ot.Normal(10)
>>> graph = distribution.drawMarginal1DSurvivalFunction(2, -6.0, 6.0, 100)
>>> view = View(graph)
>>> view.show()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawMarginal1DSurvivalFunction
OT_Distribution_drawMarginal1DSurvivalFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawMarginal2DSurvivalFunction_doc
"Draw the cumulative distribution function of a couple of margins.

Parameters
----------
i : int, :math:`1 \\leq i \\leq n`
    The index of the first margin of interest.
j : int, :math:`1 \\leq i \\neq j \\leq n`
    The index of the second margin of interest.
x_min : list of 2 floats
    The starting values that are used for meshing the x- and y- axes.
x_max : list of 2 floats, :math:`x_{\\max} > x_{\\min}`
    The ending values that are used for meshing the x- and y- axes.
n_points : list of 2 ints
    The number of points that are used for meshing the x- and y- axes.
logScaleX : bool
    Flag to tell if the plot is done on a logarithmic scale for X. Default is *False*.
logScaleY : bool
    Flag to tell if the plot is done on a logarithmic scale for Y. Default is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graphical representation of the marginal SurvivalFunction of the requested couple of
    margins.

See Also
--------
computeSurvivalFunction, getMarginal, viewer.View, ResourceMap

Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> distribution = ot.Normal(10)
>>> graph = distribution.drawMarginal2DSurvivalFunction(2, 3, [-6.0] * 2, [6.0] * 2, [100] * 2)
>>> view = View(graph)
>>> view.show()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawMarginal2DSurvivalFunction
OT_Distribution_drawMarginal2DSurvivalFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeUpperTailDependenceMatrix_doc
"Compute the upper tail dependence coefficients.

We assume that the distribution is :math:`d`-dimensional, with :math:`d \\geq 2` and that its copula is
denoted by :math:`C`.
The upper tail dependence matrix is :math:`(\\chi^{ij})_{1 \\leq i, j \\leq d}` where :math:`\\chi^{ij}` is
the upper tail dependence coefficient of the bivariate distribution :math:`(X_i, X_j)`.
It is defined by:

.. math::

    \\chi^{ij} = \\lim_{u \\to 1} \\chi^{ij}(u)

where  :math:`u \\mapsto \\chi^{ij}(u)` is the upper tail dependence function of the bivariate distribution
:math:`(X_i, X_j)` defined by:

.. math::

    \\chi(u) = 2 - \\frac{\\log C(u,u)}{\\log u}, \\forall u \\in [0,1]

Refer to :ref:`tail_dependence` to get more details.

Returns
-------
dep : :class:`~openturns.CorrelationMatrix`
    The dependence matrix.

Examples
--------
>>> import openturns as ot
>>> copula = ot.FrankCopula()
>>> chi = copula.computeUpperTailDependenceMatrix()[1, 0]"
%enddef
%feature("docstring") OT::DistributionImplementation::computeUpperTailDependenceMatrix
OT_Distribution_computeUpperTailDependenceMatrix_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeUpperExtremalDependenceMatrix_doc
"Compute the upper extremal dependence coefficients.

We assume that the distribution is :math:`d`-dimensional, with :math:`d \\geq 2` and that its copula is
denoted by :math:`C`.
The upper extremal dependence matrix is :math:`(\\bar{\\chi}^{ij})_{1 \\leq i, j \\leq d}` where
:math:`\\bar{\\chi}^{ij}` is the upper extremal dependence coefficient of the bivariate distribution
:math:`(X_i, X_j)`.
It is defined by:

.. math::

    \\bar{\\chi}^{ij} = \\lim_{u \\to 1} \\bar{\\chi}^{ij}(u)

where  :math:`u \\mapsto \\bar{\\chi}^{ij}(u)` is the upper extremal dependence function of the bivariate
distribution :math:`(X_i, X_j)` defined by:

.. math::

    \\bar{\\chi}(u) = \\frac{2 \\log 1-u}{\\log \\bar{C}(u,u)} - 1, \\forall u \\in [0,1]

Refer to :ref:`tail_dependence` to get more details.

Returns
-------
dep : :class:`~openturns.CorrelationMatrix`
    The dependence matrix.

Examples
--------
>>> import openturns as ot
>>> copula = ot.FrankCopula()
>>> chib = copula.computeUpperExtremalDependenceMatrix()[1, 0]"
%enddef
%feature("docstring") OT::DistributionImplementation::computeUpperExtremalDependenceMatrix
OT_Distribution_computeUpperExtremalDependenceMatrix_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeLowerTailDependenceMatrix_doc
"Compute the lower tail dependence coefficients.

We assume that the distribution is :math:`d`-dimensional, with :math:`d \\geq 2` and that
its copula is denoted by :math:`C`.
The lower tail dependence matrix is :math:`(\\chi_L^{ij})_{1 \\leq i, j \\leq d}` where
:math:`\\chi_L^{ij}` is the lower tail dependence coefficient of the bivariate distribution
:math:`(X_i, X_j)`.
It is defined by:

.. math::

    \\chi_L^{ij} = \\lim_{u \\to 0} \\chi_L^{ij}(u)

where  :math:`u \\mapsto \\chi_L^{ij}(u)` is the lower tail dependence function of the bivariate
distribution :math:`(X_i, X_j)` defined by:

.. math::

    \\chi_L(u) = \\frac{\\log (1 - C(u,u))}{\\log (1-u)}, \\forall u \\in [0,1]

Refer to :ref:`tail_dependence` to get more details.

Returns
-------
dep : :class:`~openturns.CorrelationMatrix`
    The dependence matrix.

Examples
--------
>>> import openturns as ot
>>> copula = ot.FrankCopula()
>>> chiL = copula.computeLowerTailDependenceMatrix()[1, 0]"
%enddef
%feature("docstring") OT::DistributionImplementation::computeLowerTailDependenceMatrix
OT_Distribution_computeLowerTailDependenceMatrix_doc

// ---------------------------------------------------------------------

%define OT_Distribution_computeLowerExtremalDependenceMatrix_doc
"Compute the lower extremal dependence coefficients.

We assume that the distribution is :math:`d`-dimensional, with :math:`d \\geq 2` and that
its copula is denoted by :math:`C`.
The lower extremal dependence matrix is :math:`(\\bar{\\chi}_L^{ij})_{1 \\leq i, j \\leq d}` where
:math:`\\\\bar{\\chi}_L^{ij}` is the lower extremal dependence coefficient of the bivariate distribution
:math:`(X_i, X_j)`.
It is defined by:

.. math::

    \\bar{\\chi}_L^{ij} = \\lim_{u \\to 0} \\bar{\\chi}_L^{ij}(u)

where  :math:`u \\mapsto \\bar{\\chi}_L^{ij}(u)` is the lower extremal dependence function
of the bivariate distribution :math:`(X_i, X_j)` defined by:

.. math::
    
    \\bar{\\chi}_L(u) = \\frac{2 \\log u}{\\log C(u,u)} - 1, \\forall u \\in [0,1]

Refer to :ref:`tail_dependence` to get more details.

Returns
-------
dep : :class:`~openturns.CorrelationMatrix`
    The dependence matrix.

Examples
--------
>>> import openturns as ot
>>> copula = ot.FrankCopula()
>>> chiLb = copula.computeLowerExtremalDependenceMatrix()[1, 0]"
%enddef
%feature("docstring") OT::DistributionImplementation::computeLowerExtremalDependenceMatrix
OT_Distribution_computeLowerExtremalDependenceMatrix_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawUpperTailDependenceFunction_doc
"Draw the upper tail dependence function.

We assume that the distribution is bivariate and that its copula is denoted by :math:`C`.
The upper tail dependence function :math:`u \\mapsto \\chi(u)` is defined by:

.. math::

    \\chi(u) = 2 - \\frac{\\log C(u,u)}{\\log u}, \\forall u \\in [0,1]

Refer to :ref:`tail_dependence` to get more details.

Returns
-------
graph : :class:`~openturns.Graph`
    Graph of the function :math:`u \\mapsto \\chi(u)`.

Examples
--------
>>> import openturns as ot
>>> copula = ot.FrankCopula()
>>> graph = copula.drawUpperTailDependenceFunction()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawUpperTailDependenceFunction
OT_Distribution_drawUpperTailDependenceFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawUpperExtremalDependenceFunction_doc
"Draw the upper extremal dependence function.

We assume that the distribution is bivariate and that its copula is denoted by :math:`C`.
The upper extremal dependence function :math:`u \\mapsto \\bar{\\chi}(u)` is defined by:

.. math::

    \\bar{\\chi}(u) = \\frac{2 \\log 1-u}{\\log \\bar{C}(u,u)} - 1, \\forall u \\in [0,1]

Refer to :ref:`tail_dependence` to get more details.

Returns
-------
graph : :class:`~openturns.Graph`
    Graph of the function :math:`u \\mapsto \\bar{\\chi}(u)`.

Examples
--------
>>> import openturns as ot
>>> copula = ot.FrankCopula()
>>> graph = copula.drawUpperExtremalDependenceFunction()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawUpperExtremalDependenceFunction
OT_Distribution_drawUpperExtremalDependenceFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawLowerTailDependenceFunction_doc
"Draw the lower tail dependence function.

We assume that the distribution is bivariate and that its copula is denoted by :math:`C`.
The lower tail dependence  function :math:`u \\mapsto \\chi_L(u)` is defined by:

.. math::

    \\chi_L(u) = \\frac{\\log (1 - C(u,u))}{\\log (1-u)}, \\forall u \\in [0,1]

Refer to :ref:`tail_dependence` to get more details.

Returns
-------
graph : :class:`~openturns.Graph`
    Graph of the function :math:`u \\mapsto \\chi_L(u)`.

Examples
--------
>>> import openturns as ot
>>> copula = ot.FrankCopula()
>>> graph = copula.drawLowerTailDependenceFunction()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawLowerTailDependenceFunction
OT_Distribution_drawLowerTailDependenceFunction_doc

// ---------------------------------------------------------------------

%define OT_Distribution_drawLowerExtremalDependenceFunction_doc
"Draw the lower extremal dependence function.

We assume that the distribution is bivariate and that its copula is denoted by :math:`C`.
The lower extremal dependence function :math:`u \\mapsto \\bar{\\chi}(u)` is defined by:

.. math::
    
    \\bar{\\chi}_L(u) = \\frac{2 \\log u}{\\log C(u,u)} - 1, \\forall u \\in [0,1]
    
Refer to :ref:`tail_dependence` to get more details.

Returns
-------
graph : :class:`~openturns.Graph`
    The graph of the function :math:`u \\mapsto \\bar{\\chi}_L(u)`.

Examples
--------
>>> import openturns as ot
>>> copula = ot.FrankCopula()
>>> graph = copula.drawLowerExtremalDependenceFunction()"
%enddef
%feature("docstring") OT::DistributionImplementation::drawLowerExtremalDependenceFunction
OT_Distribution_drawLowerExtremalDependenceFunction_doc
