%feature("docstring") OT::MethodOfMomentsFactory
"Estimation by method of moments.

Parameters
----------
distribution : :class:`~openturns.Distribution`
    The distribution defining the parametric model to be adjusted to data.
momentsOrder : sequence of int
    The orders of moments to estimate (1 for mean, 2 for variance, etc)
bounds : :class:`~openturns.Interval`, optional
    Parameter bounds

Notes
-----
This method fits a scalar distribution to data of dimension 1, using the method of moments.

Let :math:`(\\vect{x}_1, \\dots, \\vect{x}_n)` denote the sample, :math:`F_{\\vect{\\theta}}`
the  cumulative distribution function we want to fit to the sample,
and :math:`\\vect{\\theta} \\in  \\Theta \\subset\\Rset^p`  its parameter vector.

Let :math:`K` denote the number of parameters of the distribution and we assume that 
the :math:`K` first moments of the distribution exist.

Let :math:`(\\mu_1, \\dots, \\mu_K)` denote the :math:`K` first central moments of the sample and
and :math:`(m_1, \\dots, m_K)` those of the parametric model.

The estimator :math:`\\hat{\\theta}` minimizes the sum of slacks between
:math:`(\\mu_1, \\dots, \\mu_K)` and :math:`(m_1, \\dots, m_K)`. It is defined by:

.. math::

    \\Delta = \\argmin_{\\vect{\\theta} \\in \\Rset^K} \\left[ \\dfrac{|\\mu_1-m_1|}{\\sigma} \\right]^2 + \\sum_{i=2}^K \\left[ \\dfrac{|\\mu_i|^{1/i}-|m_i|^{1/i}}{\\sigma} \\right]^2


See also
--------
DistributionFactory

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Normal(0.9, 1.7)
>>> sample = distribution.getSample(10)
>>> orders = [1, 2]  # mean, variance
>>> factory = ot.MethodOfMomentsFactory(ot.Normal(), orders)
>>> inf_distribution = factory.build(sample)

With parameter bounds:

>>> bounds = ot.Interval([0.8, 1.6], [1.0, 1.8])
>>> factory = ot.MethodOfMomentsFactory(ot.Normal(), orders, bounds)
>>> inf_distribution = factory.build(sample)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setOptimizationAlgorithm
"Accessor to the solver.

Parameters
----------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the moments."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getOptimizationAlgorithm
"Accessor to the solver.

Returns
-------
solver : :class:`~openturns.OptimizationAlgorithm`
    The solver used for numerical optimization of the moments."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setOptimizationBounds
"Accessor to the optimization bounds.

Parameters
----------
bounds : :class:`~openturns.Interval`
    The bounds used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getOptimizationBounds
"Accessor to the optimization bounds.

Returns
-------
bounds : :class:`~openturns.Interval`
    The bounds used for numerical optimization of the likelihood."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setKnownParameter
"Accessor to the known parameters.

Parameters
----------
values : sequence of float
    Values of fixed parameters.
indices : sequence of int
    Indices of fixed parameters.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Beta(2.3, 2.2, -1.0, 1.0)
>>> sample = distribution.getSample(10)
>>> orders = [3, 4]  # skewness, kurtosis
>>> factory = ot.MethodOfMomentsFactory(ot.Beta(), orders)
>>> # set (a,b) out of (r, t, a, b)
>>> factory.setKnownParameter([-1.0, 1.0], [2, 3])
>>> inf_distribution = factory.build(sample)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getKnownParameterValues
"Accessor to the known parameters indices.

Returns
-------
values : :class:`~openturns.Point`
    Values of fixed parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getKnownParameterIndices
"Accessor to the known parameters indices.

Returns
-------
indices : :class:`~openturns.Indices`
    Indices of fixed parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::buildFromMoments
"Build from moments.

Parameters
----------
moments : sequence of float
    Consists in the mean followed by consecutive central moments from order 2
    (variance), of total size at least the distribution parameter dimension.

Returns
-------
dist : :class:`~openturns.Distribution`
    Estimated distribution.

Notes
-----
Depending on the parametric model choosed, not all moments define a valid
distribution, so it should only used with empirical moments from the same model.

Examples
--------
>>> import openturns as ot
>>> distribution = ot.Beta(2.3, 2.2, -1.0, 1.0)
>>> factory = ot.MethodOfMomentsFactory(ot.Beta(), [1, 2, 3, 4])
>>> cm = [distribution.getCentralMoment(i + 2)[0] for i in range(3)]
>>> moments = [distribution.getMean()[0]] + cm
>>> inf_distribution = factory.buildFromMoments(moments)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::setMomentOrders
"Accessor to the moment orders.

Parameters
----------
momentsOrder : sequence of int
    The orders of moments to estimate (1 for mean, 2 for variance, etc)"

// ---------------------------------------------------------------------

%feature("docstring") OT::MethodOfMomentsFactory::getMomentOrders
"Accessor to the moment orders.

Returns
-------
momentsOrder : sequence of int
    The orders of moments to estimate (1 for mean, 2 for variance, etc)"
