%feature("docstring") OT::DiracCovarianceModel
"Dirac covariance function.

Available constructors:
    DiracCovarianceModel(*spatialDim=1*)

    DiracCovarianceModel(*spatialDim, amplitude*)

    DiracCovarianceModel(*spatialDim, amplitude, spatialCorrelation*)

    DiracCovarianceModel(*spatialDim, spatialCovariance*)

Parameters
----------
spatialDim : int
    Spatial dimension :math:`n`.
    By default, equal to 1.
amplitude : sequence of positive floats
    Amplitude of the process :math:`\\vect{\\sigma}\\in \\Rset^d`.
    Its size is the dimension :math:`d` of the process.
    By default, equal to :math:`[1]`.
spatialCorrelation : :class:`~openturns.CorrelationMatrix`
    Correlation matrix :math:`\\mat{R} \\in \\cS^+_d([-1, 1])`.
    By default, Identity matrix.
spatialCovariance : :class:`~openturns.CovarianceMatrix`
    Covariance matrix :math:`\\mat{C}^{stat} \\in \\cS_d^+(\\Rset)`.
    By default, Identity matrix.

Notes
-----
The *Dirac* covariance function is a stationary covariance function with dimension :math:`d \\geq 1`.

We consider the stochastic process :math:`X: \\Omega \\times\\cD \\mapsto \\Rset^d`, where :math:`\\omega \\in \\Omega` is an event, :math:`\\cD` is a domain of :math:`\\Rset^n`.

The  *Dirac* covariance function is defined by:

.. math::
    
    C(\\vect{s}, \\vect{t}) =  1_{\\{\\vect{s}=\\vect{t}\\}} \\, \\mbox{Diag}(\\vect{\\sigma}) \\, \\mat{R}\\,  \\mbox{Diag}(\\vect{\\sigma}), \\quad \\forall (\\vect{s}, \\vect{t}) \\in \\cD

where :math:`\\mat{R} \\in \\cS_d^+([-1,1])` is the spatial correlation matrix. We can define the spatial covariance matrix :math:`\\mat{C}^{stat}` as:

.. math::
    
    \\mat{C}^{stat} = \\mbox{Diag}(\\vect{\\sigma}) \\, mat{R}\\,  \\mbox{Diag}(\\vect{\\sigma})

The correlation function :math:`\\rho` writes:

.. math::

    \\rho(\\vect{s}, \\vect{t}) = 1_{\\{\\vect{s}=\\vect{t}\\}}


See Also
--------
CovarianceModel

Examples
--------
Create a standard Dirac covariance function:


>>> import openturns as ot
>>> covModel = ot.DiracCovarianceModel(2)
>>> t = [0.1, 0.3]
>>> s = [0.1, 0.3]
>>> print(covModel(s, t))
[[ 1 ]]
>>> tau = [0.1, 0.3]
>>> print(covModel(tau))
[[ 0 ]]

Create a  Dirac covariance function specifying the amplitude vector:

>>> covModel2 = ot.DiracCovarianceModel(2, [1.5, 2.5])

Create a  Dirac covariance function specifying the amplitude vector and the correlation matrix:

>>> corrMat = ot.CorrelationMatrix(2)
>>> corrMat[1,0] = 0.1
>>> covModel3 =  ot.DiracCovarianceModel(2, [1.5, 2.5], corrMat)"
