%define OT_LeastSquaresMethod_doc
"Base class for least square solvers.

Available constructors:
    LeastSquaresMethod(*proxy, weight, indices*)

    LeastSquaresMethod(*proxy, indices*)

    LeastSquaresMethod(*design*)

Parameters
----------
proxy : :class:`~openturns.DesignProxy`
    Input sample
weight : sequence of float
    Output weights
indices : sequence of int
    Indices allowed in the basis
design : 2-d sequence of float
    A priori known design matrix

See also
--------
CholeskyMethod, SVDMethod, QRMethod

Notes
-----
Solve the least-squares problem:

.. math::

    \\vect{a} = \\argmin_{\\vect{b} \\in \\Rset^P} ||y - \\vect{b}^{\\intercal} \\vect{\\Psi}(\\vect{U})||^2

Examples
--------
>>> import openturns as ot
>>> A = ot.Matrix([[1, 1], [1, 2], [1, 3], [1, 4]])
>>> y = [6, 5, 7, 10]
>>> method = ot.LeastSquaresMethod(A)
>>> x = method.solve(y)
>>> print(x)
[3.5,1.4]
"
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation
OT_LeastSquaresMethod_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getInputSample_doc
"Input sample accessor.

Returns
-------
inputSample : :class:`~openturns.Sample`
    Input sample."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getInputSample
OT_LeastSquaresMethod_getInputSample_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getWeight_doc
"Accessor to the weights.

Returns
-------
weight : :class:`~openturns.Point`
    Weights."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getWeight
OT_LeastSquaresMethod_getWeight_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getBasis_doc
"Accessor to the basis.

Returns
-------
basis : collection of :class:`~openturns.Function`
    Basis."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getBasis
OT_LeastSquaresMethod_getBasis_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getCurrentIndices_doc
"Current indices accessor.

Returns
-------
indices : :class:`~openturns.Indices`
    Indices of the current decomposition in the global basis."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getCurrentIndices
OT_LeastSquaresMethod_getCurrentIndices_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getInitialIndices_doc
"Initial indices accessor.

Returns
-------
indices : :class:`~openturns.Indices`
    Initial indices of the terms in the global basis."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getInitialIndices
OT_LeastSquaresMethod_getInitialIndices_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_solve_doc
"Solve the least-squares problem.

.. math::

    \\vect{a} = \\argmin_{\\vect{x} \\in \\Rset^P} ||M\\vect{x}-\\vect{b}||^2

Parameters
----------
b : sequence of float
    Second term of the equation

Returns
-------
a : :class:`~openturns.Point`
    The solution."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::solve
OT_LeastSquaresMethod_solve_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_solveNormal_doc
"Solve the least-squares problem using normal equation.

.. math::

    M^T*M*x=M^T*b

Parameters
----------
b : sequence of float
    Second term of the equation

Returns
-------
x : :class:`~openturns.Point`
    The solution."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::solveNormal
OT_LeastSquaresMethod_solveNormal_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getGramInverse_doc
"Get the inverse Gram matrix of input sample.

.. math::

    G^{-1} = (X^T * X)^{-1}

Returns
-------
c : :class:`~openturns.CovarianceMatrix`
    The inverse Gram matrix."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getGramInverse
OT_LeastSquaresMethod_getGramInverse_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getGramInverseDiag_doc
"Get the diagonal of the inverse Gram matrix.

.. math::

    diag(G^{-1}) = diag((X^T * X)^{-1})

Returns
-------
d : :class:`~openturns.Point`
    The diagonal of the inverse Gram matrix."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getGramInverseDiag
OT_LeastSquaresMethod_getGramInverseDiag_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getGramInverseTrace_doc
"Get the trace of the inverse Gram matrix.

.. math::

    Tr(G^{-1}) = Tr(x^T * x)^{-1}

Returns
-------
x : float
    The trace of inverse Gram matrix."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getGramInverseTrace
OT_LeastSquaresMethod_getGramInverseTrace_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getH_doc
"Get the projection matrix H.

.. math::

    H = X * (X^T * X)^{-1} * X^T

Returns
-------
h : :class:`~openturns.SymmetricMatrix`
    The projection matrix H."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getH
OT_LeastSquaresMethod_getH_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_getHDiag_doc
"Get the diagonal of the projection matrix H.

.. math::

    H = X * (X^T * X)^{-1} * X^T

Returns
-------
d : :class:`~openturns.Point`
    The diagonal of H."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::getHDiag
OT_LeastSquaresMethod_getHDiag_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_computeWeightedDesign_doc
"Build the design matrix.

Parameters
----------
whole : bool, defaults to False
    Whether to use the initial indices instead of the current indices

Returns
-------
psiAk : :class:`~openturns.Matrix`
    The design matrix"
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::computeWeightedDesign
OT_LeastSquaresMethod_computeWeightedDesign_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_trashDecomposition_doc
"Drop the current decomposition."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::trashDecomposition
OT_LeastSquaresMethod_trashDecomposition_doc

// ---------------------------------------------------------------------

%define OT_LeastSquaresMethod_update_doc
"Update the current decomposition.

Parameters
----------
addedIndices : sequence of int
    Indices of added basis terms.
conservedIndices : sequence of int
    Indices of conserved basis terms.
removedIndices : sequence of int
    Indices of removed basis terms."
%enddef
%feature("docstring") OT::LeastSquaresMethodImplementation::update
OT_LeastSquaresMethod_update_doc


