%feature("docstring") OT::UserDefinedCovarianceModel
"Covariance model defined by the User.

Parameters
----------
mesh : :class:`~openturns.Mesh`
    A mesh which contains `N` vertices.
matrix : :class:`~openturns.CovarianceMatrix`
    A symmetric positive definite matrix of dimension :math:`Nd` corresponding to the discretization of the model over the given mesh.

Notes
-----
Let :math:`\\mat{\\Sigma}` be the given covariance matrix, :math:`s_i` and :math:`s_j` be two vertices of the mesh, with :math:`i,j\\in\\{0,\\dots,N-1\\}`. The :math:`d\\times d` sub-matrix :math:`\\mat{S}=(\\Sigma_{id+k,jd+\\ell})_{k,\\ell\\in\\{0,\\dots,d-1}` is equal to :math:`C(s_i,s_j)`, where :math:`C` is the user defined covariance model.

Examples
--------
>>> import openturns as ot
>>> import math as m

Create the covariance function at (s,t):

>>> def C(s, t):
...     return m.exp( -4.0 * abs(s - t) / (1 + (s * s + t * t)))

Create the time grid:

>>> N = 32
>>> a = 4.0
>>> myMesh = ot.IntervalMesher([N]).build(ot.Interval(-a, a))

Create the large covariance matrix:

>>> myCovariance = ot.CovarianceMatrix(myMesh.getVerticesNumber())
>>> for k in range(myMesh.getVerticesNumber()):
...     t = myMesh.getVertices()[k]
...     for l in range(k + 1):
...         s = myMesh.getVertices()[l]
...         myCovariance[k, l] = C(s[0], t[0])

Create the covariance model:

>>> myCovarianceModel = ot.UserDefinedCovarianceModel(myMesh, myCovariance)
"

// ---------------------------------------------------------------------
%feature("docstring") OT::UserDefinedCovarianceModel::getMesh
"Accessor to the mesh.

Returns
-------
mesh : :class:`~openturns.Mesh`
    The mesh associated to the collection of covariance matrices.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::UserDefinedCovarianceModel::getTimeGrid
"Accessor to the time grid.

Returns
-------
mesh : :class:`~openturns.RegularGrid`
    The time grid associated to the collection of covariance matrices when the mesh can be interpreted as a regular time grid.
"
