"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const http = require("http");
const https = require("https");
const url_1 = require("url");
const defaults_1 = require("./defaults");
const proxy_errors_1 = require("./proxy-errors");
function get(opts, useHttps = false) {
    return __awaiter(this, void 0, void 0, function* () {
        const send = useHttps ? https.get : http.get;
        return new Promise((resolve, reject) => {
            const req = send(opts, (res) => {
                if (res.statusCode < 200 || res.statusCode > 299) {
                    reject(res);
                }
                else {
                    resolve(res);
                }
            });
            req.on("error", e => reject(e));
        });
    });
}
/**
 * Validate the given proxy setting
 * @param setting Proxy setting to validate.
 *
 * @throws {ProxyConnectionRefusedError} if it cannot connect to the proxy
 * @throws {ProxyAuthenticationRequiredError} if proxy settings doesn't have credentials but is required
 * @throws {ProxyInvalidCredentialsError} if proxy settings has credentials but proxy denies the request(407)
 * @throws {GetProxyError} for other errors
 */
function validateProxySetting(setting) {
    return __awaiter(this, void 0, void 0, function* () {
        const auth = setting.getAuthorizationHeader();
        const testUrl = new url_1.URL(defaults_1.defaults.testUrl);
        try {
            yield get({
                host: setting.host,
                port: setting.port,
                path: defaults_1.defaults.testUrl,
                headers: {
                    "Connection": "close",
                    "Host": testUrl.host,
                    "Proxy-Authorization": auth,
                    "User-Agent": "Mozilla/5.0",
                },
                agent: false,
            });
            return true;
        }
        catch (e) {
            if (e.code === "ECONNREFUSED") {
                throw new proxy_errors_1.ProxyConnectionRefusedError(setting);
            }
            else if (e.statusCode === 407) {
                if (setting.credentials) {
                    throw new proxy_errors_1.ProxyInvalidCredentialsError(setting);
                }
                else {
                    throw new proxy_errors_1.ProxyAuthenticationRequiredError(setting);
                }
            }
            else {
                throw new proxy_errors_1.GetProxyError(setting, `Error validating proxy. Returned ${e.statusCode} ${e.statusMessage}`);
            }
        }
    });
}
exports.validateProxySetting = validateProxySetting;
//# sourceMappingURL=validate.js.map