"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const core_1 = require("@oclif/core");
const fs_1 = tslib_1.__importDefault(require("fs"));
const js_yaml_1 = tslib_1.__importDefault(require("js-yaml"));
const mustache_1 = tslib_1.__importDefault(require("mustache"));
const fsPromises = tslib_1.__importStar(require("node:fs/promises"));
const path = tslib_1.__importStar(require("node:path"));
const converter_1 = tslib_1.__importDefault(require("../classes/converter"));
const utils_1 = tslib_1.__importDefault(require("../classes/utils"));
const utils_2 = require("../lib/utils");
const scripts = {
    /* eslint-disable no-useless-escape */
    bin: (config) => `#!/usr/bin/env bash
set -e
echoerr() { echo "$@" 1>&2; }
get_script_dir () {
  SOURCE="\${BASH_SOURCE[0]}"
  # While \$SOURCE is a symlink, resolve it
  while [ -h "\$SOURCE" ]; do
    DIR="\$( cd -P "\$( dirname "\$SOURCE" )" && pwd )"
    SOURCE="\$( readlink "\$SOURCE" )"
    # If \$SOURCE was a relative symlink (so no "/" as prefix, need to resolve it relative to the symlink base directory
    [[ \$SOURCE != /* ]] && SOURCE="\$DIR/\$SOURCE"
  done
  DIR="\$( cd -P "\$( dirname "\$SOURCE" )" && pwd )"
  echo "\$DIR"
}
DIR=\$(get_script_dir)
export ${config.scopedEnvVarKey('UPDATE_INSTRUCTIONS')}="update with \\"sudo apt update && sudo apt install ${config.bin}\\""
\$DIR/node \$DIR/run "\$@"
`,
};
/**
 *
 */
class Deb extends core_1.Command {
    static args = {
        pathSource: core_1.Args.string({ description: 'pathSource', name: 'pathSource', required: false }),
    };
    static description = 'Create a deb package from your npm package';
    static flags = {
        help: core_1.Flags.help({ char: 'h' }),
        all: core_1.Flags.boolean({ char: 'a', description: 'all architectures' }),
        codename: core_1.Flags.string({ char: 'c', description: 'codename' }),
        mantain: core_1.Flags.boolean({ char: 'm' }),
        release: core_1.Flags.string({ char: 'r', description: 'release' }),
        manpages: core_1.Flags.boolean({ char: 'M', description: 'refresh manpages on the sources' }),
        verbose: core_1.Flags.boolean({ char: 'v', description: 'verbose' }),
    };
    static summary = 'Pack CLI into debian package.';
    /**
     *
     */
    async run() {
        const { args, flags } = await this.parse(Deb);
        if (process.platform !== 'linux') {
            this.log('debian packing must be run on linux');
            this.exit(0);
        }
        let manpages = flags.manpages;
        let all = flags.all;
        if (all === undefined) {
            all = false;
        }
        let { codename } = flags;
        if (codename === undefined) {
            codename = '';
        }
        let { release } = flags;
        if (release === undefined) {
            release = '1';
        }
        const { verbose } = flags;
        const echo = utils_1.default.setEcho(verbose);
        // Crea configurazione per il pacchetto
        const here = process.cwd() + '/';
        let pathSource = here;
        if (args.pathSource !== undefined) {
            if (pathSource.substring(pathSource.length, -1) === '/') {
                pathSource = args.pathSource;
            }
            else {
                pathSource = args.pathSource + '/';
            }
        }
        if (fs_1.default.existsSync(`${here}/perrisbrewery`)) {
            this.log('perrisbrewery configurations already exists');
        }
        else {
            fs_1.default.mkdirSync(`${here}/perrisbrewery`);
            await (0, utils_2.exec)(`cp -r ${path.resolve(__dirname, '../perrisbrewery.sample/*')} ${here}/perrisbrewery`, echo);
            this.log('perrisbrewery dir created in: ' + pathSource);
            this.log('Edit configuration in template e scripts. Include /perribrewery/workdir in your .gitignore.');
            this.log('Finally run pb to rebuild your packages with manpages, scripts, etc');
            process.exit(0);
        }
        if (!fs_1.default.existsSync(pathSource + 'package.json')) {
            console.log('package.json not found in ' + pathSource);
            process.exit(0);
        }
        // Decido le architetture da costruire
        let debArchs = ['amd64'];
        if (all) {
            debArchs = ['amd64', 'arm64', 'i386'];
        }
        else {
            if (process.arch === 'x64') {
                debArchs = ['amd64'];
            }
            else if (process.arch === 'arm64') {
                debArchs = ['arm64'];
            }
            else if (process.arch === 'ia32') {
                debArchs = ['i386'];
            }
        }
        if (codename === 'bionic') {
            debArchs = ['amd64'];
        }
        // per operare sul valor for .. of
        for (const debArch of debArchs) {
            this.log('');
            this.log('building arch: ' + debArch);
            const content = fs_1.default.readFileSync(pathSource + 'package.json', 'utf8');
            const packageJson = JSON.parse(content);
            let { mantainer } = packageJson;
            if (mantainer === undefined) {
                mantainer = packageJson.author;
            }
            if (mantainer === undefined) {
                mantainer = 'made on Perris\' Brewery';
            }
            const { description } = packageJson;
            const packageVersion = packageJson.version;
            const packageRelease = release;
            const packageName = packageJson.name;
            let packageNameVersioned = `${packageName}_${packageVersion}-${packageRelease}_${debArch}`;
            if (codename === 'bionic') {
                packageNameVersioned = `${packageName}_${packageVersion}-bionic-${packageRelease}_${debArch}`;
            }
            const { files } = packageJson;
            const binName = Object.keys(packageJson.bin)[0];
            const packageRoot = path.join(here, 'dist');
            const packageDir = path.join(packageRoot, packageNameVersioned);
            await Promise.all([
                fsPromises.mkdir(path.join(packageDir, 'DEBIAN'), { recursive: true }),
                fsPromises.mkdir(path.join(packageDir, 'usr', 'bin'), { recursive: true }),
                fsPromises.mkdir(path.join(packageDir, 'usr', 'lib', packageName), { recursive: true }),
                fsPromises.mkdir(path.join(packageDir, 'usr', 'lib', packageName, 'manpages', 'doc', 'man'), { recursive: true }),
            ]);
            this.log('creating package skel complete');
            // find package dependencies
            let dependenciesFile = "dependencies.yaml";
            if (codename === 'bionic') {
                console.log("codename: bionic");
                dependenciesFile = "dependencies-bionic.yaml";
            }
            let fileContents = fs_1.default.readFileSync(`${here}/perrisbrewery/template/${dependenciesFile}`, 'utf8');
            const dep = js_yaml_1.default.load(fileContents);
            let packages = dep.common;
            packages = packages.concat(dep.arch[debArch]);
            packages.sort();
            // create debian control file
            const depends = array2comma(packages);
            const template = fs_1.default.readFileSync('./perrisbrewery/template/control.template', 'utf8');
            const view = {
                arch: debArch,
                depends,
                description,
                mantainer,
                name: packageName,
                priority: 'standard',
                section: 'main',
                version: `${packageVersion}-${packageRelease}`,
            };
            fs_1.default.writeFileSync(`${packageDir}/DEBIAN/control`, mustache_1.default.render(template, view));
            this.log('>>> creating debian control file complete');
            // include debian scripts
            await (0, utils_2.exec)(`cp ./perrisbrewery/scripts/* ${packageDir}/DEBIAN/`, echo);
            this.log('>>> included debian scripts: postinst, postrm, preinst, prerm');
            // create man page
            await (0, utils_2.exec)(`cp ./README.md  ${packageDir}/DEBIAN/`, echo);
            const converter = new converter_1.default(pathSource + '/README.md');
            await converter.readme2md(packageDir, packageName, packageVersion, binName, packageNameVersioned, verbose);
            await converter.md2man(packageDir, packageName, packageVersion, binName, verbose);
            await converter.md2html(packageDir, packageName, packageVersion, binName, verbose);
            this.log('>>> created man page complete');
            if (manpages) {
                this.log('>>> refresh manpages on the sources');
                await (0, utils_2.exec)(`rm -rf ${here}/manpages`, echo);
                await (0, utils_2.exec)(`cp ${packageDir}/usr/lib/${packageName}/manpages ${here} -R`, echo);
            }
            // copia i file del pacchetto
            const rootLib = `${packageDir}/usr/lib/${packageName}`;
            await (0, utils_2.exec)(`cp -r ${pathSource}/LICENSE ${rootLib}`, echo);
            await (0, utils_2.exec)(`cp -r ${pathSource}/node_modules  ${rootLib}`, echo);
            await (0, utils_2.exec)(`cp -r ${pathSource}/package.json  ${rootLib}`, echo);
            // copia il lock file
            if (fs_1.default.existsSync(pathSource + '/pnpm-lock.yaml')) {
                await (0, utils_2.exec)(`cp -r ${pathSource}/pnpm-lock.yaml  ${rootLib}`, echo);
            }
            if (fs_1.default.existsSync(pathSource + '/yarn.lock')) {
                await (0, utils_2.exec)(`cp -r ${pathSource}/yarn.lock  ${rootLib}`, echo);
            }
            if (fs_1.default.existsSync(pathSource + '/npm-shrinkwrap.json')) {
                await (0, utils_2.exec)(`cp -r ${pathSource}/npm-shrinkwrap.json  ${rootLib}`, echo);
            }
            // copia i file del pacchetto
            for (const file in files) {
                await (0, utils_2.exec)(`cp -r ${pathSource}/${files[file]} ${rootLib}`, echo);
            }
            this.log('>>> imported node package complete');
            // create link to node on rootLib
            await (0, utils_2.exec)(`ln -s /usr/bin/node ${rootLib}/bin/node`);
            this.log('>>> created link node');
            // create binName
            fs_1.default.writeFileSync(`${rootLib}/bin/${binName}`, scripts.bin(this.config));
            await (0, utils_2.exec)(`chmod 755 ${rootLib}/bin/${binName}`);
            this.log(`>>> created exec ${binName}`);
            const curDir = process.cwd();
            process.chdir(`${packageDir}/usr/bin`);
            await (0, utils_2.exec)(`ln -sf ../lib/${packageName}/bin/${binName} ${binName}`);
            process.chdir(curDir);
            this.log(`>>> created a link on /usr/bin/ for ${binName}`);
            await (0, utils_2.exec)(`sudo chown -R root "${packageDir}"`);
            await (0, utils_2.exec)(`sudo chgrp -R root "${packageDir}"`);
            await (0, utils_2.exec)(`dpkg-deb --build ${packageDir}`);
            await (0, utils_2.exec)(`sudo rm -rf ${packageRoot}/${packageNameVersioned}`);
            this.log(`finished building ${packageNameVersioned}.deb`);
            this.log(`Creating sha256sum on ${packageDir}.deb`);
            let savedDir = process.cwd();
            process.chdir(packageRoot);
            await (0, utils_2.exec)(`sha256sum ${packageNameVersioned}.deb > ${packageNameVersioned}.deb.sha256`);
            process.chdir(savedDir);
        }
        this.log('Complete!');
    }
}
exports.default = Deb;
/**
 * @param packages array packages
 */
function array2comma(packages) {
    return packages.join(',\n         ');
    //return packages.join(', ')
}
