"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityPlugin = void 0;
var _configSchema = require("@osd/config-schema");
var _server = require("../../../src/core/server");
var _shared = require("../common/constants/shared");
var _migrations = require("./adaptors/integrations/migrations");
var _opensearch_observability_plugin = require("./adaptors/opensearch_observability_plugin");
var _ppl_plugin = require("./adaptors/ppl_plugin");
var _ppl_parser = require("./parsers/ppl_parser");
var _register_settings = require("./plugin_helper/register_settings");
var _index = require("./routes/index");
var _observability_saved_object = require("./saved_objects/observability_saved_object");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class ObservabilityPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    this.logger = initializerContext.logger.get();
  }
  async setup(core, deps) {
    const {
      assistantDashboards,
      dataSource
    } = deps;
    this.logger.debug('Observability: Setup');
    const router = core.http.createRouter();
    const dataSourceEnabled = !!dataSource;
    const openSearchObservabilityClient = core.opensearch.legacy.createClient('opensearch_observability', {
      plugins: [_ppl_plugin.PPLPlugin, _opensearch_observability_plugin.OpenSearchObservabilityPlugin]
    });
    if (dataSourceEnabled) {
      dataSource.registerCustomApiSchema(_ppl_plugin.PPLPlugin);
      dataSource.registerCustomApiSchema(_opensearch_observability_plugin.OpenSearchObservabilityPlugin);
    }
    // @ts-ignore
    core.http.registerRouteHandlerContext('observability_plugin', (_context, _request) => {
      return {
        logger: this.logger,
        observabilityClient: openSearchObservabilityClient
      };
    });
    const obsPanelType = {
      name: 'observability-panel',
      hidden: false,
      namespaceType: 'single',
      mappings: {
        dynamic: false,
        properties: {
          title: {
            type: 'text'
          },
          description: {
            type: 'text'
          }
        }
      },
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: dataSourceEnabled ? '' : `/app/${_shared.observabilityPanelsID}#/${obj.id}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return `Observability Settings [${obj.id}]`;
        }
      },
      migrations: {
        '3.0.0': doc => ({
          ...doc,
          description: ''
        }),
        '3.0.1': doc => ({
          ...doc,
          description: 'Some Description Text'
        }),
        '3.0.2': doc => ({
          ...doc,
          dateCreated: parseInt(doc.dateCreated || '0', 10)
        })
      }
    };
    const integrationInstanceType = {
      name: 'integration-instance',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/installed/${obj.id}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          templateName: {
            type: 'text'
          },
          dataSource: {
            type: 'text'
          },
          creationDate: {
            type: 'date'
          },
          assets: {
            type: 'nested'
          }
        }
      }
    };
    const integrationTemplateType = {
      name: 'integration-template',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/available/${obj.attributes.name}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          var _obj$attributes$displ;
          return (_obj$attributes$displ = obj.attributes.displayName) !== null && _obj$attributes$displ !== void 0 ? _obj$attributes$displ : obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          version: {
            type: 'text'
          },
          displayName: {
            type: 'text'
          },
          license: {
            type: 'text'
          },
          type: {
            type: 'text'
          },
          labels: {
            type: 'text'
          },
          author: {
            type: 'text'
          },
          description: {
            type: 'text'
          },
          sourceUrl: {
            type: 'text'
          },
          statics: {
            type: 'nested'
          },
          components: {
            type: 'nested'
          },
          assets: {
            type: 'nested'
          },
          sampleData: {
            type: 'nested'
          }
        }
      },
      migrations: {
        '3.0.0': _migrations.migrateV1IntegrationToV2Integration
      }
    };
    core.savedObjects.registerType(obsPanelType);
    core.savedObjects.registerType(integrationInstanceType);
    core.savedObjects.registerType(integrationTemplateType);

    // Register server side APIs
    (0, _index.setupRoutes)({
      router,
      client: openSearchObservabilityClient,
      dataSourceEnabled,
      logger: this.logger
    });
    core.savedObjects.registerType((0, _observability_saved_object.getVisualizationSavedObject)(dataSourceEnabled));
    core.savedObjects.registerType((0, _observability_saved_object.getSearchSavedObject)(dataSourceEnabled));
    core.savedObjects.registerType(_observability_saved_object.notebookSavedObject);
    core.capabilities.registerProvider(() => ({
      observability: {
        show: true
      }
    }));
    assistantDashboards === null || assistantDashboards === void 0 || assistantDashboards.registerMessageParser(_ppl_parser.PPLParsers);
    (0, _register_settings.registerObservabilityUISettings)(core.uiSettings);
    core.uiSettings.register({
      'observability:defaultDashboard': {
        name: 'Observability default dashboard',
        value: '',
        description: 'The default dashboard to display in Observability overview page',
        schema: _configSchema.schema.string(),
        scope: core.workspace.isWorkspaceEnabled() ? _server.UiSettingScope.WORKSPACE : _server.UiSettingScope.GLOBAL
      }
    });
    core.uiSettings.register({
      'observability:overviewCardsDisplay': {
        name: 'Observability overview cards',
        value: true,
        description: 'Show the Observability overview page cards',
        schema: _configSchema.schema.boolean(),
        scope: core.workspace.isWorkspaceEnabled() ? _server.UiSettingScope.WORKSPACE : _server.UiSettingScope.GLOBAL
      }
    });
    return {};
  }
  start(_core) {
    this.logger.debug('Observability: Started');
    return {};
  }
  stop() {}
}
exports.ObservabilityPlugin = ObservabilityPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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