"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.backendToUiReportsList = exports.backendToUiReportDefinitionsList = exports.backendToUiReportDefinition = exports.backendToUiReport = void 0;
var _model = require("../../../model");
var _backendModel = require("../../../model/backendModel");
var _constants = require("../constants");
var _moment = _interopRequireDefault(require("moment"));
var _url = require("url");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const backendToUiReport = (backendReportInstance, serverBasePath) => {
  const {
    inContextDownloadUrlPath,
    beginTimeMs,
    endTimeMs,
    tenant,
    status,
    lastUpdatedTimeMs: reportLastUpdatedTimeMs,
    createdTimeMs: reportCreatedTimeMs,
    reportDefinitionDetails: backendReportDefinitionDetails
  } = backendReportInstance;
  const {
    reportDefinition: {
      source: {
        type: sourceType,
        id: sourceId
      },
      delivery
    }
  } = backendReportDefinitionDetails;
  const baseUrl = getBaseUrl(sourceType, sourceId);
  let report = {
    // inContextDownloadUrlPath may not exist for report instance created from scheduled job
    query_url: inContextDownloadUrlPath || getUiQueryUrl(baseUrl, beginTimeMs, endTimeMs, tenant),
    time_from: beginTimeMs,
    time_to: endTimeMs,
    last_updated: reportLastUpdatedTimeMs,
    time_created: reportCreatedTimeMs,
    state: getUiReportState(status, delivery),
    report_definition: backendToUiReportDefinition(backendReportDefinitionDetails, serverBasePath)
  };

  // Add severbasePath back to query_url
  report.query_url = serverBasePath + report.query_url;
  return report;
};
exports.backendToUiReport = backendToUiReport;
const backendToUiReportsList = (backendReportsList, serverBasePath) => {
  const res = backendReportsList.map(backendReport => {
    return {
      _id: backendReport.id,
      _source: backendToUiReport(backendReport, serverBasePath)
    };
  });
  return res;
};
exports.backendToUiReportsList = backendToUiReportsList;
const backendToUiReportDefinition = (backendReportDefinitionDetails, serverBasePath) => {
  const {
    lastUpdatedTimeMs,
    createdTimeMs,
    reportDefinition: {
      name,
      isEnabled,
      source: {
        type: sourceType,
        description,
        id: sourceId,
        origin
      },
      format: {
        fileFormat,
        duration,
        header,
        footer,
        limit
      },
      trigger: {
        triggerType,
        schedule
      },
      delivery
    }
  } = backendReportDefinitionDetails;
  const baseUrl = getBaseUrl(sourceType, sourceId);
  const reportSource = getUiReportSource(sourceType);
  let uiReportDefinition = {
    report_params: {
      report_name: name,
      report_source: reportSource,
      description: description,
      core_params: reportSource === _constants.REPORT_TYPE.savedSearch ? getDataReportCoreParams(limit, sourceId, fileFormat, duration, baseUrl, origin) : getVisualReportCoreParams(fileFormat, header, footer, duration, baseUrl, origin)
    },
    trigger: getUiTriggerParams(triggerType, schedule, createdTimeMs, isEnabled),
    delivery: getUiDeliveryParams(delivery),
    //TODO:
    time_created: createdTimeMs,
    last_updated: lastUpdatedTimeMs,
    status: getUiReportDefinitionStatus(isEnabled)
  };
  // validate to assign default values to some fields for UI model
  uiReportDefinition = _model.reportDefinitionSchema.validate(uiReportDefinition);
  uiReportDefinition.report_params.core_params.base_url = serverBasePath + uiReportDefinition.report_params.core_params.base_url;
  return uiReportDefinition;
};
exports.backendToUiReportDefinition = backendToUiReportDefinition;
const backendToUiReportDefinitionsList = (backendReportDefinitionDetailsList, serverBasePath) => {
  const res = backendReportDefinitionDetailsList.map(backendReportDefinitionDetails => {
    return {
      _id: backendReportDefinitionDetails.id,
      _source: {
        // TODO: this property can be removed, but need UI changes as well
        report_definition: backendToUiReportDefinition(backendReportDefinitionDetails, serverBasePath)
      }
    };
  });
  return res;
};
exports.backendToUiReportDefinitionsList = backendToUiReportDefinitionsList;
const getVisualReportCoreParams = (fileFormat, header = '', footer = '', duration, baseUrl, origin) => {
  let res = {
    base_url: baseUrl,
    report_format: getUiReportFormat(fileFormat),
    header: header,
    footer: footer,
    time_duration: duration,
    origin: origin
  };
  return res;
};

// queryUrl = baseUrl + time range
const getUiQueryUrl = (baseUrl, beginTimeMs, endTimeMs, tenant) => {
  const timeFrom = (0, _moment.default)(beginTimeMs).toISOString();
  const timeTo = (0, _moment.default)(endTimeMs).toISOString();
  let queryUrl = `${baseUrl}?_g=(time:(from:'${timeFrom}',to:'${timeTo}'))`;
  if (tenant !== undefined) {
    if (tenant === '') {
      tenant = 'global';
    } else if (tenant === '__user__') {
      tenant = 'private';
    }
    queryUrl = addTenantToURL(queryUrl, tenant);
  }
  return queryUrl;
};
const getBaseUrl = (sourceType, sourceId) => {
  //TODO: AES domain has different prefix, need figure out a general solution
  const baseUrl = `${_backendModel.URL_PREFIX_DICT[sourceType]}${sourceId}`;
  return baseUrl;
};
const getDataReportCoreParams = (limit = _constants.DEFAULT_MAX_SIZE, sourceId, fileFormat, duration, baseUrl, origin) => {
  let res = {
    base_url: baseUrl,
    report_format: getUiReportFormat(fileFormat),
    limit: limit,
    time_duration: duration,
    saved_search_id: sourceId,
    origin: origin
  };
  return res;
};
const getUiScheduleParams = (schedule, createdTimeMs, isEnabled) => {
  let res = {
    trigger_params: {
      enabled_time: createdTimeMs,
      enabled: isEnabled,
      schedule_type: schedule && 'cron' in schedule ? _constants.SCHEDULE_TYPE.cron : _constants.SCHEDULE_TYPE.recurring,
      schedule: schedule
    }
  };
  return res;
};
const getUiTriggerType = backendField => {
  let res;
  for (let [ui, backendFieldList] of Object.entries(_backendModel.TRIGGER_TYPE_DICT)) {
    for (let item of backendFieldList) {
      if (item === backendField) {
        res = ui;
      }
    }
  }
  return res;
};
const getUiReportFormat = backendField => {
  let res;
  for (let [ui, backend] of Object.entries(_backendModel.REPORT_FORMAT_DICT)) {
    if (backend === backendField) {
      res = ui;
    }
  }
  return res;
};
const getUiReportState = (status, delivery) => {
  let res;
  for (let [ui, backend] of Object.entries(_backendModel.REPORT_STATE_DICT)) {
    if (backend === status) {
      // distinguish "shared" and "created"
      if (status === _backendModel.BACKEND_REPORT_STATE.success && delivery) {
        res = _constants.REPORT_STATE.shared;
      } else {
        res = ui;
      }
    } else if (status === _backendModel.BACKEND_REPORT_STATE.scheduled) {
      // corner case
      res = _constants.REPORT_STATE.pending;
    }
  }
  return res;
};
const getUiReportSource = type => {
  let res;
  for (let [ui, backend] of Object.entries(_backendModel.REPORT_SOURCE_DICT)) {
    if (backend === type) {
      res = ui;
    }
  }
  return res;
};
const getUiReportDefinitionStatus = isEnabled => {
  return isEnabled ? _constants.REPORT_DEFINITION_STATUS.active : _constants.REPORT_DEFINITION_STATUS.disabled;
};
const getUiTriggerParams = (triggerType, schedule, createdTimeMs, isEnabled) => {
  let res = {
    trigger_type: getUiTriggerType(triggerType),
    ...(getUiTriggerType(triggerType) === _constants.TRIGGER_TYPE.schedule && getUiScheduleParams(schedule, createdTimeMs, isEnabled))
  };
  return res;
};

// Delivery
const getUiDeliveryParams = delivery => {
  const opensearchDashboardsUserDeliveryParams = {
    configIds: [],
    title: '',
    textDescription: '',
    htmlDescription: ''
  };
  let params;
  if (delivery) {
    const {
      ...rest
    } = delivery;
    params = {
      ...rest
    };
  } else {
    params = opensearchDashboardsUserDeliveryParams;
  }
  return params;
};

// helper function to add tenant info to url(if tenant is available)
const addTenantToURL = (url, userRequestedTenant) => {
  // build fake url from relative url
  const fakeUrl = `http://opensearch.com${url}`;
  const tenantKey = 'security_tenant';
  const tenantKeyAndValue = tenantKey + '=' + encodeURIComponent(userRequestedTenant);
  const {
    pathname,
    search
  } = (0, _url.parse)(fakeUrl);
  const queryDelimiter = !search ? '?' : '&';

  // The url parser returns null if the search is empty. Change that to an empty
  // string so that we can use it to build the values later
  if (search && search.toLowerCase().indexOf(tenantKey) > -1) {
    // If we for some reason already have a tenant in the URL we skip any updates
    return url;
  }

  // A helper for finding the part in the string that we want to extend/replace
  const valueToReplace = pathname + (search || '');
  const replaceWith = valueToReplace + queryDelimiter + tenantKeyAndValue;
  return url.replace(valueToReplace, replaceWith);
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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