"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metaData = exports.getSelectedFields = exports.getOpenSearchData = exports.convertToExcel = exports.convertToCSV = exports.buildRequestBody = void 0;
var _elasticBuilder = _interopRequireDefault(require("elastic-builder"));
var _json2Csv = _interopRequireDefault(require("json-2-csv"));
var _lodash = _interopRequireDefault(require("lodash"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _common = require("../../../../../src/plugins/data/common");
var _excelBuilder = require("./excelBuilder");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

var metaData = exports.metaData = {
  saved_search_id: null,
  report_format: null,
  start: null,
  end: null,
  fields: null,
  type: null,
  timeFieldName: null,
  sorting: null,
  fields_exist: false,
  selectedFields: [],
  paternName: null,
  searchSourceJSON: [],
  dateFields: []
};

// Get the selected columns by the user.
const getSelectedFields = async columns => {
  const selectedFields = [];
  let fields_exist = false;
  for (let column of columns) {
    if (column !== '_source') {
      fields_exist = true;
      selectedFields.push(column);
    } else {
      fields_exist = false;
      selectedFields.push('_source');
    }
  }
  metaData.fields_exist = fields_exist;
  metaData.selectedFields = selectedFields;
};

// Build the OpenSearch query from the meta data
// is_count is set to 1 if we building the count query but 0 if we building the fetch data query
exports.getSelectedFields = getSelectedFields;
const buildRequestBody = (report, allowLeadingWildcards, is_count) => {
  let esbBoolQuery = _elasticBuilder.default.boolQuery();
  const searchSourceJSON = report._source.searchSourceJSON;
  const savedObjectQuery = JSON.parse(searchSourceJSON).query;
  const savedObjectFilter = JSON.parse(searchSourceJSON).filter;
  const savedObjectConfig = {
    allowLeadingWildcards: allowLeadingWildcards,
    queryStringOptions: {},
    ignoreFilterIfFieldNotInIndex: false
  };
  const QueryFromSavedObject = (0, _common.buildOpenSearchQuery)(undefined, savedObjectQuery, savedObjectFilter, savedObjectConfig);
  // Add time range
  if (report._source.timeFieldName && report._source.timeFieldName.length > 0) {
    esbBoolQuery.must(_elasticBuilder.default.rangeQuery(report._source.timeFieldName).format('epoch_millis').gte(report._source.start - 1).lte(report._source.end + 1));
  }
  if (is_count) {
    return _elasticBuilder.default.requestBodySearch().query(esbBoolQuery);
  }

  // Add sorting to the query
  let esbSearchQuery = _elasticBuilder.default.requestBodySearch().query(esbBoolQuery).version(true);
  let sorting = report._source.sorting;

  // We expect a list of [field, order] pairs for sorting. In some migration paths, though it's not
  // clear why, this list can get unnested in the case of one sort, [["field", "asc"]] becomes
  // ["field", "asc"]. The true root cause remains a mystery, so we work around it.
  // See: https://github.com/opensearch-project/dashboards-reporting/issues/371
  if (sorting.length > 0 && typeof sorting[0] === "string") {
    sorting = [sorting];
  }
  if (sorting.length > 0) {
    const sorts = sorting.map(element => {
      return _elasticBuilder.default.sort(element[0], element[1]);
    });
    esbSearchQuery.sorts(sorts);
  }

  // add selected fields to query
  if (report._source.fields_exist) {
    esbSearchQuery.source({
      includes: report._source.selectedFields
    });
  }
  // Add a customizer to merge queries to generate request body
  let requestBody = _lodash.default.mergeWith({
    query: QueryFromSavedObject
  }, esbSearchQuery.toJSON(), (objValue, srcValue) => {
    if (_lodash.default.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  });
  requestBody = addDocValueFields(report, requestBody);
  return requestBody;
};

// Fetch the data from OpenSearch
exports.buildRequestBody = buildRequestBody;
const getOpenSearchData = (arrayHits, report, params, dateFormat, timezone) => {
  let hits = [];
  for (let valueRes of arrayHits) {
    for (let data of valueRes.hits) {
      const fields = data.fields;
      // get all the fields of type date and format them to excel format
      let tempKeyElement = [];
      for (let dateField of report._source.dateFields) {
        let keys;
        keys = dateField.split('.');
        const dateValue = data._source[dateField];
        const fieldDateValue = fields !== undefined ? fields[dateField] : undefined;
        const isDateFieldPresent = isKeyPresent(data._source, dateField);
        if (isDateFieldPresent) {
          // if its not a nested date field
          if (keys.length === 1) {
            // if conditions to determine if the date field's value is an array or a string
            if (typeof dateValue === 'string') {
              data._source[keys] = _momentTimezone.default.utc(dateValue).tz(timezone).format(dateFormat);
            } else if (dateValue.length !== 0 && dateValue instanceof Array && fieldDateValue !== undefined) {
              fieldDateValue.forEach((element, index) => {
                data._source[keys][index] = _momentTimezone.default.utc(element).tz(timezone).format(dateFormat);
              });
            } else {
              data._source[keys] = [];
            }
            // else to cover cases with nested date fields
          } else {
            let keyElement = keys.shift();
            // if conditions to determine if the date field's value is an array or a string
            if (fieldDateValue !== undefined && typeof fieldDateValue === 'string') {
              keys.push(_momentTimezone.default.utc(fieldDateValue).tz(timezone).format(dateFormat));
            } else if (dateValue.length !== 0 && dateValue instanceof Array && fieldDateValue !== undefined) {
              let tempArray = [];
              fieldDateValue.forEach(index => {
                tempArray.push(_momentTimezone.default.utc(index).tz(timezone).format(dateFormat));
              });
              keys.push(tempArray);
            } else {
              keys.push([]);
            }
            const nestedJSON = arrayToNestedJSON(keys);
            let keyLength = Object.keys(data._source);
            // to check if the nested field have anyother keys apart from date field
            if (tempKeyElement.includes(keyElement) || keyLength.length > 1) {
              data._source[keyElement] = {
                ...data._source[keyElement],
                ...nestedJSON
              };
            } else {
              data._source[keyElement] = nestedJSON;
              tempKeyElement.push(keyElement);
            }
          }
        }
      }
      delete data['fields'];
      if (report._source.fields_exist === true) {
        let result = traverse(data, report._source.selectedFields);
        hits.push(params.excel ? sanitize(result) : result);
      } else {
        hits.push(params.excel ? sanitize(data) : data);
      }
      // Truncate to expected limit size
      if (hits.length >= params.limit) {
        return hits;
      }
    }
  }
  return hits;
};

// Convert the data to Csv format
exports.getOpenSearchData = getOpenSearchData;
const convertToCSV = async (dataset, csvSeparator) => {
  let convertedData = [];
  const options = {
    delimiter: {
      field: csvSeparator,
      eol: '\n'
    },
    emptyFieldValue: ' '
  };
  await _json2Csv.default.json2csvAsync(dataset[0], options).then(csv => {
    convertedData = csv;
  });
  return convertedData;
};
exports.convertToCSV = convertToCSV;
function flattenHits(hits, result = {}, prefix = '') {
  Object.entries(hits).forEach(([key, value]) => {
    if (value !== null && typeof value === 'object' && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenHits(value, result, `${prefix}${key}.`);
    } else {
      result[`${prefix.replace(/^_source\./, '')}${key}`] = value;
    }
  });
  return result;
}
function flattenObject(obj = {}, parentKey = '', result = {}) {
  for (const [key, value] of Object.entries(obj)) {
    const newKey = parentKey ? `${parentKey}.${key}` : key;
    if (typeof value == 'object' && value !== null && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenObject(value, newKey, result);
    } else if (Array.isArray(value)) {
      result[newKey] = JSON.stringify(value);
    } else {
      result[newKey] = value;
    }
  }
  return result;
}
function flattenArray(array = []) {
  return array.map(item => flattenObject(item));
}
const convertToExcel = async dataset => {
  const flatDataset = flattenArray(dataset[0]);
  const excelBuilder = new _excelBuilder.ExcelBuilder();
  const base64 = await excelBuilder.addHeaders(flatDataset).addRows(flatDataset).updateColumnWidths().applyHeaderStyles().applyAutoFilter().applyFreeze().getWorkbookAsBase64();
  return 'data:application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;base64,' + base64;
};

//Return only the selected fields
exports.convertToExcel = convertToExcel;
function traverse(data, keys, result = {}) {
  // Flatten the data if necessary (ensure all nested fields are at the top level)
  data = flattenHits(data);
  keys.forEach(key => {
    const value = _lodash.default.get(data, key, undefined);
    if (value !== undefined) {
      result[key] = value;
    } else {
      const flattenedValues = {};
      Object.keys(data).forEach(dataKey => {
        if (dataKey.startsWith(key + '.')) {
          result[dataKey] = data[dataKey];
        }
        const arrayValue = data[dataKey];
        if (Array.isArray(arrayValue)) {
          arrayValue.forEach(item => {
            if (typeof item === 'object' && item !== null) {
              Object.keys(item).forEach(subKey => {
                const newKey = `${dataKey}.${subKey}`;
                if (!flattenedValues[newKey]) {
                  flattenedValues[newKey] = [];
                }
                flattenedValues[newKey].push(item[subKey]);
              });
            }
          });
        }
      });
      Object.keys(flattenedValues).forEach(newKey => {
        result[newKey] = flattenedValues[newKey];
      });
    }
  });
  return result;
}

/**
 * Escape special characters if field value prefixed with.
 * This is intend to avoid CSV injection in Microsoft Excel.
 * @param doc   document
 */
function sanitize(doc) {
  for (const field in doc) {
    if (doc[field] == null) continue;
    if (doc[field].toString().startsWith('+') || doc[field].toString().startsWith('-') && typeof doc[field] !== 'number' || doc[field].toString().startsWith('=') || doc[field].toString().startsWith('@')) {
      doc[field] = "'" + doc[field];
    }
  }
  return doc;
}
function arrayToNestedJSON(arr) {
  if (arr.length === 0) {
    return null;
  } else if (arr.length === 1) {
    return arr[0];
  } else {
    const key = arr[0];
    const rest = arr.slice(1);
    return {
      [key]: arrayToNestedJSON(rest)
    };
  }
}
function isKeyPresent(data, key) {
  if (typeof data === 'object' && data !== null) {
    if (key in data) {
      return true;
    }
    for (const value of Object.values(data)) {
      if (isKeyPresent(value, key)) {
        return true;
      }
    }
  }
  return false;
}
const addDocValueFields = (report, requestBody) => {
  const docValues = [];
  for (const dateType of report._source.dateFields) {
    docValues.push({
      field: dateType,
      format: 'date_hour_minute_second_fraction'
    });
  }
  // elastic-builder doesn't provide function to build docvalue_fields with format,
  // this is a workaround which appends docvalues field to the request body.
  requestBody = {
    ...requestBody,
    docvalue_fields: docValues
  };
  return requestBody;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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