# (c) Copyright 2009-2011, 2015. CodeWeavers, Inc.

import bottlequery
import bottlewrapper
import cxobjc
import distversion


class BottleCollection(cxobjc.Proxy):

    bottleList = cxobjc.array_property()

    def __init__(self):
        self.bottleList = []
        self.ignored_bottles = []

        self.changeDelegates = []
        self.bottleChangeDelegates = []
        self.refresh()

    def init(self):
        self = cxobjc.Proxy.nsobject_init(self)
        if self is not None:
            self.__init__()
        return self

    @cxobjc.namedSelector(b'refresh')
    def refresh(self):
        bottle_list = bottlequery.get_bottle_list()
        changed = False

        # Remove deleted bottles
        for bottle in self.bottles():
            if bottle.name not in bottle_list and not bottle.is_renaming:
                self.removeBottle(bottle.name)
                changed = True

        # Remove existing bottles from the new bottle list
        for bottle in self.bottles():
            if bottle.name in bottle_list:
                bottle_list.remove(bottle.name)
            elif bottle.changeablename in bottle_list:
                bottle_list.remove(bottle.changeablename)

        # Remove ignored bottles from the new bottle list
        for bottle_name in self.ignored_bottles:
            if bottle_name in bottle_list:
                bottle_list.remove(bottle_name)

        # Add new bottles
        for bottle_name in bottle_list:
            self.addBottle(bottle_name)
            changed = True

        if changed:
            for delegate in self.changeDelegates:
                delegate.bottleCollectionChanged()

    @cxobjc.namedSelector(b'reset')
    def reset(self):
        for bottle_name in self.bottle_names():
            self.removeBottle(bottle_name)

        self.refresh()

    @cxobjc.python_method
    def add_ignored_bottle(self, new_name):
        self.ignored_bottles.append(new_name)

    @cxobjc.python_method
    def remove_ignored_bottle(self, new_name):
        self.ignored_bottles.remove(new_name)

    @cxobjc.namedSelector(b'refreshDefaultBottle')
    def refreshDefaultBottle(self):
        default_bottle_name = bottlequery.get_default_bottle()
        for bottle in self.bottleList:
            bottle.is_default = (bottle.name == default_bottle_name)

    @cxobjc.namedSelector(b'bottleWithName:')
    def bottle_with_name(self, bottle_name):
        for bottle in self.bottleList:
            if bottle.name == bottle_name:
                return bottle

        return None

    @cxobjc.python_method
    def bottle_names(self):
        return [bottle.name for bottle in self.bottleList]

    @cxobjc.python_method
    def bottles(self):
        return list(self.bottleList)

    @cxobjc.python_method
    def addBottle(self, bottle_name):
        # Make sure this isn't already in the list.
        for bottle in self.bottles():
            if bottle.name == bottle_name:
                return

        if distversion.IS_MACOSX:
            bottle = bottlewrapper.BottleWrapper.alloc().initWithName_(bottle_name)
        else:
            bottle = bottlewrapper.BottleWrapper(bottle_name)

        self.bottleList.append(bottle)
        bottle.add_menu_observers()

        for delegate in self.bottleChangeDelegates:
            bottle.add_change_delegate(delegate)

        bottle.add_change_delegate(self)

        self.refreshDefaultBottle()

    @cxobjc.python_method
    def removeBottle(self, bottle_name):
        bottle = self.bottle_with_name(bottle_name)
        if not bottle:
            return

        self.bottleList.remove(bottle)
        bottle.remove_change_delegate(self)
        bottle.remove_observers()
        self.refreshDefaultBottle()

    @cxobjc.namedSelector(b'addChangeDelegate:')
    def addChangeDelegate(self, inDelegate):
        self.changeDelegates.append(inDelegate)

    @cxobjc.python_method
    def removeChangeDelegate(self, inDelegate):
        self.changeDelegates.remove(inDelegate)

    @cxobjc.python_method
    def addBottleChangeDelegate(self, inDelegate):
        self.bottleChangeDelegates.append(inDelegate)
        for bottleobj in self.bottles():
            bottleobj.add_change_delegate(inDelegate)

    @cxobjc.python_method
    def removeBottleChangeDelegate(self, inDelegate):
        self.bottleChangeDelegates.remove(inDelegate)
        for bottleobj in self.bottles():
            bottleobj.remove_change_delegate(inDelegate)


@cxobjc.method(BottleCollection, 'sharedCollection')
def sharedCollection():
    # pylint: disable=W0601
    global sharedBottleCollection
    try:
        return sharedBottleCollection
    except NameError:
        if distversion.IS_MACOSX:
            sharedBottleCollection = BottleCollection.alloc().init()
        else:
            sharedBottleCollection = BottleCollection()

    return sharedBottleCollection
