# (c) Copyright 2022. CodeWeavers, Inc.

import traceback

from gi.repository import GLib
from gi.repository import Gtk

import bottlecollection
import bottlemanagement
import bottlequery

import cxdiag
import cxguitools
import cxproduct
import cxutils

import pyop
import ratingdialog

from cxutils import cxgettext as _


def archive_bottle(bottle, window):
    file_chooser = Gtk.FileChooserDialog(
        title=_("Choose an Archive Name and Location"),
        transient_for=window,
        action=Gtk.FileChooserAction.SAVE)

    cxguitools.add_filters(file_chooser, cxguitools.FILTERS_CXARCHIVES)
    file_chooser.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
    button = file_chooser.add_button(Gtk.STOCK_SAVE, Gtk.ResponseType.OK)
    button.get_style_context().add_class('suggested-action')
    file_chooser.set_current_name(bottle.name + ".cxarchive")

    file_chooser.set_do_overwrite_confirmation(True)

    cxguitools.set_default_extension(file_chooser, 'cxarchive')

    if file_chooser.run() == Gtk.ResponseType.OK:
        filename = file_chooser.get_filename()

        def callback(operation):
            if not operation.exit_status[0]:
                cxguitools.CXMessageDlg(
                    primary=_('An error occurred while archiving %s') % operation.bottle.name,
                    secondary=operation.exit_status[1], message_type=Gtk.MessageType.ERROR,
                    parent=window)

        archiveOp = ArchiveBottleOperation(bottle, filename, callback)
        pyop.sharedOperationQueue.enqueue(archiveOp)

    file_chooser.destroy()


def delete_bottle(bottle, window):
    primary_text = _("Are you sure you want to delete the '%s' bottle?") % bottle.name
    secondary_text = _("Deleting this bottle will permanently erase all Windows applications contained in it, as well as any documents stored in its C: drive.")

    buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
               (Gtk.STOCK_DELETE, Gtk.ResponseType.OK, "destructive-action"))

    def delete_bottle_callback():
        if bottle.is_deleted:
            bottlecollection.sharedCollection().refresh()

    def delete_bottle_response(response):
        if response != Gtk.ResponseType.OK:
            return

        bottle.delete(delete_bottle_callback)

        config = cxproduct.get_config()
        ask_for_ratings = config['CrossOver'].get('AskForRatings', '1') != '0'
        if bottle.appid and ask_for_ratings:
            ratingdialog.RatingController(bottle, window, True, False)

    cxguitools.CXMessageDlg(primary=primary_text, secondary=secondary_text, button_array=buttons,
                            response_function=delete_bottle_response,
                            message_type=Gtk.MessageType.WARNING, parent=window)


def open_c_drive(bottle):
    if not bottle.is_usable:
        return

    drive = bottlequery.get_system_drive(bottle.name)
    args = ['xdg-open', drive]
    try:
        cxutils.run(args, background=True)
    except: # pylint: disable=W0702
        traceback.print_exc()


def quit_bottle(bottle, force, callback=None):
    if force:
        bottle.force_quit(callback)
    else:
        bottle.quit(callback)


def set_esync_state(bottle, state, window, callback=None):
    if state != (bottle.is_esync_enabled_state == bottle.STATUS_ESYNC_ENABLED):
        if state:
            primary_text = _("Some Windows applications will not work correctly with ESync enabled.")
            buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
                       (_("Reboot Bottle and Enable ESync"), Gtk.ResponseType.OK))
        else:
            primary_text = None
            buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
                       (_("Reboot Bottle and Disable ESync"), Gtk.ResponseType.OK))

        secondary_text = _("Changing this setting will reboot the bottle and cause any running applications in the bottle to quit. Are you sure you want to do this?")

        def toggle_esync_response(response):
            if response == Gtk.ResponseType.OK:
                if state:
                    bottle.enable_esync()
                else:
                    bottle.disable_esync()

                quit_bottle(bottle, False, callback)
            else:
                bottle.bottle_changed()

        cxguitools.CXMessageDlg(primary=primary_text, secondary=secondary_text, button_array=buttons,
                                response_function=toggle_esync_response,
                                message_type=Gtk.MessageType.WARNING, parent=window)


def set_highres_state(bottle, state, window):
    if state == bottle.is_high_resolution_enabled():
        return

    if state:
        primary_text = _("Many Windows applications render incorrectly in high resolution mode.")
        secondary_text = _("Changing this setting will reboot the bottle and cause any running applications in the bottle to quit. Are you sure you want to do this?")
        buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
                   (_("Reboot Bottle and Enable High Resolution Mode"), Gtk.ResponseType.OK))
    else:
        primary_text = _("Changing this setting will reboot the bottle and cause any running applications in the bottle to quit.")
        secondary_text = _("Are you sure you want to do this?")
        buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
                   (_("Reboot Bottle and Disable High Resolution Mode"), Gtk.ResponseType.OK))

    def response_function(response):
        if response != Gtk.ResponseType.OK:
            bottle.bottle_changed()
            return

        if state:
            bottle.enable_high_resolution(int(float(cxdiag.get(None).properties.get('display.dpi', 0))))
        else:
            bottle.disable_high_resolution()

    cxguitools.CXMessageDlg(primary=primary_text, secondary=secondary_text, button_array=buttons,
                            response_function=response_function,
                            message_type=Gtk.MessageType.WARNING, parent=window)


def set_preview_state(bottle, state, window, wait=False):
    if state == bottle.is_preview_enabled():
        return

    if state:
        primary_text = _("Enabling bottles for CrossOver Preview may cause them to break permanently.")
        secondary_text = _("CrossOver Preview contains untested changes that may permanently break bottles. This is very unlikely, but not impossible. Are you sure you want to enable the '%s' bottle for CrossOver Preview?") % bottle.name
        buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
                   (_("Enable for CrossOver Preview"), Gtk.ResponseType.OK))
    else:
        primary_text = _("Disabling this bottle for CrossOver Preview will cause any running applications in the bottle to quit.")
        secondary_text = _("Are you sure you want to do this?")
        buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
                   (_("Disable for CrossOver Preview"), Gtk.ResponseType.OK))

    loop = GLib.MainLoop()

    def disable_preview_callback():
        bottle.disable_preview()
        loop.quit()

    def toggle_preview_response(response):
        if response == Gtk.ResponseType.OK:
            if state:
                bottle.enable_preview()
            else:
                quit_bottle(bottle, False, disable_preview_callback)
                return
        else:
            bottle.bottle_changed()

        loop.quit()

    cxguitools.CXMessageDlg(primary=primary_text, secondary=secondary_text, button_array=buttons,
                            response_function=toggle_preview_response,
                            message_type=Gtk.MessageType.WARNING, parent=window)

    if wait:
        loop.run()


class ArchiveBottleOperation(pyop.PythonOperation):

    def __init__(self, bottle, archive_path, callback):
        pyop.PythonOperation.__init__(self)
        self.exit_status = None
        self.bottle = bottle
        self.archive_path = archive_path
        self.callback = callback

    def __unicode__(self):
        return "ArchiveBottleOperation for " + self.bottle.name

    def enqueued(self):
        self.bottle.add_status_override(self.bottle.STATUS_ARCHIVING)
        pyop.PythonOperation.enqueued(self)

    def main(self):
        self.exit_status = bottlemanagement.archive_bottle(self.bottle.name, self.archive_path)

    def finish(self):
        self.bottle.remove_status_override(self.bottle.STATUS_ARCHIVING)
        self.callback(self)
        pyop.PythonOperation.finish(self)
